/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/*****************************************************************************
 * decode_rsakey.c
 *---------------------------------------------------------------------------
 * ASN.1 decode RSA key per algorithm-specific syntax
 *****************************************************************************/

#include "config.h"
#include "general.h"

#include "crypto.h"
#include "asn1.h"
#include "bbuf.h"
#include "allocate.h"
#include "util.h"

#include "./rsa.h"

/*****************************************************************************
 * decode_rsakeyparms()
 *---------------------------------------------------------------------------
 * ASN.1 decode RSA key length (as an INTEGER) from bbuf
 *****************************************************************************/

int decode_rsakeyparms(alg, parms, len)
int alg;
struct bbuf *parms;
int *len;
{
    int r, dummy;

    /* DOUBLE-CHECK FUNCTION PARAMETERS */
    if (parms == NULLBB) 
	return(NOTOK);

    r = decode_int(parms, len, &dummy);

    return(r);
}

/*****************************************************************************
 * predecode_rsakeydata()
 *---------------------------------------------------------------------------
 * ASN.1 decode RSA key enough to determine whether public or private
 *****************************************************************************/

int predecode_rsakeydata(data, alg)
struct bbuf *data;
int *alg;
{
    int r;
    int v;
    int dlen, llen, pl;
    struct bbuf bb;

    /* CHECK INPUT LENGTH */
    if (data->length < 2) 
        return(ASN1_ERR01);

    /* CHECK TAG */
    if (data->data[0] != (CLASS_UNIV|FORM_CONS|CONS_SEQ)) 
	return(ASN1_ERR02);

    /* DECODE LENGTH */
    if ((r = decode_len(&(data->data[1]), data->length-1, &dlen, &llen)) != OK)
	return(r);

    /* DECODE AND CHECK VERSION */
    bb.data = &data->data[1 + llen];
    bb.length = dlen;
    if ((r = decode_int(&bb, &v, &pl)) != OK || v != PRIVATE_KEY_VERSION) {
	*alg |= PUB;
        return(OK);
    }

    *alg |= PRIV;
    return(OK);
}

/*****************************************************************************
 * decode_rsapubkeydata()
 *---------------------------------------------------------------------------
 * ASN.1 decode RSA public key (see encode_rsakey.c)
 *****************************************************************************/

int decode_rsapubkeydata(data, Pu)
struct bbuf *data;
R_RSA_PUBLIC_KEY *Pu;
{
    int r;
    int dlen, llen, dp, dl, pl;
    struct bbuf bb;
    struct bbuf *n = NULLBB, *e = NULLBB;

    /* CHECK INPUT LENGTH */
    if (data->length < 2) 
        return(ASN1_ERR01);

    /* CHECK TAG */
    if (data->data[0] != (CLASS_UNIV|FORM_CONS|CONS_SEQ)) 
	return(ASN1_ERR02);

    /* DECODE LENGTH */
    if ((r = decode_len(&(data->data[1]), data->length-1, &dlen, &llen)) != OK)
	return(r);

    /* DECODE MODULUS N */
    bb.data = &data->data[dp = 1 + llen];
    bb.length = dl = dlen;
    if ((r = decode_ints(&bb, &n, &pl)) != OK)
        goto cleanup;

    /* DECODE ENCRYPTION EXPONENT E */
    bb.data = &data->data[dp+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &e, &pl)) != OK)
        goto cleanup;

    /* FILL IN RSAREF PUBLIC KEY STRUCTURE */
    if (n->data[0] == NULLUC) Pu->bits = (n->length-1)*8;
    else Pu->bits = n->length*8;
    if ((r = bbuf2buf(n, Pu->modulus, MAX_RSA_MODULUS_LEN)) != OK) 
	goto cleanup;
    if ((r = bbuf2buf(e, Pu->exponent, MAX_RSA_MODULUS_LEN)) != OK) 
	goto cleanup;

    r = OK;

 cleanup:

    FREE_BBUF(n);
    FREE_BBUF(e);

    return(r);
}


/*****************************************************************************
 * decode_rsaprikeydata()
 *---------------------------------------------------------------------------
 * ASN.1 decode RSA private key (see encode_rsakey.c)
 *****************************************************************************/

int decode_rsaprikeydata(data, Pr)
struct bbuf *data;
R_RSA_PRIVATE_KEY *Pr;
{
    int r;
    int v;
    int dlen, llen, dp2, dl, pl;
    struct bbuf bb;
    struct bbuf *n = NULLBB, *e = NULLBB, *d = NULLBB, *p = NULLBB,
		*q = NULLBB, *dp = NULLBB, *dq = NULLBB, *cr = NULLBB;

    /* CHECK INPUT LENGTH */
    if (data->length < 2) 
        return(ASN1_ERR01);

    /* CHECK TAG */
    if (data->data[0] != (CLASS_UNIV|FORM_CONS|CONS_SEQ)) 
	return(ASN1_ERR02);

    /* DECODE LENGTH */
    if ((r = decode_len(&(data->data[1]), data->length-1, &dlen, &llen)) != OK)
	return(r);

    /* DECODE VERSION */
    bb.data = &data->data[dp2 = 1 + llen];
    bb.length = dl = dlen;
    if ((r = decode_int(&bb, &v, &pl)) != OK)
        goto cleanup;
    if (v != PRIVATE_KEY_VERSION) {
	r = NOTOK;
	goto cleanup;
    }

    /* DECODE MODULUS N */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl = dlen;
    if ((r = decode_ints(&bb, &n, &pl)) != OK)
        goto cleanup;

    /* DECODE ENCRYPTION EXPONENT E */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &e, &pl)) != OK)
        goto cleanup;

    /* DECODE DECRYPTION EXPONENT D */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &d, &pl)) != OK)
        goto cleanup;

    /* DECODE PRIME P */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &p, &pl)) != OK)
        goto cleanup;

    /* DECODE PRIME Q */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &q, &pl)) != OK)
        goto cleanup;

    /* DECODE EXPONENT DP */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &dp, &pl)) != OK)
        goto cleanup;

    /* DECODE EXPONENT DQ */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &dq, &pl)) != OK)
        goto cleanup;

    /* DECODE COEFFICIENT CR */
    bb.data = &data->data[dp2+=pl];
    bb.length = dl -= pl;
    if ((r = decode_ints(&bb, &cr, &pl)) != OK)
        goto cleanup;

    /* FILL IN RSAREF PRIVATE KEY STRUCTURE */
    if (n->data[0] == NULLUC) Pr->bits = (n->length-1)*8;
    else Pr->bits = n->length*8;
    if ((r = bbuf2buf(n, Pr->modulus, MAX_RSA_MODULUS_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(e, Pr->publicExponent, MAX_RSA_MODULUS_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(d, Pr->exponent, MAX_RSA_MODULUS_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(p, Pr->prime[0], MAX_RSA_PRIME_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(q, Pr->prime[1], MAX_RSA_PRIME_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(dp, Pr->primeExponent[0], MAX_RSA_PRIME_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(dq, Pr->primeExponent[1], MAX_RSA_PRIME_LEN)) != OK)
	goto cleanup;
    if ((r = bbuf2buf(cr, Pr->coefficient, MAX_RSA_PRIME_LEN)) != OK)
	goto cleanup;

    r = OK;

 cleanup:

    FREE_BBUF(n);
    FREE_BBUF(e);
    FREE_BBUF(d);
    FREE_BBUF(p);
    FREE_BBUF(q);
    FREE_BBUF(dp);
    FREE_BBUF(dq);
    FREE_BBUF(cr);

    return(r);
}

/****************************************************************************/

static int bbuf2buf(bb, b, l)
struct bbuf *bb;
unsigned char *b;
int l;
{
    unsigned char *p = bb->data;
    int l2 = bb->length;

    /* ZEROIZE DATA */
    BZERO(b, l);

    /* CHECK LENGTH */
    if (l2 == l+1 && *p == NULLUC) {
        /* IGNORE EXTRA LEADING NULL BYTE */
	p++;
	l2--;
    }
    if (l2 > l) 
	return(NOTOK);

    /* COPY DATA */
    BCOPY(p, b+l-l2, l2);

    return(OK);
}

/****************************************************************************/
