/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

//------------------------------------------------------------
// includes
//------------------------------------------------------------

#include "x509.h"
#include "functiontrace.h"
#include "pkixvalidator.h"
#include "tpdefines.h"


//------------------------------------------------------------
// method: validateNameChaining 
//------------------------------------------------------------

void
pkix_validator::validateNameChaining(pkix_certificate& subject,
                                     pkix_certificate& issuer)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateNameChaining(pkix_certificate&,pkix_certificate&)");

   // collect data about name forms we have to work with

   bool subjectNull                     = subject.isSubjectNull();
   bool issuerNull                      = subject.isIssuerNull();
   bool subjectAltNamePresent           = subject.isSubjectAlternativeNamePresent();
   bool subjectAltNameCritical          = subject.isSubjectAlternativeNameCritical();
   bool issuerAltNamePresent            = subject.isIssuerAlternativeNamePresent();
   bool issuerAltNameCritical           = subject.isIssuerAlternativeNameCritical();

   bool subjectNull_inIssuer            = issuer.isSubjectNull();
   bool issuerNull_inIssuer             = issuer.isIssuerNull();
   bool subjectAltNamePresent_inIssuer  = issuer.isSubjectAlternativeNamePresent();
   bool subjectAltNameCritical_inIssuer = issuer.isSubjectAlternativeNameCritical();

   //------------------------------------------------------------
   // Comment:
   //
   // At one one point, spec says that CAs must have DNs.  This
   // would seem to guarantee that we could always chain on DNs.
   // Later, in validation section, spec says if no DNs, then 
   // chaining should use critical subjectAltNames and issuerAltNames.
   // We'll be flexible and chain on either.  It is an error for
   // subject to have an issuer DN but for issuer not to have a
   // subject DN (i.e., both must be present or both must be absent).
   // A related issue is validating subjectAltNames if subject DN
   // is null.
   //------------------------------------------------------------

   if ( issuerNull != subjectNull_inIssuer )
      throw TPException(TP_NAME_CHAINING_FAILURE);

   if ( false == issuerNull == subjectNull_inIssuer )
   {
      //------------------------------------------------------------
      // DN name chaining: operator does proper comparison in PKIX sense
      //------------------------------------------------------------

      if ( issuer.tbsCertificate.subject != subject.tbsCertificate.issuer )
            throw TPException(TP_NAME_CHAINING_FAILURE);
   }
   else
   {
      //------------------------------------------------------------
      // No DN name chaining
      //------------------------------------------------------------

      // throw TPException(TP_NAME_CHAINING_NON_NULL_SUBJECT_ISSUER_DN_REQUIRED);

      if ( !(issuerAltNamePresent && issuerAltNameCritical && subjectAltNamePresent_inIssuer && subjectAltNameCritical_inIssuer) )
;//         throw TPException(TP_NAME_CHAINING_NULL_SUBJECT_NAME_REQUIRES_CRITICAL_ALTNAMES);

      if ( !(subjectAltNamePresent && subjectAltNameCritical) )
;//         throw TPException(TP_NAME_CHAINING_NULL_SUBJECT_NAME_REQUIRES_CRITICAL_SUBJECTALTNAME_EXT);
   }

   //------------------------------------------------------------
   // UniqueID Checking:
   // (a) Expect uniqueID's must be both present/absent
   // (b) If present, uniqueID's should also chain
   //------------------------------------------------------------
   
   if ( subject.isIssuerUniqueIDPresent() != issuer.isSubjectUniqueIDPresent() )
      throw TPException(TP_NAME_CHAINING_INCONSISTENT_UNIQUE_ID_USAGE);

   if ( subject.isIssuerUniqueIDPresent() && issuer.isSubjectUniqueIDPresent() )
   {
      if ( !(subject.tbsCertificate.issuerUniqueID == issuer.tbsCertificate.subjectUniqueID) )
         throw TPException(TP_NAME_CHAINING_FAILURE);
   }

   //------------------------------------------------------------
   // SubjectAltName Checking:
   // (a) SubjectAltName should be present and critical if subject null
   // (b) SubjectAltName extension should contain only name forms we recognize;
   // (c) if subject null, should chain on critical issuer and subject AltNames
   // (d) subjectAltName and issuerAltName fields should contain
   //     only supported name form
   //------------------------------------------------------------
   
   if ( subjectAltNamePresent && subjectAltNameCritical )
   {
      validateOnlySupportedNameForms( subject.subjectAltName() );
   }
   if ( issuerAltNamePresent && issuerAltNameCritical )
   {
      validateOnlySupportedNameForms( subject.issuerAltName() );
   }
}


//------------------------------------------------------------
// method: validateNameConstraints 
//------------------------------------------------------------

void
pkix_validator::validateNameConstraints(pkix_certificate& subject)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateNameConstraints(pkix_certificate&)");

   // no requirement that NameConstraints be present

   if ( !subject.isNameConstraintsPresent() )
   {
      return;
   }

   // but if present, must be critical

   if ( !subject.isNameConstraintsCritical() )
      throw TPException(TP_NAME_CONSTRAINTS_MUST_BE_CRITICAL_EXT);

   //------------------------------------------------------------
   // (1) Does name constraints usage match that specfied in PKIX? (min = 0, no max)
   // (2) Do constraints on unsupported (or PKIX unspecified) name forms appear?
   //     e.g., constraint on OTHERNAME (unspecified) or x400Address (unsupported)
   //------------------------------------------------------------

   XNameConstraints& nameConstraints = subject.nameConstraints();
   if ( nameConstraints.permittedSubtrees.is_present() )
   {
      validateNoMinMaxInSubtrees(nameConstraints.permittedSubtrees);
      validateOnlySupportedNameForms(nameConstraints.permittedSubtrees);
   }
   if ( nameConstraints.excludedSubtrees.is_present() )
   {
      validateNoMinMaxInSubtrees(nameConstraints.excludedSubtrees);
      validateOnlySupportedNameForms(nameConstraints.excludedSubtrees);
   }
}


//------------------------------------------------------------
// method: validateOnlySupportedNameForms 
//------------------------------------------------------------

void
pkix_validator::validateOnlySupportedNameForms(GeneralNames& altNames)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateOnlySupportedNameForms(GeneralNames&)");

   for (int i = 0; i < altNames.get_child_count(); i++)
   {
      validateOnlySupportedNameForms( *altNames[i] );
   }
}


//------------------------------------------------------------
// method: validateOnlySupportedNameForms 
//------------------------------------------------------------

void
pkix_validator::validateOnlySupportedNameForms(GeneralSubtrees& trees)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateOnlySupportedNameForms(GeneralSubtrees&)");

   for (int i = 0; i < trees.get_child_count(); i++)
   {
      validateOnlySupportedNameForms( trees[i]->base );
   }
}


//------------------------------------------------------------
// method: validateOnlySupportedNameForms 
//------------------------------------------------------------

void
pkix_validator::validateOnlySupportedNameForms(GeneralName& name)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateOnlySupportedNameForms(GeneralName&)");

   int type = name.selected();

   if ( !(type == TYPE_rfc822Name                || 
          type == TYPE_dNSName                   ||
          type == TYPE_directoryname             ||
          type == TYPE_uniformResourceIdentifier) )
   {
      //------------------------------------------------------------
      // Unsupported and unspecified types
      // TYPE_otherName:    constraint semantics unspecified in PKIX
      // TYPE_ediPartyName: constraint semantics unspecified in PKIX
      // TYPE_registeredID: constraint semantics unspecified in PKIX
      // TYPE_iPAddress:    unsupported for AUGUST
      // TYPE_x400Address:  unsupported for AUGUST
      //------------------------------------------------------------
      throw TPException(TP_UNSUPPORTED_NAME_FORM);
   }
}


//------------------------------------------------------------
// method: validateNoMinMaxInSubtrees 
//------------------------------------------------------------

void
pkix_validator::validateNoMinMaxInSubtrees(GeneralSubtrees& trees)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::validateNoMinMaxInSubtrees(GeneralSubtrees&)");

   int status = 0;

   long value;

   int count = trees.get_child_count();
   for (int i = 0; i < count; i++)
   {
      if ((status = trees[i]->minimum.get_value(value)) != 0)
         throw TPASNException(status);
      if ( value != 0 )
         throw TPException(TP_NAME_CONSTRAINTS_MINUMUM_FIELD_MUST_BE_ZERO);
      if ( trees[i]->maximum.is_present() )
         throw TPException(TP_NAME_CONSTRAINTS_MAXIMUM_FIELD_NOT_ALLOWED);
   }
}


//------------------------------------------------------------
// method: enforceNameConstraintsState 
//------------------------------------------------------------

void
pkix_validator::enforceNameConstraintsState(pkix_certificate& subject)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::enforceNameConstraintsState(pkix_certificate&)");

   //------------------------------------------------------------
   // Subject certificate can have subject distinguished name 
   // and/or subject alternative name extension; name constraints 
   // must be applied to both equally;
   // Special legacy check done in case where PKCS-9 EmailAddress 
   // appears in subject name but no subjectAltNames
   // present; in this case, name constraints must be applied!
   //------------------------------------------------------------

   bool legacyCheck = !subject.isSubjectNull()                    && 
                      !subject.isSubjectAlternativeNamePresent()  &&
                       subject.containsEmailAddress();
   
   //------------------------------------------------------------
   // (b) Verify subject name/subjectAltName is consistent with
   //     constrained subtrees state variable:
   //     Rule: No match is considered a failure
   //------------------------------------------------------------

   if ( !subject.isSubjectNull() )
   {
      GeneralName subjectNameForm;
      convertDNToGeneralName(subject.tbsCertificate.subject, subjectNameForm);
      if ( !_permittedNameSet.is_permitted(subjectNameForm) )
         throw TPException(TP_NAME_CONTRAINTS_NO_MATCHING_PERMITTED_SUBTREE);
   }

   if ( subject.isSubjectAlternativeNamePresent() )
   {
      XsubjectAltName& subjectAltName = subject.subjectAltName();
      if ( !_permittedNameSet.is_permitted(subjectAltName) )
         throw TPException(TP_NAME_CONTRAINTS_NO_MATCHING_PERMITTED_SUBTREE);
   }

   if (legacyCheck)
   {
      GeneralName& emailAddress = subject.emailAddress();
      if ( !_permittedNameSet.is_permitted(emailAddress) )
         throw TPException(TP_NAME_CONTRAINTS_NO_MATCHING_PERMITTED_SUBTREE);
   }

   //------------------------------------------------------------
   // (c) Verify subject name/subjectAltName is consistent with
   //     excluded subtrees state variable
   //     Rule: Any match is considered a failure
   //------------------------------------------------------------
   
   if ( !subject.isSubjectNull() )
   {
      GeneralName subjectNameForm;
      convertDNToGeneralName(subject.tbsCertificate.subject, subjectNameForm);
      if ( _excludedNameSet.is_excluded(subjectNameForm) )
         throw TPException(TP_NAME_CONTRAINTS_NAME_FORM_IS_ON_EXCLUDED_SUBTREE);
   }

   if ( subject.isSubjectAlternativeNamePresent() )
   {
      XsubjectAltName& subjectAltName = subject.subjectAltName();
      if ( _excludedNameSet.is_excluded(subjectAltName) )
         throw TPException(TP_NAME_CONTRAINTS_NAME_FORM_IS_ON_EXCLUDED_SUBTREE);
   }
   
   if (legacyCheck)
   {
      GeneralName& emailAddress = subject.emailAddress();
      if ( _excludedNameSet.is_excluded(emailAddress) )
         throw TPException(TP_NAME_CONTRAINTS_NAME_FORM_IS_ON_EXCLUDED_SUBTREE);
   }
}


//------------------------------------------------------------
// method: updateNameConstraintsState 
//------------------------------------------------------------

void
pkix_validator::updateNameConstraintsState(pkix_certificate& subject)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::updateNameConstraintsState(pkix_certificate&)");

   // can only update state if current certificate has name constraints

   if ( !subject.isNameConstraintsPresent() )
   {
      return;
   }

   //------------------------------------------------------------
   // (j) if permittedSubtrees is present in the certificate, 
   //     set the constrained subtrees state variable to the 
   //     intersection of its previous value and the value 
   //     indicated in the extension field
   // (k) if excludedSubtrees is present in the certificate, 
   //     set the excluded subtrees state variable to the 
   //     union of its previous value and the value 
   //     indicated in the extension field
   //------------------------------------------------------------

   XNameConstraints& nameConstraints = subject.nameConstraints();
   if ( nameConstraints.permittedSubtrees.is_present() )
   {
      _permittedNameSet.update( nameConstraints.permittedSubtrees );
   }
   if ( nameConstraints.excludedSubtrees.is_present() )
   {
      _excludedNameSet.update( nameConstraints.excludedSubtrees );
   }  
}


//------------------------------------------------------------
// method: convertDNToGeneralName 
//------------------------------------------------------------

void 
pkix_validator::convertDNToGeneralName(asn_x500name& name, GeneralName& gname)
{
   TPTRACE(jonahtp_trace_info, "void pkix_validator::convertDNToGeneralName(asn_x500name&,GeneralName&)");

   int status = 0;

   // create a second name and change class/tag

   buffer_t copyBuffer;
   if ((status = name.write(copyBuffer)) != 0)
      throw TPASNException(status);

   asn_x500name nameCopy;
   if ((status = nameCopy.read(copyBuffer)) != 0)
      throw TPASNException(status);

   nameCopy.set_tag(4); 
   nameCopy.set_class(CLASS_CONTEXT_SPECIFIC);

   copyBuffer.clear();
   if ((status = nameCopy.write(copyBuffer)) != 0)
      throw TPASNException(status);

   // convert asn_x500name to a GeneralName

   if ((status = gname.read(copyBuffer)) != 0)
      throw TPASNException(status);
}


