/*
 * Copyright (c) 1992, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * John Heidemann of the UCLA Ficus project.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)cryptfs_vnops.c      8.6 (Berkeley) 5/27/95
 *
 * Ancestors:
 *      @(#)lofs_vnops.c        1.2 (Berkeley) 6/18/92
 *      $Id: vnode.c,v 1.1.1.1 1998/11/05 21:06:28 ezk Exp $
 *      ...and...
 *      @(#)cryptfs_vnodeops.c 1.20 92/07/07 UCLA Ficus project
 *
 * $Id: vnode.c,v 1.1.1.1 1998/11/05 21:06:28 ezk Exp $
 */

/*
 * Cryptfs Layer
 *
 * (See mount_cryptfs(8) for more information.)
 *
 * The cryptfs layer duplicates a portion of the file system
 * name space under a new name.  In this respect, it is
 * similar to the loopback file system.  It differs from
 * the loopback fs in two respects:  it is implemented using
 * a stackable layers techniques, and its "cryptfs-node"s stack above
 * all lower-layer vnodes, not just over directory vnodes.
 *
 * The cryptfs layer has two purposes.  First, it serves as a demonstration
 * of layering by proving a layer which does nothing.  (It actually
 * does everything the loopback file system does, which is slightly
 * more than nothing.)  Second, the cryptfs layer can serve as a prototype
 * layer.  Since it provides all necessary layer framework,
 * new file system layers can be created very easily be starting
 * with a cryptfs layer.
 *
 * The remainder of this man page examines the cryptfs layer as a basis
 * for constructing new layers.
 *
 *
 * INSTANTIATING NEW CRYPTFS LAYERS
 *
 * New cryptfs layers are created with mount_cryptfs(8).
 * Mount_cryptfs(8) takes two arguments, the pathname
 * of the lower vfs (target-pn) and the pathname where the cryptfs
 * layer will appear in the namespace (alias-pn).  After
 * the cryptfs layer is put into place, the contents
 * of target-pn subtree will be aliased under alias-pn.
 *
 *
 * OPERATION OF A CRYPTFS LAYER
 *
 * The cryptfs layer is the minimum file system layer,
 * simply bypassing all possible operations to the lower layer
 * for processing there.  The majority of its activity centers
 * on the bypass routine, through which nearly all vnode operations
 * pass.
 *
 * The bypass routine accepts arbitrary vnode operations for
 * handling by the lower layer.  It begins by examing vnode
 * operation arguments and replacing any cryptfs-nodes by their
 * lower-layer equivlants.  It then invokes the operation
 * on the lower layer.  Finally, it replaces the cryptfs-nodes
 * in the arguments and, if a vnode is return by the operation,
 * stacks a cryptfs-node on top of the returned vnode.
 *
 * Although bypass handles most operations, vop_getattr, vop_lock,
 * vop_unlock, vop_inactive, vop_reclaim, and vop_print are not
 * bypassed. Vop_getattr must change the fsid being returned.
 * Vop_lock and vop_unlock must handle any locking for the
 * current vnode as well as pass the lock request down.
 * Vop_inactive and vop_reclaim are not bypassed so that
 * they can handle freeing cryptfs-layer specific data. Vop_print
 * is not bypassed to avoid excessive debugging information.
 * Also, certain vnode operations change the locking state within
 * the operation (create, mknod, remove, link, rename, mkdir, rmdir,
 * and symlink). Ideally these operations should not change the
 * lock state, but should be changed to let the caller of the
 * function unlock them. Otherwise all intermediate vnode layers
 * (such as union, umapfs, etc) must catch these functions to do
 * the necessary locking at their layer.
 *
 *
 * INSTANTIATING VNODE STACKS
 *
 * Mounting associates the cryptfs layer with a lower layer,
 * effect stacking two VFSes.  Vnode stacks are instead
 * created on demand as files are accessed.
 *
 * The initial mount creates a single vnode stack for the
 * root of the new cryptfs layer.  All other vnode stacks
 * are created as a result of vnode operations on
 * this or other cryptfs vnode stacks.
 *
 * New vnode stacks come into existance as a result of
 * an operation which returns a vnode.
 * The bypass routine stacks a cryptfs-node above the new
 * vnode before returning it to the caller.
 *
 * For example, imagine mounting a cryptfs layer with
 * "mount_cryptfs /usr/include /dev/layer/cryptfs".
 * Changing directory to /dev/layer/cryptfs will assign
 * the root cryptfs-node (which was created when the cryptfs layer was mounted).
 * Now consider opening "sys".  A vop_lookup would be
 * done on the root cryptfs-node.  This operation would bypass through
 * to the lower layer which would return a vnode representing
 * the UFS "sys".  Cryptfs_bypass then builds a cryptfs-node
 * aliasing the UFS "sys" and returns this to the caller.
 * Later operations on the cryptfs-node "sys" will repeat this
 * process when constructing other vnode stacks.
 *
 *
 * CREATING OTHER FILE SYSTEM LAYERS
 *
 * One of the easiest ways to construct new file system layers is to make
 * a copy of the cryptfs layer, rename all files and variables, and
 * then begin modifing the copy.  Sed can be used to easily rename
 * all variables.
 *
 * The umap layer is an example of a layer descended from the
 * cryptfs layer.
 *
 *
 * INVOKING OPERATIONS ON LOWER LAYERS
 *
 * There are two techniques to invoke operations on a lower layer
 * when the operation cannot be completely bypassed.  Each method
 * is appropriate in different situations.  In both cases,
 * it is the responsibility of the aliasing layer to make
 * the operation arguments "correct" for the lower layer
 * by mapping an vnode arguments to the lower layer.
 *
 * The first approach is to call the aliasing layer's bypass routine.
 * This method is most suitable when you wish to invoke the operation
 * currently being handled on the lower layer.  It has the advantage
 * that the bypass routine already must do argument mapping.
 * An example of this is cryptfs_getattrs in the cryptfs layer.
 *
 * A second approach is to directly invoke vnode operations on
 * the lower layer with the VOP_OPERATIONNAME interface.
 * The advantage of this method is that it is easy to invoke
 * arbitrary operations on the lower layer.  The disadvantage
 * is that vnode arguments must be manualy mapped.
 *
 */

#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/sysctl.h>
#include <sys/vnode.h>
#include <sys/mount.h>
#include <sys/namei.h>
#include <sys/malloc.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/ioccom.h>
#include <sys/dirent.h>

#include <vm/vm.h>
#include <vm/vm_extern.h>
#include <vm/vm_zone.h>
#include <vm/vnode_pager.h>
#include <vm/vm_prot.h>
#include <vm/vm_page.h>
#include <vm/vm_pageout.h>
#include <vm/vm_object.h>
#include <vm/vm_pager.h>

#include <machine/cputypes.h>

#include <cryptfs.h>

#define SAFETY
// #define HAVE_BMAP
#define DO_WLOCK
#define WRITE_SYNC


static int cryptfs_bug_bypass = 0;	/* for debugging: enables bypass
					 * printf'ing */
SYSCTL_INT(_debug, OID_AUTO, cryptfs_bug_bypass, CTLFLAG_RW,
	   &cryptfs_bug_bypass, 0, "");

static int cryptfs_access __P((struct vop_access_args * ap));
static int cryptfs_bwrite __P((struct vop_bwrite_args * ap));
static int cryptfs_getattr __P((struct vop_getattr_args * ap));
static int cryptfs_inactive __P((struct vop_inactive_args * ap));
static int cryptfs_lock __P((struct vop_lock_args * ap));
static int cryptfs_lookup __P((struct vop_lookup_args * ap));
static int cryptfs_print __P((struct vop_print_args * ap));
static int cryptfs_reclaim __P((struct vop_reclaim_args * ap));
static int cryptfs_setattr __P((struct vop_setattr_args * ap));
static int cryptfs_strategy __P((struct vop_strategy_args * ap));
static int cryptfs_unlock __P((struct vop_unlock_args * ap));

/*
 * This is the 10-Apr-92 bypass routine.
 *    This version has been optimized for speed, throwing away some
 * safety checks.  It should still always work, but it's not as
 * robust to programmer errors.
 *    Define SAFETY to include some error checking code.
 *
 * In general, we map all vnodes going down and unmap them on the way back.
 * As an exception to this, vnodes can be marked "unmapped" by setting
 * the Nth bit in operation's vdesc_flags.
 *
 * Also, some BSD vnode operations have the side effect of vrele'ing
 * their arguments.  With stacking, the reference counts are held
 * by the upper node, not the lower one, so we must handle these
 * side-effects here.  This is not of concern in Sun-derived systems
 * since there are no such side-effects.
 *
 * This makes the following assumptions:
 * - only one returned vpp
 * - no INOUT vpp's (Sun's vop_open has one of these)
 * - the vnode operation vector of the first vnode should be used
 *   to determine what implementation of the op should be invoked
 * - all mapped vnodes are of our vnode-type (NEEDSWORK:
 *   problems on rmdir'ing mount points and renaming?)
 */
int
cryptfs_bypass(ap)
     struct vop_generic_args /* {
				struct vnodeop_desc *a_desc;
				<other random data follows, presumably>
				} */ *ap;
{
  register struct vnode **this_vp_p;
  int error;
  struct vnode *old_vps[VDESC_MAX_VPS];
  struct vnode **vps_p[VDESC_MAX_VPS];
  struct vnode ***vppp;
  struct vnodeop_desc *descp = ap->a_desc;
  int reles, i;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d FOR=%s\n",
	      __FUNCTION__,__FILE__,__LINE__,descp->vdesc_name);

  if (cryptfs_bug_bypass)
    fist_dprint(5, "cryptfs_bypass: %s\n", descp->vdesc_name);

#ifdef SAFETY
  /* we require at least one vp */
  if (descp->vdesc_vp_offsets == NULL ||
      descp->vdesc_vp_offsets[0] == VDESC_NO_OFFSET)
    panic("cryptfs_bypass: no vp's in map.");
#endif /* not SAFETY */

  /*
   * Map the vnodes going in.  Later, we'll invoke the operation based on
   * the first mapped vnode's operation vector.
   */
  reles = descp->vdesc_flags;
  for (i = 0; i < VDESC_MAX_VPS; reles >>= 1, i++) {
    if (descp->vdesc_vp_offsets[i] == VDESC_NO_OFFSET)
      break;			/* bail out at end of list */
    vps_p[i] = this_vp_p =
      VOPARG_OFFSETTO(struct vnode **, descp->vdesc_vp_offsets[i], ap);
    /*
     * We're not guaranteed that any but the first vnode
     * are of our type.  Check for and don't map any
     * that aren't.  (We must always map first vp or vclean fails.)
     */
    if (i && (*this_vp_p == NULLVP ||
	      (*this_vp_p)->v_op != cryptfs_vnodeop_p)) {
      old_vps[i] = NULLVP;
    } else {
      old_vps[i] = *this_vp_p;
      *(vps_p[i]) = CRYPTFS_VP_TO_LOWERVP(*this_vp_p);
      /*
       * XXX - Several operations have the side effect of vrele'ing their
       * vp's.  We must account for that.  (This should go away in the
       * future.)
       */
      if (reles & 1)
	VREF(*this_vp_p);
    }
  } /* end of "for (i = 0;" loop */

  /*
   * call the operation on the lower layer with the modified argument
   * structure.
   */
  error = VCALL(*(vps_p[0]), descp->vdesc_offset, ap);

  /*
   * Maintain the illusion of call-by-value by restoring vnodes in the
   * argument structure to their original value.
   */
  reles = descp->vdesc_flags;
  for (i = 0; i < VDESC_MAX_VPS; reles >>= 1, i++) {
    if (descp->vdesc_vp_offsets[i] == VDESC_NO_OFFSET)
      break;			/* bail out at end of list */
    if (old_vps[i]) {
      *(vps_p[i]) = old_vps[i];
      if (reles & 1)
	vrele(*(vps_p[i]));
    }
  }

  /*
   * Map the possible out-going vpp (Assumes that the lower layer always
   * returns a VREF'ed vpp unless it gets an error.)
   */
  if (descp->vdesc_vpp_offset != VDESC_NO_OFFSET &&
      !(descp->vdesc_flags & VDESC_NOMAP_VPP) &&
      !error) {
    /*
     * XXX - even though some ops have vpp returned vp's, several ops
     * actually vrele this before returning.  We must avoid these ops.
     * (This should go away when these ops are regularized.)
     */
    if (descp->vdesc_flags & VDESC_VPP_WILLRELE)
      goto out;
    vppp = VOPARG_OFFSETTO(struct vnode ***, descp->vdesc_vpp_offset, ap);
    if (*vppp)
      error = cryptfs_node_create(old_vps[0]->v_mount, **vppp, *vppp);
  }
out:
  return (error);
}

/*
 * We have to carry on the locking protocol on the cryptfs layer vnodes
 * as we progress through the tree. We also have to enforce read-only
 * if this layer is mounted read-only.
 */
static int
cryptfs_lookup(ap)
     struct vop_lookup_args /* {
			       struct vnode * a_dvp;
			       struct vnode ** a_vpp;
			       struct componentname * a_cnp;
			       } */ *ap;
{
  struct componentname *cnp = ap->a_cnp;
  struct proc *p = cnp->cn_proc;
  int flags = cnp->cn_flags;
  struct vop_lock_args lockargs;
  struct vop_unlock_args unlockargs;
  struct vnode *dvp, *vp;
  int error;
  static int counter;
  CNP_VARS;

  fist_dprint(2, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if ((flags & ISLASTCN) && (ap->a_dvp->v_mount->mnt_flag & MNT_RDONLY) &&
      (cnp->cn_nameiop == DELETE || cnp->cn_nameiop == RENAME))
    return (EROFS);

  // fist_print_cn("LOOKUP0", ap->a_cnp);

  counter++;
  fist_dprint(1, "LK0 %d (dvp=0x%x)\n", counter, ap->a_dvp);
  CNP_BEFORE(ap->a_dvp);
  // fist_print_cn("LOOKUP1", ap->a_cnp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
#if 0
  if ((thiscnp->cn_flags & PARAMASK) != (lowercnp->cn_flags & PARAMASK)) {
    char buf[256];
    strcpy(buf, fist_cn_flags(thiscnp->cn_flags));
    panic("LK: this 0x%x <%s>, lower 0x%x <%s>",
	  thiscnp->cn_flags,
	  buf,
	  lowercnp->cn_flags,
	  fist_cn_flags(lowercnp->cn_flags));
  }
#endif
#ifdef DO_FILENAMES
  thiscnp->cn_flags = lowercnp->cn_flags;
#endif /* DO_FILENAMES */
#if 0
  if (had_flags)
    thiscnp->cn_flags |= had_flags;
  else
    thiscnp->cn_flags &= ~had_flags;
#endif
  // fist_print_cn("LOOKUP2", ap->a_cnp);
  CNP_AFTER(ap->a_dvp);

#ifdef DO_FILENAMES
  if (!error &&
      (thiscnp->cn_flags & MAKEENTRY) &&
      thiscnp->cn_nameiop != CREATE) {
    fist_dprint(1, "LK cache_enter\n");
    cache_enter(ap->a_dvp, *(ap->a_vpp), thiscnp);
  }
#endif /* DO_FILENAMES */

  // fist_print_cn("LOOKUP3", ap->a_cnp);
  fist_dprint(1, "LK1 %d (dvp=0x%x)\n", counter, ap->a_dvp);

  if (error == EJUSTRETURN && (flags & ISLASTCN) &&
      (ap->a_dvp->v_mount->mnt_flag & MNT_RDONLY) &&
      (cnp->cn_nameiop == CREATE || cnp->cn_nameiop == RENAME))
    error = EROFS;
  /*
   * We must do the same locking and unlocking at this layer as
   * is done in the layers below us. We could figure this out
   * based on the error return and the LASTCN, LOCKPARENT, and
   * LOCKLEAF flags. However, it is more expidient to just find
   * out the state of the lower level vnodes and set ours to the
   * same state.
   */
  dvp = ap->a_dvp;
  vp = *ap->a_vpp;
  if (dvp == vp)
    return (error);
  if (!VOP_ISLOCKED(dvp)) {
    unlockargs.a_vp = dvp;
    unlockargs.a_flags = 0;
    unlockargs.a_p = p;
    vop_nounlock(&unlockargs);
  }
  if (vp != NULLVP && VOP_ISLOCKED(vp)) {
    lockargs.a_vp = vp;
    lockargs.a_flags = LK_SHARED;
    lockargs.a_p = p;
    vop_nolock(&lockargs);
  }
  return (error);
}

/*
 * Setattr call. Disallow write attempts if the layer is mounted read-only.
 */
static int
cryptfs_setattr(ap)
     struct vop_setattr_args /* {
				struct vnodeop_desc *a_desc;
				struct vnode *a_vp;
				struct vattr *a_vap;
				struct ucred *a_cred;
				struct proc *a_p;
				} */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vattr *vap = ap->a_vap;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if ((vap->va_flags != VNOVAL || vap->va_uid != (uid_t) VNOVAL ||
       vap->va_gid != (gid_t) VNOVAL || vap->va_atime.tv_sec != VNOVAL ||
       vap->va_mtime.tv_sec != VNOVAL || vap->va_mode != (mode_t) VNOVAL) &&
      (vp->v_mount->mnt_flag & MNT_RDONLY))
    return (EROFS);
  if (vap->va_size != VNOVAL) {
    switch (vp->v_type) {
    case VDIR:
      return (EISDIR);
    case VCHR:
    case VBLK:
    case VSOCK:
    case VFIFO:
      if (vap->va_flags != VNOVAL)
	return (EOPNOTSUPP);
      return (0);
    case VREG:
    case VLNK:
    default:
      /*
       * Disallow write attempts if the filesystem is
       * mounted read-only.
       */
      if (vp->v_mount->mnt_flag & MNT_RDONLY)
	return (EROFS);
    }
  }
  return (cryptfs_bypass((struct vop_generic_args *) ap));
}

/*
 *  We handle getattr only to change the fsid.
 */
static int
cryptfs_getattr(ap)
     struct vop_getattr_args /* {
				struct vnode *a_vp;
				struct vattr *a_vap;
				struct ucred *a_cred;
				struct proc *a_p;
				} */ *ap;
{
  int error;

  fist_dprint(4, "CRYPTFS_GETATTR: vp=0x%x\n", (int) ap->a_vp);

  if ((error = cryptfs_bypass((struct vop_generic_args *) ap))) {
    printf("bypass getattr returned error %d\n", error);
    return (error);
  }
  /* Requires that arguments be restored. */
  ap->a_vap->va_fsid = ap->a_vp->v_mount->mnt_stat.f_fsid.val[0];
  return (0);
}

static int
cryptfs_access(ap)
     struct vop_access_args /* {
			       struct vnode *a_vp;
			       int a_mode;
			       struct ucred *a_cred;
			       struct proc *a_p;
			       } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  mode_t mode = ap->a_mode;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Disallow write attempts on read-only layers;
   * unless the file is a socket, fifo, or a block or
   * character device resident on the file system.
   */
  if (mode & VWRITE) {
    switch (vp->v_type) {
    case VDIR:
    case VLNK:
    case VREG:
      if (vp->v_mount->mnt_flag & MNT_RDONLY)
	return (EROFS);
      break;
    default:
      break;
    }
  }
  return (cryptfs_bypass((struct vop_generic_args *) ap));
}

/*
 * We need to process our own vnode lock and then clear the
 * interlock flag as it applies only to our vnode, not the
 * vnodes below us on the stack.
 */
static int
cryptfs_lock(ap)
     struct vop_lock_args /* {
			     struct vnode *a_vp;
			     int a_flags;
			     struct proc *a_p;
			     } */ *ap;
{

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  vop_nolock(ap);
  if ((ap->a_flags & LK_TYPE_MASK) == LK_DRAIN)
    return (0);
  ap->a_flags &= ~LK_INTERLOCK;
  return (cryptfs_bypass((struct vop_generic_args *) ap));
}

/*
 * We need to process our own vnode unlock and then clear the
 * interlock flag as it applies only to our vnode, not the
 * vnodes below us on the stack.
 */
static int
cryptfs_unlock(ap)
     struct vop_unlock_args /* {
			       struct vnode *a_vp;
			       int a_flags;
			       struct proc *a_p;
			       } */ *ap;
{
  /*   struct vnode *vp = ap->a_vp; */

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  vop_nounlock(ap);
  ap->a_flags &= ~LK_INTERLOCK;
  return (cryptfs_bypass((struct vop_generic_args *) ap));
}

static int
cryptfs_inactive(ap)
     struct vop_inactive_args /* {
				 struct vnode *a_vp;
				 struct proc *a_p;
				 } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct cryptfs_node *xp = VP_TO_CRYPTFS(vp);
  struct vnode *lowervp = xp->cryptfs_lowervp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Do nothing (and _don't_ bypass).
   * Wait to vrele lowervp until reclaim,
   * so that until then our cryptfs_node is in the
   * cache and reusable.
   * We still have to tell the lower layer the vnode
   * is now inactive though.
   *
   * NEEDSWORK: Someday, consider inactive'ing
   * the lowervp and then trying to reactivate it
   * with capabilities (v_id)
   * like they do in the name lookup cache code.
   * That's too much work for now.
   */
  VOP_INACTIVE(lowervp, ap->a_p);
  VOP_UNLOCK(ap->a_vp, 0, ap->a_p);
  return (0);
}

static int
cryptfs_reclaim(ap)
     struct vop_reclaim_args /* {
				struct vnode *a_vp;
				struct proc *a_p;
				} */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct cryptfs_node *xp = VP_TO_CRYPTFS(vp);
  struct vnode *lowervp = xp->cryptfs_lowervp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Note: in vop_reclaim, vp->v_op == dead_vnodeop_p,
   * so we can't call VOPs on ourself.
   */
  /* After this assignment, this node will not be re-used. */
  xp->cryptfs_lowervp = NULLVP;
  LIST_REMOVE(xp, cryptfs_hash);
  FREE(vp->v_data, M_TEMP);
  vp->v_data = NULL;
  vrele(lowervp);
  return (0);
}

static int
cryptfs_print(ap)
     struct vop_print_args /* {
			      struct vnode *a_vp;
			      } */ *ap;
{
  register struct vnode *vp = ap->a_vp;
  printf("\ttag VT_CRYPTFS, vp=%p, lowervp=%p\n", vp, CRYPTFS_VP_TO_LOWERVP(vp));
  return (0);
}

/*
 * XXX - vop_strategy must be hand coded because it has no
 * vnode in its arguments.
 * This goes away with a merged VM/buffer cache.
 */
static int
cryptfs_strategy(ap)
     struct vop_strategy_args /* {
				 struct buf *a_bp;
				 } */ *ap;
{
  struct buf *bp = ap->a_bp;
  int error;
  struct vnode *savedvp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  printf("CRYPTFS_STRATEGY on buffer 0x%x\n", (int) bp);

  savedvp = bp->b_vp;
  bp->b_vp = CRYPTFS_VP_TO_LOWERVP(bp->b_vp);

  error = VOP_STRATEGY(bp->b_vp, bp);

  bp->b_vp = savedvp;

  return (error);
}

/*
 * XXX - like vop_strategy, vop_bwrite must be hand coded because it has no
 * vnode in its arguments.
 * This goes away with a merged VM/buffer cache.
 */
static int
cryptfs_bwrite(ap)
     struct vop_bwrite_args /* {
			       struct buf *a_bp;
			       } */ *ap;
{
  struct buf *bp = ap->a_bp;
  int error;
  struct vnode *savedvp;

  printf("FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  savedvp = bp->b_vp;
  bp->b_vp = CRYPTFS_VP_TO_LOWERVP(bp->b_vp);

  error = VOP_BWRITE(bp);

  bp->b_vp = savedvp;

  return (error);
}

/****************************************************************************/
/* verify that lowervp has an v_object.  create it if needed */
void
crypt_verify_lower_object(vnode_t *vp, char *fxn)
{
  vnode_t *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);

  /* allocate a duplicate vnode pager for lower vp if needed */
  if (lowervp->v_object) {
    return;
  }
  lowervp->v_object = vm_pager_allocate(vp->v_object->type,
					lowervp,
					vp->v_object->size,
					VM_PROT_ALL,
					0LL);
  fist_dprint(2, "VERIFY_OBJECT: fxn=%s size=%d obj=0x%x (bhv %d) lowervp=0x%x vp=0x%x\n",
	      fxn,
	      vp->v_object->size,
	      (int) lowervp->v_object,
	      lowervp->v_object->behavior,
	      (int) lowervp,
	      (int) vp);
}

/*
 * Create a page in the vnode and insert data into it.
 * This is used to synchronize data between the VM and read/write interface.
 * That that was written using write() gets pages containing the same data, so
 * that subsequent mmap() ops get valid data.
 */
void
crypt_fill_page(vnode_t *vp, char *buf, long long offset)
{
  vm_page_t pp;
  vm_offset_t kva;
  caddr_t ca;

  fist_dprint(4, "FILL_PAGE: vp=0x%x, buf=0x%x, offset=0x%x\n",
	      (int) vp, (int) buf, buf[0], buf[1], buf[2], (int) offset);
  fist_dprint(1, "FILL_PAGE: vp=0x%x, buf=0x%x [%d,%d,%d], offset=0x%x\n",
	      (int) vp, (int) buf, buf[0], buf[1], buf[2], (int) offset);

  pp = vm_page_grab(vp->v_object, OFF_TO_IDX(offset),
		    VM_ALLOC_NORMAL | VM_ALLOC_RETRY);
  if (!pp) {
    printf("vm_page_grab returned NULL for offset 0x%x!\n", (int) offset);
    return;
  }
  kva = vm_pager_map_page(pp);
  ca = (caddr_t) kva;
  bcopy(buf, ca, PAGE_SIZE);
  vm_pager_unmap_page(kva);
  vm_page_set_validclean(pp, 0, PAGE_SIZE);
  pp->flags &= ~PG_BUSY;
}

void
crypt_fill_lowerpage(vnode_t *lowervp, char *buf, long long offset)
{
  vm_page_t pp;
  vm_offset_t kva;
  caddr_t ca;

  fist_dprint(4, "FILL_LOWERPAGE: lowervp=0x%x, buf=0x%x, offset=0x%x\n",
	      (int) lowervp, (int) buf, buf[0], buf[1], buf[2], (int) offset);
  fist_dprint(1, "FILL_LOWERPAGE: vp=0x%x, buf=0x%x [%d,%d,%d], offset=0x%x\n",
	      (int) lowervp, (int) buf, buf[0], buf[1], buf[2], (int) offset);

  pp = vm_page_grab(lowervp->v_object, OFF_TO_IDX(offset),
		    VM_ALLOC_NORMAL | VM_ALLOC_RETRY);
  if (!pp) {
    printf("vm_page_grab2 returned NULL for offset 0x%x!\n", (int) offset);
    return;
  }
  kva = vm_pager_map_page(pp);
  ca = (caddr_t) kva;
  bcopy(buf, ca, PAGE_SIZE);
  vm_pager_unmap_page(kva);
  vm_page_set_validclean(pp, 0, PAGE_SIZE);
  pp->flags &= ~PG_BUSY;
}

static int
cryptfs_read(ap)
     struct vop_read_args /* {
			     struct vnode *a_vp;
			     struct uio *a_uio;
			     int  a_ioflag;
			     struct ucred *a_cred;
			     } */ *ap;
{
  /* easy mappings */
  vnode_t *vp = ap->a_vp;
  uio_t *uiop = ap->a_uio;
  int ioflag = ap->a_ioflag;
  cred_t *cr = ap->a_cred;

  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  caddr_t current_base;
  int i, bytes_read;
  int num_pages, resid;
  long long start_loffset, end_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;

  fist_dprint(4, "fist_crypt_read vp %x\n", (int) vp);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_read", uiop);
#endif /* FIST_DEBUG */

  cleartext_start_loffset = uiop->uio_offset;
  cleartext_end_loffset = uiop->uio_offset + uiop->uio_resid;
  start_loffset = cleartext_start_loffset & ~(PAGE_SIZE - 1);
  end_loffset = cleartext_end_loffset & ~(PAGE_SIZE - 1);
  /* if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGE_SIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGE_SHIFT;

  fist_dprint(6,
	      "READ: so=%d eo=%d cs=%d es=%d res=%d np=%d ps=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      resid,
	      num_pages,
	      PAGE_SIZE);

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = PAGE_SIZE;
    temp_iovec[i].iov_base = kmem_zalloc(PAGE_SIZE);
    fist_dprint(6, "READ allocated %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
  }

  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = resid;

  hidden_vp = CRYPTFS_VP_TO_LOWERVP(vp);

  /*
   * pass operation to hidden filesystem, and return status
   */

  error = VOP_READ(hidden_vp, &temp_uio, ioflag, cr);

  if (error) {
    fist_dprint(4, "VOP_READ in read returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }

  current_loffset = start_loffset;
  for (i = 0; i < num_pages; i++) {
    bytes_read = PAGE_SIZE - temp_iovec[i].iov_len;
    if (bytes_read == 0)
      break;

    temp_iovec[i].iov_base -= bytes_read;
    current_base = temp_iovec[i].iov_base;

    /* decode the page/block */
    crypt_decode_block(__FUNCTION__, __LINE__,
		       current_base, bytes_read, vp, cr);

    /*
     * save the original size, for kmem_free.
     * no need for it w/ cryptfs; size is always PAGE_SIZE, hence this line
     * is commented out:
     *		temp_iovec[i].iov_len = uiop->uio_iov[i].iov_len;
     */
    /* treat first and last iovec separately, not all data in them is needed */
    if (current_loffset + bytes_read > cleartext_end_loffset) {
      bytes_read = cleartext_end_loffset - current_loffset;
    }
    if (i == 0) {
      bytes_read -= cleartext_start_loffset - start_loffset;
      current_loffset += cleartext_start_loffset - start_loffset;
      current_base += cleartext_start_loffset - start_loffset;
    }
    if ((error = fist_uiomove(current_base, bytes_read, UIO_READ, uiop)))
      /*
       * XXX: we have to see the exact semantics of returning with an
       * EFAULT from read
       */
      break;
    current_loffset += bytes_read;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "READ freeing %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
    kmem_free(temp_iovec[i].iov_base);
  }
  kmem_free(temp_iovec);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_read (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

static int
cryptfs_write(ap)
     struct vop_read_args /* {
			     struct vnode *a_vp;
			     struct uio *a_uio;
			     int  a_ioflag;
			     struct ucred *a_cred;
			     } */ *ap;
{
  /* easy mappings */
  vnode_t *vp = ap->a_vp;
  uio_t *uiop = ap->a_uio;
  int ioflag = ap->a_ioflag;
  cred_t *cr = ap->a_cred;
  struct proc *p = curproc;	/* XXX */

  int error = EPERM;
  vnode_t *hidden_vp;
  vattr_t va;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  iovec_t *free_iovec;		/* for freeing allocated memory */
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, real_start_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;
  int hidden_ioflag = (ioflag & ~IO_APPEND);

  fist_dprint(4, "fist_crypt_write vp=0x%x ioflag=0x%x offset=0x%x resid=%d iovcnt=%x\n",
	      (int) vp, ioflag, (int) uiop->uio_offset, uiop->uio_resid, uiop->uio_iovcnt);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_write (START)", uiop);
#endif /* FIST_DEBUG */

  hidden_vp = CRYPTFS_VP_TO_LOWERVP(vp);

  /* we don't want anybody to do updates while we write, so lock the vnode */
#ifdef DO_WLOCK
  VREF(hidden_vp);		/* XXX: is this needed? */
  vn_lock(hidden_vp, LK_EXCLUSIVE | LK_RETRY | LK_CANRECURSE, p);
#endif

  /* get the attributes, length is necessary for correct updates */
  if ((error = VOP_GETATTR(hidden_vp, &va, cr, p))) {
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    /* XXX to be checked */
    goto out;
  }

  /* just in case someone tries to pull a fast one */
  if (uiop->uio_resid == 0) {
    error = 0;
    goto out;
  }

  cleartext_start_loffset = uiop->uio_offset;
  cleartext_end_loffset = uiop->uio_offset + uiop->uio_resid;

  if (ioflag & IO_APPEND) {
    fist_dprint(6, "WRITE: turning off append flag\n");
    cleartext_start_loffset += va.va_size;
    cleartext_end_loffset += va.va_size;
  }

  start_loffset = MIN(cleartext_start_loffset, va.va_size) & ~(PAGE_SIZE - 1);
  real_start_loffset = cleartext_start_loffset & ~(PAGE_SIZE - 1);
  first_page_bytes = MIN(cleartext_start_loffset, va.va_size) - start_loffset;
  /* must use this to avoid shifting a quad w/ gcc */
  real_first_page = (int)(real_start_loffset - start_loffset) >> PAGE_SHIFT;
  end_loffset = cleartext_end_loffset & ~(PAGE_SIZE - 1);
  ASSERT(first_page_bytes <= PAGE_SIZE);
  /*
   * if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it
   */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGE_SIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGE_SHIFT;

  if (num_pages == 1)
    first_page_bytes = PAGE_SIZE;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = free_iovec[i].iov_len = PAGE_SIZE;
    /* we need the pages to be zeroed out */
    temp_iovec[i].iov_base = free_iovec[i].iov_base = kmem_zalloc(PAGE_SIZE);
  }

  fist_dprint(6,
	      "WRITE: so=%d eo=%d cso=%d ceo=%d rso=%d res=%d np=%d rfp=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      (int) real_start_loffset,
	      resid,
	      num_pages,
	      real_first_page
	      );

  current_loffset = start_loffset;

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_READ;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = first_page_bytes;
  fist_print_uios("WRITE (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }
  fist_print_uios("WRITE (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGE_SIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGE_SIZE;
  /* decode block read */
  crypt_decode_block(__FUNCTION__, __LINE__,
		     temp_iovec[0].iov_base, bytes_read, vp, cr);

  /*
   * if num_pages == 1, we already read the page... don't clobber it
   * if num_pages > 1, then we must read the last page, and decode it
   * completely, before clobbering it.
   * XXX: if end offset is on page boundary, we don't have to do this.
   */
  if (num_pages > 1) {
    /* read last block XXX check length of file */
    temp_uio.uio_iov = temp_iovec + (num_pages - 1);
    temp_uio.uio_iovcnt = 1;
    temp_uio.uio_offset = end_loffset - PAGE_SIZE;
    temp_uio.uio_segflg = UIO_SYSSPACE;
    temp_uio.uio_rw = UIO_READ;
    temp_uio.uio_procp = uiop->uio_procp;
    temp_uio.uio_resid = PAGE_SIZE;

    fist_print_uios("WRITE (before VOP_READ 2)", &temp_uio);
    error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
    fist_print_uios("WRITE (after VOP_READ 3)", &temp_uio);
    if (error) {
      fist_dprint(4, "VOP_READ returned error - not good\n");
      /* XXX to be checked */
      goto out_free;
    }
    bytes_read = PAGE_SIZE - temp_iovec[num_pages - 1].iov_len;
    temp_iovec[num_pages - 1].iov_base -= bytes_read;
    temp_iovec[num_pages - 1].iov_len = PAGE_SIZE;
    /* decode block read */
    crypt_decode_block(__FUNCTION__, __LINE__,
		       temp_iovec[num_pages-1].iov_base, bytes_read, vp, cr);
  }
  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  for (i = 0; i < num_pages; i++) {
    if (i >= real_first_page) {
      bytes_read = PAGE_SIZE;
      current_base = temp_iovec[i].iov_base;
      if (i == real_first_page) {
#define real_first_page_offset (cleartext_start_loffset - real_start_loffset)
	bytes_read -= real_first_page_offset;
	current_loffset += real_first_page_offset;
	current_base += real_first_page_offset;
#undef real_first_page_offset
      }
      if (current_loffset + bytes_read > cleartext_end_loffset) {
	bytes_read = cleartext_end_loffset - current_loffset;
      }
      if ((error = fist_uiomove(current_base, bytes_read, UIO_WRITE, uiop)))
	break;
    }
    /* update/create VM page with this new/updated data before encoding it */
    crypt_fill_page(vp, temp_iovec[i].iov_base, current_loffset);
    /* encode block before writing */
    crypt_encode_block(__FUNCTION__, __LINE__,
		       temp_iovec[i].iov_base, PAGE_SIZE, vp, cr);
#if 0
    if (0)
      crypt_fill_lowerpage(hidden_vp, temp_iovec[i].iov_base, current_loffset);
    if (vp->v_object)
      vm_pager_deallocate(vp->v_object);
#endif
    current_loffset += bytes_read;
  }
  fist_print_uios("WRITE (after for loop 4)", &temp_uio);

  if (va.va_size < end_loffset)
    if (va.va_size < cleartext_end_loffset)
      resid -= end_loffset - cleartext_end_loffset;
    else
      resid -= end_loffset - va.va_size;

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("WRITE (before write)", &temp_uio);
  /*
   * XXX: Must do IO_SYNC else zfod pages get written arbitrarily into
   * large files.  This slows write performance 5 times!
   */
#ifdef WRITE_SYNC
  error = VOP_WRITE(hidden_vp, &temp_uio, (hidden_ioflag | IO_SYNC), cr);
#else /* not WRITE_SYNC */
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cr);
#endif /* not WRITE_SYNC */
  fist_print_uios("WRITE (after write)", &temp_uio);

  if (temp_uio.uio_offset < cleartext_end_loffset) {
    /* incomplete write: this case is an error and should not happen */
    uiop->uio_offset = temp_uio.uio_offset;
    uiop->uio_resid = cleartext_end_loffset - temp_uio.uio_offset;
  } else {
    /*
     * we may have written more than what was asked of us to preserve the
     * encoding over a whole page
     */
    uiop->uio_offset = cleartext_end_loffset;
    uiop->uio_resid = 0;
  }
  /* if IO_APPEND was used, return offset of 0 to upper level */
  if (ioflag & IO_APPEND) {
    uiop->uio_offset = 0;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    fist_dprint(6, "PRINT_BASE2 %d: 0x%x (len=%d)\n", i,
		free_iovec[i].iov_base,
		free_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base);
  }
  kmem_free(free_iovec);
  kmem_free(temp_iovec);

out:
#ifdef DO_WLOCK
  VOP_UNLOCK(hidden_vp, 0, p);
  vrele(hidden_vp);
#endif

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_write (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

static int
cryptfs_ioctl(ap)
     struct vop_ioctl_args /* {
			      struct vnode *a_vp;
			      u_long  a_command;
			      caddr_t  a_data;
			      int  a_fflag;
			      struct ucred *a_cred;
			      struct proc *a_p;
			      } */ *ap;
{
  int error = EPERM;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);
  int val = 0;

  fist_dprint(4, "CRYPTFS_IOCTL: vp=0x%x, lowervp=0x%x\n", (int) vp, (int) lowervp);

  switch (ap->a_command) {
  case FIST_IOCTL_GET_DEBUG_VALUE:
    val = fist_get_debug_value();
    *ap->a_data = val;
    error = 0;
    goto out;
    break;

  case FIST_IOCTL_SET_DEBUG_VALUE:
    val = (int) (*ap->a_data);
    if (val < 0 || val > 20) {
      error = EINVAL;
      goto out;
    }
    fist_dprint(6, "IOCTL: got arg %d\n", val);
    fist_set_debug_value(val);
    error = 0;
    goto out;
    break;
  }

  /* default to repeating ioctl on lowervp */
  error = VOP_IOCTL(lowervp,
		    ap->a_command,
		    ap->a_data,
		    ap->a_fflag,
		    ap->a_cred,
		    ap->a_p);
out:
  print_location();
  return (error);
}

/****************************************************************************/
/*
 * get page routine
 * This is called only if the pages are NOT in the cache!
 * XXX: ap->a_offset is always ignored
 */
static int
cryptfs_getpages(ap)
     struct vop_getpages_args /* {
				 struct vnode *a_vp;
				 vm_page_t *a_m;
				 int a_count;
				 int a_reqpage;
				 vm_ooffset_t a_offset;
			      } */ *ap;
{
  int error = VM_PAGER_ERROR;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);
  int reqpage = ap->a_reqpage;
  int bytecount = ap->a_count;
  vm_page_t thispp;
  int i, resid;
  int pagecount = round_page(bytecount) / PAGE_SIZE;
  vm_offset_t thiskva;
  caddr_t thisca;
  uio_t temp_uio;
  iovec_t *free_iovecp, *free_iovarr;
  cred_t *cr = curproc->p_ucred;

  fist_dprint(2, "CRYPTFS_GETPAGES: vp=0x%x, lowervp=0x%x, m=0x%x, pages=%d, bytes=%d, reqpage=%d, offset=%d\n", /* XXX: change level to 4 */
	      (int) vp,
	      (int) lowervp,
	      (int) ap->a_m,
	      pagecount,
	      bytecount,
	      reqpage,
	      (int) ap->a_offset
	      );
  crypt_verify_lower_object(vp, __FUNCTION__);

  /* prepare for a VOP_READ on the lowervp data straight into this vp's pages */
  temp_uio.uio_resid = bytecount;
  temp_uio.uio_offset = IDX_TO_OFF(ap->a_m[0]->pindex);
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_READ;
  temp_uio.uio_procp = curproc;
  temp_uio.uio_iovcnt = pagecount;
  temp_uio.uio_iov = free_iovarr = (iovec_t *) kmem_zalloc(pagecount * sizeof(iovec_t));
  free_iovecp = (iovec_t *) kmem_zalloc(pagecount * sizeof(iovec_t));
  if (!temp_uio.uio_iov || !free_iovecp) {
    error = VM_PAGER_AGAIN;
    printf("GetPages: nore more memory for temp uio\n");
    goto out;
  }

  /* setup pointers to each page to read */
  for (i = 0; i < pagecount; i++) {
    temp_uio.uio_iov[i].iov_base = free_iovecp[i].iov_base = kmem_zalloc(PAGE_SIZE);
    if (!free_iovecp[i].iov_base) {
      error = VM_PAGER_AGAIN;
      printf("GetPages: nore more memory for temp iovecs\n");
      goto out;
    }
    temp_uio.uio_iov[i].iov_len = free_iovecp[i].iov_len = PAGE_SIZE;
  }

  /* do the actual VOP_READ */
  error = VOP_READ(lowervp, &temp_uio, (IO_VMIO | IO_SYNC), curproc->p_ucred);
  if (error) {
    printf("GETPAGES: read on behalf of vmio failed with error %d\n", error);
    error = VM_PAGER_ERROR;
    goto out;
  }

  /* if residual is non-zero, do nothing since rest of pages were zalloc'ed */
  resid = temp_uio.uio_resid;
  if (resid > PAGE_SIZE)
    panic("GETPAGES: temp_iovec.uio_resid is %d > PAGE_SIZE\n", resid);

  /* copy and decode all the bytes */
  for (i = 0; i < pagecount; i++) {
    thispp = ap->a_m[i];
    thiskva = vm_pager_map_page(thispp);
    thisca = (caddr_t) thiskva;
    bcopy(free_iovecp[i].iov_base, thisca, PAGE_SIZE);
    crypt_decode_block(__FUNCTION__ ,__LINE__,
		       thisca, PAGE_SIZE, vp, cr);
    vm_pager_unmap_page(thiskva);
  }

  /* set status of pages to valid and non-busy as needed */
  for (i = 0; i < pagecount; i++) {
    thispp = ap->a_m[i];
    thispp->valid = VM_PAGE_BITS_ALL;
    if (i == reqpage)
      thispp->flags |= PG_BUSY;	/* requested page must be busy */
    else
      thispp->flags &= ~PG_BUSY; /* XXX: others need not? */
    fist_print_page("getpages: thispp0", thispp);
  }

  error = VM_PAGER_OK;
out:
  if (free_iovecp) {
    for (i = 0; i < pagecount; i++)
      if (free_iovecp[i].iov_base)
	kmem_free(free_iovecp[i].iov_base);
    kmem_free(free_iovecp);
  }
  if (free_iovarr)
    kmem_free(free_iovarr);
  print_location();
  return (error);
}

/*
 * put page routine
 * XXX: ap->a_offset is always ignored
 */
static int
cryptfs_putpages(ap)
     struct vop_putpages_args /* {
				 struct vnode *a_vp;
				 vm_page_t *a_m;
				 int a_count;
				 int a_sync;
				 int *a_rtvals;
				 vm_ooffset_t a_offset;
				 } */ *ap;
{
  int error;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);
  int bytecount = ap->a_count;
  int pagecount = round_page(bytecount) / PAGE_SIZE;
  vm_page_t thispp;
  int i, resid;
  vm_offset_t thiskva;
  caddr_t thisca;
  uio_t temp_uio;
  iovec_t *free_iovecp, *free_iovarr;
  cred_t *cr = curproc->p_ucred;

  fist_dprint(2, "CRYPTFS_PUTPAGES: vp=0x%x, lowervp=0x%x, sync=0x%x\n",
	      (int) vp, (int) lowervp, ap->a_sync);

  /* prepare for a VOP_WRITE on the lowervp data straight into this vp's pages */
  temp_uio.uio_resid = bytecount;
  temp_uio.uio_offset = IDX_TO_OFF(ap->a_m[0]->pindex);
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_WRITE;
  temp_uio.uio_procp = curproc;
  temp_uio.uio_iovcnt = pagecount;
  temp_uio.uio_iov = free_iovarr = (iovec_t *) kmem_zalloc(pagecount * sizeof(iovec_t));
  free_iovecp = (iovec_t *) kmem_zalloc(pagecount * sizeof(iovec_t));
  if (!temp_uio.uio_iov || !free_iovecp) {
    error = VM_PAGER_AGAIN;
    printf("PutPages: nore more memory for temp uio\n");
    goto out;
  }

  /* setup pointers to each page to write */
  for (i = 0; i < pagecount; i++) {
    temp_uio.uio_iov[i].iov_base = free_iovecp[i].iov_base = kmem_zalloc(PAGE_SIZE);
    if (!free_iovecp[i].iov_base) {
      error = VM_PAGER_AGAIN;
      printf("PutPages: nore more memory for temp iovecs\n");
      goto out;
    }
    temp_uio.uio_iov[i].iov_len = free_iovecp[i].iov_len = PAGE_SIZE;
  }

  /* copy and decode all the bytes */
  for (i = 0; i < pagecount; i++) {
    thispp = ap->a_m[i];
    thiskva = vm_pager_map_page(thispp);
    thisca = (caddr_t) thiskva;
    bcopy(thisca, free_iovecp[i].iov_base, PAGE_SIZE);
    crypt_encode_block(__FUNCTION__, __LINE__,
		       free_iovecp[i].iov_base, PAGE_SIZE, vp, cr);
    vm_pager_unmap_page(thiskva);
  }

  /* do the actual VOP_WRITE */
  error = VOP_WRITE(lowervp, &temp_uio, (IO_VMIO | IO_SYNC), curproc->p_ucred);
  if (error) {
    printf("PUTPAGES: write on behalf of vmio failed with error %d\n", error);
    error = VM_PAGER_ERROR;
    goto out;
  }

  /* if residual is non-zero, do nothing since rest of pages were zalloc'ed */
  resid = temp_uio.uio_resid;
  if (resid > PAGE_SIZE)
    panic("PUTPAGES: temp_iovec.uio_resid is %d > PAGE_SIZE\n", resid);

  /* set status of pages to valid and non-busy as needed */
  for (i = 0; i < pagecount; i++) {
    thispp = ap->a_m[i];
    thispp->valid = VM_PAGE_BITS_ALL;
    thispp->flags &= ~PG_BUSY;
    fist_print_page("putpages: thispp0", thispp);
  }

  error = VM_PAGER_OK;
out:
  if (free_iovecp) {
    for (i = 0; i < pagecount; i++)
      if (free_iovecp[i].iov_base)
	kmem_free(free_iovecp[i].iov_base);
    kmem_free(free_iovecp);
  }
  if (free_iovarr)
    kmem_free(free_iovarr);
  print_location();
  return (error);
}

#ifdef HAVE_BMAP
static int
cryptfs_bmap(ap)
     struct vop_bmap_args /* {
			     struct vnode *a_vp;
			     daddr_t a_bn;
			     struct vnode **a_vpp;
			     daddr_t *a_bnp;
			     int *a_runp;
			     int *a_runb;
			     } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);

  fist_dprint(4, "CRYPT_BMAP:0x%x:0x%x\n", (int) ap->a_vpp, (int) lowervp);
  printf("CRYPT_BMAP:0x%x:0x%x\n", (int) ap->a_vpp, (int) lowervp);

  crypt_verify_lower_object(vp, __FUNCTION__);

#if 0
  ap->a_vp = lowervp;
  error = VCALL(lowervp, VOFFSET(vop_bmap), ap);
  return (error);
#endif

#if 1
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  return (error);
#endif

#if 0
  /* this code also doesn't panic */
  if (ap->a_vpp != NULL)
    *ap->a_vpp = ap->a_vp;
  if (ap->a_bnp != NULL)
    *ap->a_bnp = ap->a_bn;
  if (ap->a_runp != NULL)
    *ap->a_runp = 0;
  if (ap->a_runb != NULL)
    *ap->a_runb = 0;
  return (error);
#endif

#if 0
  error = EINVAL;
  return (error);
#endif
}
#endif /* HAVE_BMAP */

static int
cryptfs_mmap(ap)
     struct vop_mmap_args /* {
			     struct vnode *a_vp;
			     int  a_fflags;
			     struct ucred *a_cred;
			     struct proc *a_p;
			     } */ *ap;
{
  int error;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = CRYPTFS_VP_TO_LOWERVP(vp);

  fist_dprint(4, "CRYPTFS_MMAP: vp=0x%x\n", (int) vp);
  printf("CRYPTFS_MMAP: vp=0x%x\n", (int) vp);

  crypt_verify_lower_object(vp, __FUNCTION__);
  error = VOP_MMAP(lowervp, ap->a_fflags, ap->a_cred, ap->a_p);

  return (error);
}


#ifdef DO_FILENAMES

static int
cryptfs_mkdir(ap)
     struct vop_mkdir_args /* {
			      struct vnode *a_dvp;
			      struct vnode **a_vpp;
			      struct componentname *a_cnp;
			      struct vattr *a_vap;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_MKDIR: lowerdvp=0x%x\n", (int) lowerdvp);

//  fist_print_cn("MKDIR1", ap->a_cnp);
  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
//  fist_print_cn("MKDIR2", thiscnp);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
cryptfs_rmdir(ap)
     struct vop_rmdir_args /* {
			      struct vnode *a_dvp;
			      struct vnode *a_vp;
			      struct componentname *a_cnp;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  vnode_t *lowervp = CRYPTFS_VP_TO_LOWERVP(thisvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_RMDIR: lowerdvp=0x%x lowervp=0x%x\n",
	      (int) lowerdvp, (int) lowervp);

  CNP_BEFORE(thisvp);
//  fist_print_cn("RMDIR0", lowercnp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisvp);

  print_location();
  return (error);
}

static int
cryptfs_create(ap)
     struct vop_create_args /* {
			       struct vnode *a_dvp;
			       struct vnode **a_vpp;
			       struct componentname *a_cnp;
			       struct vattr *a_vap;
			       } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_CREATE: lowerdvp=0x%x\n", (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);
//  fist_print_cn("CREATE1", lowercnp);

  print_location();
  return (error);
}

static int
cryptfs_remove(ap)
     struct vop_remove_args /* {
			       struct vnode *a_dvp;
			       struct vnode *a_vp;
			       struct componentname *a_cnp;
			       } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(ap->a_dvp);
  vnode_t *lowervp = CRYPTFS_VP_TO_LOWERVP(ap->a_vp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_REMOVE: lowerdvp=0x%x lowervp=0x%x\n",
	      (int) lowerdvp, (int) lowervp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  return (error);
}

static int
cryptfs_rename(ap)
     struct vop_rename_args  /* {
				struct vnode *a_fdvp;
				struct vnode *a_fvp;
				struct componentname *a_fcnp;
				struct vnode *a_tdvp;
				struct vnode *a_tvp;
				struct componentname *a_tcnp;
				} */ *ap;
{
  int error;
  vnode_t *thisfdvp = ap->a_fdvp;
  vnode_t *thisfvp = ap->a_fvp;
  vnode_t *thistdvp = ap->a_tdvp;
  vnode_t *thistvp = ap->a_tvp;	/* this one comes in as NULL! */
  vnode_t *lowerfdvp = CRYPTFS_VP_TO_LOWERVP(thisfdvp);
  vnode_t *lowerfvp = CRYPTFS_VP_TO_LOWERVP(thisfvp);
  vnode_t *lowertdvp = CRYPTFS_VP_TO_LOWERVP(thistdvp);
  vnode_t *lowertvp = NULL;
  /* CNP_VARS; */
  cn_t *thisfcnp, *lowerfcnp, *thistcnp, *lowertcnp;

  if (thistvp != NULL)		/* comes in as null to this function */
    lowertvp = CRYPTFS_VP_TO_LOWERVP(thistvp);

  fist_dprint(2, "CRYPTFS_RENAME: thisfdvp=0x%x thisfvp=0x%x thistdvp=0x%x thistvp=0x%x\n",
	      (int) thisfdvp, (int) thisfvp,
	      (int) thistdvp, (int) thistvp);
  fist_dprint(2, "CRYPTFS_RENAME: lowerfdvp=0x%x lowerfvp=0x%x lowertdvp=0x%x lowertvp=0x%x\n",
	      (int) lowerfdvp, (int) lowerfvp,
	      (int) lowertdvp, (int) lowertvp);

  /* CNP_BEFORE(thisvp); */
  thisfcnp = ap->a_fcnp;	/* save original */
  lowerfcnp = cryptfs_new_cnp((thisfdvp), thisfcnp);
  ap->a_fcnp = lowerfcnp;	/* prepare for bypass */
  thistcnp = ap->a_tcnp;	/* save original */
  lowertcnp = cryptfs_new_cnp((thistdvp), thistcnp);
  ap->a_tcnp = lowertcnp;	/* prepare for bypass */

  error = cryptfs_bypass((struct vop_generic_args *) ap);

  /* CNP_AFTER(thisvp); */
  if ((thisfcnp->cn_flags & PARAMASK) != (lowerfcnp->cn_flags & PARAMASK))
    panic("%s: FLAGS CHANGED fthis:0x%x flower:0x%x",
	  __FUNCTION__, thisfcnp->cn_flags, lowerfcnp->cn_flags);
  if ((thistcnp->cn_flags & PARAMASK) != (lowertcnp->cn_flags & PARAMASK))
    panic("%s: FLAGS CHANGED tthis:0x%x tlower:0x%x",
	  __FUNCTION__, thistcnp->cn_flags, lowertcnp->cn_flags);
  cryptfs_update_cnp((thisfdvp), &lowerfcnp, thisfcnp, error);
  ap->a_fcnp = thisfcnp;	/* update for caller */
  cryptfs_update_cnp((thistdvp), &lowertcnp, thistcnp, error);
  ap->a_tcnp = thistcnp;	/* update for caller */

  print_location();
  return (error);
}

static int
cryptfs_link(ap)
     struct vop_link_args /* {
			     struct vnode *a_tdvp;
			     struct vnode *a_vp;
			     struct componentname *a_cnp;
			     } */ *ap;
{
  int error;
  vnode_t *thistdvp = ap->a_tdvp;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowertdvp = NULL;
  vnode_t *lowervp = NULL;
  CNP_VARS;

  /* MUST make sure we only hard link into our own file system! */
  if (thisvp->v_op != cryptfs_vnodeop_p)
    return EXDEV;

  lowertdvp = CRYPTFS_VP_TO_LOWERVP(thistdvp);
  lowervp = CRYPTFS_VP_TO_LOWERVP(thisvp);
  fist_dprint(2, "CRYPTFS_LINK: lowertdvp=0x%x lowervp=0x%x\n",
	      (int) lowertdvp, (int) lowervp);

  CNP_BEFORE(thistdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thistdvp);

  print_location();
  return (error);
}

static int
cryptfs_symlink(ap)
     struct vop_symlink_args /* {
				struct vnode *a_dvp;
				struct vnode **a_vpp;
				struct componentname *a_cnp;
				struct vattr *a_vap;
				char *a_target;
				} */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  char *lower_target, *this_target;
  int lower_target_len = 0;
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_SYMLINK: lowerdvp=0x%x target=\"%s\"\n",
	      (int) lowerdvp, ap->a_target);

  fist_crypt_encodefilename(thisdvp->v_mount,
			    ap->a_target,
			    &lower_target,
			    &lower_target_len,
			    DO_DOTS,
			    FALSE,
			    ap->a_cnp->cn_cred);
  this_target = ap->a_target;	/* save pointer */
  ap->a_target = lower_target;	/* set to encoded name */
  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);
  ap->a_target = this_target;	/* restore pointer */
  kmem_free(lower_target);

  print_location();
  return (error);
}

static int
cryptfs_readlink(ap)
     struct vop_readlink_args /* {
				 struct vnode *a_vp;
				 struct uio *a_uio;
				 struct ucred *a_cred;
				 } */ *ap;
{
  int error;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowervp = CRYPTFS_VP_TO_LOWERVP(thisvp);
  uio_t *uiop = ap->a_uio;
  cred_t *cr = ap->a_cred;
  iovec_t temp_iovec;
  uio_t temp_uio;
  caddr_t temp_addr2free;
  int bytes_read;
  int temp_length, target_real_length;
  char *temp_name;

  fist_dprint(2, "CRYPTFS_READLINK: lowervp=0x%x\n",
	      (int) lowervp);

  if (fist_get_userpass(thisvp->v_mount, cr) == NULL)
    return EACCES;

  temp_iovec.iov_len = PAGE_SIZE;
  temp_iovec.iov_base = temp_addr2free = kmem_zalloc(PAGE_SIZE);
  if (!temp_iovec.iov_base) {
    printf("no more memory in readlink\n");
    error = ENOMEM;
    goto out;
  }
  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = 0;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = uiop->uio_resid;

  error = VOP_READLINK(lowervp, &temp_uio, cr);
  if (error)
    goto out_free;

  bytes_read = PAGE_SIZE - temp_iovec.iov_len;

  if (fist_crypt_decodefilename(thisvp->v_mount,
				temp_iovec.iov_base - bytes_read,
				bytes_read,
				&temp_name,
				&temp_length,
				DO_DOTS,
				cr) != 0) {
    /* a checksum error had occured: skip entry */
    panic("symlink value encrypted with different key that link itself");
  }
  /* must find real string length, which is guaranteed null terminated here */
  target_real_length = strlen(temp_name) + 1;
  fist_dprint(4, "cryptfs_readlink DECODE len=%d, real_len=%d, bytes_read=%d, name=\"%s\"",
	      temp_length, target_real_length, bytes_read, temp_name);

  fist_uiomove(temp_name, target_real_length, UIO_READ, uiop);
  /* already OK: uiop->uio_resid and uiop->uio_loffset */

out_free:
  kmem_free(temp_name);
  kmem_free(temp_addr2free);

out:
  print_location();
  return (error);
}

#if 0
static int
cryptfs_cachedlookup(ap)
     struct vop_cachedlookup_args /* {
				     struct vnode *a_dvp;
				     struct vnode **a_vpp;
				     struct componentname *a_cnp;
				     } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_CACHEDLOOKUP: lowerdvp=0x%x\n",
	      (int) lowerdvp);
  panic("CRYPTFS_CACHEDLOOKUP: lowerdvp=0x%x\n", (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}
#endif

static int
cryptfs_whiteout(ap)
     struct vop_whiteout_args /* {
				 struct vnode *a_dvp;
				 struct componentname *a_cnp;
				 int a_flags;
				 } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_WHITEOUT: lowerdvp=0x%x\n",
	      (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
cryptfs_mknod(ap)
     struct vop_mknod_args /* {
			      struct vnode *a_dvp;
			      struct vnode **a_vpp;
			      struct componentname *a_cnp;
			      struct vattr *a_vap;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_MKNOD: lowerdvp=0x%x\n",
	      (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

/*
 * UFS abort op, called after namei() when a CREATE/DELETE isn't actually
 * done. If a buffer has been saved in anticipation of a CREATE, delete it.
 */
static int
cryptfs_abortop(ap)
     struct vop_abortop_args /* {
				struct vnode *a_dvp;
				struct componentname *a_cnp;
				} */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = CRYPTFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "CRYPTFS_ABORTOP: lowerdvp=0x%x\n",
	      (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = cryptfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
cryptfs_readdir(ap)
     struct vop_readdir_args /* {
				struct vnode *a_vp;
				struct uio *a_uio;
				struct ucred *a_cred;
				int *a_eofflag;
				int *a_ncookies;
				u_long **a_cookies;
				} */ *ap;
{
  int error = EPERM;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowervp = CRYPTFS_VP_TO_LOWERVP(thisvp);
  cred_t *cr = ap->a_cred;
  uio_t *uiop = ap->a_uio;
  uio_t temp_uio;
  iovec_t temp_iovec;
  int aux, bytes_read, length, temp_length, tmp, old_reclen;
  char *temp_name;

  fist_dprint(2, "CRYPTFS_READDIR: lowervp=0x%x ncookies=0x%x\n",
	      (int) lowervp, (int) ap->a_ncookies);

  if (ap->a_ncookies)
    panic("CRYPTFS_READDIR0: *ncookies=%d (fix the code)\n");

  if (fist_get_userpass(thisvp->v_mount, cr) == NULL)
    return EACCES;

  if (uiop->uio_iovcnt != 1)
    panic("CRYPTFS_READDIR: iovecnt not 1 (=%d)", uiop->uio_iovcnt);

  temp_iovec.iov_len = uiop->uio_resid;
  temp_iovec.iov_base = kmem_zalloc(uiop->uio_resid);

  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = uiop->uio_offset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = uiop->uio_resid;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_READDIR(lowervp, &temp_uio, cr,
		      ap->a_eofflag, ap->a_ncookies, ap->a_cookies);

  bytes_read = uiop->uio_resid - temp_uio.uio_resid;
  temp_iovec.iov_base -= bytes_read;
  temp_iovec.iov_len += bytes_read;

  if (error)
    goto clean_up;

#define crt_dirent ((struct dirent *)(temp_iovec.iov_base + aux))
  for (aux = 0; aux < bytes_read; aux += old_reclen) {
    old_reclen = crt_dirent->d_reclen;
    fist_dprint(2, "\nRD: old_reclen=%d old_namlen=%d\n", old_reclen, crt_dirent->d_namlen);
    if (fist_crypt_decodefilename(thisvp->v_mount,
				  crt_dirent->d_name,
#if 0
				  crt_dirent->d_reclen - sizeof(struct dirent) + 2,
#else
				  crt_dirent->d_namlen,
#endif
				  &temp_name, /* null terminated */
				  &temp_length, /* w/ terminating null */
				  SKIP_DOTS,
				  cr) == 0) {
      /*
       * We copy the dirent to userspace only if the csum matched
       */
      strcpy(crt_dirent->d_name, temp_name); /* ok, b/c of terminating null */
      length = temp_length - 256 + sizeof(struct dirent);
      fist_dprint(2, "RD: new length is %d, temp_length %d, struct dirent: %d\n",
		  length, temp_length, sizeof(struct dirent));
      if ((tmp = length & 3))
	length += 4 - tmp;
      crt_dirent->d_reclen = length;
      crt_dirent->d_namlen = temp_length - 1;
      kmem_free(temp_name);
      fist_dprint(2, "RD: adding entry \"%s\" of length %d\n",
		  crt_dirent->d_name, crt_dirent->d_reclen);
      error = fist_uiomove(temp_iovec.iov_base + aux,
			   crt_dirent->d_reclen, UIO_READ, uiop);
      if (error)
	goto clean_up;
    }
  }
  uiop->uio_offset = temp_uio.uio_offset;

clean_up:
  kmem_free(temp_iovec.iov_base);
  print_location();
  return (error);
}
#endif /* DO_FILENAMES */

/*
 * Global vfs data structures
 */
vop_t **cryptfs_vnodeop_p;
static struct vnodeopv_entry_desc cryptfs_vnodeop_entries[] =
{
  {&vop_default_desc,		(vop_t *) cryptfs_bypass},
  {&vop_access_desc,		(vop_t *) cryptfs_access},
  {&vop_bwrite_desc,		(vop_t *) cryptfs_bwrite},
  {&vop_getattr_desc,		(vop_t *) cryptfs_getattr},
  {&vop_inactive_desc,		(vop_t *) cryptfs_inactive},
  {&vop_lock_desc,		(vop_t *) cryptfs_lock},
#ifdef DO_FILENAMES
  {&vop_lookup_desc,		(vop_t *) vfs_cache_lookup}, /* DO_FILENAMES */
#else /* not DO_FILENAMES */
  {&vop_lookup_desc,		(vop_t *) cryptfs_lookup}, /* DO_FILENAMES */
#endif /* not DO_FILENAMES */
  {&vop_print_desc,		(vop_t *) cryptfs_print},
  {&vop_reclaim_desc,		(vop_t *) cryptfs_reclaim},
  {&vop_setattr_desc,		(vop_t *) cryptfs_setattr},
  {&vop_strategy_desc,		(vop_t *) cryptfs_strategy},
  {&vop_unlock_desc,		(vop_t *) cryptfs_unlock},
  /* EZK added these */
  { &vop_read_desc,		(vop_t *) cryptfs_read },
  { &vop_write_desc,		(vop_t *) cryptfs_write },
  { &vop_ioctl_desc,		(vop_t *) cryptfs_ioctl },
  { &vop_getpages_desc,		(vop_t *) cryptfs_getpages },
  { &vop_putpages_desc,		(vop_t *) cryptfs_putpages },
#ifdef HAVE_BMAP
  { &vop_bmap_desc,		(vop_t *) cryptfs_bmap },
#endif /* HAVE_BMAP */
  { &vop_mmap_desc,		(vop_t *) cryptfs_mmap },
#ifdef DO_FILENAMES
  { &vop_mkdir_desc,		(vop_t *) cryptfs_mkdir },
  { &vop_rmdir_desc,		(vop_t *) cryptfs_rmdir },
  { &vop_create_desc,		(vop_t *) cryptfs_create },
  { &vop_remove_desc,		(vop_t *) cryptfs_remove },
  { &vop_rename_desc,		(vop_t *) cryptfs_rename },
  { &vop_link_desc,		(vop_t *) cryptfs_link },
  { &vop_symlink_desc,		(vop_t *) cryptfs_symlink },
  { &vop_readlink_desc,		(vop_t *) cryptfs_readlink },
  { &vop_readdir_desc,		(vop_t *) cryptfs_readdir },
  /* also these are new but need to handle DO_NAMES */
#if 0
  { &vop_cachedlookup_desc,	(vop_t *) cryptfs_cachedlookup },
#else
  { &vop_cachedlookup_desc,	(vop_t *) cryptfs_lookup },
#endif
  { &vop_whiteout_desc,		(vop_t *) cryptfs_whiteout },
  { &vop_mknod_desc,		(vop_t *) cryptfs_mknod },
  { &vop_abortop_desc,		(vop_t *) cryptfs_abortop },
#endif /* DO_FILENAMES */

  {NULL, NULL}
};
static struct vnodeopv_desc cryptfs_vnodeop_opv_desc = {
  &cryptfs_vnodeop_p, cryptfs_vnodeop_entries
};

VNODEOP_SET(cryptfs_vnodeop_opv_desc);
