/* t_tcp.c */ 

#include "t_tcp.h"


#include <string.h> 
#include <stdlib.h>


#include "eaddr.h"
#include "emsg.h"

#include "t_config.h"



/* 
 * In order to use swap_order, define the "LITTLE_ENDIAN" flag or 
 * not as appropriate your platform.
 */
UInt32 swap_order(UInt32 num)
{
	UInt32 returnval;

#ifdef LITTLE_ENDIAN

	#define NUM num
	#define MOST_SIG_BYTE ((Byte)(NUM >> 24))
	#define SECOND_MOST_SIG_BYTE ((Byte)((NUM >> 16) & 0xFF))
	#define THIRD_MOST_SIG_BYTE ((Byte)((NUM >> 8) & 0xFF))
	#define LEAST_SIG_BYTE ((Byte)(NUM & 0xFF))

	#define NEW_MOST_SIG_BYTE ((UInt32)(LEAST_SIG_BYTE << 24))
	#define NEW_SECOND_MOST_SIG_BYTE ((UInt32)(THIRD_MOST_SIG_BYTE << 16))
	#define NEW_THIRD_MOST_SIG_BYTE ((UInt32)(SECOND_MOST_SIG_BYTE << 8))
	#define NEW_LEAST_SIG_BYTE ((UInt32)MOST_SIG_BYTE)

	returnval = (NEW_MOST_SIG_BYTE | NEW_SECOND_MOST_SIG_BYTE |
		NEW_THIRD_MOST_SIG_BYTE | NEW_LEAST_SIG_BYTE);

	#undef NEW_LEAST_SIG_BYTE
	#undef NEW_THIRD_MOST_SIG_BYTE
	#undef NEW_SECOND_MOST_SIG_BYTE
	#undef NEW_MOST_SIG_BYTE

	#undef LEAST_SIG_BYTE
	#undef THIRD_MOST_SIG_BYTE
	#undef SECOND_MOST_SIG_BYTE
	#undef MOST_SIG_BYTE

	#undef NUM

#else

	returnval = num;

#endif

	return returnval;
} /* swap_order */



/* Feel free to choose your own magic number, but if you want
 * your Ecash application to talk to other people's Ecash
 * applications over TCP/IP you have to agree on the same rules 
 * for wrapping up blocks of data for TCP/IP transmission!  
 * DigiCash will be coordinating these conventions.  See our web
 * pages.  For TCP/IP the standard convention will probably be 
 * exactly like the demo convention coded here.  (Using this 
 * magic number and the 3-byte size prefixes.) 
 */

#define TCP_PREFIX_MAGIC_NUMBER 0xa0b99083



int
test_tcp_recv_msg(EC_Msg **msg_ptr_out, struct tcp_conn *conn)
{
  Byte *data_ptr = NULL;
  UInt32 temp_uint32 = 0;

  /* We can transport the contents of the EC_Msg however we want,
	but for compatibility with DigiCash Ecash(tm) clients and
	DigiCash mints (very important!), we will use the DigiCash
	tradition of prefixing the data with a 4-byte magic, 3-byte
	length, and 1-byte padding. */

  /* read the magic number */
  if (tcp_read(conn, (char *)(&temp_uint32), sizeof(temp_uint32)) == -1)
	return -1;

  temp_uint32 = swap_order(temp_uint32); /* to local ordering */

  if (temp_uint32 != TCP_PREFIX_MAGIC_NUMBER) {
	return -1; /* bad magic */
  }	

  /* read the length */
  if (tcp_read(conn, (char *)(&temp_uint32), sizeof(temp_uint32)) == -1)
	return -1;

  temp_uint32 = swap_order(temp_uint32); /* to local ordering */

  temp_uint32 = temp_uint32 >> 8;

  if (temp_uint32 == 0) {
	data_ptr = NULL;

	return 0; /* Um...  A zero length chunk of data? */
  }

  /* allocate space for that much data */
  data_ptr = (Byte *) malloc (sizeof(Byte) * temp_uint32);

  /* read the data */
  if (tcp_read(conn, data_ptr, temp_uint32))
	return -1;

  if ((data_ptr != NULL) && (temp_uint32 != 0)) {
	(*msg_ptr_out) = EC_msg_new(temp_uint32, data_ptr);
  }

  if (data_ptr)
	free(data_ptr);

  return 0;
} /* test_tcp_recv_msg */



int
test_tcp_send_msg(const EC_Msg *msg_lnk, struct tcp_conn *conn)
{
  Byte *data_ptr = NULL;
  UInt32 data_size = 0;
  UInt32 temp_uint32 = 0;

  data_ptr = EC_msg_get_contents(msg_lnk);
  data_size = EC_msg_get_size(msg_lnk);
  
  temp_uint32 = swap_order(TCP_PREFIX_MAGIC_NUMBER); /* to network ordering */

  /* write the magic number */
  if (tcp_rwrite(conn, (char *)(&temp_uint32), sizeof(temp_uint32)))
	return -1;

  if (data_size == 0) 
	return -1;

  temp_uint32 = swap_order((((long)data_size) << 8)); /* to network ordering */

  /* write the size and the pad char */
  if (tcp_rwrite(conn, (char *)(&temp_uint32), sizeof(temp_uint32)))
	return -1;

  /* write the data */
  if (tcp_rwrite(conn, (char *)data_ptr, data_size))
	return -1;

  if (data_ptr)
	free (data_ptr);

  return 0;
} /* test_tcp_send_msg */


Char *test_tcp_addr_to_hostname(const EC_Address *addr_lnk) 
{
  size_t temp_size = 0;
  size_t name_len = 0;
  void *temp_ptr = NULL;
  void *find_sep = NULL;

  char *hostname_str = NULL;

  temp_size = EC_addr_get_size(addr_lnk);

  temp_ptr = EC_addr_get_contents(addr_lnk);

  find_sep = memchr(temp_ptr, ':', temp_size);

  if (find_sep == NULL) {
	name_len = temp_size;
  } else {
	name_len = (char *)find_sep - (char *)temp_ptr;
  }

  hostname_str = (char *)malloc(name_len + 1);

  memcpy(hostname_str, temp_ptr, name_len);

  hostname_str[name_len] = '\0';

  free(temp_ptr);

  return hostname_str;
} /* test_tcp_addr_to_hostname */

UInt32 test_tcp_addr_to_portnum(const EC_Address *addr_lnk)
{
  size_t temp_size = 0;
  int magnitude = 0;
  void *temp_ptr = NULL;
  void *find_sep = NULL;
  void *temp_ptr_2 = NULL;
  
  unsigned short int portnum = 0;

  temp_size = EC_addr_get_size(addr_lnk);

  temp_ptr = EC_addr_get_contents(addr_lnk);

  find_sep = memchr(temp_ptr, ':', temp_size);

  if ((find_sep != NULL) && (((char *)find_sep - (char *)temp_ptr) > 0)) {
	portnum = 0;
	magnitude = 1;
	for (temp_ptr_2 = ((char *)temp_ptr) + temp_size - 1;
	((char *)temp_ptr_2) > ((char *)find_sep);
	((char *)temp_ptr_2)--) {
	portnum = portnum + (magnitude * ((*((char *)temp_ptr_2)) - '0'));
	magnitude = magnitude * 10;
	}
  } else {
	portnum = 5885; /* default */
  }

  free(temp_ptr);

  return portnum;
} /* test_tcp_addr_to_portnum */

EC_Address *test_hostname_to_addr(const Char *hostname, UInt32 port)
{
  Byte *contents_ptr = NULL;
  UInt32 size = 0;
  UInt32 stringlen = 0;
  UInt32 portnum_temp = 0;
  UInt32 temp = 0;
  EC_Address *new_addr_ptr = NULL;

  size = stringlen = strlen(hostname);
  if (size < 1)
	return NULL;

  size++; /* space for the ':' between hostname and portnum */

  /* how much space for portnum ? */
  portnum_temp = port;
  while (portnum_temp > 0) {
	size++;
	portnum_temp = portnum_temp / 10;
  }

  /* okay allocate space */
  contents_ptr = (Byte *) malloc (sizeof(Byte) * size);
  if (contents_ptr == NULL)
	return NULL;

  /* copy hostname */
  memcpy(contents_ptr, hostname, stringlen);
  
  /* insert ':' */
  contents_ptr[stringlen] = ':';

  /* encode portnum */
  portnum_temp = port;
  temp = size - 1;
  while (portnum_temp > 0) {
	contents_ptr[temp] = ((unsigned char)(portnum_temp % 10)) + '0';
	temp--;
	portnum_temp = portnum_temp / 10;
  }

  new_addr_ptr = EC_addr_new(size, contents_ptr);

  return new_addr_ptr;
} /* test_hostname_to_addr */

int test_tcp_addr_cmp(const EC_Address *addr_1_lnk, 
  const EC_Address *addr_2_lnk)
{
  UInt32 addr_1_size = 0;
  UInt32 addr_2_size = 0;
  Byte *addr_1_contents = NULL;
  Byte *addr_2_contents = NULL;
  int retval = 0;

  if (addr_1_lnk == NULL) {
	return (addr_2_lnk != NULL);
  }

  addr_1_size = EC_addr_get_size(addr_1_lnk);
  addr_2_size = EC_addr_get_size(addr_2_lnk);

  if (addr_1_size == addr_2_size) {
	addr_1_contents = EC_addr_get_contents(addr_1_lnk);
	addr_2_contents = EC_addr_get_contents(addr_2_lnk);
	
	retval = memcmp(addr_1_contents, addr_2_contents, addr_1_size);

	free(addr_1_contents);
	free(addr_2_contents);

	return retval;
  }

  return 1;
} /* test_tcp_addr_cmp */

