#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "client.h"
#include "des.h"  /* Just for the pass phrase routines */

/* Make a cash deposit from a wallet */

int lucre_deposit_cash(EC_W_Wallet wallet, UInt32 amount)
{
    EC_M_Msg		msg = NULL;
    EC_M_Payment	payment = NULL;
    EC_Errno		err;
    int			seqno;
    int			ret = -1;

    if (!wallet) return -1;

    /* Create a payment */
    msg = EC_M_new_msg();
    CHECK(new_msg, !msg);
    err = EC_W_create_payment(wallet, amount, wallet->userrec->currency,
				wallet->userrec->username,
				wallet->userrec->bankID,
				"cash deposit", &seqno);
    CHECK(create_payment, err);

    /* Get it from the database */
    payment = EC_W_paydb_get(wallet, seqno);
    CHECK(paydb_get, !payment);

    /* Deposit it as cash */
    ret = lucre_deposit_payment(wallet, msg, &payment, EC_M_DEPFLAGS_CASH);

clean:
    EC_M_free_msg(msg);
    EC_M_free_payment(payment);

    return ret;
}

static int moneychanger_deposit(EC_W_Wallet wallet, EC_M_Msg msg,
    EC_M_Payment *payment, EC_M_Depflags depflags)
{
    int ret = -1;
    EC_Errno err = EC_ERR_NONE;
    UInt32 changeamt;
    UInt32 amt;
    UInt32 *seqlist = NULL;
    UInt32 s,numseqs;
    char amtbuf[80];
    sockfd sfd = NULL;
    UInt32 minpayments = (depflags == EC_M_DEPFLAGS_CASH) ? 8 : 1;

    /* Put this payment into the database */
    err = EC_W_noaccount_recdb_put(wallet, *payment, depflags);
    CHECK(noaccount_recdb_put, err);
    *payment = NULL;

    /* For each payment now in the database, try to clear it */
    err = EC_W_recdb_get_all_seqno(wallet, &seqlist, &numseqs);
    CHECK(recdb_get_all_seqno, err);
    changeamt = 0;
    for(s=0; s<numseqs; ++s) {
	err = EC_W_noaccount_deposit_1(wallet, msg, seqlist[s], minpayments);
	CHECK(noaccount_deposit_1, err);

	sfd = bank_socket(wallet);
	CHECK(bank_socket, !sfd);
	err = sfd->write(sfd, msg);
	CHECK(write, err);
	EC_M_clear_msg(msg);
	err = sfd->read(sfd, msg);
	sfd->close(sfd);
	if (err) continue;

	err = EC_W_noaccount_deposit_2(wallet, msg, seqlist[s], &amt);
	if (err) continue;
	changeamt += amt;

	printf("Deposit #%d: ", seqlist[s]);
	EC_U_curramt2str(wallet->userrec->currency, amt,
	    amtbuf, sizeof(amtbuf));
	printf("%s accepted\n", amtbuf);
    }

    EC_U_curramt2str(wallet->userrec->currency, changeamt,
	amtbuf, sizeof(amtbuf));
    printf("\nTotal: %s accepted\n", amtbuf);
    ret = 0;

clean:
    EC_G_free(seqlist);
    return ret;
}

int lucre_deposit_payment(EC_W_Wallet wallet, EC_M_Msg msg,
    EC_M_Payment *payment, EC_M_Depflags depflags)
{
    int			ret = -1;
    int			seqno;
    int			accepted, amount;
    EC_Errno		err;
    EC_M_Dep_ack	dep_ack = NULL;
    int			i;
    int			numdeps;
    sockfd		sfd = NULL;

    /* If we use a moneychanger instead of a bank, this needs to be handled
	specially. */
    if (wallet->userrec->userID == 0) {
	return moneychanger_deposit(wallet, msg, payment, depflags);
    }

    /* First check for uncompleted payments */
    EC_M_clear_msg(msg);
    err = EC_W_deposit_all_payments_1(wallet, msg, &numdeps);
    CHECK(deposit_all_payments_1, err);
    if (numdeps) {
	sfd = bank_socket(wallet);
	CHECK(bank_socket, !sfd);
	err = sfd->write(sfd, msg);
	CHECK(encode, err);
	EC_M_clear_msg(msg);
	err = sfd->read(sfd, msg);
	sfd->close(sfd);
	CHECK(decode, err);
	err = EC_W_deposit_all_payments_2(wallet, msg, &dep_ack);
	CHECK(deposit_all_payments_2, err);
	if (dep_ack) for(i=0;i<dep_ack->numacks;++i) {
	    printf("Deposit #%d: ", dep_ack->dep_1ack[i]->seqno);
	    if (dep_ack->dep_1ack[i]->result == 3) {
		char amtbuf[80];

		EC_U_curramt2str(wallet->userrec->currency,
		    dep_ack->dep_1ack[i]->amount, amtbuf, sizeof(amtbuf));
		printf("%s accepted\n", amtbuf);
	    } else {
		printf("rejected (reason = %d)\n",
			dep_ack->dep_1ack[i]->amount);
	    }
	}
	EC_M_free_dep_ack(dep_ack);
	dep_ack = NULL;
    }

    /* If this is a cash deposit, we'll need to seal it first */
    if (depflags == EC_M_DEPFLAGS_CASH) {
	EC_W_seal_payment(wallet, *payment);
    }

    /* Now do the payment */
    sfd = bank_socket(wallet);
    CHECK(bank_socket, !sfd);
    err = EC_W_recdb_put(wallet, *payment, depflags, &seqno);
    CHECK(recdb_put, err);
    *payment = NULL;
    err = EC_W_deposit_payment_1(wallet, msg, seqno);
    CHECK(deposit_payment_1, err);
    err = sfd->write(sfd, msg);
    CHECK(encode, err);
    EC_M_clear_msg(msg);
    err = sfd->read(sfd, msg);
    sfd->close(sfd);
    CHECK(decode, err);
    err = EC_W_deposit_payment_2(wallet, msg, seqno, &accepted, &amount);
    CHECK(deposit_payment_2, err);

    printf("Deposit #%d: ", seqno);
    if (accepted) {
	char amtbuf[80];

	EC_U_curramt2str(wallet->userrec->currency, amount,
	    amtbuf, sizeof(amtbuf));
	printf("%s accepted\n", amtbuf);
    } else {
	printf("rejected (reason = %d)\n", amount);
    }
    ret = 0;

clean:
    EC_M_free_dep_ack(dep_ack);
    return ret;
}

int lucre_moneychangers_block_create(EC_W_Wallet wallet, char *mfilename)
{
    int fd = 1;
    EC_M_Msg msg = NULL;
    int opened = 0;
    EC_Errno err;

    msg = EC_M_new_msg();
    CHECK(new_msg, !msg);

    if (mfilename && *mfilename) {
	fd = open(mfilename, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR);
	opened = 1;
    }
    CHECK(open, fd < 0);

    create_moneychangers_block(wallet, msg);
    err = EC_M_ATE_encode(msg, "MONEYCHANGERS BLOCK", NULL, NULL, &fd);

clean:
    if (fd >= 0 && opened) close(fd);
    EC_M_free_msg(msg);
    if (err) return -1;
    return 0;
}

int lucre_moneychangers_block_parse(EC_W_Wallet wallet, char *mfilename)
{
    int fd = -1;
    EC_M_Msg msg = NULL;
    EC_Errno err;

    if (!mfilename || !*mfilename) return -1;

    msg = EC_M_new_msg();
    CHECK(new_msg, !msg);

    fd = open(mfilename, O_RDONLY);
    CHECK(open, fd < 0);

    err = EC_M_ATE_decode(msg, NULL, &fd);
    CHECK(decode, err);

    parse_moneychangers_block(wallet, msg);

clean:
    if (fd >= 0) close(fd);
    EC_M_free_msg(msg);
    if (err) return -1;
    return 0;
}
