#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>
#include "client.h"
#include "ssl.h"
#include "rand.h"

#define MONEYCHANGERS_FNAME "moneychangers"

/* Create a data block containing a bank_mkey, a curr, and the
    moneychangers list */
EC_Errno create_moneychangers_block(EC_W_Wallet wallet, EC_M_Msg msg)
{
    EC_Errno err = EC_ERR_NONE;
    EC_M_Curr curr = NULL;
    EC_M_Bank_mkey bank_mkey = NULL;
    char *mfile = NULL;
    int mfilelen = 0;
    EC_W_Locktype oldlock;
    
    /* Get the CURR record */
    curr = EC_W_curr_lookup(wallet, wallet->userrec->bankID,
			    wallet->userrec->currency, 0);
    if (curr) {
	EC_M_compile_curr(curr, msg);
	EC_M_free_curr(curr);
    }

    /* Get the BANK_MKEY record */
    bank_mkey = EC_W_bankkeys_lookup(wallet, wallet->userrec->bankID, 0);
    if (bank_mkey) {
	EC_M_compile_bank_mkey(bank_mkey, msg);
	EC_M_free_bank_mkey(bank_mkey);
    }

    /* Get the moneychangers file */
    err = EC_W_wallet_templock(wallet, EC_W_LOCK_READ, &oldlock);
    if (!err) {
	FILE *fh;
	char *listfname = EC_W_wallet_mkfname(wallet->name,
			    MONEYCHANGERS_FNAME, "");
	if (listfname) fh = fopen(listfname, "r");
	EC_G_free(listfname);
	if (fh) {
	    char buf[1024];
	    size_t res;
	    char *newmfile;
	    int ok = 0;

	    mfile = EC_G_strdup("Moneychangers: ");
	    if (mfile) mfilelen = strlen(mfile)+1;
	    while(mfile) {
		res = fread(buf, 1, 1024, fh);
		if (res == 0) {
		    ok = 1;
		    break;
		}
		/* Append buf to the mlist */
		newmfile = EC_G_realloc(mfile, mfilelen + res);
		if (!newmfile) break;
		mfile = newmfile;
		memmove(mfile+mfilelen-1, buf, res);
		mfilelen += res;
	    }
	    fclose(fh);
	    EC_M_compile_string(mfile, msg);
	    EC_G_free(mfile);
	}
    }
    EC_W_wallet_lock(wallet, oldlock);
    return EC_ERR_NONE;
}

/* Parse a data block like the one above */
int parse_moneychangers_block(EC_W_Wallet wallet, EC_M_Msg msg)
{
    EC_Errno err;
    int didit = 0;
    EC_W_Locktype oldlock;

    err = EC_W_wallet_templock(wallet, EC_W_LOCK_WRITE, &oldlock);
    while (!err) {
	EC_M_Fieldtype fld;
	EC_M_Rectype rec;

	err = EC_M_examine_msg(&fld, &rec, msg);
	if (err) break;
	if (fld == EC_M_FIELD_NONE) break;
	if (fld == EC_M_FIELD_STRING) {
	    char *instr = NULL;
	    err = EC_M_decompile_string(&instr, msg);
	    if (!err) {
		if (!strncasecmp(instr, "Moneychangers: ", 15)) {
		    /* Write the data */
		    char *listfname = EC_W_wallet_mkfname(wallet->name,
					MONEYCHANGERS_FNAME, "");
		    if (listfname) {
			FILE *fh = fopen(listfname, "w");
			EC_G_free(listfname);
			if (fh) {
			    fwrite(instr+15, strlen(instr+15), 1, fh);
			    didit = 1;
			    fclose(fh);
			}
		    }
		}
	    }
	    EC_G_free(instr);
	} else if (fld == EC_M_FIELD_SOR) {
	    if (rec == EC_M_REC_CURR) {
		EC_M_Curr curr = NULL;
		err = EC_M_decompile_curr(&curr, msg);
		if (!err) err = EC_W_curr_write(wallet,
				    wallet->userrec->bankID, curr);
		EC_M_free_curr(curr);
	    } else if (rec == EC_M_REC_BANK_MKEY) {
		EC_M_Bank_mkey bank_mkey = NULL;
		err = EC_M_decompile_bank_mkey(&bank_mkey, msg);
		if (!err) err = EC_W_bankkeys_write(wallet, bank_mkey);
		EC_M_free_bank_mkey(bank_mkey);
	    } else {
		err = EC_M_transfer_field(msg, NULL);
	    }
	} else {
	    err = EC_M_transfer_field(msg, NULL);
	}
    }
    EC_W_wallet_lock(wallet, oldlock);
    return didit;
}

/* Make a socket connection to a specified addr and port */
int make_socket(char *host, unsigned short port)
{
    struct protoent *proto;
    int tcpproto;
    struct sockaddr_in addr;
    struct hostent *hostent;
    int ev;
    int sock;

    proto = getprotobyname("tcp");
    if (proto) {
	tcpproto = proto->p_proto;
    } else {
	tcpproto = 6;
    }
    sock = socket(AF_INET, SOCK_STREAM, tcpproto);
    if (sock < 0) {
	return -1;
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    /* Is this a number or a name? */
    if (host[0] >= '0' && host[0] <= '9') {
	addr.sin_addr.s_addr = inet_addr(host);
    } else {
	hostent = gethostbyname(host);
	if (!hostent) {
	    close(sock);
	    return -1;
	}
	addr.sin_addr.s_addr = **(unsigned long **)&(hostent->h_addr);
    }

    ev = connect(sock, (struct sockaddr *)&addr, sizeof(addr));
    if (ev < 0) {
	close(sock);
	return -1;
    }
    return sock;
}

int make_listen(unsigned short port)
{
    struct protoent *proto;
    int tcpproto;
    struct sockaddr_in addr;
    int ev;
    int sock;

    proto = getprotobyname("tcp");
    if (proto) {
	tcpproto = proto->p_proto;
    } else {
	tcpproto = 6;
    }
    sock = socket(AF_INET, SOCK_STREAM, tcpproto);
    if (sock < 0) {
	return -1;
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr.s_addr = INADDR_ANY;

    ev = bind(sock, (struct sockaddr *)&addr, sizeof(addr));
    if (ev < 0) {
	close(sock);
	return -1;
    }
    listen(sock, 5);

    return sock;
}

/* Functions for direct TCP connections */
static EC_Errno tcp_write(sockfd sfd, EC_M_Msg msg)
{
    return EC_M_BTE_encode(msg, NULL, sfd->data);
}

static EC_Errno tcp_read(sockfd sfd, EC_M_Msg msg)
{
    return EC_M_BTE_decode(msg, NULL, sfd->data);
}

static void tcp_close(sockfd sfd)
{
    int fd;
    
    if (!sfd) return;

    fd = *(int *)sfd->data;
    close(fd);
    free(sfd->data);
    free(sfd);
}

/* Functions for HTTP connections */
typedef struct {
    int fd;
    char *uri;
} http_data;

typedef struct {
    Byte *data;
    UInt32 len;
} databuf;

static Int32 write_to_buf(Byte *data, UInt32 len, void *state)
{
    databuf *w = state;
    Byte *newdata = realloc(w->data, w->len+len);
    if (!newdata) return 0;
    w->data = newdata;
    memmove(w->data+w->len, data, len);
    w->len += len;

    return len;
}

static EC_Errno http_req(sockfd sfd, int requestP)
{
    char reqheaders[] = { "GET %s%cecash_request HTTP/1.0\r\n\r\n" };
    char normalheaders[] = { "GET %s%cHTTP/1.0\r\n\r\n" };
    char *theheaders;
    char *dheaders;
    http_data *data = sfd->data;
    int outlen;
    char *outd;

    /* Construct the request */
    theheaders = requestP ? reqheaders : normalheaders;
    dheaders = malloc(strlen(theheaders) + strlen(data->uri) + 10);
    if (!dheaders) {
	return EC_ERR_INTERNAL;
    }
    sprintf(dheaders, theheaders, data->uri, requestP ?
			(strchr(data->uri, '?') ? '&' : '?') : ' ');

    /* Output the request */
    outlen = strlen(dheaders);
    outd = dheaders;
    while (outlen) {
	int res = write(data->fd, outd, outlen);
	if (res <= 0) break;
	outd += res;
	outlen -= res;
    }
    free(dheaders);

    return EC_ERR_NONE;
}

static EC_Errno http_write(sockfd sfd, EC_M_Msg msg)
{
    EC_Errno err = EC_ERR_NONE;
    databuf w;
    char headers[] = {  "POST %s HTTP/1.0\r\n"
			"X-Ecash-leserver-Client: %s %s\n"
			"Content-Type: application/ecash\r\n"
			"Content-Length: %d\r\n\r\n" };
    char *dheaders;
    http_data *data = sfd->data;
    int outlen;
    char *outd;

    /* Read the msg into a databuf */
    w.data = NULL;
    w.len = 0;
    err = EC_M_ATE_encode(msg, NULL, NULL, write_to_buf, &w);
    if (err) {
	free(w.data);
	return err;
    }

    /* Construct the HTTP headers */
    dheaders = malloc(strlen(headers) + strlen(data->uri) + strlen(EC_LIB_NAME)
			+ strlen(LUCRE_VERSION_STR) + 10);
    if (!dheaders) {
	free(w.data);
	return EC_ERR_INTERNAL;
    }
    sprintf(dheaders, headers, data->uri, EC_LIB_NAME, LUCRE_VERSION_STR,
		w.len);

    /* Output the headers */
    outlen = strlen(dheaders);
    outd = dheaders;
    while (outlen) {
	int res = write(data->fd, outd, outlen);
	if (res <= 0) break;
	outd += res;
	outlen -= res;
    }
    free(dheaders);

    /* Output the data */
    outlen = w.len;
    outd = w.data;
    while (outlen) {
	int res = write(data->fd, outd, outlen);
	if (res <= 0) break;
	outd += res;
	outlen -= res;
    }
    free(w.data);

    return EC_ERR_NONE;
}

static EC_Errno http_read(sockfd sfd, EC_M_Msg msg)
{
    http_data *data = sfd->data;

    return EC_M_ATE_decode(msg, NULL, &(data->fd));
}

static void http_close(sockfd sfd)
{
    http_data *data;
    
    if (!sfd) return;

    data = sfd->data;
    if (data) {
	close(data->fd);
	free(data->uri);
	free(data);
    }
    free(sfd);
}

/* Functions for HTTP SSL connections */
typedef struct {
    int fd;
    SSL_CTX *ssl_ctx;
    SSL *ssl;
    char *uri;
} http_ssl_data;

static EC_Errno http_ssl_req(sockfd sfd, int requestP)
{
    char reqheaders[] = { "GET %s%cecash_request HTTP/1.0\r\n\r\n" };
    char normalheaders[] = { "GET %s%cHTTP/1.0\r\n\r\n" };
    char *theheaders;
    char *dheaders;
    http_ssl_data *data = sfd->data;

    /* Construct the request */
    theheaders = requestP ? reqheaders : normalheaders;
    dheaders = malloc(strlen(theheaders) + strlen(data->uri) + 10);
    if (!dheaders) {
	return EC_ERR_INTERNAL;
    }
    sprintf(dheaders, theheaders, data->uri, requestP ?
			(strchr(data->uri, '?') ? '&' : '?') : ' ');

    /* Output the request */
    SSL_write(data->ssl, dheaders, strlen(dheaders));
    free(dheaders);

    return EC_ERR_NONE;
}

static EC_Errno http_ssl_write(sockfd sfd, EC_M_Msg msg)
{
    EC_Errno err = EC_ERR_NONE;
    databuf w;
    char headers[] = {  "POST %s HTTP/1.0\r\n"
			"X-Ecash-leserver-Client: %s %s\n"
			"Content-Type: application/ecash\r\n"
			"Content-Length: %d\r\n\r\n" };
    char *dheaders;
    http_ssl_data *data = sfd->data;

    /* Read the msg into a databuf */
    w.data = NULL;
    w.len = 0;
    err = EC_M_ATE_encode(msg, NULL, NULL, write_to_buf, &w);
    if (err) {
	free(w.data);
	return err;
    }

    /* Construct the HTTP headers */
    dheaders = malloc(strlen(headers) + strlen(data->uri) + strlen(EC_LIB_NAME)
			+ strlen(LUCRE_VERSION_STR) + 10);
    if (!dheaders) {
	free(w.data);
	return EC_ERR_INTERNAL;
    }
    sprintf(dheaders, headers, data->uri, EC_LIB_NAME, LUCRE_VERSION_STR,
		w.len);

    /* Output the headers */
    SSL_write(data->ssl, dheaders, strlen(dheaders));
    free(dheaders);

    /* Output the data */
    SSL_write(data->ssl, w.data, w.len);
    free(w.data);
    
    return EC_ERR_NONE;
}

static Int32 read_from_ssl(Byte *data, UInt32 len, void *state)
{
    SSL *ssl = state;

    return SSL_read(ssl, data, len);
}

static EC_Errno http_ssl_read(sockfd sfd, EC_M_Msg msg)
{
    http_ssl_data *data = sfd->data;

    return EC_M_ATE_decode(msg, read_from_ssl, data->ssl);
}

static void http_ssl_close(sockfd sfd)
{
    http_ssl_data *data;
    
    if (!sfd) return;

    data = sfd->data;
    if (data) {
	SSL_free(data->ssl);
	SSL_CTX_free(data->ssl_ctx);
	close(data->fd);
	free(data->uri);
	free(data);
    }
    free(sfd);
}

typedef enum { URL_METHOD_NONE, URL_METHOD_HTTP, URL_METHOD_HTTPS } URLMethod;

typedef struct {
    URLMethod method;
    char *host;
    unsigned short port;
    char *uri;
} URLData;

typedef struct mchngfh_s {
    char *id;
    UInt32 version;
    UInt32 numchangers;
    URLData *changers;
    UInt32 readsofar;
    UInt32 offset;
} * mchngfh;

static int copy_urldata(URLData *to, URLData *from)
{
    char *host = NULL, *uri = NULL;

    if (!to || !from) return 0;

    host = strdup(from->host);
    uri = strdup(from->uri);
    if (!host || !uri) {
	free(host);
	free(uri);
	return 0;
    }
    to->method = from->method;
    to->host = host;
    to->port = from->port;
    to->uri = uri;

    return 1;
}

static void free_in_mchngfh(mchngfh mcl)
{
    int i;

    free(mcl->id);
    mcl->id = NULL;
    mcl->version = 0;
    for (i=0;i<mcl->numchangers;++i) {
	free(mcl->changers[i].host);
	free(mcl->changers[i].uri);
    }
    free(mcl->changers);
    mcl->changers = NULL;
    mcl->numchangers = 0;
    mcl->readsofar = 0;
    mcl->offset = 0;
}

static int parseURL(char *url, URLData *data)
{
    unsigned short defport;
    char *sptr, *ptr;
    char *colon, *slash;
    
    if (!data) return -1;
    data->method = URL_METHOD_NONE;
    data->host = NULL;
    data->port = 0;
    data->uri = NULL;

    sptr = strdup(url);
    ptr = sptr;
    if (!ptr) return 0;

    /* The next has better look like a URL */
    if (!strncmp(ptr, "http://", 7)) {
	data->method = URL_METHOD_HTTP;
	defport = 80;
	ptr += 7;
    } else if (!strncmp(ptr, "https://", 8)) {
	data->method = URL_METHOD_HTTPS;
	defport = 443;
	ptr += 8;
    } else {
	/* Fail */
	free(ptr);
	return 0;
    }

    /* Now look until a : or a /, whichever is first. */
    colon = strchr(ptr, ':');
    slash = strchr(ptr, '/');
    if (!colon && !slash) {
	/* No : or / */
	data->host = strdup(ptr);
	data->port = defport;
	data->uri = strdup("/");
    } else if (!colon) {
	/* /, no : */
	*slash = '\0';
	data->host = strdup(ptr);
	data->port = defport;
	*slash = '/';
	data->uri = strdup(slash);
    } else if (!slash) {
	/* :, no / */
	*colon = '\0';
	data->host = strdup(ptr);
	data->port = atoi(colon+1);
	data->uri = strdup("/");
    } else if (colon < slash) {
	/* : before / */
	*colon = '\0';
	data->host = strdup(ptr);
	*slash = '\0';
	data->port = atoi(colon+1);
	*slash = '/';
	data->uri = strdup(slash);
    } else {
	/* / before : */
	*slash = '\0';
	data->host = strdup(ptr);
	data->port = defport;
	*slash = '/';
	data->uri = strdup(slash);
    }

    if (!data->host || !data->uri) {
	/* Something failed! */
	free(data->host);
	data->host = NULL;
	free(data->uri);
	data->uri = NULL;
	free(sptr);
	return 0;
    }

    free(sptr);
    return 1;
}

static int download_new_moneychanger_list(EC_W_Wallet wallet, char *url)
{
    URLData urldata;
    int res;
    EC_M_Msg msg;
    EC_Errno err = EC_ERR_NONE;
    sockfd sfd = NULL;

    res = parseURL(url, &urldata);
    if (!res) return 0;

    /* Make a connection, depending on whether it is HTTP or HTTPS */
    if (urldata.method == URL_METHOD_HTTPS) {
	http_ssl_data *rdata = NULL;
	int ok = 0;

	rdata = malloc(sizeof(http_ssl_data));
	if (!rdata) goto dsslbail;
	rdata->fd = -1;
	rdata->ssl_ctx = NULL;
	rdata->ssl = NULL;
	rdata->uri = NULL;

	sfd = malloc(sizeof(struct sockfd_s));
	if (!sfd) goto dsslbail;

	sfd->kind = SOCKFD_HTTP_SSL;
	sfd->write = http_ssl_write;
	sfd->read = http_ssl_read;
	sfd->close = http_ssl_close;
	sfd->data = rdata;

	msg = EC_M_new_msg();
	if (!msg) goto dsslbail;
	rdata->ssl_ctx = SSL_CTX_new();
	if (!rdata->ssl_ctx) goto dsslbail;
	rdata->ssl = SSL_new(rdata->ssl_ctx);
	if (!rdata->ssl) goto dsslbail;
	rdata->fd = make_socket(urldata.host, urldata.port);
	if (rdata->fd < 0) goto dsslbail;
	rdata->uri = strdup(urldata.uri);
	if (!rdata->uri) goto dsslbail;
	SSL_set_fd(rdata->ssl, rdata->fd);
	res = SSL_connect(rdata->ssl);
	if (res <= 0) goto dsslbail;
	err = http_ssl_req(sfd, 0);
	if (err) goto dsslbail;
	err = http_ssl_read(sfd, msg);
	if (err) goto dsslbail;
	http_ssl_close(sfd);
	sfd = NULL;
	rdata = NULL;
	ok = 1;

    dsslbail:
	if (!ok) {
	    EC_M_free_msg(msg);
	    msg = NULL;
	    http_ssl_close(sfd);
	    free(urldata.host);
	    free(urldata.uri);
	    return 0;
	}
	free(urldata.host);
	free(urldata.uri);
    } else if (urldata.method == URL_METHOD_HTTP) {
	http_data *rdata = NULL;
	int ok = 0;

	rdata = malloc(sizeof(http_data));
	if (!rdata) goto dhttpbail;
	rdata->fd = -1;
	rdata->uri = NULL;

	sfd = malloc(sizeof(struct sockfd_s));
	if (!sfd) goto dhttpbail;

	sfd->kind = SOCKFD_HTTP;
	sfd->write = http_write;
	sfd->read = http_read;
	sfd->close = http_close;
	sfd->data = rdata;

	msg = EC_M_new_msg();
	if (!msg) goto dhttpbail;
	rdata->fd = make_socket(urldata.host, urldata.port);
	if (rdata->fd < 0) goto dhttpbail;
	rdata->uri = strdup(urldata.uri);
	if (!rdata->uri) goto dhttpbail;
	err = http_req(sfd, 0);
	if (err) goto dhttpbail;
	err = http_read(sfd, msg);
	if (err) goto dhttpbail;
	http_close(sfd);
	sfd = NULL;
	rdata = NULL;
	ok = 1;

    dhttpbail:
	if (!ok) {
	    EC_M_free_msg(msg);
	    msg = NULL;
	    http_close(sfd);
	    free(urldata.host);
	    free(urldata.uri);
	    return 0;
	}
	free(urldata.host);
	free(urldata.uri);
    } else {
	free(urldata.host);
	free(urldata.uri);
	return 0;
    }

    /* Now parse what we got back */
    return parse_moneychangers_block(wallet, msg);
}

static mchngfh moneychanger_list_open(EC_W_Wallet wallet)
{
    mchngfh retval;
    char *listfname;
    FILE *fh;
    char linebuf[300];
    char *ptr, *start, *end;
    EC_W_Locktype oldlock;
    EC_Errno err;

    err = EC_W_wallet_templock(wallet, EC_W_LOCK_READ, &oldlock);
    if (err) {
	free(retval);
	return NULL;
    }

    listfname = EC_W_wallet_mkfname(wallet->name, MONEYCHANGERS_FNAME, "");
    if (!listfname) {
	EC_W_wallet_lock(wallet, oldlock);
	return NULL;
    }

    retval = malloc(sizeof(struct mchngfh_s));
    if (!retval) {
	EC_G_free(listfname);
	EC_W_wallet_lock(wallet, oldlock);
	return NULL;
    }

    fh = fopen(listfname, "r");
    EC_G_free(listfname);
    if (!fh) {
	free(retval);
	EC_W_wallet_lock(wallet, oldlock);
	return NULL;
    }
    retval->id = NULL;
    retval->version = 0;
    retval->numchangers = 0;
    retval->changers = NULL;
    retval->readsofar = 0;
    retval->offset = 0;

    /* Parse the file */
    while (fgets(linebuf, 299, fh)) {
	/* Remove the end of line */
	ptr = strchr(linebuf, '\r');
	if (ptr) *ptr = '\0';
	ptr = strchr(linebuf, '\n');
	if (ptr) *ptr = '\0';

	/* Find the first word */
	start = linebuf;
	while(*start && isspace(*start)) ++start; ptr = start;
	while(*ptr && !isspace(*ptr)) ++ptr; end = ptr;
	while(*ptr && isspace(*ptr)) ++ptr; *end = '\0';

	if (!strcasecmp(start, "Version")) {
	    /* This is the version identifier */
	    char *id = ptr;
	    char *version;
	    while(*ptr && !isspace(*ptr)) ++ptr; end = ptr;
	    while(*ptr && isspace(*ptr)) ++ptr; *end = '\0';
	    free(retval->id);
	    retval->id = strdup(id);
	    if (!retval->id) goto bail;
	    version = ptr;
	    while(*ptr && !isspace(*ptr)) ++ptr; end = ptr;
	    while(*ptr && isspace(*ptr)) ++ptr; *end = '\0';
	    retval->version = atoi(version);
	} else if (!strcasecmp(start, "AutoRefresh")) {
	    /* This indicates how often and where to get a new copy */
	    char *update = ptr;
	    UInt32 updatesecs, offset;
	    time_t now;
	    struct stat st;

	    while(*ptr && !isspace(*ptr)) ++ptr; end = ptr;
	    while(*ptr && isspace(*ptr)) ++ptr; *end = '\0';
	    if (sscanf(update, "%u%n", &updatesecs, &offset) > 0) {
		if (update[offset] && strchr("smhdwSMHDW", update[offset])) {
		    switch(update[offset]) {
                            case 'w': case 'W': updatesecs *= 7;
                            case 'd': case 'D': updatesecs *= 24;
                            case 'h': case 'H': updatesecs *= 60;
                            case 'm': case 'M': updatesecs *= 60;
		    }
		}
	    }

	    /* See if our copy has expired */
	    now = time(NULL);
	    if (!fstat(fileno(fh), &st) && (now > st.st_mtime) &&
		(now - st.st_mtime > updatesecs)) {
		/* We need a new version */
		int res;

		res = download_new_moneychanger_list(wallet, ptr);
		if (res) {
		    free_in_mchngfh(retval);
		    free(retval);
		    fclose(fh);
		    EC_W_wallet_lock(wallet, oldlock);
		    return moneychanger_list_open(wallet);
		}
	    }
	} else if (!strcasecmp(start, "ChangeURL")) {
	    /* Add this one to the list */
	    char *url = ptr;
	    URLData *newchangers = NULL;

	    while(*ptr && !isspace(*ptr)) ++ptr; end = ptr;
	    while(*ptr && isspace(*ptr)) ++ptr; *end = '\0';
	    newchangers = realloc(retval->changers, (retval->numchangers+1) *
				    sizeof(URLData));
	    if (newchangers) {
		int res = parseURL(url, &(newchangers[retval->numchangers]));
		if (res) ++retval->numchangers;
		retval->changers = newchangers;
	    }
	}
    }

    /* Choose a random place to start */
    if (retval->numchangers) {
	RAND_bytes((char *)&retval->offset, sizeof(retval->offset));
	retval->offset %= retval->numchangers;
    }
    goto out;

bail:
    free_in_mchngfh(retval);
    free(retval);
    retval = NULL;

out:
    if (fh) fclose(fh);
    EC_W_wallet_lock(wallet, oldlock);
    return retval;
}

static int moneychanger_list_read(mchngfh mcl, URLData *data)
{
    int res;

    if (!mcl || !data) return 0;

    if (mcl->readsofar == mcl->numchangers) return 0;

    /* Return a copy of the next one */
    res = copy_urldata(data, &(mcl->changers[(mcl->readsofar + mcl->offset) %
	mcl->numchangers]));
    
    if (!res) return 0;

    ++mcl->readsofar;
    return 1;
}

static void moneychanger_list_close(mchngfh mcl)
{
    free_in_mchngfh(mcl);
    free(mcl);
}

/* If we use a moneychanger (as opposed to a bank), figure out which
    one to use. */
static sockfd moneychanger_socket(EC_W_Wallet wallet)
{
    mchngfh mcl = NULL;
    EC_Errno err = EC_ERR_NONE;
    EC_M_Msg msg = NULL;
    EC_M_Payreq payreq = NULL;
    sockfd retval = NULL;
    int changeok = 0;
    int res;
    URLData urldata;

    /* Open the list of moneychangers suitable for this wallet */
    mcl = moneychanger_list_open(wallet);
    if (!mcl) {
	return NULL;
    }

    /* For each one in the list (until we get a good one) */
    while(moneychanger_list_read(mcl, &urldata) > 0) {
	if (urldata.method == URL_METHOD_HTTPS) {
	    http_ssl_data *rdata = NULL;

	    rdata = malloc(sizeof(http_ssl_data));
	    if (!rdata) goto sslbail;
	    rdata->fd = -1;
	    rdata->ssl_ctx = NULL;
	    rdata->ssl = NULL;
	    rdata->uri = NULL;

	    /* Create the eventual return value */
	    retval = malloc(sizeof(struct sockfd_s));
	    if (!retval) goto sslbail;

	    retval->kind = SOCKFD_HTTP_SSL;
	    retval->write = http_ssl_write;
	    retval->read = http_ssl_read;
	    retval->close = http_ssl_close;
	    retval->data = rdata;

	    msg = EC_M_new_msg();
	    if (!msg) goto sslbail;
	    rdata->ssl_ctx = SSL_CTX_new();
	    if (!rdata->ssl_ctx) goto sslbail;
	    rdata->ssl = SSL_new(rdata->ssl_ctx);
	    if (!rdata->ssl) goto sslbail;
	    rdata->fd = make_socket(urldata.host, urldata.port);
	    if (rdata->fd < 0) goto sslbail;
	    rdata->uri = strdup(urldata.uri);
	    if (!rdata->uri) goto sslbail;
	    SSL_set_fd(rdata->ssl, rdata->fd);
	    res = SSL_connect(rdata->ssl);
	    if (res <= 0) goto sslbail;
	    err = http_ssl_req(retval, 1);
	    if (err) goto sslbail;
	    err = http_ssl_read(retval, msg);
	    if (err) goto sslbail;
	    http_ssl_close(retval);
	    retval = NULL;
	    rdata = NULL;
	    err = EC_M_decompile_payreq(&payreq, msg);
	    if (err) goto sslbail;

	    /* Is this an OK request? */
	    if (payreq->currency != wallet->userrec->currency
		|| payreq->amount != 0)
		goto sslbail;
	    /* Check that it accepts change */
	    changeok = 0;
	    while(!err) {
		EC_M_Fieldtype fld;
		EC_M_Rectype rec;

		err = EC_M_examine_msg(&fld, &rec, msg);
		if (err) break;
		if (fld == EC_M_FIELD_NONE) break;
		if (fld == EC_M_FIELD_STRING) {
		    char *instr = NULL;
		    err = EC_M_decompile_string(&instr, msg);
		    if (!err) {
			if (!strcasecmp(instr, "Change: 1")) changeok = 1;
			if (!strcasecmp(instr, "Change: 0")) changeok = 0;
		    }
		    EC_G_free(instr);
		} else {
		    err = EC_M_transfer_field(msg, NULL);
		}
	    }
	    if (err || !changeok) goto sslbail;
	    
	    /* Looks good; set up the real sockfd */
	    retval = malloc(sizeof(struct sockfd_s));
	    if (!retval) goto sslbail;

	    rdata = malloc(sizeof(http_ssl_data));
	    if (!rdata) goto sslbail;
	    rdata->fd = -1;
	    rdata->ssl_ctx = NULL;
	    rdata->ssl = NULL;
	    rdata->uri = NULL;

	    retval->kind = SOCKFD_HTTP_SSL;
	    retval->write = http_ssl_write;
	    retval->read = http_ssl_read;
	    retval->close = http_ssl_close;
	    retval->data = rdata;

	    rdata->ssl_ctx = SSL_CTX_new();
	    if (!rdata->ssl_ctx) goto sslbail;
	    rdata->ssl = SSL_new(rdata->ssl_ctx);
	    if (!rdata->ssl) goto sslbail;
	    rdata->fd = make_socket(urldata.host, urldata.port);
	    if (rdata->fd < 0) goto sslbail;
	    rdata->uri = strdup(urldata.uri);
	    if (!rdata->uri) goto sslbail;
	    SSL_set_fd(rdata->ssl, rdata->fd);
	    res = SSL_connect(rdata->ssl);
	    if (res <= 0) goto sslbail;

	    /* All OK */
	    EC_M_free_msg(msg);
	    EC_M_free_payreq(payreq);
	    free(urldata.host);
	    free(urldata.uri);
	    moneychanger_list_close(mcl);
	    return retval;

	sslbail:
	    /* Clean up */
	    EC_M_free_msg(msg);
	    msg = NULL;
	    EC_M_free_payreq(payreq);
	    payreq = NULL;
	    http_ssl_close(retval);
	    free(urldata.host);
	    free(urldata.uri);
	} else if (urldata.method == URL_METHOD_HTTP) {
	    http_data *rdata = NULL;

	    rdata = malloc(sizeof(http_data));
	    if (!rdata) goto sslbail;
	    rdata->fd = -1;
	    rdata->uri = NULL;

	    /* Create the eventual return value */
	    retval = malloc(sizeof(struct sockfd_s));
	    if (!retval) goto sslbail;

	    retval->kind = SOCKFD_HTTP;
	    retval->write = http_write;
	    retval->read = http_read;
	    retval->close = http_close;
	    retval->data = rdata;

	    msg = EC_M_new_msg();
	    if (!msg) goto httpbail;
	    rdata->fd = make_socket(urldata.host, urldata.port);
	    if (rdata->fd < 0) goto httpbail;
	    rdata->uri = strdup(urldata.uri);
	    if (!rdata->uri) goto httpbail;
	    err = http_req(retval, 1);
	    if (err) goto httpbail;
	    err = http_read(retval, msg);
	    if (err) goto httpbail;
	    http_close(retval);
	    retval = NULL;
	    rdata = NULL;
	    err = EC_M_decompile_payreq(&payreq, msg);
	    if (err) goto httpbail;

	    /* Is this an OK request? */
	    if (payreq->currency != wallet->userrec->currency
		|| payreq->amount != 0)
		goto httpbail;
	    /* Check that it accepts change */
	    changeok = 0;
	    while(!err) {
		EC_M_Fieldtype fld;
		EC_M_Rectype rec;

		err = EC_M_examine_msg(&fld, &rec, msg);
		if (err) break;
		if (fld == EC_M_FIELD_NONE) break;
		if (fld == EC_M_FIELD_STRING) {
		    char *instr = NULL;
		    err = EC_M_decompile_string(&instr, msg);
		    if (!err) {
			if (!strcasecmp(instr, "Change: 1")) changeok = 1;
			if (!strcasecmp(instr, "Change: 0")) changeok = 0;
		    }
		    EC_G_free(instr);
		} else {
		    err = EC_M_transfer_field(msg, NULL);
		}
	    }
	    if (err || !changeok) goto httpbail;
	    
	    /* Looks good; set up the real sockfd */
	    retval = malloc(sizeof(struct sockfd_s));
	    if (!retval) goto httpbail;

	    rdata = malloc(sizeof(http_data));
	    if (!rdata) goto httpbail;
	    rdata->fd = -1;
	    rdata->uri = NULL;

	    retval->kind = SOCKFD_HTTP;
	    retval->write = http_write;
	    retval->read = http_read;
	    retval->close = http_close;
	    retval->data = rdata;

	    rdata->fd = make_socket(urldata.host, urldata.port);
	    if (rdata->fd < 0) goto httpbail;
	    rdata->uri = strdup(urldata.uri);
	    if (!rdata->uri) goto httpbail;

	    /* All OK */
	    EC_M_free_msg(msg);
	    EC_M_free_payreq(payreq);
	    free(urldata.host);
	    free(urldata.uri);
	    moneychanger_list_close(mcl);
	    return retval;

	httpbail:
	    /* Clean up */
	    EC_M_free_msg(msg);
	    msg = NULL;
	    EC_M_free_payreq(payreq);
	    payreq = NULL;
	    if (retval) free(retval);
	    if (rdata) {
		if (rdata->fd >= 0) close(rdata->fd);
		if (rdata->uri) free(rdata->uri);
		free(rdata);
	    }
	    free(urldata.host);
	    free(urldata.uri);
	}
    }
    moneychanger_list_close(mcl);
    return NULL;
}

/* Try each of the listed addresses in an attempt to connect to the bank */
sockfd bank_socket(EC_W_Wallet wallet)
{
    EC_M_Bank_mkey      bank_mkey = NULL;
    int                 i;
    int                 fd;
    sockfd		retval = NULL;

    if (!wallet) return NULL;

    /* If we use a moneychanger, not a bank, handle it specially */
    if (wallet->userrec->userID == 0) {
	return moneychanger_socket(wallet);
    }

    /* Get the bank's addresses */
    bank_mkey = EC_W_bankkeys_lookup(wallet, wallet->userrec->bankID, 0);
    if (!bank_mkey) return NULL;

    for (i=0;i<bank_mkey->numaddrs;++i) {
        fd = make_socket(bank_mkey->bankaddr[i], bank_mkey->bankport);
        if (fd >= 0) {
            EC_M_free_bank_mkey(bank_mkey);
	    retval = malloc(sizeof(struct sockfd_s));
	    if (!retval) {
		close(fd);
		return NULL;
	    }
	    /* This is a direct TCP connection */
	    retval->kind = SOCKFD_DIRECT_TCP;
	    retval->write = tcp_write;
	    retval->read = tcp_read;
	    retval->close = tcp_close;
	    retval->data = malloc(sizeof(int));
	    if (retval->data) *(int *)retval->data = fd;
	    else {
		close(fd);
		free(retval);
		return NULL;
	    }
            return retval;
        }
    }

    EC_M_free_bank_mkey(bank_mkey);
    return NULL;
}
