#include <stdio.h>
#include <string.h>
#include "libopgp.h"
#include <rsa.h>
#include <dh.h>
#include <dsa.h>

/*--------------------------------------------------*/
/* do DSA verify function */

static int checkdsa(BIGNUM * r, BIGNUM * u1, DSA * dsakey,
                    UCHAR * hash, int hlen)
{
  BN_CTX *ctx = BN_CTX_new();
  BIGNUM *t1 = BN_new(), *t2 = BN_new(), *t3 = BN_new(), *u2 = NULL;
  int ret;

  u2 = BN_mod_inverse(u1, dsakey->q, ctx);
  BN_bin2bn(hash, hlen, t3);
  BN_mod_mul(t3, t3, u2, dsakey->q, ctx);
  BN_mod_mul(u2, r, u2, dsakey->q, ctx);
  BN_mod_exp(t1, dsakey->g, t3, dsakey->p, ctx);
  BN_mod_exp(t2, dsakey->pub_key, u2, dsakey->p, ctx);
  BN_mod_mul(t3, t1, t2, dsakey->p, ctx);
  BN_mod(t3, t3, dsakey->q, ctx);
  DSA_free(dsakey);
  BN_CTX_free(ctx), BN_free(u2), BN_free(t1), BN_free(t2), BN_free(t3);
  ret = BN_ucmp(t3, r);
  return ret;
}

#ifndef NOELGSIG
/*--------------------------------------------------*/
/* do ElG verify function */

static int checkelg(BIGNUM * a, BIGNUM * b, DH * dhkey,
                    UCHAR * hash, int hlen)
{
  BN_CTX *ctx = BN_CTX_new();
  BIGNUM *t = BN_new(), *u = BN_new();
  UCHAR hbuf[1024];
  unsigned int k;

  BN_sub(t, dhkey->p, BN_value_one());
  BN_gcd(u, a, t, ctx);
  BN_set_word(t, 2);
  if (BN_ucmp(u, t) > 0) {
    BN_print_fp(stderr, u);
    fprintf(stderr, " = gcd(p-1,r)\n");
    fprintf(stderr, "WARNING: Weak El Gamal Signature\n");
    exit(-1);
  }
  memset(hbuf, 0xff, 1024);
  hbuf[0] = 0, hbuf[1] = 1;
  k = BN_num_bytes(dhkey->p);
  hbuf[k - hlen - 1] = 0;
  memcpy(&hbuf[k - hlen], hash, hlen);
  BN_bin2bn(hbuf, k, u);
  BN_mod_exp(t, dhkey->g, u, dhkey->p, ctx);
  BN_mod_exp(u, a, b, dhkey->p, ctx);
  BN_mod_exp(b, dhkey->pub_key, a, dhkey->p, ctx);
  BN_mod_mul(a, b, u, dhkey->p, ctx);
  k = BN_ucmp(t, a);
  BN_CTX_free(ctx), BN_free(u), BN_free(t), DH_free(dhkey);
  return k;
}
#endif

#ifndef NO_RSA
/*--------------------------------------------------*/
/* do RSA verify function */

static int checkrsa(BIGNUM * a, RSA * key, UCHAR * hash, int hlen)
{
  UCHAR dbuf[1024];
  int j, ll;

  j = BN_bn2bin(a, dbuf);
  ll = BN_num_bytes(key->n);
  while (j < ll)
    memmove(&dbuf[1], dbuf, j++), dbuf[0] = 0;
  j = RSA_public_decrypt(ll, dbuf, dbuf, key, RSA_PKCS1_PADDING);
  RSA_free(key);
  return (j != hlen || memcmp(dbuf, hash, j));
}
#endif

/*--------------------------------------------------*/
int PGP_sigck(UCHAR * sigp, void *hctx)
{
  UCHAR hbuf[100], *bp;
  keyid_t keyid = 0;
  unsigned int j, sgalg = 17, halg = 2;
  int i, k = 0;
  BIGNUM *r, *u1 = NULL;

  j = *sigp++;
  if (j == 3) {
    k = *sigp++;                /* extramd length - fixed on old ver */
    PGP_hblk(hctx, sigp, k);
    sigp += k;
    for (j = 0; j < 8; j++)
      keyid = (keyid << 8) + *sigp++;
    sgalg = *sigp++;
    halg = *sigp++;             /* hash algorithm */
  } else if (j == 4) {
    bp = sigp;
    sigp++;                     /* type */
    sgalg = *sigp++;
    halg = *sigp++;
    for (i = 0; i < 2; i++) {
      j = *sigp++ * 256, j += *sigp++;
      if (!i) {
        j += 6;
        PGP_hblk(hctx, --bp, j);
        bp = hbuf;
        *bp++ = 4, *bp++ = 0xff;
        *bp++ = j >> 24, *bp++ = j >> 16, *bp++ = j >> 8, *bp++ = j;
        PGP_hblk(hctx, hbuf, 6);
        j -= 6;
      }
      while (j > 0) {           /* find keyid */
        /* NOTE: I take only the first, but also check unhashed */
        if ((sigp[1] & 0x7f) == 16 && !keyid)
          for (k = 2; k < 10; k++)
            keyid = (keyid << 8) + sigp[k];
        j -= *sigp + 1;
        sigp += *sigp + 1;
      }
      if (j)
        return -2;
    }
  } else
    return -2;

  PGP_hfin(hbuf, hctx);

  if (*sigp++ != hbuf[0] || *sigp++ != hbuf[1])
    return -1;
  if (sgalg != 17 && sgalg != 16 && sgalg != 20 && sgalg != 1 && sgalg != 3)
    return -3;
  r = PGP_mpiBN(&sigp);
  if (sgalg != 1 && sgalg != 3)
    u1 = PGP_mpiBN(&sigp);

  i = PGP_hlen[halg];
  if (sgalg != 17) {
    /* prefix the DER stuff */
    k = PGP_hpfxlen[halg];
    memmove(&hbuf[k], hbuf, i);
    memcpy(hbuf, PGP_hpfx[halg], k);
    i += k;
  }
  for (;;) {
    void *sigkey;

    if (0 >= (k = j = PGP_gtkey(&sigkey, NULL, &keyid)))
      break;
    if (sgalg == 17)
      k = checkdsa(r, u1, sigkey, hbuf, i);
#ifndef NO_RSA
    else if (sgalg == 1 || sgalg == 3)
      k = checkrsa(r, sigkey, hbuf, i);
#endif
#ifndef NOELGSIG
    else if (sgalg == 16 || sgalg == 20)
      k = checkelg(r, u1, sigkey, hbuf, i);
#endif
    if (j == sgalg && !k)
      break;
    keyid = NEXT_KEY;
  }
  BN_free(r);
  if (sgalg != 1 && sgalg != 3)
    BN_free(u1);
  return k;
}
