// Base64OutputStream.java

import java.io.*;

/**
 * Do RFCxxxx style conversion of binary to ascii.
 * Based on base64.cpp in Wei Dai's Crypto10 library.
 * @see Base64InputStream
 */
public class Base64OutputStream extends FilterOutputStream {
static final int MAX_LINE_LENGTH = 64;
static final String vec =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
static final byte padding = '=';

/**
 * Underlying output stream
 */
OutputStream os;

/**
 * Output a checksum afterwards, for compatibility with PGP
 */
boolean checksum;

/**
 * Accumulated checksum, if requested
 */
PGPBase64CRC cksum;

/**
 * Internal data buffer
 */
byte inBuf[];

/**
 * Bytes in internal data buffer
 */
int inBufSize;

/**
 * Number of bytes output so far on line
 */
int lineLength;
       
/**
 * @param os underlying output stream
 * @param checksum true to output a PGP style checksum at the end
 */
public Base64OutputStream(OutputStream os, boolean checksum)
{
    super(os);
    this.os = os;
    this.checksum = checksum;
    lineLength = 0;
    inBufSize = 0;
    inBuf = new byte[3];
    if (checksum)
        cksum = new PGPBase64CRC();
}

public void write (int b) throws IOException
{
    inBuf[inBufSize++] = (byte)b;
    if (checksum)
        cksum.CRCByte(b);
    if (inBufSize==3)
        EncodeQuantum();
}

public void write (byte[]b, int off, int len) throws IOException
{
    for (int i=0; i<len; ++i)
        write(b[off+i]);
}

public void write (byte[]b) throws IOException
{
    write(b, 0, b.length);
}

protected void LineBreak() throws IOException
{
    os.write('\n');
    lineLength = 0;
}

protected void EncodeQuantum() throws IOException
{
    int out;
    char c;

    out=((inBuf[0] & 0xFC) >> 2);
    c = vec.charAt(out);
    os.write(c);

    out=((inBuf[0] & 0x03) << 4) | ((inBuf[1]&0xF0) >> 4);
    c = vec.charAt(out);
    os.write(c);

    out=((inBuf[1] & 0x0F) << 2) | ((inBuf[2]&0xC0) >> 6);
    c = (char)(inBufSize > 1 ? vec.charAt(out) : padding);
    os.write(c);

    out=inBuf[2] & 0x3F;
    c = (char)(inBufSize > 2 ? vec.charAt(out) : padding);
    os.write(c);

    inBufSize=0;
    lineLength+=4;

    if (lineLength>=MAX_LINE_LENGTH)
        LineBreak();
}


public void close() throws IOException
{
    if (inBufSize!=0)
    {
        for (int i=inBufSize;i<3;i++)
            inBuf[i]=0;
        EncodeQuantum();
    }

    if (lineLength!=0) // force a line break unless the current line is empty
        LineBreak();
    if (checksum) {
        os.write(padding);
        int cks = cksum.getCRC();
        inBuf[0] = (byte)(cks>>16);
        inBuf[1] = (byte)(cks>>8);
        inBuf[2] = (byte)(cks);
        inBufSize = 3;
        EncodeQuantum();
        LineBreak();
    }
}

} // Base64OutputStream
