// PGPBase64CRC.java

/**
 * Class for calculating the CRC for PGP's Base64 checksum
 * Adapted from PGP, by Philip Zimmermann, source file armor.c.
 */

public class PGPBase64CRC {

/* CRC Routines. */
/*	These CRC functions are derived from code in chapter 19 of the book 
	"C Programmer's Guide to Serial Communications", by Joe Campbell.
	Generalized to any CRC width by Philip Zimmermann.
*/

static final int CRCBITS = 24;
static final int CRCMASK = (1<<CRCBITS)-1;
static final int CRCHIBIT = (1<<(CRCBITS-1));
static final int CRCSHIFTS = CRCBITS - 8;

/*
 * Notes on making a good 24-bit CRC--
 * The primitive irreducible polynomial of degree 23 over GF(2),
 * 040435651 (octal), comes from Appendix C of "Error Correcting Codes,
 * 2nd edition" by Peterson and Weldon, page 490.  This polynomial was
 * chosen for its uniform density of ones and zeros, which has better
 * error detection properties than polynomials with a minimal number of
 * nonzero terms.  Multiplying this primitive degree-23 polynomial by
 * the polynomial x+1 yields the additional property of detecting any
 * odd number of bits in error, which means it adds parity.  This 
 * approach was recommended by Neal Glover.
 *
 * To multiply the polynomial 040435651 by x+1, shift it left 1 bit and
 * bitwise add (xor) the unshifted version back in.  Dropping the unused 
 * upper bit (bit 24) produces a CRC-24 generator bitmask of 041446373 
 * octal, or 0x864cfb hex.  
 *
 * You can detect spurious leading zeros or framing errors in the 
 * message by initializing the CRC accumulator to some agreed-upon 
 * nonzero "random-like" value, but this is a bit nonstandard.  
 */

static final int PRZCRC=0x864cfb;	 /* PRZ's 24-bit CRC generator polynomial */
static final int CRCINIT=0xB704CE;	 /* Init value for CRC accumulator */

/**
 * Table for speeding up CRC's
 */
int crctable[] = new int[256];

/**
 * Accumulated CRC value
 */
int crc;

public PGPBase64CRC ()
{
    reset();
}

/**
 * Accumulate a buffer's worth of bytes into a CRC accumulator.
 * @param buf buffer data
 * @param off offset into buffer
 * @param len number of bytes to use
 */
public void CRCBytes(byte buf[], int off, int len)
{
    while (len-- > 0) {
	crc = (crc<<8) ^ crctable[(crc>>CRCSHIFTS) ^ buf[off++]];
    }
    crc &= CRCMASK;
} /* crcbytes */

/**
 * Accumulate a single byte into a CRC accumulator.
 * @param n data value to accumulate
 */
public void CRCByte(int n)
{
    crc = (crc<<8) ^ crctable[(crc>>CRCSHIFTS) ^ (n & 0xff)];
    crc &= CRCMASK;
} /* crcbytes */

/**
 * Return accumulated CRC
 */
public int getCRC()
{
    return crc;
}

/**
 * Reset to start reading a new CRC
 */
public void reset()
{
    mk_crctbl(PRZCRC);
    crc = CRCINIT;
}


/*
 * mk_crctbl derives a CRC lookup table from the CRC polynomial.
 * The table is used later by the crcbytes function given below.
 * mk_crctbl only needs to be called once at the dawn of time.
 *
 * The theory behind mk_crctbl is that table[i] is initialized
 * with the CRC of i, and this is related to the CRC of i>>1,
 * so the CRC of i>>1 (pointed to by p) can be used to derive
 * the CRC of i (pointed to by q).
 */

public void mk_crctbl(int poly)
{
	int i;
	int t;
	int itwo = 0;	/* 2*i, or 2*i+1 */
	crctable[itwo++] = 0;
	crctable[itwo++] = poly;
	for (i = 1; i < 128; i++) {
		t = crctable[i];
		if ((t & CRCHIBIT) != 0) {
			t <<= 1;
			crctable[itwo++] = t ^ poly;
			crctable[itwo++] = t;
		} else {
			t <<= 1;
			crctable[itwo++] = t;
			crctable[itwo++] = t ^ poly;
		}
	}
}


} // PGPBase64CRC
