import java.io.*;
import java.util.Random;

public class PGPPublicKey {
static final int KEYIDLEN = 8;
    
public int version;
public int timestamp;
public int validity;
public int algorithm;
public Bignum n;
public Bignum e;

public int numids;
public String userids[];


/**
 * Create a PGP public key from an input stream.
 * Assumes have already read the ctb from the stream.
 * That is passed as first arg, remainder of string follows.
 * @param ctb Cipher Type Byte of PGP
 * @param is DataInputStream pointing just past ctb
 */
public PGPPublicKey(int ctb, DataInputStream is) throws IOException
{
    if (PGPCTB.ctbType(ctb) != PGPCTB.PKC)
        throw new IOException("Incorrect PGP public key format");
    int length = PGPCTB.ctbLength(ctb, is);
    initfromstream(is);
}

/**
 * Create a PGP public key from an input stream.
 * Assumes have already read the ctb and length from the stream.
 * These are passed as first args, remainder of string follows.
 * @param ctb Cipher Type Byte of PGP
 * @param length length field from stream, -1 if no length available
 * @param is DataInputStream pointing just past ctb
 */
public PGPPublicKey (int ctb, int length, DataInputStream is) throws IOException
{
    if (PGPCTB.ctbType(ctb) != PGPCTB.PKC)
        throw new IOException("Incorrect PGP public key format");
    initfromstream(is);
}

/**
 * Create a PGP public key from an array which holds the key in PGP format.
 * Assumes we are pointing past ctb, which is passed as first arg.
 * @param ctb Cipher Type Byte of PGP
 * @param b array
 * @param off offset of value after ctb.
 */
public PGPPublicKey(int ctb, byte b[], int off)
{
    int length;
    int lenlen = ctb & 3;
    switch (lenlen) {
        case 0:  length = b[off++]&0xff; break;
        case 1:  length = bytes.BArrToShortBig(b, off) & 0xffff; off += 2; break;
        case 2:  length = bytes.BArrToIntBig(b, off); off += 4; break;
        case 3:  length = -1; break;
    }
    initfromarray (b, off);
}

/**
 * Create a PGP public key from an array which holds the key in PGP format.
 * Assumes we are pointing past ctb and length, which are passed as first args.
 * @param ctb Cipher Type Byte of PGP
 * @param length length field from array, or -1 if no length available.
 * @param b array
 * @param off offset of value after ctb.
 */
public PGPPublicKey(int ctb, int length, byte b[], int off)
{
    initfromarray (b, off);
}


/* Private routine to do the actual initialization from a stream */
void initfromstream(DataInputStream is) throws IOException
{
    version = is.readUnsignedByte();
    timestamp = is.readInt();
    validity = is.readUnsignedShort();
    algorithm = is.readUnsignedByte();
    n = PGPBignumIO.fromStream(is);
    e = PGPBignumIO.fromStream(is);
    readids(is);
}

void readids(DataInputStream is) throws IOException
{
    int ctb;
    int length;
    for ( ; ; ) {
        ctb = is.read();
        if (ctb < 0)
            break;
        length = PGPCTB.ctbLength(ctb, is);
        byte b[] = new byte[length];
        is.read(b);
        if (PGPCTB.ctbType(ctb)==PGPCTB.UID) {
            String newids[] = new String[numids+1];
            if (userids != null)
                System.arraycopy(userids, 0, newids, 0, numids);
            userids = newids;
            userids[numids++] = new String(b, 0);
        }
    }
}

/* Private routine to do actual initialization from an array */
void initfromarray(byte b[], int off)
{
    version = b[off++] & 0xff;
    timestamp = bytes.BArrToIntBig(b, off);
    off += 4;
    validity = bytes.BArrToShortBig(b, off) & 0xffff;
    off += 2;
    algorithm = b[off++] & 0xff;
    int nlen = (bytes.BArrToShortBig(b, off) + 7) / 8;
    off += 2;
    n = new Bignum(b, off, nlen);
    off += nlen;
    int elen = (bytes.BArrToShortBig(b, off) + 7) / 8;
    off += 2;
    e = new Bignum(b, off, elen);
    off += elen;
}
    

public void dump (PrintStream o, String s)
{
    if (s!=null)
        o.println("PGP public key "+s+":");
    else
        o.println("PGP public key:");
    o.println("  version: "+version);
    o.println("  timestamp: 0x"+Integer.toString(timestamp, 16));
    o.println("  validity: "+validity);
    o.println("  algorithm: "+algorithm);
    o.println("  n: "+n.toString(16));
    o.println("  e: "+e.toString(16));
}

public void dumpids (DataOutputStream o) throws IOException
{
    for (int i=0; i<userids.length; ++i) {
        o.writeChars(userids[i]);
        o.writeChars("\n");
    }
}


/* Service routines for encrypt */

protected static int checksum(byte b[])
{
    int cs = 0;
    for (int i=0; i<b.length; ++i)
        cs += b[i] & 0xff;
    return cs;
}
    
protected static void padkey(byte r[], byte ckey[], Random rnd)
{
    int i = 0;
    r[i++] = 0;
    r[i++] = 2;
    while (i < r.length-4-ckey.length) {
        byte rb;
        while ((rb = (byte)rnd.nextInt()) == 0)
            ;
        r[i++] = rb;
    }
    r[i++] = 0;
    r[i++] = 1;
    int j = 0;
    while (j < ckey.length)
        r[i++] = ckey[j++];
    int csum = checksum(ckey);
    r[i++] = (byte)(csum >> 8);
    r[i++] = (byte)csum;
}


/**
 * PGP RSA encrypt data from input stream, writing to output stream.
 * @param is InputStream to collect data.  Must be such that available()
 * reliably tells how many bytes are in the whole stream.
 * @param os OutputStream to produce results to.  May be a
 * PGPBase64OutputStream to produce ascii encoded data.
 * @param rnd Random object to generate session key and padding.  Best
 * to be a cryptographically strong extension to Random.
 */
public void encrypt(InputStream is, OutputStream os, Random rnd)
        throws IOException
{
    DataOutputStream out = (os instanceof DataOutputStream)?
                (DataOutputStream)os : new DataOutputStream(os);
    byte bkey[] = new byte[Idea.ideaKeySize()];
    for (int i=0; i<bkey.length; ++i)
        bkey[i] = (byte)rnd.nextInt();
    PGPConvKey ckey = new PGPConvKey(bkey);
    int nbytes = (n.countbits() + 7) / 8;
    byte r[] = new byte[nbytes];
    padkey (r, bkey, rnd);
    Bignum m = new Bignum(r);
    Bignum c = Bignum.modexp2(m, e, n);
    int length = 10 + PGPBignumIO.nbytes(c);
    PGPCTB.writectbLength(PGPCTB.PKE, length, out);
    out.write (3);
    byte keyid[] = new byte[nbytes];
    n.getBytes(keyid, 0);
    out.write (keyid, nbytes-KEYIDLEN, KEYIDLEN);
    out.write (1);
    PGPBignumIO.toStream(c, out);
    ckey.encrypt(is, out, rnd);
}


}
