/****************************************************************************
  This file is part of the Freedom Remailer.  It is mostly borrowed from
  Matt Ghio's statistics functions as given in his remailer distribution,
  but significant changes have been made by John B. Fleming and Johannes
  Kroeger.  Changes are
  (C) 1995 John B. Fleming (jfleming@indiana.edu)
  (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/file.h>
#include "freedom.h"

void updatestats (enum stats_flag flag)
{
  char month[24][5];
  struct tm *curtime;
  time_t now;
  pid_t pid = getpid();
  FILE *file;
  int hour, date[24], currenthour, x, lockfd;
  int Tm, Tp, Tl, Tw, Tu, Th, Ts, Tk;
  int Hm[24], Hp[24], Hl[24], Hw[24], Hu[24], Hh[24], Hs[24], Hk[24];

  now = time(NULL);
  curtime = localtime(&now);

  /* don't manipulate the stats file before locking stats.lock exclusively */
  lockfd = open("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  if (!(file = fopen(STATS_FILE, "r"))) {
    /* the stats file does not exist */
    /* make a new empty stats file and zero variables */
    file = fopen(STATS_FILE, "w");
    fprintf(file, "0\n");
    fprintf(file, "0 0 0 0 0 0 0 0\n");
    for (x = 0; x < 24; x++) { /* tmp for month. First day, no messages */
      strcpy(month[x], "tmp");
      date[x] = 0;
      Hm[x] = Hp[x] = Hl[x] = Hw[x] = Hu[x] = Hh[x] = Hs[x] = Hk[x] = 0;
      fprintf(file, "tmp 0 0 0 0 0 0 0 0 0\n");
    }
    fclose(file);
    Tm = Tp = Tl = Tw = Tu = Th = Ts = Tk = 0;
  }
  else {
    fscanf(file, "%d", &hour);
    fscanf(file, "%d %d %d %d %d %d %d %d", &Tm, &Tp, &Tl, &Tw, &Tu,
	   &Th, &Ts, &Tk);
    for (x = 0; x < 24; x++)
      fscanf(file, "%s %d %d %d %d %d %d %d %d %d", month[x], &date[x],
	     &Hm[x], &Hp[x], &Hl[x], &Hw[x], &Hu[x], &Hh[x], &Hs[x], &Hk[x]);
    fclose(file);
  }

  currenthour = curtime->tm_hour;
  x = hour;
  while (x != currenthour) {
    if (x > 0) {
      strcpy(month[x], month[x-1]);
      date[x] = date[x - 1];
    }
    else {
      strftime(month[0], 5, "%b", curtime);
      date[0] = curtime->tm_mday;
    }
    Hm[x] = Hp[x] = Hl[x] = Hw[x] = Hu[x] = Hh[x] = Hs[x] = Hk[x] = 0;
    x++;
    if (x > 23)
      x=0;
  }

  if (hour != currenthour) {
    Hm[hour] = Tm;
    Hp[hour] = Tp;
    Hl[hour] = Tl;
    Hw[hour] = Tw;
    Hu[hour] = Tu;
    Hh[hour] = Th;
    Hs[hour] = Ts;
    Hk[hour] = Tk;
    Tm = Tp = Tl = Tw = Tu = Th = Ts = Tk = 0;
  }

  if (flag == STATS_MESSAGE)
    Tm++;
  else if (flag == STATS_PGP)
    Tp++;
  else if (flag == STATS_LATENT)
    Tl++;
  else if (flag == STATS_WWW)
    Tw++;
  else if (flag == STATS_POST)
    Tu++;
  else if (flag == STATS_HELP)
    Th++;
  else if (flag == STATS_STAT)
    Ts++;
  else if (flag == STATS_KEY)
    Tk++;

  file = fopen(STATS_FILE, "w");
  fprintf(file, "%d\n", currenthour);
  fprintf(file, "%d %d %d %d %d %d %d %d\n", Tm, Tp, Tl, Tw, Tu, Th, Ts, Tk);
  for (x = 0; x < 24; x++)
    fprintf(file, "%s %d %d %d %d %d %d %d %d %d\n", month[x], date[x],
	    Hm[x], Hp[x], Hl[x], Hw[x], Hu[x], Hh[x], Hs[x], Hk[x]);
  fclose(file);
  flock(lockfd, LOCK_UN);
  close(lockfd);
}

void mailstats(const char *to)
{
  char sendmail[BUFSIZ], month[24][5];
  struct tm *curtime;
  time_t now;
  pid_t pid = getpid();
  FILE *file;
  int hour, x, x0, y, date[24], currenthour, lockfd;
  int Tm, Tp, Tl, Tw, Tu, Th, Ts, Tk, mmax = 0;
  int Hm[24], Hp[24], Hl[24], Hw[24], Hu[24], Hh[24], Hs[24], Hk[24];
  float mscale;

  now = time(NULL);
  curtime = localtime(&now);
  currenthour = curtime->tm_hour;


  /* don't manipulate the stats file before locking stats.lock exclusively */
  lockfd = open("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  if ((file = fopen(STATS_FILE, "r"))) {
    fscanf(file, "%d", &hour);
    fscanf(file, "%d %d %d %d %d %d %d %d", &Tm, &Tp, &Tl, &Tw, &Tu,
	   &Th, &Ts, &Tk);
    for (x = 0; x < 24; x++)
      fscanf(file, "%s %d %d %d %d %d %d %d %d %d", month[x], &date[x],
	     &Hm[x], &Hp[x], &Hl[x], &Hw[x], &Hu[x], &Hh[x], &Hs[x], &Hk[x]);
    fclose(file);

    x = hour;
    while (x != currenthour) {
      if (x > 0) {
	strcpy(month[x], month[x-1]);
	date[x] = date[x - 1];
      }
      else {
	strftime(month[0], 5, "%b", curtime);
	date[0] = curtime->tm_mday;
      }
      Hm[x] = Hp[x] = Hl[x] = Hw[x] = Hu[x] = Hh[x] = Hs[x] = Hk[x] = 0;
      x++;
      if (x > 23)
	x=0;
    }

    if (hour != currenthour) {
      Hm[hour] = Tm;
      Hp[hour] = Tp;
      Hl[hour] = Tl;
      Hw[hour] = Tw;
      Hu[hour] = Tu;
      Hh[hour] = Th;
      Hs[hour] = Ts;
      Hk[hour] = Tk;
      Tm = Tp = Tl = Tw = Tu = Th = Ts = Tk = 0;
    }

    sprintf(sendmail, "%s", SENDMAIL);
    file = (FILE *) popen(sendmail, "w");
    fprintf(file, "From: %s <%s>\n", REMAILER_NAME, REMAILER_ADDR);
    fprintf(file, "Subject: Re: Remailer Statistics\n");
    fprintf(file, "To: %s\n\n", to);
    fprintf(file, "Statistics for last 24 hours from anonymous remailer\n");
    fprintf(file, "%s\n\n", REMAILER_NAME);
    fprintf(file, "Number of messages per hour from "
		  "%s %2d %02d:00 to %s %2d %02d:59\n\n",
		  month[23], date[23], currenthour, month[0], date[0],
		  (currenthour + 23) % 24);
    for (x = 0; x < 24; x++)
      mmax = (Hm[x] > mmax) ? Hm[x] : mmax;
    mscale = (mmax > 67) ? 67.0 / mmax : 1.0;
    for (x0 = 0; x0 < 24; x0++) {
      x = (x0 + currenthour) % 24;
      fprintf(file, "%02d:00 (%3d) ", x, Hm[x]);
      if (Hm[x] > 0) {
	for (y = 0; y < Hp[x] * mscale; y++)
	  fprintf (file, "*");
	for (y = 0; y < (Hm[x] - Hp[x]) * mscale; y++)
	  fprintf (file, "+");
	Tm += Hm[x];
	Tp += Hp[x];
	Tl += Hl[x];
	Tw += Hw[x];
	Tu += Hu[x];
	Th += Hh[x];
	Ts += Hs[x];
	Tk += Hk[x];
      }
      fprintf(file, "\n");
    }
    fprintf(file, "\n");
    fprintf(file, "Total messages remailed in last 24 hours: \t%5d\n", Tm);
    if (ALLOW_PGP)
      fprintf(file, "Number of messages encrypted with PGP: \t\t%5d\n", Tp);
    fprintf(file, "Number of messages queued with latency: \t%5d\n", Tl);
    if (ALLOW_WWW)
      fprintf(file, "Number of WWW/HTML requests processed: \t\t%5d\n", Tw);
    if (ALLOW_POST)
      fprintf(file, "Number of messages posted to Usenet: \t\t%5d\n", Tu);
    fprintf(file, "Number of help requests: \t\t\t%5d\n", Th);
    if (ALLOW_PGP)
      fprintf(file, "Number of key requests: \t\t\t%5d\n", Tk);
    fprintf(file, "Number of stats requests: \t\t\t%5d\n", Ts);
    pclose(file);
  }
  flock(lockfd, LOCK_UN);
  close(lockfd);
}
