/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <syslog.h>
#include "freedom.h"

#ifdef USE_RX
#include <rxposix.h>	/* for use with GNU Rx regexp library */
/* #include <rx.h> */	/* for use with libc regexp functions */

int
rxmatch(const char *string, const char *pattern)
{
  int err, match;
  char errmsg[80];
  regex_t compiled;

  if ((err = regcomp(&compiled, pattern, REG_EXTENDED | REG_ICASE))) {
    regerror(err, &compiled, errmsg, sizeof(errmsg));
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Couldn't compile regex %s: %s", pattern, errmsg);
    regfree(&compiled);
    return 0;
  }
  else {
    match = regexec(&compiled, string, 0, NULL, 0);
    regfree(&compiled);
    return !match;
  }
}
#else /* !USE_RX */
int
rxmatch(const char *string, const char *pattern)
{
  return strifind(string, pattern);
}
#endif /* !USE_RX */

void
chop(char *string)
{
  unsigned i, j;

  for (i = 0, j = 0; i < strlen(string) && string[i] != '\n'; i++) {
    if((j > 0) || !isspace(string[i])) {
      string[j] = string[i];
      j++;
    }
  }
  string[j] = '\0';
}

int
strileft(const char *string, const char *keyword)
{
  unsigned i;

  for (i = 0; i < strlen(keyword); i++) {
    if (tolower(string[i]) != tolower(keyword[i]))
      return 0;
  }
  return 1;
}

int
strifind(const char *string, const char *keyword)
{
  unsigned i, j;

  for (i = 0, j = 0; i < strlen(string); i++) {
    if (tolower(string[i]) == tolower(keyword[j])) {
      if (++j >= strlen(keyword))
	return 1;
    }
    else j = 0; /* reset search on non-match */
  }
  return 0;
}

int
strieq(const char *s1, const char *s2)
{
  unsigned i;

  if (strlen(s1) != strlen(s2))
    return 0;

  for (i = 0; i <= strlen(s1); i++) {
    if (tolower(s1[i]) != tolower(s2[i]))
      return 0;
  }
  return 1;
}

int
blocked(const char *address, const char *blockfilename)
{
  FILE *infile;
  char line[BUFSIZ];
  int block = 0;

  if (!(infile = fopen(blockfilename, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", blockfilename);
    return 0;
  }

  while (fgets(line, sizeof(line), infile)) {
    chop(line);
    if (line[0] == '#' || strlen(line) == 0)
      continue; /* skip blank or comment lines */
    else if (rxmatch(address, line)) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "%s matches %s in %s",
	       address, line, basename(blockfilename));
      block = 1;
      break;
    }
  }
  fclose(infile);
  return block;
}

int
print_blocked(FILE *outfile, const char *blockfilename)
{
  FILE *infile;
  char line[BUFSIZ];

  if (!(infile = fopen(blockfilename, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", blockfilename);
    return 1;
  }

  while (fgets(line, sizeof(line), infile)) {
    if (line[0] == '#' || line[0] == '\n')
      continue; /* skip blank or comment lines */
    else
      fputs(line, outfile);
  }
  return fclose(infile);
}

void
cut(const char *origfilename, const char *cutfilename, const char *cutmarks,
    const char *header, const int fakeheader)
{
  char tempfilename[BUFSIZ], line[BUFSIZ], tmpline[BUFSIZ];
  FILE *infile, *outfile;
  int cut = 0, headers = 1, linenum = 0;

  sprintf(tempfilename, "%sT", origfilename);
  rename(origfilename, tempfilename);

  if (!(infile = fopen(tempfilename, "r"))) {
    outfile = fopen(origfilename, "w");
    fclose(outfile);
    return;
  }
  outfile = fopen(origfilename, "w");

  while (fgets(line, sizeof(line), infile)) {
    strcpy(tmpline, line);
    chop(tmpline);
    if (!headers) {
      if ((linenum > 1) || (strlen(line) > 1))
	linenum++;
      if ((linenum == 1) && strstr(line, "::")) {
	headers = 1;
	fputs(line, outfile);
      }
      else {
	if (!cut && !strcmp(tmpline, cutmarks)) {
	  fclose(outfile);
	  if (USE_SYSLOG)
	    syslog(LOG_DEBUG, "Cutting message");
	  outfile = fopen(cutfilename, "w");
	  if (fakeheader)
	    fputs("Cut-Recursive: Yes\n\n", outfile);
	  cut = 1;
	}
	else if (linenum > 0)
	  fputs(line, outfile);
      }
    }
    else {
      if (strlen(line) <= 1) {
	headers = 0;
	fputs("\n", outfile);
      }
      else {
	if (!strileft(line, header))
	  fputs(line, outfile);
      }
    }
  }
  fclose(infile);
  fclose(outfile);
  unlink(tempfilename);
}

void
mailfile(const char *filename, const char *to)
{
  FILE *infile, *outfile;
  char sendmail[BUFSIZ], line[BUFSIZ];

  if (blocked(to, DEST_BLOCK)) {
    if (USE_SYSLOG)
      syslog(LOG_NOTICE, "Ignoring request");
  }
  else {
    if (USE_SYSLOG)
      syslog(LOG_INFO, "Mailing %s", basename(filename));
    sprintf(sendmail, "%s", SENDMAIL);
    if ((infile = fopen(filename, "r"))) {
      outfile = popen(sendmail, "w");
      fprintf(outfile, "From: %s <%s>\n"
		 "To: %s\n", REMAILER_NAME, REMAILER_ADDR, to);
      while (fgets(line, sizeof(line), infile))
	fputs(line, outfile);
      fclose(infile);
      pclose(outfile);
    }
  }
}

/* get the basename of file <name> */
const char *
basename(const char *name)
{
  char *tmp = strrchr(name, '/');
  return (tmp ? tmp + 1 : name);
}
