/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <syslog.h>
#include "freedom.h"

#ifdef USE_RX
#include "pcre.h"

int
rxmatch(const char *string, const char *pattern)
{
  int errptr, match;
  const char *error;
  pcre *compiled;

  if ((compiled = pcre_compile(pattern, PCRE_CASELESS, &error, &errptr))) {
    match = pcre_exec(compiled, NULL, string, strlen(string),
		      PCRE_CASELESS, NULL, 0);
    free(compiled);
    return (match >= 0);
  }
  else {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Error in regex at offset %d: %s", errptr, error);
    return 0;
  }
}
#else /* !USE_RX */
int
rxmatch(const char *string, const char *pattern)
{
  return strifind(string, pattern);
}
#endif /* !USE_RX */

void
flushleft(char *string, const unsigned pos)
{
  unsigned i, j;

  for (i = pos, j = 0; i <= strlen(string); i++) {
    if(j > 0 || !isspace(string[i])) {
      string[j] = string[i];
      j++;
    }
  }
}

void
chop(char *string)
{
  size_t l = strlen(string);

  if (l > 0 && string[l - 1] == '\n') {
    string[l - 1] = '\0';
    if (l > 1 && string[l - 2] == '\r')
      string[l - 2] = '\0';
  }
}

int
strileft(const char *string, const char *keyword)
{
  unsigned i;

  for (i = 0; i < strlen(keyword); i++) {
    if (tolower(string[i]) != tolower(keyword[i]))
      return 0;
  }
  return 1;
}

int
strifind(const char *string, const char *keyword)
{
  unsigned i, j;

  for (i = 0, j = 0; i < strlen(string); i++) {
    if (tolower(string[i]) == tolower(keyword[j])) {
      if (++j >= strlen(keyword))
	return 1;
    }
    else j = 0; /* reset search on non-match */
  }
  return 0;
}

int
strieq(const char *s1, const char *s2)
{
  unsigned i;

  if (strlen(s1) != strlen(s2))
    return 0;

  for (i = 0; i <= strlen(s1); i++) {
    if (tolower(s1[i]) != tolower(s2[i]))
      return 0;
  }
  return 1;
}

int
rxfind(const char *address, const char *listname)
{
  FILE *infile;
  char line[BUFSIZ];
  int match = 0;

  if (!(infile = fopen(listname, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", listname);
    return 0;
  }

  while (fgets(line, sizeof(line), infile)) {
    chop(line);
    if (line[0] == '#' || strlen(line) == 0)
      continue; /* skip blank or comment lines */
    else if (rxmatch(address, line)) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "%s matches %s in %s", address, line, listname);
      match = 1;
      break;
    }
  }
  fclose(infile);
  return match;
}

int
findline(const char *address, const char *listname)
{
  FILE *infile;
  char line[BUFSIZ];
  int linenum = 0;

    if (!(infile = fopen(listname, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", listname);
    return 0;
  }

  while (fgets(line, sizeof(line), infile)) {
    linenum++;
    if (strifind(line, address)) {
      fclose(infile);
      return linenum;
    }
  }
  fclose(infile);
  return 0;
}

int
print_list(FILE *outfile, const char *listname)
{
  FILE *infile;
  char line[BUFSIZ];

  if (!(infile = fopen(listname, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", listname);
    return 1;
  }

  while (fgets(line, sizeof(line), infile)) {
    if (line[0] == '#' || line[0] == '\n')
      continue; /* skip blank or comment lines */
    else
      fputs(line, outfile);
  }
  return fclose(infile);
}

void
cut(const char *origfilename, const char *cutfilename, const char *cutmarks,
    const char *header)
{
  char tempfilename[BUFSIZ], line[BUFSIZ], tmpline[BUFSIZ];
  FILE *infile, *outfile;
  int cut = 0, headers = 1, linenum = 0;

  sprintf(tempfilename, "%sT", origfilename);
  rename(origfilename, tempfilename);
  outfile = fopen(origfilename, "w");
  if (!(infile = fopen(tempfilename, "r"))) {
    fclose(outfile);
    return;
  }

  while (fgets(line, sizeof(line), infile)) {
    strcpy(tmpline, line);
    chop(tmpline);
    if (!headers) {
      if (linenum > 1 || strlen(line) > 1)
	linenum++;
      if (linenum == 1 && !strncmp(line, "::", 2)) {
	headers = 1;
	fputs(line, outfile);
      }
      else {
	if (!cut && !strcmp(tmpline, cutmarks)) {
	  if (!strcmp(header, "encrypt-key:"))
	    fputs(line, outfile);
	  fclose(outfile);
	  if (USE_SYSLOG)
	    syslog(LOG_DEBUG, "Cutting message");
	  outfile = fopen(cutfilename, "w");
	  if (!strcmp(header, "cutmarks:"))
	    fputs("Recursive: Cut\n\n", outfile);
	  cut = 1;
	}
	else if (linenum > 0)
	  fputs(line, outfile);
      }
    }
    else {
      if (!strileft(line, header))
	fputs(line, outfile);
      if (strlen(tmpline) == 0)
	headers = 0;
    }
  }
  fclose(infile);
  fclose(outfile);
  unlink(tempfilename);
}

void
mailfile(const char *filename, const char *to)
{
  FILE *infile, *outfile;
  char sendmail[BUFSIZ], line[BUFSIZ];

  if (rxfind(to, DEST_BLOCK)) {
    if (USE_SYSLOG)
      syslog(LOG_NOTICE, "Ignoring request");
  }
  else {
    if (USE_SYSLOG)
      syslog(LOG_INFO, "Mailing %s to %s", filename, to);
    sprintf(sendmail, "%s", SENDMAIL);
    if ((infile = fopen(filename, "r"))) {
      outfile = popen(sendmail, "w");
      fprintf(outfile, "From: %s <%s>\n"
		 "To: %s\n", REMAILER_NAME, REMAILER_ADDR, to);
      while (fgets(line, sizeof(line), infile))
	fputs(line, outfile);
      fclose(infile);
      pclose(outfile);
    }
  }
}
