/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include "freedom.h"

int
run_cmd(const char *command, const char *option, const char *arg,
	FILE *infile, FILE *outfile)
{
  FILE *passfile;
  int pipefd[2], lockfd, status;
  pid_t pid = getpid();
  char line[BUFSIZ];

  /* PGP needs exclusive access to randseed.bin */
  if (!strcmp(command, PGP)) {
    lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0600);
    flock(lockfd, LOCK_EX);
    write(lockfd, &pid, sizeof(pid_t));
  }

  /* write passphrase to pipe */
  pipe(pipefd);
  if (!strcmp(option, "-d")) {
    if ((passfile = fopen(PASS_FILE, "r"))) { /* get passphrase from file */
      fgets(line, sizeof(line), passfile);
      write(pipefd[1], line, strlen(line));
      fclose(passfile);
    }
    else {
      if (USE_SYSLOG)
	syslog(LOG_WARNING, "Can't open %s: %m", PASS_FILE);
    }
  }
  else if (!strcmp(option, "-c")) { /* get passphrase from argument string */
    write(pipefd[1], arg, strlen(arg));
    write(pipefd[1], "\n", 1);
  }
  close(pipefd[1]);

  if (!fork()) { /* child */
    if (infile) {
      dup2(fileno(infile), STDIN_FILENO);
      fclose(infile);
    }
    if (outfile) {
      dup2(fileno(outfile), STDOUT_FILENO);
      fclose(outfile);
    }

    /* connect passphrase pipe to child and set environment */
    if (!strcmp(option, "-d") || !strcmp(option, "-c"))
      sprintf(line, "%d", dup(pipefd[0]));
    if (!strcmp(command, PGP)) {
      setenv("PGPPATH", PGP_DIR, 1);
      setenv("PGPPASSFD", line, 1);
    }
    close(pipefd[0]);
    fclose(stderr); /* shut up! */

    if (!strcmp(command, PGP)) {
      if (!strcmp(option, "-d")) /* decrypt infile to outfile */
	execl(command, "pgp", "+batchmode", "+verbose=0", "-f", NULL);
      else if (!strcmp(option, "-e")) /* RSA-encrypt infile to outfile */
	execl(command, "pgp", "+batchmode", "+verbose=0", "+armorlines=0",
	      "-fat", option, arg, NULL);
      else if (!strcmp(option, "-c")) /* IDEA-encrypt infile to outfile */
	execl(command, "pgp", "+batchmode", "+verbose=0", "+armorlines=0",
	      "-fat", option, NULL);
      else if (!strcmp(option, "-a")) /* ASCII-armore infile to outfile */
	execl(command, "pgp", "+batchmode", "+verbose=0", "+armorlines=0",
	      "-f", option, NULL);
      else if (!strcmp(option, "-kv")) { /* look for public key in keyring */
	if (!outfile)
	  fclose(stdout); /* we don't need the output, only the exit status */
	execl(command, "pgp", "+batchmode", "+verbose=0", option, arg, NULL);
      }
    }
    else if (!strcmp(command, GPG)) {
      if (!strcmp(option, "-d")) /* decrypt infile to outfile */
	execl(command, "gpg", "--homedir", GPG_DIR, "--passphrase-fd", line,
	      "--batch", option, NULL);
    }
    else if (!strcmp(command, MIXMASTER)) {
      if (!strcmp(option, "-l")) { /* remix infile to outfile */
	sprintf(line, "%d", findline(arg, TYPE2_LIST));
	execl(command, "mixmaster", "-f", "-o", "-", option, line, NULL);
      }
    }
    else if (!strcmp(command, WGET)) {
      if (!strcmp(option, "-O")) /* append WWW page to file arg */
	execl(command, "wget", "-q", "-goff", "-s", option, arg, NULL);
    }
  }
  else /* parent */
    wait(&status);

  if (!strcmp(command, PGP)) {
    flock(lockfd, LOCK_UN);
    close(lockfd);
  }
  if (WIFEXITED(status))
    return WEXITSTATUS(status);
  else
    return -1;
}

int
append_url(const char *urlfilename, const char *toappendto, const char *url)
{
  int status;
  FILE *infile, *outfile;
  struct stat urlfilestat;

  /* URL is blocked; write error message to outfile */
  if (rxfind(url, URL_BLOCK)) {
    if (USE_SYSLOG)
      syslog(LOG_NOTICE, "Ignoring request");
    outfile = fopen(toappendto, "a");
    fprintf(outfile, "You are not allowed to download the URL\n"
		     "%s\n"
		     "through the %s.\n", url, REMAILER_NAME);
    fclose(outfile);
    return 0;
  }

  /* call Wget to retrieve the requested URL */
  if (USE_SYSLOG)
    syslog(LOG_INFO, "Downloading URL %s", url);
  run_cmd(WGET, "-O", urlfilename, NULL, NULL);
  if (USE_SYSLOG)
    syslog(LOG_INFO, "Download finished");

  stat(urlfilename, &urlfilestat);
  if (urlfilestat.st_size == 0L ||
      (SIZE_LIMIT && urlfilestat.st_size > SIZE_LIMIT)) {
    if (urlfilestat.st_size == 0L) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "Invalid URL %s", url);
    }
    else if (SIZE_LIMIT && urlfilestat.st_size > SIZE_LIMIT) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "Oversized URL %s: %ld bytes",
	       url, urlfilestat.st_size);
      outfile = fopen(toappendto, "a");
      fprintf(outfile, "You are not allowed to download more than\n"
		       "%d bytes\n"
		       "through the %s.\n", SIZE_LIMIT, REMAILER_NAME);
      fclose(outfile);
    }
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Deleting %s", urlfilename);
    unlink(urlfilename);
    if (WIFEXITED(status))
      return WEXITSTATUS(status);
    else
      return -1;
  }

  /* call PGP to ASCII-armor and append URL file to message */
  infile = fopen(urlfilename, "r");
  outfile = fopen(toappendto, "a");
  fputs("\n", outfile);
  fflush(outfile);
  run_cmd(PGP, "-a", NULL, infile, outfile);
  fclose(outfile);
  fclose(infile);
  unlink(urlfilename);
  if (WIFEXITED(status))
    return WEXITSTATUS(status);
  else
    return -1;
}
