#include <stdio.h>
#include <stdlib.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <math.h>
#include <sys/wait.h>
#include "rlist.h"

char *spool_fn = "/var/spool/mail/remailer-list3";
char *state_fn = "/usr/local/remailer-list3/state";
char *report_com = "/usr/local/remailer-list3/format";
char *dump_fn = "/usr/local/remailer-list3/dump";
char *ping_address = "remailer-list3@kiwi.cs.berkeley.edu";
char *premail_com = "/usr/local/remailer-list3/premail +premailrc=/usr/local/remailer-list3/.premailrc -t";

int state_dirty;

int num_remailers;
summ *summaries;
int state_f;
int now;

int main (int argc, char **argv) {
   int add, toggle, run;
   int begin;
   int last_report, last_ping;

   init_state ();

   add = 0;
   toggle = 0;
   run = 1;
   argc--, argv++;
   while (argc) {
      if (add) {
	 add_remailer (argv[0]);
	 add = 0;
      } else if (toggle) {
	 toggle_remailer (argv[0]);
	 toggle = 0;
      } else if (!strcmp (argv[0], "-add")) {
	 add = 1;
	 run = 0;
      } else if (!strcmp (argv[0], "-toggle")) {
	 toggle = 1;
	 run = 0;
      }
      argc--, argv++;
   }

   setlinebuf (stdout);
   begin = gettime ();
/* report_all (); */
   last_report = 0;
   last_ping = 0;
   while (run) {
      gettime ();
      state_dirty = 0;
      handle_mail ();
      if (now >= last_ping + 0) {
	 do_pings ();
	 last_ping = now;
      }
      if (now > last_report + 300) {
	 make_report ();
	 last_report = now;
      }
      if (state_dirty) update_state ();
      sleep (1);
/*    if (now >= begin + 600) run = 0; */
   }
   done_state ();
   exit (0);
}

void init_state (void) {
   /* Read the prelude of the state file into num_remailers and summaries.
      Leave state_f open but unlocked. */
   struct stat stat_buf;
   int status;

   status = stat (state_fn, &stat_buf);
   if (status == 0) {
      state_f = open (state_fn, O_RDWR);
      if (state_f < 0) {
	 fprintf (stderr, "Can't open state file\n");
	 exit (1);
      }
      status = lockf (state_f, F_LOCK, 0);
      lseek (state_f, 0, SEEK_SET);
      read (state_f, &num_remailers, sizeof (int));
      summaries = (summ *) malloc (sizeof (summ) * num_remailers);
      read (state_f, summaries, sizeof (summ) * num_remailers);
   } else {
      /* Create the status file */
      printf ("creating state file...\n");
      state_f = open (state_fn, O_RDWR | O_CREAT, 0644);
      if (state_f < 0) {
	 fprintf (stderr, "Can't create state file\n");
	 exit (1);
      }
      lockf (state_f, F_LOCK, 0);
      lseek (state_f, 0, SEEK_SET);
      num_remailers = 0;
      summaries = (summ *) malloc (4);
      write (state_f, &num_remailers, sizeof (int));
   }
   lockf (state_f, F_ULOCK, 0);
}

void touch_state (void) {
   if (!state_dirty) lockf (state_f, F_LOCK, 0);
   state_dirty = 1;
}

void update_state (void) {
   if (!state_dirty) lockf (state_f, F_LOCK, 0);
   lseek (state_f, 0, SEEK_SET);
   write (state_f, &num_remailers, sizeof (int));
   write (state_f, summaries, sizeof (summ) * num_remailers);
   lockf (state_f, F_ULOCK, 0);
}

void done_state (void) {
   /* Close state file, deallocate structures */
   close (state_f);
   free (summaries);
}

int add_remailer (char *remailer_name) {
/* Add the remailer to the summary */
   int i, j;
   int off, num;
   ping *buf;

   i = lookup_remailer (remailer_name);
   if (i >= 0) {
      printf ("Remailer %s already present\n", remailer_name);
      if (summaries[i].flags & 1) {
	 printf ("Pinging of remailer %s enabled\n", remailer_name);
	 summaries[i].flags &= ~1;
	 update_state ();
      }
      return 1;
   }
   printf ("adding remailer %s\n", remailer_name);
   /* First, check to see if there's space after the summary */
   evict (num_remailers * sizeof (summ) + sizeof (int), sizeof (summ));
   /* Then, reallocate the summary */
   i = num_remailers++;
   summaries = (summ *) realloc (summaries, num_remailers * sizeof (summ));
   strncpy (summaries[i].nickname, remailer_name, 16);
   summaries[i].last_ping = 0;
   summaries[i].last_response = 0;
   summaries[i].last_error = 0;
   summaries[i].last_update = 0;
   summaries[i].flags = 0;
   num = 256;
   off = allocate (num * sizeof (ping),
		   num_remailers * sizeof (summ) + sizeof (int));
   printf ("allocation: off = %d\n", off);
   buf = (ping *) malloc (num * sizeof (ping));
   j = 0;
   while (j != num) {
      buf[j].ping_time = 0;
      buf[j].response_time = 0;
      j++;
   }
   summaries[i].off_pings = off;
   summaries[i].num_pings = num;
   free (put_pings (i, buf));
   summaries[i].stats0.latency = 0;
   summaries[i].stats0.rel = 0;
   summaries[i].stats0.tc = 1;
   summaries[i].stats0.reserved = 0;
   summaries[i].stats1.latency = 0;
   summaries[i].stats1.rel = 0;
   summaries[i].stats1.tc = 1;
   summaries[i].stats1.reserved = 0;
   update_state ();
}

int toggle_remailer (char *remailer_name) {
/* Toggle pinging of the remailer */
   int i;

   i = lookup_remailer (remailer_name);
   if (i < 0) {
      printf ("Remailer %s not found\n", remailer_name);
      return 1;
   }
   summaries[i].flags ^= 1;
   if (summaries[i].flags & 1)
      printf ("Pinging of remailer %s disabled\n", remailer_name);
   else
      printf ("Pinging of remailer %s enabled\n", remailer_name);
   update_state ();
   return 0;
}

int conflict (int start, int size) {
   /* Determine whether the range indicated contains any ping data. */
   int i;
   int cend;
   int pstart, pend;

   cend = start + size * sizeof (ping);
   i = 0;
   while (i != num_remailers) {
      pstart = summaries[i].off_pings;
      pend = pstart + summaries[i].num_pings * sizeof (ping);
      if ((start >= pstart && start < pend) ||
	  (pstart >= start && pstart < cend))
	 return 1;
      i++;
   }
   return 0;
}

int evict (int start, int size) {
   /* Evict any ping records that live in the specified range. Return value
      tells whether any records were evicted. */
   ping *buf;
   int evicted;
   int i;
   int cend;
   int pstart, pend;

   cend = start + size;
   evicted = 0;
   i = 0;
   while (i != num_remailers) {
      pstart = summaries[i].off_pings;
      pend = pstart + summaries[i].num_pings * sizeof (ping);
      if ((start >= pstart && start < pend) ||
	  (pstart >= start && pstart < cend)) {
	 buf = get_pings (i);
	 summaries[i].off_pings =
	    allocate (summaries[i].num_pings * sizeof (ping), cend);
	 free (put_pings (i, buf));
	 evicted = 1;
      }
      i++;
   }
   return evicted;
}

int allocate (int size, int start) {
/* Return an offset to a free space with at least (size) bytes, beginning
   no earlier than offset start. */
   int alloc_ptr;
   int i;
   int pstart, pend;

   /* Extremely crude algorithm. At some point, I may want to extend the
      thing to look for holes, but it's not necessary. */
   alloc_ptr = start;
   i = 0;
   while (i != num_remailers) {
      pstart = summaries[i].off_pings;
      pend = pstart + summaries[i].num_pings * sizeof (ping);
      if (alloc_ptr < pend)
	 alloc_ptr = pend;
      i++;
   }
   return alloc_ptr;
}

ping *get_pings (int i) {
   /* Get the ping data for remailer number i - alloc the buffer and
    return it. */
   ping *buf;
   int buf_size;

   lseek (state_f, summaries[i].off_pings, SEEK_SET);
   buf_size = summaries[i].num_pings * sizeof (ping);
   buf = (ping *) malloc (buf_size);
   read (state_f, buf, buf_size);
   return buf;
}

ping *put_pings (int i, ping *buf) {
   /* Store the ping data for remailer number i - return the buffer. */
   int buf_size;

   lseek (state_f, summaries[i].off_pings, SEEK_SET);
   buf_size = summaries[i].num_pings * sizeof (ping);
   write (state_f, buf, buf_size);
   return buf;
}

int get_msg (char *buf, int buf_size) {
/* Get a message from the spool, store in the buffer (limited to buf_size
   bytes, truncate if exceeded). Return 0 for success. */
   int f;
   int status;

   f = open (spool_fn, O_RDWR);
   if (f < 0) {
      fprintf (stderr, "Can't open spool file!\n");
      exit (1);
   }
   status = lockf (f, F_LOCK, 0);
   lseek (f, 0, SEEK_SET);
   /* Read the message, handle the remainder of the spool file. */
   status = lockf (f, F_ULOCK, 0);
   close (f);
   return 0;
}

int gettime (void) {
/* Return time in standard Unix format (seconds since 00:00 UTC 1 Jan 1970.
   This routine also sets the global variable "now". */
   struct timeval t;
   struct timezone tz;

   gettimeofday (&t, &tz);
   now = t.tv_sec;
   return now;
}

void do_pings (void) {
   int i, ii;
   int rot;

   if (!num_remailers) return; /* to avoid div by zero next */
   rot = rand () % num_remailers;
   i = 0;
   while (i != num_remailers) {
      ii = (i + rot) % num_remailers;
      if (now >= ping_time (ii) && !(summaries[ii].flags & 1)) {
	 ping_remailer (ii);
	 return; /* only do one ping per event loop */
      }
      i++;
   }
}

int ping_time (int i) {
   /* When should we ping the remailer? */
   int t, t_candidate;

   t = summaries[i].last_ping + 1800 + (rand () & 15);
   if (summaries[i].stats1.latency != 0) {
      t_candidate = summaries[i].last_ping + summaries[i].stats1.latency * 0.5;
      if (t_candidate > t) t = t_candidate;
   }
   if (summaries[i].last_response) {
      t_candidate = summaries[i].last_ping
	 + (summaries[i].last_ping - summaries[i].last_response) * 0.2;
      if (t_candidate > t) t = t_candidate;
   }
   return t;
}

void ping_remailer (int i) {
   char nick[17];

   strncpy (nick,summaries[i].nickname, 16);
   nick[16] = 0;
   printf ("pinging remailer %s at %d\n", nick, now);
   if (!send_ping (i, now)) {
      touch_state ();
      insert_ping (i, now);
      summaries[i].last_ping = now;
   }
}

void insert_ping (int i, int when) {
   /* insert the ping for time "when" in remailer i's ping records */
   ping *buf;
   int j;

   buf = get_pings (i);
   roll_ping (&(summaries[i].stats0), buf);
   j = 0;
   while (j != summaries[i].num_pings - 1) {
      buf[j] = buf[j + 1];
      j++;
   }
   buf[j].ping_time = when;
   buf[j].response_time = 0;
   free (put_pings (i, buf));
}

int send_ping (int i, int when) {
   /* Invoke premail to actually send the ping. Return 0 on success */
   FILE *f;
   char nick[17];
   char tmpfn[32];
   char pre_com[128];
   int status;

   sprintf (tmpfn, "/tmp/rlist.%d", getpid ());
   f = fopen (tmpfn, "w");
   if (f == NULL) {
      fprintf (stderr, "Error invoking premail\n");
      return 1;
   }
   strncpy (nick, summaries[i].nickname, 16);
   nick[16] = 0;
   fprintf (f, "To: %s\n", ping_address);
   fprintf (f, "Subject: remailer list ping %s %d\n", nick, when);
   fprintf (f, "Chain: %s\n", nick);
   fprintf (f, "No-Reply: \n");
   fprintf (f, "\n");
   fprintf (f, "itch %s %d\n", nick, when);
   fprintf (f, "knee\n");
   fprintf (f, "sand\n");
   fclose (f);
   sprintf (pre_com, "%s < %s", premail_com, tmpfn);
   status = system (pre_com);
   if (!WIFEXITED(status) || WEXITSTATUS(status)) {
      fprintf (stderr, "premail error, status = %d\n", status);
      return 1;
   }      
   unlink (tmpfn);
   return 0;
}

void roll_ping (stats *s, ping *p) {
   /* Roll the statistics forward over the ping p[1] (i.e. incorporate that
      ping into the statistics. Note: this looks at both p[0] and p[1]. */
   double raw_rel, raw_lat;
   double tc, tc0, tc1, tc2;
   double ratio;

   raw_lat = 0;
   if (p[1].response_time != 0) {
      /* positive ping */
      raw_rel = 1;
      raw_lat = p[1].response_time - p[1].ping_time;
      tc2 = 1;
   } else {
      raw_rel = 0;
      if (s->latency == 0)
	 tc2 = 1;
      else {
	 ratio = (now - p[1].ping_time) / s->latency;
	 if (ratio < 1) tc2 = 0.01 * ratio;
	 else if (ratio < 5) tc2 = - .2375 + 0.2475 * ratio;
	 else tc2 = 1;
      }
   }
   if (p[0].ping_time)
      tc0 = 1 - exp ((p[1].ping_time - p[0].ping_time) * -2.314814e-06);
   /* 2.314814e-06 = 5 days */
   else {
      raw_rel *= 0.9;
      tc0 = 1;
   }
   tc1 = s->tc;
   tc = tc2 * (tc1 + tc0 * (1 - tc1));
   s->rel += tc * (raw_rel - s->rel);
   if (p[1].response_time != 0)
      s->latency += tc * (raw_lat - s->latency);
   if (p[1].ping_time)
      s->tc = tc1 / (1 + tc1);
}

void update_stats (int i) {
   /* Update the stats1 field of remailer i */
   ping *buf;
   int j;
   stats s;

   buf = get_pings (i);
   s = summaries[i].stats0;
   j = 0;
   while (j != summaries[i].num_pings - 1) {
      roll_ping (&s, buf + j);
      j++;
   }
   summaries[i].stats1 = s;
   free (buf);
}

void handle_mail (void) {
   struct stat stat_buf;
   int status;
   int f, f1;
   char *buf, *p;

   status = stat (spool_fn, &stat_buf);
   if (status == 0 && stat_buf.st_size != 0) {
      printf ("spool file is of size %d\n", stat_buf.st_size);
      f = open (spool_fn, O_RDWR);
      if (f < 0) {
	 fprintf (stderr, "Can't open spool file!\n");
	 exit (1);
      }
      status = lockf (f, F_LOCK, 0);
      lseek (f, 0, SEEK_SET);
      status = stat (spool_fn, &stat_buf);
      /* For the time being, process the whole thing in one gulp */
      buf = (char *) malloc (stat_buf.st_size);
      read (f, buf, stat_buf.st_size);
      p = buf;
      while (p != buf + stat_buf.st_size)
	 p = handle_msg (p, buf + stat_buf.st_size - p);
      free (buf);
      /* Delete the mail spool */
      f1 = open (spool_fn, O_WRONLY | O_TRUNC, 0600);
      if (f1 >= 0) close (f1);
      status = lockf (f, F_ULOCK, 0);
      close (f);
   }
}

char *handle_msg (char *buf, int size) {
   /* Handle a single message. Return the pointer to right after the
      message. */
   char *p, *q;
   int i, when;
   int state; /* 0 = at "From " line
		 1 = in header
		 2 = in body
		 90-99 = error */
   int ok;

   ok = 0;
   p = buf;
   state = 0;
   while (p != buf + size) {
      if (state == 0) {
	 if (prefix ("From ", p))
	    state = 1;
	 else {
	    fprintf (stderr, "Malformed message (no From at beginning)\n");
	    state = 90;
	 }
      } else if (state == 1 || state == 91) {
	 /* state == 91 allows multiple Subject lines, a la Mixmaster bug */
	 if (prefixi ("Subject: ", p)) {
	    q = p + strlen ("Subject: ");
	    while (q != buf + size && *q != '\n' &&
		   !prefix ("remailer list ping ", q))
	       q++;
	    if (q == buf + size || *q == '\n')
	       state = 91;
	    else {
	       q += strlen ("remailer list ping ");
	       i = lookup_remailer (q);
	       if (i < 0)
		  state = 92;
	       else {
		  while (q != buf + size && *q != ' ' && *q != '\n')
		     q++;
		  if (q == buf + size || *q == '\n')
		     state = 93;
		  else {
		     when = atoi (q);
		     ok = !register_response (i, when);
		     if (state == 91)
			printf ("response had double Subject: field\n");
		  }
	       }
	    }
	 } else {
	    if (prefix ("\n", p))
	       state = 2;
	 }
      } else if (state >= 2) {
	 if (prefix ("From ", p)) {
	    if (!ok)
	       dump_msg (buf, p - buf, state);
	    return p;
	 }
      }
      while (p != buf + size && *p != '\n')
	 p++;
      if (p != buf + size) p++;
   }
   if (!ok)
      dump_msg (buf, p - buf, state);
   return p;
}

int prefix (char *s1, char *s2) {
   /* Tell whether s1 is a prefix of s2 */
   return !strncmp (s1, s2, strlen (s1));
}

int prefixi (char *s1, char *s2) {
   /* Tell whether s1 is a case-insensitive prefix of s2. */
   return !strncasecmp (s1, s2, strlen (s1));
}

int lookup_remailer (char *name) {
   /* Return number of remailer named by name, or -1.
      Name may be either a space or null terminated string. */
   int i, j;
   char *q;

   i = 0;
   while (i != num_remailers) {
      q = summaries[i].nickname;
      j = 0;
      while (j != 16 && name[j] != ' ' && name[j] != 0 && name[j] == q[j])
	 j++;
      if (j == 16 || ((name[j] == ' ' || name[j] == 0) && q[j] == 0))
	  return i;
      i++;
   }
   return -1;
}

int register_response (int i, int when) {
   /* Record the response in the ping record. Return 0 for success. */
   ping *buf;
   int j;
   char nick[16];

   strncpy (nick, summaries[i].nickname, 16);
   nick[16] = 0;
   buf = get_pings (i);
   j = 0;
   while (j != summaries[i].num_pings) {
      if (buf[j].ping_time == when) {
	 if (buf[j].response_time) {
	    fprintf (stderr,
		     "Warning: remailer %s ping %d multiple response\n",
		     nick, when);
	 } else {
	    touch_state ();
	    summaries[i].last_response = now;
	    buf[j].response_time = now;
	    free (put_pings (i, buf));
	    printf ("%s responded to ping %d at %d\n", nick, when, now);
	    return 0;
	 }
      }
      j++;
   }
   free (buf);
   return 1;
}

void dump_msg (char *buf, int size, int code) {
   /* Dump the message so that humans can look at it */
   int f;
   char line[80];

   f = open (dump_fn, O_WRONLY | O_CREAT, 0644);
   if (f < 0) return;
   lseek (f, 0, SEEK_END);
   sprintf (line, "!Dump code %d\n", code);
   write (f, line, strlen (line));
   write (f, buf, size);
   close (f);
}

void report (int i) {
   /* Print a report on remailer i - for debugging. */
   char nick[17];

   strncpy (nick, summaries[i].nickname, 16);
   nick[16] = 0;
   update_stats (i);
   printf ("%s, %d %d %d %d rel=%6.2f, lat=%f\n", nick,
	   summaries[i].last_ping, summaries[i].last_response,
	   summaries[i].last_error, summaries[i].last_update,
	   summaries[i].stats1.rel * 100, summaries[i].stats1.latency);
}

void report_all (void) {
   int i;

   i = 0;
   while (i != num_remailers) {
      report (i);
      i++;
   }
}

void debug_rolling (int i) {
   ping *buf;
   stats s;
   int j;

   buf = get_pings (i);
   s = summaries[i].stats0;
   j = 0;
   while (j != summaries[i].num_pings - 1) {
      printf ("%d, %d rel=%2.2f, lat=%f\n", buf[j].ping_time,
	      buf[j].response_time, s.rel, s.latency);
      roll_ping (&s, buf + j);
      j++;
   }
   free (buf);
}

void make_report (void) {
   FILE *f;
   int i;
   double rel;
   int lat, hr, min, sec;
   char nick[17];
   char latstr[10];
   char *p;
   char hist[13];
   char tmpfn[32];
   char rep_com[128];

   sprintf (tmpfn, "/tmp/rlist.%d", getpid ());
   f = fopen (tmpfn, "w");
   if (f == NULL) {
      printf ("Error creating report file\n");
      return;
   }
   i = 0;
   while (i != num_remailers) {
      touch_state ();
      update_stats (i);
      if (summaries[i].stats1.latency != 0 &&
	  !(summaries[i].flags & 1)) {
	 strncpy (nick, summaries[i].nickname, 16);
	 nick[16] = 0;
	 report_hist (i, hist);
	 rel = summaries[i].stats1.rel;
	 if (summaries[i].last_response != 0)
	    rel -= (now - summaries[i].last_response) * 1.157407e-07;
	 lat = summaries[i].stats1.latency;
	 p = latstr;
	 hr = lat / 3600;
	 if (hr < 100) *p++ = ' '; else *p++ = '0' + hr / 100;
	 if (hr < 10) *p++ = ' '; else *p++ = '0' + (hr / 10) % 10;
	 if (hr < 1) *p++ = ' '; else *p++ = '0' + hr % 10;
	 if (lat < 3600) *p++ = ' '; else *p++ = ':';
	 min = (lat % 3600) / 60;
	 if (lat < 600) *p++ = ' '; else *p++ = '0' + min / 10;
	 if (lat < 60) *p++ = ' '; else *p++ = '0' + min % 10;
	 *p++ = ':';
	 sec = lat % 60;
	 *p++ = '0' + sec / 10;
	 *p++ = '0' + sec % 10;
	 *p++ = 0;
	 if (nick[0] == '(')
	    fprintf (f, "%-16s %s%s %6.2f%%\n",
		     nick, hist, latstr, rel * 100);
	 else
	    fprintf (f, "%-8s <------------------------------> %s%s %6.2f%%\n",
		     nick, hist, latstr, rel * 100);
      }
      i++;
   }
   fclose (f);
   sprintf (rep_com, "%s < %s", report_com, tmpfn);
   if (system (rep_com)) 
      fprintf (stderr, "Error executing command: %s\n", rep_com);
   unlink (tmpfn);
}

#define DAY 86400
void report_hist (int i, char *hbuf) {
   /* Report the history of remailer i into buf (12 characters) + null */
   ping *buf;
   char hist[16];
   int j, k;
   int vnow;
   int elapsed, this_hist;
   char *sym = " #*+-._ ";
   int off, last;

   buf = get_pings (i);
   k = 0;
   while (k != 16) {
      hist[k] = 0;
      k++;
   }
   vnow = now - 3600; /* "virtual now" - embargo */
   j = 0;
   while (j != summaries[i].num_pings) {
      if (buf[j].ping_time && buf[j].ping_time < vnow) {
	 if (buf[j].response_time) {
	    elapsed = buf[j].response_time - buf[j].ping_time;
	    if (elapsed < 300) this_hist = 1;
	    else if (elapsed < 3600) this_hist = 2;
	    else if (elapsed < 14400) this_hist = 3;
	    else if (elapsed < 86400) this_hist = 4;
	    else if (elapsed < 172800) this_hist = 5;
	    else this_hist = 6;
	 } else
	    this_hist = 7;
	 k = 15 + buf[j].ping_time / DAY - vnow / DAY;
	 if (k >= 0 && this_hist > hist[k])
	    hist[k] = this_hist;
      }
      j++;
   }

   touch_state ();
   /* incorporate old history */
   off = vnow / DAY - summaries[i].last_update / DAY;
   last = 15 + buf[0].ping_time / DAY - vnow / DAY;
   k = 0;
   while (k <= last) {
      hist[k] = summaries[i].hist[k + off];
      k++;
   }
   k = 0;
   while (k != 16) {
      summaries[i].hist[k] = hist[k];
      k++;
   }
   summaries[i].last_update = vnow;
   free (buf);
   k = 0;
   while (k != 12) {
      hbuf[k] = sym[hist[k + 4]];
      k++;
   }
   hbuf[k] = 0;
}
