/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomrs.c
 * User interface aid: command handler for R through S
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrtrack.h"
#include "efunction.h"
#include "conlay.h"
#include "tecgen.h"

void us_redraw(INTSML count, char *par[])
{
	/* save highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* re-draw the status display */
	us_redostatus(NOWINDOWFRAME);

	/* redraw the color screen */
	us_drawmenu(0, NOWINDOWFRAME);

	/* restore highlighting */
	(void)us_pophighlight(0);
}

void us_rename(INTSML count, char *par[])
{
	char prompt[80], newfile[100],*newpar[10], *oldname, si[10], sj[10];
	REGISTER INTSML i, savei, k, command, savecommand, variable, savevariable, len;
	REGISTER INTBIG varnewkey, varoldkey;
	REGISTER char *ch, *pt, *netname, *savenetname, **newlist, *str, *which;
	REGISTER NODEPROTO *np, *lnt, *onp, *savenp, *curfacet;
	REGISTER NODEINST *ni;
	REGISTER CELL *c, *lc, *savec;
	REGISTER PORTPROTO *pp, *savepp;
	REGISTER LIBRARY *lib, *olib, *savelib;
	REGISTER TECHNOLOGY *tech, *otech, *savetech;
	REGISTER ARCPROTO *ap, *oat, *saveap;
	REGISTER VARIABLE *macvar, *savemacvar, *var;
	REGISTER POPUPMENU *opm;
	POPUPMENU *pm, *savepm;
	REGISTER NETWORK *net;
	HIGHLIGHT high;
	REGISTER USERCOM *item;
	COMMANDBINDING commandbinding;
	extern COMCOMP us_renameop, us_renamenp;
	extern AIDENTRY *net_aid;

	/* get the former name */
	if (count == 0)
	{
		count = ttygetparam("Old name: ", &us_renameop, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	pt = par[0];
	curfacet = getcurfacet();

	/* see if any primitive nodeprotos have that name */
	np = getnodeproto(pt);
	if (np != NONODEPROTO && np->primindex == 0) np = NONODEPROTO;

	/* see if any cells have that name */
	c = getcell(pt);

	/* see if any arcprotos have that name */
	ap = getarcproto(pt);

	/* see if any libraries have that name */
	lib = getlibrary(pt);

	/* see if any ports in this facet have that name */
	if (curfacet == NONODEPROTO) pp = NOPORTPROTO; else
		for(pp = curfacet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (namesame(pp->protoname, pt) == 0) break;
	}

	/* see if any macros have that name */
	macvar = us_getmacro(pt);

	/* see if any technologies have that name */
	tech = gettechnology(pt);

	/* see if any user commands have that name */
	for(i=0; us_lcommand[i].name != 0; i++)
		if (namesame(us_lcommand[i].name, pt) == 0) break;
	if (us_lcommand[i].name == 0) command = -1; else command = i;

	/* see if any database variables have that name */
	for(i=0; i<el_numnames; i++)
		if (namesame(el_namespace[i], pt) == 0) break;
	if (i >= el_numnames) variable = -1; else variable = i;

	/* see if any popup menus have that name */
	for(pm = us_firstpopupmenu; pm != NOPOPUPMENU; pm = pm->nextpopupmenu)
		if (namesame(pm->name, pt) == 0) break;

	/* see if any networks have that name */
	netname = 0;
	if (curfacet != NONODEPROTO)
	{
		len = strlen(pt);
		for(net = curfacet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		{
			ch = net->netname;
			for(k=0; k<net->namecount; k++)
			{
				if (namesamen(ch, pt, len) == 0 && (ch[len] == '[' || ch[len] == 0))
				{
					netname = pt;
					break;
				}
				ch += strlen(ch) + 1;
			}
			if (netname != 0) break;
		}
	}

	/* special case: if port and network are possible, exclude the network */
	if (pp != NOPORTPROTO && netname != 0) netname = 0;

	/* see how many different things have that name */
	i = 0;
	if (np != NONODEPROTO) i++;
	if (c != NOCELL) i++;
	if (ap != NOARCPROTO) i++;
	if (lib != NOLIBRARY) i++;
	if (pp != NOPORTPROTO) i++;
	if (macvar != NOVARIABLE) i++;
	if (tech != NOTECHNOLOGY) i++;
	if (command >= 0) i++;
	if (variable >= 0) i++;
	if (pm != NOPOPUPMENU) i++;
	if (netname != 0) i++;

	/* quit if the name doesn't exist */
	if (i == 0)
	{
		us_abortcommand("Nothing named %s", pt);
		return;
	}

	/* if name matches more than one type, exclude matches that are not exact */
	if (i != 1)
	{
		savenp = np;
		saveap = ap;
		savelib = lib;
		savepp = pp;
		savemacvar = macvar;
		savetech = tech;
		savecommand = command;
		savevariable = variable;
		savepm = pm;
		savec = c;
		savenetname = netname;
		savei = i;

		if (np != NONODEPROTO && namesame(np->primname, pt) != 0)
		{
			np = NONODEPROTO;   i--;
		}
		if (c != NOCELL && namesame(c->cellname, pt) != 0)
		{
			c = NOCELL;   i--;
		}
		if (ap != NOARCPROTO && namesame(ap->protoname, pt) != 0)
		{
			ap = NOARCPROTO;   i--;
		}
		if (lib != NOLIBRARY && namesame(lib->libname, pt) != 0)
		{
			lib = NOLIBRARY;   i--;
		}
		if (pp != NOPORTPROTO && namesame(pp->protoname, pt) != 0)
		{
			pp = NOPORTPROTO;   i--;
		}
		if (macvar != NOVARIABLE && namesame(&makename(macvar->key)[11], pt) != 0)
		{
			macvar = NOVARIABLE;   i--;
		}
		if (tech != NOTECHNOLOGY && namesame(tech->techname, pt) != 0)
		{
			tech = NOTECHNOLOGY;   i--;
		}
		if (command >= 0 && namesame(us_lcommand[command].name, pt) != 0)
		{
			command = -1;   i--;
		}
		if (variable >= 0 && namesame(el_namespace[variable], pt) != 0)
		{
			variable = -1;   i--;
		}
		if (pm != NOPOPUPMENU && namesame(pm->name, pt) != 0)
		{
			pm = NOPOPUPMENU;   i--;
		}
		if (netname != 0 && namesame(netname, pt) != 0)
		{
			netname = 0;   i--;
		}

		if (i <= 0)
		{
			np = savenp;
			ap = saveap;
			lib = savelib;
			pp = savepp;
			macvar = savemacvar;
			tech = savetech;
			command = savecommand;
			variable = savevariable;
			pm = savepm;
			c = savec;
			netname = savenetname;
			i = savei;
		}
	}

	/* build the ambiguity string */
	(void)strcpy(prompt, "Rename the");
	i = 0;
	if (np != NONODEPROTO)
	{
		(void)strcat(prompt, " Primitive"); i++;
	}
	if (c != NOCELL)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Cell"); i++;
	}
	if (ap != NOARCPROTO)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Arc"); i++;
	}
	if (lib != NOLIBRARY)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Library"); i++;
	}
	if (pp != NOPORTPROTO)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " poRt"); i++;
	}
	if (macvar != NOVARIABLE)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Macro"); i++;
	}
	if (tech != NOTECHNOLOGY)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Technology"); i++;
	}
	if (command >= 0)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " commanD"); i++;
	}
	if (variable >= 0)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Variable"); i++;
	}
	if (pm != NOPOPUPMENU)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " pop-Up-menu"); i++;
	}
	if (netname != 0)
	{
		if (i) (void)strcat(prompt, " or");
		(void)strcat(prompt, " Network"); i++;
	}
	(void)strcat(prompt, ": ");

	/* if name is more than one type of object, ask which */
	while (i != 1)
	{
		if (count >= 3)
		{
			which = par[2];
			count = 2;
		} else
		{
			which = ttygetline(prompt);
			if (which == 0) return;
		}
		switch (*which)
		{
			case 'p':   case 'P':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				macvar=NOVARIABLE; tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'c':   case 'C':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				macvar=NOVARIABLE; tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  np=NONODEPROTO;
				netname = 0;       i=1;               break;
			case 'a':   case 'A':
				np=NONODEPROTO;    lib=NOLIBRARY;     pp=NOPORTPROTO;
				macvar=NOVARIABLE; tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'l':   case 'L':
				ap=NOARCPROTO;     np=NONODEPROTO;    pp=NOPORTPROTO;
				macvar=NOVARIABLE; tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'r':   case 'R':
				ap=NOARCPROTO;     lib=NOLIBRARY;     np=NONODEPROTO;
				macvar=NOVARIABLE; tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'm':   case 'M':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    tech=NOTECHNOLOGY; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 't':   case 'T':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    macvar=NOVARIABLE; command = -1;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'd':   case 'D':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    tech=NOTECHNOLOGY; macvar=NOVARIABLE;
				variable = -1;     pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'v':   case 'V':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    tech=NOTECHNOLOGY; macvar=NOVARIABLE;
				command = -1;      pm = NOPOPUPMENU;  c = NOCELL;
				netname = 0;       i=1;               break;
			case 'u':   case 'U':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    tech=NOTECHNOLOGY; macvar=NOVARIABLE;
				variable = -1;     command = -1;      c = NOCELL;
				netname = 0;       i=1;               break;
			case 'n':   case 'N':
				ap=NOARCPROTO;     lib=NOLIBRARY;     pp=NOPORTPROTO;
				np=NONODEPROTO;    tech=NOTECHNOLOGY; macvar=NOVARIABLE;
				variable = -1;     command = -1;      c = NOCELL;
				pm=NOPOPUPMENU;    i=1;               break;
			case 0:   us_abortedmsg();  return;
		}
	}

	/* get new name */
	if (count <= 1)
	{
		count = ttygetparam("New name: ", &us_renamenp, MAXPARS-1, &par[1]) + 1;
		if (count == 1)
		{
			us_abortedmsg();
			return;
		}
	}
	pt = par[1];
	for(ch = pt; *ch != 0; ch++) if (*ch == ' ' || *ch == '\t')
	{
		us_abortcommand("Name cannot have embedded spaces or tabs");
		return;
	}

	/* handle primitive nodeproto name change */
	if (np != NONODEPROTO)
	{
		/* check for duplicate name */
		if (strcmp(np->primname, pt) == 0)
		{
			ttyputmsg("Primitive name has not changed");
			return;
		}
		for(lnt = np->tech->firstnodeproto; lnt != NONODEPROTO; lnt = lnt->nextnodeproto)
			if (np != lnt && namesame(lnt->primname, pt) == 0)
		{
			us_abortcommand("Already a primitive of that name");
			return;
		}

		/* change the node name */
		ttyputverbose("Primitive %s renamed to %s", np->primname, pt);
		lnt = us_curnodeproto;
		if (lnt == np) us_setnodeproto(NONODEPROTO);
		(void)setval((INTBIG)np, VNODEPROTO, "primname", (INTBIG)pt, VSTRING);
		if (lnt == np) us_setnodeproto(np);
	}

	/* handle cell name name change */
	if (c != NOCELL)
	{
		/* check for duplicate name */
		if (strcmp(c->cellname, pt) == 0)
		{
			ttyputmsg("Cell name has not changed");
			return;
		}
		for(lc = c->lib->firstcell; lc != NOCELL; lc = lc->nextcell)
			if (c != lc && namesame(lc->cellname, pt) == 0)
		{
			us_abortcommand("Already a cell of that name");
			return;
		}

		/*
		 * because highlighting is stored as strings with the cell name
		 * spelled out, it is necessary to clear all highlighting if any
		 * of it refers to the renamed cell.
		 */
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
		if (var != NOVARIABLE)
		{
			len = (INTSML)getlength(var);
			for(i=0; i<len; i++)
			{
				if (us_makehighlight(((char **)var->addr)[i], &high) != 0) continue;
				if (high.facet->cell == c)
				{
					us_clearhighlightcount();
					break;
				}
			}
		}

		/* change the node name */
		ttyputverbose("Cell %s renamed to %s", c->cellname, pt);
		lnt = us_curnodeproto;
		if (lnt != NONODEPROTO && lnt->cell == c) us_setnodeproto(NONODEPROTO);

		/* find all instances of this cell */
		for(np = c->firstincell; np != NONODEPROTO; np = np->nextincell)
			for(onp = np; onp != NONODEPROTO; onp = onp->lastversion)
				for(ni = onp->firstinst; ni != NONODEINST; ni = ni->nextinst)
					if ((ni->userbits&NEXPAND) == 0)
						startobjectchange((INTBIG)ni, VNODEINST);

		(void)setval((INTBIG)c, VCELL, "cellname", (INTBIG)pt, VSTRING);
		if (lnt != NONODEPROTO && lnt->cell == c) us_setnodeproto(lnt);

		/* find all instances of this cell */
		for(np = c->firstincell; np != NONODEPROTO; np = np->nextincell)
			for(onp = np; onp != NONODEPROTO; onp = onp->lastversion)
				for(ni = onp->firstinst; ni != NONODEINST; ni = ni->nextinst)
					if ((ni->userbits&NEXPAND) == 0)
						endobjectchange((INTBIG)ni, VNODEINST);
	}

	/* handle arcproto name change */
	if (ap != NOARCPROTO)
	{
		/* check for duplicate name */
		if (strcmp(ap->protoname, pt) == 0)
		{
			ttyputmsg("Arc name has not changed");
			return;
		}
		for(oat = ap->tech->firstarcproto; oat != NOARCPROTO; oat = oat->nextarcproto)
			if (ap != oat && namesame(pt, oat->protoname) == 0)
		{
			us_abortcommand("Already an arc of that name");
			return;
		}

		/* change the arc name */
		allocstring(&oldname, ap->protoname, el_tempcluster);
		ttyputverbose("Arc prototype %s renamed to %s", describearcproto(ap), pt);
		oat = us_curarcproto;
		if (oat == ap) us_setarcproto(NOARCPROTO, 1);
		(void)setval((INTBIG)ap, VARCPROTO, "protoname", (INTBIG)pt, VSTRING);
		if (oat == ap) us_setarcproto(ap, 1);

		/* change any component menu entries that mention this arc */
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_menu);
		if (var != NOVARIABLE)
		{
			for(i=0; i<us_menuy*us_menux; i++)
			{
				us_parsebinding(((char **)var->addr)[i], &commandbinding);
				item = us_makecommand(commandbinding.command);
				if (namesame(item->comname, "getproto") == 0)
				{
					if (item->count >= 2 && namesame(item->word[0], "arc") == 0)
					{
						if (namesame(item->word[1], oldname) == 0)
						{
							/* rename this item */
							if (us_menupos <= 1)
							{
								(void)sprintf(si, "%d", i%us_menux);
								(void)sprintf(sj, "%d", i/us_menux);
							} else
							{
								(void)sprintf(si, "%d", i/us_menuy);
								(void)sprintf(sj, "%d", i%us_menuy);
							}
							newpar[0] = "set";          newpar[1] = "menu";
							newpar[2] = "background";   newpar[3] = "red";
							newpar[4] = sj;             newpar[5] = si;
							newpar[6] = "getproto";     newpar[7] = "arc";
							newpar[8] = describearcproto(ap);
							us_bind(9, newpar);
						}
					}
				}
				us_freeusercom(item);
				us_freebindingparse(&commandbinding);
			}
		}
		efree(oldname);
	}

	/* handle library name change */
	if (lib != NOLIBRARY)
	{
		/* get pure library name if path was given */
		ch = skippath(pt);

		/* remove any ".elib" extension */
		k = 0;
		for(str = ch; *str != 0; str++)
			if (namesame(str, ".elib") == 0)
		{
			k = 1;
			*str = 0;
			break;
		}

		/* check for duplicate name */
		if (strcmp(lib->libname, ch) == 0 && ch == pt)
		{
			ttyputverbose("Library name has not changed");
			return;
		}
		for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
			if (olib != lib && namesame(olib->libname, ch) == 0)
		{
			us_abortcommand("Already a library of that name");
			return;
		}

		/* change the library name */
		ttyputverbose("Library %s renamed to %s", lib->libname, ch);
		(void)setval((INTBIG)lib, VLIBRARY, "libname", (INTBIG)ch, VSTRING);

		/* change the library file name too */
		if (k != 0) *str = '.';
		if (ch == pt)
		{
			/* no path given: use old path */
			(void)strcpy(newfile, lib->libfile);
			ch = skippath(newfile);
			(void)strcpy(ch, pt);
			pt = newfile;
		}
		(void)setval((INTBIG)lib, VLIBRARY, "libfile", (INTBIG)pt, VSTRING);
	}

	/* handle port name change */
	if (pp != NOPORTPROTO)
	{
		/* rename the port */
		us_renameport(pp, pt);
	}

	/* handle macro change */
	if (macvar != NOVARIABLE)
	{
		/* check for duplicate name */
		if (namesame(&makename(macvar->key)[11], pt) == 0)
		{
			ttyputmsg("Macro name has not changed");
			return;
		}
		var = us_getmacro(pt);
		if (var != macvar)
		{
			us_abortcommand("Already a macro of that name");
			return;
		}

		/* make sure macro name isn't overloading existing command or popup */
		for(i=0; us_lcommand[i].name != 0; i++)
			if (namesame(pt, us_lcommand[i].name) == 0)
		{
			us_abortcommand("There is a command with that name");
			return;
		}
		for(opm=us_firstpopupmenu; opm!=NOPOPUPMENU; opm=opm->nextpopupmenu)
			if (namesame(pt, opm->name) == 0)
		{
			us_abortcommand("There is a popup menu with that name");
			return;
		}

		/* save the macro data */
		len = (INTSML)getlength(macvar);
		newlist = (char **)emalloc(len * (sizeof (char *)), el_tempcluster);
		if (newlist == 0) return;
		for(i=0; i<len; i++)
			(void)allocstring(&newlist[i], ((char **)var->addr)[i], el_tempcluster);

		/* change the macro name */
		ttyputverbose("Macro %s renamed to %s", &makename(macvar->key)[11], pt);
		(void)delvalkey((INTBIG)us_aid, VAID, (INTBIG)macvar->key);
		(void)initinfstr();
		(void)addstringtoinfstr("USER_macro_");
		(void)addstringtoinfstr(pt);
		(void)setval((INTBIG)us_aid, VAID, returninfstr(), (INTBIG)newlist,
			VSTRING|VISARRAY|(len<<VLENGTHSH)|VDONTSAVE);
		for(i=0; i<len; i++) efree(newlist[i]);
		efree((char *)newlist);
	}

	/* handle technology change */
	if (tech != NOTECHNOLOGY)
	{
		/* check for duplicate name */
		if (strcmp(tech->techname, pt) == 0)
		{
			ttyputmsg("Technology name has not changed");
			return;
		}
		for(otech = el_technologies; otech != NOTECHNOLOGY; otech = otech->nexttechnology)
			if (otech != tech && namesame(otech->techname, pt) == 0)
		{
			us_abortcommand("Already a technology of that name");
			return;
		}

		/* change the technology name */
		ttyputverbose("Technology %s renamed to %s", tech->techname, pt);
		(void)setval((INTBIG)tech, VTECHNOLOGY, "techname", (INTBIG)pt, VSTRING);
	}

	/* handle user command change */
	if (command >= 0)
	{
		/* check for duplicate name */
		if (strcmp(us_lcommand[command].name, pt) == 0)
		{
			ttyputmsg("Command name has not changed");
			return;
		}
		for(i=0; us_lcommand[i].name != 0; i++)
			if (i != command && namesame(pt, us_lcommand[i].name) == 0)
		{
			us_abortcommand("Already a command of that name");
			return;
		}

		/* make sure command name isn't overloading existing macro or popup */
		if (us_getmacro(pt) != NOVARIABLE)
		{
			us_abortcommand("There is a macro with that name");
			return;
		}
		for(opm=us_firstpopupmenu; opm!=NOPOPUPMENU; opm=opm->nextpopupmenu)
			if (namesame(pt, opm->name) == 0)
		{
			us_abortcommand("There is a popup menu with that name");
			return;
		}

		/*
		 * change the command name
		 * Note: this allocates space that is never freed !!!
		 */
		ttyputverbose("Command %s renamed to %s", us_lcommand[command].name, pt);
		if (allocstring(&us_lcommand[command].name, pt, us_aid->cluster) != 0)
			ttyputerr("Out of memory!");
	}

	/* handle variable change */
	if (variable >= 0)
	{
		/* check for duplicate name */
		if (namesame(el_namespace[variable], pt) == 0)
		{
			ttyputmsg("Variable name has not changed");
			return;
		}
		for(i=0; i<el_numnames; i++)
			if (namesame(pt, el_namespace[i]) == 0)
		{
			us_abortcommand("Already a variable of that name");
			return;
		}

		/* change the variable name */
		ttyputverbose("Variable %s renamed to %s", el_namespace[variable], pt);
		(void)renameval(el_namespace[variable], pt);
	}

	/* handle popup menu change */
	if (pm != NOPOPUPMENU)
	{
		/* check for duplicate name */
		if (namesame(pm->name, pt) == 0)
		{
			ttyputmsg("Popup menu name has not changed");
			return;
		}
		for(opm=us_firstpopupmenu; opm!=NOPOPUPMENU; opm=opm->nextpopupmenu)
			if (namesame(pt, opm->name) == 0)
		{
			us_abortcommand("Already a popup menu with that name");
			return;
		}

		/* make sure popup name isn't overloading existing command or menu */
		for(i=0; us_lcommand[i].name != 0; i++)
			if (namesame(pt, us_lcommand[i].name) == 0)
		{
			us_abortcommand("There is a command with that name");
			return;
		}
		if (us_getmacro(pt) != NOVARIABLE)
		{
			us_abortcommand("There is a macro with that name");
			return;
		}

		/* change the popup name */
		ttyputverbose("Popup menu %s renamed to %s", pm->name, pt);

		/* find the old popup menu */
		(void)initinfstr();
		(void)addstringtoinfstr("USER_binding_popup_");
		(void)addstringtoinfstr(pm->name);
		varoldkey = makekey(returninfstr());
		var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, varoldkey);
		if (var == NOVARIABLE)
		{
			us_abortcommand("Cannot find popup menu %s", pm->name);
			return;
		}
		len = (INTSML)getlength(var);

		/* create the new popup menu with the new name and old data */
		(void)initinfstr();
		(void)addstringtoinfstr("USER_binding_popup_");
		(void)addstringtoinfstr(pt);
		varnewkey = makekey(returninfstr());
		(void)setvalkey((INTBIG)us_aid, VAID, varnewkey, (INTBIG)var->addr,
			VSTRING|VISARRAY|VDONTSAVE|(len<<VLENGTHSH));

		/* now delete the former popup menu */
		(void)delvalkey((INTBIG)us_aid, VAID, varoldkey);
	}

	/* handle network name change */
	if (netname != 0)
	{
		(void)askaid(net_aid, "rename", (INTBIG)netname, (INTBIG)pt, (INTBIG)net->parent);
		return;
	}
}

void us_replace(INTSML count, char *par[])
{
	REGISTER INTSML universal, thisfacet, connected, total, len;
	REGISTER char *pt;
	REGISTER NODEPROTO *np, *oldntype, *facet, *curfacet;
	REGISTER ARCPROTO *ap, *oldatype;
	REGISTER NODEINST *ni, *newno, *lni, *onlynewno;
	REGISTER ARCINST *ai, *newar, *lai, *onlynewar;
	REGISTER PORTARCINST *pi, *opi;
	REGISTER HIGHLIGHT *high;
	REGISTER LIBRARY *lib;
	extern COMCOMP us_replacep;

	/* find highlighted object to be replaced */
	high = us_getonehighlight();
	if (high == NOHIGHLIGHT) return;
	if ((high->status&HIGHTYPE) != HIGHFROM)
	{
		us_abortcommand("Find an object to be replaced");
		return;
	}
	curfacet = us_needfacet();
	if (curfacet == NONODEPROTO) return;

	/* handle node replacement */
	if (high->fromgeom->entrytype == OBJNODEINST)
	{
		/* get node to be replaced */
		ni = high->fromgeom->entryaddr.ni;

		/* disallow replacing if lock is on */
		if (us_canedit(ni->parent, ni->proto, 1) != 0) return;

		/* get nodeproto to replace it with */
		if (count == 0)
		{
			count = ttygetparam("Node name: ", &us_replacep, MAXPARS, par);
			if (count == 0)
			{
				us_abortedmsg();
				return;
			}
		}
		np = getnodeproto(par[0]);
		if (np == NONODEPROTO)
		{
			us_abortcommand("Nothing called '%s'", par[0]);
			return;
		}

		/* sanity check */
		oldntype = ni->proto;
		if (oldntype == np)
		{
			us_abortcommand("Node already of type %s", describenodeproto(np));
			return;
		}

		/* get any arguments to the replace */
		connected = thisfacet = universal = 0;
		if (count > 1)
		{
			len = strlen(pt = par[1]);
			if (namesamen(pt, "connected", len) == 0) connected = 1;
			if (namesamen(pt, "this-facet", len) == 0) thisfacet = 1;
			if (namesamen(pt, "universally", len) == 0) universal = 1;
		}

		/* clear highlighting */
		us_clearhighlightcount();

		/* replace the nodeinst */
		onlynewno = us_replacenodeinst(ni, np);
		if (onlynewno == NONODEINST)
		{
			us_abortcommand("%s does not fit in the place of %s", describenodeproto(np),
				describenodeproto(oldntype));
			(void)us_addhighlight(high);
			return;
		}

		/* do additional replacements if requested */
		total = 1;
		if (universal != 0)
		{
			/* replace in all facets of library if "universally" used */
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
				for(facet = lib->firstnodeproto; facet != NONODEPROTO;
					facet = facet->nextnodeproto)
			{
				for(lni = facet->firstnodeinst; lni != NONODEINST; lni = lni->nextnodeinst)
					if (lni->proto == oldntype)
				{
					/* disallow replacing if lock is on */
					if (us_canedit(facet, lni->proto, 1) != 0) continue;

					newno = us_replacenodeinst(lni, np);
					if (newno != NONODEINST) total++;
					if (stopping("Replace") != 0) break;
				}
			}
			ttyputmsg("All %d %s nodes in the library replaced with %s", total,
				describenodeproto(oldntype), describenodeproto(np));
		} else if (thisfacet != 0)
		{
			/* replace throughout this facet if "this-facet" used */
			for(lni = curfacet->firstnodeinst; lni != NONODEINST; lni = lni->nextnodeinst)
				if (lni->proto == oldntype)
			{
				/* disallow replacing if lock is on */
				if (us_canedit(curfacet, lni->proto, 1) != 0) continue;

				newno = us_replacenodeinst(lni, np);
				if (newno != NONODEINST) total++;
				if (stopping("Replace") != 0) break;
			}
			ttyputmsg("All %d %s nodes in facet %s replaced with %s", total,
				describenodeproto(oldntype), describenodeproto(ni->parent), describenodeproto(np));
		} else if (connected != 0)
		{
			/* replace all connected to this in the facet if "connected" used */
			for(lni = curfacet->firstnodeinst; lni != NONODEINST; lni = lni->nextnodeinst)
				if (lni->proto == oldntype)
			{
				for(pi = lni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					for(opi = onlynewno->firstportarcinst; opi != NOPORTARCINST; opi = opi->nextportarcinst)
					{
						if (pi->conarcinst->network == opi->conarcinst->network) break;
					}
					if (opi != NOPORTARCINST) break;
				}
				if (pi == NOPORTARCINST) continue;

				/* disallow replacing if lock is on */
				if (us_canedit(curfacet, lni->proto, 1) != 0) continue;

				newno = us_replacenodeinst(lni, np);
				if (newno != NONODEINST) total++;
				if (stopping("Replace") != 0) break;
			}
			ttyputmsg("All %d %s nodes connected to this replaced with %s", total,
				describenodeproto(oldntype), describenodeproto(np));
		} else ttyputmsg("Node %s replaced with %s", describenodeproto(oldntype),
			describenodeproto(np));

		/* clean up */
		us_setnodeproto(np);
		high->fromgeom = onlynewno->geom;
		high->fromport = NOPORTPROTO;
		high->frompoint = 0;
		(void)us_addhighlight(high);
	} else
	{
		/* get arc to be replaced */
		ai = high->fromgeom->entryaddr.ai;

		/* disallow replacement if lock is on */
		if (us_canedit(ai->parent, NONODEPROTO, 1) != 0) return;

		/* get arcproto to replace it with */
		if (count == 0)
		{
			count = ttygetparam("Arc name: ", &us_replacep, MAXPARS, par);
			if (count == 0)
			{
				us_abortedmsg();
				return;
			}
		}
		ap = getarcproto(par[0]);
		if (ap == NOARCPROTO)
		{
			us_abortcommand("Nothing called '%s'", par[0]);
			return;
		}

		/* sanity check */
		oldatype = ai->proto;
		if (oldatype == ap)
		{
			us_abortcommand("Arc already of type %s", describearcproto(ap));
			return;
		}

		/* get any arguments to the replace */
		connected = thisfacet = universal = 0;
		if (count > 1)
		{
			len = strlen(pt = par[1]);
			if (namesamen(pt, "connected", len) == 0) connected = 1;
			if (namesamen(pt, "this-facet", len) == 0) thisfacet = 1;
			if (namesamen(pt, "universally", len) == 0) universal = 1;
		}

		/* remove highlighting */
		us_clearhighlightcount();

		/* replace the arcinst */
		startobjectchange((INTBIG)ai, VARCINST);
		onlynewar = replacearcinst(ai, ap);
		if (onlynewar == NOARCINST)
		{
			us_abortcommand("%s does not fit in the place of %s", describearcproto(ap),
				describearcproto(oldatype));
			(void)us_addhighlight(high);
			return;
		}
		endobjectchange((INTBIG)onlynewar, VARCINST);

		/* do additional replacements if requested */
		total = 1;
		if (universal != 0)
		{
			/* replace in all facets of library if "universally" used */
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
				for(facet = lib->firstnodeproto; facet != NONODEPROTO;
					facet = facet->nextnodeproto)
			{
				for(lai = facet->firstarcinst; lai != NOARCINST; lai = lai->nextarcinst)
					if (lai->proto == oldatype)
				{
					/* disallow replacing if lock is on */
					if (us_canedit(facet, NONODEPROTO, 1) != 0) continue;

					startobjectchange((INTBIG)lai, VARCINST);
					newar = replacearcinst(lai, ap);
					if (newar != NOARCINST)
					{
						total++;
						endobjectchange((INTBIG)newar, VARCINST);
					}
					if (stopping("Replace") != 0) break;
				}
			}
			ttyputmsg("All %d %s arcs in the library replaced with %s", total,
				describearcproto(oldatype), describearcproto(ap));
		} else if (thisfacet != 0)
		{
			/* replace throughout this facet if "this-facet" used */
			for(lai = curfacet->firstarcinst; lai != NOARCINST; lai = lai->nextarcinst)
				if (lai->proto == oldatype)
			{
				startobjectchange((INTBIG)lai, VARCINST);
				newar = replacearcinst(lai, ap);
				if (newar != NOARCINST)
				{
					total++;
					endobjectchange((INTBIG)newar, VARCINST);
				}
				if (stopping("Replace") != 0) break;
			}
			ttyputmsg("All %d %s arcs in facet %s replaced with %s", total,
				describearcproto(oldatype), describenodeproto(ai->parent), describearcproto(ap));
		} else if (connected != 0)
		{
			/* replace all connected to this if "connected" used */
			for(lai = curfacet->firstarcinst; lai != NOARCINST; lai = lai->nextarcinst)
				if (lai->proto == oldatype)
			{
				if (lai->network != onlynewar->network) continue;
				startobjectchange((INTBIG)lai, VARCINST);
				newar = replacearcinst(lai, ap);
				if (newar != NOARCINST)
				{
					total++;
					endobjectchange((INTBIG)newar, VARCINST);
				}
				if (stopping("Replace") != 0) break;
			}
			ttyputmsg("All %d %s arcs connected to this replaced with %s", total,
				describearcproto(oldatype), describearcproto(ap));
		} else ttyputmsg("Arc %s replaced with %s", describearcproto(oldatype),
			describearcproto(ap));

		/* clean up */
		us_setarcproto(ap, 1);
		high->fromgeom = onlynewar->geom;
		(void)us_addhighlight(high);
	}
}

void us_rotate(INTSML count, char *par[])
{
	REGISTER NODEINST *ni, *theni;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp, *thepp;
	REGISTER ARCINST *ai, **ailist, **newailist;
	REGISTER GEOM **list;
	REGISTER VARIABLE *var;
	REGISTER HIGHLIGHT *high;
	REGISTER INTSML amt, startangle, endangle, rotatemore;
	INTBIG xstart, ystart, xend, yend, gx, gy, cx, cy, rotcx, rotcy, aicount,
		x, y, thex, they;
	REGISTER INTBIG lx, hx, ly, hy, nicount, dist, bestdist, i, j;
	XARRAY transtz, rot, transfz, t1, t2;

	/* handle interactive rotation */
	if (count == 1 && namesamen(par[0], "interactively", strlen(par[0])) == 0)
	{
		ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
		if (ni == NONODEINST)
		{
			us_abortcommand("Must highlight one node for interactive rotation");
			return;
		}

		/* disallow rotating if lock is on */
		if (us_canedit(ni->parent, ni->proto, 1) != 0) return;

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		if (us_demandxy(&xstart, &ystart) != 0) return;
		us_rotateinit(ni);
		trackcursor(0, us_ignoreup, us_rotatebegin, us_rotatedown,
			us_stopandpoponchar, us_dragup, TRACKDRAGGING);
		if (el_pleasestop != 0) return;
		if (us_demandxy(&xend, &yend) != 0) return;
		startangle = figureangle((ni->lowx+ni->highx)/2, (ni->lowy+ni->highy)/2, xstart, ystart);
		endangle = figureangle((ni->lowx+ni->highx)/2, (ni->lowy+ni->highy)/2, xend, yend);
		if (startangle == endangle)
		{
			ttyputverbose("Null node rotation");
			(void)us_pophighlight(0);
			return;
		}
		if (ni->transpose == 0) amt = endangle - startangle; else
			amt = startangle - endangle;
		while (amt < 0) amt += 3600;
		while (amt > 3600) amt -= 3600;

		/* do the rotation */
		startobjectchange((INTBIG)ni, VNODEINST);
		modifynodeinst(ni, 0, 0, 0, 0, amt, 0);
		endobjectchange((INTBIG)ni, VNODEINST);

		/* restore highlighting */
		(void)us_pophighlight(1);
		return;
	}

	/* determine rotation amount */
	if (count < 1)
	{
		us_abortcommand("Usage: rotate ANGLE");
		return;
	}
	amt = (INTSML)atofr(par[0]);
	amt = amt * 10 / WHOLE;
	rotatemore = 0;
	if (count >= 2 && namesamen(par[1], "more", strlen(par[1])) == 0)
	{
		count--;
		par++;
		rotatemore++;
	}

	/* get all highlighted objects for rotation */
	list = us_gethighlighted(OBJNODEINST|OBJARCINST);
	if (list[0] == NOGEOM)
	{
		us_abortcommand("Must highlight node(s) to be rotated");
		return;
	}
	np = geomparent(list[0]);

	/* disallow rotating if lock is on */
	if (us_canedit(np, NONODEPROTO, 1) != 0) return;

	/* figure out which nodes get rotated */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	nicount = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype != OBJNODEINST) continue;
		ni = list[i]->entryaddr.ni;
		if (us_canedit(np, ni->proto, 1) != 0) return;
		ni->temp1 = 1;
		if (nicount == 0)
		{
			lx = ni->lowx;   hx = ni->highx;
			ly = ni->lowy;   hy = ni->highy;
		} else
		{
			if (ni->lowx < lx) lx = ni->lowx;
			if (ni->highx > hx) hx = ni->highx;
			if (ni->lowy < ly) ly = ni->lowy;
			if (ni->highy > hy) hy = ni->highy;
		}
		theni = ni;
		nicount++;
	}

	/* must be at least 1 node */
	if (nicount <= 0)
	{
		us_abortcommand("Must select at least 1 node for rotation");
		return;
	}

	/* if multiple nodes, find the center one */
	if (nicount > 1)
	{
		theni = NONODEINST;
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni->temp1 == 0) continue;
			dist = computedistance((lx+hx)/2, (ly+hy)/2, (ni->lowx+ni->highx)/2,
				(ni->lowy+ni->highy)/2);
			if (theni == NONODEINST || dist < bestdist)
			{
				theni = ni;
				bestdist = dist;
			}
		}
	}

	/* compute rotation, given the node */
	if (rotatemore != 0)
	{
		if (theni->transpose != 0) amt = -amt;
		if (amt < 0) amt += 3600;
	} else
	{
		amt = (amt - theni->rotation) % 3600;
		if (theni->transpose != 0) amt = -amt;
		if (amt < 0) amt += 3600;
	}
	if (amt == 0)
	{
		ttyputverbose("Null rotation");
		return;
	}

	/* handle rotation about the grab point */
	i = strlen(par[1]);
	if (count >= 2 && namesamen(par[1], "sensibly", i) == 0)
	{
		if (nicount == 1)
		{
			if (theni->proto->primindex == 0)
			{
				var = getvalkey((INTBIG)theni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
				if (var != NOVARIABLE)
				{
					par[1] = "about-grab-point";
				}
			}
		}
	}
	if (count >= 2 && namesamen(par[1], "about-grab-point", i) == 0 && i >= 7)
	{
		if (nicount > 1)
		{
			us_abortcommand("Must highlight one node for rotation about the grab-point");
			return;
		}
		ni = theni;

		/* disallow rotating if lock is on */
		if (us_canedit(ni->parent, ni->proto, 1) != 0) return;

		/* find the grab point */
		corneroffset(ni, ni->proto, ni->rotation, ni->transpose, &gx, &gy, 0);
		gx += ni->lowx;   gy += ni->lowy;

		/* build transformation for this operation */
		transid(transtz);   transtz[2][0] = -gx;   transtz[2][1] = -gy;
		makeangle(amt, 0, rot);
		transid(transfz);   transfz[2][0] = gx;    transfz[2][1] = gy;
		transmult(transtz, rot, t1);
		transmult(t1, transfz, t2);
		cx = (ni->lowx+ni->highx)/2;   cy = (ni->lowy+ni->highy)/2;
		xform(cx, cy, &gx, &gy, t2);
		gx -= cx;   gy -= cy;

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* do the rotation */
		startobjectchange((INTBIG)ni, VNODEINST);

		/* rotate and translate */
		modifynodeinst(ni, gx, gy, gx, gy, amt, 0);

		/* end change */
		endobjectchange((INTBIG)ni, VNODEINST);

		/* restore highlighting */
		(void)us_pophighlight(1);
		return;
	}

	/* handle rotation about a trace point */
	if (count >= 2 && namesamen(par[1], "about-trace-point", i) == 0 && i >= 7)
	{
		if (nicount > 1)
		{
			us_abortcommand("Must highlight one node for rotation about an outline point");
			return;
		}
		ni = theni;

		/* disallow rotating if lock is on */
		if (us_canedit(ni->parent, ni->proto, 1) != 0) return;

		/* get the trace information */
		var = gettrace(ni);
		if (var == NOVARIABLE)
		{
			us_abortcommand("Highlighted node must have outline information");
			return;
		}

		/* find the pivot point */
		high = us_getonehighlight();
		i = high->frompoint;   if (i != 0) i--;
		makerot(ni, t1);
		gx = (ni->highx + ni->lowx) / 2;
		gy = (ni->highy + ni->lowy) / 2;
		xform(((INTBIG *)var->addr)[i*2]+gx, ((INTBIG *)var->addr)[i*2+1]+gy, &gx, &gy, t1);

		/* build transformation for this operation */
		transid(transtz);   transtz[2][0] = -gx;   transtz[2][1] = -gy;
		makeangle(amt, 0, rot);
		transid(transfz);   transfz[2][0] = gx;    transfz[2][1] = gy;
		transmult(transtz, rot, t1);
		transmult(t1, transfz, t2);
		cx = (ni->lowx+ni->highx)/2;   cy = (ni->lowy+ni->highy)/2;
		xform(cx, cy, &gx, &gy, t2);
		gx -= cx;   gy -= cy;

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* do the rotation */
		startobjectchange((INTBIG)ni, VNODEINST);

		/* rotate and translate */
		modifynodeinst(ni, gx, gy, gx, gy, amt, 0);

		/* end change */
		endobjectchange((INTBIG)ni, VNODEINST);

		/* restore highlighting */
		(void)us_pophighlight(1);
		return;
	}

	/* see which nodes already connect to the main rotation node (theni) */
	for(ni = theni->parent->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	theni->temp1 = 1;
	for(ai = theni->parent->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		ai->temp1 = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype != OBJARCINST) continue;
		ai = list[i]->entryaddr.ai;
		ai->temp1 = 1;
	}
	us_spreadrotateconnection(theni);

	/* now make sure that it is all connected */
	aicount = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype != OBJNODEINST) continue;
		ni = list[i]->entryaddr.ni;
		if (ni == theni) continue;
		if (ni->temp1 != 0) continue;

		thepp = theni->proto->firstportproto;
		pp = ni->proto->firstportproto;
		portposition(theni, thepp, &thex, &they);
		portposition(ni, pp, &x, &y);
		ai = newarcinst(gen_invisiblearc, 0, FIXED, theni, thepp, thex, they,
			ni, pp, x, y, np);
		if (ai == NOARCINST) break;
		endobjectchange((INTBIG)ai, VARCINST);
		newailist = (ARCINST **)emalloc((aicount+1) * (sizeof (ARCINST *)), el_tempcluster);
		if (newailist == 0) break;
		for(j=0; j<aicount; j++) newailist[j] = ailist[j];
		if (aicount > 0) efree((char *)ailist);
		ailist = newailist;
		ailist[aicount] = ai;
		aicount++;
	}

	/* make all selected arcs temporarily rigid */
	us_modarcbits(6, 0, "", list);

	/* save highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* see if there is a snap point */
	if (us_getonesnappoint(&rotcx, &rotcy) == 0)
	{
		/* no snap point, use center of node */
		rotcx = (theni->lowx + theni->highx) / 2;
		rotcy = (theni->lowy + theni->highy) / 2;
	}

	/* build transformation for this operation */
	transid(transtz);   transtz[2][0] = -rotcx;   transtz[2][1] = -rotcy;
	makeangle(amt, 0, rot);
	transid(transfz);   transfz[2][0] = rotcx;    transfz[2][1] = rotcy;
	transmult(transtz, rot, t1);
	transmult(t1, transfz, t2);
	cx = (theni->lowx+theni->highx)/2;   cy = (theni->lowy+theni->highy)/2;
	xform(cx, cy, &gx, &gy, t2);
	gx -= cx;   gy -= cy;

	/* do the rotation */
	startobjectchange((INTBIG)theni, VNODEINST);
	modifynodeinst(theni, gx, gy, gx, gy, amt, 0);
	endobjectchange((INTBIG)theni, VNODEINST);

	/* delete intermediate arcs used to constrain */
	for(i=0; i<aicount; i++)
	{
		startobjectchange((INTBIG)ailist[i], VARCINST);
		killarcinst(ailist[i]);
	}
	if (aicount > 0) efree((char *)ailist);

	/* restore highlighting */
	(void)us_pophighlight(1);
}

#define	MAXPORTTYPE 13

void us_show(INTSML count, char *par[])
{
	char line[100], *activity, *name, *colorname, *colorsymbol, *dumpfilename, *truename;
	REGISTER char *pt, *matchspec, *str, **keybindings, **buttonbindings;
	INTBIG plx, ply, phx, phy, keyindex[NUMKEYS], porttype[MAXPORTTYPE], len, wid, xp, yp;
	INTSML shortcols, numtypes, num_found, idummy16;
	REGISTER INTBIG i, j, k, m;
	REGISTER INTSML l, verbose, maxlen, total, columns, rows, key, but, menu,
		popup, shortcolwidth, shortrows, x, y, first, givedates, editlist,
		graphiclist, contentslist, fun, keytotal, done;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np, *wnp, **sortindex;
	REGISTER PORTPROTO *pp, **pplist;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER ARCPROTO *ap;
	REGISTER POPUPMENU *pm, *wantpm;
	REGISTER USERCOM *rb;
	REGISTER HIGHLIGHT *high;
	REGISTER LIBRARY *lib, *olib;
	REGISTER VIEW *v;
	REGISTER WINDOWPART *w;
	REGISTER TECHNOLOGY *tech;
	FILE *dumpfile;
	extern COMCOMP us_showp;
	extern char *us_castofthousands[];
	extern INTBIG cla_changeclock;	/* from layout constraints: "conlay.c" */
	CONSTRAINT *con;
	REGISTER VARIABLE *var, *varkey, *varbutton;
	REGISTER NETWORK *net;
	COMMANDBINDING commandbinding;

	if (count == 0)
	{
		count = ttygetparam("Show option: ", &us_showp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pt = par[0]);

	if (namesamen(pt, "aids", l) == 0 && l >= 1)
	{
		ttyputmsg(" Which Aid       Information");
		for(i=0; i<el_maxaid; i++)
		{
			(void)initinfstr();
			if ((el_aids[i].aidstate&AIDON) == 0) (void)addstringtoinfstr("Off"); else
				(void)addstringtoinfstr("On");
			if ((el_aids[i].aidstate&AIDBG) != 0) (void)addstringtoinfstr(", Background");
			if ((el_aids[i].aidstate&AIDFIX) != 0) (void)addstringtoinfstr(", Correcting");
			if ((el_aids[i].aidstate&AIDLANG) != 0) (void)addstringtoinfstr(", Interpreted");
			if ((el_aids[i].aidstate&AIDINCREMENTAL) != 0) (void)addstringtoinfstr(", Incremental");
			if ((el_aids[i].aidstate&AIDANALYSIS) != 0) (void)addstringtoinfstr(", Analysis");
			if ((el_aids[i].aidstate&AIDSYNTHESIS) != 0) (void)addstringtoinfstr(", Synthesis");
			ttyputmsg("%-16s %s", el_aids[i].aidname, returninfstr());
		}
		return;
	}

	if (namesamen(pt, "bindings", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			us_abortcommand("Usage: show bindings key|menu|button|popup|all|short");
			return;
		}
		l = strlen(pt = par[1]);

		if (namesamen(pt, "short", l) == 0 && l >= 1)
		{
			varkey = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_keys);
			varbutton = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_buttons);
			if (varkey == NOVARIABLE || varbutton == NOVARIABLE)
			{
				ttyputerr("Cannot find key and button bindings");
				return;
			}
			keytotal = (INTSML)getlength(varkey);
			keybindings = (char **)varkey->addr;
			buttonbindings = (char **)varbutton->addr;

			/* print the button bindings */
			j = buttoncount();
			if (j > 0)
			{
				(void)initinfstr();
				for(i=0; i<(MESSAGESWIDTH-25)/2; i++) (void)addtoinfstr('-');
				(void)addstringtoinfstr(" Single Button Commands: ");
				for(i=0; i<(MESSAGESWIDTH-25)/2; i++) (void)addtoinfstr('-');
				ttyputmsg("%s", returninfstr());

				/* count the number of bound buttons, compute longest name */
				for(i=0, k=0, j=0; i<buttoncount(); i++)
				{
					us_parsebinding(buttonbindings[i], &commandbinding);
					if (*commandbinding.command != 0)
					{
						k++;
						for(l=0; commandbinding.command[l] != 0; l++)
							if (commandbinding.command[l] == ' ') break;
						j = maxi(j, l + strlen(buttonname((INTSML)i, &idummy16)));
					}
					us_freebindingparse(&commandbinding);
				}

				/* compute number of rows and columns */
				shortcols = (INTSML)mini(MESSAGESWIDTH / (j+4), k);
				shortcolwidth = MESSAGESWIDTH / shortcols;
				shortrows = (k+shortcols-1) / shortcols;

				/* print the buttons */
				i = -1;
				for(j=0; j<shortrows; j++)
				{
					(void)initinfstr();
					for(m=0; m<shortcols; m++)
					{
						/* find next bound button */
						for (;;)
						{
							i++;
							if (i >= buttoncount()) break;
							us_parsebinding(buttonbindings[i], &commandbinding);
							if (*commandbinding.command != 0) break;
							us_freebindingparse(&commandbinding);
						}
						if (i >= buttoncount()) break;

						/* place button name */
						pt = buttonname((INTSML)i, &idummy16);
						(void)addstringtoinfstr(pt);
						(void)addstringtoinfstr(": ");
						k = strlen(pt);

						/* place command name */
						for(l=0; commandbinding.command[l] != 0; l++)
							if (commandbinding.command[l] == ' ') break;
								else (void)addtoinfstr(commandbinding.command[l]);
						us_freebindingparse(&commandbinding);
						k += l;

						/* pad out the field if not at the end */
						if (m<shortcols-1)
							for(k = k+4; k < shortcolwidth; k++)
								(void)addtoinfstr(' ');
					}
					ttyputmsg("%s", returninfstr());
				}
			}

			/* print the key bindings */
			(void)initinfstr();
			for(i=0; i<(MESSAGESWIDTH-22)/2; i++) (void)addtoinfstr('-');
			(void)addstringtoinfstr(" Single Key Commands: ");
			for(i=0; i<(MESSAGESWIDTH-22)/2; i++) (void)addtoinfstr('-');
			ttyputmsg("%s", returninfstr());

			/* count the number of bound keys, compute longest name */
			for(i=0, k=0, j=0; i<keytotal; i++)
			{
				keyindex[i] = -1;
				us_parsebinding(keybindings[i], &commandbinding);
				if (*commandbinding.command != 0)
				{
					keyindex[i] = k++;
					for(l=0; commandbinding.command[l] != 0; l++)
						if (commandbinding.command[l] == ' ') break;
					j = maxi(j, l);
				}
				us_freebindingparse(&commandbinding);
			}

			/* compute number of rows and columns */
			shortcols = MESSAGESWIDTH / (j+4);
			shortcolwidth = MESSAGESWIDTH / shortcols;
			shortrows = (k+shortcols-1) / shortcols;

			/* print the keys */
			for(j=0; j<shortrows; j++)
			{
				(void)initinfstr();
				for(m=0; m<shortcols; m++) for(i=0; i<keytotal; i++)
					if (keyindex[i] == m*shortrows+j)
				{
					/* place key name */
					pt = us_keyname((INTSML)i);
					if (strlen(pt) < 3) (void)addtoinfstr(' ');
					if (strlen(pt) < 2) (void)addtoinfstr(' ');
					(void)addstringtoinfstr(pt);
					(void)addtoinfstr(' ');

					/* place command name */
					us_parsebinding(keybindings[i], &commandbinding);
					for(l=0; commandbinding.command[l] != 0; l++)
						if (commandbinding.command[l] == ' ') break; else
							(void)addtoinfstr(commandbinding.command[l]);
					us_freebindingparse(&commandbinding);

					/* pad out the field if not at the end */
					if (m<shortcols-1)
						for(k=l+4; k < shortcolwidth; k++)
							(void)addtoinfstr(' ');
				}
				ttyputmsg("%s", returninfstr());
			}
			ttyputmsg("Type '-help' if you need it");
			return;
		}

		key = but = menu = popup = 0;
		if (namesamen(pt, "all", l) == 0 && l >= 1)
		{
			but++;   key++;   menu++;   popup++;
		}
		if (namesamen(pt, "key", l) == 0 && l >= 1) key++;
		if (namesamen(pt, "menu", l) == 0 && l >= 1) menu++;
		if (namesamen(pt, "button", l) == 0 && l >= 1) but++;
		if (namesamen(pt, "popup", l) == 0 && l >= 1) popup++;

		if (key == 0 && but == 0 && menu == 0 && popup == 0)
		{
			us_abortcommand("Unknown show bindings option: %s", pt);
			return;
		}

		/* if key bindings are requested, list them */
		if (key)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_keys);
			if (var == NOVARIABLE)
			{
				ttyputerr("Cannot find key binding attributes");
				return;
			}
			l = (INTSML)getlength(var);
			for(i=0; i<l; i++)
			{
				us_parsebinding(((char **)var->addr)[i], &commandbinding);
				if (*commandbinding.command != 0)
					ttyputmsg("Key %s: %s", us_keyname((INTSML)i), commandbinding.command);
				us_freebindingparse(&commandbinding);
			}
		}

		/* if button bindings are requested, list them */
		if (but)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_buttons);
			if (var == NOVARIABLE)
			{
				ttyputerr("Cannot find button binding attributes");
				return;
			}
			l = (INTSML)getlength(var);
			for(i=0; i<l; i++)
			{
				us_parsebinding(((char **)var->addr)[i], &commandbinding);
				if (*commandbinding.command != 0) ttyputmsg("Button %s: %s",
					buttonname((INTSML)i, &idummy16), commandbinding.command);
				us_freebindingparse(&commandbinding);
			}
		}

		/* if menu bindings are requested, list them */
		if (menu)
		{
			var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_menu);
			if (var == NOVARIABLE)
			{
				ttyputerr("Cannot find menu binding attributes");
				return;
			}
			for(x=0; x<us_menux; x++) for(y=0; y<us_menuy; y++)
			{
				if (us_menupos <= 1)
					str = ((char **)var->addr)[y * us_menux + x]; else
						str = ((char **)var->addr)[x * us_menuy + y];
				us_parsebinding(str, &commandbinding);
				(void)initinfstr();
				(void)sprintf(line, "Menu row %d column %d: ", y, x);
				(void)addstringtoinfstr(line);
				if (*commandbinding.command == 0) (void)addstringtoinfstr("NOT DEFINED"); else
					(void)addstringtoinfstr(commandbinding.command);
				if (commandbinding.menumessage != 0)
				{
					(void)addstringtoinfstr(" [message=\"");
					(void)addstringtoinfstr(commandbinding.menumessage);
					(void)addstringtoinfstr("\"]");
				} else if (commandbinding.nodeglyph != NONODEPROTO)
				{
					(void)addstringtoinfstr(" [node=");
					(void)addstringtoinfstr(describenodeproto(commandbinding.nodeglyph));
					(void)addtoinfstr(']');
				} else if (commandbinding.arcglyph != NOARCPROTO)
				{
					(void)addstringtoinfstr(" [arc=");
					(void)addstringtoinfstr(describearcproto(commandbinding.arcglyph));
					(void)addtoinfstr(']');
				}
				if (commandbinding.backgroundcolor != 0)
				{
					if (ecolorname(commandbinding.backgroundcolor, &colorname, &colorsymbol) != 0)
						colorname = "**UNKNOWN**";
					(void)sprintf(line, " [background=%s]", colorname);
					(void)addstringtoinfstr(line);
				}
				ttyputmsg("%s", returninfstr());
				us_freebindingparse(&commandbinding);
			}
		}

		/* if popup menu bindings are requested, list them */
		if (popup)
		{
			if (count <= 2) wantpm = NOPOPUPMENU; else
			{
				wantpm = us_getpopupmenu(par[2]);
				if (wantpm == NOPOPUPMENU)
				{
					us_abortcommand("Cannot find popup menu '%s'", par[2]);
					return;
				}
			}

			for(pm=us_firstpopupmenu; pm!=NOPOPUPMENU; pm=pm->nextpopupmenu)
			{
				if (wantpm != NOPOPUPMENU && wantpm != pm) continue;
				ttyputmsg("Popup menu %s has %d entries:", pm->name, pm->total);
				for(i=0; i<pm->total; i++)
				{
					rb = pm->list[i].response;
					(void)initinfstr();
					(void)sprintf(line, "Entry %ld: ", i);
					(void)addstringtoinfstr(line);
					if (rb->active < 0) (void)addstringtoinfstr("NOT DEFINED"); else
					{
						(void)addstringtoinfstr(rb->comname);
						(void)us_appendargs(rb);
					}
					if (rb->message != 0)
					{
						(void)addstringtoinfstr(" [message=\"");
						(void)addstringtoinfstr(rb->message);
						(void)addstringtoinfstr("\"]");
					}
					ttyputmsg("   %s", returninfstr());
				}
			}
		}
		return;
	}

	if (namesamen(pt, "cells", l) == 0 && l >= 1)
	{
		np = getcurfacet();
		if (np != NONODEPROTO) lib = np->cell->lib; else
			lib = el_curlib;
		matchspec = 0;
		while (count >= 2)
		{
			l = strlen(pt = par[1]);
			if (namesamen(pt, "matching", l) == 0)
			{
				if (count < 3)
				{
					us_abortcommand("USAGE: show cells matching MATCHSPEC");
					return;
				}
				if (matchspec != 0)
				{
					us_abortcommand("Can only have one 'matching' clause");
					return;
				}
				matchspec = par[2];
				par++;
				count--;
			} else if (namesamen(pt, "library", l) == 0)
			{
				if (count < 3)
				{
					us_abortcommand("USAGE: show cells library LIBNAME");
					return;
				}
				lib = getlibrary(par[2]);
				if (lib == NOLIBRARY)
				{
					us_abortcommand("No library called %s", par[2]);
					return;
				}
				par++;
				count--;
			} else
			{
				us_abortcommand("Bad SHOW CELLS option: %s", pt);
				return;
			}
			par++;
			count--;
		}

		/* allocate array for sorting entries */
		sortindex = us_sortlib(lib, matchspec, 1);
		if (sortindex == 0)
		{
			ttyputerr("No memory");
			return;
		}

		/* compute the longest cell name and the number of facets */
		maxlen = 0;   total = 0;
		for(i=0; sortindex[i] != NONODEPROTO; i++)
		{
			l = strlen(sortindex[i]->cell->cellname);
			maxlen = (INTSML)maxi(maxlen, l);
			total++;
		}

		/* short list in columns */
		ttyputmsg("----- Cells in library %s -----", lib->libname);
		maxlen += 2;  columns = MESSAGESWIDTH / maxlen;
		if (columns <= 0) columns = 1;
		rows = (total + columns - 1) / columns;
		for(j=0; j<rows; j++)
		{
			(void)initinfstr();
			for(k=0; k<columns; k++)
			{
				i = j + k*rows;
				if (i >= total) continue;
				np = sortindex[i];
				pt = np->cell->cellname;
				(void)addstringtoinfstr(pt);
				l = strlen(pt);
				if (k != columns-1)
					for(i=l; i<maxlen; i++) (void)addtoinfstr(' ');
			}
			ttyputmsg("%s", returninfstr());
		}

		/* free the sort list */
		efree((char *)sortindex);
		return;
	}

	if (namesamen(pt, "dates", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			np = us_needfacet();
			if (np == NONODEPROTO) return;
		} else
		{
			np = getnodeproto(par[1]);
			if (np == NONODEPROTO || np->primindex != 0)
			{
				us_abortcommand("No such facet: %s", par[1]);
				return;
			}
		}

		/* give requested info */
		if (np->creationdate == 0)
			ttyputmsg("Facet %s has no recorded creation date", describenodeproto(np)); else
				ttyputmsg("Facet %s was created %s", describenodeproto(np),
					timetostring(np->creationdate));
		ttyputmsg("Version %d was last revised %s", np->version, timetostring(np->revisiondate));
		return;
	}

	if (namesamen(pt, "environment", l) == 0 && l >= 1)
	{
		ttyputmsg("This is Electric, version %s", el_version);
		pt = languagename();
		if (pt != NOSTRING) ttyputmsg("Includes built-in %s", pt);
		ttyputmsg("Default library directory is %s", el_libdir);

		if (count > 1 && namesamen(par[1], "authors", strlen(par[1])) == 0)
		{
			ttyputmsg("Electric was written by Steven M. Rubin");
			ttyputmsg("   and a cast of thousands:");
			for(i=0; us_castofthousands[i] != 0; i++)
			ttyputmsg("      %s", us_castofthousands[i]);
		}
		return;
	}
	if (namesamen(pt, "facets", l) == 0 && l >= 1)
	{
		givedates = editlist = graphiclist = contentslist = 0;
		dumpfilename = 0;
		np = getcurfacet();
		if (np != NONODEPROTO) lib = np->cell->lib; else
			lib = el_curlib;
		matchspec = 0;
		while (count >= 2)
		{
			l = strlen(pt = par[1]);
			if (namesamen(pt, "dates", l) == 0) givedates++; else
			if (namesamen(pt, "edit", l) == 0) editlist++; else
			if (namesamen(pt, "graphically", l) == 0) graphiclist++; else
			if (namesamen(pt, "contained-in-this", l) == 0) contentslist++; else
			if (namesamen(pt, "matching", l) == 0)
			{
				if (count < 3)
				{
					us_abortcommand("USAGE: show facets matching MATCHSPEC");
					return;
				}
				if (matchspec != 0)
				{
					us_abortcommand("Can only have one 'matching' clause");
					return;
				}
				matchspec = par[2];
				par++;
				count--;
			} else if (namesamen(pt, "library", l) == 0)
			{
				if (count < 3)
				{
					us_abortcommand("USAGE: show facets library LIBNAME");
					return;
				}
				lib = getlibrary(par[2]);
				if (lib == NOLIBRARY)
				{
					us_abortcommand("No library called %s", par[2]);
					return;
				}
				par++;
				count--;
			} else if (namesamen(pt, "file", l) == 0)
			{
				if (count < 3)
				{
					us_abortcommand("USAGE: show facets file FILENAME");
					return;
				}
				dumpfilename = par[2];
				par++;
				count--;
			} else
			{
				us_abortcommand("Bad SHOW FACETS option: %s", pt);
				return;
			}
			par++;
			count--;
		}

		if (contentslist != 0)
		{
			if (givedates != 0 || editlist != 0 || graphiclist != 0 ||
				matchspec != 0 || dumpfilename != 0)
					ttyputerr("Other options for 'contained-in-this' ignored");

			wnp = us_needfacet();
			if (wnp == NONODEPROTO) return;
			for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
				for(np = olib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
					np->temp1 = 0;
			for(ni = wnp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				np = ni->proto;
				if (np->primindex != 0) continue;
				np->temp1++;
			}
			first = 0;
			for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
				for(np = olib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
					if (np->temp1 != 0)
			{
				if (first == 0)
					ttyputmsg("Facet instances appearing in %s", describenodeproto(wnp));
				first++;
				ttyputmsg("   %ld instances of %s", np->temp1, describenodeproto(np));
			}
			if (first == 0)
				ttyputmsg("There are no facet instances in %s", describenodeproto(wnp));
			return;
		}

		/* graph the facets if requested */
		if (graphiclist != 0)
		{
			if (givedates != 0 || editlist != 0 || contentslist != 0 ||
				matchspec != 0 || dumpfilename != 0)
					ttyputerr("Other options for 'graphically' ignored");
			us_graphfacets();
			return;
		}

		/* allocate array for sorting entries */
		sortindex = us_sortlib(lib, matchspec, 0);
		if (sortindex == 0)
		{
			ttyputerr("No memory");
			return;
		}

		/* compute the longest facet name and the number of facets */
		maxlen = 0;   total = 0;
		for(i=0; sortindex[i] != NONODEPROTO; i++)
		{
			l = strlen(nldescribenodeproto(sortindex[i]));
			maxlen = (INTSML)maxi(maxlen, l);
			total++;
		}

		/* short list in columns */
		if (givedates == 0 && editlist == 0)
		{
			if (dumpfilename != 0)
				ttyputerr("Cannot write to file unless 'dates' option specified");
			ttyputmsg("----- Facets in library %s -----", lib->libname);
			maxlen += 2;  columns = MESSAGESWIDTH / maxlen;
			if (columns <= 0) columns = 1;
			rows = (total + columns - 1) / columns;
			for(j=0; j<rows; j++)
			{
				(void)initinfstr();
				for(k=0; k<columns; k++)
				{
					i = j + k*rows;
					if (i >= total) continue;
					np = sortindex[i];
					pt = nldescribenodeproto(np);
					(void)addstringtoinfstr(pt);
					l = strlen(pt);
					if (k != columns-1)
						for(i=l; i<maxlen; i++) (void)addtoinfstr(' ');
				}
				ttyputmsg("%s", returninfstr());
			}

			/* free the sort list */
			efree((char *)sortindex);
			return;
		}

		/* full list with dates */
		if (givedates != 0)
		{
			/* create the dump file if requested */
			if (dumpfilename != 0)
			{
				dumpfile = xopen(dumpfilename, FILETYPETEXT|FILETYPEWRITE, 0, &truename);
				if (dumpfile == 0)
				{
					us_abortcommand("Cannot write %s", dumpfilename);
					return;
				}
				pt = timetostring(getcurrenttime());
				fprintf(dumpfile, "List of facets in library %s created on %s\n", lib->libname, pt);
				fprintf(dumpfile, "Facet\tVersion\tCreation date\tRevision Date\tSize\tUsage\tLock\tInst-lock\tCell-lib\tDRC\n");
				for(j=0; j<total; j++)
					fprintf(dumpfile, "%s\n", us_makefacetline(sortindex[j], -1));
				xclose(dumpfile);
				ttyputmsg("Wrote %s", truename);
			} else
			{
				maxlen = (INTSML)maxi(maxlen+2, 7);
				(void)initinfstr();
				(void)addstringtoinfstr("Facet");
				for(i=5; i<maxlen; i++) (void)addtoinfstr('-');
				(void)addstringtoinfstr("Version-----Creation date");
				(void)addstringtoinfstr("---------Revision Date------------Size-------Usage-L-I-C-D");
				ttyputmsg("%s", returninfstr());
				for(j=0; j<total; j++)
					ttyputmsg("%s", us_makefacetline(sortindex[j], maxlen));
			}

			/* free the sort list */
			efree((char *)sortindex);
			return;
		}

		/* editable list */
		if (editlist != 0)
		{
			if (dumpfilename != 0)
				ttyputerr("Cannot write to file when 'edit' option specified");
			w = us_wantnewwindow(0);
			if (w == NOWINDOWPART) return;
			(void)initinfstr();
			us_describeeditor(&name);
			(void)addstringtoinfstr(name);
			(void)addstringtoinfstr(" editor of facet names in library ");
			(void)addstringtoinfstr(lib->libname);
			if (us_makeeditor(w, returninfstr(), &idummy16, &idummy16) == NOWINDOWPART) return;
			w->charhandler = us_facetedithandler;
			us_suspendgraphics(w);
			maxlen = (INTSML)maxi(maxlen+2, 7);
			(void)initinfstr();
			(void)addstringtoinfstr("Facet");
			for(i=5; i<maxlen; i++) (void)addtoinfstr('-');
			(void)addstringtoinfstr("Version-----Creation date");
			(void)addstringtoinfstr("---------Revision Date------------Size-------Usage-L-I-C-D");
			us_addline(w, 0, returninfstr());
			for(j=0; j<total; j++)
				us_addline(w, j+1, us_makefacetline(sortindex[j], maxlen));
			us_resumegraphics(w);
			us_describeeditor(&name);
			if (namesame(name, "emacs") == 0)
				ttyputmsg("Use M(=) to edit the facet on the current line");

			/* free the sort list */
			efree((char *)sortindex);
			return;
		}
	}

	if (namesamen(pt, "history", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			showhistorylist(-1);
			return;
		}
		showhistorylist((INTSML)myatoi(par[1]));
		return;
	}

	if (namesamen(pt, "libraries", l) == 0 && l >= 1)
	{
		ttyputmsg("----- Libraries -----");
		maxlen = 0;
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			maxlen = (INTSML)maxi(maxlen, strlen(lib->libname));
		maxlen += 3;
		k = 0;
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		{
			(void)strcpy(line, lib->libname);
			if ((lib->userbits&(LIBCHANGEDMAJOR | LIBCHANGEDMINOR)) != 0)
			{
				(void)strcat(line, "*");
				k++;
			}
			if (lib->curnodeproto != NONODEPROTO)
			{
				while ((INTBIG)strlen(line) < maxlen) (void)strcat(line, " ");
				(void)strcat(line, "top facet: ");
				(void)strcat(line, nldescribenodeproto(lib->curnodeproto));
				if (strcmp(lib->libname, lib->libfile) != 0)
				{
					(void)strcat(line, " (disk file: ");
					(void)strcat(line, lib->libfile);
					(void)strcat(line, ")");
				}
			}
			ttyputmsg(line);
		}
		if (k != 0) ttyputmsg("   (* means library has changed)");
		return;
	}

	if (namesamen(pt, "macros", l) == 0 && l >= 1)
	{
		if (count >= 2)
		{
			var = us_getmacro(par[1]);
			if (var != NOVARIABLE)
			{
				us_printmacro(var);
				return;
			}
			us_abortcommand("No macro named %s", par[1]);
			return;
		}
		us_printmacros();
		return;
	}

	if (namesamen(pt, "networks", l) == 0 && l >= 1)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		total = 0;
		for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
			if (net->namecount != 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(describenetwork(net));
			(void)addstringtoinfstr("'");
			if (net->signals > 1)
			{
				(void)sprintf(line, " (bus with %d signals)", net->signals);
				(void)addstringtoinfstr(line);
			}
			if (net->arccount == 0 && net->portcount == 0 && net->buslinkcount == 0)
				(void)addstringtoinfstr(" not connected"); else
			{
				(void)addstringtoinfstr(" connected on");
				if (net->arccount != 0)
				{
					(void)sprintf(line, " %d arc", net->arccount);
					(void)addstringtoinfstr(line);
					if (net->arccount > 1) (void)addtoinfstr('s');
				}
				if (net->portcount != 0)
				{
					if (net->arccount != 0) (void)addstringtoinfstr(" and");
					(void)sprintf(line, " %d port", net->portcount);
					(void)addstringtoinfstr(line);
					if (net->portcount > 1) (void)addtoinfstr('s');
				}
				if (net->buslinkcount > 0)
				{
					(void)sprintf(line, " %d bus", net->buslinkcount);
					(void)addstringtoinfstr(line);
					if (net->buslinkcount > 1) (void)addstringtoinfstr("ses");
				}
			}
			ttyputmsg("Network '%s", returninfstr());
			total++;
		}
		if (total == 0) ttyputmsg("There are no named networks");
		return;
	}

	if (namesamen(pt, "object", l) == 0 && l >= 1)
	{
		verbose = 0;
		if (count > 1)
		{
			l = strlen(par[1]);
			if (namesamen(par[1], "short", l) == 0 && l >= 1) verbose = -1;
			if (namesamen(par[1], "long", l) == 0 && l >= 1) verbose = 1;
		}

		high = us_getonehighlight();
		if (high == NOHIGHLIGHT) return;

		if ((high->status&HIGHTYPE) == HIGHTEXT)
		{
			/* describe highlighted text */
			if (high->fromvar != NOVARIABLE)
			{
				ttyputmsg("%s variable '%s' is on %s", us_variableattributes(high->fromvar, -1),
					describevariable(high->fromvar, -1, -1), geomname(high->fromgeom));
				return;
			}
			if (high->fromport != NOPORTPROTO)
			{
				(void)initinfstr();
				activity = us_describeportbits(high->fromport);
				if (*activity != 0)
				{
					(void)addstringtoinfstr(activity);
					(void)addstringtoinfstr(" port name '");
				} else (void)addstringtoinfstr("Port name '");
				(void)addstringtoinfstr(high->fromport->protoname);
				(void)addstringtoinfstr("' is on ");
				(void)addstringtoinfstr(geomname(high->fromgeom));
				(void)addstringtoinfstr(" (label ");
				(void)addstringtoinfstr(us_describetextdescript(
					high->fromport->textdescript));
				if ((high->fromport->userbits&PORTDRAWN) != 0)
					(void)addstringtoinfstr(",always-drawn");
				if ((high->fromport->userbits&BODYONLY) != 0)
					(void)addstringtoinfstr(",only-on-body");
				(void)addstringtoinfstr(") ");
				ttyputmsg("%s", returninfstr());
				return;
			}
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				ni = high->fromgeom->entryaddr.ni;
				(void)initinfstr();
				(void)addstringtoinfstr("Facet name '");
				(void)addstringtoinfstr(describenodeproto(ni->proto));
				(void)addstringtoinfstr("' (label ");
				(void)addstringtoinfstr(us_describetextdescript(ni->textdescript));
				(void)addstringtoinfstr(") ");
				ttyputmsg("%s", returninfstr());
				return;
			}
		}
		if ((high->status&HIGHTYPE) != HIGHFROM)
		{
			us_abortcommand("Find a single node or arc first");
			return;
		}

		/* describe a nodeinst */
		if (high->fromgeom->entrytype == OBJNODEINST)
		{
			/* give basic information about nodeinst */
			ni = high->fromgeom->entryaddr.ni;
			np = ni->proto;
			nodesizeoffset(ni, &plx, &ply, &phx, &phy);
			(void)initinfstr();
			if (np->primindex != 0) (void)addstringtoinfstr("Node "); else
				(void)addstringtoinfstr("Facet ");
			(void)addstringtoinfstr(describenodeinst(ni));
			(void)addstringtoinfstr(" is ");
			(void)addstringtoinfstr(latoa(ni->highx-ni->lowx-plx-phx));
			(void)addtoinfstr('x');
			(void)addstringtoinfstr(latoa(ni->highy-ni->lowy-ply-phy));

			/* special case for serpentine transistors: print true size */
			if (np->primindex != 0 && (np->userbits&HOLDSTRACE) != 0)
			{
				fun = (INTSML)((np->userbits&NFUNCTION) >> NFUNCTIONSH);
				if (fun == NPTRANMOS || fun == NPTRADMOS || fun == NPTRAPMOS)
				{
					var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
					if (var != NOVARIABLE)
					{
						transistorsize(ni, &len, &wid);
						if (len != -1 && wid != -1)
						{
							(void)addstringtoinfstr(" (actually ");
							(void)addstringtoinfstr(latoa(len));
							(void)addtoinfstr('x');
							(void)addstringtoinfstr(latoa(wid));
							(void)addtoinfstr(')');
						}
					}
				}
			}
			(void)addstringtoinfstr(", ");
			if (ni->transpose != 0) (void)addstringtoinfstr("transposed and ");
			(void)addstringtoinfstr("rotated ");
			(void)addstringtoinfstr(frtoa(ni->rotation*WHOLE/10));
			(void)addstringtoinfstr(", center (");
			(void)addstringtoinfstr(latoa((ni->highx+ni->lowx)/2));
			(void)addtoinfstr(',');
			(void)addstringtoinfstr(latoa((ni->highy+ni->lowy)/2));
			(void)addtoinfstr(')');
			ttyputmsg("%s", returninfstr());

			/* reset the "chat" indicator on the port prototypes */
			for(pp=np->firstportproto; pp!=NOPORTPROTO; pp=pp->nextportproto)
				pp->temp1 = 0;

			/* always describe the highlighted port */
			if (high->fromport != NOPORTPROTO)
				us_chatportproto(ni, high->fromport);

			/* describe all arcs and ports if not short option */
			if (verbose >= 0)
			{
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					us_chatportproto(ni, pe->proto);
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					us_chatportproto(ni, pi->proto);
			}

			/* long option: describe aid information, variables, etc */
			if (verbose > 0)
			{
				/* talk about every port in long option */
				for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					us_chatportproto(ni, pp);

				/* print aid information */
				us_printnodeaidinfo(ni);

				/* print variables */
				for(i=0; i<ni->numvar; i++)
					ttyputmsg("%s variable '%s' is %s", us_variableattributes(&ni->firstvar[i], -1),
						makename(ni->firstvar[i].key), describevariable(&ni->firstvar[i], -1, -1));

				/* describe contents */
				if (np->primindex == 0) us_describecontents(ni);
			}
		} else if (high->fromgeom->entrytype == OBJARCINST)
		{
			/* print the basic information about the arcinst */
			ai = high->fromgeom->entryaddr.ai;

			/* compute the arc length and width */
			wid = ai->width - arcwidthoffset(ai);
			len = ai->length;
			if ((ai->userbits&NOEXTEND) == 0) len += wid; else
			{
				if ((ai->userbits&NOTEND0) != 0) len += wid/2;
				if ((ai->userbits&NOTEND1) != 0) len += wid/2;
			}

			/* build a string of constraints and properties */
			(void)initinfstr();
			(void)addstringtoinfstr(describearcinst(ai));
			(void)addstringtoinfstr(" arc is ");
			if (el_curconstraint == cla_constraint)
			{
				if (((ai->userbits&FIXED) == 0 || ai->changed == cla_changeclock+3) &&
					ai->changed != cla_changeclock+2)
				{
					if (ai->changed == cla_changeclock+3) (void)addstringtoinfstr("temporarily ");
					(void)addstringtoinfstr("stretchable, ");
					if ((ai->userbits&FIXANG) == 0) (void)addstringtoinfstr("not ");
					(void)addstringtoinfstr("fixed-angle, ");
					if ((ai->userbits&CANTSLIDE) != 0) (void)addstringtoinfstr("non");
					(void)addstringtoinfstr("slidable, ");
				} else
				{
					if (ai->changed == cla_changeclock+2) (void)addstringtoinfstr("temporarily ");
					(void)addstringtoinfstr("rigid, ");
				}
			} else
			{
				pt = (char *)(*(el_curconstraint->request))("describearc", (INTBIG)ai);
				if (*pt != 0)
				{
					(void)addstringtoinfstr("constrained to ");
					(void)addstringtoinfstr(pt);
					(void)addstringtoinfstr(", ");
				}
			}
			if ((ai->userbits&ISNEGATED) != 0) (void)addstringtoinfstr("negated, ");
			if ((ai->userbits&ISDIRECTIONAL) != 0) (void)addstringtoinfstr("directional, ");
			if ((ai->userbits&(NOTEND0|NOTEND1)) != 0)
			{
				(void)addstringtoinfstr("with ");
				switch (ai->userbits & (NOTEND0|NOTEND1))
				{
					case NOTEND0:         (void)addstringtoinfstr("tail");             break;
					case NOTEND1:         (void)addstringtoinfstr("head");             break;
					case NOTEND0|NOTEND1: (void)addstringtoinfstr("head and tail");    break;
				}
				(void)addstringtoinfstr(" skipped, ");
			}
			if ((ai->userbits&REVERSEEND) != 0) (void)addstringtoinfstr("ends reversed, ");

			/* add bus width */
			if (ai->network != NONETWORK && ai->network->signals > 1)
			{
				(void)sprintf(line, "has %d signals, ", ai->network->signals);
				(void)addstringtoinfstr(line);
			}

			/* add in width and length */
			(void)addstringtoinfstr(latoa(wid));
			(void)addstringtoinfstr(" wide and ");
			(void)addstringtoinfstr(latoa(len));
			(void)addstringtoinfstr(" long");

			/* if the arc is unnamed but on a network, report this */
			var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
			if (var == NOVARIABLE)
			{
				if (ai->network != NONETWORK && ai->network->namecount != 0)
				{
					(void)addstringtoinfstr(", on network ");
					(void)addstringtoinfstr(describenetwork(ai->network));
				}
			}

			/* print the message */
			ttyputmsg("%s", returninfstr());

			/* tell about the ends if default or long option */
			if (verbose >= 0)
			{
				ttyputmsg("Tail on port %s of %s at (%s,%s) runs %ld degrees to",
					ai->end[0].portarcinst->proto->protoname, describenodeinst(ai->end[0].nodeinst),
						latoa(ai->end[0].xpos), latoa(ai->end[0].ypos),
							(ai->userbits & AANGLE) >> AANGLESH);
				ttyputmsg("Head on port %s of %s at (%s,%s)",
					ai->end[1].portarcinst->proto->protoname, describenodeinst(ai->end[1].nodeinst),
						latoa(ai->end[1].xpos), latoa(ai->end[1].ypos));
			}

			/* tell about aid values and variables if long option */
			if (verbose > 0)
			{
				/* print aid information */
				us_printarcaidinfo(ai);

				/* print variables */
				for(i=0; i<ai->numvar; i++)
					ttyputmsg("%s variable '%s' is %s", us_variableattributes(&ai->firstvar[i], -1),
						makename(ai->firstvar[i].key), describevariable(&ai->firstvar[i], -1, -1));
			}
		}
		return;
	}

	if (namesamen(pt, "ports", l) == 0 && l >= 2)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		numtypes = 0;
		if (count == 1) par[count++] = "all";
		for (i=1; i<count; i++)
		{
			l = strlen(par[i]);
			if (namesamen(par[i], "clock", l) == 0 && l >= 1)
			{
				porttype[numtypes++] = CLKPORT;		/* basic clock */
				porttype[numtypes++] = C1PORT;		/* clock phase 1 */
				porttype[numtypes++] = C2PORT;		/* clock phase 2 */
				porttype[numtypes++] = C3PORT;		/* clock phase 3 */
				porttype[numtypes++] = C4PORT;		/* clock phase 4 */
				porttype[numtypes++] = C5PORT;		/* clock phase 5 */
				porttype[numtypes++] = C6PORT;		/* clock phase 6 */
			} else if (namesamen(par[i], "input", l) == 0 && l >= 1)
				porttype[numtypes++] = INPORT;		/* input */
			else if (namesamen(par[i], "output", l) == 0 && l >= 1)
				porttype[numtypes++] = OUTPORT;		/* output */
			else if (namesamen(par[i], "bidirectional", l) == 0 && l >= 1)
				porttype[numtypes++] = BIDIRPORT;	/* bidirectional */
			else if (namesamen(par[i], "power", l) == 0 && l >= 1)
				porttype[numtypes++] = PWRPORT;		/* power */
			else if (namesamen(par[i], "ground", l) == 0 && l >= 2)
				porttype[numtypes++] = GNDPORT;		/* ground */
			else if (namesamen(par[i], "reference", l) == 0 && l >= 1)
			{
				porttype[numtypes++] = REFOUTPORT;	/* ref out */
				porttype[numtypes++] = REFINPORT;	/* ref in */
			} else if (namesamen(par[i], "generic", l) == 0 && l >= 2)
				porttype[numtypes++] = 0;
			else if (namesamen(par[i], "all", l) == 0 && l >= 1)
			{
				porttype[numtypes++] = 0;		/* generic */
				porttype[numtypes++] = CLKPORT;		/* basic clock */
				porttype[numtypes++] = C1PORT;		/* clock phase 1 */
				porttype[numtypes++] = C2PORT;		/* clock phase 2 */
				porttype[numtypes++] = C3PORT;		/* clock phase 3 */
				porttype[numtypes++] = C4PORT;		/* clock phase 4 */
				porttype[numtypes++] = C5PORT;		/* clock phase 5 */
				porttype[numtypes++] = C6PORT;		/* clock phase 6 */
				porttype[numtypes++] = INPORT;		/* input */
				porttype[numtypes++] = OUTPORT;		/* output */
				porttype[numtypes++] = BIDIRPORT;	/* bidirectional */
				porttype[numtypes++] = PWRPORT;		/* power */
				porttype[numtypes++] = GNDPORT;		/* ground */
				porttype[numtypes++] = REFOUTPORT;	/* ref out */
				porttype[numtypes++] = REFINPORT;	/* ref in */
			} else
			{
				 us_abortcommand("Invalid show port option '%s'", par[i]);
				 return;
			}
		}

		/* compute the associated facet to check */
		wnp = contentsview(np);
		if (wnp == NONODEPROTO) wnp = iconview(np);
		if (wnp == np) wnp = NONODEPROTO;

		/* count the number of ports */
		num_found = 0;
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			m = pp->userbits & STATEBITS;
			for(i=0; i<numtypes; i++) if (porttype[i] == m) break;
			if (i >= numtypes) continue;
			num_found++;
		}
		if (num_found == 0)
		{
			ttyputmsg("There are no ports on facet %s", describenodeproto(np));
			return;
		}

		/* make a list of ports */
		pplist = (PORTPROTO **)emalloc(num_found * (sizeof (PORTPROTO *)), el_tempcluster);
		if (pplist == 0) return;
		num_found = 0;
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			m = pp->userbits & STATEBITS;
			for(i=0; i<numtypes; i++) if (porttype[i] == m) break;
			if (i >= numtypes) continue;
			pplist[num_found++] = pp;
		}

		/* sort ports by name within type */
		done = 0;
		while (done == 0)
		{
			done = 1;
			for(j=1; j<num_found; j++)
			{
				if ((pplist[j]->userbits&STATEBITS) > (pplist[j-1]->userbits&STATEBITS))
					continue;
				if ((pplist[j]->userbits&STATEBITS) == (pplist[j-1]->userbits&STATEBITS) &&
					namesame(pplist[j]->protoname, pplist[j-1]->protoname) >= 0) continue;
				pp = pplist[j];
				pplist[j] = pplist[j-1];
				pplist[j-1] = pp;
				done = 0;
			}
		}

		/* describe each port */
		ttyputmsg("----- Ports on facet %s -----", describenodeproto(np));
		for(j=0; j<num_found; j++)
		{
			pp = pplist[j];

			(void)initinfstr();
			activity = us_describeportbits(pp);
			if (*activity == 0) activity = "Unknown";
			(void)addstringtoinfstr(activity);
			(void)addstringtoinfstr(" port '");
			(void)addstringtoinfstr(pp->protoname);

			portposition(pp->subnodeinst, pp->subportproto, &xp, &yp);
			(void)addstringtoinfstr("' connects at (");
			(void)addstringtoinfstr(latoa(xp));
			(void)addstringtoinfstr(", ");
			(void)addstringtoinfstr(latoa(yp));
			(void)addstringtoinfstr(") to");
			m = 0;
			for(i=0; pp->connects[i] != NOARCPROTO; i++)
			{
				ap = pp->connects[i];
				if (ap->tech == gen_tech) continue;
				if (m > 0) (void)addstringtoinfstr(",");
				(void)addstringtoinfstr(" ");
				(void)addstringtoinfstr(ap->protoname);
				m++;
			}

			/* check for the port in the associated facet */
			if (wnp != NONODEPROTO)
			{
				if (equivalentport(np, pp, wnp) == NOPORTPROTO)
				{
					(void)addstringtoinfstr(" *** no equivalent in ");
					(void)addstringtoinfstr(describenodeproto(wnp));
				}
			}

			ttyputmsg("%s", returninfstr());
		}
		if (wnp != NONODEPROTO)
		{
			for(pp = wnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				m = pp->userbits & STATEBITS;
				for(i=0; i<numtypes; i++) if (porttype[i] == m) break;
				if (i >= numtypes) continue;
				if (equivalentport(wnp, pp, np) == NOPORTPROTO)
					ttyputmsg("*** Port %s, found in facet %s, is missing here",
						pp->protoname, describenodeproto(wnp));
			}
		}
		efree((char *)pplist);
		return;
	}

	if (namesamen(pt, "primitives", l) == 0 && l >= 2)
	{
		/* list the primitive facet names */
		ttyputmsg("----- Primitive Node Prototypes (%s) -----", el_curtech->techname);
		maxlen = 0;
		for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			maxlen = (INTSML)maxi(maxlen, strlen(np->primname));
		maxlen += 2;  columns = MESSAGESWIDTH / maxlen;
		total = 0;
		for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if ((total % columns) == 0) (void)strcpy(line, np->primname); else
				(void)strcat(line, np->primname);
			if (np->nextnodeproto!=NONODEPROTO && (total%columns) != columns-1)
				for(i=strlen(np->primname); i<maxlen; i++)
					(void)strcat(line, " ");
			if ((total % columns) == columns-1) ttyputmsg(line);
			total++;
		}
		if ((total % columns) != 0) ttyputmsg(line);
		return;
	}

	if (namesamen(pt, "solvers", l) == 0 && l >= 2)
	{
		ttyputmsg("Current constraint solver is %s (%s)", el_curconstraint->conname,
			el_curconstraint->condesc);
		ttyputmsg("Other constraint solvers:");
		for(i=0; el_constraints[i].conname != 0; i++)
		{
			con = &el_constraints[i];
			if (con == el_curconstraint) continue;
			ttyputmsg("  %s (%s)", con->conname, con->condesc);
		}
		return;
	}

	if (namesamen(pt, "technologies", l) == 0 && l >= 1)
	{
		ttyputmsg("Current technology is %s", el_curtech->techname);
		ttyputmsg("----- Technologies -----");
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
			ttyputmsg("%-9s %s", tech->techname, tech->techdescript);
		return;
	}

	if (namesamen(pt, "usage", l) == 0 && l >= 1)
	{
		if (count <= 1)
		{
			us_abortcommand("Usage: show usage FACET");
			return;
		}
		np = getnodeproto(par[1]);
		if (np == NONODEPROTO || np->primindex != 0)
		{
			us_abortcommand("'%s' is not a facet", par[1]);
			return;
		}
		if (np->firstinst == NONODEINST)
		{
			ttyputmsg("Facet %s is not used anywhere", describenodeproto(np));
			return;
		}

		for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
			for(wnp = olib->firstnodeproto; wnp != NONODEPROTO; wnp = wnp->nextnodeproto)
				wnp->temp1 = 0;
		for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
			ni->parent->temp1++;
		ttyputmsg("Facet %s is used in these locations:", describenodeproto(np));
		for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
			for(wnp = olib->firstnodeproto; wnp != NONODEPROTO; wnp = wnp->nextnodeproto)
				if (wnp->temp1 != 0)
					ttyputmsg("  %ld %s in facet %s", wnp->temp1,
						makeplural("instance", wnp->temp1), describenodeproto(wnp));
		return;
	}

	if (namesamen(pt, "views", l) == 0 && l >= 1)
	{
		if (count == 1)
		{
			ttyputmsg("Current views:");
			for(v = el_views; v != NOVIEW; v = v->nextview)
				ttyputmsg("%s (short name %s)", v->viewname, v->sviewname);
			return;
		}
		np = getnodeproto(par[1]);
		if (np == NONODEPROTO || np->primindex != 0)
		{
			us_abortcommand("'%s' is not a facet", par[1]);
			return;
		}

		(void)initinfstr();
		first = 0;
		for(wnp = np->cell->firstincell; wnp != NONODEPROTO; wnp = wnp->nextincell)
		{
			if (first != 0) (void)addstringtoinfstr(", ");
			first++;
			(void)addstringtoinfstr(describenodeproto(wnp));
		}
		ttyputmsg("View contains: %s", returninfstr());
		return;
	}

	us_abortcommand("Bad SHOW option: %s", par[0]);
}

void us_size(INTSML count, char *par[])
{
	REGISTER INTSML l, cursorbased, isin, otherin, nodecount, arccount, serptrans, fun,
		justnodes, justarcs, fixedcorner, usetransformation, flipxy;
	REGISTER INTBIG i, j, k, dxs, dys, lx, hx, ly, hy, dist, bestdist, otherx, othery,
		rot, wid, dx, dy;
	INTBIG offxl, offyl, offxh, offyh, truewid, rx,ry, otheralign, xcur, ycur, xs, ys;
	XARRAY trans;
	REGISTER NODEINST *ni, *nodetosize;
	REGISTER ARCINST *ai;
	REGISTER NODEPROTO *np;
	REGISTER char *pp;
	REGISTER GEOM **list, *geom;
	GEOM *fakelist[2];
	REGISTER VARIABLE *var;
	static POLYGON *poly = NOPOLYGON;
	extern COMCOMP us_sizep, us_sizeyp, us_sizewp;
	HIGHLIGHT high;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	/* see if text on an invisible pin is selected (in which case, size the pin) */
	nodetosize = NONODEINST;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE && getlength(var) == 1)
	{
		(void)us_makehighlight(((char **)var->addr)[0], &high);
		if ((high.status&HIGHTYPE) == HIGHTEXT && high.fromvar != NOVARIABLE &&
			high.fromgeom->entrytype == OBJNODEINST)
		{
			ni = high.fromgeom->entryaddr.ni;
			if (ni->proto == gen_invispinprim) nodetosize = ni;
		}
	}

	/* get list of highlighted objects to be sized */
	if (nodetosize != NONODEINST)
	{
		list = fakelist;
		fakelist[0] = nodetosize->geom;
		fakelist[1] = NOGEOM;
	} else
	{
		list = us_gethighlighted(OBJARCINST|OBJNODEINST);
	}
	if (list[0] == NOGEOM)
	{
		us_abortcommand("Find object(s) to resize");
		return;
	}

	/* disallow sizing if lock is on */
	np = geomparent(list[0]);
	if (us_canedit(np, NONODEPROTO, 1) != 0) return;

	/* see how many are selected */
	nodecount = arccount = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype == OBJNODEINST)
		{
			nodecount++;
			nodetosize = list[i]->entryaddr.ni;
		} else
		{
			arccount++;
		}
	}

	/* see if a serpentine transistor is being scaled */
	serptrans = 0;
	if (nodecount == 1 && arccount == 0)
	{
		if (us_canedit(np, nodetosize->proto, 1) != 0) return;
		if (nodetosize->proto->primindex != 0 && (nodetosize->proto->userbits&HOLDSTRACE) != 0)
		{
			fun = (INTSML)((nodetosize->proto->userbits&NFUNCTION) >> NFUNCTIONSH);
			if (fun == NPTRANMOS || fun == NPTRADMOS || fun == NPTRAPMOS)
			{
				var = getvalkey((INTBIG)nodetosize, VNODEINST, VINTEGER|VISARRAY, el_trace);
				if (var != NOVARIABLE) serptrans = 1;
			}
		}
	}

	cursorbased = justnodes = justarcs = 0;
	while (count > 0)
	{
		l = strlen(pp = par[0]);
		if (namesamen(pp, "corner-fixed", l) == 0 && l >= 2)
		{
			cursorbased = 1;
			count--;
			par++;
			continue;
		}
		if (namesamen(pp, "center-fixed", l) == 0 && l >= 2)
		{
			cursorbased = 2;
			count--;
			par++;
			continue;
		}
		if (namesamen(pp, "grab-point-fixed", l) == 0)
		{
			us_abortcommand("Cannot size about the grab-point yet");
			return;
		}
		if (namesamen(pp, "nodes", l) == 0)
		{
			justnodes = 1;
			count--;
			par++;
			continue;
		}
		if (namesamen(pp, "arcs", l) == 0)
		{
			justarcs = 1;
			count--;
			par++;
			continue;
		}
		break;
	}

	if (count <= 0)
	{
		if ((nodecount > 1 || justnodes != 0) && justarcs == 0)
		{
			if (serptrans != 0)
				count = ttygetparam("Transistor width: ", &us_sizep, MAXPARS, par); else
					count = ttygetparam("X size: ", &us_sizep, MAXPARS, par);
			if (count == 0)
			{
				us_abortedmsg();
				return;
			}
			l = strlen(pp = par[0]);
		} else if ((arccount > 1 || justarcs != 0) && justnodes == 0)
		{
			count = ttygetparam("Width: ", &us_sizewp, MAXPARS, par);
			if (count == 0)
			{
				us_abortedmsg();
				return;
			}
			l = strlen(pp = par[0]);
		}
	}

	if ((nodecount != 0 || justnodes != 0) && justarcs == 0)
	{
		if (cursorbased != 0)
		{
			if (nodecount != 1 || arccount != 0)
			{
				us_abortcommand("Can only size one node at a time");
				return;
			}
			if (nodetosize->proto->primindex == 0)
			{
				us_abortcommand("Cannot change facet size");
				return;
			}

			if (serptrans != 0)
			{
				us_abortcommand("No cursor scaling on serpentine transistors");
				ttyputmsg("Use explicit width instead");
				return;
			}

			/* save highlighting */
			us_pushhighlight();
			us_clearhighlightcount();

			/* figure out the new size of the nodeinst */
			nodesizeoffset(nodetosize, &offxl, &offyl, &offxh, &offyh);
			lx = nodetosize->lowx+offxl;   hx = nodetosize->highx-offxh;
			ly = nodetosize->lowy+offyl;   hy = nodetosize->highy-offyh;

			if (cursorbased == 2)
			{
				/* center of node is fixed and all corners stretch */
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, us_alignment/2);

				/* adjust the cursor position if selecting interactively */
				if ((us_aid->aidstate&INTERACTIVE) != 0)
				{
					us_sizeinit(nodetosize);
					trackcursor(0, us_ignoreup, us_sizebegin, us_sizecdown,
						us_stopandpoponchar, us_dragup, TRACKDRAGGING);
					if (el_pleasestop != 0) return;
					if (us_demandxy(&xcur, &ycur) != 0) return;
					gridalign(&xcur, &ycur, us_alignment/2);
				}

				/* transform cursor to account for node orientation */
				makeangle((INTSML)((3600 - nodetosize->rotation)%3600), nodetosize->transpose, trans);
				xform(xcur-(hx+lx)/2, ycur-(hy+ly)/2, &rx, &ry, trans);
				xcur = (hx+lx)/2 + rx;
				ycur = (hy+ly)/2 + ry;

				/* compute new size of the nodeinst */
				j = (hx+lx)/2;   i = abs(j - xcur);
				lx = j-i;   hx = j+i;
				j = (hy+ly)/2;   i = abs(j - ycur);
				ly = j-i;   hy = j+i;
				lx -= offxl;   hx += offxh;
				ly -= offyl;   hy += offyh;
			} else if (cursorbased == 1)
			{
				/* closest corner of node stretches to cursor */
				/* adjust the cursor position if selecting interactively */
				fixedcorner = -1;
				if ((us_aid->aidstate&INTERACTIVE) != 0)
				{
					us_sizeinit(nodetosize);
					trackcursor(1, us_sizedown, us_sizebegin, us_sizedown,
						us_stopandpoponchar, us_dragup, TRACKDRAGGING);
					if (el_pleasestop != 0) return;
					if (us_demandxy(&xcur, &ycur) != 0) return;
					gridalign(&xcur, &ycur, us_alignment);
					fixedcorner = us_sizeterm();
				}
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, us_alignment);
				makerot(nodetosize, trans);

				/* determine which corner is fixed */
				if (fixedcorner < 0)
				{
					xform(lx, ly, &rx, &ry, trans);
					bestdist = abs(rx - xcur) + abs(ry - ycur);
					fixedcorner = 1;	/* lower-left */
					xform(hx, ly, &rx, &ry, trans);
					dist = abs(rx - xcur) + abs(ry - ycur);
					if (dist < bestdist)
					{
						bestdist = dist;   fixedcorner = 2;	/* lower-right */
					}
					xform(lx, hy, &rx, &ry, trans);
					dist = abs(rx - xcur) + abs(ry - ycur);
					if (dist < bestdist)
					{
						bestdist = dist;   fixedcorner = 3;	/* upper-left */
					}
					xform(hx, hy, &rx, &ry, trans);
					dist = abs(rx - xcur) + abs(ry - ycur);
					if (dist < bestdist) fixedcorner = 4;	/* upper-right */
				}

				/* check the lower-left corner */
				switch (fixedcorner)
				{
					case 1:		/* lower-left */
						otherx = hx;   othery = hy;   break;
					case 2:		/* lower-right */
						otherx = lx;   othery = hy;   break;
					case 3:		/* upper-left */
						otherx = hx;   othery = ly;   break;
					case 4:		/* upper-right */
						otherx = lx;   othery = ly;   break;
				}

				/* transform the cursor back through the node's orientation */
				xform(otherx, othery, &rx, &ry, trans);
				lx = mini(xcur, rx);
				ly = mini(ycur, ry);
				hx = maxi(xcur, rx);
				hy = maxi(ycur, ry);

				makeangle((INTSML)((3600 - nodetosize->rotation)%3600), nodetosize->transpose, trans);
				xform(lx-(hx+lx)/2, ly-(hy+ly)/2, &rx, &ry, trans);
				xform(hx-(hx+lx)/2, hy-(hy+ly)/2, &xs, &ys, trans);
				rx += (hx+lx)/2;   ry += (hy+ly)/2;
				xs += (hx+lx)/2;   ys += (hy+ly)/2;
				lx = mini(xs, rx) - offxl;
				ly = mini(ys, ry) - offyl;
				hx = maxi(xs, rx) + offxh;
				hy = maxi(ys, ry) + offyh;
			} else
			{
				/* grab-point fixed and all corners stretch */
				ttyputerr("Cannot do grab-point stretching yet");
				return;
			}

			/* make sure size is actually changing */
			if (lx == nodetosize->lowx && ly == nodetosize->lowy && hx == nodetosize->highx &&
				hy == nodetosize->highy)
			{
				ttyputverbose("Null node scaling");
				(void)us_pophighlight(0);
				return;
			}

			if ((nodetosize->proto->userbits&NSQUARE) != 0)
			{
				/* make sure the node is square */
				xs = hx - lx;   ys = hy - ly;
				if (xs != ys)
				{
					if (xs < ys)
					{
						lx = (lx + hx) / 2 - ys/2;   hx = lx + ys;
					} else
					{
						ly = (ly + hy) / 2 - xs/2;   hy = ly + xs;
					}
				}
			}

			/* modify the node */
			startobjectchange((INTBIG)nodetosize, VNODEINST);
			us_scaletraceinfo(nodetosize, lx, hx, ly, hy);
			modifynodeinst(nodetosize, lx-nodetosize->lowx, ly-nodetosize->lowy,
				hx-nodetosize->highx, hy-nodetosize->highy, 0, 0);
			endobjectchange((INTBIG)nodetosize, VNODEINST);

			/* adjust text descriptors on sized invisible pins */
			us_adjustdisplayabletext(nodetosize);

			/* restore highlighting */
			(void)us_pophighlight(1);
		} else
		{
			/* handle serpentine transistors specially */
			if (serptrans != 0)
			{
				i = atofr(pp);
				if (i <= 0)
				{
					us_abortcommand("Width must be positive");
					return;
				}

				/* save highlighting */
				us_pushhighlight();
				us_clearhighlightcount();

				/* size the node */
				startobjectchange((INTBIG)nodetosize, VNODEINST);
				(void)setvalkey((INTBIG)nodetosize, VNODEINST, el_transistor_width, i, VFRACT);
				endobjectchange((INTBIG)nodetosize, VNODEINST);

				/* restore highlighting */
				(void)us_pophighlight(1);
				return;
			}

			/* get the new X size */
			i = atola(pp);
			if (i&1) i++;
			if (i < 0)
			{
				us_abortcommand("X size must be positive");
				return;
			}

			/* get the new Y size */
			if (count <= 1)
			{
				if ((nodetosize->proto->userbits&NSQUARE) != 0) par[1] = pp; else
				{
					count = ttygetparam("Y size: ", &us_sizeyp, MAXPARS-1, &par[1]) + 1;
					if (count == 1)
					{
						us_abortedmsg();
						return;
					}
				}
			}
			j = atola(par[1]);
			if (j&1) j++;
			if (j < 0)
			{
				us_abortcommand("Y size must not be negative");
				return;
			}

			/* see if transformations should be considered */
			usetransformation = 0;
			if (count > 2 && namesamen(par[2], "use-transformation", l) == 0)
				usetransformation = 1;

			/* save highlighting */
			us_pushhighlight();
			us_clearhighlightcount();

			for(k=0; list[k] != NOGEOM; k++)
			{
				geom = list[k];
				if (geom->entrytype != OBJNODEINST) continue;
				nodetosize = geom->entryaddr.ni;
				if (nodetosize->proto->primindex == 0) continue;

				/* see if X and Y size factors should be flipped */
				flipxy = 0;
				if (usetransformation != 0)
				{
					rot = nodetosize->rotation;
					if (nodetosize->transpose != 0) rot = (rot + 900) % 3600;
					if (rot == 900 || rot == 2700) flipxy = 1;
				}

				nodesizeoffset(nodetosize, &offxl, &offyl, &offxh, &offyh);
				if (flipxy != 0)
				{
					if (pp[0] != 0) dy = i; else
					{
						dy = (nodetosize->highy - nodetosize->lowy) - offyl - offyh;
					}
					if (par[1][0] != 0) dx = j; else
					{
						dx = (nodetosize->highx - nodetosize->lowx) - offxl - offxh;
					}
				} else
				{
					if (pp[0] != 0) dx = i; else
					{
						dx = (nodetosize->highx - nodetosize->lowx) - offxl - offxh;
					}
					if (par[1][0] != 0) dy = j; else
					{
						dy = (nodetosize->highy - nodetosize->lowy) - offyl - offyh;
					}
				}
				if ((nodetosize->proto->userbits&NSQUARE) != 0 && dx != dy)
				{
					dx = dy = maxi(dx, dy);
					if (nodecount == 1 && arccount == 0)
						ttyputmsg("Warning: node must be square, making it %sx%s",
							latoa(dx), latoa(dy));
				}
				dxs = dx+offxl+offxh - (nodetosize->highx - nodetosize->lowx);
				dys = dy+offyl+offyh - (nodetosize->highy - nodetosize->lowy);
				if (dxs == 0 && dys == 0)
				{
					if (nodecount == 1 && arccount == 0)
						ttyputverbose("Null node scaling");
					continue;
				}

				/* size the node */
				startobjectchange((INTBIG)nodetosize, VNODEINST);
				us_scaletraceinfo(nodetosize, nodetosize->lowx-dxs/2, nodetosize->highx+dxs/2,
					nodetosize->lowy-dys/2, nodetosize->highy+dys/2);
				modifynodeinst(nodetosize, -dxs/2, -dys/2, dxs/2, dys/2, 0, 0);
				endobjectchange((INTBIG)nodetosize, VNODEINST);

				/* adjust text descriptors on sized invisible pins */
				us_adjustdisplayabletext(nodetosize);

				/* solve constraints */
				(*el_curconstraint->solve)(np);
			}

			/* restore highlighting */
			(void)us_pophighlight(1);
		}
	} else if ((arccount != 0 || justarcs != 0) && justnodes == 0)
	{
		if (cursorbased != 0 && (arccount != 1 || nodecount != 0))
		{
			us_abortcommand("Can only use cursor-based scaling on one arc");
			return;
		}

		/* save highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		for(j=0; list[j] != NOGEOM; j++)
		{
			geom = list[j];
			if (geom->entrytype != OBJARCINST) continue;
			ai = geom->entryaddr.ai;
			ai->end[0].nodeinst->temp1 = 0;
			ai->end[1].nodeinst->temp1 = 0;
		}

		for(j=0; list[j] != NOGEOM; j++)
		{
			geom = list[j];
			if (geom->entrytype != OBJARCINST) continue;
			ai = geom->entryaddr.ai;
			if (cursorbased != 0)
			{
				/* get location of cursor */
				if (us_demandxy(&xcur, &ycur))
				{
					(void)us_pophighlight(0);
					return;
				}
				gridalign(&xcur, &ycur, us_alignment/2);

				/* adjust the cursor position if selecting interactively */
				if ((us_aid->aidstate&INTERACTIVE) != 0)
				{
					us_sizeainit(ai);
					trackcursor(0, us_ignoreup, us_sizeabegin, us_sizeadown,
						us_stopandpoponchar, us_dragup, TRACKDRAGGING);
					if (el_pleasestop != 0) return;
					if (us_demandxy(&xcur, &ycur) != 0) return;
					gridalign(&xcur, &ycur, us_alignment/2);
				}

				/* figure out the new size of the arcinst (manhattan only) */
				if (ai->end[0].xpos == ai->end[1].xpos)
					wid = abs(xcur - ai->end[0].xpos) * 2; else
						wid = abs(ycur - ai->end[0].ypos) * 2;
			} else
			{
				wid = atola(pp);
				if (wid&1) wid++;
				if (wid < 0)
				{
					us_abortcommand("Width must not be negative");
					(void)us_pophighlight(0);
					return;
				}
			}
			truewid = arcwidthoffset(ai) + wid;
			if (truewid == ai->width)
			{
				ttyputverbose("Null arc scaling");
				continue;
			}

			/* handle edge alignment if possible */
			dxs = dys = 0;
			if (us_edgealignment != 0 && (ai->end[0].xpos == ai->end[1].xpos ||
				ai->end[0].ypos == ai->end[1].ypos))
			{
				if (ai->end[0].xpos == ai->end[1].xpos)
				{
					/* vertical arc */
					otherx = us_alignvalue(ai->end[0].xpos + wid/2, us_edgealignment, &otheralign) -
						wid/2;
					otheralign -= wid/2;
					isin = otherin = 0;
					for(i=0; i<2; i++)
					{
						shapeportpoly(ai->end[i].nodeinst, ai->end[i].portarcinst->proto, poly, 0);
						if (isinside(otherx, ai->end[i].ypos, poly) != 0) isin++;
						if (isinside(otheralign, ai->end[i].ypos, poly) != 0) otherin++;
					}
					if (isin == 2) dxs = otherx - ai->end[0].xpos; else
						if (otherin == 2) dxs = otheralign - ai->end[0].xpos;
				}
				if (ai->end[0].ypos == ai->end[1].ypos)
				{
					/* horizontal arc */
					othery = us_alignvalue(ai->end[0].ypos + wid/2,
						us_edgealignment, &otheralign) - wid/2;
					otheralign -= wid/2;
					isin = otherin = 0;
					for(i=0; i<2; i++)
					{
						shapeportpoly(ai->end[i].nodeinst, ai->end[i].portarcinst->proto, poly, 0);
						if (isinside(ai->end[i].xpos, othery, poly) != 0) isin++;
						if (isinside(ai->end[i].xpos, otheralign, poly) != 0) otherin++;
					}
					if (isin == 2) dys = othery - ai->end[0].ypos; else
						if (otherin == 2) dys = otheralign - ai->end[0].ypos;
				}
			}
			startobjectchange((INTBIG)ai, VARCINST);
			if (modifyarcinst(ai, truewid - ai->width, dxs, dys, dxs, dys) != 0)
				ttyputerr("Cannot set arc width to %s", latoa(truewid)); else
					endobjectchange((INTBIG)ai, VARCINST);

			/* scale the pins on either end */
			if (dxs == 0 && dys == 0) for(i=0; i<2; i++)
			{
				/* get a node that hasn't been scaled already */
				ni = ai->end[i].nodeinst;
				if (ni->temp1 != 0) continue;
				ni->temp1 = 1;

				/* make sure it is a pin */
				if (ni->proto->primindex == 0) continue;
				if (((ni->proto->userbits&NFUNCTION) >> NFUNCTIONSH) != NPPIN) continue;

				/* see if it is already large enough */
				nodesizeoffset(ni, &offxl, &offyl, &offxh, &offyh);
				lx = ni->lowx+offxl;   hx = ni->highx-offxh;
				ly = ni->lowy+offyl;   hy = ni->highy-offyh;
				if (hx - lx >= wid && hy - ly >= wid) continue;

				/* increase its size */
				if (hx - lx < wid) dx = (wid - (hx-lx)) / 2; else dx = 0;
				if (hy - ly < wid) dy = (wid - (hy-ly)) / 2; else dy = 0;
				startobjectchange((INTBIG)ni, VNODEINST);
				modifynodeinst(ni, -dx, -dy, dx, dy, 0, 0);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}

		/* restore highlighting */
		(void)us_pophighlight(1);
	}
}

void us_spread(INTSML count, char *par[])
{
	REGISTER NODEINST *ni, *no1, *no2, *inno;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai;
	INTBIG plx, ply, phx, phy;
	REGISTER INTBIG amount, xc1, yc1, xc2, yc2, i, slx, shx, sly, shy;
	REGISTER INTSML tot, doit, again, moved, mustbehor;
	static POLYGON *poly = NOPOLYGON;

	if (el_curconstraint != cla_constraint)
	{
		us_abortcommand("Must use the layout constraint system to spread");
		return;
	}

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	/* get the nodeinst about which things must spread */
	ni = (NODEINST *)us_getobject(OBJNODEINST, 0);
	if (ni == NONODEINST) return;
	nodesizeoffset(ni, &plx, &ply, &phx, &phy);
	slx = ni->lowx + plx;
	shx = ni->highx - phx;
	sly = ni->lowy + ply;
	shy = ni->highy - phy;

	/* disallow spreading if lock is on */
	if (us_canedit(ni->parent, NONODEPROTO, 1) != 0) return;

	/* get the direction of spread */
	if (count == 0)
	{
		us_abortcommand("Usage: spread DIRECTION [AMOUNT]");
		return;
	}
	if (*par[0] != 'u' && *par[0] != 'd' && *par[0] != 'r' && *par[0] != 'l')
	{
		us_abortcommand("Direction must be 'left', 'right', 'up', or 'down'");
		return;
	}

	/* get the amount to spread */
	if (count >= 2) amount = atola(par[1]); else
	{
		if (ni->proto->primindex != 0)
		{
			/* get design-rule surround around this node as spread amount */
			tot = nodepolys(ni);   amount = 0;
			for(i=0; i<tot; i++)
			{
				shapenodepoly(ni, (INTSML)i, poly);
				amount = maxi(amount, maxdrcsurround(ni->proto->tech, poly->layer));
			}

			/* also count maximum width of arcs connected to this nodeinst */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				if (pi->conarcinst == NOARCINST) continue;
				amount = maxi(amount, pi->conarcinst->width);
			}

			/* if all else fails, spread by "ni->proto->tech->deflambda" */
			if (amount < ni->proto->tech->deflambda) amount = ni->proto->tech->deflambda;
		} else
		{
			/* for facet, use facet size along spread axis */
			if (*par[0] == 'l' || *par[0] == 'r') amount = ni->highx - ni->lowx; else
				amount = ni->highy - ni->lowy;
		}
	}

	/* initialize by turning the marker bits off */
	for(inno = ni->parent->firstnodeinst; inno != NONODEINST; inno = inno->nextnodeinst)
		inno->userbits &= ~SPREADN;
	for(ai = ni->parent->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		ai->userbits &= ~SPREADA;

	/* set "already done" flag for nodes manhattan connected on spread line */
	if (*par[0] == 'l' || *par[0] == 'r') mustbehor = 0; else
		mustbehor = 1;
	us_manhattantravel(ni, mustbehor);

	/* set "already done" flag for nodes that completely cover spread node or are in its line */
	for(inno = ni->parent->firstnodeinst; inno != NONODEINST; inno = inno->nextnodeinst)
	{
		nodesizeoffset(inno, &plx, &ply, &phx, &phy);
		if (*par[0] == 'l' || *par[0] == 'r')
		{
			if (inno->lowx + plx < slx && inno->highx - phx > shx)
				inno->userbits |= SPREADN;
			if ((inno->lowx+inno->highx)/2 == (slx+shx)/2)
				inno->userbits |= SPREADN;
		} else
		{
			if (inno->lowy + ply < sly && inno->highy - phy > shy)
				inno->userbits |= SPREADN;
			if ((inno->lowy+inno->highy)/2 == (sly+shy)/2)
				inno->userbits |= SPREADN;
		}
	}

	/* mark those arcinsts that should stretch during spread */
	for(ai = ni->parent->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		no1 = ai->end[0].nodeinst;   no2 = ai->end[1].nodeinst;
		xc1 = (no1->lowx + no1->highx) / 2;   yc1 = (no1->lowy + no1->highy) / 2;
		xc2 = (no2->lowx + no2->highx) / 2;   yc2 = (no2->lowy + no2->highy) / 2;

		/* if one node is along spread line, make it "no1" */
		if ((no2->userbits&SPREADN) != 0)
		{
			inno = no1;  no1 = no2;  no2 = inno;
			i = xc1;     xc1 = xc2;  xc2 = i;
			i = yc1;     yc1 = yc2;  yc2 = i;
		}

		/* if both nodes are along spread line, leave arc alone */
		if ((no2->userbits&SPREADN) != 0) continue;

		i = 1;

		if ((no1->userbits&SPREADN) != 0)
		{
			/* handle arcs connected to spread line */
			switch (*par[0])
			{
				case 'l': if (xc2 <= slx) i = 0;   break;
				case 'r': if (xc2 >= shx) i = 0;  break;
				case 'u': if (yc2 >= shy) i = 0;  break;
				case 'd': if (yc2 <= sly) i = 0;   break;
			}
		} else
		{
			/* handle arcs that cross the spread line */
			switch (*par[0])
			{
				case 'l': if (xc1 > slx && xc2 <= slx) i = 0; else
					if (xc2 > slx && xc1 <= slx) i = 0;
					break;
				case 'r': if (xc1 < shx && xc2 >= shx) i = 0; else
					if (xc2 < shx && xc1 >= shx) i = 0;
					break;
				case 'u': if (yc1 > shy && yc2 <= shy) i = 0; else
					if (yc2 > shy && yc1 <= shy) i = 0;
					break;
				case 'd': if (yc1 < sly && yc2 >= sly) i = 0; else
					if (yc2 < sly && yc1 >= sly) i = 0;
					break;
			}
		}
		if (i == 0) ai->userbits |= SPREADA;
	}

	/* save highlighting */
	us_pushhighlight();
	us_clearhighlightcount();

	/* now look at every nodeinst in the facet */
	moved = 0;
	again = 1;
	while (again)
	{
		again = 0;
		for(inno = ni->parent->firstnodeinst; inno != NONODEINST; inno = inno->nextnodeinst)
		{
			if (stopping("Spread") != 0) break;

			/* ignore this nodeinst if it has been spread already */
			if ((inno->userbits & SPREADN) != 0) continue;

			/* make sure nodeinst is on proper side of requested spread */
			xc1 = (inno->highx+inno->lowx) / 2;
			yc1 = (inno->highy+inno->lowy) / 2;
			doit = 0;
			switch (*par[0])
			{
				case 'l': if (xc1 < slx)  doit++;   break;
				case 'r': if (xc1 > shx) doit++;   break;
				case 'u': if (yc1 > shy) doit++;   break;
				case 'd': if (yc1 < sly)  doit++;   break;
			}
			if (doit == 0) continue;

			/* set every connecting nodeinst to be "spread" */
			for(ai = ni->parent->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				if ((ai->userbits & SPREADA) != 0)
				{
					/* make arc temporarily unrigid */
					(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST,
						CHANGETYPETEMPUNRIGID, 0);
				} else
				{
					/* make arc temporarily rigid */
					(void)(*el_curconstraint->setobject)((INTBIG)ai, VARCINST,
						CHANGETYPETEMPRIGID, 0);
				}
			}
			us_nettravel(inno, SPREADN);

			/* move this nodeinst in proper direction to do spread */
			startobjectchange((INTBIG)inno, VNODEINST);
			switch(*par[0])
			{
				case 'l':
					modifynodeinst(inno, -amount, 0, -amount, 0, 0,0);
					break;
				case 'r':
					modifynodeinst(inno, amount, 0, amount, 0, 0,0);
					break;
				case 'u':
					modifynodeinst(inno, 0, amount, 0, amount, 0,0);
					break;
				case 'd':
					modifynodeinst(inno, 0, -amount, 0, -amount, 0,0);
					break;
			}
			endobjectchange((INTBIG)inno, VNODEINST);

			/* set loop iteration flag and node spread flag */
			moved++;
			again++;
			break;
		}
	}

	/* warn if nothing was done */
	if (moved == 0) ttyputverbose("Nothing changed");

	/* restore highlighting */
	(void)us_pophighlight(1);
}

void us_system(INTSML count, char *par[])
{
	REGISTER INTSML l;
	REGISTER char *pt;

	if (count == 0) l = strlen(pt = "x"); else
		l = strlen(pt = par[0]);

	if (namesamen(pt, "setstatusfont", l) == 0 && l >= 1)
	{
		setmessagesfont();
		return;
	}
#ifdef	WIN32
	if (namesamen(pt, "print", l) == 0 && l >= 1)
	{
		void gra_printwindow(void);

		gra_printwindow();
		return;
	}
	us_abortcommand("Usage: system print | setstatusfont");
#else
	us_abortcommand("Usage: system setstatusfont");
#endif
}
