/*
 * Electric(tm) VLSI Design System
 *
 * File: simspice.c
 * SPICE list generator: write a SPICE format file for the current facet
 * Written by: Steven M. Rubin, Static Free Software
 * Improved by: Sid Penstone, Queen's University
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 * Modified to get areas correctly by Oct.6/87 S.R.Penstone, Queen's U.
 * Pools diffusion areas (see notes below)
 * Revision Nov.12-- fixed error that applied mask scale factor twice
 * Revised Dec.2/87 SRP - separate out the diffusion and transistor types
 * Revision Dec.30/87, to leave out the poly gate on a transistor
 * Revision Mar.29/89 to declare atof() as double (SRP)
 * Revised June 6/89 to ask for cell name during parse-output operation (QU)
 * Revised June 6/89 to no longer look for X in 1st column of spice output (QU)
 * Revised Aug 31/89 to verify cell name during parse-output operation (QU)
 * Revised Aug 31/89 to look for layout view (QU)
 * Revised Nov 27/89 merged with version in Electric 4.04 (QU)
 * Revised Nov 28/89 merged with version from Chris Schneider at U. of Manitoba (QU)
 * Revised Mar 89 to support the use of node names, external model files,
 *    remote execution (UNIX), .NODESET and special sources, meters (fixed
 *    bug), varargs (UNIX), interactive use with spice2 spice3 or hspice
 *    simulators... by L. Swab (QU).  Some code by C. Schneider.
 * Revised June 7/90 to check for prototype ground net when exporting
 * subcircuit ports; if it is ground, do not export it. (SRP) This will
 * be looked at again.
 * Revised Dec.7/90 to start handling ARRAY types
 * MAJOR CHANGES Dec/90 SRP:
 * prepare to split source files into two files: simspice.c and simspicerun.c to
 * separate execution from writing
 * Create separate function to write 2 port types, so it can be called
 * Do not write  two terminal elements that are shorted out 91-1-30 SRP
 * Mar. 1991:  Include a substrate connection for bipolar transistors; use the
 * subnet defined by a substrate connection or default to ground.
 * Nov.18/91 Test the temp bits in the cell in case we
 * are doing an extraction from icons...
 * Nov. 29/91: Added option to write a trailer file, tech:~.SIM_spice_trailer_file
 * modified comment header to describe capabilities
 * Removed bug in writing substrate of bipolar transistors
 * 920113: Created default ground net in facets; output only the first
 * name of multiply-named networks; output a reference list of named arcs
 * Do not complain about posnet in subcircuits
 * SRP920603: Added option to include behavioral file for a facet, or a cell.
 * The variable "SPICE_behave_file" can be attached to the facet (in case it is
 * an icon, or to any facet in the cell on the nodeproto.
 * Changed the name of the behavior file variable to "sim_spice_behave_file"
 * SRP920604
 * We should only count diffusion connections to the network, to calculate
 * the drain and source areas correctly!
 * Removed trapdoor in sim_spice_arcarea() for well layers SRP920618:
 * Added function to look for diffusion arc function
 * Added check for isolated diffusion that will not be modelled correctly
 * Caught multiple polygons on different arc in sim_spice_evalpolygon()
 * Changed call to mrgdonefacet() and evalpolygon() to use (float) area
 * Changed conditions for error messages about wells and substrates (SRP)
 * Identify ground node even if the port is not flagged SRP920623
 * SRP920712: Changed storage in nets for diffarea[] to float as well
 *     "    Changed to write facet pins in numerical order
 * Fixed bug in sim_spice_nodearea() that didi not traverse the arcs of
 * a complex facet
 * Wrote temporary local version of decsribenetwork() to get around bug
 * in infinitstr() code
 * RLW920716: modified sim_spicewritefacet() to indicate the type of
 *     unconnected diffusion on a node
 * RLW920721: Changed error and warning messages to use the following
 *     descriptor convention (all mutli-line messages have the extra lines
 *     indented):
 *                      Warning: - general warning message
 *                      WARNING: - special warning message (i.e. not fully
 *                                  connected transistor)
 *                      ERROR:   - all error messages
 *            Changed some of the messages to use "ttyputerr()" instead of
 *              "ttyputmsg()"
 *
 * SRP921116: Changed warning messages in spice file to comments so that
 *      simulator can test file
 *
 * TADR20000805 Added dependent sources CCVS CCCS VCVS VCCS
 *
 * To properly simulate a facet, it must have the following (many of these
 * commands are found in the "spice.mac" command file):
 *
 * 1) Power and ground must be exports and explicitly connected to sources.
 *    You can do this with a power source (the primitive node prototype
 *    "schematic:source") which must be connected to power at the top and to
 *    ground at the bottom.  The source should then be parameterized to
 *    indicate the amount and whether it is voltage or current.  For example,
 *    to make a 5 volt supply, create a source node and use:
 *        setsource v "DC 5"
 *
 * 2) All input ports must be exports and connected to the positive side
 *    of sources (again, the "schematic:source" node).  Time values may be
 *    placed on the source nodes.  For example, if the source is to produce
 *    values that are 0 volts from 0 to 10NS and then 5 volts, use:
 *        setsource v "PWL(0NS 0 10NS 0 11NS 5)"
 *    constant input values can use the same form as for power and ground.
 *
 *    A .NODESET source can be created by connecting the positive end of the
 *    source to the appropriate node, and typing:
 *        setsource n VALUE
 *
 *    Special sources such as VCVS's can be created with the "special" source.
 *    The letter following the "s" in the parameterization will appear as
 *    the first letter in the deck.  For example, to create a VCVS, create
 *    a source and connect it to the approprite nodes.  Then parameterize it
 *    with
 *        setsource se "NC+ NC-"
 *    Where NC+ and NC- are the names of ports which specify the control
 *    voltages.  This would produce a spice card of the form:
 *        EX N+ N- NC+ NC-
 *    Where X is a unique name or number, and N+ and N- are the nodes which
 *    are connected to the source.
 *
 * 3) All values that are being watched must be exports and have meters
 *    placed on them.  The primitive nodeproto "schematic:meter" can be placed
 *    anywhere and it should have the top and bottom ports connected
 *    appropriately.  For the meter to watch voltage from 0 to 5 volts, use:
 *        setmeter "(0 5)"
 *    To watch the current through any voltage source, parameterize it in the
 *    usual way, but add the "m" option, eg.
 *        setsource vm ...
 *        setsource vdm ...
 *
 * 4) Determine the level of simulation by saying:
 *        setlevel 2
 *    This will extract the appropriate header from the technology.
 *    Alternately, you can specify a file to read the header info from
 *    with the command:
 *        variable set tech:~.SIM_spice_model_file FILENAME
 *
 * 5) Determine the type of analysis with another source node.  For transient
 *    analysis that displays half NS intervals, runs for 20NS, starts at time
 *    0, and uses internal steps of 0.1NS, create an unconnected source node
 *    and issue this statement:
 *        setsource vt ".5NS 20NS 0NS .1NS"
 *    For DC analysis you must connect the power side of the source node to
 *    the DC point and then use:
 *        setsource vd "0V 5V .1V"
 *    For AC analysis, create a source node and parameterize it with (eg.)
 *        setsource va "DEC 10 1 10K"
 *    There must be exactly one of these source nodes.
 *
 * 6) Define the spice format to use when producing the deck and parsing
 *    the output:
 *        telltool simulation spice format [2 | 3 | hspice]
 *
 * 7) Run spice with the command:
 *        ontool simulation
 *    This generates a deck and runs the simulator.  The results from .PRINT
 *    commands will be converted to a facet in the current library that
 *    contains a plot of the data.
 *
 * 8) You can also run SPICE on another machine (or otherwise outside of
 *    Electric).  To do this, supress SPICE execution with:
 *        telltool simulation not execute
 *    which will cause deck generation only.  Then run SPICE externally
 *    and convert the output listing to a plot with:
 *        telltool simulation spice parse-output FILE
 *
 * 9) You can replace the internal spice header file that is part of the
 *        technology by defining the variable "SIM_spice_model_file" on the
 *        technology, which is the name of a file. You can add a trailer to the
 *    spice deck by attaching a variable "SIM_spice_trailer_file" to the
 *    technology, that is the name of a file. These variables are most
 *    easily created in the Info.variables window, by clicking on
 *    "Current Technology", then "New Attribute", defining "SIM_spice_model_file"
 *    with the name of the file. Remember to click on "Set Attribute".
 *    Include a full path if the file will not be in the current directory at
 *    run time.
 *
 * 10) You can call up special bipolar and jfet models that are defined in
 *    your header file by including an appropriate string in the variable
 *    "SIM_spice_model" that is attached to transistors when they
 *    are created. When the 'Transistor size' window appears, enter the
 *    model name, ex: LARGE_NPN
 *        The extractor will use the string in the call in the
 *    spice file, ex:   Q1 3 4 0 0 LARGE_NPN
 *    (in fact, any string that does not include the character '/' will
 *    be used to describe the transistor in the spice file; you can
 *    use this to define the attributes of individual transistors in your
 *    circuit. The character '/' is reserved as a separator for length
 *    and width values, and if present, causes the default type to be
 *    invoked.)
 *
 * 11) You can use the contents of a file to replace the extracted description
 *    of any facet, by attaching the name of the file as a variable
 *    called "SIM_spice_behave_file" to the prototype.  The extractor will always
 *    use the file, if found, instead of extracting the facet network, but it will
 *    still extract any subfacets in the facet. These in turn could also be described by
 *    behavior files. If an icon or different view is used, it can have a
 *    different behavior file than the other views.
 *
 */

/*
 * Extraction notes: Layers on arcs and nodes that overlap are
 * merged into polygons, and the area and perimeter of the resulting polygon
 * is computed. Overlapping areas are thus eliminated. The resultinmg areas
 * are used to calculate the capacitance on the net, with the following special
 * treatment: If the node or arc has multiple layers, the layer that gives the
 * largest capacitance is left as the only active capacitance, and the other
 * layers have an their area equal to their area on this port of the node
 * removed.
 * BUT, if the node or arc has a diffusion layer, that layer is always assumed
 * dominant, and the area of the nondominant layers are subtracted from
 * their accumulated area. This is not quite correct, when the diffusion area
 * only partly covers the other areas.
 * The NPCAPAC nodes (series capacitors) are defined to have a dominant
 * diffusion layer, so that their nondiffusion layers are cancelled out. In
 * order to cancel out the perimeter capacity of a top-plate layer, there
 * should be an identical-sized layer with a nonzero area capacitance value,
 * and a negative edge capacitance value equal to that of the layer to be
 * cancelled out.
 * The diffusion areas are gathered up according to whether they are declared
 * as n-type, or p-type, or undefined. DMOS are assumed n-type. The number
 * of n-transistors and p-transistors on the net are counted up, and the
 * corresponding diffusion shared equally.
 * It is assumed that the technology file has correctly used the bits that
 * define layer functions.
 * MOS Transistors must have a correct labelling of the source and drain, or
 * there may be problems in the simulations if the wrong end is connected to
 * the substrate. In this extractor, the order of extraction will be gate,
 * source, gate, drain, based on the order of the ports in the technology file.
 * This will be correct for PMOS with the Vdd at the top. The extracted values
 * source and drain areas will correspond to this order.
 * pMOS-transistors as taken from the technology file prototype will have their
 * source at the top (more positive end), and nMOS-transistors taken in the
 * prototype position will have to be rotated to have their drain at the top.
 * Other device types will output collector, base, emitter, corresponding to
 * extraction of the first three ports from the prototype in the tech file.
 * Otherwise manual editing of the SPICE file is required.
 */

#include "config.h"
#if SIMTOOL

#include "global.h"
#include "sim.h"
#include "efunction.h"
#include "tecschem.h"
#include <math.h>

/* the spice node types returned by "sim_spice_nodetype" */
#define SPICEFACET     -1	/* sub-facet */
#define SPICEUNKNOWN    0	/* unimportant node (all connected) */
#define SPICENMOS       1	/* nMOS transistor */
#define SPICENMOS4      2	/* nMOS 4-port transistor */
#define SPICEDMOS       3	/* DMOS transistor */
#define SPICEDMOS4      4	/* DMOS 4-port transistor */
#define SPICEPMOS       5	/* PMOS transistor */
#define SPICEPMOS4      6	/* PMOS 4-port transistor */
#define SPICENPN        7	/* NPN transistor */
#define SPICENPN4       8	/* NPN 4-port transistor */
#define SPICEPNP        9	/* PNP transistor */
#define SPICEPNP4       10	/* PNP 4-port transistor */
#define SPICENJFET      11	/* N-JFET transistor */
#define SPICENJFET4     12	/* N-JFET 4-port transistor */
#define SPICEPJFET      13	/* P-JFET transistor */
#define SPICEPJFET4     14	/* P-JFET 4-port transistor */
#define SPICEDMES       15	/* Depletion MESFET transistor */
#define SPICEDMES4      16	/* Depletion MESFET 4-port transistor */
#define SPICEEMES       17	/* Enhancement MESFET transistor */
#define SPICEEMES4      18	/* Enhancement MESFET 4-port transistor */
#define SPICEREF        19	/* Self-referential transistor */
#define SPICEMETER      20	/* meter */
#define SPICESOURCEV    21	/* voltage source */
#define SPICESOURCEC    22	/* current source */
#define SPICESOURCECM   23	/* current meter */
#define SPICESOURCET    24	/* transient analysis specification */
#define SPICESOURCEDC   25	/* DC analysis specification */
#define SPICESOURCEAC   26	/* AC analysis specification */
#define SPICESOURCEN    27	/* NodeSet source */
#define SPICESOURCEX    28	/* Extension source */
#define SPICESOURCEB    29	/* Bulk source */
#define SPICESOURCES    30	/* Special source */
#define SPICERESISTOR   31	/* resistor */
#define SPICECAPACITOR  32	/* capacitor */
#define SPICEINDUCTOR   33	/* inductor */
#define SPICEDIODE      34	/* diode */
#define SPICEDIODEZ     35	/* Zener diode */
#define SPICEARRAY      36	/* Multiport/function device */
#define SPICESUBSTRATE  37	/* connection to substrate */
#define SPICETRANS      38	/* generic (special) transistor */
#define SPICEGROUND     39	/* A node connected to ground */
#define SPICEPOWER      40	/* A node connected to power */
#define SPICECCCS       41	/* Current Controlled Current Source */
#define SPICECCVS       42	/* Current Controlled Voltage Source */
#define SPICEVCCS       43	/* Voltage Controlled Current Source */
#define SPICEVCVS       44	/* Voltage Controlled Voltage Source */
#define SPICETLINE      45	/* Transmission Line */

/* Types of spice sources */
#define SPICENORMAL     0	/* normal source */
#define SPICEAC	        1	/* .AC card */
#define SPICEDC	        2	/* .DC card */
#define SPICETRAN       3	/* .TRAN card */
#define SPICENODESET    4	/* .NODESET card */
#define SPICEBULK       5	/* bulk potential */
#define SPICEEXTENSION  6	/* extension CS901002 */

#define DIFFTYPES       3	/* Types of diffusions & transistors plus 1 */
#define ISNONE	        0
#define ISNTYPE	        1
#define ISPTYPE	        2

static float    sim_spice_min_resist;					/* spice minimum resistance */
static float    sim_spice_min_capac;					/* spice minimum capacitance */
static char    *sim_spice_ac;							/* AC analysis message */
static char    *sim_spice_dc;							/* DC analysis message */
static char    *sim_spice_tran;							/* Transient analysis message */
static POLYGON *sim_polygonfree = NOPOLYGON;			/* list of free simulation polygons */
static float    sim_spice_mask_scale = 1.0;				/* Mask shrink factor (default =1) */
static float   *sim_spice_extra_area = 0;				/* Duplicated area on each layer */
static INTSML   sim_spice_diffusion_index[DIFFTYPES];	/* diffusion layers indices */
static INTSML   sim_spice_layer_count;
static INTBIG   sim_spice_unnamednum;
       INTBIG   sim_spice_level;						/* key for "SIM_spice_level" */
       INTBIG   sim_spice_state;						/* key for "SIM_spice_state" */
       INTSML   spice_state;							/* value of "SIM_spice_state" */
       INTSML   sim_spice_debug = 0;					/* SPICE debugging */
       INTBIG   sim_listingfile;						/* key for "SIM_listingfile" */
       char	    sim_deck_facetname[100] = "";			/* Name extracted from .SPO file */
static char    *sim_spice_printcard;					/* the .PRINT or .PLOT card */

/* working memory for "sim_spice_edge_capacitance()" */
static INTBIG *sim_spice_capacvalue = 0;

/******************** SPICE NET QUEUE ********************/

#define NOSPNET   ((SPNET *)-1)

typedef struct Ispnet
{
	NETWORK       *network;					/* network object associated with this */
	INTBIG         netnumber;				/* internal unique net number */
	float          diffarea[DIFFTYPES];		/* area of diffusion */
	float          diffperim[DIFFTYPES];	/* perimeter of diffusion */
	float          resistance;				/* amount of resistance */
	float          capacitance;				/* amount of capacitance */
	INTSML         components[DIFFTYPES];	/* number of components connected to net */
	struct Ispnet *nextnet;					/* next in linked list */
} SPNET;

static SPNET *sim_spice_firstnet;					/* first in list of nets in this facet */
static SPNET *sim_spice_netfree = NOSPNET;			/* list of free nets */
static SPNET *sim_spice_power;						/* net of power */
static SPNET *sim_spice_ground;						/* net of ground */
static SPNET *sim_spice_cur_net;					/* for polygon merging */

/* prototypes for local routines */
static POLYGON   *sim_allocpolygon(void);
static void       sim_freepolygon(POLYGON*);
static SPNET     *sim_allocspnet(void);
static void       sim_freespnet(SPNET*);
static void       sim_spice_writeheader(CELL*, FILE*, TECHNOLOGY*);
static INTSML     sim_spicewritefacet(NODEPROTO*, FILE*, INTSML);
static INTSML     sim_spice_nodetype(NODEINST*);
static SPNET     *sim_spice_getnet(NODEINST*, NETWORK*);
static void       sim_spice_nodearea(SPNET*, NODEINST*, PORTPROTO*);
static void       sim_spice_arcarea(SPNET*, ARCINST*);
static void       sim_spice_writeports(FILE*, NODEPROTO*);
static char      *sim_spice_element_name(NODEINST*, char, INTBIG*);
static char      *sim_spice_node_name(SPNET*);
static void       sim_spice_xputs(char*, FILE*);
static char      *sim_spice_describesource(NODEINST*);
static void       sim_spice_store_box(TECHNOLOGY*, INTSML, INTBIG, INTBIG, INTBIG, INTBIG);
static void       sim_spice_evalpolygon(INTSML, TECHNOLOGY*, INTBIG*, INTBIG*, INTSML);
static INTSML     sim_spice_layerisdiff(TECHNOLOGY*, INTSML);
static INTSML     sim_spice_arcisdiff(ARCINST*);
static float      sim_spice_capacitance(TECHNOLOGY*, INTSML);
static float      sim_spice_edge_capacitance(TECHNOLOGY*, INTSML);
static NODEPROTO *sim_spice_topview(NODEPROTO*);
static void       sim_spice_write_twoport(FILE*, NODEINST*, INTBIG, char*, INTBIG*, INTBIG);
static void       sim_spice_writetrailer(CELL*, FILE*, TECHNOLOGY*);
static char      *sim_spice_net_name(NETWORK *net);
static INTBIG     sim_spiceisexported(SPNET *spnet, NODEPROTO *np);

/*
 * Routine to free all memory associated with this module.
 */
void sim_freespicememory(void)
{
	REGISTER POLYGON *poly;
	REGISTER SPNET *spnet;

	while (sim_polygonfree != NOPOLYGON)
	{
		poly = sim_polygonfree;
		sim_polygonfree = sim_polygonfree->nextpolygon;
		freepolygon(poly);
	}

	while (sim_spice_netfree != NOSPNET)
	{
		spnet = sim_spice_netfree;
		sim_spice_netfree = sim_spice_netfree->nextnet;
		efree((char *)spnet);
	}
	if (sim_spice_capacvalue != 0) efree((char *)sim_spice_capacvalue);
}

/******************** SPICE DECK GENERATION ********************/

/*
 * procedure to write a spice deck to describe facet "np"
 */
void sim_writespice(NODEPROTO *np)
{
	FILE *spfile;
	REGISTER INTSML retval;
	REGISTER INTBIG analysiscards;
	char deckfile[256], *pt;
	REGISTER VARIABLE *var;
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *onp;
	REGISTER TECHNOLOGY *tech, *t;

	/* make sure network tool is on */
	if ((net_tool->toolstate&TOOLON) == 0)
	{
		ttyputerr(_("Network tool must be running...turning it on"));
		toolturnon(net_tool, 0);
		ttyputerr(_("...now reissue the simulation command"));
		return;
	}

	if (np == NONODEPROTO)
	{
		ttyputerr(_("Must have a facet to edit"));
		return;
	}
	sim_simnt = np;

	/* use the current technology */
	tech = el_curtech;
	sim_spice_layer_count = tech->layercount;

	/* get the SPICE state */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_state);
	if (var != NOVARIABLE) spice_state = (INTSML)var->addr; else spice_state = 0;

	/* get the overall minimum resistance and capacitance */
	var = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT, "SIM_spice_min_resistance");
	if (var != NOVARIABLE) sim_spice_min_resist = castfloat(var->addr); else
		sim_spice_min_resist = 0.0;
	var = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT, "SIM_spice_min_capacitance");
	if (var != NOVARIABLE) sim_spice_min_capac = castfloat(var->addr); else
		sim_spice_min_capac = 0.0;
	var = getval((INTBIG)tech, VTECHNOLOGY, VFLOAT, "SIM_spice_mask_scale");
	if (var != NOVARIABLE) sim_spice_mask_scale = castfloat(var->addr); else
		sim_spice_mask_scale = 1.0;
	sim_spice_extra_area = (float *)emalloc(sizeof(float) * sim_spice_layer_count,
		sim_tool->cluster);

	/* get the layer resistance and capacitance arrays for each technology */
	for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
	{
		var = getval((INTBIG)t, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_resistance");
		t->temp1 = (var == NOVARIABLE ? 0 : var->addr);
		var = getval((INTBIG)t, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_capacitance");
		t->temp2 = (var == NOVARIABLE ? 0 : var->addr);
	}

	/* start writing the spice deck */
	(void)strcpy(deckfile, np->cell->cellname);
	(void)strcat(deckfile, ".spi");
	pt = deckfile;
	spfile = xcreate(deckfile, sim_filetypespice, _("SPICE File"), &pt);
	if (pt != 0) (void)strcpy(deckfile, pt);
	if (spfile == NULL)
	{
		ttyputerr(_("Cannot create SPICE file: %s"), deckfile);
		return;
	}

	/* write header to spice deck */
	(void)sim_spice_writeheader(np->cell, spfile, tech);

	/* reset bits on all facets */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
	{
		onp->temp1 = 0;
		onp->cell->temp1 = 0;
	}
	sim_spice_unnamednum = 1;

	/*
	 * initialize the .PRINT or .PLOT card.
	 * As we find sources/meters, we will tack things onto this string
	 * and 'reallocstring' it.
	 * It is printed and then freed in sim_spice_writefacet.
	 */
	if (allocstring(&sim_spice_printcard, "", sim_tool->cluster)) return;

	/* we don't know the type of analysis yet... */
	sim_spice_ac = sim_spice_dc = sim_spice_tran = NULL;

	/* initialize the polygon merging system */
	mrginit();

	/* recursively write all facets below this one */
	retval = sim_spicewritefacet(np, spfile, 1);
	if (retval < 0) ttyputnomemory();
	if (retval > 0)
		ttyputmsg(_("Back-annotation information has been added (library must be saved)"));

	/* handle AC, DC, and TRAN analysis cards */
	analysiscards = 0;
	if (sim_spice_dc != NULL) analysiscards++;
	if (sim_spice_tran != NULL) analysiscards++;
	if (sim_spice_ac != NULL) analysiscards++;
	if (analysiscards == 0)
		ttyputerr(_("Warning: should have a DC, Transient or AC source node"));
	if (analysiscards > 1)
		ttyputerr(_("Warning: can only have one DC, Transient or AC source node"));
	if (sim_spice_tran != NULL)
	{
		sim_spice_xprintf(spfile, "%s\n", sim_spice_tran);
		efree(sim_spice_tran);
	} else if (sim_spice_ac != NULL)
	{
		sim_spice_xprintf(spfile, "%s\n", sim_spice_ac);
		efree(sim_spice_ac);
	} else if (sim_spice_dc != NULL)
	{
		sim_spice_xprintf(spfile, "%s\n", sim_spice_dc);
		efree(sim_spice_dc);
	}

	sim_spice_writetrailer(np->cell, spfile, tech);
	sim_spice_xprintf(spfile, ".END\n");
	ttyputmsg(_("%s written"), deckfile);
	xclose(spfile);
	efree((char *)sim_spice_extra_area);

	/* finish polygon merging subsystem */
	mrgterm();

	/* run spice (if requested) */
	var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_dontrun);
	if (var != NOVARIABLE && var->addr != SIMRUNNO)
	{
		ttyputmsg(_("Running SPICE..."));
		var = getvalkey((INTBIG)sim_tool, VTOOL, VSTRING, sim_listingfile);
		if (var == NOVARIABLE) sim_spice_execute(deckfile, ""); else
			sim_spice_execute(deckfile, (char *)var->addr);
	}
}

/*
 * routine to write facet "np" and its all referenced subnodeprotos in SPICE
 * format to file "f".  If "top" is nonzero, this is the top facet.  The
 * spice level is "tool:sim.SIM_spice_level" and "sim_spice_min_resist" and
 * "sim_spice_min_capac" are the minimum required resistance and capacitance.
 * Returns negative on error, positive if back annotation was added.
 */
INTSML sim_spicewritefacet(NODEPROTO *np, FILE *f, INTSML top)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *subnt, *tnp, *cnp;
	REGISTER PORTPROTO *pp, *posport, *negport, *tpp, *otpp, *biasport;
	PORTPROTO *gate, *source, *drain, *gatedummy;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai;
	REGISTER SPNET *spnet, *nspnet, *gaten, *gatedn, *sourcen, *drainn, *posnet, *negnet,
		*subnet, *tempspnet, *biasn;
	float a, b;
	REGISTER INTSML backannotate;
	REGISTER INTBIG netindex, state, j, first, found, meterit, i, nodetype, retval;
	INTBIG subfacetindex, nodeindex, resistnum, capacnum, inductnum, diodenum;
	REGISTER VARIABLE *var;
	REGISTER NETWORK *net;
	char *extra, *info, line[100];
	REGISTER char sourcetype;
	INTBIG lx, ly;
	INTBIG sim_spice_sourceindex, bipolars, nmostrans, pmostrans;
	char *dummy, c, *filename;
	FILE *modelfile;
	char *uncon_diff_type;

	/* stop if requested */
	if (stopping(STOPREASONDECK)) return(-1);

	/* look for a model file on the current facet */
	var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_spice_behave_file");

	/* if this facet's view has a behavioral model attached, then use the facet */
	if (var == NOVARIABLE)
	{
		/* if not, try to use view in same view as top-level facet */
		np = sim_spice_topview(np);
	}

	/* first pass through the node list */
	/* if there are any sub-facets that have not been written, write them */
	backannotate = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		subnt = ni->proto;
		if (subnt->primindex != 0) continue;

		/* ignore recursive references (showing icon in contents) */
		if (subnt->cell == np->cell) continue;

		/* use the contents view if this is an icon */
		if (subnt->cellview == el_iconview)
		{
			cnp = contentsview(subnt);
			if (cnp != NONODEPROTO) subnt = cnp;
		}

		/* make sure the subfacet hasn't already been written */
		if (subnt->temp1 != 0) continue;
		if (subnt->cell->temp1 != 0) continue;
		retval = sim_spicewritefacet(subnt, f, 0);
		if (retval < 0) return(-1);
		if (retval > 0) backannotate = 1;
	}

	/* make sure that all nodes have names on them */
	if (asktool(net_tool, "name-nodes", (INTBIG)np) != 0) backannotate++;
	if (asktool(net_tool, "name-nets", (INTBIG)np) != 0) backannotate++;

	/* mark this facet as written */
	np->temp1 = 1;
	np->cell->temp1 = 1;

	/* look on the current view of the cell for models */
	var = getval((INTBIG)np, VNODEPROTO, VSTRING, "SIM_spice_behave_file");
	if (var != NOVARIABLE)
	{
		modelfile = xopen((char *)var->addr, sim_filetypespice,  el_libdir, &filename);
		if (modelfile != NULL)
		{
			/* copy file */
			c = (char)xgetc(modelfile);
			while (!xeof(modelfile))
			{
				(void)xputc(c, f);
				c = (char)xgetc(modelfile);
			}
			return(backannotate);
		} else ttyputerr(_("Warning: cannot open model file: %s"), (char *)var->addr);
	}

	/* zero the temp1 for arcs and nodes to compute area */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst) ai->temp1 = 0;
	for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork) net->temp1 = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = ni->temp2 = 0;
	bipolars = nmostrans = pmostrans = 0;

	/* create linked list of electrical nets in this facet */
	sim_spice_firstnet = NOSPNET;
	sim_spice_power = sim_spice_ground = NOSPNET;

	/* must have a default node 0 in subfacets */
	nodeindex = subfacetindex = 1;
	netindex = 2;	   /* save 1 for the substrate */

	/* look at all arcs in the facet */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		/* don't count non-electrical arcs */
		if (((ai->proto->userbits&AFUNCTION)>>AFUNCTIONSH) == APNONELEC) continue;

		/* see if this arc has the same net as another arc */
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
			if (spnet->network == ai->network) break;
		if (spnet == NOSPNET)
		{
			/* add this net to the list */
			spnet = sim_allocspnet();
			if (spnet == NOSPNET) return(-1);
			spnet->network = ai->network;
			ai->network->temp1 = (INTBIG)spnet;
			spnet->netnumber = netindex++;

			/* reset */
			for (j = 0; j < sim_spice_layer_count; j++) sim_spice_extra_area[j] = 0.0;
			for (j = 0; j < DIFFTYPES; j++) sim_spice_diffusion_index[j] = -1;
			spnet->nextnet = sim_spice_firstnet;
			sim_spice_firstnet = spnet;

			if (sim_spice_debug != 0) ttyputmsg(M_("Starting net %ld"), spnet->netnumber);
			sim_spice_arcarea(spnet, ai); /* builds areas, etc */

			/* get merged polygons so far */
			sim_spice_cur_net = spnet;
			mrgdonefacet(sim_spice_evalpolygon);
		}
	}

	/* now add any unwired but exported ports to the list of nets */
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* determine the net number of the export */
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
			if (spnet->network == pp->network) break;
		if (spnet != NOSPNET)
		{
			if (portispower(pp) != 0) sim_spice_power = spnet;
			if (portisground(pp) != 0)
			{
				sim_spice_ground = spnet;
				spnet->netnumber = 0;
			}
		} else
		{
			/* add this net to the list */
			spnet = sim_allocspnet();
			if (spnet == NOSPNET) return(-1);
			spnet->network = pp->network;
			pp->network->temp1 = (INTBIG)spnet;
			if (portisground(pp) != 0)
			{
				spnet->netnumber = 0;
				sim_spice_ground = spnet;
			} else spnet->netnumber = netindex++;
			if (portispower(pp) != 0) sim_spice_power = spnet;
			ni = pp->subnodeinst;

			/* reset */
			for (j = 0; j < sim_spice_layer_count; j++) sim_spice_extra_area[j] = 0.0;
			for (j = 0; j < DIFFTYPES; j++) sim_spice_diffusion_index[j] = -1;
			sim_spice_nodearea(spnet, ni, pp->subportproto);

			/* get merged polygons so far */
			sim_spice_cur_net = spnet;
			mrgdonefacet(sim_spice_evalpolygon);

			/* count the number of components on the net */
			state = sim_spice_nodetype(ni);
			switch (state)
			{
				case SPICEEMES:
				case SPICEEMES4:
				case SPICEDMES:
				case SPICEDMES4:
				case SPICEDMOS:
				case SPICEDMOS4:
				case SPICENMOS:
				case SPICENMOS4: nodetype = ISNTYPE; break;
				case SPICEPMOS:
				case SPICEPMOS4: nodetype = ISPTYPE; break;
				default:         nodetype = ISNONE;  break;
			}

			/* only count drains and source connections in counting transistors */
			if (nodetype == ISNONE)
				spnet->components[nodetype]++; /* We don't use this, anyhow */
			else
			{
				transistorports(ni, &gate, &gatedummy, &source, &drain);
				if (pp->subportproto == source || pp->subportproto == drain)
				{
					spnet->components[nodetype]++;
				}
			}
			spnet->nextnet = sim_spice_firstnet;
			sim_spice_firstnet = spnet;
		}
	}

	/* create Spice net information for all remaining nets in the facet */
	for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
	{
		if (net->temp1 != 0) continue;

		/* add this net to the list */
		spnet = sim_allocspnet();
		if (spnet == NOSPNET) return(-1);
		net->temp1 = (INTBIG)spnet;
		spnet->network = net;
		spnet->netnumber = netindex++;
		spnet->nextnet = sim_spice_firstnet;
		sim_spice_firstnet = spnet;
	}

	posnet = negnet = subnet = NOSPNET;

	/* second pass through the node list */ 
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		state = sim_spice_nodetype(ni);
		if (state == SPICESOURCEB)
		{
			posport = ni->proto->firstportproto;
			negport = posport->nextportproto;
			posnet = sim_spice_getnet(ni, posport->network);
			negnet = sim_spice_getnet(ni, negport->network);
			if (posnet == NOSPNET || negnet == NOSPNET)
				ttyputerr(_("Warning: %s component not properly connected"),
					describenodeinst(ni));
			continue;
		}
		if (state == SPICEUNKNOWN || state == SPICEMETER || state == SPICESOURCEV ||
			state == SPICESOURCEC || state == SPICESOURCECM || state == SPICESOURCET ||
			state == SPICESOURCEDC || state == SPICESOURCEAC || state == SPICESOURCEN ||
			state == SPICESOURCEX || state == SPICESOURCES || state == SPICECCCS ||
			state == SPICECCVS || state == SPICEVCCS || state == SPICEVCVS) continue;
		if (state == SPICESUBSTRATE)
		{
			if (subnet == NOSPNET)
				subnet = sim_spice_getnet(ni, ni->proto->firstportproto->network);
			continue;
		}
		if (state == SPICEPOWER)
		{
			tempspnet = sim_spice_getnet(ni, ni->proto->firstportproto->network);
			if (tempspnet == NOSPNET) continue;
			if (sim_spice_power == NOSPNET) sim_spice_power = tempspnet; else
			{
				if (tempspnet->network != sim_spice_power->network)
					ttyputerr(_("Warning: there may be inconsistent power networks in %s"),
						describenodeproto(np));
			}
		}
		if (state == SPICEGROUND)
		{
			tempspnet = sim_spice_getnet(ni, ni->proto->firstportproto->network);
			if (tempspnet == NOSPNET) continue;
			tempspnet->netnumber = 0;
			if (sim_spice_ground == NOSPNET) sim_spice_ground = tempspnet; else
			{
				if (tempspnet->network != sim_spice_ground->network)
					ttyputerr(_("Warning: there may be inconsistent ground networks in %s"),
						describenodeproto(np));
			}
		}
		switch (state)
		{
			case SPICENPN:
			case SPICENPN4:
			case SPICEPNP:
			case SPICEPNP4:
			case SPICETRANS:
				nodetype = ISNONE;
				bipolars++;
				break;
			case SPICEEMES:
			case SPICEEMES4:
			case SPICEDMES:
			case SPICEDMES4:
			case SPICEDMOS:
			case SPICEDMOS4:
			case SPICENMOS:
			case SPICENMOS4:
				nodetype = ISNTYPE;
				nmostrans++;
				break;
			case SPICEPMOS:
			case SPICEPMOS4:
				nodetype = ISPTYPE;
				pmostrans++;
				break;
			default:
				nodetype = ISNONE;
				break;
		}

		/*
		 * find all wired ports on component and increment their count,
		 * but only if they are a drain or source
		 */
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			ai = pi->conarcinst;
			for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
			{
				if (spnet->network == ai->network)
				{
					if (nodetype == ISNONE) spnet->components[nodetype]++; else
					{
						transistorports(ni, &gate, &gatedummy, &source, &drain);
						if (pi->proto == source || pi->proto == drain)
						{
							spnet->components[nodetype]++;
						}
					}
				}
			}
		}
	}

	/* use ground net for substrate */
	if (subnet == NOSPNET) subnet = sim_spice_ground;

	if (sim_spice_power == NOSPNET && top)
	{
		ttyputerr(_("Warning: cannot find power in the circuit"));
		ttyputmsg(_("     Must have power export or use a 'DC' source node"));
	}
	if (sim_spice_ground == NOSPNET && top)
	{
		ttyputerr(_("Warning: cannot find ground in the circuit"));
		ttyputmsg(_("     Must have ground export or use a 'DC' source node"));
	}

	if (pmostrans != 0 && posnet == NOSPNET)
	{
		if (sim_spice_power == NOSPNET)
		{
			ttyputerr(_("Warning: no power connection for P-transistor wells in %s"),
				describenodeproto(np));
		} else posnet = sim_spice_power;
	}
	if (nmostrans != 0 && negnet == NOSPNET)
	{
		if (sim_spice_ground == NOSPNET)
		{
			ttyputerr(_("Warning: no connection for N-transistor wells in %s"),
				describenodeproto(np));
		} else negnet = sim_spice_ground;
	}

	if (bipolars != 0 && subnet == NOSPNET)
	{
		ttyputerr(_("Warning: no explicit connection to the substrate in cell %s"),
			describenodeproto(np));
		ttyputmsg(_("     A connection to ground will be assumed if necessary."));
		subnet = sim_spice_ground;
	}

	/* generate header for subckt or top-level facet */
	if (!top)
	{
		sim_spice_xprintf(f, "\n.SUBCKT %s", np->cell->cellname);
		sim_spice_writeports(f, np);
		sim_spice_xprintf(f, "\n");
	} else
		sim_spice_xprintf(f, "\n*** TOP LEVEL FACET: %s\n", describenodeproto(np));

	/* generate pin descriptions for reference (when not using node names) */
	/* report on the ground network */
	if (sim_spice_power != NOSPNET && sim_spice_power->network != NONETWORK)
		sim_spice_xprintf(f, "** POWER NET: %s\n",
			describenetwork(sim_spice_power->network));
	if (sim_spice_ground != NOSPNET && sim_spice_ground->network != NONETWORK)
		sim_spice_xprintf(f, "** GROUND NET: 0 (%s)\n",
			describenetwork(sim_spice_ground->network));

	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* do not write the ground net */
		if (sim_spice_ground != NOSPNET && sim_spice_ground->network == pp->network)
			continue;
		if (pp->network->signals > 1)
		{
			/* a bus: write each part */
			for(i=0; i<pp->network->signals; i++)
			{
				net = pp->network->networklist[i];

				/* skip if already mentioned */
				for(tpp = np->firstportproto; tpp != pp; tpp = tpp->nextportproto)
				{
					if (net == tpp->network) break;
					if (tpp->network->signals > 1)
					{
						for(j=0; j<tpp->network->signals; j++)
							if (tpp->network->networklist[j] == net) break;
						if (j < tpp->network->signals) break;
					}
				}
				if (tpp != pp) continue;

				sim_spice_xprintf(f, "** PORT %s", sim_spice_net_name(net));
				if ((spice_state&SPICENODENAMES) == 0 && net->namecount > 0)
					sim_spice_xprintf(f, " (network: %s)", describenetwork(net));
				sim_spice_xprintf(f, "\n");
			}
		} else
		{
			/* skip if already mentioned */
			for(tpp = np->firstportproto; tpp != pp; tpp = tpp->nextportproto)
			{
				if (pp->network == tpp->network) break;
				if (tpp->network->signals > 1)
				{
					for(j=0; j<tpp->network->signals; j++)
						if (tpp->network->networklist[j] == pp->network) break;
					if (j < tpp->network->signals) break;
				}
			}
			if (tpp != pp) continue;

			/* write the net name */
			sim_spice_xprintf(f, "** PORT %s", sim_spice_net_name(pp->network));
			if ((spice_state&SPICENODENAMES) == 0 && pp->network->namecount > 0)
				sim_spice_xprintf(f, " (network: %s)", describenetwork(pp->network));
			sim_spice_xprintf(f, "\n");
		}
	}

	/* now run through all components in the facet */
	resistnum = capacnum = inductnum = diodenum = 1;

	/* third pass through the node list, print it this time */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		/* get the type of this node */
		state = sim_spice_nodetype(ni);

		/* handle sub-facet calls */
		if (state == SPICEFACET)
		{
			/* ignore recursive references (showing icon in contents) */
			if (ni->proto->cell == np->cell) continue;

			sim_spice_xputs(sim_spice_element_name(ni, 'X', &subfacetindex), f);

			/*
			 * We must be consistent in the ordering of ports.  However,
			 * subfacets may be another view (eg. an icon).  Therefore, we
			 * always try to use the same view - in this case, the view
			 * of the top-level facet.
			 */
			tnp = sim_spice_topview(ni->proto);
			i = 0;
			for(tpp = tnp->firstportproto; tpp != NOPORTPROTO; tpp = tpp->nextportproto)
			{
				/* ensure that common net numbers get written only once */
				for(otpp = tnp->firstportproto; otpp != tpp; otpp = otpp->nextportproto)
					if (otpp->network == tpp->network) break; /* already done */
				if (otpp != tpp) continue;

				pp = equivalentport(tnp, tpp, ni->proto);
				if (pp == NOPORTPROTO) continue;

				/*
				 * write the ground net.
				 * It's valid to pass ground as a parameter to a
				 * sub-circuit, and the sub-circuit header has the
				 * matching parameter, as long as the net was not
				 * specified as Ground in the sub-circuit.
				 *
				 * if (spnet->netnumber != 0)
				 * WE WILL CHECK THE NET NUMBER IN THE PROTOTYPE
				 */
				spnet = sim_spice_getnet(ni, pp->network);
				if (spnet == NOSPNET)
				{
					/* we have a problem here */
					if ((spice_state&SPICENODENAMES) != 0)
						sprintf(line, " UNNAMED%ld", sim_spice_unnamednum++); else
							sprintf(line, " %ld", netindex++);
					sim_spice_xputs(line, f);
					i++;
					continue;
				}
				if (spnet->network->signals > 1)
				{
					/* a bus: write each part */
					for(j=0; j<spnet->network->signals; j++)
					{
						net = spnet->network->networklist[j];
						for(subnet = sim_spice_firstnet; subnet != NOSPNET; subnet = subnet->nextnet)
							if (net == subnet->network) break;
						if (subnet == NOSPNET)
						{
							/* we have a problem here */
							if ((spice_state&SPICENODENAMES) != 0)
								sprintf(line, " UNNAMED%ld", sim_spice_unnamednum++); else
									sprintf(line, " %ld", netindex++);
							sim_spice_xputs(line, f);
							i++;
							break;
						}
						sim_spice_xputs(sim_spice_node_name(subnet), f);
					}
				} else
				{
					/* write the net name */
					if (spnet->netnumber != 0)
						sim_spice_xputs(sim_spice_node_name(spnet), f);
				}
			}

			sim_spice_xprintf(f, " %s\n", ni->proto->cell->cellname);
			if (i != 0)
			{
				sim_spice_xprintf(f, "**WARNING: subfacet %s is not fully connected\n",
					describenodeinst(ni));
				ttyputmsg(_("WARNING: subfacet %s is not fully connected in facet %s"),
					describenodeinst(ni), describenodeproto(np));
			}
			continue;
		}

		/* handle resistors, inductors, capacitors, and diodes */
		if (state == SPICERESISTOR || state == SPICECAPACITOR || state == SPICEINDUCTOR ||
			state == SPICEDIODE || state == SPICEDIODEZ)
		{
			switch (state)
			{
				case SPICERESISTOR:		/* resistor */
					var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_resistancekey);
					if (var == NOVARIABLE) extra = ""; else
						extra = describesimplevariable(var);
					sim_spice_write_twoport(f, ni, state, extra, &resistnum, 1);
					break;
				case SPICECAPACITOR:	/* capacitor */
					var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_capacitancekey);
					if (var == NOVARIABLE) extra = ""; else
						extra = describesimplevariable(var);
					sim_spice_write_twoport(f, ni, state, extra, &capacnum, 1);
					break;
				case SPICEINDUCTOR:		/* inductor */
					var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_inductancekey);
					if (var == NOVARIABLE) extra = ""; else
						extra = describesimplevariable(var);
					sim_spice_write_twoport(f, ni, state, extra, &inductnum, 1);
					break;
				case SPICEDIODE:		/* diode */
				case SPICEDIODEZ:		/* Zener diode */
					var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_diodekey);
					if (var == NOVARIABLE) extra = ""; else
						extra = describesimplevariable(var);
					sim_spice_write_twoport(f, ni, state, extra, &diodenum, 1);
					break;
			}
			continue;
		}
		if (state == SPICEARRAY)
		{
			sim_spice_xprintf(f, "*** ARRAY %s encountered\n", describenodeinst(ni));
			continue;
		}

		/* the default is to handle everything else as a transistor : */
		if (state == SPICEUNKNOWN || state == SPICEMETER || state == SPICESOURCEV ||
			state == SPICESOURCEC || state == SPICESOURCECM || state == SPICESOURCET ||
			state == SPICESOURCEDC || state == SPICESOURCEAC || state == SPICESOURCEN ||
			state == SPICESOURCEX || state == SPICESOURCEB || state == SPICESOURCES ||
			state == SPICESUBSTRATE || state == SPICEGROUND || state == SPICEPOWER)
				continue;

		/* get the source, gates, and drain for the transistor */
		transistorports(ni, &gate, &gatedummy, &source, &drain);

		/* determine the net numbers for these parts of the transistor */
		gaten = sim_spice_getnet(ni, gate->network);     /* base */
		sourcen = sim_spice_getnet(ni, source->network); /* emitter */
		drainn = sim_spice_getnet(ni, drain->network);   /* collector */
		if (gatedummy != NOPORTPROTO)
			gatedn = sim_spice_getnet(ni, gatedummy->network);

		/* make sure transistor is connected to nets */
		if (sourcen == NOSPNET || gaten == NOSPNET || drainn == NOSPNET)
		{
			sim_spice_xprintf(f, "*** WARNING: transistor %s not fully connected:",
				describenodeinst(ni));
			sim_spice_xprintf(f, "\n*** Output line would be: ");
			sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
			sim_spice_xputs(sim_spice_node_name(drainn), f);
			sim_spice_xputs(sim_spice_node_name(gaten), f);
			sim_spice_xputs(sim_spice_node_name(sourcen), f);
			sim_spice_xputs("\n", f);
			continue;
		}

		/* print source, gate, drain, and substrate */

		/* get any special model information */
		info = NULL;
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_spicemodelkey);
		if (var != NOVARIABLE) info = (char *)var->addr;

		switch (state)
		{
			case SPICEREF:			/* self-referential transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'X', &subfacetindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(negnet), f);
				sim_spice_xprintf(f, " %s", ni->proto->primname);
				break;
			case SPICEVCVS:			/* current controlled Current source */
				sim_spice_xputs(sim_spice_element_name(ni, 'E', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gatedn), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				break;
			case SPICEVCCS:			/* current controlled Current source */
				sim_spice_xputs(sim_spice_element_name(ni, 'G', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gatedn), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				break;
			case SPICECCVS:			/* current controlled Current source */
				sim_spice_xputs("*CCVS\n", f); 
				sim_spice_xputs(sim_spice_element_name(ni, 'V', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(" 0V\n", f);
				sim_spice_xputs(sim_spice_element_name(ni, 'H', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gatedn), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(" ", f);  /* y */
				sim_spice_xputs(sim_spice_element_name(ni, 'V', &nodeindex), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				sim_spice_xputs("\n*", f);
				break;
			case SPICECCCS:			/* current controlled Current source */
				sim_spice_xputs("*CCCS\n", f); 
				sim_spice_xputs(sim_spice_element_name(ni, 'V', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(" 0V\n", f);
				sim_spice_xputs(sim_spice_element_name(ni, 'F', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gatedn), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(" ", f);  /* y */
				sim_spice_xputs(sim_spice_element_name(ni, 'V', &nodeindex), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				sim_spice_xputs("\n*", f);
				break;
			case SPICETLINE:	    /* transmission line source */
				sim_spice_xputs(sim_spice_element_name(ni, 'T', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(gatedn), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				break;
			case SPICENMOS:			/* NMOS (Enhancement) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(negnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " N");
				break;
			case SPICENMOS4:		/* NMOS (Complementary) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " N");
				break;
			case SPICEDMOS:			/* DMOS (Depletion) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(negnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " D");
				break;
			case SPICEDMOS4:		/* DMOS (Depletion) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " D");
				break;
			case SPICEPMOS:			/* PMOS (Complementary) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(posnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " P");
				break;
			case SPICEPMOS4:		/* PMOS (Complementary) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'M', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " P");
				break;
			case SPICENPN:			/* NPN (Junction) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (subnet != NOSPNET) sim_spice_xputs(sim_spice_node_name(subnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " NBJT");
				break;
			case SPICENPN4:			/* NPN (Junction) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " NBJT");
				break;
			case SPICEPNP:			/* PNP (Junction) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (subnet != NOSPNET) sim_spice_xputs(sim_spice_node_name(subnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " PBJT");
				break;
			case SPICEPNP4:			/* PNP (Junction) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " PBJT");
				break;
			case SPICENJFET:		/* NJFET (N Channel) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'J', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (info != NULL) sim_spice_xprintf(f,  " %s", info); else
					sim_spice_xprintf(f, " NJFET");
				break;
			case SPICENJFET4:		/* NJFET (N Channel) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'J', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " NJFET");
				break;
			case SPICEPJFET:		/* PJFET (P Channel) transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'J', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " PJFET");
				break;
			case SPICEPJFET4:		/* PJFET (P Channel) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'J', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					biasport = pp;
				biasn = sim_spice_getnet(ni, biasport->network);
				sim_spice_xputs(sim_spice_node_name(biasn), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info); else
					sim_spice_xprintf(f, " PJFET");
				break;
			case SPICEDMES:			/* DMES (Depletion) transistor */
			case SPICEDMES4:		/* DMES (Depletion) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Z', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xprintf(f, " DMES");
				break;
			case SPICEEMES:			/* EMES (Enhancement) transistor */
			case SPICEEMES4:		/* EMES (Enhancement) 4-port transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Z', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xprintf(f, " EMES");
				break;
			case SPICETRANS:		/* special transistor */
				sim_spice_xputs(sim_spice_element_name(ni, 'Q', &nodeindex), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs(sim_spice_node_name(gaten), f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				/* sim_spice_xputs(sim_spice_node_name(subnet), f); */
				if (subnet != NOSPNET) sim_spice_xputs(sim_spice_node_name(subnet), f);
				if (info != NULL) sim_spice_xprintf(f, " %s", info);
				break;
		}

		/* do not print area for self-referential transistors */
		if (state == SPICEREF) continue;

		/* compute length and width (or area for nonMOS transistors) */
		transistorsize(ni, &lx, &ly);
		if (lx >= 0 && ly >= 0)
		{
			a = sim_spice_mask_scale * lx;
			b = sim_spice_mask_scale * ly;
			if (state == SPICENMOS  || state == SPICEDMOS  || state == SPICEPMOS ||
				state == SPICENMOS4 || state == SPICEDMOS4 || state == SPICEPMOS4)
			{
				sim_spice_xprintf(f, " L=%3.2fU W=%3.2fU", scaletodispunit((INTBIG)a, DISPUNITMIC),
					scaletodispunit((INTBIG)b, DISPUNITMIC));
			} else if ((state == SPICENJFET || state == SPICEPJFET || state == SPICEDMES ||
				state == SPICEEMES) && ((spice_state & SPICETYPE) == SPICEHSPICE))
			{
				sim_spice_xprintf(f, " AREA=%3.2f W=%3.2fU L=%3.2fU",
					scaletodispunitsq((INTBIG)(a*b), DISPUNITMIC), scaletodispunit((INTBIG)b, DISPUNITMIC),
						scaletodispunit((INTBIG)a, DISPUNITMIC));
			} else
				sim_spice_xprintf(f, " AREA=%3.2f", scaletodispunitsq((INTBIG)(a*b), DISPUNITMIC));
		}

		/* make sure transistor is connected to nets */
		if (sourcen == NOSPNET || gaten == NOSPNET || drainn == NOSPNET)
		{
			sim_spice_xprintf(f, "\n");
			continue;
		}

		/* compute area of source and drain */
		if (state == SPICENMOS  || state == SPICEDMOS  || state == SPICEPMOS ||
			state == SPICENMOS4 || state == SPICEDMOS4 || state == SPICEPMOS4)
		{
			switch (state)
			{
				case SPICEDMOS:
				case SPICEDMOS4:
				case SPICENMOS:
				case SPICENMOS4: i = ISNTYPE; break;
				case SPICEPMOS:
				case SPICEPMOS4: i = ISPTYPE; break;
				default:         i = ISNONE;  break;
			}

			/* we should not look at the ISNONE entry of components[],
			 * but the diffareas  will be zero anyhow,
			 */
			if (sourcen->components[i] != 0)
			{
				a = scaletodispunitsq((INTBIG)(sourcen->diffarea[i] / sourcen->components[i]),
					DISPUNITMIC);
				if (a > 0.0) sim_spice_xprintf(f, " AS=%5.2fP", a);
			}
			if (drainn->components[i] != 0)
			{
				b = scaletodispunitsq((INTBIG)(drainn->diffarea[i] / drainn->components[i]),
					DISPUNITMIC);
				if (b > 0.0) sim_spice_xprintf(f, " AD=%5.2fP", b);
			}

			/* compute perimeters of source and drain */
			if (sourcen->components[i] != 0)
			{
				a = scaletodispunit((INTBIG)(sourcen->diffperim[i] / sourcen->components[i]),
					DISPUNITMIC);
				if (a > 0.0) sim_spice_xprintf(f, " PS=%5.2fU", a);
			}
			if (drainn->components[i] != 0)
			{
				b = scaletodispunit((INTBIG)(drainn->diffperim[i] / drainn->components[i]),
					DISPUNITMIC);
				if (b > 0.0) sim_spice_xprintf(f, " PD=%5.2fU", b);
			}
		}
		sim_spice_xprintf(f, "\n");
	}

	/* print resistances and capacitances */
	if ((spice_state&SPICERESISTANCE) != 0)
	{
		/* print parasitic capacitances */
		first = 1;
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
		{
			spnet->resistance = scaletodispunitsq((INTBIG)spnet->resistance, DISPUNITMIC);
			if (spnet->resistance > sim_spice_min_resist)
			{
				if (first != 0)
				{
					first = 0;
					sim_spice_xprintf(f, "** Extracted Parasitic Elements:\n");
				}
				sim_spice_xprintf(f, "R%d ? ? %9.2f\n", resistnum++, spnet->resistance);
			}

			if (spnet->netnumber == 0) continue;
			if (spnet->capacitance > sim_spice_min_capac)
			{
				if (first != 0)
				{
					first = 0;
					sim_spice_xprintf(f, "** Extracted Parasitic Elements:\n");
				}
				sim_spice_xprintf(f, "C%d%s 0 %9.2fF\n", capacnum++,
					sim_spice_node_name(spnet), spnet->capacitance);
			}
		}
	}

	/* sources */
	sim_spice_sourceindex = 1;
	first = 1;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		/* only want primitive source nodes */
		if (ni->proto->primindex == 0) continue;
		state = sim_spice_nodetype(ni);
		if (state != SPICESOURCEV && state != SPICESOURCEC && state != SPICESOURCECM &&
			state != SPICESOURCET && state != SPICESOURCEDC && state != SPICESOURCEAC &&
			state != SPICESOURCEN && state != SPICESOURCEX && state != SPICESOURCEB && 
			state != SPICESOURCES) continue;
		if (first != 0)
		{
			first = 0;
			sim_spice_xprintf(f, "** Sources and special nodes:\n");
		}
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_spicemodelkey);
		if (var == NOVARIABLE) extra = 0; else
			extra = (char *)var->addr;

		/* handle the source */
		source = ni->proto->firstportproto;
		sourcen = sim_spice_getnet(ni, source->network);
		if (source->nextportproto == NOPORTPROTO) drainn = NOSPNET; else
			drainn = sim_spice_getnet(ni, source->nextportproto->network);

		/* determine the type of source node */
		found = SPICENORMAL;
		meterit = 0;
		switch (state)
		{
			case SPICESOURCEV:
				sourcetype = 'V';
				break;
			case SPICESOURCEC:
				sourcetype = 'I';
				break;
			case SPICESOURCECM:
				meterit = 1;
				break;
			case SPICESOURCET:
				found = SPICETRAN;
				break;
			case SPICESOURCEDC:
				found = SPICEDC;
				break;
			case SPICESOURCEAC:
				found = SPICEAC;
				break;
			case SPICESOURCEN:
				found = SPICENODESET;
				break;
			case SPICESOURCEX:
				found = SPICEEXTENSION;
				break;
			case SPICESOURCEB:
				found = SPICEBULK;
				break;
			case SPICESOURCES:
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					extra = "V";
				}
				sourcetype = extra[0];
				extra++;
				break;
		}

		/* make sure both connections are exports */
		if (found != SPICEEXTENSION)
		{
			if (sourcen != NOSPNET)
			{
				if (sim_spiceisexported(sourcen, np) == 0)
					ttyputerr(_("Warning: top of %s in facet %s not connected to an export"),
						sim_spice_describesource(ni), describenodeproto(np));
			}
			if (drainn != NOSPNET)
			{
				if (sim_spiceisexported(drainn, np) == 0)
					ttyputerr(_("Warning: bottom of %s in facet %s not connected to an export"),
						sim_spice_describesource(ni), describenodeproto(np));
			}
		}

		switch (found)
		{
			case SPICENORMAL:      /* handle normal source specification */
				if (sourcen == NOSPNET || drainn == NOSPNET)
				{
					ttyputerr(_("ERROR: %s in facet %s is not fully connected"),
						sim_spice_describesource(ni), describenodeproto(np));
				}
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}

				/* write the card */
				dummy = sim_spice_element_name(ni, sourcetype, &sim_spice_sourceindex);
				sim_spice_xputs(dummy, f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xprintf(f, " %s\n", extra);

				if (meterit != 0)
				{
					if (!top)
						ttyputerr(_("Warning: cannot meter source in subfacet: %s"),
							describenodeproto(ni->proto));
					else
					{
						/* add this source to the spice .PRINT card */
						(void)initinfstr();
						(void)addstringtoinfstr(sim_spice_printcard);
						(void)addstringtoinfstr(" I(");
						(void)addstringtoinfstr(dummy);
						(void)addtoinfstr(')');
						if (reallocstring(&sim_spice_printcard, returninfstr(), sim_tool->cluster))
							return(-1);
					}
				}
				break;

			case SPICETRAN:     /* handle transient analysis specification */
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}
				if (sim_spice_tran != 0)
				{
					ttyputerr(_("Warning: should be only one Transient source"));
					efree(sim_spice_tran);
				}
				if (top == 0)
					ttyputerr(_("Warning: Transient source node found in subfacet %s"),
						describenodeproto(np));

				/* create the .TRAN card */
				(void)initinfstr();
				(void)addstringtoinfstr(".TRAN ");
				(void)addstringtoinfstr(extra);
				if (allocstring(&sim_spice_tran, returninfstr(), sim_tool->cluster))
					return(-1);
				break;

			case SPICEDC:       /* handle DC analysis specification */
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}
				if (sourcen == NOSPNET || drainn == NOSPNET)
				{
					ttyputerr(_("ERROR: DC %s in facet %s is not fully connected"),
						sim_spice_describesource(ni), describenodeproto(np));
				}
				if (sim_spice_dc != 0)
				{
					ttyputerr(_("Warning: should be only one DC source"));
					efree(sim_spice_dc);
				}
				if (top == 0)
					ttyputerr(_("Warning: DC source node found in subfacet %s"), describenodeproto(np));

				/* write the source to the deck */
				dummy = sim_spice_element_name(ni, sourcetype, &sim_spice_sourceindex);
				sim_spice_xputs(dummy, f);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs("\n", f);

				if (meterit != 0)
				{
					if (!top)
						ttyputerr(_("Warning: cannot meter source in subfacet: %s"),
							describenodeproto(ni->proto));
					else
					{
						/* add this source to the spice .PRINT card */
						(void)initinfstr();
						(void)addstringtoinfstr(sim_spice_printcard);
						(void)addstringtoinfstr(" I(");
						(void)addstringtoinfstr(dummy);
						(void)addtoinfstr(')');
						if (reallocstring(&sim_spice_printcard, returninfstr(), sim_tool->cluster))
							return(-1);
					}
				}

				/* create the .DC card */
				(void)initinfstr();
				(void)addstringtoinfstr(".DC ");
				(void)addstringtoinfstr(dummy);
				(void)addtoinfstr(' ');
				(void)addstringtoinfstr(extra);
				if (allocstring(&sim_spice_dc, returninfstr(), sim_tool->cluster))
					return(-1);
				break;

			case SPICEAC:			/* handle ac analysis specification */
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}
				if (sim_spice_ac != 0)
				{
					ttyputerr(_("Warning: should be only one AC source"));
					efree(sim_spice_ac);
				}
				if (top == 0) ttyputerr(_("Warning: Transient source node found in subfacet %s"),
					describenodeproto(np));

				/* create the .AC card */
				(void)initinfstr();
				(void)addstringtoinfstr(".AC ");
				(void)addstringtoinfstr(extra);
				if (allocstring(&sim_spice_ac, returninfstr(), sim_tool->cluster)) return(-1);
				break;

			case SPICENODESET:		/* handle NODESET specification */
				if (sourcen == NOSPNET)
				{
					ttyputerr(_("ERROR: %s in facet %s has unconnected positive terminal"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}

				/* no space in V(NName), because HSPICE can't handle line break after ")" */
				sim_spice_xprintf(f, ".NODESET V(%s)=%s\n", sim_spice_node_name(sourcen)+1, extra);
				break;

			case SPICEEXTENSION:	/* handle SPICEEXTENSION */
				if (extra == 0)
				{
					ttyputerr(_("ERROR: %s in facet %s has missing SPICE card"),
						sim_spice_describesource(ni), describenodeproto(np));
					break;
				}
				sim_spice_xprintf(f, "X%s", extra);
				sim_spice_xputs(sim_spice_node_name(sourcen), f);
				sim_spice_xputs(sim_spice_node_name(drainn), f);
				sim_spice_xputs("\n", f);
				break;

			case SPICEBULK:			/* source to set substrate potential */
				/* We don't really have to do anything... but make a comment */
				/* print on a separate line, otherwise too long */
				sim_spice_xprintf(f, "*** P-well tied to%s\n", sim_spice_node_name(negnet));
				sim_spice_xprintf(f, "*** N-well tied to%s\n", sim_spice_node_name(posnet));
				break;
		}
	}

	/*
	 * Now we're finished writing the subcircuit.
	 * Only the top-level facet can contain meters.
	 */
	if (!top)
	{
		sim_spice_xprintf(f, ".ENDS %s\n", np->cell->cellname);

		/* free the net modules */
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = nspnet)
		{
			nspnet = spnet->nextnet;
			sim_freespnet(spnet);
		}
		return(backannotate);
	}

	/* now look for meters in the facet */
	first = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		/* only want primitive meter nodes */
		if (ni->proto->primindex == 0) continue;
		state = sim_spice_nodetype(ni);
		if (state != SPICEMETER) continue;

		/* get the connections on the meter */
		source = ni->proto->firstportproto;
		sourcen = sim_spice_getnet(ni, source->network);
		drain = source->nextportproto;
		drainn = sim_spice_getnet(ni, drain->network);
		if (sourcen == NOSPNET)
		{
			ttyputerr(_("ERROR: Top of %s component in facet %s is not connected"),
				describenodeinst(ni), describenodeproto(np));
			continue;
		}

		/* make sure meter is connected to exports */
		if (sim_spiceisexported(sourcen, np) == 0)
			ttyputerr(_("Warning: top of %s component in facet %s not connected to an export"),
				describenodeinst(ni), describenodeproto(np));
		if (drainn != NOSPNET)
		{
			if (sim_spiceisexported(drainn, np) == 0)
				ttyputerr(_("Warning: bottom of %s component in facet %s not connected to anexport"),
					describenodeinst(ni), describenodeproto(np));
		}
		if (top == 0) ttyputerr(_("Warning: Meter node found in subfacet %s"), describenodeproto(np));

		/* begin printing the meter card */
		if (first == 0)
		{
			first++;
			sim_spice_xputs((spice_state&SPICEPLOT ? ".PLOT" : ".PRINT"), f);

			if (sim_spice_tran) sim_spice_xputs(" TRAN", f); else
				if (sim_spice_dc) sim_spice_xputs(" DC", f); else
					if (sim_spice_ac) sim_spice_xputs(" AC", f);

			/* write what we have already (from metered sources) */
			sim_spice_xputs(sim_spice_printcard, f);
			efree(sim_spice_printcard);
		}

		/* see what kind of meter it is */
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_meterkey);
		if (var != NOVARIABLE) extra = describesimplevariable(var); else extra = "";
		if (drainn == NOSPNET || drainn->netnumber == 0)
		{
			/* no space in V(NName), because HSPICE can't handle line break after ")" */
			sim_spice_xprintf(f, " V(%s)", sim_spice_node_name(sourcen)+1);
			if (*extra != 0) sim_spice_xprintf(f, " %s", extra);
			if (sim_spice_ac)
			{
				/* no space in V(NName), because HSPICE can't handle line break after ")" */
				sim_spice_xprintf(f, " VP(%s)", sim_spice_node_name(sourcen)+1);
				if (*extra != 0) sim_spice_xprintf(f, " %s", extra);
			}
		} else
		{
			/* no space in V(NName), because HSPICE can't handle line break after ")" */
			sim_spice_xprintf(f, " V(%s,",  sim_spice_node_name(sourcen)+1);

			/* can't use sim_spice_node_name() twice in call, because static storage is used */
			sim_spice_xprintf(f, "%s)", sim_spice_node_name(drainn)+1);
			if (*extra != 0) sim_spice_xprintf(f, " %s", extra);
			if (sim_spice_ac)
			{
				/* no space in V(NName), because HSPICE can't handle line break after ")" */
				sim_spice_xprintf(f, " VP(%s,", sim_spice_node_name(sourcen)+1);
				sim_spice_xprintf(f, "%s)", sim_spice_node_name(drainn)+1);
				if (*extra != 0) sim_spice_xprintf(f, " %s", extra);
			}
		}
	}
	if (first != 0) sim_spice_xprintf(f, "\n"); else
	{
		/* if no voltmeters, print metered sources anyway */
		if (*sim_spice_printcard != '\0')
		{
			sim_spice_xputs((spice_state&SPICEPLOT ? ".PLOT" : ".PRINT"), f);

			if (sim_spice_tran) sim_spice_xputs(" TRAN", f); else
				if (sim_spice_dc) sim_spice_xputs(" DC", f); else
					if (sim_spice_ac) sim_spice_xputs(" AC", f);

			/* write what we have already (from metered sources) */
			sim_spice_xputs(sim_spice_printcard, f);
			sim_spice_xprintf(f, "\n");
		}
		efree(sim_spice_printcard);
	}

	/* miscellaneous checks */
	for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
	{
		for (i = 0; i < DIFFTYPES; i++)
		{
			if (spnet->diffarea[i] == 0.0 || spnet->components[i] != 0) continue;

			/* do not issue errors for active area on supply rails (probably well contacts) */
			if (spnet == sim_spice_power || spnet == sim_spice_ground) continue;
			switch (i)
			{
				case ISNTYPE:
					uncon_diff_type = " N-type";
					break;
				case ISPTYPE:
					uncon_diff_type = " P-type";
					break;
				case ISNONE:
				default:
					uncon_diff_type = "";
					break;
			}
			ttyputerr(_("Warning: SPICE node%s has unconnected%s device diffusion"),
				sim_spice_node_name(spnet), uncon_diff_type);
		}
	}

	/* free the net modules */
	for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = nspnet)
	{
		nspnet = spnet->nextnet;
		sim_freespnet(spnet);
	}
	return(backannotate);
}

/******************** DECK GENERATION SUPPORT ********************/

/*
 * write a header for "cell" to spice deck "spfile"
 * The model cards come from a file specified by tech:~.SIM_spice_model_file
 * or else tech:~.SIM_spice_header_level%d
 * The spice model file can be located in el_libdir
 */
void sim_spice_writeheader(CELL *cell, FILE *spfile, TECHNOLOGY *tech)
{
	REGISTER INTSML i, j, level;
	UINTBIG cdate, rdate;
	INTBIG rnum;
	REGISTER VARIABLE *var;
	char c, hvar[80], *filename;
	REGISTER char **name;
	FILE *modelfile;

	/* Print the header line for SPICE  */
	sim_spice_xprintf(spfile, "*** FACET %s FROM LIBRARY %s ***\n", cell->cellname,
		cell->lib->libname);
	cdate = sim_simnt->creationdate;
	if (cdate != 0) sim_spice_xprintf(spfile, "*** FACET CREATED %s\n", timetostring(cdate));
	rnum = sim_simnt->version;
	if (rnum != 0) sim_spice_xprintf(spfile, "*** VERSION %ld", rnum); else
		sim_spice_xputs("***", spfile);
	rdate = sim_simnt->revisiondate;
	if (rdate != 0) sim_spice_xprintf(spfile, " LAST REVISED %s\n", timetostring(rdate));
	sim_spice_xprintf(spfile, "*** EXTRACTED BY ELECTRIC DESIGN SYSTEM, VERSION %s\n",
		el_version);
	sim_spice_xprintf(spfile,"*** UC SPICE *** , MIN_RESIST %f, MIN_CAPAC %fFF\n",
		sim_spice_min_resist, sim_spice_min_capac);
	sim_spice_xputs(".OPTIONS NOMOD NOPAGE\n", spfile);

	/* get spice model/option cards from file if specified */
	var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING, "SIM_spice_model_file");
	if (var != NOVARIABLE)
	{
		if ((modelfile = xopen((char *)var->addr, sim_filetypespice, el_libdir, &filename)) != NULL)
		{
			/* copy file */
			c = (char)xgetc(modelfile);
			while (!xeof(modelfile))
			{
				(void)xputc(c, spfile);
				c = (char)xgetc(modelfile);
			}
			xclose(modelfile);
		} else ttyputerr(_("Warning: cannot open model file: %s"), (char **)var->addr);
	} else
	{
		/* get the spice level and header from technology */
		var = getvalkey((INTBIG)sim_tool, VTOOL, VINTEGER, sim_spice_level);
		if (var == NOVARIABLE) level = 1; else
			level = (INTSML)var->addr;
		(void)sprintf(hvar, "SIM_spice_header_level%d", level);
		var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, hvar);
		if (var != NOVARIABLE)
		{
			name = (char **)var->addr;
			j = (INTSML)getlength(var);
			for(i=0; i<j; i++) sim_spice_xprintf(spfile, "%s\n", name[i]);
		} else ttyputerr(_("Warning: no model cards for SPICE level %d in %s technology"),
			level, tech->techname);
	}

	/* write some extra stuff for HSPICE */
	if ((spice_state & SPICETYPE) == SPICEHSPICE)
		sim_spice_xputs(".OPTIONS INGOLD=2 POST=ASCII TNOM=27 PLIM\n", spfile);
}

/*
 * Write a trailer from an external file, defined as a variable on
 * the current technology in this library: tech:~.SIM_spice_trailer_file
 * if it is available.
 */
void sim_spice_writetrailer(CELL *cell, FILE *spfile, TECHNOLOGY *tech)
{
	VARIABLE *var;
	FILE    *trailerfile;
	char    c, *truename;

	/* get spice trailer cards from file if specified */
	var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING, "SIM_spice_trailer_file");
	if (var != NOVARIABLE)
	{
		trailerfile = xopen((char *)var->addr, sim_filetypespice, el_libdir, &truename);
		if (trailerfile != NULL)
		{
			/* copy file */
			c = (char)xgetc(trailerfile);
			while (!xeof(trailerfile))
			{
				(void)xputc(c, spfile);
				c = (char)xgetc(trailerfile);
			}
			xclose(trailerfile);
		} else
			ttyputerr(_("Warning: cannot open trailer file: %s"), (char **)var->addr);
	}
}

/*
 * routine to write the port net numbers of facet "np" to file "f".  The
 * numbers must be unique and must not include net zero (ground)
 */
void sim_spice_writeports(FILE *f, NODEPROTO *np)
{
	REGISTER PORTPROTO *pp, *opp;
	REGISTER NETWORK *net;
	REGISTER INTBIG i, j;

	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* do not write the ground net */
		if (sim_spice_ground != NOSPNET && sim_spice_ground->network == pp->network)
			continue;

		if (pp->network->signals > 1)
		{
			/* a bus: write each part */
			for(i=0; i<pp->network->signals; i++)
			{
				net = pp->network->networklist[i];

				/* skip if already mentioned */
				for(opp = np->firstportproto; opp != pp; opp = opp->nextportproto)
				{
					if (net == opp->network) break;
					if (opp->network->signals > 1)
					{
						for(j=0; j<opp->network->signals; j++)
							if (opp->network->networklist[j] == net) break;
						if (j < opp->network->signals) break;
					}
				}
				if (opp != pp) continue;

				sim_spice_xprintf(f, " %s", sim_spice_net_name(net));
			}
		} else
		{
			/* skip if already mentioned */
			for(opp = np->firstportproto; opp != pp; opp = opp->nextportproto)
			{
				if (pp->network == opp->network) break;
				if (opp->network->signals > 1)
				{
					for(j=0; j<opp->network->signals; j++)
						if (opp->network->networklist[j] == pp->network) break;
					if (j < opp->network->signals) break;
				}
			}
			if (opp != pp) continue;

			/* write the net name */
			sim_spice_xprintf(f, " %s", sim_spice_net_name(pp->network));
		}
	}
}

/*
 * Function to write a two port device to the file. If the flag 'report'
 * is set, then complain about the missing connections.
 * Determine the port connections from the portprotos in the instance
 * prototype. Get the part number from the 'part' number value;
 * increment it. The type of device is declared in type; extra is the string
 * data acquired before calling here.
 * If the device is connected to the same net at both ends, do not
 * write it. Is this OK?
 */
void sim_spice_write_twoport(FILE *f, NODEINST *ni, INTBIG type, char *extra,
	INTBIG *part, INTBIG report)
{
	REGISTER PORTPROTO *pp1, *pp2;
	REGISTER SPNET *end1, *end2;

	pp1 = ni->proto->firstportproto;
	pp2 = pp1->nextportproto;
	end1 = sim_spice_getnet(ni, pp1->network);
	end2 = sim_spice_getnet(ni, pp2->network);

	/* make sure the component is connected to nets */
	if (end1 == NOSPNET || end2 == NOSPNET)
	{
		sim_spice_xprintf(f, "*** WARNING: %s component not fully connected\n*** ",
			describenodeinst(ni));
	}
	if (end1 != NOSPNET && end2 != NOSPNET)
		if (end1->netnumber == end2->netnumber)
	{
		if (report)
			sim_spice_xprintf(f, "*** WARNING: %s component appears to be shorted on net %s\n",
				describenodeinst(ni), sim_spice_node_name(end1));
		return;
	}

	/* next line is not really necessary any more */
	switch (type)
	{
		case SPICERESISTOR:		/* resistor */
			sim_spice_xputs(sim_spice_element_name(ni, 'R', part), f);
			break;
		case SPICECAPACITOR:	/* capacitor */
			sim_spice_xputs(sim_spice_element_name(ni, 'C', part), f);
			break;
		case SPICEINDUCTOR:		/* inductor */
			sim_spice_xputs(sim_spice_element_name(ni, 'L', part), f);
			break;
		case SPICEDIODE:		/* diode */
		case SPICEDIODEZ:		/* Zener diode */
			sim_spice_xputs(sim_spice_element_name(ni, 'D', part), f);
			break;
	}
	sim_spice_xputs(sim_spice_node_name(end2), f);
	sim_spice_xputs(sim_spice_node_name(end1), f);  /* note order */
	if (type == SPICEDIODE || type == SPICEDIODEZ)
	{
		if (extra[0] != 0) sim_spice_xprintf(f, " %s\n", extra); else
			sim_spice_xprintf(f, " DIODE\n");
	} else
		sim_spice_xprintf(f, " %s\n", extra);
}

/******************** PARASITIC CALCULATIONS ********************/

/*
 * routine to recursively determine the area of diffusion and capacitance
 * associated with port "pp" of nodeinst "ni".  If the node is mult_layer, then
 * determine the dominant capacitance layer, and add its area; all other
 * layers will be added as well to the extra_area total.
 * Continue out of the ports on a complex facet
 */
void sim_spice_nodearea(SPNET *spnet, NODEINST *ni, PORTPROTO *pp)
{
	REGISTER INTBIG tot, i, function;
	REGISTER INTBIG fun;
	XARRAY trans;
	INTBIG lx, hx, ly, hy;
	INTSML j, dominant;
	REGISTER POLYGON *poly, *lastpoly, *firstpoly;
	REGISTER PORTARCINST *pi;
	REGISTER TECHNOLOGY *tech;
	float worst, cap;

	/* make sure this node hasn't been completely examined */
	if (ni->temp1 == 2) return;
	if (ni->temp2 == (INTBIG)pp->network) return;	/* Is this recursive? */
	ni->temp2 = (INTBIG)pp->network;

	/* facets have no area or capacitance (for now) */
	if (ni->proto->primindex != 0)  /* No area for complex nodes */
	{
		/* assign new state of this node */
		function = sim_spice_nodetype(ni);
		if (function == SPICEFACET || function == SPICEMETER || function == SPICESOURCEV ||
			function == SPICESOURCEC || function == SPICESOURCECM || function == SPICESOURCET ||
			function == SPICESOURCEDC || function == SPICESOURCEAC || function == SPICESOURCEN ||
			function == SPICESOURCEX || function == SPICESOURCEB || function == SPICESOURCES ||
			function == SPICEGROUND || function == SPICEPOWER) ni->temp1 = 2; else
				ni->temp1 = 1;

		/* initialize to examine the polygons on this node */
		tech = ni->proto->tech;
		makerot(ni, trans);

		/*
		 * NOW!  A fudge to make sure that well capacitors mask out the capacity
		 * to substrate of their top plate polysilicon  or metal
		 */
		if (function == SPICECAPACITOR) dominant = -1; else dominant = -2;
		if (function == SPICENMOS || function == SPICENMOS4 ||
			function == SPICEPMOS || function == SPICEPMOS4 ||
			function == SPICEDMOS || function == SPICEDMOS4 ||
			function == SPICEEMES || function == SPICEEMES4 ||
			function == SPICEDMES || function == SPICEDMES4)
				function = SPICENMOS;   /* One will do */

		/* make linked list of polygons */
		lastpoly = firstpoly = NOPOLYGON;
		tot = nodeEpolys(ni, 0, NOWINDOWPART);
		for(i=0; i<tot; i++)
		{
			poly = sim_allocpolygon();
			if (poly == NOPOLYGON) break;
			shapeEnodepoly(ni, i, poly);

			/* make sure this layer connects electrically to the desired port */
			if (poly->portproto == NOPORTPROTO)
			{
				sim_freepolygon(poly);   continue;
			}
			if (poly->portproto->network != pp->network)
			{
				sim_freepolygon(poly);   continue;
			}

			/* don't bother with layers without capacity */
			if ((sim_spice_layerisdiff(tech, poly->layer) == ISNONE) &&
				(sim_spice_capacitance(tech, poly->layer) == 0.0))
			{
				sim_freepolygon(poly);   continue;
			}

			/* leave out the gate capacitance of transistors */
			if (function == SPICENMOS)
			{
				fun = layerfunction(tech, poly->layer);
				if ((fun & LFPSEUDO) == 0 && layerispoly(fun))
				{
					sim_freepolygon(poly);   continue;
				}
			}
			if (lastpoly != NOPOLYGON) lastpoly->nextpolygon = poly; else
				firstpoly = poly;
			lastpoly = poly;
		}

		/* do we need to test the layers? */
		if (dominant != -1)
		{
			if (tot != 0 && firstpoly != NOPOLYGON) dominant = firstpoly->layer;

			/* find the layer that will contribute the maximum capacitance */
			if (tot > 1 && tech == el_curtech)
			{
				worst = 0.0;
				for(poly = firstpoly; poly != NOPOLYGON; poly = poly->nextpolygon)
				{
					if (sim_spice_layerisdiff(tech, poly->layer) != ISNONE)
					{
						dominant = -1;      /* flag for diffusion on this port */
						break;
					} else
					{
						cap = (float)fabs(areapoly(poly));
						if (cap * sim_spice_capacitance(tech, poly->layer) > worst)
						{
							worst = cap;
							dominant = poly->layer;
						}
					}
				}
			}
		}

		for(poly = firstpoly; poly != NOPOLYGON; poly = poly->nextpolygon)
		{
			/* get the area of this polygon */
			xformpoly(poly, trans);
			j = isbox(poly, &lx, &hx, &ly, &hy);
			if (tech != el_curtech || j == 0)
			{
				if (sim_spice_debug != 0)
					ttyputmsg(M_("%s POLYGON ON %s NODE IGNORED IN %s TECHNOLOGY"),
						layername(tech, poly->layer), ni->proto->cell->cellname, tech->techname);
				continue;
			}
			sim_spice_store_box(tech, poly->layer, hx-lx, hy-ly, (lx+hx)/2, (ly+hy)/2);
			if (sim_spice_debug != 0)
				ttyputmsg(M_("BOX: %ld to %ld; %ld to %ld; layer %s"), lx, hx, ly, hy,
					layername(tech,poly->layer));
			if (sim_spice_layerisdiff(tech, poly->layer) == ISNONE &&
				poly->layer != dominant)
					sim_spice_extra_area[poly->layer] += (float)fabs(areapoly(poly));
		}

		/* free the polygons */
		while (firstpoly != NOPOLYGON)
		{
			poly = firstpoly;
			firstpoly = firstpoly->nextpolygon;
			sim_freepolygon(poly);
		}
	}
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		if (pi->proto->network == pp->network)
			sim_spice_arcarea(spnet, pi->conarcinst);
	ni->temp2 = 0;      /* reset for next net pass */
}

/*
 * routine to recursively determine the area of diffusion, capacitance, (NOT
 * resistance) on arc "ai". If the arc contains active device diffusion, then
 * it will contribute to the area of sources and drains, and the other layers
 * will be ignored. This is not quite the same as the rule used for
 * contact (node) structures. Note: the earlier version of this
 * function assumed that diffusion arcs would always have zero capacitance
 * values for the other layers; this produces an error if any of these layers
 * have non-zero values assigned for other reasons. So we will check for the
 * function of the arc, and if it contains active device, we will ignore any
 * other layers
 */
void sim_spice_arcarea(SPNET *spnet, ARCINST *ai)
{
	REGISTER INTBIG i, tot;
	REGISTER TECHNOLOGY *tech;
	static POLYGON *poly = NOPOLYGON;
	INTBIG lx, hx, ly, hy;
	INTSML j, isdiffarc;

	if (ai->temp1 != 0) return;
	ai->temp1++;

	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, sim_tool->cluster);

	tot = arcpolys(ai, NOWINDOWPART);
	tech = ai->proto->tech;
	isdiffarc = sim_spice_arcisdiff(ai);    /* check arc function */
	for(i=0; i<tot; i++)
	{
		shapearcpoly(ai, i, poly);
		j = isbox(poly, &lx, &hx, &ly, &hy);
		if (tech != el_curtech || j == 0)
		{
			if (sim_spice_debug != 0)
				ttyputmsg(M_("%s POLYGON ON %s ARC IGNORED IN %s TECHNOLOGY"),
					layername(tech, poly->layer), describearcinst(ai), tech->techname);
			continue;
		}
		if ((layerfunction(tech, poly->layer)&LFPSEUDO) != 0) continue;
		if (sim_spice_layerisdiff(tech, poly->layer) != ISNONE ||
			(!isdiffarc && sim_spice_capacitance(tech, poly->layer) > 0.0))
				sim_spice_store_box(tech, poly->layer, hx-lx, hy-ly, (lx+hx)/2, (ly+hy)/2);
		if (sim_spice_debug != 0)
			ttyputmsg(M_("BOX: %ld to %ld; %ld to %ld; layer %s"), lx, hx, ly, hy,
				layername(tech,poly->layer));
	}

	/* propagate to all of the nodes on this arc */
	for(i=0; i<2; i++)
		sim_spice_nodearea(spnet, ai->end[i].nodeinst, ai->end[i].portarcinst->proto);
}

/*
 * Routine to store a box on layer "layer" into the box merging system
 */
void sim_spice_store_box(TECHNOLOGY *tech, INTSML layer, INTBIG xs, INTBIG ys, INTBIG xc, INTBIG yc)
{
	INTSML type;

	if ((type = sim_spice_layerisdiff(tech, layer)) != ISNONE)
	{
		if (sim_spice_diffusion_index[type] < 0)
			sim_spice_diffusion_index[type] = layer; else
				layer = sim_spice_diffusion_index[type];
	}
	mrgstorebox(layer, tech, xs, ys, xc, yc);
}

/*
 * routine to obtain a polygon from the box merging system
 */
void sim_spice_evalpolygon(INTSML layer, TECHNOLOGY *tech, INTBIG *xbuf, INTBIG *ybuf, INTSML count)
{
	REGISTER INTBIG perim;
	float area;
	REGISTER INTSML i, j;

	if (sim_spice_debug != 0)
	{
		ttyputmsg(M_("** Evaluating polygon on Layer: %s; Net: %ld"),
			layername(tech, layer), sim_spice_cur_net->netnumber);
	}

	/* compute perimeter */
	perim = 0;
	for(i=0; i<count; i++)
	{
		if (i == 0) j = count-1; else j = i-1;
		perim += computedistance(xbuf[j], ybuf[j], xbuf[i], ybuf[i]);
	}

	/* get area */
	area = areapoints(count, xbuf, ybuf);
	if (sim_spice_extra_area[layer] != 0.0)
	{
		area -= sim_spice_extra_area[layer];
		sim_spice_extra_area[layer] = 0.0; /* but only once */
	}

	i = sim_spice_layerisdiff(tech, layer);
	if (i != ISNONE)
	{
		sim_spice_cur_net->diffarea[i] += area * sim_spice_mask_scale * sim_spice_mask_scale;
		sim_spice_cur_net->diffperim[i] += perim * sim_spice_mask_scale;
	} else
	{
		sim_spice_cur_net->capacitance += scaletodispunitsq((INTBIG)(sim_spice_capacitance(
			tech, layer) * area), DISPUNITMIC) *
				sim_spice_mask_scale * sim_spice_mask_scale;
		sim_spice_cur_net->capacitance += scaletodispunit((INTBIG)(sim_spice_edge_capacitance(
			tech, layer) * perim), DISPUNITMIC) *
				sim_spice_mask_scale;
	}
	if (sim_spice_debug != 0)
		ttyputmsg(M_("Area: %f , Perim: %ld Cap: %f"), area, perim, sim_spice_cur_net->capacitance);
}

/*
 * routine to find the capacitance for the arc on layer "layer"
 */
float sim_spice_capacitance(TECHNOLOGY *tech, INTSML layer)
{
	if (layer < 0 || tech->temp2 == 0) return(0.0);
	return(castfloat(((INTBIG *)tech->temp2)[layer]));
}

/*
 * Routine to return the fringing capacitance of layer "layer" in tech "tech"
 */
float sim_spice_edge_capacitance(TECHNOLOGY *tech, INTSML layer)
{
	REGISTER INTBIG addr;
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *t;

	if (layer < 0) return(0.0);
	if (sim_spice_capacvalue == 0)
	{
		sim_spice_capacvalue = emalloc(el_maxtech * SIZEOFINTBIG, sim_tool->cluster);
		if (sim_spice_capacvalue == 0) return(0.0);
		for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
		{
			var = getval((INTBIG)t, VTECHNOLOGY, VFLOAT|VISARRAY, "SIM_spice_edge_capacitance");
			sim_spice_capacvalue[t->techindex] = (var == NOVARIABLE ? 0 : var->addr);
		}
	}
	addr = sim_spice_capacvalue[tech->techindex];
	if (addr == 0) return(0.0);
	return(castfloat(((INTBIG *)(addr))[layer]));
}

/******************** TEXT ROUTINES ********************/

/*
 * Function to return a spice "element" name.
 * The first character (eg. R,L,C,D,X,...) is specified by "first".
 * the rest of the name comes from the name on inst "ni".
 * If there is no node name, a unique number specified by "counter" is used
 * and counter is incremented.
 *
 * Warning: This routine is not re-entrant.  You must use the returned string
 * before calling the routine again.
 */
char *sim_spice_element_name(NODEINST *ni, char first, INTBIG *counter)
{
	VARIABLE *varname;
	static char s[80];  /* this is overkill - spice only recognizes 8 chars */

	varname = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
	if (varname == NOVARIABLE)
	{
		ttyputerr(_("SPICE generation warning: no name on node %s"), describenodeinst(ni));
		(void)sprintf(s, "%c%d", first, (*counter)++);
	} else
	{
		(void)sprintf(s, "%c%s", first, (char *)varname->addr);
	}

	return(s);
}

/*
 * The ground node is always written as node 0 (name is never used)
 * in SPICE2.  Unknown nets are assigned as node '*'.
 *
 * Warning: This routine is not re-entrant.  You must use the returned string
 * before calling the routine again.
 */
char *sim_spice_node_name(SPNET *spnet)
{
	static char s[80];
	REGISTER INTSML namegroundzero;

	if (spnet == NOSPNET) return(" *");

	/* decide whether or not to name the ground net "0" */
	namegroundzero = 1;
	/* if ((spice_state & SPICETYPE) != SPICE2) namegroundzero = 0; */

	if ((spice_state&SPICENODENAMES) != 0 &&
		(spnet->netnumber != 0 || namegroundzero == 0))
	{
		if ((spnet->network->namecount > 0))
		{
			(void)sprintf(s, " %s", spnet->network->netname);
			if (isdigit(s[1])) (void)sprintf(s, " %ld", spnet->netnumber);
		} else
			(void)sprintf(s, " %ld", spnet->netnumber);
	} else
		(void)sprintf(s, " %ld", spnet->netnumber);

	return(s);
}

/*
 * This is almost identical to sim_spice_node_name, but takes a network
 * instead.
 *
 * Warning: This routine is not re-entrant.  You must use the returned string
 * before calling the routine again.
 */
char *sim_spice_net_name(NETWORK *net)
{
	static char s[80];
	REGISTER SPNET *spnet;
	REGISTER INTSML namegroundzero;

	if (net == NONETWORK) return("0");

	/* decide whether or not to name the ground net "0" */
	namegroundzero = 1;
	/* if ((spice_state & SPICETYPE) != SPICE2) namegroundzero = 0; */

	spnet = (SPNET *)net->temp1;
	if ((spice_state&SPICENODENAMES) != 0)
	{
		if (net->namecount > 0 &&
			(spnet->netnumber != 0 || namegroundzero == 0))
		{
			(void)strcpy(s, net->netname);
			if (isdigit(s[0])) (void)sprintf(s, "%ld", spnet->netnumber);
			return(s);
		}
	}
	(void)sprintf(s, "%ld", spnet->netnumber);
	return(s);
}

char *sim_spice_describesource(NODEINST *ni)
{
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_spicemodelkey);
	if (var == NOVARIABLE) return(describenodeinst(ni));
	(void)initinfstr();
	(void)addstringtoinfstr(describenodeinst(ni));
	(void)addtoinfstr('/');
	(void)addstringtoinfstr(describesimplevariable(var));
	return(returninfstr());
}

/******************** SUPPORT ********************/

/*
 * Routine to return nonzero if the network associated with spice net "spnet" in
 * facet "np" is exported.
 */
INTBIG sim_spiceisexported(SPNET *spnet, NODEPROTO *np)
{
	REGISTER PORTPROTO *pp;
	REGISTER INTBIG i;

	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp->network == spnet->network) return(1);
		if (pp->network->signals > 1)
		{
			for(i=0; i<pp->network->signals; i++)
				if (pp->network->networklist[i] == spnet->network) return(1);
		}
	}
	return(0);
}

/*
 * routine to return the type of node in "ni", according to the defines at
 * the top of this module
 */
INTSML sim_spice_nodetype(NODEINST *ni)
{
	REGISTER INTBIG nodetype;

	if (ni->proto->primindex == 0) return(SPICEFACET);
	nodetype = nodefunction(ni);
	switch (nodetype)
	{
		case NPTRANMOS:     return(SPICENMOS);
		case NPTRADMOS:     return(SPICEDMOS);
		case NPTRAPMOS:     return(SPICEPMOS);
		case NPTRANPN:      return(SPICENPN);
		case NPTRAPNP:      return(SPICEPNP);
		case NPTRANJFET:    return(SPICENJFET);
		case NPTRAPJFET:    return(SPICEPJFET);
		case NPTRADMES:     return(SPICEDMES);
		case NPTRAEMES:     return(SPICEEMES);
		case NPTRANSREF:    return(SPICEREF);
		case NPTRANS:       return(SPICETRANS);
		case NPTRA4NMOS:    return(SPICENMOS4);
		case NPTRA4DMOS:    return(SPICEDMOS4);
		case NPTRA4PMOS:    return(SPICEPMOS4);
		case NPTRA4NPN:     return(SPICENPN4);
		case NPTRA4PNP:     return(SPICEPNP4);
		case NPTRA4NJFET:   return(SPICENJFET4);
		case NPTRA4PJFET:   return(SPICEPJFET4);
		case NPTRA4DMES:    return(SPICEDMES4);
		case NPTRA4EMES:    return(SPICEEMES4);
		case NPMETER:       return(SPICEMETER);
		case NPSOURCEV:     return(SPICESOURCEV);
		case NPSOURCEC:     return(SPICESOURCEC);
		case NPSOURCECM:    return(SPICESOURCECM);
		case NPSOURCET:     return(SPICESOURCET);
		case NPSOURCEDC:    return(SPICESOURCEDC);
		case NPSOURCEAC:    return(SPICESOURCEAC);
		case NPSOURCEN:     return(SPICESOURCEN);
		case NPSOURCEX:     return(SPICESOURCEX);
		case NPSOURCEB:     return(SPICESOURCEB);
		case NPSOURCES:     return(SPICESOURCES);
		case NPRESIST:      return(SPICERESISTOR);
		case NPCAPAC:
		case NPECAPAC:      return(SPICECAPACITOR);
		case NPINDUCT:      return(SPICEINDUCTOR);
		case NPDIODE:       return(SPICEDIODE);
		case NPDIODEZ:      return(SPICEDIODEZ);
		case NPSUBSTRATE:   return(SPICESUBSTRATE);
		case NPARRAY:       return(SPICEARRAY);
		case NPCONGROUND:   return(SPICEGROUND);
		case NPCONPOWER:    return(SPICEPOWER);
		case NPCCCS:        return(SPICECCCS);
		case NPCCVS:        return(SPICECCVS);
		case NPVCCS:        return(SPICEVCCS);
		case NPVCVS:        return(SPICEVCVS);
		case NPTLINE:       return(SPICETLINE);
	}
	return(SPICEUNKNOWN);
}

/*
 * routine to return nonzero if layer "layer" is on diffusion
 * Return the type of the diffusion
 */
INTSML sim_spice_layerisdiff(TECHNOLOGY *tech, INTSML layer)
{
	REGISTER INTBIG i;

	i = layerfunction(tech, layer);
	if ((i&LFPSEUDO) != 0) return(ISNONE);
	if ((i&LFTYPE) != LFDIFF) return(ISNONE);
	if ((i&LFPTYPE) != 0) return(ISPTYPE);
	if ((i&LFNTYPE) != 0) return(ISNTYPE);
	return(ISNTYPE);		/* Default to N-type  */
}

/*
 * routine to return value if arc contains device active diffusion
 * Return the type of the diffusion, else ISNONE
 */
INTSML sim_spice_arcisdiff(ARCINST *ai)
{
	REGISTER INTBIG i;

	i = (ai->proto->userbits&AFUNCTION)>>AFUNCTIONSH;
	switch (i)
	{
		case APDIFFP:
			return(ISPTYPE);
		case APDIFFN:
			return(ISNTYPE);
		case APDIFF:
			return(ISNTYPE);	/* Default device is n-type */
		default:
			return(ISNONE);	/* Default to Unknown  */
	}
}

/*
 * routine to search the net list for this facet and return the net number
 * associated with nodeinst "ni", network "net"
 */
SPNET *sim_spice_getnet(NODEINST *ni, NETWORK *net)
{
	REGISTER SPNET *spnet;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;

	/* search for arcs electrically connected to this port */
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		if (pi->proto->network == net)
	{
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
			if (pi->conarcinst->network == spnet->network) return(spnet);
	}

	/* search for exports on the node, connected to this port */
	for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
		if (pe->proto->network == net)
	{
		for(spnet = sim_spice_firstnet; spnet != NOSPNET; spnet = spnet->nextnet)
			if (pe->exportproto->network == spnet->network) return(spnet);
	}
	return(NOSPNET);
}

/*
 * Function to return the same view as top-level facet.
 * "sim_simnt" must be set before calling this function.
 * Returns the current view if there is no top-level view.
 */
NODEPROTO *sim_spice_topview(NODEPROTO *np)
{
	REGISTER NODEPROTO *rnp;

	if (np == NONODEPROTO) return(NONODEPROTO);

	if (np->primindex != 0) return(NONODEPROTO);    /* must be complex */

	/* Now look for views */
	for(rnp = np->cell->firstincell; rnp != NONODEPROTO; rnp = rnp->nextincell)
		if (rnp->cellview == sim_simnt->cellview) return(rnp);

	/* keep what we have */
	return(np);
}

/******************** LOW-LEVEL OUTPUT ROUTINES ********************/

/*
 * Formatted output to file "stream".  All spice output is in upper case.
 * The buffer can contain no more than 256 chars including the newline
 * and null characters.
 * Doesn't return anything.
 */
void sim_spice_xprintf(FILE *stream, char *format, ...)
{
	va_list ap;
	char s[256];

	var_start(ap, format);
	evsnprintf(s, 256, format, ap);
	va_end(ap);
	sim_spice_xputs(s, stream);
}

/*
 * Routine to write string "s" onto stream "stream" in all upper case.
 */
void sim_spice_xputs(char *s, FILE *stream)
{
	char *pt;
	static INTSML i=0;
	char *lastspace;

	/* put in line continuations, if over 78 chars long */
	lastspace = NULL;
	for (pt = s; *pt; pt++)
	{
		if ((spice_state & SPICETYPE) == SPICE2)
		{
			if (islower(*pt)) *pt = toupper(*pt);
		}
		if (*pt == '\n')
		{
			i = 0;
			lastspace = NULL;
		} else
		{
			if (*pt == ' ' || *pt == '/') lastspace = pt;
			++i;
			if (i >= 78)
			{
				if (lastspace != NULL)
				{
					*lastspace = '\0';
					(void)xputs(s, stream);
					(void)xputs("\n+  ", stream);
					s = lastspace + 1;
					i = 9 + pt-s+1;
					lastspace = NULL;
				} else
				{
					(void)xputs("\n+  ", stream);
					i = 9 + 1;
				}
			}
		}
	}
	(void)xputs(s, stream);
}

/******************** MEMORY ALLOCATION ********************/

POLYGON *sim_allocpolygon(void)
{
	REGISTER POLYGON *poly;

	if (sim_polygonfree != NOPOLYGON)
	{
		poly = sim_polygonfree;
		sim_polygonfree = poly->nextpolygon;
	} else
	{
		poly = allocpolygon(4, sim_tool->cluster);
		if (poly == NOPOLYGON) return(NOPOLYGON);
	}
	poly->nextpolygon = NOPOLYGON;
	return(poly);
}

void sim_freepolygon(POLYGON *poly)
{
	poly->nextpolygon = sim_polygonfree;
	sim_polygonfree = poly;
}

/*
 * routine to allocate and initialize a new net module from the pool
 * (if any) or memory
 */
SPNET *sim_allocspnet(void)
{
	REGISTER SPNET *spnet;
	REGISTER INTBIG j;

	if (sim_spice_netfree == NOSPNET)
	{
		spnet = (SPNET *)emalloc(sizeof (SPNET), sim_tool->cluster);
		if (spnet == 0) return(NOSPNET);
	} else
	{
		/* take module from free list */
		spnet = sim_spice_netfree;
		sim_spice_netfree = spnet->nextnet;
	}

	/* Initialize it to empty values */
	spnet->resistance = 0.0;
	spnet->capacitance = 0.0;
	spnet->network = NONETWORK;
	for (j = 0; j < DIFFTYPES; j++)
	{
		spnet->diffperim[j] = 0.0;
		spnet->diffarea[j] = 0.0;
		spnet->components[j] = 0;
	}
	return(spnet);
}

/*
 * routine to return net module "spnet" to the pool of free modules
 */
void sim_freespnet(SPNET *spnet)
{
	spnet->nextnet = sim_spice_netfree;
	sim_spice_netfree = spnet;
}

#endif  /* SIMTOOL - at top */
