#!/usr/bin/env python

# compare two mpeg transport streams and report differences

# usage: cmp-mpeg-ts expected-ts actual-ts


import getopt, sys

MPEG_PKT_LEN = 188

verbose_p = 0

def read_pkt (file):
    p = file.read (MPEG_PKT_LEN)
    if p and p[0] != '\x47':
        sys.stderr.write (("MPEG packet doesn't start with 0x47 at file offset %d\n"
                          % (file.tell() - MPEG_PKT_LEN)))
    return p
    
# returns number of packets with errors (nerrors, total_packets)

def compare (expected, actual, expected_fn, actual_fn):
    global verbose_p
    npackets = 0
    nerrors = 0
    column = 0
    while 1:
        ep = read_pkt (expected)
        ap = read_pkt (actual)
        if ep == '' or ap == '':
            if ep == '' and ap == '':
                return (nerrors, npackets)
            if ep == '':
                sys.stderr.write ('Unexpected EOF on %s\n' % expected_fn)
            else:
                sys.stderr.write ('Unexpected EOF on %s\n' % actual_fn)
            return (nerrors + 1, npackets)
        else:
            if ep == ap:
                # OK
                if verbose_p:
                    sys.stdout.write ('.')
            else:
                # don't match
                if verbose_p:
                    sys.stdout.write ('#')
                nerrors = nerrors + 1

            if verbose_p:
                # break lines every 80 and also at end of a field (312 segs)
                column = column + 1
                if column % 80 == 0:
                    sys.stdout.write ('\n')
                if column == 312:
                    sys.stdout.write ('\n')
                    column = 0

        npackets = npackets + 1
        

    
def usage ():
    print "usage: %s [-v] [-h] [--help] expected_mpeg_ts actual_mpeg_ts" % sys.argv[0]
    
def main ():
    global verbose_p
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hv", ["help", "verbose"])
    except getopt.GetoptError:
        # print help information and exit:
        usage ()
        sys.exit (2)
    output = None
    for o, a in opts:
        if o in ("-h", "--help"):
            usage ()
            sys.exit (0)
        if o in ("-v", "--verbose"):
            verbose_p = 1

    if len (args) != 2:
        usage ()
        sys.exit (2)
    # ...

    expected_fn = args[0]
    expected_ts = open (expected_fn, "r")
    actual_fn = args[1]
    actual_ts = open (actual_fn, "r")

    nerrors, npackets = compare (expected_ts, actual_ts, expected_fn, actual_fn)
    if verbose_p:
        sys.stdout.write ('\n')
        sys.stdout.flush ()
    if nerrors > 0:
        sys.stderr.write ('%d incorrect packets detected out of %d total (error rate: %.1e)\n'
                          % (nerrors, npackets, float(nerrors) / float(npackets)))
        sys.exit (1);
    else:
        sys.exit (0)

main ()
