#!/usr/bin/env python
#
# Copyright 2003 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

import sys, getopt
from GnuRadio import *


def build_graph (input_filename, output_filename, output_is_20M_p = True):
    '''This implements a complete ATSC transmitter.

    The input is an MPEG transport stream.
    The output is 16-bit samples at either 21.52 or 20.0 Msamples/sec
      with an IF of 5.75 MHz.'''

    INPUT_RATE = ATSC_DATA_SEGMENT_RATE * 312./313.
    OUTPUT_RATE = 2 * ATSC_SYMBOL_RATE
    NMFTAPS = 139                                       # of matched filter taps
    IF_freq = 5.75e6                                    # output freq of tuner module
    MIXER_gain = 128

    source = VrFileSource_mpeg_packet (INPUT_RATE, input_filename, 0)
    randomizer = GrAtscRandomizer ()
    rs_encoder = GrAtscRSEncoder ()
    interleaver = GrAtscInterleaver ()
    trellis_encoder = GrAtscTrellisEncoder ()
    field_sync_mux = GrAtscFieldSyncMux ()
    mapper = GrAtscSymbolMapperF ()

    # mix with cos/sin Fsym/4 and interpolate by 2
    weaver_head = GrWeaverModHeadFF (2)

    # matched filter taps
    # gain of 2 to make up for the gain of 1/2 of the interpolate by 2
    mf_taps = gr_firdes.root_raised_cosine (2.0,
                                            OUTPUT_RATE,
                                            ATSC_SYMBOL_RATE/2,
                                            .115, NMFTAPS)
    mfI = GrFIRfilterFFF (1, mf_taps)
    mfQ = GrFIRfilterFFF (1, mf_taps)

    # we drop the interpolators in here, instead of at the end,
    # because the bandwidth of interest is much lower here then after
    # the up-conversion to the IF.  This allows us to use interpolators
    # that have fewer taps
    
    if output_is_20M_p:
        interpI = GrAtscConvert2xTo20 ()
        interpQ = GrAtscConvert2xTo20 ()

    # mix with cos/sin IF_freq, sum components
    weaver_tail = GrWeaverModTailFS (IF_freq, MIXER_gain)
    
    # final sink
    sink = VrFileSink_short (output_filename)

    fg = gr_FlowGraph ()
    fg.connect (source, randomizer)
    fg.connect (randomizer, rs_encoder)
    fg.connect (rs_encoder, interleaver)
    fg.connect (interleaver, trellis_encoder)
    fg.connect (trellis_encoder, field_sync_mux)
    fg.connect (field_sync_mux, mapper)
    fg.connect (mapper, weaver_head)
    fg.connectn (weaver_head, 0, mfI)
    fg.connectn (weaver_head, 1, mfQ)

    if output_is_20M_p:
        fg.connect (mfI, interpI)
        fg.connect (mfQ, interpQ)
        fg.connect (interpI, weaver_tail)
        fg.connect (interpQ, weaver_tail)
    else:
        fg.connect (mfI, weaver_tail)
        fg.connect (mfQ, weaver_tail)

    fg.connect (weaver_tail, sink)
    return fg


def usage ():
    sys.stderr.write ("usage: %s [-s {20 | 2x}] -f <filename> -o <filename>\n" % sys.argv[0])
    sys.exit (1)
    
    
def main ():
    atsc_tx_FLAGS = ''
    input_filename = None
    output_filename = None
    output_is_20M_p = True

    try:
        opts, args = getopt.getopt (sys.argv[1:],
                                    "s:f:o:", ["20", "2x"])
    except getopt.GetoptError:
        usage ()

    for o, a in opts:
        if o == '-f':
            input_filename = a
        if o == '-o':
            output_filename = a
        if o == '--20':
            output_is_20M_p = True
        if o == '--2x':
            output_is_20M_p = False
        if o == '-s':
            if a == '20':
                output_is_20M_p = True
            elif a == '2x':
                output_is_20M_p = False
            else:
                usage ()

    if input_filename == None or output_filename == None:
        usage ()
        
    fg = build_graph (input_filename, output_filename, output_is_20M_p)

    fg.start ()                         # fork thread(s) and return immediately
    # your GUI main loop here
    fg.wait ()                          # wait (forever)


if __name__ == '__main__':
    main ()
