/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/

/**
 * @file mhd/test_typst.c
 * @brief Tests for Typst-MHD logic
 * @author Christian Grothoff <christian@grothoff.org>
 */
#include "taler/platform.h"
#include "taler/taler_util.h"
#include "taler/taler_mhd_lib.h"

static int global_ret;

static struct TALER_MHD_TypstContext *tc;

static int keep_output;

static void
do_shutdown (void *cls)
{
  if (NULL != tc)
  {
    TALER_MHD_typst_cancel (tc);
    tc = NULL;
  }
}


/**
 * Function called with the result of a #TALER_MHD_typst() operation.
 *
 * @param cls closure
 * @param tr result of the operation
 */
static void
result_cb (void *cls,
           const struct TALER_MHD_TypstResponse *tr)
{
  tc = NULL;
  if (TALER_EC_NONE != tr->ec)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "PDF generation failed\n");
    global_ret = 1;
  }
  else
  {
    GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                "PDF created at %s\n",
                tr->details.filename);
  }
  GNUNET_SCHEDULER_shutdown ();
}


/**
 * Main function that will be run.
 */
static void
run (void *cls,
     char *const *args,
     const char *cfgfile,
     const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  char *datadir
    = GNUNET_OS_installation_get_path (TALER_EXCHANGE_project_data (),
                                       GNUNET_OS_IPK_DATADIR);
  json_t *forms[] = {
    GNUNET_JSON_PACK (
      GNUNET_JSON_pack_string ("VQF_MEMBER_NUMBER",
                               "12345"),
      GNUNET_JSON_pack_string ("FILE_NUMBER",
                               "-1"),
      GNUNET_JSON_pack_string ("DATADIR",
                               datadir)),
    GNUNET_JSON_PACK (
      GNUNET_JSON_pack_string ("VQF_MEMBER_NUMBER",
                               "54321"),
      GNUNET_JSON_pack_string ("FILE_NUMBER",
                               "-1"),
      GNUNET_JSON_pack_string ("DATADIR",
                               datadir)),
    NULL,
  };
  struct TALER_MHD_TypstDocument docs[] = {
    {
      .form_name = "test_typst_1.typ",
      .data = forms[0],
    },
    {
      .form_name = "test_typst_1.typ",
      .data = forms[1],
    },
  };

  GNUNET_SCHEDULER_add_shutdown (&do_shutdown,
                                 NULL);
  tc = TALER_MHD_typst (cfg,
                        keep_output ? true : false,
                        "test-typst",
                        2,
                        docs,
                        &result_cb,
                        NULL);
  for (unsigned int i = 0; NULL != forms[i]; i++)
    json_decref (forms[i]);
  GNUNET_free (datadir);
}


int
main (int argc,
      const char *const argv[])
{
  const char *argvx[] = {
    "test_typst",
    "-c", "test_typst.conf",
    "-L", "INFO",
    NULL
  };
  struct GNUNET_GETOPT_CommandLineOption options[] = {
    GNUNET_GETOPT_option_flag ('k',
                               "keep-output",
                               "do not delete directory at the end of the test",
                               &keep_output),
    GNUNET_GETOPT_OPTION_END
  };

  (void) argc;
  (void) argv;
  GNUNET_log_setup ("test-typst",
                    "INFO",
                    NULL);
  if (GNUNET_OK !=
      GNUNET_PROGRAM_run (TALER_EXCHANGE_project_data (),
                          5, (char **) argvx,
                          "test_typst",
                          "Test typst",
                          options,
                          &run,
                          NULL))
  {
    GNUNET_break (0);
    return 1;
  }
  return global_ret;
}


/* end of test_typst.c */
