import {
  LocalNotificationBanner,
  notifyInfo,
  RouteDefinition,
  useBankCoreApiContext,
  useLocalNotification,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useSessionState } from "../hooks/session.js";
import { ConversionRateClassForm } from "./admin/ConversionRateClassForm.js";
import { useState } from "preact/hooks";
import { TalerCorebankApi } from "@gnu-taler/taler-util";
import { HttpStatusCode } from "@gnu-taler/taler-util";
import { assertUnreachable } from "@gnu-taler/taler-util";
import { TalerErrorCode } from "@gnu-taler/taler-util";

interface Props {
  routeCancel: RouteDefinition;
  onCreated: (id: number) => void;
}
export function NewConversionRateClass({
  routeCancel,
  onCreated,
}: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: credentials } = useSessionState();
  const token =
    credentials.status !== "loggedIn" ? undefined : credentials.token;
  const {
    lib: { bank: api },
  } = useBankCoreApiContext();

  const [notification, notify, handleError] = useLocalNotification();

  const [submitData, setSubmitData] = useState<
    TalerCorebankApi.ConversionRateClassInput | undefined
  >();

  async function doCreate() {
    if (!submitData || !token) return;
    await handleError(async () => {
      const resp = await api.createConversionRateClass(token, submitData);
      if (resp.type === "ok") {
        notifyInfo(i18n.str`Conversion rate class created.`);
        onCreated(resp.body.conversion_rate_class_id);
        return;
      }
      switch (resp.case) {
        case HttpStatusCode.Unauthorized: {
          break;
        }
        case TalerErrorCode.BANK_NAME_REUSE: {
          break;
        }
        case HttpStatusCode.Forbidden: {
          break;
        }
        case HttpStatusCode.NotFound: {
          break;
        }
        case HttpStatusCode.NotImplemented: {
          break;
        }
        default: {
          assertUnreachable(resp);
        }
      }
    });
  }

  return (
    <div class="grid grid-cols-1 gap-x-8 gap-y-8 pt-6 md:grid-cols-3 bg-gray-100 my-4 px-4 pb-4 rounded-lg">
      <LocalNotificationBanner notification={notification} />

      <div class="px-4 sm:px-0">
        <h2 class="text-base font-semibold leading-7 text-gray-900">
          <i18n.Translate>New conversion rate class</i18n.Translate>
        </h2>
      </div>

      <ConversionRateClassForm onChange={setSubmitData}>
        <div class="flex items-center justify-between gap-x-6 border-t border-gray-900/10 px-4 py-4 sm:px-8">
          <a
            href={routeCancel.url({})}
            name="cancel"
            class="text-sm font-semibold leading-6 text-gray-900"
          >
            <i18n.Translate>Cancel</i18n.Translate>
          </a>
          <button
            type="submit"
            name="create"
            class="disabled:opacity-50 disabled:cursor-default cursor-pointer rounded-md bg-indigo-600 px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            disabled={!submitData}
            onClick={(e) => {
              e.preventDefault();
              doCreate();
            }}
          >
            <i18n.Translate>Create</i18n.Translate>
          </button>
        </div>
      </ConversionRateClassForm>
    </div>
  );
}
