/*
 * This file belongs to FreeMiNT.  It's not in the original MiNT 1.12
 * distribution.  See the file Changes.MH for a detailed log of changes.
 */

/* @(#)ssystem2.c FreeMiNT
 * $Id$
 * by jerry g geiger
 *  - jerry@zedat.fu-berlin.de  or jerry@merlin.abacus.de
 * almost completely rewritten by Draco, draco@mi.com.pl, Warszawa, 4.XII.1997. 
 * 
 * added time related stuff, Guido, gufl0000@stud.uni-sb.de Mar/Apr 1998.
 * 

 General purpose: access vital system variables and constants without need
 to switch to Supervisor mode. Prototype:

 long ARGS_ON_STACK s_system (int mode, ulong arg1, ulong arg2);

 */

# include "ssystem.h"

# include "version.h"
# include "cookie.h"
# include "time.h"


# ifdef VM_EXTENSION
extern int vm_in_use;		/* from main.c */
# endif
extern int no_mem_prot;
extern int secure_mode;
extern int time_slice;		/* from proc.c */
extern short forcefastload;	/* from mem.c */
extern long sync_time;		/* from main.c */
extern long MiNT_time;		/* from build.c */
extern long MiNT_date;		/* from build.c */

# if 0
short run_level = 1;		/* default runlevel */
# endif
short disallow_single = 0;


/*
 * definitions in version.h
 */

static long MiNT_version =
	((ulong) MAJ_VERSION << 24)	|
	((ulong) MIN_VERSION << 16)	|
	((ulong) PAT_LEVEL << 8)	|
	(ulong) BETA_IDENT;


/* 
 * find cookie cookiep->tag and return its value in
 * cookiep->value, return E_OK (0)
 * return ERROR (-1) if not found
 */

static long
get_cookie (ulong arg1, long *arg2)
{
	COOKIE *cjar = *CJAR;	/* for compatibility */
	int slotnum = 0;	/* number of already taken slots */
	
	TRACE(("entering get_cookie(): tag=%8x arg2=%8x", arg1, arg2));
	
	/* If arg1 == 0, we return the value of NULL slot
	 */
	if (arg1 == 0)
	{
		TRACE(("get_cookie(): searching for NULL slot"));
		while (cjar->tag)
			cjar++;
		
		/* If arg2 is a zero, the value is returned in the d0, otherwise
		 * the arg2 is considered a pointer where the value should be put to
		 */
		if (arg2 != 0)
		{
			*arg2 = cjar->value;
			TRACE(("exit get_cookie(): NULL value written to %8x", arg2));
			return E_OK;
		}
		else
		{
			TRACE(("exit get_cookie(): NULL value returned in d0"));
			return cjar->value;
		}

	}
	
	/* if the high word of arg1 is zero, this is the slot number
	 * to look at. The first slot is number 1.
	 */
	if ((arg1 & 0xffff0000UL) == 0)
	{
		TRACE(("get_cookie(): looking for entry number %d",arg1));
		while (cjar->tag)
		{
			cjar++;
			slotnum++;
		}
		slotnum++;
		if (arg1 > slotnum)
		{
			DEBUG(("get_cookie(): entry number too big"));
			return ERROR;
		}
		cjar = *CJAR;
		slotnum = 1;
		while (slotnum != arg1)
		{
			slotnum++;
			cjar++;
		}
		if (arg2)
		{
			*arg2 = cjar->tag;
			TRACE(("get_cookie(): tag returned at %8x",arg2));
			return E_OK;
		}
		else
		{
			TRACE(("get_cookie(): tag returned in d0"));
			return cjar->tag;
		}
	}	
	
	/* all other values of arg1 mean tag id to search for */

	TRACE(("get_cookie(): searching for tag %8x", arg1));	
	while (cjar->tag)
	{
		if (cjar->tag == arg1)
		{
			if (arg2)
			{
				*arg2 = cjar->value;
				TRACE(("get_cookie(): value returned at %8x",arg2));
				return E_OK;
			}
			else
			{
				TRACE(("get_cookie(): value returned in d0"));
				return cjar->value;
			}
		}
		cjar++;
	}
	
	DEBUG(("get_cookie(): lookup failed"));
	return ERROR;
}


/* 
 * add cookie cookiep->tag to cookie list or change it's value
 * if already existing
 * 
 */

static long
set_cookie (ulong arg1, ulong arg2)
{
	COOKIE *cjar = *CJAR;	/* for compatibility. */
	long n = 0;
	
	/* 0x0000xxxx feature of GETCOOKIE may be confusing, so
	 * prevent users from using slotnumber HERE :)
	 */
	TRACE(("entering set_cookie(): tag %8x arg2 %8x", arg1, arg2));
	if (	((arg1 & 0xff000000UL) == 0) ||
		((arg1 & 0x00ff0000UL) == 0) ||
		((arg1 & 0x0000ff00UL) == 0) ||
		((arg1 & 0x000000ffUL) == 0))
	{
		DEBUG(("set_cookie(): invalid tag id %8x", arg1));
		return ERROR;	/* Bad tag id */
	}
	
	TRACE(("set_cookie(): jar lookup"));
	
	while (cjar->tag)
	{
		n++;
		if (cjar->tag == arg1)
		{
			cjar->value = arg2;
			TRACE(("set_cookie(): old entry %8x updated"));
			return E_OK;
		}
		cjar++;
	}
	
	n++;
	if (n < cjar->value)
	{
		n = cjar->value;
		cjar->tag = arg1;
		cjar->value = arg2;
		
		cjar++;
		cjar->tag = 0L;
		cjar->value = n;
		
		TRACE(("set_cookie(): new entry"));
		return E_OK;
	}
	
	/* LIST exhausted :-) */
	DEBUG(("set_cookie(): unable to place an entry, jar full"));
	return ENSMEM;
}

long ARGS_ON_STACK
s_system (int mode, ulong arg1, ulong arg2)
{
	int isroot = 0;		/* check euid -> 1, ruid --> 2	*/
	ulong *lpointer;
	ushort *wpointer;
	uchar *bpointer;
	ulong *sysbase;
	short *mfp;
	
	register long r = E_OK;
	
	TRACE(("enter s_system(): mode %d", mode));
	
	/* don't expand this with ruid check, daemons aren't supposed
	 * to run this code.
	 */
	isroot = (curproc->euid == 0);
	
	switch (mode) 
	{
		case -1:
		{
			TRACE(("exit s_system(): call exists"));
			break;
		}
		/*
		 * Kernel information
		 */
		case OSNAME:
		{
			r = 0x4d694e54L; 	/* 'MiNT' */
			break;
		}
		case OSXNAME:
		{
			r = VERS_IDENT;
			break;
		}					
		case OSVERSION:
		{
			r = MiNT_version;
			break;
		}					
		case TOSHEADER:
		{
			/* We return a complete and corrected OSHEADER. */
			sysbase = *((ulong **)(0x4f2L));
			arg1 &= 0x000000fe;	/* 256 bytes allowed */
			if (((ushort *)sysbase)[1] < 0x102 && arg1 >= 0x20 && arg1 <= 0x2c)
			{
				/* BUG: RAM-TOS before 1986 could differ! */
				static const long sysbase0x100[2][4] =
				     {{0x56FA, 0xE1B, 0x602C, 0},
				      {0x7E0A, 0xE1B, 0x873C, 0}};
				r = sysbase0x100[(((ushort *)sysbase)[14] >> 1) == 4][(arg1 - 0x20) / sizeof(ulong)];
			}
			else
			{
				/* Correct a bug in TOS >= 1.02 during AUTO folder
				 * or with AHDI, correcting an AHDI bug.
				 * Only necessary for os_conf!
				 */
				if (arg1 == 0x1a)	/* special to read os_conf */
				{
					sysbase = (ulong *)(sysbase[2]);
					r = ((ushort *)sysbase)[14];
				}
				else if (arg1 == 0x1c)
				{
					r = ((ushort *)sysbase)[15];
					sysbase = (ulong *)(sysbase[2]);
					r |= (ulong)(((ushort *)sysbase)[14]) << 16;
				}
				else
					r = sysbase[arg1 / sizeof(ulong)];
			}
			break;
		}					
		case OS_BUILD_DATE:
		{
			r = MiNT_date;
			break;
		}					
		case OS_BUILD_TIME:
		{
			r = MiNT_time;
			break;
		}					
		case COMPILE_TYPE:
		{
# ifdef CPU060
			r = 0x0000003cL;	/* 060 kernel */
# endif
# ifdef CPU040
			r = 0x00000028L;	/* 040 kernel */
# endif
# ifdef CPU030
			r = 0x0000001eL;	/* 030 kernel */
# endif
			break;			/* generic 68000 */
		}					
		case FEATURES:
		{
#ifdef VM_EXTENSION
			r =  (((!no_mem_prot)&0x01)|((vm_in_use<<1)&0x02)|(secure_mode<<2));
#else
			r =  (((!no_mem_prot)&0x01)|(secure_mode<<2));
#endif 
			break;
		}					
		/*
		 * GEMDOS variables
		 */
		case GET_LVAL:
		{
			arg1 &= 0xfffffffeUL;
			lpointer = (ulong *) arg1;
			if (arg1 < 0x08 || arg1 > 0xfffcUL)
				DEBUG (("GET_LVAL: address out of range"));
			else
				r = *lpointer;
			break;
		}					
		case GET_WVAL:
		{
			arg1 &= 0xfffffffeUL;
			wpointer = (ushort *) arg1;
			if (arg1 < 0x08 || arg1 > 0xfffeUL)
				DEBUG (("GET_WVAL: address out of range"));
			else
				r = *wpointer;
			break;
		}					
		case GET_BVAL:
		{
			bpointer = (uchar *) arg1;
			if (arg1 < 0x08 || arg1 > 0xffffUL)
				DEBUG (("GET_BVAL: address out of range"));
			else
				r = *bpointer;
			break;
		}					
		case SET_LVAL:
		{
			if (isroot == 0)
			{
				DEBUG (("SET_LVAL: access denied"));
				r = EACCDN;
				break;
			}
			arg1 &= 0xfffffffeUL;
			lpointer = (ulong *) arg1;
			if (arg1 < 0x08 || arg1 > 0xfffc)
			{
				DEBUG (("SET_LVAL: address out of range"));
				r = ERANGE;
				break;
			}
			*lpointer = arg2;
			break;
		}					
		case SET_WVAL:
		{
			if (isroot == 0)
			{
				DEBUG (("SET_WVAL: access denied"));
				r = EACCDN;
				break;
			}
			arg1 &= 0xfffffffeUL;
			wpointer = (ushort *) arg1;
			if (arg1 < 0x08 || arg1 > 0xfffe)
			{
				DEBUG (("SET_WVAL: address out of range"));
				r = ERANGE;
				break;
			}
			*wpointer = arg2;
			break;
		}					
		case SET_BVAL:
		{
			if (isroot == 0)
			{
				DEBUG (("SET_BVAL: access denied"));
				r = EACCDN;
				break;
			}
			bpointer = (uchar*) arg1;
			if (arg1 < 0x08 || arg1 > 0xffff)
			{
				DEBUG (("SET_BVAL: address out of range"));
				r = ERANGE;
				break;
			}
			*bpointer = arg2;
			break;
		}					
		/*
		 * Cookie Jar functions
		 */
		case GETCOOKIE:
		{
			r = get_cookie (arg1, (long *)arg2);
			break;
		}					
		case SETCOOKIE:
		{
			if (isroot == 0)
			{
				DEBUG (("SET_COOKIE: access denied"));
				r = EACCDN;
				break;
			}
			r = set_cookie (arg1, arg2);
			break;
		}					
		/* Hack (dirty one) for MiNTLibs */
		case TIOCMGET:
		{
			mfp = (short *) arg1;
			r = ((*mfp) & 0x00ff);
			break;
		}					
		case SYS_SLEVEL:
		{
			if (arg1 == -1)
			{
				r = (long) secure_mode;
				break;
			}
			if (isroot == 0)
			{
				DEBUG (("SYS_SLEVEL: access denied"));
				r = EACCDN;
				break;
			}
			if (arg1 < 0 || arg1 > 2)
			{
				DEBUG (("SYS_SLEVEL: arg1 out of range"));
				r = ERANGE;
				break;
			}
			secure_mode = arg1;
			break;
		}					
# if 0 /* bogus concept & code */
		case RUN_LEVEL:
		{
			if (arg1 == -1)
				return (long) run_level;
			if (isroot == 0 || disallow_single)
				return EACCDN;
			if (arg1 == 0 || arg1 == 1)
				run_level = (short) arg1;
			else
				return EACCDN;
			break;
		}					
# endif
		case CLOCK_UTC:
		{
			/* I think we shouldn't allow that if the real
			 * user-id only is root.  I think that all calls
			 * that require super-user privileges should
			 * return EACCDN if only the real user-id is
			 * root.  This does not only apply to Ssystem
			 * but to a lot of other calls too like 
			 * Pkill.  (Personal opinion of Guido).
			 * Agreed. Done. (Draco)
			 */
 			if (arg1 == -1)
 			{
				r = clock_mode;
				break;
			}
			if (arg1 < 0)
			{
				r = EINVFN;
				break;
			}
			if (isroot == 0)
			{
				r = EACCDN;
				break;
			}
			warp_clock (arg1 ? 1 : 0);
 			break;
		}
		case FORCEFASTLOAD:
		{
			if (isroot == 0)
			{
				r = EACCDN;
				break;
			}
			if (arg1 == -1)
			{
				r = (long) forcefastload;
				break;
			}
			forcefastload = (arg1 ? 1 : 0);
			break;
		}
		case SYNC_TIME:
		{
			if (isroot == 0)
			{
				r = EACCDN;
				break;
			}
			if (arg1 == -1)
			{
				r = sync_time;
				break;
			}
			if (arg1 > 0)
				sync_time = arg1;
			else
				r = ERANGE;
			break;
		}
		case TSLICE:
		{
			if (isroot == 0)
			{
				r = EACCDN;
				break;
			}
			if (arg1 == -1)
			{
				r = (long) time_slice;
				break;
			}
			if (arg1 > 0 && arg1 < 20)
				time_slice = arg1;
			else
				r = ERANGE;
			break;
		}
		default:
		{
			DEBUG (("s_system(): invalid mode %d", mode));
			r = EINVFN;	/* invalid function number */
			break;
		}
	}
	
	TRACE (("s_system() returned %ld", r));
	return r;
}
