/*
 * This file belongs to FreeMiNT.  It's not in the original MiNT 1.12
 * distribution.  See the file Changes.MH for a detailed log of changes.
 */

/* 
 * Kernel time functions.
 * 
 * Author:	Guido Flohr <gufl0000@stud.uni-sb.de> 
 * Started:	1998-03-30
 */  

# ifdef __TURBOC__
# include "include\mint.h"
# else
# include "include/mint.h"
# endif
# include "mfp.h"
# include "time.h"
# include "util.h"


# define _hz_200	(ulong *) 0x000004ba
ulong *hz_200;

ushort timestamp;
ushort datestamp;
long int clock_mode = 0;

/* Last value read at _hz_200. */
static ulong sys_lastticks;

# define SECS_PER_YEAR ((long) 364 * 24 * 3600)
/* Of course this only works because CLOCKS_PER_SEC < 1000000. */
# define MICROSECONDS_PER_CLOCK (1000000 / CLOCKS_PER_SEC)

/* The current time in UTC. */
struct timeval xtime = { 0, 0 };

/* The timezone that we're living in. */
struct timezone sys_tz = { 0, 0 };

/* Seconds west of GMT. */
static long timezone = 0;

/* Correction to apply to our kernel time to get a valid setting
 * for the hardware clock.  If the hardware clock is ticking
 * in UTC it is zero.  If it is ticking in local time it is
 * minus timezone.
 */
static long sys2tos = 0;

/* Communications variables between quick_synch and do_gettimeofday. */
static ulong timerc = 0;
static ulong last_timerc = 0xffffffff;

static ulong hardtime = 0;

static void quick_synch P_ ((void));
static ulong unix2xbios P_ ((long tv_sec));

/* Time of day in representation of the IKBD-Chip.  If somebody can
 * fix the routine that reads the values of the IKBD-clock we 
 * will be able to retrieve the system time with an accuracy of
 * one second at system start-up.
 */
struct
{
	char year;	/* Year in packed BCD-format. */
	char month;	/* Month in packed BCD-format. */
	char day;	/* Day of month in packed BCD-format. */
	char hour;	/* Hour of day in packed BCD-format. */
	char minute;	/* Minute of hour in packed BCD-format. */
	char second;	/* Second of minute in packed BCD-format. */
} ikbd_time;

/* Holds address of old clock vector.  */
long oldcvec;
/* Tells if an IKBD packet has come */
volatile short packet_came = 0;		/* volatile, f**k */

/*
 * get/set time and date original functions.  Their use is deprecated
 * now.  Use Tgettimeofday () and Tsettimeofday () instead.
 */
long ARGS_ON_STACK 
t_getdate (void) 
{ 
	return datestamp; 
}

long ARGS_ON_STACK 
t_gettime (void) 
{
	return timestamp; 
}

long ARGS_ON_STACK
t_setdate (ushort date)
{
	struct timeval tv = { 0, 0 };
	
	if (curproc->euid != 0)
	{
		DEBUG (("Tsetdate: attempt to change time by unprivileged user id %d", (int) curproc->euid));
		return EACCDN;
	}
	
	tv.tv_sec = unixtim (timestamp, date) + timezone;
	return do_settimeofday (&tv);
}

long ARGS_ON_STACK
t_settime (ushort time)
{
	struct timeval tv = { 0, 0 };
	
	if (curproc->euid != 0)
	{
		DEBUG (("Tsettime: attempt to change time by unprivileged user id %d", (int) curproc->euid));
		return EACCDN;
	}
	
	tv.tv_sec = unixtim (time, datestamp) + timezone;
	return do_settimeofday (&tv);
}

/* This function is like t_gettimeofday except that it doesn't
 * use the struct timezone.  The caller has to take care that
 * the argument TV is a valid pointer.
 *
 * The time value will be calculated from the number of
 * 200-Hz-timer-interrupts.  This will give an accuracy of
 * 5000 ms.  The microseconds inbetween are interpolated from
 * the value found in the timer-c data register.  This counter
 * down from 192 to 1 once per timer interrupt period.  This
 * provides an accuracy of little more than 26 us.
 *
 * Some programs will get confused if the time is standing still
 * for a little amount of time.  We keep track of the last timeval
 * reported plus the additional microseconds we added to the
 * calculated timevals.  In case the last timeval and the current
 * timeval are equal we add MICRO_ADJUSTMENT microseconds and 
 * increment this value.  This gives us time for about 26 us.
 * If this isn't enough we have to tell the truth however.
 */
static struct timeval last_tv = { 0, 0 };
static long micro_adjustment = 0;

long ARGS_ON_STACK
do_gettimeofday (struct timeval* tv)
{
	quick_synch ();
	
	*tv = xtime;
	
	tv->tv_usec += (192L - timerc) * MICROSECONDS_PER_CLOCK / 192L;
	
	if (last_tv.tv_sec == tv->tv_sec && last_tv.tv_usec == tv->tv_usec)
	{
		if (micro_adjustment < 25)
			micro_adjustment++;
	}
	else
	{
		micro_adjustment = 0;
	}
	
	last_tv = *tv;
	
	tv->tv_usec += micro_adjustment;
	
	if (tv->tv_usec >= 1000000)
	{
		tv->tv_usec -= 1000000;
		tv->tv_sec++;
	}
	
	return E_OK;
}

long ARGS_ON_STACK
t_gettimeofday (struct timeval* tv, struct timezone* tz)
{
	TRACE (("Tgettimeofday (tv = 0x%x, tz = 0x%x)", tv, tz));
	
	if (tz != NULL)
		memcpy (tz, &sys_tz, sizeof (sys_tz));
	
	if (tv != NULL)
		return do_gettimeofday (tv);
	
	return E_OK;
}

/* Common function for setting the kernel time.  The caller has
 * to check for appropriate privileges and valid pointer.
 */
long ARGS_ON_STACK
do_settimeofday (struct timeval* tv)
{
	ulong tos_combined;
	
	TRACE (("do_settimeofday %ld.%ld s", tv->tv_sec, 1000000 * tv->tv_usec));
	
	/* We don't allow dates before Jan 2nd, 1980.  This avoids all
	 * headaches about timezones.
	 */
# define JAN_2_1980 (10L * 365L + 2L) * 24L * 60L * 60L
	if (tv->tv_sec < JAN_2_1980)
	{
		DEBUG (("do_settimeofday: attempt to rewind time to before 1980"));
		return ERANGE;
	}
	
	/* The timeval we got is always in UTC. */
	memcpy (&xtime, tv, sizeof xtime);
	
	/* Now calculate timestamp and datestamp from that.  */
	tos_combined = unix2xbios (xtime.tv_sec - timezone);
	datestamp = (tos_combined >> 16) & 0xffff;
	timestamp = tos_combined & 0xffff;
	
	hardtime = unix2xbios (xtime.tv_sec + sys2tos);
	
	Settime (hardtime);
	
	return E_OK;
}

long ARGS_ON_STACK
t_settimeofday (struct timeval* tv, struct timezone* tz)
{
	TRACE (("Tsettimeofday (tv = 0x%x, tz = 0x%x)", tv, tz));
	
	if (curproc->euid != 0)
	{
		DEBUG (("t_settimeofday: attempt to change time by unprivileged user id %d", (int) curproc->euid));
		return EACCDN;
	}
	
	if (tz != NULL)
	{
# define TWELVE_HOURS 12 * 60
		if (tz->tz_minuteswest > TWELVE_HOURS || tz->tz_minuteswest < -TWELVE_HOURS)
		{
			DEBUG (("t_settimeofday: timezone out of range (%d minutes)", tz->tz_minuteswest));
			return ERANGE;
		}
		
		memcpy (&sys_tz, tz, sizeof (sys_tz));
		timezone = sys_tz.tz_minuteswest * 60;
		
		/* We have to distinguish now if the clock is ticking in UTC
		 * or local time.
		 */
		if (clock_mode == 0)
		{
			/* UTC. */
			sys2tos = 0;
		}
		else
		{
			sys2tos = -timezone;
		}
		
		/* Update timestamp and datestamp. */
		synch_timers ();
	}
	
	if (tv != NULL)
	{
		long retval = do_settimeofday (tv);
		hardtime = 0;
		if (retval < 0)
			return retval;
	}
	
	return E_OK;
}

/* Most programs assume that the values returned by Tgettime()
 * and Tgetdate() are in local time.  We won't disappoint them.
 * However it is favorable to have at least the high-resolution
 * kernel time running in UTC instead.  
 * 
 * This means that Tgettime(), Tsettime(), Tgetdate() and
 * Tsetdate() keep on handling local times.  All other time-
 * related new system calls (Tgettimeofday, Tsettimeofday
 * and Tadjtimex) handle UTC instead.
 */
void 
init_time ()
{
	long value = _mfpregs->tbdr;
	ulong tos_time;
	ushort *broken = (ushort *) &tos_time;
	
	/* See, a piece of code is a function, not just a long integer */
	extern long ARGS_ON_STACK newcvec();  /* In intr.spp */
	KBDVEC *kvecs;
	/* Opcode for getting ikbd clock. */
	uchar ikbd_clock_get = 0x1c;
	register short count;
	
	/* Interpolate. */
	value &= 0x000000ffL;
	value -= 192L;
	if (value < 0)
		value = 0;

	/* Get the current setting of the hardware clock. Since we only bend
	 * the vector once we don't bother about xbra stuff. We can't go
	 * the fine MiNT way (using syskey, cf. mint.h) because this routine
	 * is called before the interrupts are initialized.
	 */

	/* Draco's explanation on why this doesn't work:
	 *
	 * 1. because of a bug - this caused bombs on startup (fixed).
	 * 2. because documentation lies - the Ikbdws() writes `len' chars,
	 *    not `len+1' (also fixed).
	 * 3. because documentation lies - the pointer passed to the
	 *    clock handler doesn't point to the packet header,
	 *    but to the packet itself (fixed as well).
	 * 4. finally - because the IKBD clock is not set,
	 *    at least on Falcon030 TOS. So the returned packet
	 *    actually consists of zeros only (not fixed).
	 *
	 * Notice: even if it worked as (previously) exspected, kernel's
	 * time package might work incorrectly when no keyboard is present.
	 * 
	 */

	do {
		kvecs = (KBDVEC *) Kbdvbase ();
	}
	while (kvecs->drvstat);
	
	oldcvec = kvecs->clockvec;
	kvecs->clockvec = (long)&newcvec;
	Ikbdws (1, &ikbd_clock_get);
	for (count = 0; count < 9999 && packet_came == 0; count++);
	kvecs->clockvec = oldcvec;
	
	tos_time = Gettime ();
	datestamp = *(broken++);
	timestamp = *broken;
	hz_200 = _hz_200;
	
	sys_lastticks = *hz_200;
	
	xtime.tv_sec = unixtim (timestamp, datestamp);
	xtime.tv_usec = (sys_lastticks % CLOCKS_PER_SEC) * MICROSECONDS_PER_CLOCK
		+ value * MICROSECONDS_PER_CLOCK / 192L;
}

static void
quick_synch (void)
{
	ulong current_ticks;
	long elapsed;	/* Microseconds elapsed since last quick_synch.  
			 * Because this routine is guaranteed to be 
			 * called at least once per second we don't have 
			 * to care to much about overflows.
			 */
	
	timerc = _mfpregs->tbdr;
	current_ticks = *hz_200;
	
	/* Make sure that the clock runs monotonic.  */
	timerc &= 0x000000ffL;
	if (timerc > 192)
		timerc = 192;
	
	if (current_ticks <= sys_lastticks)
	{
		if (timerc > last_timerc)
			timerc = last_timerc;
	}
	
	last_timerc = timerc;
	
	if (current_ticks < sys_lastticks)
	{
		/* We had an overflow in _hz_200. */
		ulong uelapsed = 0xffffffff - sys_lastticks;
		uelapsed += current_ticks;
		
		elapsed = uelapsed * MICROSECONDS_PER_CLOCK;
		TRACE (("quick_synch: 200-Hz-Timer overflow"));
	}
	else
	{
		elapsed = (current_ticks - sys_lastticks) * MICROSECONDS_PER_CLOCK;
	}
	
	sys_lastticks = current_ticks;
	
	xtime.tv_usec += elapsed;
	if (xtime.tv_usec >= 1000000)
	{
		xtime.tv_sec += (xtime.tv_usec / 1000000);
		xtime.tv_usec = xtime.tv_usec % 1000000;
	}
}

/* Calculate timestamp and datestamp.  */
void ARGS_ON_STACK
synch_timers ()
{
	ulong tos_combined;
	
	quick_synch ();
	
	/* Now adjust timestamp and datestamp to be in local time.  */
	tos_combined = unix2xbios (xtime.tv_sec - timezone);
	datestamp = (tos_combined >> 16) & 0xffff;
	timestamp = tos_combined & 0xffff;
	
	if (hardtime && curproc->pid == 0)
	{
		/* Hm, if Tsetdate or Tsettime was called by the user our
		 * changes get lost. That's why this strange piece of code
		 * is here.
		 */
		hardtime = unix2xbios (xtime.tv_sec + sys2tos);
		Settime (hardtime);
		hardtime = 0;
	}
}

/* Change the kernel's notion of the system time.  */
void ARGS_ON_STACK
warp_clock (int mode)
{
	if ((mode == 0 && clock_mode == 0) || (mode != 0 && clock_mode != 0))
		return;
	
	if (clock_mode == 0)
	{
		/* Change it from UTC to local time.  */
		xtime.tv_sec += timezone;
		sys2tos = -timezone;
		clock_mode = 1;
	}
	else
	{
		/* Change it back from local time to UTC.  */
		xtime.tv_sec -= timezone;
		sys2tos = 0;
		clock_mode = 0;
	}
	
	/* Set timestamp and datestamp correctly. */
	synch_timers ();
}

/* How many days come before each month (0-12).  */
const ushort __mon_yday[2][13] =
{
  /* Normal years.  */
  { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 },
  /* Leap years.  */
  { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335 }
};

# define SECS_PER_HOUR	((long) (60 * 60))
# define SECS_PER_DAY	((long) (SECS_PER_HOUR * 24))

# ifdef __isleap
# undef __isleap
# endif
/* Lucky enough that 2000 is a leap year.  */
# define __isleap(year) ((year % 4) == 0)

static ulong
unix2xbios (long tv_sec)
{
	register long days, rem, y;
	register const ushort *ip;
	
	struct _xbios_time
	{
		unsigned year: 7;
		unsigned month: 4;
		unsigned day: 5;
		unsigned hour: 5;
		unsigned minute: 6;
		unsigned sec2: 5;
	} xbios_time;
	register struct _xbios_time *xtm = &xbios_time;
	
	ulong *retval = (ulong *) xtm;
	
	days = tv_sec / SECS_PER_DAY;
	rem = tv_sec % SECS_PER_DAY;
	xtm->hour = rem / SECS_PER_HOUR;
	rem %= SECS_PER_HOUR;
	xtm->minute = rem / 60;
	xtm->sec2 = (rem % 60) >> 1;
	
	y = 1970;
# define LEAPS_THRU_END_OF(y)	((y) / 4)
	
	while (days < 0 || days >= (__isleap (y) ? 366 : 365))
	{
		/* Guess a corrected year, assuming 365 days per year.  */
		long yg = y + days / 365 - (days % 365 < 0);
		
		/* Adjust DAYS and Y to match the guessed year.  */
		days -= ((yg - y) * 365
			+ LEAPS_THRU_END_OF (yg - 1)
			- LEAPS_THRU_END_OF (y - 1));
		y = yg;
	}
	
	xtm->year = y - 1980;
	
	ip = __mon_yday[__isleap(y)];
	for (y = 11; days < ip[y]; --y)
		continue;
	
	days -= ip[y];
	xtm->month = y + 1;
	xtm->day = days + 1;
	
	return *retval;
}

long ARGS_ON_STACK
gettime ()
{
	if (xtime.tv_sec)
	{
		return (((long) datestamp << 16) | (timestamp & 0xffff));
	}
	else
	{
		/* We're not initialized.  */
		return Gettime ();
	}
}

void ARGS_ON_STACK
settime (ulong datetime)
{
	if (hardtime)
	{
		/* Called from the kernel.  */
		Settime (hardtime);
	}
	else
	{
		struct timeval tv = { 0, 0 };
		ushort time, date;
		
		if (curproc->euid != 0)
		{
			DEBUG (("Settime: attempt to change time by unprivileged user id %d", (int) curproc->euid));
			return;
		}
		
		time = datetime & 0xffff;
		date = (datetime >> 16) & 0xffff;
		
		tv.tv_sec = unixtim (timestamp, date) + timezone;
		tv.tv_sec = unixtim (timestamp, date) + timezone;
		
		(void) do_settimeofday (&tv);
	}
}
