/*
 * This file has been modified as part of the FreeMiNT project. See
 * the file Changes.MH for details and dates.
 */

/*
 * Copyright 1990,1991,1992 Eric R. Smith.
 * Copyright 1993,1994 Atari Corporation.
 * All rights reserved.
 */

/* MiNT routines for doing console I/O */

# include "console.h"

# include "dosfile.h"
# include "tty.h"


/*
 * These routines are what Cconout, Cauxout, etc. ultimately call.
 * They take an integer argument which is the user's file handle,
 * and do the translation to a file ptr (and return an appropriate error
 * message if necessary.
 * "mode" may be RAW, COOKED, ECHO, or COOKED|ECHO.
 * note that the user may not call them directly!
 */

long
file_instat(FILEPTR *f)
{
	long r;

	if (!f) {
		return EIHNDL;
	}
	r = 1;		/* default is to assume input waiting (e.g. TOS files)*/
	if (is_terminal(f))
		(void)tty_ioctl(f, FIONREAD, &r);
	else
		(void)(*f->dev->ioctl)(f, FIONREAD, &r);
	return r;
}

long
file_outstat(FILEPTR *f)
{
	long r;

	if (!f) {
		return EIHNDL;
	}
	r = 1;		/* default is to assume output OK (e.g. TOS files) */
	if (is_terminal(f))
		(void)tty_ioctl(f, FIONWRITE, &r);
	else
		(void)(*f->dev->ioctl)(f, FIONWRITE, &r);
	return r;
}

long
file_getchar(FILEPTR *f, int mode)
{
	char c;
	long r;

	if (!f) {
		return EIHNDL;
	}
	if (is_terminal(f)) {
		return tty_getchar(f, mode);
	}
	r = (*f->dev->read)(f, &c, 1L);
	if (r != 1)
		return MiNTEOF;
	else
		return ((long)c) & 0xff;
}

long
file_putchar(FILEPTR *f, long int c, int mode)
{
	char ch;

	if (!f) {
		return EIHNDL;
	}
	if (is_terminal(f)) {
		return tty_putchar(f, c & 0x7fffffffL, mode);
	}
	ch = c & 0x00ff;
	return (*f->dev->write)(f, &ch, 1L);
}

/*
 * OK, here are the GEMDOS console I/O routines
 */

long ARGS_ON_STACK
c_conin(void)
{
	return file_getchar(curproc->handle[0], COOKED|ECHO);
}

long ARGS_ON_STACK
c_conout(int c)
{
	return file_putchar(curproc->handle[1], (long)c, COOKED);
}

long ARGS_ON_STACK
c_auxin(void)
{
	return file_getchar(curproc->handle[2], RAW);
}

long ARGS_ON_STACK
c_auxout(int c)
{
	return file_putchar(curproc->handle[2], (long)c, RAW);
}

long ARGS_ON_STACK
c_prnout(int c)
{
	return file_putchar(curproc->handle[3], (long)c, RAW);
}

long ARGS_ON_STACK
c_rawio(int c)
{
	long r;
	PROC *p = curproc;

	if (c == 0x00ff) {
		if (!file_instat(p->handle[0]))
			return 0;
		r = file_getchar(p->handle[0], RAW);
		if (r <= E_OK)
			return 0;
		return r;
	}
	else
		return file_putchar(p->handle[1], (long)c, RAW);
}

long ARGS_ON_STACK
c_rawcin(void)
{
	return file_getchar(curproc->handle[0], RAW);
}

long ARGS_ON_STACK
c_necin(void)
{
	return file_getchar(curproc->handle[0],COOKED|NOECHO);
}

long ARGS_ON_STACK
c_conws(const char *str)
{
	const char *p = str;
	long cnt = 0;

	while (*p++) cnt++;
	return f_write(1, cnt, str);
}

long ARGS_ON_STACK
c_conrs(char *buf)
{
	long size, count, r;
	char dummy;

	size = ((long)*buf) & 0xff;
        if (is_terminal(curproc->handle[0]))
        {
/*
 * TB: When reading from a terminal, f_read is used which automatically
 * breaks at the first newline
 */
                r = f_read(0, size, buf + 2);
		if (r < E_OK)
                        buf[1] = 0;
                else
                        buf[1] = (char)r;
		return (r < E_OK) ? r : E_OK;
	}
/*
 * TB: In all other cases, we must read character by character, breaking
 * at a newline or carriage return, or when the maximum number of
 * characters has been read
 */
       for (count = 0L;; count++)
       {
               if (count == size)
               {
                       buf[1] = (char)count;
                       buf[count + 2] = 0;
			return E_OK;
               }
               r = c_conin();
/* Also break on error... */
		if (r < E_OK)
               {
                       buf[count + 2] = 0;
                       buf[1] = (char)count;
                       return r;
               }
               dummy = (char)r;
/* ... or at EOF */
		if ((short)r == MiNTEOF)
               {
                       buf[count + 2] = 0;
                       buf[1] = (char)count;
			return E_OK;
               }
               if ((dummy == '\r') || (dummy == '\n'))
               {
                       buf[count + 2] = 0;
                       buf[1] = (char)count;
/*
 * When we've read a carriage return, the next char has to be skipped,
 * because it usually is a linefeed (GEMDOS-style lines end with CR/LF).
 * Problems may occur with files that have lines ending with CR only, but
 * I have not seen many yet.
 */
                       if (dummy == '\r')
                               return f_read(0, 1L, &dummy);
                       else
				return E_OK;
               }
               buf[count + 2] = dummy;

	}
}

long ARGS_ON_STACK
c_conis(void)
{
	return -(!!file_instat(curproc->handle[0]));
}

long ARGS_ON_STACK
c_conos(void)
{
	return -(!!file_outstat(curproc->handle[1]));
}

long ARGS_ON_STACK
c_prnos(void)
{
	return -(!!file_outstat(curproc->handle[3]));
}

long ARGS_ON_STACK
c_auxis(void)
{
	return -(!!file_instat(curproc->handle[2]));
}

long ARGS_ON_STACK
c_auxos(void)
{
	return -(!!file_outstat(curproc->handle[2]));
}

/* Extended GEMDOS routines */

long ARGS_ON_STACK
f_instat(int h)
{
	PROC *proc;
	int fh = h;

#if O_GLOBAL
	if (fh >= 100) {
		proc = rootproc;
		fh -= 100;
	} else
#endif
		proc = curproc;

	if (fh < MIN_HANDLE || fh >=MAX_OPEN) {
		DEBUG(("Finstat: bad handle %d", h));
		return EIHNDL;
	}
	return file_instat(proc->handle[fh]);
}

long ARGS_ON_STACK
f_outstat(int h)
{
	int fh = h;
	PROC *proc;
#if O_GLOBAL
	if (fh >= 100) {
		fh -= 100;
		proc = rootproc;
	} else
#endif
		proc = curproc;

	if (fh < MIN_HANDLE || fh >=MAX_OPEN) {
		DEBUG(("Foutstat: bad handle %d", h));
		return EIHNDL;
	}
	return file_outstat(proc->handle[fh]);
}

long ARGS_ON_STACK
f_getchar(int h, int mode)
{
	int fh = h;
	PROC *proc;

#if O_GLOBAL
	if (fh >= 100) {
		fh -= 100;
		proc = rootproc;
	} else
#endif
		proc = curproc;
	if (fh < MIN_HANDLE || fh >=MAX_OPEN) {
		DEBUG(("Fgetchar: bad handle %d", h));
		return EIHNDL;
	}
	return file_getchar(proc->handle[fh], mode);
}

long ARGS_ON_STACK
f_putchar(int h, long int c, int mode)
{
	int fh = h;
	PROC *proc;

#if O_GLOBAL
	if (fh >= 100) {
		fh -= 100;
		proc = rootproc;
	} else
#endif
		proc = curproc;

	if (fh < MIN_HANDLE || fh >=MAX_OPEN) {
		DEBUG(("Fputchar: bad handle %d", h));
		return EIHNDL;
	}
	return file_putchar(proc->handle[fh], c, mode);
}
