/*
 * This file belongs to FreeMiNT.  It's not in the original MiNT 1.12
 * distribution.  See the file Changes.MH for a detailed log of changes.
 */

/*
 * This file is dedicated to the FreeMiNT project.
 * It's not allowed to use this file for other projects without my
 * explicit permission.
 */

/*
 * begin:	1998-06
 * last change: 1998-09-10
 * 
 * Author: Frank Naumann - <fnaumann@cs.uni-magdeburg.de>
 * 
 * please send suggestions, patches or bug reports to me or
 * the MiNT mailing list
 * 
 */

# ifndef _BLOCK_IO_H
# define _BLOCK_IO_H

# ifdef __TURBOC__
# include "include\misc.h"
# include "include\types.h"
# else
# include "misc.h"
# include "types.h"
# endif


/*
 * definitions
 */

typedef struct di DI;
typedef struct unit UNIT;
typedef struct cbl CBL;

/* device identifikator */
struct di
{
	DI	*next;		/* internal: next in linked list */
	UNIT	**table;	/* internal: unit hash table */
	UNIT	*wb_queue;	/* internal: writeback queue */
	
	const ushort drv;	/* internal: BIOS device number (unique) */
	ushort	major;		/* XHDI */
	ushort	minor;		/* XHDI */
	ushort	mode;		/* internal: some flags */
	
# define BIO_WP_MODE	0x01	/* writeprotect bit (soft/hard) */
# define BIO_WB_MODE	0x02	/* writeback bit (soft) */
# define BIO_REMOVABLE	0x04	/* removable media */
# define BIO_LRECNO	0x10	/* lrecno supported */
	
	ulong	start;		/* physical start sector */
	ulong	size;		/* physical sectors */
	ulong	pssize;		/* internal: physical sector size */
	
	ushort	pshift;		/* internal: size to count calculation */
	ushort	lshift;		/* internal: logical to physical recno calculation */
	
	long	(*rwabs)(DI *di, ushort rw, void *buf, ulong size, ulong lrecno);
	long	(*dskchng)(DI *di);
	
	ushort	valid;		/* internal: DI valid */
	ushort	lock;		/* internal: DI in use */
	
	char	id[4];		/* partition id (GEM, BGM, RAW, \0D6, ...) */
	ushort	key;		/* XHDI key */
	
	char	res[18];	/* reserved for future */
};

/* cache unit */
struct unit
{
	char	*data;		/* pointer to the data */
	UNIT	*next;		/* next in hash table */
	UNIT	*wb_prev;	/* prev in writeback queue */
	UNIT	*wb_next;	/* next in writeback queue */
	CBL	*cbl;		/* the parent cache block */
	DI	*di;		/* device identifikator */
	long	sector;		/* start sector on medium */
	long	size;		/* size in bytes */
	ulong	stat;		/* access statistic */
	short	pos;		/* location in block */
	short	dirty;		/* unit is modified (flag) */
	short	lock;		/* unit is locked (counter) */
	short	res;		/* reserved for future */	
};


/*
 * interface
 */

typedef struct
{
	short	version;	/* buffer cache version */
	short	reserved;	/* reserved for future */
	long	(*config)	(const ushort drv, const long config, const long mode);
	
/* config: */
# define BIO_WP		1	/* configuring writeprotect feature */
# define BIO_WB		2	/* configuring writeback mode */
# define BIO_MAX_BLOCK	10	/* maximum cacheable blocksize */
# define BIO_DEBUGLOG	100	/* only for debugging, kernel internal */
# define BIO_DEBUG_T	101	/* only for debugging, kernel internal */
	
	/* DI management */
	DI *	(*get_di)	(ushort drv);
	DI *	(*res_di)	(ushort drv);
	void	(*free_di)	(DI *di);
	
	/* physical/logical calculation init */
	void	(*set_pshift)	(DI *di, ulong physical);
	void	(*set_lshift)	(DI *di, ulong logical);
	
	/* cached block I/O */
	UNIT *	(*lookup)	(DI *di, long sector, long blocksize);
	UNIT *	(*getunit)	(DI *di, long sector, long blocksize);
	UNIT *	(*read)		(DI *di, long sector, long blocksize);
	long	(*write)	(UNIT *u);
	long	(*l_read)	(DI *di, long sector, long blocks, long blocksize, void *buf);
	long	(*l_write)	(DI *di, long sector, long blocks, long blocksize, void *buf);
	
	/* optional feature */
	void	(*pre_read)	(DI *di, long *sector, long blocks, long blocksize);
	
	/* synchronization */
	void	(*lock)		(UNIT *u);
	void	(*unlock)	(UNIT *u);
	
	/* update functions */
	void	(*mark_modified)(UNIT *u);
	void	(*sync_drv)	(DI *di);
	
	/* cache management */
	long	(*validate)	(DI *di, long maxblocksize);
	void	(*invalidate)	(DI *di);
	
	long	res[6];		/* reserved for future */
} BIO;


/*
 * useful makros
 */

# define BIO_RWABS(di, rw, buf, size, recno) \
				((*(di->rwabs))(di, rw, buf, size, recno))
# define BIO_DSKCHNG(di)	((*(di->dskchng))(di))

# define BIO_WP_CHECK(di)	((di->mode & BIO_WP_MODE) ? ENABLE : DISABLE)
# define BIO_WB_CHECK(di)	((di->mode & BIO_WB_MODE) ? ENABLE : DISABLE)
# define BIO_LR_CHECK(di)	((di->mode & BIO_LRECNO) ? ENABLE : DISABLE)

# ifdef USE_INLINE_FUNCS
FASTFN void bio_MARK_MODIFIED (BIO *bio, UNIT *u){ if (!u->dirty) (bio)->mark_modified (u); }
FASTFN void bio_SYNC_DRV (BIO *bio, DI *di){ if (!BIO_WB_CHECK (di)) (bio)->sync_drv (di); }
# else
# define bio_MARK_MODIFIED(bio, u){ if (!u->dirty) (bio)->mark_modified (u); }
# define bio_SYNC_DRV(bio, di)	{ if (!BIO_WB_CHECK (di)) (bio)->sync_drv (di); }
# endif


# endif /* _BLOCK_IO_H */
