/*
 * This file has been modified as part of the FreeMiNT project. See
 * the file Changes.MH for details and dates.
 */

/*
 * Copyright 1990,1991,1992 Eric R. Smith.
 * Copyright 1992,1993 Atari Corporation.
 * All rights reserved.
 */

/*
 * misc. utility routines
 */

# include "util.h"

/*
 * given an address, find the corresponding memory region in this program's
 * memory map
 */

MEMREGION *
addr2mem (virtaddr a)
{
	register long i;

	for (i = 0; i < curproc->num_reg; i++)
	{
		if (a == curproc->addr[i])
		{
			return curproc->mem[i];
		}
	}
	
	return NULL;
}

/*
 * given a pid, return the corresponding process
 */

PROC *
pid2proc (int pid)
{
	register PROC *p;

	for (p = proclist; p; p = p->gl_next)
	{
		if (p->pid == pid)
			return p;
	}
	
	return NULL;
}

/*
 * return a new pid
 */

static int _maxpid = 2;
	
int
newpid (void)
{
	register int i;
# ifndef NDEBUG
	register int j = 0;
# endif
	
	do {
		i = _maxpid++;
		if (_maxpid >= 1000) _maxpid = 2;
		assert (j++ < 1000);
	}
	while (pid2proc (i));
	
	return i;
}

/*
 * set up to run the init program as PID 1
 */
int
set_pid_1 (void)
{
	if (pid2proc (1))
		/* should never happen, only called once */
		return -1;
	
	_maxpid = 1;
	return 0;
}

/*
 * zero out a block of memory, quickly; the block must be word-aligned,
 * and should be long-aligned for speed reasons
 */

void
zero (char *place, long size)
{
	register long cruft;
	register long blocksize;
	
	cruft = size % 256;	/* quickzero does 256 byte blocks */
	blocksize = size / 256;	/* divide by 256 */
	if (blocksize > 0)
	{
		quickzero (place, blocksize);
		place += (blocksize * 256);
	}
	while (cruft > 0)
	{
		*place++ = 0;
		cruft--;
	}
}

# ifdef JUNK_MEM
void
fillwjunk (long *place, long size)
{
	while (size > 0)
	{
		*place++ = size;
		size -= 4;
	}
}
# endif

/*
 * convert a time in milliseconds to a GEMDOS style date/time
 * timeptr[0] gets the time, timeptr[1] the date.
 * BUG/FEATURE: in the conversion, it is assumed that all months have
 * 30 days and all years have 360 days.
 */

void ARGS_ON_STACK 
ms_time (ulong ms, short int *timeptr)
{
	register ulong secs = ms;
	register short tsec, tmin, thour;
	register short tday, tmonth, tyear;

	secs /= 1000;
	tsec = secs % 60;
	
	secs /= 60;		/* secs now contains # of minutes */
	tmin = secs % 60;
	
	secs /= 60;		/* secs now contains # of hours */
	thour = secs % 24;
	
	secs /= 24;		/* secs now contains # of days */
	tday = secs % 30;
	
	secs /= 30;
	tmonth = secs % 12;
	tyear = secs / 12;
	
	*timeptr++ = (thour << 11) | (tmin << 5) | (tsec >> 1);
	*timeptr = (tyear << 9) | ((tmonth + 1) << 5) | (tday + 1);
}

/*
 * unixtim (time, date): convert a Dos style (time, date) pair into
 * a Unix time (seconds from midnight Jan 1., 1970)
 */

static int const
mth_start[13] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 };

long ARGS_ON_STACK 
unixtim (ushort time, ushort date)
{
	register int sec, min, hour;
	register int mday, mon, year;
	register long s;
	
	sec	= (time & 31) << 1;
	min	= (time >> 5) & 63;
	hour	= (time >> 11) & 31;
	
	mday	= date & 31;
	mon	= ((date >> 5) & 15) - 1;
	year	= 80 + ((date >> 9) & 127);
	
	/* calculate tm_yday here */
	s = (mday - 1) + mth_start[mon] + /* leap year correction */
		(((year % 4) != 0 ) ? 0 : (mon > 1));
	
	s = (sec) + (min * 60L) + (hour * 3600L) +
		(s * 86400L) + ((year - 70) * 31536000L) +
		((year - 69) / 4) * 86400L;
	
	return s;
}

/* convert a Unix time into a DOS time. The longword returned contains
 * the time word first, then the date word.
 * BUG: we completely ignore any time zone information.
 */
# define SECS_PER_MIN		(60L)
# define SECS_PER_HOUR		(3600L)
# define SECS_PER_DAY		(86400L)
# define SECS_PER_YEAR		(31536000L)
# define SECS_PER_LEAPYEAR	(SECS_PER_DAY + SECS_PER_YEAR)

static int
days_per_mth[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

long ARGS_ON_STACK 
dostim (long time)
{
	register int tm_hour, tm_min, tm_sec;
	register int tm_year, tm_mon, tm_mday;
	register long t = time;
	
	if (t <= 0) return 0;
	
	tm_year = 70;
	while (t >= SECS_PER_YEAR)
	{
		if ((tm_year & 0x3) == 0)
		{
			if (t < SECS_PER_LEAPYEAR)
				break;
			t -= SECS_PER_LEAPYEAR;
		}
		else
		{
			t -= SECS_PER_YEAR;
		}
		tm_year++;
	}
	
	tm_mday = (int)(t / SECS_PER_DAY);
	days_per_mth[1] = (tm_year & 0x3) ? 28 : 29;
	{
		register long i;
		for (i = 0; tm_mday >= days_per_mth[i]; i++)
			tm_mday -= days_per_mth[i];
	
		tm_mon = i + 1;
	}
	tm_mday++;
	
	t = t % SECS_PER_DAY;
	tm_hour = (int)(t / SECS_PER_HOUR);
	
	t = t % SECS_PER_HOUR;
	tm_min = (int)(t / SECS_PER_MIN);
	tm_sec = (int)(t % SECS_PER_MIN);
	
	if (tm_year < 80)
	{
		tm_year = 80;
		tm_mon = tm_mday = 1;
		tm_hour = tm_min = tm_sec = 0;
	}
	
	{
        	register ulong time;
        	register ulong date;
		
		time = (tm_hour << 11) | (tm_min << 5) | (tm_sec >> 1);
		date = ((tm_year - 80) & 0x7f) << 9;
		date |= ((tm_mon) << 5) | (tm_mday);
		
		return (time << 16) | date;
	}
}

/*
 * Case insensitive string comparison. note that this only returns
 * 0 (match) or nonzero (no match), and that the returned value
 * is not a reliable indicator of any "order".
 */

int ARGS_ON_STACK 
strnicmp (register const char *str1, register const char *str2, register int len)
{
	register char c1, c2;

	do {
		c1 = *str1++; if (isupper(c1)) c1 = _tolower(c1);
		c2 = *str2++; if (isupper(c2)) c2 = _tolower(c2);
	}
	while (--len >= 0 && c1 && c1 == c2);
	
	if (len < 0 || c1 == c2)
		return 0;
	
	return c1 - c2;
}

int ARGS_ON_STACK 
stricmp(const char *str1, const char *str2)
{
	return strnicmp (str1, str2, 0x7fff);
}


/*
 * string utilities: strlwr() converts a string to lower case,
 *                   strupr() converts it to upper case
 */

char * ARGS_ON_STACK 
strlwr (char *s)
{
	char c;
	char *old = s;
	
	while ((c = *s) != 0)
	{
		if (isupper (c))
		{
			*s = _tolower(c);
		}
		s++;
	}
	
	return old;
}

char * ARGS_ON_STACK 
strupr (char *s)
{
	char c;
	char *old = s;

	while ((c = *s) != 0)
	{
		if (islower (c))
		{
			*s = _toupper(c);
		}
		s++;
	}
	
	return old;
}

# ifdef OWN_LIB

/*
 * Case sensitive comparison functions.
 */

int
strncmp (register const char *str1, register const char *str2, register int len)
{
	register char c1, c2;

	do {
		c1 = *str1++;
		c2 = *str2++;
	}
	while (--len >= 0 && c1 && c1 == c2);
	
	if (len < 0) return 0;
	
	return c1 - c2;
}

int
strcmp (register const char *str1, register const char *str2)
{
	register char c1, c2;
	
	do {
		c1 = *str1++;
		c2 = *str2++;
	}
	while (c1 && c1 == c2);
	
	return c1 - c2;
}

/*
 * some standard string functions
 */

char *
strcat (char *dst, const char *src)
{
	register char *_dscan;
	
	for (_dscan = dst; *_dscan; _dscan++) ;
	while ((*_dscan++ = *src++) != 0) ;
	
	return dst;
}

char *
strcpy (char *dst, const char *src)
{
	register char *_dscan = dst;
	
	while ((*_dscan++ = *src++) != 0) ;
	
	return dst;
}

char *
strncpy (char *dst, const char *src, int len)
{
	register char *_dscan = dst;
	
	while (--len >= 0 && (*_dscan++ = *src++) != 0)
		continue;
	while (--len >= 0)
		*_dscan++ = 0;
	
	return dst;
}

int
strlen (const char *scan)
{
	register const char *_start = scan + 1;
	while (*scan++) ;
	return (int)((long) scan - (long) _start);
}

/*
 * strrchr: find the last occurence of a character in a string
 */

char *
strrchr (const char *str, register int which)
{
	register uchar c, *s;
	register char *place;
	
	s = (uchar *)str;
	place = 0;
	do {
		c = *s++;
		if (c == which)
			place = (char *) s - 1;
	}
	while (c);
	
	return place;
}

uchar _ctype[256] =
{
	_CTc, _CTc, _CTc, _CTc,				/* 0x00..0x03 */
	_CTc, _CTc, _CTc, _CTc,				/* 0x04..0x07 */
	_CTc, _CTc|_CTs, _CTc|_CTs, _CTc|_CTs,		/* 0x08..0x0B */
	_CTc|_CTs, _CTc|_CTs, _CTc, _CTc,		/* 0x0C..0x0F */

	_CTc, _CTc, _CTc, _CTc,				/* 0x10..0x13 */
	_CTc, _CTc, _CTc, _CTc,				/* 0x14..0x17 */
	_CTc, _CTc, _CTc, _CTc,				/* 0x18..0x1B */
	_CTc, _CTc, _CTc, _CTc,				/* 0x1C..0x1F */

	_CTs, _CTp, _CTp, _CTp,				/* 0x20..0x23 */
	_CTp, _CTp, _CTp, _CTp,				/* 0x24..0x27 */
	_CTp, _CTp, _CTp, _CTp,				/* 0x28..0x2B */
	_CTp, _CTp, _CTp, _CTp,				/* 0x2C..0x2F */

	_CTd|_CTx, _CTd|_CTx, _CTd|_CTx, _CTd|_CTx,	/* 0x30..0x33 */
	_CTd|_CTx, _CTd|_CTx, _CTd|_CTx, _CTd|_CTx,	/* 0x34..0x37 */
	_CTd|_CTx, _CTd|_CTx, _CTp, _CTp,		/* 0x38..0x3B */
	_CTp, _CTp, _CTp, _CTp,				/* 0x3C..0x3F */

	_CTp, _CTu|_CTx, _CTu|_CTx, _CTu|_CTx,		/* 0x40..0x43 */
	_CTu|_CTx, _CTu|_CTx, _CTu|_CTx, _CTu,		/* 0x44..0x47 */
	_CTu, _CTu, _CTu, _CTu,				/* 0x48..0x4B */
	_CTu, _CTu, _CTu, _CTu,				/* 0x4C..0x4F */

	_CTu, _CTu, _CTu, _CTu,				/* 0x50..0x53 */
	_CTu, _CTu, _CTu, _CTu,				/* 0x54..0x57 */
	_CTu, _CTu, _CTu, _CTp,				/* 0x58..0x5B */
	_CTp, _CTp, _CTp, _CTp,				/* 0x5C..0x5F */

	_CTp, _CTl|_CTx, _CTl|_CTx, _CTl|_CTx,		/* 0x60..0x63 */
	_CTl|_CTx, _CTl|_CTx, _CTl|_CTx, _CTl,		/* 0x64..0x67 */
	_CTl, _CTl, _CTl, _CTl,				/* 0x68..0x6B */
	_CTl, _CTl, _CTl, _CTl,				/* 0x6C..0x6F */

	_CTl, _CTl, _CTl, _CTl,				/* 0x70..0x73 */
	_CTl, _CTl, _CTl, _CTl,				/* 0x74..0x77 */
	_CTl, _CTl, _CTl, _CTp,				/* 0x78..0x7B */
	_CTp, _CTp, _CTp, _CTc,				/* 0x7C..0x7F */

	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0x80..0x8F */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0x90..0x9F */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xA0..0xAF */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xB0..0xBF */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xC0..0xCF */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xD0..0xDF */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xE0..0xEF */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0  /* 0xF0..0xFF */
};

int
toupper (int c)
{
	return (islower (c) ? _toupper (c) : c);
}

int
tolower(int c)
{
	return (isupper (c) ? _toupper (c) : c);
}

/*
 * converts a decimal string to an integer
 */

long
atol (const char *s)
{
	long d = 0;
	int negflag = 0;
	int c;

	while (*s && isspace (*s)) s++;
	while (*s == '-' || *s == '+')
	{
		if (*s == '-')
			negflag ^= 1;
		s++;
	}
	while ((c = *s++) != 0 && isdigit (c))
	{
		d = 10 * d + (c - '0');
	}
	if (negflag) d = -d;
	
	return d;
}

# endif /* OWN_LIB */
