// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/slog.h>
#include <cc++/process.h>
#include "server.h"

using namespace server;
using namespace ost;
using namespace std;

#ifdef	WIN32

static bool test = true;

static BOOL WINAPI stop(DWORD code)
{
        if(code == CTRL_LOGOFF_EVENT)
                return TRUE;

	BayonneService::stop();
	BayonneDriver::stop();

	if(code)
		errlog("failed", "server exiting; reason=%d", code);
	else
		errlog("notice", "server exiting; normal termination");

	exit(code);
	return TRUE;
}

#else
static int mainpid;

static RETSIGTYPE stop(int signo)
{
	if(signo > 0 && getpid() != mainpid)
	{
		kill(mainpid, signo);
		return;
	}

        signal(SIGINT, SIG_IGN);
        signal(SIGABRT, SIG_IGN);
        signal(SIGTERM, SIG_IGN);
        signal(SIGQUIT, SIG_IGN);

	BayonneService::stop();
	BayonneDriver::stop();

        if(signo == -2)
                errlog("failed", "server exiting; no timeslots allocated");
        else if(signo)
                errlog("failed", "server exiting; reason=%d", signo);
        else
                errlog("notice", "server exiting; normal termination");

#ifdef	HAVE_LIBEXEC
	Libexec::cleanup();
#endif

	Thread::sleep(100);

	::exit(signo);
}
#endif

static void version(void)
{
        cout << VERSION << endl;
        exit(0);
}

static void loadStack(const char *id)
{
	BayonneDriver *d;

	if(!loadModule(id, true))
	{
failed:
		cerr << "bayonne cannot load " << id << " stack" << endl;
		stop(-1);
	}
	d = BayonneDriver::get(id);
	if(!d)
		goto failed;

	d->start();
}	

static void loading(void)
{
	char ** keys;
	const char *cp;
	const char *dname = keyserver.getLast("driver");
	unsigned count = keyoptions.getCount("modules");
	BayonneDriver *driver;

#ifdef	HAVE_LIBEXEC
	Libexec::allocate();
#endif

	if(!loadModule(keyoptions.getLast("driver")))
		stop(-1);

	if(!count)
		goto start;

	keys = (char **)keyoptions.getList("modules");

	while(*keys)
		loadModule(*keys++);

start:
	driver = BayonneDriver::get(dname);
	if(!driver)
		stop(-1);

	driver->start();
	count = Bayonne::getTimeslotsUsed();
	if(!count)
		stop(-2);

	Bayonne::start_driver = driver;

	cp = keyoptions.getLast("sip.stack");
	if(cp && atoi(cp) > 0)
		loadStack("sip");

	cp = keyoptions.getLast("h323.stack");
	if(cp && atoi(cp) > 0)
		loadStack("h323");

	count = Bayonne::getTimeslotsUsed();

#ifdef	HAVE_LIBEXEC
	Libexec::startup();
#endif

	Bayonne::reload();
	BayonneService::start();
	errlog("notice", "%s driver started; %d timeslot(s) used", dname, count);
#ifndef	WIN32
        Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);
#endif
}

static void logging(void)
{
	slog.open("bayonne", Slog::classDaemon);
	const char *level = keyserver.getLast("logging");

	if(!stricmp(level, "notice"))
		slog.level(Slog::levelNotice);
	else if(!stricmp(level, "info"))
		slog.level(Slog::levelInfo);
	else if(!stricmp(level, "error"))
		slog.level(Slog::levelError);
	else if(!stricmp(level, "debug"))
		slog.level(Slog::levelDebug);
}

static void testing(char **argv)
{
	testConfig(*argv);		
	parseConfig(argv);
	logging();

	errlog("notice", "testing %s on %s %s; driver=%s, timeslots=%s",
		VERSION, 
		keyserver.getLast("cpu"), keyserver.getLast("platform"),
		keyserver.getLast("driver"), keyserver.getLast("timeslots"));

	loading();
	Runtime::process();
}

static void starting(char **argv, bool trace)
{
	loadConfig();
	if(trace)
		keyserver.setValue("logging", "debug");

	parseConfig(argv);
        Process::setScheduler(keyengine.getLast("scheduler"));
#ifdef	WIN32
//	Process::setUser(keyserver.getLast("user"));
#else
	Process::setPriority(atoi(keyengine.getLast("priority")));
	if(!getuid())
	{
		if(!Process::setUser(keyserver.getLast("user")))
		{
			errlog("fatal", "%s: cannot set user",
				keyserver.getLast("user"));
			stop(-1);
		}
	}
	if(!trace)
		Process::detach();

	mainpid = getpid();
#endif
	
	logging();
        errlog("notice", "starting %s on %s %s; driver=%s, timeslots=%s",           
		VERSION,
                keyserver.getLast("cpu"), keyserver.getLast("platform"),
		keyserver.getLast("driver"), keyserver.getLast("timeslots"));

	loading();
	Runtime::process();
}

static void banner(void)
{
	cout << "SERVER VERSION " << VERSION << "; ";
	cout << keyserver.getLast("node") << " ";
	cout << keyserver.getLast("platform") << " ";
	cout << keyserver.getLast("cpu") << " ";
	cout << endl;
}

int main(int argc, char **argv)
{
	char *cp;

        if(argc < 2)
        {
                cerr << "use: bayonne --option [args...]" << endl;
                exit(-1);
        }
        cp = argv[1];
        if(!strncmp(cp, "--", 2))
                ++cp;

#ifdef	WIN32
	SetConsoleTitle("Bayonne");
        SetConsoleCtrlHandler((PHANDLER_ROUTINE)stop, TRUE);
#else
	mainpid = getpid();
	Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);
#endif

	if(!stricmp(cp, "-version"))
		version();
	else if(!stricmp(cp, "-test"))
		testing(argv);
	else if(!stricmp(cp, "-trace"))
		starting(argv, true);
	else if(!stricmp(cp, "-start"))
		starting(argv, false);
	else if(!stricmp(cp, "-dump-install"))
	{
		loadConfig();
		banner();
		dumpConfig(keypaths);
		exit(0);
	}
	else if(!stricmp(cp, "-dump-testing"))
	{
		testConfig(*argv);
		banner();
		dumpConfig(keypaths);
		exit(0);
	}
#ifdef	WIN32
	else if(!stricmp(cp, "-dump-registry"))
	{
		// we assume these objects are registry initialized
		// and dump them without further processing.
		cout << "[paths]" << endl;
		dumpConfig(keypaths);
		cout << "[server]" << endl;
		dumpConfig(keyserver);
                cout << "[engine]" << endl;
                dumpConfig(keyengine);
		exit(0);
	}
#endif

	cerr << "bayonne: " << *(argv + 1) << ": unknown option" << endl;
	exit(-1);
	return -1;
}
