/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <cstdio>
#include <stdlib.h>
#include <atsc_sssr.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>


static void
usage (const char *name)
{
  fprintf (stderr, "usage: %s <nodc.rxout> <symbols.out>\n", name);
  exit (1);
}

static off_t
filesize (int fd)
{
  struct stat	statbuf;

  if (fstat (fd, &statbuf) != 0){
    perror ("fstat");
    exit (1);
  }
  return statbuf.st_size;
}

int main (int argc, char **argv)
{
  const char 	*in_filename;
  const char	*out_filename;
  FILE		*in_file;
  FILE		*out_file;
  bool		seg_locked;

  float		sample_rate = 2 * ATSC_SYMBOL_RATE;

  if (argc != 3)
    usage (argv[0]);

  in_filename = argv[1];
  out_filename = argv[2];

  if ((in_file = fopen (in_filename, "rb")) == 0){
    perror (in_filename);
    exit (1);
  }

  if ((out_file = fopen (out_filename, "wb")) == 0){
    perror (out_filename);
    exit (1);
  }

  off_t bytelen = filesize (fileno (in_file));
  off_t len = bytelen / sizeof (float);		// length in floats
  
  void *p = mmap (0, bytelen, PROT_READ, MAP_SHARED, fileno (in_file), 0);
  if (p == MAP_FAILED){
    perror ("mmap");
    exit (1);
  }

  float *samples = (float *) p;
  int	 si = 500000;		// skip transients while FPLL settles
  
  atsc_sssr		sssr;
  atsc_interpolator	interp (sample_rate / ATSC_SYMBOL_RATE);
  
  float  interp_sample;
  int	 symbol_index;
  double timing_adjustment = 0;

  while (interp.update (samples, len, &si, timing_adjustment, &interp_sample)){
    sssr.update (interp_sample, &seg_locked, &symbol_index, &timing_adjustment);

    fprintf (out_file, "%d %3d %10g\n", seg_locked, symbol_index, interp_sample);
  }

  exit (0);
}
