/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <GrFIRfilterFFF.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>
#include <VrInterpolatingSigProc.h>
#include <gr_firdes.h>
#include <gr_fir_builder.h>
#include <VrNullSink.h>
#include <VrFileSource.h>
#include <VrFileSink.h>
#include <GrReal.h>
#include <getopt.h>
#include <VrSigSource.h>
#include <atsc_root_raised_cosine_bandpass.h>
#include <atsc_types.h>
#include <atsc_consts.h>
#include <GrAtscSymbolMapper.h>
#include <GrWeaverModHead.h>
#include <GrWeaverModTail.h>


double output_rate = 2 * ATSC_SYMBOL_RATE;
static const double IF_freq = 5.75e6;		// output freq of tuner module

//static const int NMFTAPS = 93;			// # of matched filter taps
static const int NMFTAPS = 139;			// # of matched filter taps

// freq of hdtv suppressed carrier pilot tone.  
//
// This assumes that the tuner is actually tuned to the middle of the 6
// MHz channel.  The pilot is 0.31 MHz up from the bottom edge of the channel

static const double pilot_freq = IF_freq - 3e6 + 0.31e6;


static const double MIXER_GAIN = 128.0;	// linear value
//static const double MIXER_GAIN = 1.0;		// linear value


static void 
usage (const char *name)
{
  cerr << "usage: " << name << " [-g] [-r] [-o <filename>] -f <filename>\n";
  cerr << "  -g : no gui\n";
  cerr << "  -r : continuously repeat the file contents (loop)\n";
  cerr << "  -f <filename> : take input from filename\n";
  cerr << "  -o <filename> : put output in filename, else VrNullSink\n";

  exit (1);
}

int main (int argc, char **argv)
{
  try {
    
    bool use_gui_p = true;
    bool repeat_p = false;		// continuous loop through file
    char *input_filename = 0;
    char *output_filename = 0;

    int c;
    while ((c = getopt (argc, argv, "grf:o:")) != EOF){
      switch (c){
      case 'g':	use_gui_p = false;    		break;
      case 'r': repeat_p = true;		break;
      case 'f': input_filename = optarg;	break;
      case 'o': output_filename = optarg;	break;
      default:
	usage (argv[0]);
      }
    }

    if (input_filename == 0)
      usage (argv[0]);
    
    if (optind != argc)
      usage (argv[0]);

    VrGUI *guimain = 0;
    VrGUILayout *horiz = 0;
    VrGUILayout *vert = 0;
    VrGUILayout *vert2 = 0;

    if (use_gui_p){
      guimain = new VrGUI(argc, argv);
      horiz = guimain->top->horizontal();
      vert = horiz->vertical();
      vert2 = horiz->horizontal()->vertical();
    }

    cerr << "Output Sampling Rate: " << output_rate << endl;
    cerr << "Pilot Freq: " << pilot_freq << endl;

    VrSource<atsc_data_segment>		*source = 0;
    GrAtscSymbolMapper<float>		*mapper = 0;
    GrWeaverModHead<float,float>	*weaver_head = 0;
    GrWeaverModTail<float,float>	*weaver_tail = 0;
    GrFIRfilterFFF			*mfI = 0;
    GrFIRfilterFFF			*mfQ = 0;
    VrSink<float>			*final_sink = 0;

    VrSink<float>	*final_fft = 0;
    VrSink<float>	*mfI_fft = 0;
    VrSink<float>	*mfQ_fft = 0;
    VrSink<float>	*mapper_log = 0;
    
    bool logging_p = output_filename != 0;
    

    // ================================================================
    //   Get segment data from external file
    // ================================================================

    // --> atsc_data_segment
    source = new VrFileSource<atsc_data_segment>(ATSC_DATA_SEGMENT_RATE,
						 input_filename,
						 repeat_p);

    // ================================================================
    //   Pass it through the symbol mapper
    //   (includes symbol mapping & pilot addition)
    // ================================================================

    mapper = new GrAtscSymbolMapper<float>();

    if (logging_p)
      mapper_log = new VrFileSink<float>("mapper.out");

    
    // ================================================================
    // 			Build Weaver Modulator
    // ================================================================

    // mix with cos/sin Fsym/4 and interpolate by 2
    weaver_head = new GrWeaverModHead<float,float>(2);

    // gain of 2 to make up for the gain of 1/2 of the interpolate by 2
    mfI = new GrFIRfilterFFF (1, gr_firdes::root_raised_cosine (2.0,
								output_rate,
								ATSC_SYMBOL_RATE/2,
								.115, NMFTAPS));

    mfQ = new GrFIRfilterFFF (1, gr_firdes::root_raised_cosine (2.0,
								output_rate,
								ATSC_SYMBOL_RATE/2,
								.115, NMFTAPS));

    if (use_gui_p){
      mfI_fft = new GrFFTSink<float>(vert, -20, 50, 512);
      mfQ_fft = new GrFFTSink<float>(vert, -20, 50, 512);
    }

    // mix with cos/sin IF_freq, sum components
    weaver_tail = new GrWeaverModTail<float,float>(IF_freq, MIXER_GAIN);


    // ================================================================
    // final sink is either a file sink or a null sink
    // ================================================================

    if (output_filename)
      final_sink = new VrFileSink<float>(output_filename);

    else
      final_sink = new VrNullSink<float>();

    if (use_gui_p){
      final_fft = new GrFFTSink<float>(vert2, -20, 50, 512);
    }

    // connect the modules together

    VrMultiTask *m = new VrMultiTask ();
    m->add (final_sink);

    NWO_CONNECT  (source, mapper);
    NWO_CONNECT  (mapper, weaver_head);
    NWO_CONNECTN (weaver_head, 0, mfI);
    NWO_CONNECTN (weaver_head, 1, mfQ);
    NWO_CONNECT  (mfI, weaver_tail);
    NWO_CONNECT  (mfQ, weaver_tail);
    NWO_CONNECT  (weaver_tail, final_sink);
    
    if (logging_p){
      NWO_CONNECT (mapper, mapper_log);
      m->add (mapper_log);
    }

    if (use_gui_p){
      NWO_CONNECT (weaver_tail, final_fft);
      NWO_CONNECT (mfI, mfI_fft);
      NWO_CONNECT (mfQ, mfQ_fft);
      m->add (final_fft);
      m->add (mfI_fft);
      m->add (mfQ_fft);
    }


    m->start ();
    if (use_gui_p)
      guimain->start ();

    while (1){
      if (use_gui_p)
	guimain->processEvents(10 /*ms*/);
      m->process();
    }	

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
