/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 *
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 *
*/

#ifndef _GRAUDIOSINK_H_
#define _GRAUDIOSINK_H_

#define AUDIOOUTCHUNKSIZE 64

extern "C" {
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <assert.h>
}
#include <VrSink.h>
#include <fstream>
#include <string>

template<class iType>
class GrAudioSink : public VrSink<iType> {
protected:
  int 	      audiofd;
  short       *buffer;
  std::string device_name;
  iType	      scale_factor;	
public:
  virtual const char *name() { return "GrAudioSink"; }

  virtual int work3(VrSampleRange output,
		    VrSampleRange inputs[], void *i[]);

  virtual void initialize();

  GrAudioSink(iType input_range = 32767, const char* dev = "/dev/dsp")
    : audiofd(-1), device_name(dev) {
    setOutputSize (AUDIOOUTCHUNKSIZE);
    scale_factor = 32767 / input_range;
  }

  virtual ~GrAudioSink() {
    delete [] buffer;
    close(audiofd);
  }
};


template<class iType> void
GrAudioSink<iType>::initialize()
{
  if(audiofd==-1) {
    int	temp = 0x7fff0004;

    if ((audiofd = open(device_name.c_str(), O_WRONLY/*|O_NONBLOCK*/)) < 0) {
      cerr << "GrAudioSink: ";
      perror (device_name.c_str ());
      exit(1);
    }
    if ((ioctl(audiofd,SNDCTL_DSP_SETFRAGMENT,&temp)) < 0) {
      fprintf(stderr, "GrAudioSink: set fragment returned %d\n", errno);
      exit(1);
    }
  }

  // -eb docs say not to do this...
  // ioctl(audiofd,SNDCTL_DSP_RESET);

  int format = AFMT_S16_NE;  // FIXME check endianness
  int origf=format;
  if((ioctl(audiofd,SNDCTL_DSP_SETFMT,&format)) < 0) {
    cerr << "GrAudioSink: " << device_name << " IOCTL failed with " << errno << "\n";
    exit(1);
  }

  if(origf!=format) {
    fprintf(stderr, "GrAudioSink: Warning: unable to support format %d\n", origf);
    fprintf(stderr, "  card requested %d instead.\n", format);
  }

  // set to stereo no matter what.  Some hardware only does stereo
  int channels = 2;
  if (ioctl (audiofd, SNDCTL_DSP_CHANNELS, &channels) < 0){
    perror ("GrAudioSink: SNDCTL_DSP_CHANNELS failed");
    exit (1);
  }
    
  if (channels != 2) {
    fprintf(stderr, "GrAudioSink: could not set STEREO mode\n");
    exit(1);
  }

  buffer = new short[getOutputSize() * 2];   

  int n = getNumberInputs ();
  assert (n >= 1 && n <= 2);

  if(n==2) 
    if(getInputSamplingFrequencyN(0)!=getInputSamplingFrequencyN(1))
      fprintf(stderr, "GrAudioSink Warning: left and right at different freq\n");
  
  int sf = (int) getSamplingFrequency();
  printf("GrAudioSink: sampling frequency is %d\n",sf);

  if ((ioctl(audiofd, SNDCTL_DSP_SPEED, &sf)) < 0) {
    cerr << device_name << ": invalid sampling frequency...defaulting to 8 Khz\n";
    sf = 8000;
    if ((ioctl(audiofd,SNDCTL_DSP_SPEED, &sf)) < 0) {
      fprintf(stderr, "Couldn't even manage that...aborting\n");
      exit(1);
    }
  }
  if (sf != getSamplingFrequency ())
    fprintf (stderr, "GrAudioSink Warning: sound card default to %d Hz\n", sf);
}

template<class iType> int
GrAudioSink<iType>::work3(VrSampleRange output, VrSampleRange inputs[], void *ai[])
{
  iType **i = (iType **)ai;
  unsigned size = output.size;
  
  // size is guaranteed to be a multple of getOutputSize
  assert ((size % getOutputSize()) == 0);

  while(size > 0) {
    if(getNumberInputs()==1) {
      for(unsigned int k=0;k<getOutputSize();k++) {
	buffer[2*k] = (short)(scale_factor * i[0][k]);
	buffer[2*k+1] = (short)(scale_factor * i[0][k]);
      }
    }
    else { 
      for(unsigned int k=0;k<getOutputSize();k++) {
	buffer[2*k] = (short)(scale_factor * i[0][k]);
	buffer[2*k+1] = (short)(scale_factor * i[1][k]);
      }
    }			     
	  
    unsigned int count =  write(audiofd,buffer,4*getOutputSize());
    // 2 for stereo times 2 bytes in a short
    if(count<0) {
      printf("AudioSink write error, errno: %d\n", errno);
      exit(1);
    } 
    else {
      if(count!=4*getOutputSize())
      	printf("AudioSink: warning: not all bytes written!\n");
    }
    size-=getOutputSize();
    i[0]+=getOutputSize();
    if(getNumberInputs()==2)
      i[1]+=getOutputSize();
  }
  return output.size;
}

template<> int
GrAudioSink<VrComplex>::work3(VrSampleRange output, VrSampleRange inputs[], void *ai[])
{
  VrComplex **i = (VrComplex **)ai;
  unsigned size = output.size;
  
  // size is guaranteed to be a multple of getOutputSize
  assert ((size % getOutputSize()) == 0);

  if(getNumberInputs()!=1)
    {
      cerr << "GrAudioSink:  Can only output one VrComplex stream\n";
      exit(-1);      
    }

  while(size > 0) {
    for(unsigned int k=0;k<getOutputSize();k++) {
      buffer[2*k] = (short)real(scale_factor * i[0][k]);
      buffer[2*k+1] = (short)imag(scale_factor * i[0][k]);
    }
  	  
    unsigned int count =  write(audiofd,buffer,4*getOutputSize());
    // 2 for stereo times 2 bytes in a short
    if(count<0) {
      printf("AudioSink write error, errno: %d\n", errno);
      exit(1);
    } 
    else {
      if(count!=4*getOutputSize())
	printf("AudioSink: warning: not all bytes written!\n");
    }
    size-=getOutputSize();
    i[0]+=getOutputSize();
    if(getNumberInputs()==2)
      i[1]+=getOutputSize();
  }
  return output.size;
}

#endif
