/* -*- Mode: c++ -*- */

/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _VRCHIRPSOURCE_H_
#define _VRCHIRPSOURCE_H_

extern "C" {
#include <math.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
}

#include <VrSource.h>


template<class oType> 
class VrChirpSource : public VrSource<oType> {

public:
  virtual const char *name () { return "VrChirpSource"; }

  virtual int work2(VrSampleRange output, void *o[]);

  VrChirpSource (double sample_freq,
		 double amplitude,
		 double chirp_sweep_freq);

  virtual void initialize ();
  

protected:
  double	sampling_freq;		// Hz
  double	amplitude;
  double	phase;

  double	chirp_sweep_freq;
  double	chirp_incr;
  double	chirp_min_freq;
  double	chirp_max_freq;
  double	chirp_current_freq;
};

template<class oType> 
VrChirpSource<oType>::VrChirpSource (double a_sampling_freq,
				     double a_amplitude,
				     double a_chirp_sweep_freq)
{
  sampling_freq = a_sampling_freq;
  setSamplingFrequency (sampling_freq);
  amplitude = a_amplitude;
  chirp_sweep_freq = a_chirp_sweep_freq;
  phase = 0.0;
}

template<class oType> void
VrChirpSource<oType>::initialize()
{
  chirp_min_freq = 0;
  chirp_max_freq = sampling_freq / 2;
  chirp_current_freq = chirp_min_freq;

  chirp_incr = (((chirp_max_freq - chirp_min_freq) * chirp_sweep_freq / 2)
		/ sampling_freq);
#if 1
  printf ("chirp_min_freq     = %g\n", chirp_min_freq);
  printf ("chirp_max_freq     = %g\n", chirp_max_freq);
  printf ("chirp_current_freq = %g\n", chirp_current_freq);
  printf ("chirp_incr         = %g\n", chirp_incr);
#endif
}

template<class oType> int
VrChirpSource<oType>::work2(VrSampleRange output, void *ao[])
{ 
  oType **ov = (oType **)ao;
  oType *o = ov[0];
  unsigned int size = output.size;

  sync (output.index);

  double	two_pi_over_sf = 2 * M_PI / sampling_freq;


  while (size-- > 0) {
    *o++ = (oType) (amplitude * sin (phase));
    phase += two_pi_over_sf * chirp_current_freq;

    chirp_current_freq += chirp_incr;
    if (chirp_current_freq >= chirp_max_freq)
      chirp_current_freq = chirp_min_freq;
  }

  return output.size;
}

#endif 
