/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GR_OSCOPEGUTS_H_
#define _GR_OSCOPEGUTS_H_

#include <gr_TriggerMode.h>

/*!
 * \brief guts of oscilloscope trigger and buffer module
 *
 * This module processes sets of samples provided the \p processSample
 * method.  When appropriate given the updateRate, sampleRate and
 * trigger conditions, processSample will periodically write output
 * records of captured data to output_fd.  For each trigger event,
 * nchannels records will be written.  Each record consists of
 * getSamplesPerOutputRecord binary floats.  The trigger instant
 * occurs at the 1/2 way point in the buffer.  Thus, output records
 * consist of 50% pre-trigger data and 50% post-trigger data.
 */

class gr_OscopeGuts {
private:
  static const int	MAX_CHANNELS = 16;
  enum ScopeState 	{ HOLD_OFF, LOOK_FOR_TRIGGER, POST_TRIGGER };

  int			d_nchannels;		// how many channels
  int			d_output_fd;		// file desc used to write output records
  gr_TriggerMode	d_trigger_mode;		
  int			d_trigger_channel;	// which channel to watch for trigger condition
  double		d_sample_rate;		// input sample rate in Hz
  double		d_update_rate;		// approx freq to produce an output record (Hz)
  double		d_trigger_level;

  int			d_obi;			// output buffer index 
  float		       *d_buffer[MAX_CHANNELS];

  ScopeState		d_state;
  int			d_decimator_count;
  int			d_decimator_count_init;
  int			d_hold_off_count;
  int			d_hold_off_count_init;
  int			d_post_trigger_count;
  int			d_post_trigger_count_init;
  float			d_prev_sample;			// used for trigger checking

  // NOT IMPLEMENTED
  gr_OscopeGuts (const gr_OscopeGuts &rhs);		// no copy constructor
  gr_OscopeGuts &operator= (const gr_OscopeGuts &rhs);	// no assignment operator

  void triggerChanged ();
  void updateRateOrDecimationChanged ();
  int  foundTrigger (float sample);	// returns -1, 0, +1
  void writeOutputRecords ();

  void enterHoldOff ();			// called on state entry
  void enterLookForTrigger ();
  void enterPostTrigger ();

public:
  // CREATORS
  gr_OscopeGuts (int nchannels, double sample_rate, int output_fd);
  ~gr_OscopeGuts ();

  // MANIPULATORS

  /*!
   * \p channel_data points to nchannels float values.  These are the values
   * for each channel at this sample time.
   */
  void processSample (const float *channel_data);

  bool setUpdateRate (double update_rate);
  bool setDecimationCount (int decimation_count);
  bool setTriggerChannel (int channel);
  bool setTriggerMode (gr_TriggerMode mode);
  bool setTriggerLevel (double trigger_level);
  bool setTriggerLevelAuto ();				// set to 50% level


  // ACCESSORS
  int getNumChannels () const;
  double getSamplingRate () const;
  double getUpdateRate () const;
  int getDecimationCount () const;
  int getTriggerChannel () const;
  gr_TriggerMode getTriggerMode () const;
  double getTriggerLevel () const;

  // # of samples written to each output record.
  int getSamplesPerOutputRecord () const;
};

#endif /* _GR_OSCOPEGUTS_H_ */
