#!/usr/bin/env python
#
# Copyright 2003 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from GnuRadio import *

#
#  FM radio guts
#
class fm_radio_guts (object):
    def __init__ (self):
        pass
    
    def set_volume (self, volume_index):
        # range check volume_index
        self.volume_index = volume_index

    def get_volume (self):
        return self.volume_index

    def incr_volume (self, incr):
        self.set_volume (self.get_volume () + incr)

    def set_channel (self, channel_number):
        # range check channel
        self.channel_number = channel_number

    def get_channel (self):
        return self.channel_number

    def incr_channel (self, incr):
        self.set_channel (self, self.get_channel () + incr)



#
# return a gr_FlowGraph
#
def build_graph ():
    input_rate = 20e6
    IF_freq = 5.75e6

    chan_taps = 75
    CFIR_decimate = 125
    fm_demod_gain = 2200

    RFIR_decimate = 5

    quad_rate = input_rate / CFIR_decimate
    audio_rate = quad_rate / RFIR_decimate

    volume = 1.0
    
    src = make_GrMC4020SourceS (input_rate,
                                MCC_CH3_EN | MCC_ALL_5V)

    offset_fixer = VrFixOffsetSS ()

    channel_coeffs = gr_firdes.low_pass (1.0,          # gain
                                         input_rate,   # sampling rate
                                         250e3,        # low pass cutoff freq
                                         8*100e3,      # width of trans. band
                                         gr_firdes.WIN_HAMMING)


    chan_filter = GrFreqXlatingFIRfilterSCF (CFIR_decimate,
                                             channel_coeffs, IF_freq)

    fm_demod = VrQuadratureDemodCF (volume * fm_demod_gain)

    width_of_transition_band = audio_rate / 32
    audio_coeffs = gr_firdes.low_pass (1.0,            # gain
                                       quad_rate,      # sampling rate
                                       audio_rate/2 - width_of_transition_band,
                                       width_of_transition_band,
                                       gr_firdes.WIN_HAMMING)

    audio_filter = GrFIRfilterFSF (RFIR_decimate, audio_coeffs)

    final_sink = GrAudioSinkS (32767, "/dev/dsp")
    

    fg = gr_FlowGraph ()

    fg.connect (src, offset_fixer)
    fg.connect (offset_fixer, chan_filter)
    fg.connect (chan_filter, fm_demod)
    fg.connect (fm_demod, audio_filter)
    fg.connect (audio_filter, final_sink)

    return fg


if __name__ == '__main__':
    fg = build_graph ()
    fg.start ()                         # fork thread(s) and return immediately
    # your GUI mail loop goes here...
    fg.wait ();                         # wait (forever)
