/*
 * 
 * Copyright 1996 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 */
#ifndef _LINUX_GUPPI_H
#define _LINUX_GUPPI_H

#include <linux/ioctl.h>
#include <linux/pci.h>

#ifdef __KERNEL__
#include <asm/spinlock.h>
#endif

#ifndef   PCI_VENDOR_ID_TNS  
#define   PCI_VENDOR_ID_TNS     0x6969
#endif

#ifndef   PCI_DEVICE_ID_GUPPI
#define   PCI_DEVICE_ID_GUPPI   0x9696 
#endif

#define   NR_CARDS              1

/*
 *  Thingys for extracting fields from registers
 */

/* Control Register Masks */

#define GUP_TXDATA_FIFO_RS     0x00000001
#define GUP_RXDATA_FIFO_RS     0x00000002
#define GUP_TXPG_FIFO_RS       0x00000004
#define GUP_RXPG_FIFO_RS       0x00000008
#define GUP_RXREG_RS           0x00000010
#define GUP_TXREG_RS           0x00000020
#define GUP_BCK_CTRL           0x000003c0
#define GUP_BCK_RS             0x00000400
#define GUP_PROG_FLAGS         0x00000800
#define GUP_USE_AEFLAG         0x00001000
#define GUP_BCK_EN             0x00002000
#define GUP_TX_EN              0x00004000
#define GUP_RX_EN              0x00008000
#define GUP_FAVOR_TX           0x00010000
#define GUP_FAVOR_RX           0x00020000
#define GUP_TXDONEINT_EN       0x00040000
#define GUP_RXPGAEINT_EN       0x00080000
#define GUP_TXPGAEINT_EN       0x00100000
#define GUP_BCKSTATINT_EN      0x00200000

#define GUP_TXDONE_INT_RS      0x00400000
#define GUP_RXPGAE_INT_RS      0x00800000
#define GUP_TXPGAE_INT_RS      0x01000000
#define GUP_BCKSTAT_INT_RS     0x02000000
#define GUP_BCSTATINT_REG      0x3c000000

#define GUP_RX_RS              0x0080001a
#define GUP_TX_RS              0x01400025
#define GUP_ALL_RS             0x03c0003f
#define GUP_INT_MASK           0x03c00000

/* Status Register Masks */
// AE = almost-empty flag (can generate interrupt)
// AF = almost-full flag (can stall Xlinix)

#define GUP_BCKID              0x0000000f
#define GUP_BCKSTAT            0x000000f0
#define GUP_TXDAT_FLAGS        0x00001f00
  #define GUP_TXDAT_EF         0x00000100 
  #define GUP_TXDAT_AE         0x00000200 
  #define GUP_TXDAT_HF         0x00000400 
  #define GUP_TXDAT_AF         0x00000800 
  #define GUP_TXDAT_FF         0x00001000  

#define GUP_RXDAT_FLAGS        0x0003e000
  #define GUP_RXDAT_EF         0x00002000 
  #define GUP_RXDAT_AE         0x00004000 
  #define GUP_RXDAT_HF         0x00008000 
  #define GUP_RXDAT_AF         0x00010000 
  #define GUP_RXDAT_FF         0x00020000  

#define GUP_TXPG_FLAGS         0x007c0000
  #define GUP_TXPG_EF          0x00040000 
  #define GUP_TXPG_AE          0x00080000 
  #define GUP_TXPG_HF          0x00100000 
  #define GUP_TXPG_AF          0x00200000 
  #define GUP_TXPG_FF          0x00400000  

#define GUP_RXPG_FLAGS         0x0f800000
  #define GUP_RXPG_EF          0x00800000 
  #define GUP_RXPG_AE          0x01000000 
  #define GUP_RXPG_HF          0x02000000 
  #define GUP_RXPG_AF          0x04000000 
  #define GUP_RXPG_FF          0x08000000  

#define GUP_TX_PENDING         0x10000000
#define GUP_RX_PENDING         0x20000000
#define GUP_LAST_TRANS         0x40000000

/*
 * Offsets from base address to various CSR registers 
 */

#define GUP_CONTROL_OFF        0x00000000
#define GUP_STATUS_OFF         0x00000004
#define GUP_TXPGADDR_OFF       0x00000008
#define GUP_RXPGADDR_OFF       0x0000000c

/*
 * Offsets from base address into FIFO spaces
 */

#define GUP_DATA_FIFO_OFF      0x00000400
#define GUP_TXPG_FIFO_OFF      0x00000800
#define GUP_RXPG_FIFO_OFF      0x00000c00

/*
 * Guppi structures (I/O buffers, control registers, and general board info)
 */

struct guppi_status {
  unsigned int index; //index of the first valid page (this page is valid if num>0)
  unsigned int num; //number of valid pages at index

  //number of pages that had to be thrown away becuase
  //the buffer was full since the last status check
  //if this is >0 then num should ==
  //the number of pages in the entire buffer
  unsigned int lost; 
};

#ifdef __KERNEL__

struct guppi_buf {
  char                  *buf; //pointer to the data
  unsigned long         *busaddrs; //list of physical addresses
};

struct   guppi_reg {
  volatile unsigned int gup_ctrl;      /* GuPPI Control Register */
  volatile unsigned int gup_stat;      /* GuPPI Status Register  */
  volatile unsigned int gup_txpgreg;   /* GuPPI Transmit Page Register */
  volatile unsigned int gup_rxpgreg;   /* GuPPI Receive Page Register */
};

struct   guppi_board {
  unsigned long        base_addr;
  unsigned long        phys_addr;
  int                  irq;
  //unsigned char        gup_bus;
  struct pci_dev *dev;
  unsigned int         minor;
  unsigned int         users;
  spinlock_t           lock;

  /* stuff for receive starts here */
  unsigned int rxbufsize; //in pages
  unsigned char        is_rxing;
  unsigned int         rxoverruns;
  struct guppi_buf *rx_buffer;

  //indexs are all into the rx_buffer->physaddrs[] array
  unsigned int rx_empty_index; //start of empty pages
  unsigned int rx_user_index; //start of user's pages
  //number of empty pages = (rx_user_index - rx_empty_index) % rxbufsize
  //unless rx_user_index == rx_empty_index in which case
  //       0 if rx_user_num > 0;
  //       rxbufsize if rx_user_num = 0;
 
  //number of pages user knows about and is working with
  unsigned int rx_user_num;

  struct guppi_buf *rxjunkpage;


  /* stuff for transmit starts here */
  unsigned int txbufsize; //in pages
  unsigned char        is_txing;
  unsigned int         txoverruns;
  struct guppi_buf *tx_buffer;

  //indexs are all into the tx_buffer->physaddrs[] array
  unsigned int tx_empty_index; //start of empty pages
  unsigned int tx_user_index; //start of user's pages
  //number of empty pages = (tx_user_index - tx_empty_index) % txbufsize
  //unless tx_user_index == tx_empty_index in which case
  //       0 if tx_user_num > 0;
  //       txbufsize if tx_user_num = 0;
 
  //number of pages user knows about and is working with
  unsigned int tx_user_num;

  struct guppi_buf *txjunkpage;

  struct guppi_board   *next;
};

/*
 * GuPPI major device number 
 */

#define GUPPI_MAJOR  127

/*
 * Size of Transmit Data FIFO (in Dwords)
 */

#define TXFIFOSIZE   1024

/*
 * Size of Receive Data FIFO (in Dwords)
 */

#define RXFIFOSIZE   1024

//low water mark (almost empty) on receive fifo(generate interrupt)
//FIX: what should this be
#define RXAEPAGES    400 

//high water mark (almost full) on receive fifo
#define RXAFPAGES    (RXFIFOSIZE-2)

//how close to get to the high water mark
//(if the Guppi gets within 2 (?) of AF, the flag gets asserted...
//FIX what should this be
#define RXSAFETY 20

//low water mark (almost empty) on receive fifo(generate interrupt)
//FIX: what should this be
#define TXAEPAGES    400 

//high water mark (almost full) on receive fifo
#define TXAFPAGES    (TXFIFOSIZE-2)

//how close to get to the high water mark
//(if the Guppi gets within 2 (?) of AF, the flag gets asserted...
//FIX what should this be
#define TXSAFETY 20

#endif

/*
 * Guppi IOCTL commands
 *   note a file descriptor must be for reading OR writing, not both
 *   so we can use the same codes for start/stop, etc.
 */

/* Read a Dword from the 21050 configuration space */
#define GIOCBRDRD   _IOR( 'G', 0x00, unsigned long )

/* Write a Dword from the 21050 configuration space */
#define GIOCBRDWR   _IOW( 'G', 0x01, unsigned long )

/* Read a Dword from the GuPPI configuration space */
#define GIOCGUPRD   _IOR( 'G', 0x02, unsigned long )

/* Write a Dword from the GuPPI configuration space */
#define GIOCGUPWR   _IOW( 'G', 0x03, unsigned long )

/* Read a Dword from the Host Bridge configuration space */
#define GIOCTSCRD   _IOR( 'G', 0x04, unsigned long )

/* Write a Dword from the Host Bridge configuration space */
#define GIOCTSCWR   _IOW( 'G', 0x05, unsigned long )

/* Set standard buffer size to some number of pages */
#define GIOCSETBUFSIZE  _IOW( 'G', 0x06, unsigned int )

/* Start up the DMA/interrupt driver */
#define GIOCSTART _IO( 'G', 0x07 )

/* Stop the DMA/interrupt driver */
#define GIOCSTOP  _IO( 'G', 0x08 )

/* Tell the driver the user is done with a range of pages and get the */
/*  range of pages the user can use as well as the number of overruns */
#define GIOCSETGETSTATUS _IOWR( 'G', 0x09, struct guppi_status )

/* guppi_init procedure to load driver as part of built-in kernel */
extern int guppi_init(void);

#endif /* _LINUX_GUPPI_H */
