/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 *
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 *
*/

#ifndef _VRAUDIOSINK_H_
#define _VRAUDIOSINK_H_

#define AUDIOOUTCHUNKSIZE 64

extern "C" {
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <assert.h>
}
#include <VrSink.h>
#include <fstream>
#include <string>

template<class iType>
class VrAudioSink : public VrSink<iType> {
protected:
  int 	      audiofd;
  iType       *mixArray;
  bool	      convert_mono_to_stereo_p;
  std::string device_name;
  
public:
  virtual const char *name() { return "VrAudioSink"; }

  virtual int work3(VrSampleRange output,
		    VrSampleRange inputs[], void *i[]);

  virtual void initialize();

  VrAudioSink(const char* dev = "/dev/dsp")
    : audiofd(-1), convert_mono_to_stereo_p(false), device_name(dev) {
    setOutputSize (AUDIOOUTCHUNKSIZE);
  }

  virtual ~VrAudioSink() {
    delete [] mixArray;
    close(audiofd);
  }
};


template<class iType> void
VrAudioSink<iType>::initialize()
{
  long format;

  if(audiofd==-1) {
    int	temp = 0x7fff0004;

    if ((audiofd = open(device_name.c_str(), O_WRONLY/*|O_NONBLOCK*/)) < 0) {
      cerr << "VrAudioSink: ";
      perror (device_name.c_str ());
      exit(1);
    }
    if ((ioctl(audiofd,SNDCTL_DSP_SETFRAGMENT,&temp)) < 0) {
      fprintf(stderr, "VrAudioSink: set fragment returned %d\n", errno);
      exit(1);
    }
  }

  if(sizeof(iType)==1)
    format = AFMT_MU_LAW;
  if(sizeof(iType)==2)
    format = AFMT_S16_NE;
  else {
    fprintf(stderr, "VrAudioSink: cannot handle data type of size %d\n", sizeof (iType));
    exit(1);
  }

  // -eb docs say not to do this...
  // ioctl(audiofd,SNDCTL_DSP_RESET);

  int origf=format;
  if((ioctl(audiofd,SNDCTL_DSP_SETFMT,&format)) < 0) {
    cerr << "VrAudioSink: " << device_name << " IOCTL failed with " << errno << "\n";
    exit(1);
  }

  if(origf!=format) {
    fprintf(stderr, "VrAudioSink: Warning: unable to support format %d\n", origf);
    fprintf(stderr, "  card requested %ld instead.\n", format);
  }


  int n = getNumberInputs ();
  assert (n >= 1 && n <= 2);

  // set number of channels no matter what.  Some hardware only does stereo

  int channels = n;
  if (ioctl (audiofd, SNDCTL_DSP_CHANNELS, &channels) < 0){
    perror ("VrAudioSink: SNDCTL_DSP_CHANNELS failed");
    exit (1);
  }
    
  if (channels != n){
    if (n == 2){
      fprintf(stderr, "VrAudioSink: could not set STEREO mode\n");
      exit(1);
    }
    else {
      // user asked for MONO, but got STEREO.  We can work around this...
      convert_mono_to_stereo_p = true;
    }
  }

  if (convert_mono_to_stereo_p)
    mixArray=new iType[getOutputSize()*2];
  else if(getNumberInputs()==2) {		//Stereo
    mixArray=new iType[getOutputSize()*2];
    if(getInputSamplingFrequencyN(0)!=getInputSamplingFrequencyN(1))
      fprintf(stderr, "VrAudioSink Warning: left and right at different freq\n");
  } else
    mixArray = NULL;


  int sf = (int) getSamplingFrequency();
  printf("VrAudioSink: sampling frequency is %d\n",sf);

  if ((ioctl(audiofd, SNDCTL_DSP_SPEED, &sf)) < 0) {
    cerr << device_name << ": invalid sampling frequency...defaulting to 8 Khz\n";
    sf = 8000;
    if ((ioctl(audiofd,SNDCTL_DSP_SPEED, &sf)) < 0) {
      fprintf(stderr, "Couldn't even manage that...aborting\n");
      exit(1);
    }
  }
  if (sf != getSamplingFrequency ())
    fprintf (stderr, "VrAudioSink Warning: sound card default to %d Hz\n", sf);
}

template<class iType> int
VrAudioSink<iType>::work3(VrSampleRange output, VrSampleRange inputs[], void *ai[])
{
  iType **i = (iType **)ai;
  unsigned size = output.size;
  int bytes = getOutputSize()*sizeof(iType);
  iType *myPtr;

  if(getNumberInputs()==2 || convert_mono_to_stereo_p)		//Stereo
      bytes*=2;

  // size is guaranteed to be a multple of getOutputSize
  assert ((size % getOutputSize()) == 0);

  while(size > 0) {
    myPtr = i[0];

    if(getNumberInputs()==2) {					//Stereo
      for(unsigned int ii=0;ii<getOutputSize();ii++) {
	mixArray[ii*2]  =i[0][ii];
	mixArray[ii*2+1]=i[1][ii];
      }
      myPtr=mixArray;
      i[1]+=getOutputSize();	// advance channel 1 here, channel 0 at bottom
    }

    else if (convert_mono_to_stereo_p){
      for(unsigned int ii=0;ii<getOutputSize();ii++) {
	mixArray[ii*2]  =i[0][ii];
	mixArray[ii*2+1]=i[0][ii];
      }
      myPtr=mixArray;
    }
    
    int count =  write(audiofd,myPtr,bytes);
    if(count<0) {
      printf("AudioSink write error, errno: %d\n", errno);
      exit(1);
    } else {
      if(count!=bytes)
      	printf("AudioSink: warning: not all bytes written!\n");
    }
    size-=getOutputSize();
    i[0]+=getOutputSize();
  }
  return output.size;
}
#endif
