/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRFSKMOD_H_
#define _VRFSKMOD_H_

#include <VrInterpolatingSigProc.h>

/****
  FSK Modulator
  VrFSKMod(int f0,int f1,int bit_rate)
    f0,f1:  frequencies of bits 0 and 1, in Hz
    bit_rate:  data rate, in bits per sample
****/

template<class oType> 
class VrFSKMod : public VrInterpolatingSigProc<bit,oType> {
protected:
  int freq[2];
  int bitLength;
  int dataRate;
  oType* waveform[2];
  oType* zero;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrFSKMod(int f0,int f1,int bit_rate);
};

template<class oType> void
VrFSKMod<oType>::work(int n)
{

  for (int i=0; i < n/bitLength; i++,incReadPtr(1)) {
    bit inputBit = inputRead(0);
    if (inputBit == -1) { // NULL bit
      outputWrite(zero,bitLength);
    }
    else {
      outputWrite(&(waveform[inputBit][0]),bitLength);
    }
  }

  return;
}

template<class oType> void
VrFSKMod<oType>::initialize() {

  int a = (1 << (8*sizeof(oType) - 1)) - 1;
  int sf = getOutputSamplingFrequency();

  /* to ensure continuous phase (each bit begins and ends at 0 phase):
     effective data rate is adjusted sf/bitLength
     f0 and f1 adjusted to multiples of the effective data rate */

  bitLength = sf / dataRate;  
  setOutputSize (bitLength);

  if ((bitLength * dataRate) != sf) {
    dataRate = sf/bitLength;
    cout << "FSKMod: Adjusting data rate to: " << dataRate << endl;
  }
  
  int separation = freq[1] - freq[0];
  if ((separation % dataRate) != 0) {
    separation = (separation/dataRate)*dataRate + dataRate;
    cout << "FSKMod: Adjusting frequency separation to: "
	 << separation << endl;
  }

  if ((freq[0] % dataRate) != 0) {
    freq[0] = (freq[0]/dataRate)*dataRate + dataRate;
    freq[1] = freq[0] + separation;
    cout << "FSKMod: Adjusting frequencies to: "
	 << freq[0] << " and " << freq[1] << endl;
  }

  /* allocate and create waveforms */
  waveform[0] = (oType*)malloc(sizeof(oType)*bitLength);
  waveform[1] = (oType*)malloc(sizeof(oType)*bitLength);
  zero = (oType*)new oType[bitLength];
  
  for (int i=0; i<bitLength; i++) {
    waveform[0][i] = (oType)(a*sin(2*M_PI*i*freq[0]/(float)sf));
    waveform[1][i] = (oType)(a*sin(2*M_PI*i*freq[1]/(float)sf));
  }

  memset(zero,0,bitLength*sizeof(oType));

  interp = bitLength;
}

template<class oType> 
VrFSKMod<oType>::VrFSKMod(int f0, int f1, int bit_rate)
  :dataRate(bit_rate)
{
  freq[0] = f0;
  freq[1] = f1;
}

#endif

