/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gr_check_counting_s.h>
#include <gr_io_signature.h>
#include <stdlib.h>
#include <stdio.h>

gr_check_counting_s_sptr
gr_make_check_counting_s ()
{
  return gr_check_counting_s_sptr (new gr_check_counting_s ());
}

gr_check_counting_s::gr_check_counting_s ()
  : gr_sync_block ("gr_check_counting",
		   gr_make_io_signature (1, 1, sizeof (short)),
		   gr_make_io_signature (0, 0, 0)),
    d_state(SEARCHING), d_history (0), d_current_count (0),
    d_total_errors (0), d_total_shorts (0)
{
  enter_SEARCHING ();
}

int
gr_check_counting_s::work (int noutput_items,
			  gr_vector_const_void_star &input_items,
			  gr_vector_void_star &output_items)
{
  unsigned short *in = (unsigned short *) input_items[0];

  for (int i = 0; i < noutput_items; i++){
    unsigned short	x = in[i];
    
    switch (d_state){

    case SEARCHING:
      if (x == d_current_count){
	right ();
	d_current_count = d_current_count + 1;
	if (right_three_times ())
	  enter_LOCKED ();
      }
      else {
	wrong ();
	d_current_count = x + 1;
      }
      break;

    case LOCKED:
      if (x == d_current_count){
	right ();
	d_current_count = d_current_count + 1;
      }
      else {
	wrong ();
	log_error (d_current_count, x);
	d_current_count = d_current_count + 1;
	if (wrong_three_times ())
	  enter_SEARCHING ();
      }
      break;

    default:
      abort ();
    }

    d_total_shorts++;
  }

  return noutput_items;
}

void
gr_check_counting_s::enter_SEARCHING ()
{
  d_state = SEARCHING;

  fprintf (stdout, "gr_check_counting: enter_SEARCHING at offset %8ld (0x%08lx)\n",
	   d_total_shorts, d_total_shorts);
}

void
gr_check_counting_s::enter_LOCKED ()
{
  d_state = LOCKED;

  fprintf (stdout, "gr_check_counting: enter_LOCKED at offset %8ld (0x%08lx)\n",
	   d_total_shorts, d_total_shorts);
}

void
gr_check_counting_s::log_error (unsigned short expected, unsigned short actual)
{
  fprintf (stdout, 
"gr_check_counting: expected %5d (0x%04x) got %5d (0x%04x) offset %8ld (0x%08lx)\n",
	   expected, expected, actual, actual, d_total_shorts, d_total_shorts);
}
