/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { useState } from "preact/hooks";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import {
  OfficerAccount,
  opFixedSuccess,
  TalerExchangeResultByMethod2,
  TalerHttpError
} from "@gnu-taler/taler-util";
import {
  buildPaginatedResult,
  useExchangeApiContext,
} from "@gnu-taler/web-util/browser";
import _useSWR, { mutate, SWRHook } from "swr";
import { useOfficer } from "./officer.js";
const useSWR = _useSWR as unknown as SWRHook;

export const PAGINATED_LIST_SIZE = 10;
// when doing paginated request, ask for one more
// and use it to know if there are more to request
export const PAGINATED_LIST_REQUEST = PAGINATED_LIST_SIZE + 1;

/**
 * @param account
 * @param args
 * @returns
 */
export function useCurrentDecisions({
  investigated,
}: {
  investigated?: boolean;
} = {}) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;
  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  const [offset, setOffset] = useState<string>();

  async function fetcher([officer, offset, investigation]: [
    OfficerAccount,
    string | undefined,
    boolean | undefined,
  ]) {
    return await api.getAmlDecisions(officer, {
      order: "dec",
      offset,
      active: true,
      investigation,
      limit: PAGINATED_LIST_REQUEST,
    });
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod2<"getAmlDecisions">,
    TalerHttpError
  >(
    !session ? undefined : [session, offset, investigated, "getAmlDecisions"],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.records,
    offset,
    setOffset,
    (d) => String(d.rowid),
    PAGINATED_LIST_REQUEST,
  );
}

export function revalidateAccountDecisions() {
  return mutate(
    (key) => Array.isArray(key) && key[key.length - 1] === "getAmlDecisions",
    undefined,
    { revalidate: true },
  );
}
/**
 * @param account
 * @param args
 * @returns
 */
export function useAccountDecisions(accountStr: string) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;
  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  const [offset, setOffset] = useState<string>();

  async function fetcher([officer, account, offset]: [
    OfficerAccount,
    string,
    string | undefined,
  ]) {
    return await api.getAmlDecisions(officer, {
      order: "dec",
      offset,
      account,
      limit: PAGINATED_LIST_REQUEST,
    });
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod2<"getAmlDecisions">,
    TalerHttpError
  >(
    !session ? undefined : [session, accountStr, offset, "getAmlDecisions"],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.records,
    offset,
    setOffset,
    (d) => String(d.rowid),
    PAGINATED_LIST_REQUEST,
  );
}

/**
 * FIXME: The return type doesn't look quite right.
 *
 * @param account
 * @param args
 * @returns
 */
export function useAccountActiveDecision(accountStr?: string) {
  const officer = useOfficer();
  const session =
    accountStr !== undefined && officer.state === "ready"
      ? officer.account
      : undefined;
  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  async function fetcher([officer, account]: [
    OfficerAccount,
    string,
    string | undefined,
  ]) {
    return await api.getAmlDecisions(officer, {
      order: "dec",
      account,
      active: true,
      limit: PAGINATED_LIST_REQUEST,
    });
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod2<"getAmlDecisions">,
    TalerHttpError
  >(
    !session ? undefined : [session, accountStr, "getAmlDecisions"],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  if (!data.body.records.length) return opFixedSuccess(undefined);
  return opFixedSuccess(data.body.records[0]);
}
