/***************************************
  $Revision: 1.4 $

  Radix payload (rp) - user level functions for storing data in radix trees

  rp_search = search the loaded radix trees using an ascii key

  Status: NOT REVIEWED, TESTED
  
  Design and implementation by: Marek Bukowy
  
  ******************/ /******************
  Copyright (c) 1999                              RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/


#include <rp.h>
/*+++++++++++++++
  translates a query into a binary prefix (or prefixes, if range).
  for registry+space (or if they are zero, for all
  registries/spaces)
  finds tree 
  calls RX_bin_search (returning node copies).
  will not put duplicate entries (composed inetnums).
  returns some sort of error code :-) 
  
  Cuts the number of answers from RX_bin_search down to max_count,
  but since some of the answers may have been "normalized" in the
  underlying functions (multiple occurences removed), 
  the result is _at_most_ max_count.
  
  appends to a given list of data blocks (not nodes!)

  The EXLESS search on inetnum tree should return the shortest range 
  that was found, by means of comparing span (size) of the range.
  If there are more of size equal to the smallest one, they are also
  returned.

  returns RX_OK or a code from an underlying function
++++++++++++*/
/******************************************************************************/
/*+ search +*/
er_ret_t
RP_asc_search ( 
               rx_srch_mt search_mode, 
               int par_a,
               int par_b,
               char *key,          /*+ search term: (string) prefix/range/IP +*/
               int   reg_id,
	       rp_attr_t  attr,    /*+ extra tree id (within the same reg/spc/fam +*/
               GList **anslist,    /*+ answers go here, please +*/
               int    max_count    /*+ max # of answers. RX_ALLANS == unlimited +*/
               )
{ 
  GList    *preflist = NULL;
  GList      *datlist = NULL;
  er_ret_t   err; 
  ip_range_t testrang;
  int        locked = 0;
  rx_srch_mt first_pass_mode = search_mode;
  ip_keytype_t key_type;
  ip_space_t   spc_id;
  rx_fam_t   fam_id = RP_attr2fam( attr );
  rx_tree_t   *mytree;

  /*  abort on error (but unlock the tree) */  
  ER_dbg_va(FAC_RX, ASP_RX_SRCH_BOT,
	    "RP_asc_search:  query: mode %d (par %d) for %s",
	    search_mode, par_a, key);
  
  /*   0. make a list of prefixes to search for */
  err = rx_asc_make_preflist(search_mode, key, fam_id, 
			     &preflist, &testrang, &first_pass_mode);
  
  /* 1. find the tree */
  if( NOERR(err) ) {
    spc_id = IP_pref_b2_space( g_list_first(preflist)->data );
    err = RP_tree_get( &mytree, reg_id, spc_id, attr );
  }

  if( NOERR(err) ) {
    /* 2. lock the tree */
    TH_acquire_read_lock( &(mytree->rwlock) );
    locked = 1;
    
    /* 3. perform the search on prefixes the query was converted to. */
     err = rx_preflist_search ( first_pass_mode, par_a, par_b, 
				mytree, &preflist, &datlist);
  }
  if( NOERR(err) ) {
    /* 4. process the data pointers obtained from the search */
    err = rx_asc_process_datlist( first_pass_mode, fam_id, &datlist,
				  &testrang, anslist );
  }
  if( NOERR(err) ) {
    /* 5. an inetnum/composed/exless was forced to exact in the first go.
       So if the exact did not match yet, an encompassing prefix must 
       be searched in exless mode */
    
    if( first_pass_mode != search_mode && g_list_length(*anslist) == 0 ) {

      ER_dbg_va(FAC_RX, ASP_RX_SRCH_DET,
		"rx_asc_search: doing pass 2 with mode %d", search_mode);
      
      /* clean old lists */
      wr_clear_list( &preflist );
      wr_clear_list( &datlist );
      
      /* mhm. make new prefix list */
      dieif ( IP_smart_conv(key, 0, 1 /* now take encompassing one */,
			    &preflist, IP_EXPN, &key_type) != IP_OK);
      
      /* search again, this time with the real search_mode */
      err = rx_preflist_search ( search_mode, par_a, par_b, 
				mytree, &preflist, &datlist);
      
      if( err == RX_OK ) {
	/*  process the data pointers obtained from the search */
	err = rx_asc_process_datlist( search_mode, fam_id, &datlist, 
				      &testrang, anslist );
      }
    }
  }

  if( locked ) {
    /* 100. unlock the tree */
    TH_release_read_lock( &(mytree->rwlock) );
  }

  /* clean up */
  wr_clear_list( &preflist ); 
  wr_clear_list( &datlist );  

  /* NOTE if error occured, anslist may be partly filled in. */
  return err;
}
     
