/***************************************
  $Revision: 1.11 $

  Utilities (ut). memwrap.c - memory allocation wrappers. 
                              Facilitate easy changing a memory allocation
			      library and provide uniform error codes.

  Status: NOT REVUED, TESTED, 

  Design and implementation by: Marek Bukowy

  ******************/ /******************
  Copyright (c) 1999                              RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/

#include <stdlib.h>
#include <erroutines.h>
#include <stubs.h>
#include <glib.h>

#define USE_LOGGING

#ifdef USE_LOGGING
/* flag whether logging now active */
static int UT_memory_logging = 0;

void 
UT_memory_log (int active) 
{
    if (active) {
        UT_memory_logging = 1;
    } else {
        UT_memory_logging = 0;
    }
}

static void
UT_alloc_log (const void *ptr, int len, const char* file, int line) 
{
    if (UT_memory_logging) {
        ER_dbg_va(FAC_UT, ASP_UT_MEM, 
	    "allocated %d bytes at address %p in %s:%d",
	    len, ptr, file, line);
    }
}

static void
UT_free_log (const void *ptr, const char* file, int line) 
{
    if (UT_memory_logging) {
        ER_dbg_va(FAC_UT, ASP_UT_MEM, 
	    "freed memory at address %p in %s:%d", 
	    ptr, file, line); 
    }
}

static void
wr_free_list_log (const void *ptr, const char* file, int line) 
{
    if (UT_memory_logging) {
        ER_dbg_va(FAC_UT, ASP_UT_MEM,
	    "freeing list and elements at address %p in %s:%d", 
	    ptr, file, line); 
    }
}
#else

void 
UT_memory_log (int active)
{
    ER_perror(FAC_UT, UT_NOMEMLOG, 
        "logging not supported, recompile %s to enable", __FILE__);
}

/* if logging is disabled, then these functions are NOOP's */
#define UT_alloc_log(ptr,len,file,line)
#define UT_free_log(ptr,file,line) 
#define UT_free_list_log(ptr,file,line) 

#endif /* USE_LOGGING */

void *
UT_malloc_real (size_t size, const char *file, int line)
{
    void *ptr;

    ptr = malloc(size);
    if (ptr == NULL) {
        ER_perror(FAC_UT, UT_OUTMEM, 
            "malloc(%u) out of memory at %s:%d", size, file, line);
        die;
    }
    UT_alloc_log(ptr, size, file, line); 
    return ptr;
}

void *
UT_calloc_real (size_t num, size_t size, const char *file, int line)
{
    void *ptr;

    ptr = calloc(num, size);
    if (ptr == NULL) {
        ER_perror(FAC_UT, UT_OUTMEM, 
            "calloc(%u, %u) out of memory at %s:%d", num, size, file, line);
        die; 
    }
    UT_alloc_log(ptr, size * num, file, line);
    return ptr;
}

void *
UT_realloc_real (void *ptr, size_t size, const char *file, int line)
{
    char *tmp_ptr;
    
    tmp_ptr = realloc(ptr, size);
    if (tmp_ptr == NULL ) {
        ER_perror(FAC_UT, UT_OUTMEM, 
            "realloc(%p, %u) out of memory at %s:%d", ptr, size, file, line);
        die; 
    } 
    UT_free_log(ptr, file, line);
    UT_alloc_log(tmp_ptr, size, file, line);
    return tmp_ptr;
}

void 
UT_free_real (void *ptr, const char *file, int line)
{
    dieif(ptr == NULL);
    free(ptr);
    UT_free_log(ptr, file, line); 
}

char *
UT_strdup_real (const char *str, const char *file, int line)
{
    char *area;

    area = UT_malloc_real(strlen(str) + 1, file, line);
    strcpy(area, str);
    
    return area;
}


/* legacy functions */

void 
wr_log_set (int value) 
{
    UT_memory_log(value);
}

er_ret_t 
wr_real_malloc (void **ptr, size_t size, const char* file, int line) 
{
    *ptr = UT_malloc_real(size, file, line);
    return UT_OK;
}

er_ret_t 
wr_real_calloc (void **ptr, size_t num, size_t size, const char* file, 
                int line) 
{
    *ptr = UT_calloc_real(num, size, file, line);
    return UT_OK;
}


er_ret_t  
wr_real_realloc (void **ptr, size_t size, const char* file, int line) 
{
    *ptr = UT_realloc_real(*ptr, size, file, line);
    return UT_OK;
}

er_ret_t 
wr_real_free (void *ptr, const char* file, int line) 
{
    UT_free_real(ptr, file, line);
    return UT_OK;
}


/* make a copy and return the pointer to the allocated area, like strdup() */
char *
wr_real_string (const char *text, const char *file, int line)
{
    return UT_strdup_real(text, file, line);
}

/* for GList's foreach */
static
void
wr_free_list_element (void *cpy, void *trash)
{
    wr_real_free(cpy, __FILE__, __LINE__);
}

/* for GList's foreach */
void
wr_real_clear_list (GList **list, const char* file, int line)
{
    /* allow NULL argument */
    if( *list != NULL ) {
	wr_free_list_log(*list, file, line); 
	g_list_foreach(*list, wr_free_list_element, NULL);
	g_list_free(*list);
	*list = NULL;
    }
}

