#
# File:		tcprelay.pl
# Author:	Kazumasa Utashiro
# Modified:	G. Paul Ziemba
# From:		tcprelay,v 1.2 1992/04/13 19:10:28 utashiro
# Date:		93.01.25
# SCCS:		@(#)tcprelay.pl	1.10 9/29/94
# Purpose:	application-level tcp stream relay; handles ftp, too.
#
;#
;# tcprelay: application level tcp bridge
;#
;# Copyright (c) 1990,1991,1992 Kazumasa Utashiro
;# Software Research Associates, Inc., Japan <utashiro@sra.co.jp>
;#
;# Version 1.0, Oct 29 1990
;# Version 1.1, Jan 21 1991
;; $rcsid = '$Id: tcprelay,v 1.2 1992/04/13 19:10:28 utashiro Exp $';#'
;#
;# Usage:
;#	tcprelay [switches] servername clientname [service]
;#      tcprelay [switches] -i
;#
;# Switches:
;#	-fg:	force foregound
;#	-bg:	force backgound [default]
;#	-ftp:	force ftp mode (automatically on when connecting to ftp port)
;#
;# Description:
;#	This program relays tcp connection in application layer, which
;#	is useful when connecting across the IP disjoint gateway.
;#	Tcprelay connect to specified server and then makes local port
;#	and listen for connection from the client.  After tcprelay listen,
;#	anybody can connect to that port, so client name is required
;#	to avoid unexpected connect request.  Local port number is not
;#	explicitly defined, so you have to see message from tcprelay
;#	and invoke internet command with that number on client machine.
;#
;#		--------   ----|----   --------
;#		|client|---|gateway|---|server|
;#		--------   ----|----   --------
;#
;#	If the session seems to be ftp, tcprelay fakes PORT command
;#	in ftp interaction.  It makes connection to port in CLIENT which
;#	is specified in PORT command, and makes local socket to listen from
;#	ftp SERVER and returns that local port number to SERVER instead of
;#	the number sent from ftp CLIENT.  Use -ftp option when you want to
;#	connect to ftpd which doesn't have standard port number 21.
;#
;#	Default service is ftp, because this program is made for doing
;#	ftp originaly.
;#
;# Example:
;#	1) % tcprelay server client	: on gateway
;#	   port=xxxx			: remember port number in message
;#	2) % ftp gateway xxxx		: on client
;#
;# require 'sys/socket.ph';

sub usage {
    print "900 $0: Usage\n";
    ($myname = $0) =~ s|.*/||;
    print "Usage: $myname server client, or $myname -i\n";
    print "$rcsid\n" if $rcsid =~ /:/;
    exit 1;
}


########################################################################
#			CONSTANTS
########################################################################

$Tsockaddr	= 'S n a4 x8';
$TCP		= (getprotobyname('tcp'))[2];




    unless (do 'sys/socket.ph') {
	eval 'sub SOCK_STREAM {1;} sub AF_INET {2;} sub PF_INET {2;}';
    }

    if ($> == 0) {
	    #
	    # Shouldn't run as root!
	    #
	    ($newuid, $newgid) = &nobodyids;
	    $) = $newgid;
	    $( = $newgid;
	    $< = $newuid;
	    $> = $newuid;
    }

    #
    # Default path unless specified in Makefile
    #
    $ENV{'PATH'} = "/bin:/usr/bin:/etc:/usr/etc:/usr/ucb";

    while ($_ = $ARGV[0], /^-/) {
	    shift;
	    if (/-s$/)		{$silent = 1;		next;}
	    if (/-ftp$/)	{$ctype = 'ftp';	next;}
	    if (/-d(\d*)$/)	{$debug = $1||1;	next;}
	    if (/-l$/)		{$syslogging = 1;	next;}
	    if (/-(fg|bg)$/)	{$fg = $1 eq 'fg';	next;}
	    if (/^-c(.*)$/)	{$ConfigFile = $1 || shift; next;}
	    &usage;
    }

    $progname = $0;
    $progname =~ s:.*/::g;
    $| = 1;

    &usage if ($#ARGV < $[+1);

    &configfile;

    ($servername, $clientname, $serverport, $localport) = @ARGV;

    $serverport='ftp' unless $serverport;

    chop($localname = `hostname`);
    $localaddr = (gethostbyname($localname))[4];

    ($serveraddr = &getaddr($servername)) ||
	    die "Unknown server $servername.\n";

    ($clientaddr = &getaddr($clientname)) ||
	    die "Unknown client $clientname.\n";

    if ($serverport !~ /^\d+$/) {
	    $serverport = (getservbyname($serverport, 'tcp'))[2];
    }

    if (!defined($ctype)) {
	    if ($serverport == 21) {
		    $ctype = 'ftp';
	    } else {
		    $ctype = 'something';
	    }
    }

    $masterpid=$$;

    $SIG{'HUP'}=$SIG{'INT'}=$SIG{'QUIT'}=$SIG{'TERM'}='terminate';
    $SIG{'ALRM'} = 'IGNORE';

    &relay($masterpid, $clientaddr, $localaddr,
	$ctype, $serveraddr, $serverport);

sub terminate {
	kill -15, $masterpid;
	exit 1;
}

sub relay
{
    local(	$masterpid,	# top-level PID
		$clientaddr,	# client's IP, packed C4
		$localaddr,	# relay host's canonical IP, packed C4
		$type,		# 'ftp' or 'data' or 'something'
		$serveraddr,	# destination IP, packed C4
		$serverport)	# server port number
	= @_;

    local($newport, $that, $this, $addr, $peeraddr, $bcount);
    local($Cname, $Sname);
    local($toplevel) = ($$ eq $masterpid);

    $Cname = gethostbyaddr($clientaddr, &AF_INET) . "(" . 
	join('.', unpack('C4', $clientaddr)) . ")";

    $Sname = gethostbyaddr($serveraddr, &AF_INET) . "(" .
	join('.', unpack('C4', $serveraddr)) . ")";


    #
    # server connection
    # TBD - probably should time out here
    #
    $that = pack($Tsockaddr, &AF_INET, $serverport, $serveraddr);
    socket(S1, &PF_INET, &SOCK_STREAM, $TCP) || die "socket: $!";
    if (!connect(S1, $that)) {
	    print STDERR "connect: $Sname, port $serverport: $!\n";
	    exit 1;
    }

    select(S1); $| = 1; select(stdout);

    &logit( "connect", ($toplevel? 900: 920),
	($toplevel? "C": "Slave c") .  "onn to serv " .
	(($type eq 'ftp')? "(FTP) ": "") .  $Sname );

    #
    # client connection
    #
    $this = pack($Tsockaddr, &AF_INET, $localport, "\0\0\0\0");
    $localport = 0;
    socket(A1, &PF_INET, &SOCK_STREAM, $TCP) || die "socket: $!\n";
    bind(A1, $this) || die "bind: $!\n";
    listen(A1, 1) || die "listen: $!\n";
    $newport = (unpack($Tsockaddr, getsockname(A1)))[1];
    if (!$toplevel && !fork) {
	    close(S1);
	    close(A1);
	    return ($newport);
    }

    close(S), close(C) unless $toplevel;

    open(S,"+>&S1"); close(S1);
    open(A,"+>&A1"); close(A1);

    printf "Please connect to port=%d\n", $newport if $toplevel;

    #
    # time out here in case client has gone away
    #
    $SIG{'ALRM'} = 'client_timeout';
    &alarm(60);
    ($addr = accept(C, A)) || die "accept: $!\n";
    &alarm(0);
    close(A);

    $peeraddr = (unpack($Tsockaddr, $addr))[2];

    if ($toplevel) {
	    if ($peeraddr ne $clientaddr) {
		    printf ("910 Connection from %s is not allowed!\n",
			join('.', unpack('C4', $peeraddr)));
		    exit 1;
	    }
	    &logit("connect", "920",
		sprintf ("Conn from client %s\n", $Cname));
    }

    select(S); $| = 1; select(C); $| = 1; select(stdout);

    if ($child = fork) {
	    if ($toplevel && !$fg && ($pid = fork)) {
		    &logit("fork", "900", "Remote -> Client (pid = $pid)");
		    &logit("exit", "900", "$$: exiting");
		    exit 0;
	    }
	    $bcount = &forward('data', S, C, $serveraddr);
	    &logit("bytecount", "900", "$bcount bytes $Sname->$Cname");
    } else {
	    &logit("fork", "900", "Client -> Remote (pid = $$)");
	    $bcount = &forward($type, C, S, $serveraddr); # serveraddr needed
	    &logit("bytecount", "900", "$bcount bytes $Cname->$Sname");
    }
    &logit("exit", "900", "$$: exiting");
    exit 0;
}

sub forward
{
    local($type, $from, $to, $serveraddr) = @_;
    local($bcount, $rc);

    if ($type ne 'ftp') {
	    #
	    # Normal session
	    #
	    while($rc = read($from, $_, 4096)) {
		    print $to $_;
		    $bcount += $rc;
	    }
	    shutdown($from, 1);
	    shutdown($to, 0);
	    return $bcount;
    }

    #
    # FTP - need to spoof PORT
    #

    local($myportaddr) = &best_if_addr($serveraddr);

    &logit("connect", "900",
	"(ftp) i/f to remote: " . join('.', unpack('C4', $myportaddr)));

    while (<$from>) {
	    #
	    # Perhaps this match won't work sometimes if we get
	    # non-line chunks, since it's not line-buffered.
	    # Maybe this stream ought to be line buffered (?)
	    #
	    if (/^PORT ([\d,]+)/ && (@p = split(/,/, $1))) {
		    &logit("ftp", "920", "R PORT->$_");
		    $p = &relay($masterpid, $clientaddr, $localaddr,
			'data', pack('C4', @p), $p[4]*256 + $p[5]);

		    #
		    # output in netascii (<stuff>\r\n)
		    #
		    $_ = sprintf("PORT %d,%d,%d,%d,%d,%d\r\n",
				 unpack('C4', $myportaddr), $p/256, $p%256);
		    &logit("ftp", "920", "S PORT->$_");
	    }
	    print $to $_;
	    $bcount += length($_);
    }

    shutdown($from, 1);
    shutdown($to, 0);

    return $bcount;
}

sub getaddr
{
    local($_) = @_;
    /^[0-9\.]+$/ ? pack("C4", split(/\./)) : (gethostbyname($_))[4];
}

#
# find route to destination & return address of appropriate interface
#
sub best_if_addr
{ # remote addr
    local($serveraddr) = $_[0];
    local($server_dq) = sprintf("%d.%d.%d.%d", unpack('C4', $serveraddr));
    local($OldPath, $NsProg);

    ($ENV{'PATH'}, $NsProg, $OldPath) = &pathit($NETSTATPATH, "netstat");

    open(RT, "$NsProg -rn|") || die "$NsProg: $!";
    $ENV{'PATH'} = $OldPath;

    while (<RT>) {
	    split(?\s+?, $_);

	    if (/^default/) {
		    $IfD = $_[5];
	    }
	    next if ($_[0] !~ /^([\d\.]+)$/);
	    $Dest = $_[0];

	    next if ($_[2] !~ /U/);

	    if ($_[2] =~ /H/) {
		    #
		    # Host route
		    #
		    ($GH{$Dest}, $IfH{$Dest}) = @_[1,5];
	    } else {
		    #
		    # Net route
		    #
		    ($GN{$Dest}, $IfN{$Dest}) = @_[1,5];
	    }
    }
    close(RT);

    #
    # First see if we have a host route to the destination
    #
    foreach (keys(%GH)) {
	    if ($_ eq $server_dq) {
		    &logit("interface", 900,
			"Host route to $_: $GH{$_} via $IfH{$_}");
		    return &ifaddr($IfH{$_});
	    }
    }

    #
    # Now see if we have a net route. This algorithm probably
    # works only with nets whose subnet masks are multiples of
    # eight bits. Maybe the comparisons should really be done
    # with bit-masks.
    #
    foreach (keys(%GN)) {
	    $_a = $_;

	    s/(\.0)*$//;	# leave only net part
	    s/(\W)/\\$1/g;	# quote metacharacters

	    #
	    # Does the net part of the route destination
	    # match the leading part of the server we're trying
	    # to reach?
	    #
	    if ($server_dq =~ /^$_\./) {
		    &logit("interface", 900,
			"Net route to $_a: $GN{$_a} via $IfN{$_a}");
		    return &ifaddr($IfN{$_a});
	    }
    }

    if (defined($IfD)) {
	    &logit("interface", 900,
		"no Net or Host route, using default route: $IfD");
	    return &ifaddr($IfD);
    }

    &logit("interface", 900,
	sprintf("&bia: No Host, Net, or Default route, using %s",
	join('.', unpack('C4', $localaddr))));

    return $localaddr;
}

sub ifaddr {	# ifname
    local($ifname) = $_[0];
    local($ip);
    local($OldPath, $IfProg);

    ($ENV{'PATH'}, $IfProg, $OldPath) = &pathit($IFCONFIGPATH, "ifconfig");

    &logit("interface", 900, "ifaddr: want IP-addr for: $ifname");
    open (IFCONFIG, "$IfProg $ifname|") || die "ifconfig: $!";
    $ENV{'PATH'} = $OldPath;
    while (<IFCONFIG>) {
	    chop;
	    if (/\s+inet\s+(\S+)\s+/) {
		    $ip = $1;
		    last;
	    }
    }
    close(IFCONFIG);
    if (!defined($ip)) {
	    local($dq) = join('.', unpack('C4', $localaddr));

	    #
	    # This is a band-aid
	    #
	    &logit("interface", 900,
		"ifaddr: can't parse ifconfig output, returning $dq");
	    return $localaddr;
    }
    return pack('C4', split(/\./, $ip));
}


sub client_timeout {
    &logit("exit", 900, "client connection timed out. Exiting.");
    &terminate;
}

sub nobodyids
{
    local($uid, $gid, @_);

    @_ = getpwnam("nobody");

    if ($#_ < 0) {
	    return (65534, 65534);
    }

    return (@_[2,3]);
}
