/* $Id$
 *
 * hgdb.c:  provides functions necessary for using GDB with hypercad
 *
 * In particular, hgdb.c provides functions
 *
 *    gdb_entity_data_size():  returns the size of an entity in bytes
 *    gdb_entity_data_copy():  data copy for each kind of entity
 *    gdb_entity_data_dist():  returns a distance from one entity to
 *                             any other
 *
 * The constant definitions for use with GDB are defined in hcore.h
 * GDB entities are all given in Klein model coordinates.
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include "hcore.h"
#include "hglobals.h"

/*-----------------------------------------------------------------------
 * Function:	gdb_entity_data_size(type)
 * Description  Computes the size of an entity data object of the given type:	
 * Args  IN:	type: flag of type to compute size of
 *      OUT:	
 * Returns:	the size
 * Author:	lena
 * Date:	Sun Apr  8 15:23:37 1990
 * Notes:      
 */
gdb_entity_data_size(type)
gdb_Entity_type type;
{
  switch (type) {
  case POINT: return( sizeof(R2Point) );
  case SEGMENT: return( sizeof(kSegment) );
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	gdb_entity_data_copy(a,b,type)
 * Description:	Copies one entity to another
 * Args  IN:	a,b: destination and source (a <- b)
 *              type: type flags of entities a and b
 *      OUT:	
 * Returns:	Nothing
 * Author:	lena
 * Date:	Sun Apr  8 15:34:44 1990
 * Notes:	
 */
gdb_entity_data_copy(a,b,type)
char *a,*b;
gdb_Entity_type type;
{
  int i;
  
  switch (type) {
  case POINT:
    for (i=0; i<2; ++i) ((double *)a)[i] = ((double *)b)[i];
    break;
  case SEGMENT:
    for (i=0; i<2; ++i) {
      ((kSegment *)a)->p[i][0] = ((kSegment *)b)->p[i][0];
      ((kSegment *)a)->p[i][1] = ((kSegment *)b)->p[i][1];
    }
    break;
  default:
    break;
  }
}

/*-----------------------------------------------------------------------
 * Function:     gdb_entity_data_dist
 * Description:  Computes the distance between two entities
 * Arguments:    a,b: the two entities
 *               type_a, type_b: type flags of entities a and b
 * Returns:      Distance from a to b
 * Notes:        Returns the value GDB_INFINITY to indicate infinite
 *               distance.  At present, the GDB_INFINITY is returned
 *		 as the distance between any pair of things for which
 *		 neither thing is a point.  In other words, the only
 *		 meaningful distances which are returned must involve
 *		 at least one point.
 *		      This function computes the Euclidean distance
 *		 between objects in the (affine embedding at z=1 of
 *		 the) current model.
 */
double gdb_entity_data_dist(a,type_a,b,type_b)
char *a,*b;
gdb_Entity_type type_a, type_b;
{
  double d;

  switch (type_a) {
  case POINT:
    switch (type_b) {
    case POINT:		/* a=point, b=point */
      d =  PointPointDist(*((kPoint *)a), *((kPoint *)b));
      break;
    case SEGMENT:		/* a=point, b=segment */
	d = PointGeoSegmentDist( *((kPoint *)a), ((kSegment *)b));
	break;
    }
    break;
  case SEGMENT:
    switch (type_b) {
    case POINT:			/* a=segment, b=point */
      d = PointGeoSegmentDist( *((kPoint *)b), ((kSegment *)a));
      break;
    case SEGMENT:		/* a=segment, b=segment */
      d = SegmentSegmentDist( (kSegment *)a, (kSegment *)b);
      break;
    }
  }
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:	PointPointDist
 * Description:	Computes the Euclidean distance between two points in
 *              the current model
 * Args  IN:	a,b: kPoints (Klein coordinates)
 *      OUT:	
 * Returns:	d: the distance
 * Author:	lena
 * Date:	Mon Apr 16 23:09:15 1990
 * Notes:	
 */
double PointPointDist(a,b)
kPoint a,b;
{
  kPoint x,y;
  double d;
  
  switch(model){
  case KLEIN:
    d = R2dist(a,b);
    break;
  case POINCARE:
    KleinToPoincare(x,a);
    KleinToPoincare(y,b);
    d = R2dist(x,y);
    break;
  case UHP:
    KleinToUHP(x,a);
    KleinToUHP(y,b);
    d = R2dist(x,y);
    break;
  }
  return(d);
}

     
/*-----------------------------------------------------------------------
 * Function:	SegmentSegmentDist
 * Description:	computes a distance between geodesic segments based
 *              on their endpoints
 * Args  IN:	s1,s2: the segments
 *      OUT:	
 * Returns:	d: the distance
 * Author:	lena
 * Date:	Mon Apr  9 15:37:24 1990
 * Notes:	
 */
double SegmentSegmentDist(s1,s2)
kSegment *s1,*s2;
{
  double d1,d2,d;

  d1 =  R2distsq(s1->p[0],s2->p[0]);
  d1 += R2distsq(s1->p[1],s2->p[1]);
  d2 =  R2distsq(s1->p[0],s2->p[1]);
  d2 += R2distsq(s1->p[1],s2->p[0]);
  if (d1 < d2)
    d = sqrt(d1);
  else
    d = sqrt(d2);
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:	PointGeoSegmentDist
 * Description:	computes the euclidean distance from a point to a 
 *              geodesic segment in the current model
 * Args  IN:	a : an R2Point
 *              *b: pointer to a kSegment
 *      OUT:	
 * Returns:	d: the distance
 * Author:	lena
 * Date:	Sun Apr  8 16:51:59 1990
 * Notes:	
 */
double PointGeoSegmentDist(a,b)
kPoint a;
kSegment *b;
{
  double d;
  kPoint x,y,z,c;

  switch(model){
  case KLEIN:
    d = PointLineDist(a,b->p[0],b->p[1]);
    break;
  case POINCARE:
    KleinToPoincare(x,a);
    KleinToPoincare(y,b->p[0]);
    KleinToPoincare(z,b->p[1]);
    PoincareGeoCenter(c,y,z);
    if ((c[0] == 0.0) && (c[1] == 0.0))
      d = PointLineDist(x,y,z);
    else
      d = PointArcDist(x,y,z,c);
    break;
  case UHP:
     KleinToUHP(x,a);
     KleinToUHP(y,b->p[0]);
     KleinToUHP(z,b->p[1]);
     UHPGeoCenter(c,y,z);
     if (c[1] != 0.0) 
       d = PointLineDist(x,y,z);
     else 
       d = PointArcDist(x,y,z,c);
     break;
  default:
     break;
  }
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:	PointLineDist
 * Description:	computes euclidean distance from a point to 
 *              a straight line segment in R2
 * Args  IN:	a: an R2Point
 *              b1,b2: the endpoints of the line segment
 *      OUT:	
 * Returns:	d: the distance
 * Author:	lena
 * Date:	Sun Apr  8 17:04:30 1990
 * Notes:	the distance from a point to a line is given as the
 *              minimum of the distances to both endpoints and the
 *              orthogonal distance to the line.  Set x = b2 -b1, and
 *              y = a - b2.  Then the orthogonal distance is given by
 *             
 *                     d = || y - x ( x.y / |x|^2)||
 */
double PointLineDist(a,b1,b2)
R2Point a,b1,b2;
{
  double dot,proj,d1,d2,d;
  R2Point x,y;

  x[0] = b2[0] - b1[0];
  x[1] = b2[1] - b1[1];
  y[0] = a[0] - b1[0];
  y[1] = a[1] - b1[1];

  dot = x[0] * y[0] + x[1] * y[1];
  proj = dot / normsq(x);

  if ((0 <= proj) && ( proj <= 1)) {            
    y[0] = y[0] - proj * x[0];
    y[1] = y[1] - proj * x[1];
    d = norm(y);
    return(d);
  }
  else {
    d1 = R2dist(a,b1);
    d2 = R2dist(a,b2);
    if (d1 < d2) 
      d = d1;
     else
      d = d2;
    return(d);
  }
}

/*-----------------------------------------------------------------------
 * Function:	PointArcDist
 * Description:	computes the euclidean distance from a point to an arc
 * Args  IN:	a: the R2 point
 *              b1,b2: endpoints of the arc
 *              c: the center
n *      OUT:	
 * Returns:	d: the distance
 * Author:	lena
 * Date:	Mon Apr  9 13:03:43 1990
 * Notes:	
 */
double PointArcDist(a,b1,b2,c)
R2Point a,b1,b2,c;
{
  R2Point s,t1,t2;
  double d1,d2,d;
  double arg0,arg1,arg2;
  int i;

  for (i = 0; i<2; ++i){
    s[i] = a[i] - c[i];
    t1[i] = b1[i] - c[i];
    t2[i] = b2[i] - c[i];
  }
  arg1 = atan2(s[1],s[0]);
  arg0 = atan2(t1[1],t1[0]);
  arg2 = atan2(t2[1],t2[0]);
  if (ABS(arg0 - arg2) > M_PI) {
    if (arg0 < 0) 
      arg0 += TWO_PI;
    else 
      arg2 += TWO_PI;
    if (arg1 < 0)
      arg1 += TWO_PI;
  }
  if ((arg0 < arg1) && (arg1 < arg2))
    d = ABS(R2dist(c,a)-R2dist(c,b1));
  else if ((arg2 < arg1) && (arg1 < arg0))
    d = ABS(R2dist(c,a)-R2dist(c,b1));
  else {
    d1 = R2dist(a,b1);
    d2 = R2dist(a,b2);
    if (d1 < d2) 
      d = d1;
    else
      d = d2;
  }
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:	AddEntity
 * Description:	add an entity to the data base and then draw it
 * Args  IN:	type: the type of the entity to add
 *		*data: the entity's data
 * Returns:	name of new entity, or "-1" if error
 * Author:	mbp
 * Date:	Thu Apr 12 16:44:12 1990
 * Notes:	BUG?: This uses gdb_generate_unique name to generate
 *		names which are ascii representations of integers.
 *		This could cause problems if the name integer gets too
 *		large --- either larger than an int can store (int is
 *		used in gdb_generate_unique_name) or larger than the
 *		local array used here to store the name.
 */
char *
  AddEntity(type, data)
gdb_Entity_type type;
char *data;
{
  char name[NAMELEN];
  gdb_Entity entity;

  gdb_generate_unique_name("", name);
  gdb_add_entity(type, data, name, &entity);
  if (entity == NULL)
   return("-1");
  else {
    if (!Batching)  DrawEntity(entity);
    return(gdb_entity_name(entity));
  }
}

/*-----------------------------------------------------------------------
 * Function:	DeleteEntity
 * Description:	delete an entity
 * Args  IN:	name: name of entity to delete
 * Returns:	success status
 * Author:	mbp
 * Date:	Thu Apr 12 17:22:02 1990
 * Notes:	
 */
int
  DeleteEntity(name)
char *name;
{
  gdb_Entity entity;

  gdb_retrieve_entity(GDB_ALL_ENTITIES, GDB_NAME, name, &entity);
  if (entity == NULL) return(-1);
  gdb_delete_entity(entity);
  if (!Batching)
    DrawPicture();
  else
    BatchDeletion = 1;
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	DeleteAll
 * Description:	delete everything in the picture
 * Args:	(none)
 * Returns:	success status
 * Author:	mbp
 * Date:	Mon Apr 16 17:31:05 1990
 * Notes:	
 */
int
  DeleteAll()
{
  gdb_delete_all();
  if (!Batching)
    DrawPicture();
  else
    BatchDeletion = 1;
  return(0);
}
