/* $Id$
 *
 * zoom.c
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

/*
 * 		    Notes about Zoom Factor slider
 *
 * The Zoom Factor slider is a bit complicated because SunView does
 * not allow floating point sliders.  We therefore construct one
 * ourselves out of the following gadgets:
 *
 *  	zoom_factor_slider:	The slider bar itself, without
 *  				any labels.
 *
 *  	zoom_factor_label_msg:	The text "Zoom Factor", to be
 *  				displayed above the slider bar.
 *
 *  	zoom_factor_value_msg:	The current value of our slider;
 *  				this is a floating point number
 *  				converted to ascii and displayed
 *  				as a message item in the panel
 *  				just below the slider bar.  It
 *  				is updated every time the slider
 *  				is moved.
 *
 *  Other things associated with the slider are:
 *
 *  	zoom_factor_slider_value: The actual integer value of
 *  				zoom_factor_slider.
 *
 *  	ZOOM_FACTOR_SLIDER_DEFAULT: Constant; default initial
 *  				value of
 *  				zoom_factor_slider_value.
 *
 *  	SLIDER_VALUE_TO_ZOOM_FACTOR: Macro to convert
 *  				zoom_factor_slider_value to
 *  				floating point zoom factor
 *  				value.
 *
 *  	char *ZoomFactorString(): Function to convert
 *  				zoom_factor_slider_value to
 *  				ascii representation of zoom
 *  				factor.
 *
 *  	ZoomFactorProc():	Notify proc for
 *  				zoom_factor_slider.  Gets called
 *  				everytime zoom_factor_slider's
 *  				value changes. Updates the value
 *  				of zoom_factor_slider_value and
 *  				the display of
 *  				zoom_factor_value_msg.
 *
 */

#include <stdio.h>
#include "internal.h"
#include "../hcore.h"

static Panel_item zoom_factor_slider, zoom_factor_label_msg,
  zoom_factor_value_msg;

#define SLIDER_VALUE_TO_ZOOM_FACTOR(x)		((double)(x)/10.0)
#define ZOOM_FACTOR_SLIDER_DEFAULT		15

static int	zoom_factor_slider_value = ZOOM_FACTOR_SLIDER_DEFAULT;
static int	ZoomFactorProc();
static char	*ZoomFactorString();

Panel_item
  wCreateZoomFactorSlider(panel, sx, sy, lx, ly, vx, vy)
Panel panel;
int sx, sy, lx, ly, vx, vy;
{
  zoom_factor_slider = 
    panel_create_item(panel, PANEL_SLIDER,
		      PANEL_ITEM_X, sx,
		      PANEL_ITEM_Y, sy,		      
		      PANEL_VALUE, zoom_factor_slider_value,
		      PANEL_SLIDER_WIDTH, 120,
		      PANEL_MIN_VALUE, 11,
		      PANEL_MAX_VALUE, 20,
		      PANEL_SHOW_RANGE, FALSE,
		      PANEL_SHOW_VALUE, FALSE,
		      PANEL_NOTIFY_LEVEL, PANEL_ALL,
		      PANEL_NOTIFY_PROC, ZoomFactorProc,
		      0);
  if (zoom_factor_slider == NULL) return(NULL);
  
  zoom_factor_label_msg =
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_ITEM_X, lx,
		      PANEL_ITEM_Y, ly,
		      PANEL_LABEL_STRING, "Zoom Factor",
		      0);
  if (zoom_factor_label_msg == NULL) return(NULL);
  
  zoom_factor_value_msg =
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_ITEM_X, vx,
		      PANEL_ITEM_Y, vy,
		      PANEL_LABEL_STRING,
		      ZoomFactorString(zoom_factor_slider_value),
		      0);
  if (zoom_factor_value_msg == NULL) return(NULL);

  return(zoom_factor_slider);
}

/*-----------------------------------------------------------------------
 * Function:	ZoomFactorProc
 * Description:	respond to change in value of zoom factor slider
 * Args  IN:	item: slider's handle
 *		value: new value for slider
 *		event: the event which caused the value change
 * Notes:	This updates the display of the slider value and
 *		stores the new value in the static global variable
 *		zoom_factor_slider_value.
 */
/*ARGSUSED*/
static int
  ZoomFactorProc(item, value, event)
Panel_item item;
int value;
Event *event;
{
  panel_set(zoom_factor_value_msg,
	    PANEL_LABEL_STRING,
	      ZoomFactorString(zoom_factor_slider_value=value),
	    0);
}

/*-----------------------------------------------------------------------
 * Function:	ZoomFactorString
 * Description:	format a string for displaying the zoom factor
 * Args  IN:	value: integer value of zoom factor slider
 * Returns:	pointer to formatting string
 */
static char *
  ZoomFactorString(value)
int value;
{
  static char buf[10];

  sprintf(buf, "[%3.1f]",
	  SLIDER_VALUE_TO_ZOOM_FACTOR(value));
  return(buf);
}

/*ARGSUSED*/
int
  wZoom(item, event)
Panel_item item;
Event *event;
{
  double zoom_factor;
  hWindow hw;
  int i;

  zoom_factor = 
    SLIDER_VALUE_TO_ZOOM_FACTOR(zoom_factor_slider_value);

  if (event_shift_is_down(event))
    zoom_factor = 1 / zoom_factor;

  GethWindow(&hw);
  for (i=0; i<4; ++i)
    ScaleR2Point(hw.corner[i], hw.center, hw.corner[i], zoom_factor);
  SethWindow(&hw);
}

static int
  ScaleR2Point(new, base, old, factor)
R2Point new, base, old;
double factor;
{
  int i;

  for (i=0; i<2; ++i)
    new[i] = base[i] + factor * (old[i] - base[i]);
}
