/*  $Id: elbowd.c,v 2.2 90/07/11 13:11:20 mbp Exp Locker: mbp $
 *
 * elbowd.c: elbow daemon
 *
 * usage: elbowd [-port n] [-cont] program [args ...]
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

/*
 * This program functions as a server in the client-server IPC model.  We
 * listen for connection requests on a port, and when we receive one, we
 * start the program (with arguments) given on the command line, with
 * stdin and stdout of the program connected to the client socket.  The
 * port on which we listens for connection requests may be specified with
 * the -port option.  If not specified, we use ELBOWPORT (as defined in
 * bothpipe.h).  If the -cont option is present, we return to listen for
 * further connection requests. Otherwise we terminate after starting the
 * program.
 */

#include <signal.h>
#include <sys/wait.h>
#include <errno.h>

/* BOTHPIPE_H should be the pathname of "bothpipe.h", in quotes */
#include BOTHPIPE_H

#define MAXARGS 100
#define NEXTARG --argc; ++argv
#define ARG (*argv)

int elbowport=ELBOWPORT;	/* port for accepting connections */
int elbowsocket;		/* server socket (for accepting connections) */
int clientsocket;		/* client socket (for talking to client) */
char serverhost[80];		/* ascii name for local host */
Sockaddr elbowaddr;		/* elbow socket address */
Sockaddr clientaddr;		/* client socket adress */
int reaper();			/* takes care of deceased child processes */
int reaped=0;			/* gets set to 1 upon reaping */

int cont = 0;

extern int errno;

char *program, *pargv[MAXARGS];

char *me;

main(argc, argv)
     int argc;
     char *argv[];
{
  Sockaddr *addr_ptr;
  int clientnamelen;

  me = argv[0];
  
  if (!parse_args(argc, argv)) {
    printf("usage: elbowd [-port n] [-cont] program [args ...]\n");
    exit(1);
  }
  
  /* Create the socket */
  if ((elbowsocket=socket(AF_INET, SOCK_STREAM, 0)) == -1)
    pquit("elbowd (socket)");
  
  /* Get name of my host */
  gethostname(serverhost, sizeof(serverhost));
  
  /* Convert it to an Internet address */
  if ((addr_ptr=resolve_host(serverhost)) == NULL)
    quit("elbow: can't resolve own host address");
  bcopy((char*)addr_ptr, (char*)&elbowaddr, sizeof(elbowaddr));
  
  /* Set port number */
  elbowaddr.sin_port = elbowport;
  
  /* Bind address to socket */
  if (bind(elbowsocket,(struct sockaddr *)&elbowaddr,sizeof(elbowaddr)) != 0)
    if (errno == EADDRINUSE) {
      perror("elbowd (bind)");
      pquit("Is an elbowd already running?");
    }
    else
      pquit("elbowd (bind)");
  
  listen(elbowsocket, 5);
  signal(SIGCHLD, reaper);

  /* Process connection requests */
  do {
    clientnamelen = sizeof(clientaddr);
    clientsocket =
      accept(elbowsocket, (struct sockaddr *)&clientaddr, &clientnamelen);
    if (clientsocket < 0)
      if (reaped)
	reaped = 0;
      else
	perror("elbowd (accept)");
    else
      start_program(clientsocket);
  } while (cont);

  /* If we got this far, we're supposed to terminate after starting
   * the program.  In fact, however, we pause till the program dies
   * so we can reap it.  This is necessary to prevent zombie processes
   * from accumulating. */
  pause();

}

parse_args(argc, argv)
     int argc;
     char *argv[];
{
  char **pargv_ptr;
  static char program_name_buf[100];

  NEXTARG;			/* skip over argv[0] = me */
  if (argc < 1) return(0);

  /* check for -port option */
  if (strcmp(ARG, "-port") == 0) {
    if (argc < 2) return(0);
    NEXTARG;
    elbowport = atoi(ARG);
    NEXTARG;
    if (elbowport < IPPORT_RESERVED) {
      fprintf("elbowd: can't use reserved port; port number must be > %1d\n",
	      IPPORT_RESERVED);
      exit(1);
    }
  }

  /* check for -cont option */
  if (strcmp(ARG, "-cont") == 0) {
    cont = 1;
    NEXTARG;
  }
  
  /* Parse program and its args; pargv is a null-terminated list of
     pointers to the arguments to program. */
  program = *argv;  NEXTARG;
  extract_last_component(pargv[0]=program_name_buf, program);
  pargv_ptr = pargv+1;
  while (argc) {
    *pargv_ptr = *argv;
    ++pargv_ptr;
    NEXTARG;
  }
  *pargv_ptr = NULL;
  return(1);
}

start_program(clientsocket)
     int clientsocket;
{
  int pid;

  switch (pid = fork()) {

  case -1:			/* error */
    fprintf(stderr, "elbowd: fork failed\n");
    break;

  case 0:			/* child */
    {
      int numfds, c;

      /* Connect both stdin and stdout to the client socket */
      dup2(clientsocket, 0);
      dup2(clientsocket, 1);

      /* Close all other files, except stderr */
      numfds = getdtablesize();
      for (c=3; c<numfds; ++c)
	close(c);

      /* Start the program */
      execv(program, pargv);

      /* Should never get here! */
      quit("elbowd: execv failed; elbowd child aborting!");
    }
    break;

  default:			/* parent */
    close(clientsocket);
    break;
  }
}

reaper()
{
  union wait status;

  if (wait(&status) == -1)
    perror("elbowd (wait)");
  signal(SIGCHLD, reaper);
  reaped = 1;
}
