/* ratectl.c, bitrate control routines (linear quantization only currently) */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>
#include <math.h>

#include "config.h"
#include "global.h"

/* private prototypes */
static double var_sblk _ANSI_ARGS_((unsigned char *p, int lx));

/* rate control variables */
static int outbitcnt;
double Xi, Xp, Xb, r, d0i, d0p, d0b, avg_act;
static double R, T, d, actsum;
static int Np, Nb, S, Q;

void rc_init_seq()
{
  /* reaction parameter (constant) */
  if (r==0.0)  r = 2.0*bit_rate/picture_rate;

  /* average activity */
  if (avg_act==0.0)  avg_act = 400.0;

  /* remaining # of bits in GOP */
  R = 0;

  /* global complexity measure */
  if (Xi==0.0) Xi = 160.0*bit_rate/115.0;
  if (Xp==0.0) Xp =  60.0*bit_rate/115.0;
  if (Xb==0.0) Xb =  42.0*bit_rate/115.0;

  /* virtual buffer fullness */
  if (d0i==0.0) d0i = 10.0*r/31.0;
  if (d0p==0.0) d0p = 10.0*r/31.0;
  if (d0b==0.0) d0b = 1.4*10.0*r/31.0;

  fprintf(statfile,"\nrate control: sequence initialization\n");
  fprintf(statfile,
    " initial global complexity measures (I,P,B): Xi=%.0f, Xp=%.0f, Xb=%.0f\n",
    Xi, Xp, Xb);
  fprintf(statfile," reaction parameter: r=%.0f\n", r);
  fprintf(statfile,
    " initial virtual buffer fullness (I,P,B): d0i=%.0f, d0p=%.0f, d0b=%.0f\n",
    d0i, d0p, d0b);
  fprintf(statfile," initial avarage activity: avg_act=%.1f\n", avg_act);
}

void rc_init_GOP(n)
int n;
{
  R+= N*bit_rate/picture_rate;
  Np = N/M - 1;
  Nb = N - Np - 1;

  fprintf(statfile,"\nrate control: new group of pictures (GOP)\n");
  fprintf(statfile," target number of bits for GOP: R=%.0f\n",R);
  fprintf(statfile," number of P frames in GOP: Np=%d\n",Np);
  fprintf(statfile," number of B frames in GOP: Nb=%d\n",Nb);
}

/* Note: we need to substitute K for the 1.4 and 1.0 constants -- this can
   be modified to fit image content */

/* Step 1: compute target bits for current picture being coded */
void rc_init_pict()
{
  double Tmin;

  switch (pict_type)
  {
  case I_TYPE:
    T = R/(1.0+Np*Xp/(Xi*1.0)+Nb*Xb/(Xi*1.4));
    d = d0i;
    break;
  case P_TYPE:
    T = R/(Np+Nb*1.0*Xb/(1.4*Xp));
    d = d0p;
    break;
  case B_TYPE:
    T = R/(Nb+Np*1.4*Xp/(1.0*Xb));
    d = d0b;
    break;
  }

  Tmin = bit_rate/(8.0*picture_rate);

  if (T<Tmin)
    T = Tmin;

  S = bitcount();
  Q = 0;
  actsum = 0.0;

  fprintf(statfile,"\nrate control: start of picture\n");
  fprintf(statfile," target number of bits: T=%.0f\n",T);
}

void rc_update_pict()
{
  double X;

  S = bitcount() - S; /* total # of bits in frame */
  R-= S; /* remaining # of bits in GOP */
  X = S*((double)Q/(mb_width*mb_height));
  d+= S - T;
  avg_act = actsum/(mb_width*mb_height);

  switch (pict_type)
  {
  case I_TYPE:
    Xi = X;
    d0i = d;
    break;
  case P_TYPE:
    Xp = X;
    d0p = d;
    Np--;
    break;
  case B_TYPE:
    Xb = X;
    d0b = d;
    Nb--;
    break;
  }

  fprintf(statfile,"\nrate control: end of picture\n");
  fprintf(statfile," actual number of bits: S=%d\n",S);
  fprintf(statfile," average quantization parameter Q=%d\n",Q);
  fprintf(statfile," remaining number of bits in GOP: R=%.0f\n",R);
  fprintf(statfile,
    " global complexity measures (I,P,B): Xi=%.0f, Xp=%.0f, Xb=%.0f\n",
    Xi, Xp, Xb);
  fprintf(statfile,
    " virtual buffer fullness (I,P,B): d0i=%.0f, d0p=%.0f, d0b=%.0f\n",
    d0i, d0p, d0b);
  fprintf(statfile," remaining number of P frames in GOP: Np=%d\n",Np);
  fprintf(statfile," remaining number of B frames in GOP: Nb=%d\n",Nb);
  fprintf(statfile," average activity: avg_act=%.1f\n", avg_act);
}

/* compute initial quantization stepsize (at the beginning of picture) */
int rc_start_mb()
{
  int mquant;

  mquant = (int)floor(d*31.0/r+0.5);
  if (mquant<1)
    mquant = 1;
  if (mquant>31)
    mquant = 31;
  mquant <<= 1;

/*
  fprintf(statfile,"rc_start_mb:\n");
  fprintf(statfile,"mquant=%d\n",mquant);
*/

  return mquant;
}

/* Step 2: measure virtual buffer - estimated buffer discrepency */
int rc_calc_mquant(j,p)
int j;
unsigned char *p;
{
  int mquant;
  double dj, Qj, actj, N_actj, var;

  /* measure virtual buffer discrepency from uniform distribution model */
  dj = d + (bitcount()-S) - j*(T/(mb_width*mb_height));

  /* scale against dynamic range of mquant and the bits/picture count */
  Qj = dj*31.0/r;

  /* take minimum spatial activity measure of luminance blocks */

  /* frame */
  actj = var_sblk(p,width);
  var = var_sblk(p+8,width);
  if (var<actj) actj = var;
  var = var_sblk(p+8*width,width);
  if (var<actj) actj = var;
  var = var_sblk(p+8*width+8,width);
  if (var<actj) actj = var;
  /* field */
  var = var_sblk(p,width<<1);
  if (var<actj) actj = var;
  var = var_sblk(p+8,width<<1);
  if (var<actj) actj = var;
  var = var_sblk(p+width,width<<1);
  if (var<actj) actj = var;
  var = var_sblk(p+width+8,width<<1);
  if (var<actj) actj = var;
  actj+= 1.0;
  actsum+= actj;

  /* compute normalized activity */
  N_actj = (2.0*actj+avg_act)/(actj+2.0*avg_act);

  /* modulate mquant with combined buffer and local activity measures */
  mquant = (int)floor(Qj*N_actj+0.5);

  /* clip mquant to legal (linear) range */
  if (mquant<1)
    mquant = 1;
  if (mquant>31)
    mquant = 31;

  Q+= mquant;
  mquant <<= 1;

/*
  fprintf(statfile,"rc_calc_mquant(%d):\n",j);
  fprintf(statfile,"bitcount=%d, dj=%f, Qj=%f, actj=%f, N_actj=%f, mquant=%d\n",
    bitcount(),dj,Qj,actj,N_actj,mquant);
*/

  return mquant;
}

/* compute variance of 8x8 block */
static double var_sblk(p,lx)
unsigned char *p;
int lx;
{
  int i, j;
  unsigned int v, s, s2;

  s = s2 = 0;

  for (j=0; j<8; j++)
  {
    for (i=0; i<8; i++)
    {
      v = *p++;
      s+= v;
      s2+= v*v;
    }
    p+= lx - 8;
  }

  return s2/64.0 - (s/64.0)*(s/64.0);
}

void init_vbv_delay()
{
  outbitcnt = (int)(((7*vbv_buffer_size*16384)/8)*90000.0/bit_rate)
              -(int)(90000.0/picture_rate+0.5);
}

void calc_vbv_delay()
{
  outbitcnt += (int)(90000.0/picture_rate+0.5);
  vbv_delay = (int)(outbitcnt-bitcount()*90000.0/bit_rate);
  fprintf(statfile,"\nvbv_delay=%d\n",vbv_delay);

  if (vbv_delay<0)
  {
    if (!quiet)
      fprintf(stderr,"vbv_delay underflow: %d\n",vbv_delay);
    vbv_delay = 0;
  }

  if (vbv_delay>65535)
  {
    if (!quiet)
      fprintf(stderr,"vbv_delay overflow: %d\n",vbv_delay);
    vbv_delay = 65535;
  }
}
