/*
 * raymath.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: raymath.c,v 3.0.1.1 90/03/07 21:26:08 craig Exp $
 *
 * $Log:	raymath.c,v $
 * Revision 3.0.1.1  90/03/07  21:26:08  craig
 * patch4: Added VectCoordSys().
 * 
 * Revision 3.0  89/10/27  02:06:01  craig
 * Baseline for first official release.
 * 
 */
#include <math.h>
#include "typedefs.h"
#include "constants.h"
#include "funcdefs.h"

/*
 * Normalize a vector, return original length.
 */
double
normalize(a)
register Vector *a;
{
	double d;

	d = sqrt(a->x*a->x + a->y*a->y + a->z*a->z);
	if(d == 0.)
		return 0.;
	a->x /= d;
	a->y /= d;
	a->z /= d;
	return d;
}

/*
 * Compute cross-product of a and b, place normalized result in o.  Returns
 * length of result before normalization.
 */
double
crossp(o, a, b)
Vector *o, *a, *b;
{
	rawcrossp(o, a, b);
	return normalize(o);
}

/*
 * Compute cross-product of a and b, place result in o.
 */
rawcrossp(o, a, b)
Vector *o, *a, *b;
{
	o->x = (a->y * b->z) - (a->z * b->y);
	o->y = (a->z * b->x) - (a->x * b->z);
	o->z = (a->x * b->y) - (a->y * b->x);
}

/*
 * Calculate direction of refracted ray using Heckbert's formula.  Returns TRUE
 * if a total internal reflection occurs.
 */
refract(dir, from_index, to_index, I, N, cos1)
double from_index, to_index, cos1;
Vector *dir, I, N;
{
	double kn, cos2, k;

	if (cos1 < 0.) {
		/*
		 * Hit the 'backside' of a surface -- flip the
		 * normal.
		 */
		N.x = -N.x;
		N.y = -N.y;
		N.z = -N.z;
		cos1 = -cos1;
	}

	kn = from_index / to_index;
	cos2 = 1. - kn*kn*(1. - cos1*cos1);
	if (cos2 < 0.)
		return TRUE;		/* Total internal reflection. */
	k = kn * cos1 - sqrt(cos2);
	veccomb(kn, I, k, N, dir);
	return FALSE;
}

/*
 * Given a vector, find two additional vectors such that all three
 * are mutually perpendicular and xaxis X yaxis = vector.  The given
 * vector need not be normalized. xaxis and yaxis are normalized.
 */
VectCoordSys(vector, xaxis, yaxis)
Vector *vector, *xaxis, *yaxis;
{
	xaxis->x = vector->y;
	xaxis->y = -vector->x;
	xaxis->z = 0.;
	if (normalize(xaxis) == 0.) {
		xaxis->x = 0.;
		xaxis->y = -vector->z;
		xaxis->z = vector->y;
		if (normalize(xaxis) == 0.)
			RSwarning("LightCoordSys: Can't find X axis!\n");
		yaxis->x = (vector->y * vector->y) + (vector->z * vector->z);
		yaxis->y = -vector->x * vector->y;
		yaxis->z = -vector->x * vector->z;
		(void)normalize(yaxis);
	} else {
		yaxis->x = vector->x * vector->z;
		yaxis->y = vector->y * vector->z;
		yaxis->z = -(vector->x * vector->x) -(vector->y * vector->y);
		(void)normalize(yaxis);
	}
}

#ifdef DUMB_CPP
/*
 * Return difference between two vectors.
 */
vecsub(a, b, res)
Vector a, b, *res;
{
	res->x = a.x - b.x;
	res->y = a.y - b.y;
	res->z = a.z - b.z;
}

/*
 * Return sum of two vectors.
 */
vecadd(a, b, res)
Vector a, b, *res;
{
	res->x = a.x + b.x;
	res->y = a.y + b.y;
	res->z = a.z + b.z;
}

/*
 * Compute scalar product of a vector.
 */
scalar_prod(scale, vec, res)
double scale;
Vector vec, *res;
{
	res->x = vec.x * scale;
	res->y = vec.y * scale;
	res->z = vec.z * scale;
}

veccomb(s1, v1, s2, v2, res)
double s1, s2;
Vector v1, v2, *res;
{
	res->x = s1 * v1.x + s2 * v2.x;
	res->y = s1 * v1.y + s2 * v2.y;
	res->z = s1 * v1.z + s2 * v2.z;
}

addscaledvec(v1, s, v2, res)
Vector v1, v2, *res;
double s;
{
	res->x = v1.x + s * v2.x;
	res->y = v1.y + s * v2.y;
	res->z = v1.z + s * v2.z;
}
#endif
