/* $Id: composewin.c,v 1.3 90/06/08 22:36:33 pturner Exp Locker: pturner $
 *
 * transformations, curve fitting, etc.
 *
 */

#include <stdio.h>
#include <math.h>
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include "globals.h"
#include "defines.h"

#define define_select_set_panel3(panel,panel_item,x,y,s) \
				panel_item=panel_create_item( panel,\
				PANEL_CYCLE,\
				PANEL_LABEL_STRING,s,\
	 			PANEL_CHOICE_STRINGS,\
			"Set 0", "Set 1", "Set 2", "Set 3", "Set 4", "Set 5",\
			"Set 6", "Set 7", "Set 8", "Set 9", "Set 10", "Set 11",\
			"Set 12", "Set 13", "Set 14", 0,\
	    		PANEL_ITEM_X,x,\
	    		PANEL_ITEM_Y,y,\
			0 );

#define define_select_set_panel4(panel,panel_item,x,y) \
				panel_item=panel_create_item( panel,\
				PANEL_CYCLE,\
				PANEL_LABEL_STRING,"Select set:",\
	 			PANEL_CHOICE_STRINGS,\
			"Set 0", "Set 1", "Set 2", "Set 3", "Set 4", "Set 5",\
			"Set 6", "Set 7", "Set 8", "Set 9", "Set 10", "Set 11",\
			"Set 12", "Set 13", "Set 14", "All sets", 0,\
	    		PANEL_ITEM_X,x,\
	    		PANEL_ITEM_Y,y,\
			0 );

void drawgraph();		/* calls plotone(), called any time the graph
				 * needs to be re-drawn */

extern Frame main_frame;
extern Cursor cursor, cursor_save;	/* cursor for zoom */
extern Cursor cursor_wait;
extern Canvas canvas;
extern Pixwin *pw;
extern Pixfont *winfont;	/* font to use for all of grtool */

static void histogram();

void define_compose_popup();	/* transformations, etc. */
Frame compose_frame;
static Panel compose_panel;

/* items for linear regression */
static Panel_item toggle_degree_item;	/* degree of fit */
static Panel_item toggle_resid_item;	/* load residuals, fitted values or
					 * nothing */
static Panel_item toggle_set_regress_item;	/* which set to use */

/* finite differencing */
static Panel_item toggle_set_differ_item;
static Panel_item toggle_differ_type_item;

/* numerical integration */
static Panel_item toggle_set_int_item;
static Panel_item toggle_int_type_item;
static Panel_item int_sum_item;

/* running averages */
static Panel_item toggle_ravglen_item;
static Panel_item toggle_run_item;
static Panel_item toggle_set_runavg_item;

/* histograms */
static Panel_item histo_binw_item;
static Panel_item histo_hxmin_item;
static Panel_item histo_hxmax_item;
static Panel_item histo_type_item;
static Panel_item toggle_set_histo_item;

/* DFTs */
static Panel_item toggle_set_fourier_item;
static Panel_item toggle_load_fourier_item;
static Panel_item toggle_inv_fourier_item;
static Panel_item toggle_type_fourier_item;

/* evaluate a formula */
static Panel_item compute_formula_item;
static Panel_item compute_set_item;
static Panel_item compute_load_item;

/* load a set */
static Panel_item load_set_item;
static Panel_item load_start_item;
static Panel_item load_step_item;
static Panel_item load_to_item;

/* evaluate a formula - load the next set */
static Panel_item compute_formulax_item;
static Panel_item compute_formulay_item;
static Panel_item load_start2_item;
static Panel_item load_stop2_item;
static Panel_item load_npts_item;
static Panel_item load_to2_item;

/* cross correlation */
static Panel_item toggle_set_xcor1_item;
static Panel_item toggle_set_xcor2_item;
static Panel_item toggle_xcor_type_item;
static Panel_item xcor_lag_item;

/* splines */
Panel_item toggle_spline_item;
Panel_item spline_start_item;
Panel_item spline_stop_item;
Panel_item spline_step_item;

/* linear convolution */
static Panel_item toggle_set_linc1_item;
static Panel_item toggle_set_linc2_item;

/* apply a digital filter in set 2 to set 1 */
static Panel_item toggle_set_digf1_item;
static Panel_item toggle_set_digf2_item;

/* sample a set */
static Panel_item toggle_set_sample_item;
static Panel_item sample_start_item;
static Panel_item sample_step_item;
static Panel_item sample_type_item;
static Panel_item sample_expr_item;

static void do_compute_proc();

/*
 * allow a carriage return to activate formula
 * interpreter
 */
static Panel_setting eotproc_compose(item, event)
    Panel_item item;
    Event *event;
{
    switch (event_id(event)) {
    case '\r':
    case '\n':
	do_compute_proc(item, event);
	return PANEL_NONE;
    default:
	return (panel_text_notify(item, event));
    }

}

/*
 * forward, backward and centered differences
 */
static void forwarddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h;

    for (i = 1; i < n; i++) {
	resx[i - 1] = x[i - 1];
	h = x[i - 1] - x[i];
	if (h == 0.0) {
	    resy[i - 1] = MBIG;
	    eflag = 1;
	} else {
	    resy[i - 1] = (y[i - 1] - y[i]) / h;
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

static void backwarddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h;

    for (i = 0; i < n - 1; i++) {
	resx[i] = x[i];
	h = x[i + 1] - x[i];
	if (h == 0.0) {
	    resy[i] = MBIG;
	    eflag = 1;
	} else {
	    resy[i] = (y[i + 1] - y[i]) / h;
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

static void centereddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h1, h2;

    for (i = 1; i < n - 1; i++) {
	resx[i - 1] = x[i];
	h1 = x[i] - x[i - 1];
	h2 = x[i + 1] - x[i];
	if (h1 + h2 == 0.0) {
	    resy[i] = MBIG;
	    eflag = 1;
	} else {
	    resy[i] = (y[i + 1] - y[i - 1]) / (h1 + h2);
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

/*
 * trapezoidal rule
 */
static double trapint(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i;
    double sum = 0.0;
    double h;

    for (i = 1; i < n; i++) {
	h = (x[i] - x[i - 1]);
	if (resx != NULL) {
	    resx[i - 1] = (x[i - 1] + x[i]) * 0.5;
	}
	sum = sum + h * (y[i - 1] + y[i]) * 0.5;
	if (resy != NULL) {
	    resy[i - 1] = sum;
	}
    }
    return sum;
}

/*
 * apply a digital filter
 */
static void do_digfilter_proc(item, event)
    Panel_item item;
    Event *event;
{
    int set1, set2, digfiltset;
    double *getx(), *gety();

    set1 = (int) panel_get_value(toggle_set_digf1_item);
    set2 = (int) panel_get_value(toggle_set_digf2_item);
    if (!(isactive(set1) && isactive(set2))) {
	errwin("Set not active");
	return;
    }
    if ((getsetlength(set1) < 3) || (getsetlength(set2) < 3)) {
	errwin("Set length < 3");
	return;
    }
    digfiltset = nextset();
    if (digfiltset != (-1)) {
	activateset(digfiltset);
	setlength(digfiltset, getsetlength(set1) - getsetlength(set2) + 1);
	sprintf(buf, "Digital filter from set %d applied to set %d", set2, set1);
	filterser(getsetlength(set1), getx(set1), gety(set1), getx(digfiltset), gety(digfiltset), gety(set2), getsetlength(set2));
	setcomment(digfiltset, buf);
	updatesetminmax(digfiltset);
	update_status(digfiltset);
	drawgraph();
    }
}

/*
 * linear convolution
 */
static void do_linearc_proc(item, event)
    Panel_item item;
    Event *event;
{
    int set1, set2, linearcset, i, itmp;
    double *getx(), *gety(), *xtmp;

    set1 = (int) panel_get_value(toggle_set_linc1_item);
    set2 = (int) panel_get_value(toggle_set_linc2_item);
    if (!(isactive(set1) && isactive(set2))) {
	errwin("Set not active");
	return;
    }
    if ((getsetlength(set1) < 3) || (getsetlength(set2) < 3)) {
	errwin("Set length < 3");
	return;
    }
    linearcset = nextset();
    if (linearcset != (-1)) {
	activateset(linearcset);
	setlength(linearcset, (itmp = getsetlength(set1) + getsetlength(set2) - 1));
	sprintf(buf, "Linear convolution of set %d with set %d", set1, set2);
	linearconv(gety(set2), gety(set1), gety(linearcset), getsetlength(set2), getsetlength(set1));
	xtmp = getx(linearcset);
	for (i = 0; i < itmp; i++) {
	    xtmp[i] = i;
	}
	setcomment(linearcset, buf);
	updatesetminmax(linearcset);
	update_status(linearcset);
	drawgraph();
    }
}

/*
 * cross correlation
 */
static void do_xcor_proc(item, event)
    Panel_item item;
    Event *event;
{
    int set1, set2, xcorset, itype, lag, i, ierr;
    double *getx(), *gety(), *xtmp;

    set1 = (int) panel_get_value(toggle_set_xcor1_item);
    set2 = (int) panel_get_value(toggle_set_xcor2_item);
    if (!(isactive(set1) && isactive(set2))) {
	errwin("Set not active");
	return;
    }
    itype = (int) panel_get_value(toggle_xcor_type_item);
    lag = atoi((char *) panel_get_value(xcor_lag_item));
    if (lag == 0 || (getsetlength(set1) - 1 < lag)) {
	errwin("Lag incorrectly specified");
	return;
    }
    if ((getsetlength(set1) < 3) || (getsetlength(set2) < 3)) {
	errwin("Set length < 3");
	return;
    }
    xcorset = nextset();
    if (xcorset != (-1)) {
	activateset(xcorset);
	setlength(xcorset, lag);
	if (set1 != set2) {
	    sprintf(buf, "X-correlation of set %d and %d at lag %d", set1, set2, lag);
	} else {
	    sprintf(buf, "Autocorrelation of set %d at lag %d", set1, lag);
	}
	ierr = crosscorr(gety(set1), gety(set2), getsetlength(set1), lag, itype, getx(xcorset), gety(xcorset));
	xtmp = getx(xcorset);
	for (i = 0; i < lag; i++) {
	    xtmp[i] = i;
	}
	setcomment(xcorset, buf);
	updatesetminmax(xcorset);
	update_status(xcorset);
	drawgraph();
    }
}

/*
 * splines
 */
static void do_spline_proc(item, event)
    Panel_item item;
    Event *event;
{
    int i, set, splineset, n, len, ier;
    double u, delx, *getx(), *gety(), *x, *y, *b, *c, *d, *xtmp, *ytmp;
    double xmin, xmax, ymin, ymax, start, stop, seval();
    extern double result;

    set = (int) panel_get_value(toggle_spline_item);
    if (!isactive(set)) {
	errwin("Set not active");
	return;
    }
    if ((len = getsetlength(set)) < 3) {
	errwin("Improper set length");
	return;
    }
    getsetminmax(set, &xmin, &xmax, &ymin, &ymax);
    strcpy(buf, panel_get_value(spline_start_item));
    if (strlen(buf) == 0) {
	errwin("Start item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &xmin, &xmax, &ymin, &ymax, 0, 0, &ier);
    if (ier)
	return;
    start = result;
    strcpy(buf, panel_get_value(spline_stop_item));
    if (strlen(buf) == 0) {
	errwin("Stop item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &xmin, &xmax, &ymin, &ymax, 0, 0, &ier);
    if (ier)
	return;
    stop = result;
    n = atoi((char *) panel_get_value(spline_step_item));
    if (n <= 1) {
	errwin("Number of steps must be > 1");
	return;
    }
    delx = (stop - start) / (n - 1);
    splineset = nextset();
    if (splineset != -1) {
	activateset(splineset);
	setlength(splineset, n);
	sprintf(buf, "Spline fit from set %d", set);
	x = getx(set);
	y = gety(set);
	b = (double *) calloc(len, sizeof(double));
	c = (double *) calloc(len, sizeof(double));
	d = (double *) calloc(len, sizeof(double));
	if (b == NULL || c == NULL || d == NULL) {
	    errwin("Not enough memory for splines");
	    cxfree(b);
	    cxfree(c);
	    cxfree(d);
	    killset(splineset);
	    return;
	}
	spline(len, x, y, b, c, d);
	xtmp = getx(splineset);
	ytmp = gety(splineset);

	for (i = 0; i < n; i++) {
	    xtmp[i] = start + i * delx;
	    ytmp[i] = seval(len, xtmp[i], x, y, b, c, d);
	}
	setcomment(splineset, buf);
	updatesetminmax(splineset);
	update_status(splineset);
	cxfree(b);
	cxfree(c);
	cxfree(d);
	drawgraph();
    }
}

/*
 * numerical integration
 */
static void do_int_proc(item, event)
    Panel_item item;
    Event *event;
{
    int setno, intset, itype;
    double *getx(), *gety();
    double sum;

    setno = (int) panel_get_value(toggle_set_int_item);
    itype = (int) panel_get_value(toggle_int_type_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    if (getsetlength(setno) < 3) {
	errwin("Set length < 3");
	return;
    }
    if (itype == 0) {
	intset = nextset();
	if (intset != (-1)) {
	    activateset(intset);
	    setlength(intset, getsetlength(setno) - 1);
	    sprintf(buf, "Cumulative sum of set %d", setno);
	    sum = trapint(getx(setno), gety(setno), getx(intset), gety(intset), getsetlength(setno));
	    setcomment(intset, buf);
	    updatesetminmax(intset);
	    sprintf(buf, "%lf", sum);
	    panel_set_value(int_sum_item, buf);
	    update_status(intset);
	    drawgraph();
	}
    } else {
	sum = trapint(getx(setno), gety(setno), NULL, NULL, getsetlength(setno));
	sprintf(buf, "%lf", sum);
	panel_set_value(int_sum_item, buf);
    }
}

/*
 * finite differences
 */
static void do_differ_proc(item, event)
    Panel_item item;
    Event *event;
{
    int setno, diffset, itype;
    double *getx(), *gety();

    setno = (int) panel_get_value(toggle_set_differ_item);
    itype = (int) panel_get_value(toggle_differ_type_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    if (getsetlength(setno) < 3) {
	errwin("Set length < 3");
	return;
    }
    diffset = nextset();
    if (diffset != (-1)) {
	activateset(diffset);
	switch (itype) {
	case 0:
	    sprintf(buf, "Forward difference of set %d", setno);
	    setlength(diffset, getsetlength(setno) - 1);
	    forwarddiff(getx(setno), gety(setno), getx(diffset), gety(diffset), getsetlength(setno));
	    break;
	case 1:
	    sprintf(buf, "Backward difference of set %d", setno);
	    setlength(diffset, getsetlength(setno) - 1);
	    backwarddiff(getx(setno), gety(setno), getx(diffset), gety(diffset), getsetlength(setno));
	    break;
	case 2:
	    sprintf(buf, "Centered difference of set %d", setno);
	    setlength(diffset, getsetlength(setno) - 2);
	    centereddiff(getx(setno), gety(setno), getx(diffset), gety(diffset), getsetlength(setno));
	    break;
	}
	setcomment(diffset, buf);
	updatesetminmax(diffset);
	update_status(diffset);
	drawgraph();
    }
}

/*
 * regression
 */
static void do_regress_proc(item, event)
    Panel_item item;
    Event *event;
{
    int fitset, setno, ideg, iresid, i;
    double *getx(), *gety(), *yd, *yf;

    setno = (int) panel_get_value(toggle_set_regress_item);
    ideg = (int) panel_get_value(toggle_degree_item) + 1;
    iresid = (int) panel_get_value(toggle_resid_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    fitset = nextset();
    if (fitset != (-1)) {
	activateset(fitset);
	setlength(fitset, getsetlength(setno));
	copyx(setno, fitset);
	fitcurve(getx(setno), gety(setno), getsetlength(setno), ideg, gety(fitset));
	if (iresid) {
	    yf = gety(fitset);
	    yd = gety(setno);
	    for (i = 0; i < getsetlength(fitset); i++) {
		yf[i] = yd[i] - yf[i];
	    }
	}
	sprintf(buf, "%d deg fit of set %d", ideg, setno);
	setcomment(fitset, buf);
	updatesetminmax(fitset);
	update_status(fitset);
    }
    drawgraph();
}

/*
 * running averages, medians, min, max, std. deviation
 */
static void do_runavg_proc(item, event)
    Panel_item item;
    Event *event;
{
    int runset, runlen, runtype, setno;
    double *getx(), *gety();

    setno = (int) panel_get_value(toggle_set_runavg_item);
    runlen = atoi(panel_get_value(toggle_ravglen_item));
    runtype = (int) panel_get_value(toggle_run_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    if (runlen >= getsetlength(setno)) {
	errwin("Length of running average > set length");
	return;
    }
    if (runlen < 2) {
	errwin("Length of running average < 2");
	return;
    }
    runset = nextset();
    if (runset != (-1)) {
	activateset(runset);
	setlength(runset, getsetlength(setno) - runlen + 1);
	switch (runtype) {
	case 0:
	    runavg(getx(setno), gety(setno), getx(runset), gety(runset), getsetlength(setno), runlen);
	    sprintf(buf, "%d-pt. avg. on set %d ", runlen, setno);
	    break;
	case 1:
	    runmedian(getx(setno), gety(setno), getx(runset), gety(runset), getsetlength(setno), runlen);
	    sprintf(buf, "%d-pt. median on set %d ", runlen, setno);
	    break;
	case 2:
	    runminmax(getx(setno), gety(setno), getx(runset), gety(runset), getsetlength(setno), runlen, 0);
	    sprintf(buf, "%d-pt. min on set %d ", runlen, setno);
	    break;
	case 3:
	    runminmax(getx(setno), gety(setno), getx(runset), gety(runset), getsetlength(setno), runlen, 1);
	    sprintf(buf, "%d-pt. max on set %d ", runlen, setno);
	    break;
	case 4:
	    runstddev(getx(setno), gety(setno), getx(runset), gety(runset), getsetlength(setno), runlen);
	    sprintf(buf, "%d-pt. std dev., set %d ", runlen, setno);
	    break;
	}
	setcomment(runset, buf);
	updatesetminmax(runset);
	update_status(runset);
    }
    drawgraph();
}

/*
 * DFT
 */
static void do_fourier_proc(item, event)
    Panel_item item;
    Event *event;
{
    double *getx(), *gety();
    int i, specset, setno, it, ilen, load, invflag, type;
    double *x, *y, *xx, *yy;

    setno = (int) panel_get_value(toggle_set_fourier_item);
    load = (int) panel_get_value(toggle_load_fourier_item);
    invflag = (int) panel_get_value(toggle_inv_fourier_item);
    type = (int) panel_get_value(toggle_type_fourier_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    if ((it = getsetlength(setno)) < 2) {
	errwin("Set length < 2");
	return;
    }
    specset = nextset();
    if (specset != -1) {
	window_set(canvas, WIN_CURSOR, cursor_wait, 0);
	window_set(compose_panel, WIN_CURSOR, cursor_wait, 0);
	x = getx(setno);
	y = gety(setno);
	ilen = it;
	/* ilen = it / 2 + it % 2; */
	activateset(specset);
	setlength(specset, ilen);
	xx = getx(specset);
	yy = gety(specset);
	dft(x, y, xx, yy, it, type, invflag);
	/* fouriercoef(y, xx, yy, it); */
	if (load == 0) {
	    for (i = 0; i < ilen; i++) {
		yy[i] = hypot(xx[i], yy[i]);
		xx[i] = i;
	    }
	}
	sprintf(buf, "DFT of set %d", setno);
	setcomment(specset, buf);
	updatesetminmax(specset);
	update_status(specset);
	window_set(canvas, WIN_CURSOR, cursor_save, 0);
	window_set(compose_panel, WIN_CURSOR, cursor_save, 0);
    }
    drawgraph();
}

/*
 * DFT by FFT
 */
static void do_fft_proc(item, event)
    Panel_item item;
    Event *event;
{
    int i, load, invflag;
    double *getx(), *gety();
    double *xx, *yy, *y;
    int i2, setno, specset, type;

    setno = (int) panel_get_value(toggle_set_fourier_item);
    load = (int) panel_get_value(toggle_load_fourier_item);
    invflag = (int) panel_get_value(toggle_inv_fourier_item);
    type = (int) panel_get_value(toggle_type_fourier_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    if (getsetlength(setno) < 2) {
	errwin("Set length < 2");
	return;
    }
    if ((i2 = ilog2(getsetlength(setno))) <= 0) {
	errwin("Set length not a power of 2");
	return;
    }
    specset = nextset();
    if (specset != -1) {
	activateset(specset);
	setlength(specset, getsetlength(setno));
	xx = getx(specset);
	yy = gety(specset);
	if (type == 0) {	/* real data */
	    y = gety(setno);
	    for (i = 0; i < getsetlength(setno); i++) {
		xx[i] = y[i];
		yy[i] = 0.0;
	    }
	} else {		/* imaginary data */
	    copyx(setno, specset);
	    copyy(setno, specset);
	}
	if (invflag) {
	    invflag = 0;
	} else {
	    invflag = 1;
	}
	fft(xx, yy, getsetlength(specset), i2, invflag);
	if (load == 0) {
	    for (i = 0; i < getsetlength(specset); i++) {
		yy[i] = hypot(xx[i], yy[i]);
		xx[i] = i;
	    }
	}
	sprintf(buf, "FFT of set %d", setno);
	setcomment(specset, buf);
	updatesetminmax(specset);
	update_status(specset);
	drawgraph();
    }
}

/*
 * histograms
 */
static void do_histo_proc(item, event)
    Panel_item item;
    Event *event;
{
    int hset, hist_type;
    double binw, xmin, xmax;

    hset = (int) panel_get_value(toggle_set_histo_item);
    if (!isactive(hset)) {
	errwin("Set not active");
	return;
    }
    if (getsetlength(hset) <= 0) {
	errwin("Set length = 0");
	return;
    }
    binw = atof(panel_get_value(histo_binw_item));
    if (binw <= 0.0) {
	errwin("Bin width <= 0");
	return;
    }
    xmin = atof(panel_get_value(histo_hxmin_item));
    xmax = atof(panel_get_value(histo_hxmax_item));
    hist_type = (int) panel_get_value(histo_type_item);
    histogram(hset, binw, xmin, xmax, hist_type);
    drawgraph();
}

static void histogram(setno, bins, xmin, xmax, hist_type)
    int setno, hist_type;
    double bins, xmin, xmax;
{
    int binmax, n1, n, i, j, nbins, hset;
    double sum = 0.0, spread, xi, *x, *y;
    int *ind;

    if ((hset = nextset()) == -1) {
	return;
    }
    n = getsetlength(setno);
    spread = xmax - xmin;
    nbins = (int) (spread / bins);
    ind = (int *) calloc(nbins, sizeof(int));
    if (ind == NULL) {
	errwin("Not enough memory for histogram");
	killset(hset);
	return;
    }
    if (nbins <= 0) {
	errwin("No bins, no work to do");
	killset(hset);
	return;
    }
    /* I don't believe there is any need to do this */
    for (i = 0; i < nbins; i++) {
	ind[i] = 0;
    }

    activateset(hset);
    setlength(hset, nbins);
    j = 0;
    y = gety(setno);
    for (i = 0; i < n; i++) {
	xi = y[i];
	j = (int) ((xi - xmin) / bins);
	ind[j] = ind[j] + 1;
    }
    n1 = 0;
    binmax = 0;
    x = getx(hset);
    y = gety(hset);
    for (i = 0; i < nbins; i++) {
	x[i] = i * bins + bins / 2.0 + xmin;
	sum = sum * hist_type + ind[i];	/* hist_type = 0 => regular histo */
	y[i] = sum;
	n1 = n1 + ind[i];
	if (ind[i] > binmax) {
	    binmax = ind[i];
	}
    }
    setplotsym(hset, HISTOSYM);
    setlinesym(hset, 0);
    updatesymbols(hset);
    updatesetminmax(hset);
    sprintf(buf, "Histogram from set # %d", setno);
    setcomment(hset, buf);
    update_status(hset);
    cxfree(ind);
}

/*
 * evaluate a formula
 */
static void do_compute_proc(item, event)
    Panel_item item;
    Event *event;
{
    int i, idraw = 0, setno, loadto;
    char fstr[80];

    setno = (int) panel_get_value(compute_set_item);
    loadto = (int) panel_get_value(compute_load_item);
    strcpy(fstr, panel_get_value(compute_formula_item));
    if (strlen(fstr) == 0) {
	errwin("Define formula first");
	return;
    }
    if (setno == maxplot) {
	for (i = 0; i < maxplot; i++) {
	    if (isactive(i)) {
		formula(i, fstr);
		idraw = 1;
	    }
	}
    } else if (isactive(setno)) {
	/* both loadto and setno do double duty here */
	if (loadto) {
	    loadto = nextset();
	    if (loadto != -1) {
		activateset(loadto);
		setlength(loadto, getsetlength(setno));
		copyx(setno, loadto);
		copyy(setno, loadto);
		setno = loadto;
	    } else {
		return;
	    }
	}
	formula(setno, fstr);
	idraw = 1;
    }
    if (idraw) {
	drawgraph();
    } else {
	errwin("Set(s) not active");
    }
}

/*
 * evaluate a formula loading the next set
 */
static void do_compute_proc2(item, event)
    Panel_item item;
    Event *event;
{
    int i, setno, npts, ier, toval;
    double start, stop, step, x, y, a, b, c, d;
    extern double result;
    char fstrx[80];
    char fstry[80];

    npts = atoi((char *) panel_get_value(load_npts_item));
    if (npts < 1) {
	errwin("Number of points < 1");
	return;
    }
    setno = nextset();
    if (setno < 0) {
	return;
    }
    activateset(setno);
    setlength(setno, npts);
    strcpy(fstrx, panel_get_value(compute_formulax_item));
    if (strlen(fstrx) == 0) {
	errwin("Undefined expression for X");
	return;
    }
    fixupstr(fstrx);
    strcpy(fstry, panel_get_value(compute_formulay_item));
    if (strlen(fstry) == 0) {
	errwin("Undefined expression for Y");
	return;
    }
    fixupstr(fstry);
    strcpy(buf, panel_get_value(load_start2_item));
    if (strlen(buf) == 0) {
	errwin("Start item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &a, &b, &c, &d, 0, 0, &ier);
    if (ier)
	return;
    start = result;
    strcpy(buf, panel_get_value(load_stop2_item));
    if (strlen(buf) == 0) {
	errwin("Stop item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &a, &b, &c, &d, 0, 0, &ier);
    if (ier)
	return;
    stop = result;
    if (npts - 1 == 0) {
	errwin("Number of points = 0");
	return;
    }
    step = (stop - start) / (npts - 1);
    toval = (int) panel_get_value(load_to2_item) + 1;
    loadset(setno, toval, start, step);
    strcpy(buf, "X=");
    strcat(buf, fstrx);
    formula(setno, buf);
    strcpy(buf, "Y=");
    strcat(buf, fstry);
    formula(setno, buf);
    drawgraph();
}

/*
 * load a set
 */
static void do_load_proc(item, event)
    Panel_item item;
    Event *event;
{
    int i, ier = 0, idraw = 0, setno, toval, getsetno(), getint(), isactive();
    double start, step;
    double x, y, a, b, c, d;
    extern double result;

    setno = (int) panel_get_value(load_set_item);
    toval = (int) panel_get_value(load_to_item) + 1;
    strcpy(buf, panel_get_value(load_start_item));
    if (strlen(buf) == 0) {
	errwin("Start item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &a, &b, &c, &d, 0, 0, &ier);
    if (ier)
	return;
    start = result;
    strcpy(buf, panel_get_value(load_step_item));
    if (strlen(buf) == 0) {
	errwin("Step item undefined");
	return;
    }
    fixupstr(buf);
    scanner(buf, &x, &y, &a, &b, &c, &d, 0, 0, &ier);
    if (ier)
	return;
    step = result;
    if (setno == maxplot) {
	for (i = 0; i < maxplot; i++) {
	    if (isactive(i)) {
		loadset(i, toval, start, step);
		idraw = 1;
	    }
	}
    } else if (isactive(setno)) {
	loadset(setno, toval, start, step);
	idraw = 1;
    }
    if (idraw) {
	drawgraph();
    } else {
	errwin("Set(s) not active");
    }
}

/*
 * sample a set, by start/step or logical expression
 */
static void do_sample_proc(item, event)
    Panel_item item;
    Event *event;
{
    int setno, startno, stepno, len, npts, i, resset, typeno, ier;
    double *xres, *yres, *x, *y, *getx(), *gety();
    double a, b, c, d;
    extern double result;
    char exprstr[128];

    setno = (int) panel_get_value(toggle_set_sample_item);
    if (!isactive(setno)) {
	errwin("Set not active");
	return;
    }
    typeno = (int) panel_get_value(sample_type_item);
    len = getsetlength(setno);
    resset = nextset();
    if (resset < 0) {
	return;
    }
    if (typeno == 0) {
	startno = atoi((char *) panel_get_value(sample_start_item));
	stepno = atoi((char *) panel_get_value(sample_step_item));
	if (len <= 2) {
	    errwin("Set has <= 2 points");
	    return;
	}
	if (startno < 0) {
	    errwin("Start point < 0");
	    return;
	}
	if (stepno < 0) {
	    errwin("Step < 0");
	    return;
	}
	npts = (len - startno + 1) / stepno;
	if ((len - startno - 1) % stepno) {
	    npts++;
	}
	if (npts < 0) {
	    errwin("Number of points < 0");
	    return;
	}
	activateset(resset);
	setlength(resset, npts);
	x = getx(setno);
	y = gety(setno);
	xres = getx(resset);
	yres = gety(resset);
	for (i = 0; i < npts; i++) {
	    xres[i] = x[startno - 1 + i * stepno];
	    yres[i] = y[startno - 1 + i * stepno];
	}
	sprintf(buf, "Sample, %d, %d set #%d", startno, stepno, setno);
    } else {
	strcpy(exprstr, (char *) panel_get_value(sample_expr_item));
	if (!strlen(exprstr)) {
	    errwin("Enter logical expression first");
	    return;
	}
	xres = (double *) calloc(len, sizeof(double));
	if (xres == NULL) {
	    errwin("Can't calloc xres in sample");
	    return;
	}
	yres = (double *) calloc(len, sizeof(double));
	if (yres == NULL) {
	    errwin("Can't calloc yres in sample");
	    cxfree(xres);
	    return;
	}
	npts = 0;
	x = getx(setno);
	y = gety(setno);
	fixupstr(exprstr);
	for (i = 0; i < len; i++) {
	    scanner(exprstr, &x[i], &y[i], &a, &b, &c, &d, 0, 0, &ier);
	    if (ier) {
		cxfree(xres);
		cxfree(yres);
		killset(resset);
		return;
	    }
	    if ((int) result) {
		xres[npts] = x[i];
		yres[npts] = y[i];
		npts++;
	    }
	}
	if (npts > 0) {
	    activateset(resset);
	    setlength(resset, npts);
	    x = getx(resset);
	    y = gety(resset);
	    for (i = 0; i < npts; i++) {
		x[i] = xres[i];
		y[i] = yres[i];
	    }
	    sprintf(buf, "%d, %s", setno, exprstr);
	} else {
	    errwin("No points from sample");
	    killset(resset);
	}
	cxfree(xres);
	cxfree(yres);
    }
    if (npts > 0) {
	updatesetminmax(resset);
	setcomment(resset, buf);
	update_status(resset);
    }
}

/*
 * service the type cycle for sample
 */
static void do_sample_type_proc()
{
    int itype;

    itype = (int) panel_get_value(sample_type_item);
    panel_set(sample_start_item, PANEL_SHOW_ITEM, !itype, 0);
    panel_set(sample_step_item, PANEL_SHOW_ITEM, !itype, 0);
    panel_set(sample_expr_item, PANEL_SHOW_ITEM, itype, 0);
    panel_paint(sample_start_item, PANEL_CLEAR);
    panel_paint(sample_step_item, PANEL_CLEAR);
    panel_paint(sample_expr_item, PANEL_CLEAR);
}

static void compose_done_proc()
{
    window_set(compose_frame, WIN_SHOW, FALSE, 0);
}

void define_compose_popup()
{
    compose_frame = window_create(main_frame, FRAME,
				  WIN_Y, 50,
				  WIN_FONT, winfont,
				  FRAME_LABEL, "Transformations",
				  FRAME_SHOW_LABEL, TRUE,
			     WIN_ERROR_MSG, "Couldn't create compose_frame",
				  0);
    compose_panel = window_create(compose_frame, PANEL,
				  WIN_FONT, winfont,
				WIN_VERTICAL_SCROLLBAR, scrollbar_create(0),
			      WIN_HORIZONTAL_SCROLLBAR, scrollbar_create(0),
				  0);
    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
	  panel_button_image(compose_panel, "Evaluate formula", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(0),
		      PANEL_NOTIFY_PROC, do_compute_proc,
		      0);
    define_select_set_panel4(compose_panel, compute_set_item, ATTR_COL(1), ATTR_ROW(1));
    compute_load_item = panel_create_item(compose_panel, PANEL_CYCLE,
					  PANEL_ITEM_X, ATTR_COL(1),
					  PANEL_ITEM_Y, ATTR_ROW(2),
					  PANEL_LABEL_STRING, "Result To:",
					  PANEL_CHOICE_STRINGS,
					  "Same set",
					  "New set", 0,
					  0);
    compute_formula_item = panel_create_item(compose_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Formula:",
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(3),
					     PANEL_NOTIFY_STRING, "\n\r",
					 PANEL_NOTIFY_PROC, eotproc_compose,
					 PANEL_VALUE_DISPLAY_LENGTH, 40, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(compose_panel, "Load", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(5),
		      PANEL_NOTIFY_PROC, do_load_proc,
		      0);
    define_select_set_panel4(compose_panel, load_set_item, ATTR_COL(1), ATTR_ROW(6));
    load_to_item = panel_create_item(compose_panel, PANEL_CYCLE,
				     PANEL_ITEM_X, ATTR_COL(1),
				     PANEL_ITEM_Y, ATTR_ROW(7),
				     PANEL_LABEL_STRING, "To:",
				     PANEL_CHOICE_STRINGS,
				     "X",
				     "Y",
				     "Scratch array A",
				     "Scratch array B",
				     "Scratch array C",
				     "Scratch array D", 0,
				     0);
    load_start_item = panel_create_item(compose_panel, PANEL_TEXT,
					PANEL_LABEL_STRING, "Start:",
					PANEL_ITEM_X, ATTR_COL(1),
					PANEL_ITEM_Y, ATTR_ROW(8),
					PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    load_step_item = panel_create_item(compose_panel, PANEL_TEXT,
				       PANEL_ITEM_X, ATTR_COL(1),
				       PANEL_ITEM_Y, ATTR_ROW(9),
				       PANEL_LABEL_STRING, "Step:",
				       PANEL_VALUE_DISPLAY_LENGTH, 15, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
	     panel_button_image(compose_panel, "Load and eval", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(70),
		      PANEL_ITEM_Y, ATTR_ROW(13),
		      PANEL_NOTIFY_PROC, do_compute_proc2,
		      0);
    compute_formulax_item = panel_create_item(compose_panel, PANEL_TEXT,
					      PANEL_ITEM_X, ATTR_COL(70),
					      PANEL_ITEM_Y, ATTR_ROW(14),
					      PANEL_LABEL_STRING, "Fx: ",
					 PANEL_VALUE_DISPLAY_LENGTH, 25, 0);
    compute_formulay_item = panel_create_item(compose_panel, PANEL_TEXT,
					      PANEL_ITEM_X, ATTR_COL(70),
					      PANEL_ITEM_Y, ATTR_ROW(15),
					      PANEL_LABEL_STRING, "Fy: ",
					 PANEL_VALUE_DISPLAY_LENGTH, 25, 0);
    load_to2_item = panel_create_item(compose_panel, PANEL_CYCLE,
				      PANEL_ITEM_X, ATTR_COL(70),
				      PANEL_ITEM_Y, ATTR_ROW(16),
				      PANEL_LABEL_STRING, "To:",
				      PANEL_CHOICE_STRINGS,
				      "X",
				      "Y",
				      "Scratch array A",
				      "Scratch array B",
				      "Scratch array C",
				      "Scratch array D", 0,
				      0);
    load_start2_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_LABEL_STRING, "Start:",
					 PANEL_ITEM_X, ATTR_COL(70),
					 PANEL_ITEM_Y, ATTR_ROW(17),
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    load_stop2_item = panel_create_item(compose_panel, PANEL_TEXT,
					PANEL_ITEM_X, ATTR_COL(70),
					PANEL_ITEM_Y, ATTR_ROW(18),
					PANEL_LABEL_STRING, "Stop:",
					PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    load_npts_item = panel_create_item(compose_panel, PANEL_TEXT,
				       PANEL_ITEM_X, ATTR_COL(70),
				       PANEL_ITEM_Y, ATTR_ROW(19),
				       PANEL_LABEL_STRING, "# of pts:",
				       PANEL_VALUE_DISPLAY_LENGTH, 15, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
	     panel_button_image(compose_panel, "Differentiate", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(30),
		      PANEL_ITEM_Y, ATTR_ROW(5),
		      PANEL_NOTIFY_PROC, do_differ_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_differ_item, ATTR_COL(30), ATTR_ROW(6), "Select set:");
    toggle_differ_type_item = panel_create_item(compose_panel, PANEL_CYCLE,
					      PANEL_LABEL_STRING, "Method:",
						PANEL_CHOICE_STRINGS,
						"Forward difference",
						"Backward difference",
						"Centered difference",
						0,
						PANEL_ITEM_X, ATTR_COL(30),
						PANEL_ITEM_Y, ATTR_ROW(7),
						0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		 panel_button_image(compose_panel, "Integrate", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(30),
		      PANEL_ITEM_Y, ATTR_ROW(9),
		      PANEL_NOTIFY_PROC, do_int_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_int_item, ATTR_COL(30), ATTR_ROW(10), "Select set:");
    toggle_int_type_item = panel_create_item(compose_panel, PANEL_CYCLE,
					     PANEL_LABEL_STRING, "Load:",
					     PANEL_CHOICE_STRINGS,
					     "Cumulative sum",
					     "Sum only",
					     0,
					     PANEL_ITEM_X, ATTR_COL(30),
					     PANEL_ITEM_Y, ATTR_ROW(11),
					     0);
    int_sum_item = panel_create_item(compose_panel, PANEL_TEXT,
				     PANEL_ITEM_X, ATTR_COL(30),
				     PANEL_ITEM_Y, ATTR_ROW(12),
				     PANEL_LABEL_STRING, "Sum:",
				     PANEL_VALUE_DISPLAY_LENGTH, 15, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		    panel_button_image(compose_panel, "X-corr", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(30),
		      PANEL_ITEM_Y, ATTR_ROW(14),
		      PANEL_NOTIFY_PROC, do_xcor_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_xcor1_item, ATTR_COL(30), ATTR_ROW(15), "Select set:");
    define_select_set_panel3(compose_panel, toggle_set_xcor2_item, ATTR_COL(30), ATTR_ROW(16), "Select set:");
    toggle_xcor_type_item = panel_create_item(compose_panel, PANEL_CYCLE,
					      PANEL_LABEL_STRING, "Bias:",
					      PANEL_CHOICE_STRINGS,
					      "Biased estimate",
					      "Unbiased estimate",
					      0,
					      PANEL_ITEM_X, ATTR_COL(30),
					      PANEL_ITEM_Y, ATTR_ROW(17),
					      0);
    xcor_lag_item = panel_create_item(compose_panel, PANEL_TEXT,
				      PANEL_ITEM_X, ATTR_COL(30),
				      PANEL_ITEM_Y, ATTR_ROW(18),
				      PANEL_LABEL_STRING, "Lag:",
				      PANEL_VALUE_DISPLAY_LENGTH, 15, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		    panel_button_image(compose_panel, "Spline", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(30),
		      PANEL_ITEM_Y, ATTR_ROW(20),
		      PANEL_NOTIFY_PROC, do_spline_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_spline_item, ATTR_COL(30), ATTR_ROW(21), "Select set:");
    spline_start_item = panel_create_item(compose_panel, PANEL_TEXT,
					  PANEL_ITEM_X, ATTR_COL(30),
					  PANEL_ITEM_Y, ATTR_ROW(22),
					  PANEL_LABEL_STRING, "Start:",
					  PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    spline_stop_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(30),
					 PANEL_ITEM_Y, ATTR_ROW(23),
					 PANEL_LABEL_STRING, "Stop:",
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    spline_step_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(30),
					 PANEL_ITEM_Y, ATTR_ROW(24),
					 PANEL_LABEL_STRING, "N steps:",
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		    panel_button_image(compose_panel, "Sample", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(30),
		      PANEL_ITEM_Y, ATTR_ROW(26),
		      PANEL_NOTIFY_PROC, do_sample_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_sample_item, ATTR_COL(30), ATTR_ROW(27), "Select set:");
    sample_start_item = panel_create_item(compose_panel, PANEL_TEXT,
					  PANEL_ITEM_X, ATTR_COL(30),
					  PANEL_ITEM_Y, ATTR_ROW(29),
					  PANEL_LABEL_STRING, "Start:",
					  PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    sample_step_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(30),
					 PANEL_ITEM_Y, ATTR_ROW(30),
					 PANEL_LABEL_STRING, "Step:",
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    sample_expr_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(30),
					 PANEL_ITEM_Y, ATTR_ROW(29),
					 PANEL_LABEL_STRING, "Expr:",
					 PANEL_VALUE_DISPLAY_LENGTH, 25, 0);
    sample_type_item = panel_create_item(compose_panel, PANEL_CYCLE,
					 PANEL_ITEM_X, ATTR_COL(30),
					 PANEL_ITEM_Y, ATTR_ROW(28),
					 PANEL_LABEL_STRING, "Type:",
					 PANEL_CHOICE_STRINGS,
					 "Start/step",
					 "Logical expression",
					 0,
				     PANEL_NOTIFY_PROC, do_sample_type_proc,
					 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
	      panel_button_image(compose_panel, "Digital Filt", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(70),
		      PANEL_ITEM_Y, ATTR_ROW(5),
		      PANEL_NOTIFY_PROC, do_digfilter_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_digf1_item, ATTR_COL(70), ATTR_ROW(6), "Select set:");
    define_select_set_panel3(compose_panel, toggle_set_digf2_item, ATTR_COL(70), ATTR_ROW(7), "Weights from set:");

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
	       panel_button_image(compose_panel, "Linear Conv", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(70),
		      PANEL_ITEM_Y, ATTR_ROW(9),
		      PANEL_NOTIFY_PROC, do_linearc_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_linc1_item, ATTR_COL(70), ATTR_ROW(10), "Select set:");
    define_select_set_panel3(compose_panel, toggle_set_linc2_item, ATTR_COL(70), ATTR_ROW(11), "Convolve with set:");

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(compose_panel, "Histo", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(11),
		      PANEL_NOTIFY_PROC, do_histo_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_histo_item, ATTR_COL(1), ATTR_ROW(12), "Select set:");
    histo_binw_item = panel_create_item(compose_panel, PANEL_TEXT,
					PANEL_ITEM_X, ATTR_COL(1),
					PANEL_ITEM_Y, ATTR_ROW(13),
					PANEL_LABEL_STRING, "Bin width:",
					PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    histo_hxmin_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(1),
					 PANEL_ITEM_Y, ATTR_ROW(14),
					 PANEL_LABEL_STRING, "Minimum X:",
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    histo_hxmax_item = panel_create_item(compose_panel, PANEL_TEXT,
					 PANEL_ITEM_X, ATTR_COL(1),
					 PANEL_ITEM_Y, ATTR_ROW(15),
					 PANEL_LABEL_STRING, "Maximum X:",
					 PANEL_VALUE_DISPLAY_LENGTH, 15, 0);
    histo_type_item = panel_create_item(compose_panel, PANEL_CYCLE,
					PANEL_LABEL_STRING, "Load:",
					PANEL_CHOICE_STRINGS,
					"Histogram",
					"Cumulative Histo",
					0,
					PANEL_ITEM_X, ATTR_COL(1),
					PANEL_ITEM_Y, ATTR_ROW(16),
					0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(compose_panel, "DFT", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(18),
		      PANEL_NOTIFY_PROC, do_fourier_proc,
		      0);
    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(compose_panel, "FFT", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(7),
		      PANEL_ITEM_Y, ATTR_ROW(18),
		      PANEL_NOTIFY_PROC, do_fft_proc,
		      0);
    toggle_load_fourier_item = panel_create_item(compose_panel, PANEL_CYCLE,
						 PANEL_LABEL_STRING, "Load:",
						 PANEL_CHOICE_STRINGS,
						 "Spectrum",
						 "Coefficients",
						 0,
						 PANEL_ITEM_X, ATTR_COL(1),
						 PANEL_ITEM_Y, ATTR_ROW(20),
						 0);
    toggle_inv_fourier_item = panel_create_item(compose_panel, PANEL_CYCLE,
					     PANEL_LABEL_STRING, "Perform:",
						PANEL_CHOICE_STRINGS,
						"Transform",
						"Inverse transform",
						0,
						PANEL_ITEM_X, ATTR_COL(1),
						PANEL_ITEM_Y, ATTR_ROW(21),
						0);
    toggle_type_fourier_item = panel_create_item(compose_panel, PANEL_CYCLE,
						 PANEL_LABEL_STRING, "Type:",
						 PANEL_CHOICE_STRINGS,
						 "Real",
						 "Complex",
						 0,
						 PANEL_ITEM_X, ATTR_COL(1),
						 PANEL_ITEM_Y, ATTR_ROW(22),
						 0);
    define_select_set_panel3(compose_panel, toggle_set_fourier_item, ATTR_COL(1), ATTR_ROW(19), "Select set:");

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		   panel_button_image(compose_panel, "Running", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(24),
		      PANEL_NOTIFY_PROC, do_runavg_proc,
		      0);
    toggle_run_item = panel_create_item(compose_panel, PANEL_CYCLE,
					PANEL_CHOICE_STRINGS,
					"Average",
					"Median",
					"Minimum",
					"Maximum",
					"Std. dev.", 0,
					PANEL_ITEM_X, ATTR_COL(11),
					PANEL_ITEM_Y, ATTR_ROW(24),
					0);
    define_select_set_panel3(compose_panel, toggle_set_runavg_item, ATTR_COL(1), ATTR_ROW(25), "Select set:");
    toggle_ravglen_item = panel_create_item(compose_panel, PANEL_TEXT,
					    PANEL_LABEL_STRING, "Length:",
					    PANEL_ITEM_X, ATTR_COL(1),
					    PANEL_ITEM_Y, ATTR_ROW(26),
					 PANEL_VALUE_DISPLAY_LENGTH, 10, 0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		   panel_button_image(compose_panel, "Regress", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(28),
		      PANEL_NOTIFY_PROC, do_regress_proc,
		      0);
    define_select_set_panel3(compose_panel, toggle_set_regress_item, ATTR_COL(1), ATTR_ROW(29), "Select set:");
    toggle_degree_item = panel_create_item(compose_panel, PANEL_CYCLE,
					   PANEL_LABEL_STRING, "Degree:",
					   PANEL_CHOICE_STRINGS,
					   "Linear",
					   "Quadratic",
					   "Cubic",
					   "4th degree",
					   "5th degree", 0,
					   PANEL_ITEM_X, ATTR_COL(1),
					   PANEL_ITEM_Y, ATTR_ROW(30),
					   0);
    toggle_resid_item = panel_create_item(compose_panel, PANEL_CYCLE,
					  PANEL_LABEL_STRING, "Load:",
					  PANEL_CHOICE_STRINGS,
					  "Fitted values",
					  "Residuals", 0,
					  PANEL_ITEM_X, ATTR_COL(1),
					  PANEL_ITEM_Y, ATTR_ROW(31),
					  0);

    panel_create_item(compose_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(compose_panel, "Done", 0, winfont),
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(33),
		      PANEL_NOTIFY_PROC, compose_done_proc,
		      0);
    panel_set_value(compute_formulax_item, "x");
    window_fit(compose_panel);
    window_fit(compose_frame);
/*
* initialize the type item - either start/step or expression
*/
    do_sample_type_proc();
}
