# GPL License and Copyright Notice ============================================
#  This file is part of Wrye Bash.
#
#  Wrye Bash is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  Wrye Bash is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Wrye Bash; if not, write to the Free Software Foundation,
#  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  Wrye Bash copyright (C) 2005, 2006, 2007 Wrye 
#
# =============================================================================

"""This module provides a command line interface for working with Oblivion files 
and environment. Functions are defined here and added to the callables 
singleton. Callables.main() is then used to parse command line arguments.

Practice: 
    While in theory this module is usable by anybody, no one but me (Wrye) has 
    made use of it, so it's become my own miscellaneous toolbin and testing 
    environment. As a result, its contents tend to shift depending on what I am 
    currently working on. However, the functions (init, dumpData) and classes 
    (Callables) at the top are fairly reliably present and unchanging.

Rational Names:
    Rational names is a mod that "Rationalizes" names of many objects in the 
    Oblivion world. Many of those names were generated here by functions working 
    on the basis of the eid or pre-existing name. Naturally these tend to be 
    use-once functions, but they also tended to be modified and then used again.
"""

# Imports ----------------------------------------------------------------------
#--Standard
import os
import re
import string
import struct
import cStringIO
import sys
import types

#--Local
import bosh, bush
from bosh import _, Path

# Basics ----------------------------------------------------------------------
#------------------------------------------------------------------------------
def init(initLevel):
    """Initializes bosh environment to specified level. I.e., it initializes
    certain data arrays which can then be accessed by the function. It's typical 
    to init to level 3, which gives access to all basic data arrays (plugins, 
    modInfos and saveInfos.)

    initLevels:
        0: Settings
        1: plugins
        2: modInfos
        3: saveInfos"""
    #--Settings
    bosh.initDirs()
    bosh.initSettings(safeMode=True)
    bosh.oblivionIni = bosh.OblivionIni()
    #--MwIniFile (initLevel >= 1)
    if initLevel < 1: return
    bosh.oblivionIni = bosh.OblivionIni()
    #--ModInfos (initLevel >= 2)
    if initLevel < 2: return
    bosh.modInfos = bosh.ModInfos()
    bosh.modInfos.refresh()
    #--SaveInfos (initLevel >= 3)
    if initLevel < 3: return
    bosh.saveInfos = bosh.SaveInfos()
    bosh.saveInfos.refresh()
    #--Misc.
    if initLevel < 4: return
    bosh.screensData = bosh.ScreensData()
    bosh.screensData.refresh()

#------------------------------------------------------------------------------
class Callables:
    """A singleton set of objects (typically functions or class instances) that 
    can be called as functions from the command line. 
    
    Functions are called with their arguments, while object instances are called 
    with their method and then their functions. E.g.:
    * bish afunction arg1 arg2 arg3
    * bish anInstance.aMethod arg1 arg2 arg3"""

    #--Ctor
    def __init__(self):
        """Initialization."""
        self.callObjs = {}

    #--Add a callable
    def add(self,callObj,callKey=None):
        """Add a callable object. 
        
        callObj:
            A function or class instance. 
        callKey: 
            Name by which the object will be accessed from the command line. 
            If callKey is not defined, then callObj.__name__ is used."""
        callKey = callKey or callObj.__name__
        self.callObjs[callKey] = callObj

    #--Help
    def help(self,callKey):
        """Print help for specified callKey."""
        help(self.callObjs[callKey])

    #--Main
    def main(self):
        callObjs = self.callObjs
        #--Call key, tail
        callParts  = string.split(sys.argv[1],'.',1)
        callKey    = callParts[0]
        callTail   = (len(callParts) > 1 and callParts[1])
        #--Help request?
        if callKey == '-h':
            help(self)
            return
        #--Not have key?
        if callKey not in callObjs:
            print "Unknown function/object:", callKey
            return
        #--Callable
        callObj = callObjs[callKey]
        if type(callObj) == types.StringType:
            callObj = eval(callObj)
        if callTail:
            callObj = eval('callObj.'+callTail)
        #--Args
        args = sys.argv[2:]
        #--Keywords?
        keywords = {}
        argDex = 0
        reKeyArg  = re.compile(r'^\-(\D\w+)')
        reKeyBool = re.compile(r'^\+(\D\w+)')
        while argDex < len(args):
            arg = args[argDex]
            if reKeyArg.match(arg):
                keyword = reKeyArg.match(arg).group(1)
                value   = args[argDex+1]
                keywords[keyword] = value
                del args[argDex:argDex+2]
            elif reKeyBool.match(arg):
                keyword = reKeyBool.match(arg).group(1)
                keywords[keyword] = 1
                del args[argDex]
            else:
                argDex = argDex + 1
        #--Apply
        apply(callObj,args,keywords)
#--Callables Singleton
callables = Callables()

def mainFunction(func):
    """A function for adding functions to callables. 
    Used as a function decorator ("@mainFunction")."""
    callables.add(func)
    return func

# Common ----------------------------------------------------------------------

#------------------------------------------------------------------------------
@mainFunction
def convertFace(fileName,eid,fromEid,toEid):
    """Converts faces from one race to another."""
    init(3)
    #--Race faces
    raceInfo = bosh.modInfos[Path.get('Oblivion.esm')]
    raceFaces = bosh.PCFaces.mod_getRaceFaces(raceInfo)
    fromRace = raceFaces.get(fromEid, bosh.PCFaces.PCFace())
    toRace   = raceFaces.get(toEid,   bosh.PCFaces.PCFace())
    #--Mod Face
    modInfo = bosh.modInfos[Path.get(fileName)]
    face = bosh.PCFaces.mod_getFaces(modInfo)[eid]
    face.convertRace(fromRace,toRace)
    #--Save back over original face
    loadFactory = bosh.LoadFactory(True,bosh.MreNpc)
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    npc = modFile.NPC_.getRecordByEid(eid)
    bosh.copyattrs(face,npc,('fggs','fgga','fgts'))
    npc.setChanged()
    modFile.safeSave()

#------------------------------------------------------------------------------
@mainFunction
def genHtml(*args,**keywords):
    """Wtxt to html. Just pass through to wtxt function."""
    import wtxt
    wtxt.genHtml(*args,**keywords)

#------------------------------------------------------------------------------
@mainFunction
def importEyes(srcMod,srcRaceEid,dstMod,dstRaceEid):
    """Copies eyes and hair from one race to another."""
    init(3)
    if dstMod.lower() == 'oblivion.esm':
        raise "You don't REALLY want to overwrite Oblvion.esm, do you?"
    srcFactory = bosh.LoadFactory(False,bosh.MreRace)
    dstFactory = bosh.LoadFactory(True,bosh.MreRace)
    srcInfo = bosh.modInfos[Path.get(srcMod)]
    dstInfo = bosh.modInfos[Path.get(dstMod)]
    srcFile = bosh.ModFile(srcInfo,srcFactory)
    dstFile = bosh.ModFile(dstInfo,dstFactory)
    srcFile.load(True)
    dstFile.load(True)
    #--Get source and dest race records
    srcRace = dstRace = None
    for record in srcFile.RACE.records:
        if record.eid == srcRaceEid:
            srcRace = record
            break
    for record in dstFile.RACE.records:
        if record.eid == dstRaceEid:
            dstRace = record
            break
    if not srcRace: raise "Didn't find race (eid) %s in %s." % (srcRaceEid,srcMod)
    if not dstRace: raise "Didn't find race (eid) %s in %s." % (dstRaceEid,dstMod)
    #--Get mapper
    srcMasters = srcFile.tes4.masters[:]
    srcMasters.append(Path.get(srcMod))
    mapper = bosh.MasterMap(srcMasters,dstFile.tes4.masters)
    print mapper.map
    #--XFer eyes, hair
    dstRace.defaultHairColor = srcRace.defaultHairColor
    dstRace.defaultHairMale = mapper(srcRace.defaultHairMale)
    dstRace.defaultHairFemale = mapper(srcRace.defaultHairFemale)
    dstRace.eyes = []
    for entry in srcRace.eyes:
        dstRace.eyes.append(mapper(entry))
    dstRace.hair = []
    for entry in srcRace.hair:
        dstRace.hair.append(mapper(entry))
    dstRace.setChanged()
    #--Save Changes
    dstFile.safeSave()

#------------------------------------------------------------------------------
@mainFunction
def diffScripts(oldFile,newFile):
    """Compares scripts between old and new files and prints scripts which differ 
    from old to new to to two text files which can then be diffed by a diff utility."""
    init(3)
    oldScripts, newScripts = {},{}
    for scripts,fileName in ((oldScripts,oldFile),(newScripts,newFile)):
        loadFactory = bosh.LoadFactory(False,bosh.MreScpt)
        modInfo = bosh.modInfos[Path.get(fileName)]
        modFile = bosh.ModFile(modInfo,loadFactory)
        modFile.load(True)
        scripts.update(dict((record.eid, record.text) for record in modFile.SCPT.records))
    oldDump,newDump = [(Path.get(fileName).root()+'.mws').open('w') for fileName in (oldFile,newFile)]
    for eid in sorted(oldScripts):
        if eid in newScripts and oldScripts[eid] != newScripts[eid]:
            print 'Modified:',eid
            oldDump.write(';;;OLD %s %s\n' %( eid,'='*40))
            newDump.write(';;;NEW %s %s\n' %( eid,'='*40))
            oldDump.write(oldScripts[eid]+'\n\n')
            newDump.write(newScripts[eid]+'\n\n')
    oldDump.close()
    newDump.close()
    newScriptKeys = set(newScripts) - set(oldScripts)

#------------------------------------------------------------------------------
@mainFunction
def scriptVars(fileName=None,printAll=None):
    """Print variables for scripts for specified mod file."""
    init(3)
    loadFactory = bosh.LoadFactory(False,bosh.MreScpt)
    modInfo = bosh.modInfos[Path.get(fileName)]
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    for record in sorted(modFile.SCPT.records,key=lambda a: a.eid):
        indices = [var.index for var in record.vars]
        if printAll or (indices != range(1,len(indices)+1)):
            print '%s:  NRefs: %d Last: %d' % (record.eid, record.numRefs, record.lastIndex)
            refVars = set(record.refVars)
            for var in record.vars:
                print ' ',var.index,var.name,('','[REF]')[var.index in refVars]

# Book Mangling ---------------------------------------------------------------
@mainFunction
def bookExport(fileName=None):
    """Export data from book to text file(s)."""
    fileName = Path.get(fileName)
    init(3)
    #--Data from mod
    modInfo = bosh.modInfos[fileName]
    loadFactory= bosh.LoadFactory(False,bosh.MreBook)
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    data = {}
    texts = {}
    for book in modFile.BOOK.records:
        data[book.eid] = (book.eid,book.full,book.value,len(book.text))
        texts[book.eid] = book.text
    #--Dump book info
    if False:
        textPath = Path.get(fileName.root()+'.csv')
        out = textPath.open('w')
        #out.write('"Edit Id"\t"Name"\t"Value"\t"Text Len"\n')
        for eid in sorted(data):
            out.write('"%s"\t"%s"\t"%d"\t"%d"\n' % data[eid])
        out.close()
    #--Dump Texts
    if False:
        reTag = re.compile('<[^>]>')
        reNewLine = re.compile('\r\n')
        out = Path.get(fileName.root()+'.txt').open('w')
        for eid in sorted(data):
            out.write('#--[%s]  %s\n' % data[eid][:2])
            out.write(reTag.sub('',reNewLine.sub('\n',texts[eid])))
            out.write('\n\n')
    out.close()

@mainFunction
def bookImport(fileName=None):
    """Import data from text file into book."""
    fileName = Path.get(fileName)
    init(3)
    data = {}
    #--Import from book
    textPath = Path.get(fileName.root()+'.csv')
    ins = textPath.open()
    ins.readline() #--Skip first line
    for line in ins:
        line = line.strip()
        if not line or '\t' not in line: return
        (eid,full,value) = line.split('\t')[:3]
        eid = eid[1:-1]
        full = full[1:-1]
        value = int(value)
        #print eid,full,value
        data[eid] = value
    ins.close()
    #--Export to book
    modInfo = bosh.modInfos[fileName]
    loadFactory= bosh.LoadFactory(True,bosh.MreBook)
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    for book in modFile.BOOK.records:
        if book.eid in data:
            print '%-35s %3d %3d' % (book.eid,book.value,data[book.eid])
            book.value = data[book.eid]
            book.setChanged()
        else:
            print book.eid,'NOT----------'
    modFile.safeSave()

# Misc. Utils -----------------------------------------------------------------

#------------------------------------------------------------------------------
@mainFunction
def bsaReport(fileName,printAll='False'):
    printAll = eval(printAll)
    init(2)
    bsaFile = bosh.BsaFile(Path.get(fileName))
    bsaFile.scan()
    bsaFile.report(printAll)

#------------------------------------------------------------------------------
@mainFunction
def getIds(fileName=None):
    """Gets formids and returns as a set. Primarily for analysis of Oblivion.esm.
    NOTE: Does a low level read and hence can read formids of ALL records in all 
    groups. Including CELLs WRLDs, etc."""
    def getRecordReader(ins,flags,size):
        """Decompress record data as needed."""
        if not bosh.MreRecord.flags1(flags).compressed:
            return (ins,ins.tell()+size)
        else:
            import zlib
            sizeCheck, = struct.unpack('I',ins.read(4))
            decomp = zlib.decompress(ins.read(size-4))
            if len(decomp) != sizeCheck: 
                raise ModError(self.inName,
                    _('Mis-sized compressed data. Expected %d, got %d.') % (size,len(decomp)))
            reader = bosh.ModReader(fileName,cStringIO.StringIO(decomp))
            return (reader,sizeCheck)
    init(2)
    modInfo = bosh.modInfos[Path.get(fileName)]
    ins = bosh.ModReader(fileName,modInfo.getPath().open('rb'))
    group_records = {}
    records = group_records['TES4'] = []
    while not ins.atEnd():
        (type,size,str0,formid,uint2) = ins.unpackRecHeader()
        print '>>',type,size,formid
        if type == 'GRUP':
            records = group_records.setdefault(str0,[])
            if str0 in ('CELL','WRLD'):
                ins.seek(size-20,1)
        elif type != 'GRUP':
            eid = ''
            nextRecord = ins.tell() + size
            recs,endRecs = getRecordReader(ins,str0,size)
            while recs.tell() < endRecs:
                (type,size) = recs.unpackSubHeader()
                if type == 'EDID':
                    eid = recs.readString(size)
                    break
                ins.seek(size,1)
            records.append((formid,eid))
            ins.seek(nextRecord)
    ins.close()
    #--Report
    del group_records['TES4']
    for group in sorted(group_records):
        #print
        print group
        for formid,eid in sorted(group_records[group],key = lambda a: a[1].lower()):
            print ' ',bosh.strFormid(formid),eid

#------------------------------------------------------------------------------
@mainFunction
def gmstIds(fileName=None):
    """Updates map of GMST eids to formids in Data\Oblivion_ids.pkl, based either
    on a list of new eids or the gmsts in the specified mod file. Updated pkl file
    is dropped in Mopy directory."""
    #--Data base
    import cPickle
    formids = cPickle.load(Path.get(r'Data\Oblivion_ids.pkl').open('r'))['GMST']
    maxId = max(formids.values())
    maxId = max(maxId,0xf12345)
    maxOld = maxId
    print 'maxId',hex(maxId)
    #--Eid list?
    for eid in ['sCantEquipGeneric']:
        if eid not in formids:
            maxId += 1
            formids[eid] = maxId
            print '%08X  %08X %s' % (0,maxId,eid)
    #--Source file
    if fileName:
        init(3)
        sorter = lambda a: a.eid
        loadFactory = bosh.LoadFactory(False,bosh.MreGmst)
        modInfo = bosh.modInfos[Path.get(fileName)]
        modFile = bosh.ModFile(modInfo,loadFactory)
        modFile.load(True)
        for gmst in sorted(modFile.GMST.records,key=sorter):
            print gmst.eid, gmst.value
            if gmst.eid not in formids:
                maxId += 1
                formids[gmst.eid] = maxId
                print '%08X  %08X %s' % (gmst.formid,maxId,gmst.eid)
    #--Changes?
    if maxId > maxOld:
        outData = {'GMST':formids}
        cPickle.dump(outData,Path.get(r'Oblivion_ids.pkl').open('w'))
        print "%d news gmst ids written to Oblivion_ids.pkl" % ((maxId - maxOld),)

#------------------------------------------------------------------------------
@mainFunction
def modCheck(fileName=None):
    """Reports on various problems with mods."""
    reBadVarName = re.compile('^[_0-9]')
    init(3)
    loadFactory = bosh.LoadFactory(False,bosh.MreWeap)
    for modInfo in bosh.modInfos.data.values():
        print '\n',modInfo.name
        modFile = bosh.ModFile(modInfo,loadFactory)
        modFile.load(True)
        #--Bows with reach == 0 error? (Causes CTD if NPC tries to equip.)
        for record in modFile.WEAP.records:
            if record.weaponType == 5 and record.reach <= 0:
                print ' ',record.eid
        #--Records with poor variable names? (Names likely to cause errors.)
        for record in modFile.SCPT.records:
            badVarNames = []
            for var in record.vars:
                if reBadVarName.match(var.name): badVarNames.append(var.name)
            if badVarNames:
                print ' ',record.eid,badVarNames

#------------------------------------------------------------------------------
@mainFunction
def findSaveRecord(srcName,formid):
    """Finds specified record in save file."""
    init(3)
    srcInfo = bosh.saveInfos[Path.get(srcName)]
    srcFile = bosh.SaveFile(srcInfo)
    srcFile.load()
    #--Get src npc data
    formid = int(formid,16)
    print srcFile.getRecord(formid)

#------------------------------------------------------------------------------
@mainFunction
def renameArchives(root=r'C:\Program Files\Bethesda Softworks\Oblivion\Downloads'):
    """Renames TesSource archive files to sort a little better. 
    E.g., change 12345-2.23-My Wicked Mod-TESSource.zip to My Wicked Mod 2.23.zip."""
    reTesSource = re.compile(r'^\d{4}-(\d[^-]*)-(.+)-TESSource.(zip|rar|7z|ace|exe)$',re.I)
    reTesSourceNV = re.compile(r'^\d{4}-(.+)-TESSource.(zip|rar|7z|ace|exe)$',re.I)
    reTesSource3 = re.compile(r'^(.+)-\d+-TESSource.(zip|rar|7z|ace|exe|esp)$',re.I)
    reBracketNum = re.compile(r'\[1\]')
    for (dirPath,dirNames,fileNames) in os.walk(root):
        dirPath = Path.get(dirPath)
        for name in fileNames:
            path = dirPath.join(name)
            maTesSource = reTesSource.match(name)
            maTesSource3 = reTesSource3.match(name)
            maTesSourceNV = reTesSourceNV.match(name)
            if maTesSource:
                newName = '%s %s.%s' % maTesSource.group(2,1,3)
            elif maTesSourceNV:
                newName = '%s.%s' % maTesSourceNV.group(1,2)
            elif maTesSource3:
                newName = '%s.%s' % maTesSource3.group(1,2)
            else:
                newName = name
            newName = reBracketNum.sub('',newName)
            if newName != name:
                newPath = os.path.join(dirPath,newName)
                print newName
                os.rename(path,newPath)

#------------------------------------------------------------------------------
@mainFunction
def uncontinue():
    """Clears continue settings from settings."""
    init(0)
    settings = bosh.settings
    for key in settings.keys():
        if re.search(r'\.continue$',key):
            print key
            del settings[key]
    settings.save()

@mainFunction
def parseTest(srcName=None,dstName='Wrye Test.esp'):
    init(3)
    testClasses = (bosh.MreQust,bosh.MrePack)
    loadFactory = bosh.LoadFactory(False,*testClasses)
    #--Src file
    srcInfo = bosh.modInfos[Path.get(srcName)]
    srcFile = bosh.ModFile(srcInfo,loadFactory)
    srcFile.load(True)
    #return
    #--Dst file
    loadFactory = bosh.LoadFactory(True,*testClasses)
    dstInfo = bosh.modInfos[Path.get(dstName)]
    dstFile = bosh.ModFile(dstInfo,loadFactory)
    dstFile.convertToLongFormids()
    srcFile.convertToLongFormids()
    #--Save to test file
    for testClass in testClasses:
        type = testClass.type
        print type
        srcBlock = getattr(srcFile,type)
        dstBlock = getattr(dstFile,type)
        for record in srcBlock.records:
            dstBlock.setRecord(record.formid,record.getTypeCopy())
    #--Convert and save
    dstFile.tes4.masters = dstFile.getMastersUsed()
    dstFile.convertToShortFormids()
    dstFile.askSave(True)

@mainFunction
def parseDials(srcName=None,dstName='Wrye Test.esp'):
    init(3)
    testClasses = (bosh.MreDial,bosh.MreInfo)
    loadFactory = bosh.LoadFactory(False,*testClasses)
    #--Src file
    srcInfo = bosh.modInfos[Path.get(srcName)]
    srcFile = bosh.ModFile(srcInfo,loadFactory)
    srcFile.load(True)
    #return
    #--Dst file
    loadFactory = bosh.LoadFactory(True,*testClasses)
    dstInfo = bosh.modInfos[Path.get(dstName)]
    dstFile = bosh.ModFile(dstInfo,loadFactory)
    dstFile.convertToLongFormids()
    srcFile.convertToLongFormids()
    #--Save to test file
    srcBlock = getattr(srcFile,'DIAL')
    dstBlock = getattr(dstFile,'DIAL')
    for index,record in enumerate(srcBlock.records):
        record = record.getTypeCopy()
        dstBlock.setRecord(record.formid,record)
    #--Convert and save
    dstFile.tes4.masters = dstFile.getMastersUsed()
    dstFile.convertToShortFormids()
    dstFile.askSave(True)

# Temp ------------------------------------------------------------------------
"""Very temporary functions."""
#--Temp
@mainFunction
def temp(fileName=None):
    bosh.deprint(1,2,'bob',temp)
    return
    init(3)
    testClasses = (bosh.MreMgef,)
    loadFactory = bosh.LoadFactory(True,*testClasses)
    modInfo = bosh.modInfos[Path.get(fileName)]
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    return
    #--Save to test file
    for testClass in testClasses:
        print testClass.type
        for record in getattr(modFile,testClass.type).records:
            #print record.eid
            if reBarExt.match(record.model.path):
                record.model.path = reBarExt.sub(r'Architecture\\BarabusCrypt',record.model.path)
                print record.eid, record.model.path
                record.setChanged()
    modFile.askSave(True)

#--Record Info
class RecordInfo:
    def __init__(self):
        self.eid = self.full = self.path = self.soundOpen = self.soundClose = None

#--Temp
@mainFunction
def defRecords(fileName=None):
    """Create a bunch of records. Specialized for creating records for Daleth's Upper Class furniture."""
    #--Init
    init(3)
    type_eid_info = {
        'FURN':{},
        'STAT':{},
        'CONT':{},
        'DOOR':{},
        }
    #--This group
    dirs = (
        r'BarabusCrypt',
        r'BarabusCrypt\Exterior',
        r'BarabusCrypt\Exterior\Headstone',
        r'BarabusCrypt\Exterior\Railing',
        r'BarabusCrypt\Interior',
        r'BarabusCrypt\Interior\Corridor',
        )
    rootPath = bosh.dirs['mods'].join('meshes')
    reNif = re.compile(r'^(.+?)\.nif',re.I)
    for dir in dirs:
        for name in sorted(rootPath.join(dir).list()):
            if not reNif.match(name): continue
            info = RecordInfo()
            info.path = dir+'\\'+name
            info.eid = reNif.sub(r'cobBarab\1',name)
            if re.search('Door(Ext|Int)',name):
                type = 'DOOR'
                info.full = "Crypt Door"
            elif re.search('(Coffin0[12])',name):
                type = 'CONT'
                info.full = "Coffin"
            elif re.search('Urn01',name):
                type = 'CONT'
                info.full = "Urn"
            else:
                type = 'STAT'
            type_eid_info[type][info.eid] = info 
            print type, info.eid, info.path
    #return
    #--Add records to mod file
    classes = [bosh.MelRecord.type_class[type] for type in type_eid_info]
    loadFactory = bosh.LoadFactory(True,*classes)
    modInfo = bosh.modInfos[Path.get('Cobl Furn.esp')]
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    for type in sorted(type_eid_info):
        modBlock = getattr(modFile,type)
        baseClass = bosh.MreRecord.type_class[type]
        for eid in sorted(type_eid_info[type]):
            info = type_eid_info[type][eid]
            print info.eid, info.path
            formid = bosh.genFormId(len(modFile.tes4.masters),modFile.tes4.getNextObject())
            record = baseClass((type,0,0,formid,0))
            record.eid = eid
            record.model = record.getDefault('model')
            record.model.path = info.path
            for attr in ('full','soundOpen','soundClose'):
                value = getattr(info,attr)
                if value: setattr(record,attr,value)
            record.setChanged()
            modBlock.setRecord(record.formid,record)
    modFile.askSave(True)

#--Temp
@mainFunction
def temp1(fileName=None):
    init(3)
    saveInfo = bosh.saveInfos[Path.get(fileName)]
    saveFile = bosh.SaveFile(saveInfo)
    saveFile.load()
    def printFace(face):
        print face.pcName
        for attr in ('attributes','skills','iclass',):
            break
            print ' ',attr,getattr(face,attr)
        for entry in face.factions:
            print ' %08X %2i' % entry
        print
    #--Player face
    flags = 0x1|0x2|0x4|0x8|0x10|0x20|0x40
    face = bosh.PCFaces.save_getPlayerFace(saveFile)
    printFace(face)
    #bosh.PCFaces.save_setPlayerFace(saveFile,face,flags)
    #face = bosh.PCFaces.save_getPlayerFace(saveFile)
    #printFace(face)

    #--Created face
    key = 0xff7c1cd5
    #face = bosh.PCFaces.save_getCreatedFaces(saveFile,key)[key]
    #printFace(face)
    bosh.PCFaces.save_setCreatedFace(saveFile,key,face)
    face = bosh.PCFaces.save_getCreatedFaces(saveFile,key)[key]
    printFace(face)

    #--Save changes
    #modFile.askSave(hasChanged)

@mainFunction
def temp2(saveName="Test.ess"):
    init(3)
    #--Oblivion
    fileName = "Oblivion.esm"
    loadFactory = bosh.LoadFactory(False,bosh.MreSpel)
    modInfo = bosh.modInfos[Path.get(fileName)]
    modFile = bosh.ModFile(modInfo,loadFactory)
    modFile.load(True)
    #--Save game
    saveInfo = bosh.saveInfos[Path.get(saveName)]
    saveFile = bosh.SaveFile(saveInfo)
    saveFile.load()
    (formid,recType,recFlags,version,data) = saveFile.getRecord(7)
    npc = bosh.SreNPC(recFlags,data)
    #--Racial specials
    #~~Taking out racial specials because changing it does remove spell effects of abilities.
    #~~I.e., after changing from dark elf to imperial, you'll still have the effects of Dark Elf Resist Fire.
    #raceSpecials = bush.raceSpecials
    #if flags & 0x2 and face.race in raceSpecials: #--Don't change, if unknown race.
    #    allRaceSpecials = listJoin(*raceSpecials.values())
    #    npc.spells = [spell for spell in npc.spells if (saveFile.formids[spell] not in allRaceSpecials)]
    #    for spell in raceSpecials[face.race]:
    #        npc.spells.append(saveFile.getIref(spell))
    #--Gender
    print "Spells:"
    for iref in npc.spells:
        if iref >> 24 == 0xFF: continue
        formid = saveFile.formids[iref]
        mreSpell = modFile.SPEL.getRecord(formid)
        if mreSpell:
            print hex(formid), mreSpell.eid

# Main -------------------------------------------------------------------------
if __name__ == '__main__':
    #--No profile
    if True: 
        callables.main()
    #--Profile
    else:
        import profile,pstats
        profile.run('callables.main()','bishProfile')
        stats = pstats.Stats('bishProfile')
        stats.strip_dirs().sort_stats('time').print_stats('bish')
