# GPL License and Copyright Notice ============================================
#  This file is part of Wrye Bash.
#
#  Wrye Bash is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  Wrye Bash is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Wrye Bash; if not, write to the Free Software Foundation,
#  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  Wrye Bash copyright (C) 2005, 2006, 2007 Wrye 
#
# =============================================================================

"""This module defines provides objects and functions for working with Oblivion
files and environment. It does not provide interface functions which instead 
provided by separate modules: bish for CLI and bash/basher for GUI."""

# Localization ----------------------------------------------------------------
#--Not totally clear on this, but it seems to safest to put locale first...
import locale; locale.setlocale(locale.LC_ALL,'')
#locale.setlocale(locale.LC_ALL,'German')
import time

def formatInteger(value):
    """Convert integer to string formatted to locale."""
    return locale.format('%d',int(value),1)

def formatDate(value):
    """Convert time to string formatted to to locale's default date/time."""
    localtime = time.localtime(value)
    return time.strftime('%c',localtime)

# Imports ---------------------------------------------------------------------
#--Python
import cPickle
import cStringIO
import copy
import math
import os
import re
import shutil
import string
import struct
import sys
import types

#--Local
import bush
from bush import _

# Singletons, Constants -------------------------------------------------------
#--Constants
#..Bit-and this with the formid to get the objectindex.
oiMask = 0xFFFFFFL

#--File Singletons
oblivionIni = None
modInfos  = None  #--ModInfos singleton
saveInfos = None #--SaveInfos singleton
screensData = None #--ScreensData singleton
messages = None #--Message archive singleton

#--Settings
dirs = {} #--app, user, mods, saves, userApp
settings  = None

#--Default settings
settingDefaults = {
    'bosh.modInfos.resetMTimes':True,
    }

# Errors ----------------------------------------------------------------------
class BoshError(Exception):
    """Generic Error"""
    def __init__(self,message):
        self.message = message
    def __str__(self):
        return self.message

#------------------------------------------------------------------------------
class AbstractError(BoshError): 
    """Coding Error: Abstract code section called."""
    def __init__(self,message=_('Abstract section called.')):
        BoshError.__init__(self,message)

#------------------------------------------------------------------------------
class ArgumentError(BoshError):
    """Coding Error: Argument out of allowed range of values."""
    pass

#------------------------------------------------------------------------------
class StateError(BoshError):
    """Error: Object is corrupted."""
    pass

#------------------------------------------------------------------------------
class UncodedError(BoshError): 
    """Coding Error: Call to section of code that hasn't been written."""
    def __init__(self,message=_('Section is not coded yet.')):
        BoshError.__init__(self,message)

#------------------------------------------------------------------------------
class FileError(BoshError):
    """TES4/Tes4SaveFile Error: File is corrupted."""
    def __init__(self,inName,message):
        BoshError.__init__(self,message)
        self.inName = inName

    def __str__(self):
        if self.inName:
            return self.inName+': '+self.message
        else:
            return _('Unknown File: ')+self.message


# Util Classes ----------------------------------------------------------------
#------------------------------------------------------------------------------
class CsvReader:
    """For reading csv files."""
    def __init__(self,path):
        import csv
        self.ins = path.open('rb')
        format = ('excel','excel-tab')['\t' in self.ins.readline()]
        self.ins.seek(0)
        self.reader = csv.reader(self.ins,format)

    def __iter__(self):
        return self

    def next(self):
        return self.reader.next()

    def close(self):
        self.reader = None
        self.ins.close()

#------------------------------------------------------------------------------
class Flags(object):
    """Represents a flag field."""

    @staticmethod
    def getNames(*names):
        """Returns dictionary mapping names to indices.
        Names are either strings or (name,index) tuples.
        E.g., Flags.getNames('isQuest','isHidden',None,(4,'isDark'),(7,'hasWater'))"""
        namesDict = {}
        for index,name in enumerate(names):
            if isinstance(name,tuple):
                namesDict[name[1]] = name[0]
            elif name: #--skip if "name" is 0 or None
                namesDict[name] = index
        return namesDict

    #--Generation
    def __init__(self,value=0,names=None):
        """Initialize. Attrs, if present, is mapping of attribute names to indices. See getAttrs()"""
        self.__dict__['_field'] = value | 0L
        self.__dict__['_names'] = names or {}
    
    def __call__(self,newValue=None):
        """Retuns a clone of self, optionally with new value."""
        newFlags = copy.copy(self)
        if newValue != None: newFlags.__dict__['_field'] = newValue | 0L
        return newFlags

    #--As hex string
    def hex(self):
        """Returns hex string of value."""
        return '%08X' % (self._field,)
    def dump(self):
        """Returns value for packing"""
        return self._field

    #--As int
    def __int__(self):
        """Return as integer value for saving."""
        return self._field

    #--As list
    def __getitem__(self, index):
        """Get value by index. E.g., flags[3]"""
        return (self._field >> index) & 1 and True or False
    
    def __setitem__(self,index,value):
        """Set value by index. E.g., flags[3] = True"""
        value = ((value or 0L) and 1L) << index
        mask = 1L << index
        self._field = ((self._field & ~mask) | value)
    
    #--As class
    def __getattr__(self,name):
        """Get value by flag name. E.g. flags.isQuestItem"""
        try:
            names = self.__dict__['_names']
            index = names[name]
            return (self.__dict__['_field'] >> index) & 1 == 1
        except KeyError:
            raise AttributeError

    def __setattr__(self,name,value):
        """Set value by flag name. E.g., flags.isQuestItem = False"""
        if name in ('_field','_names'):
            self.__dict__[name] = value
        else:
            self.__setitem__(self._names[name],value)

    #--Native operations
    def __and__(self,other):
        """Bitwise and."""
        if isinstance(other,Flags): other = other._field
        return self(self._field & other)

    def __invert__(self):
        """Bitwise inversion."""
        return self(~self._field)

    def __or__(self,other):
        """Bitwise or."""
        if isinstance(other,Flags): other = other._field
        return self(self._field | other)

    def __xor__(self,other):
        """Bitwise exclusive or."""
        if isinstance(other,Flags): other = other._field
        return self(self._field ^ other)

    def getTrueAttrs(self):
        """Returns attributes that are true."""
        trueNames = [name for name in self._names if getattr(self,name)]
        trueNames.sort(key = lambda xxx: self._names[xxx])
        return tuple(trueNames)

#------------------------------------------------------------------------------
class Path(str):
    name_path = {} #--Dictionary of paths
    mtimeResets = [] #--Used by getmtime

    @staticmethod
    def getcwd():
        return Path(os.getcwd())

    @staticmethod
    def get(pathName):
        """Gets path as a singleton."""
        if isinstance(pathName,Path):
            pathName = pathName._path
        path = Path.name_path.get(pathName)
        if path: 
            return path
        else:
            return Path.name_path.setdefault(pathName,Path(pathName))

    @staticmethod
    def set(pathName):
        """Stores path."""
        if isinstance(pathName,Path):
            path = pathName
            pathName = path._pathLC
        else:
            path = Path(pathName)
        Path.name_path[pathName] = path
        return path

    #--Instance stuff --------------------------------------------------
    """A file path. May be just a directory, filename or full path."""
    def __init__(self, path):
        """Initialize."""
        if isinstance(path,Path): path = path._path
        self._path = path
        self._pathLC = path.lower()
        self._pathNormLC = os.path.normpath(path).lower()
        str.__init__(self,path)

    def __getstate__(self):
        """Used by picker. State is determined by underlying string, so return psempty tuple."""
        return (0,) #--Pseudo empty. If tuple were actually empty, then setstate wouldn't be run.

    def __setstate__(self,state):
        """Used by unpicker. Ignore state and reset from value of underlying string."""
        path = str(self)
        self._path = path
        self._pathLC = path.lower()
        self._pathNormLC = os.path.normpath(path).lower()

    def __repr__(self):
        return "Path(%s)" % str.__repr__(self)

    def open(self,*args):
        return open(self._path,*args)

    #--Path stuff -------------------------------------------------------
    #--New Paths, subpaths
    def __add__(self,other):
        if isinstance(other,Path): other = other._path
        return Path.get(self._path+other)
    def ext(self):
        """For joe\bob.ext, returns '.ext'."""
        return Path.get(os.path.splitext(self._path)[1])
    def head(self):
        """For joe\bob.ext, returns Path('joe')."""
        return Path.get(os.path.split(self._path)[0])
    def join(*args):
        return Path.get(os.path.join(*args))
    def list(self):
        """For directory: Returns list of files."""
        if not os.path.exists(self._path): return []
        return [Path.get(name) for name in os.listdir(self._path)]
    def normpath(self):
        return os.path.normpath(self) #--NOT lowercase!
    def root(self):
        """For joe\bob.ext, returns Path('joe\bob')."""
        return Path.get(os.path.splitext(self._path)[0])
    def splitext(self):
        return map(Path.get,os.path.splitext(self._path))
    def split(self):
        return map(Path.get,os.path.split(self._path))
    def tail(self):
        """For joe\bob.ext, returns Path('bob.ext')."""
        return Path.get(os.path.split(self._path)[1])

    #--File system info
    def exists(self):
        return os.path.exists(self._path)
    def isdir(self):
        return os.path.isdir(self._path)
    def isfile(self):
        return os.path.isfile(self._path)
    def getsize(self):
        return os.path.getsize(self._path)
    def getctime(self):
        return os.path.getctime(self._path)
    def samefile(self,path2):
        return os.path.samefile(self,path2)

    def getmtime(self,isModFile=False):
        """Returns mtime for path. But if mtime is outside of epoch, then resets 
        mtime to an in-epoch date and uses that."""
        mtime = int(os.path.getmtime(self))
        #--Y2038 bug? (os.path.getmtime() can't handle years over unix epoch)
        if mtime <= 0:
            import random
            #--Kludge mtime to a random time within 10 days of 1/1/2037
            mtime = time.mktime((2037,1,1,0,0,0,3,1,0))
            mtime += random.randint(0,10*24*60*60) #--10 days in seconds
            self.setmtime(mtime)
            Path.mtimeResets.append(self.tail())
        return mtime

    #--File system manipulation
    def chdir(self):
        os.chdir(self)
    def copyfile(self,destPath):
        shutil.copyfile(self,destPath)
    def makedirs(self):
        if not self.exists(): os.makedirs(self)
    def mkdir(self):
        if not self.exists(): os.mkdir(self)
    def move(self,newPath):
        shutil.move(self,newPath)
    def remove(self):
        if self.exists(): os.remove(self)
    def rename(self,newPath):
        os.rename(self._path,newPath)
    def replace(self,newPath):
        Path.get(newPath).remove()
        self.rename(newPath)
    def rmtree(self):
        """WARNING! Removes all directories and files under specified directory!"""
        #--A few sanity checks
        if not self.isdir(): 
            raise BoshError(self._path + _(' is not a directory.'))
        if 'Oblivion' not in self: 
            raise bosh.BoshError(self._path + _(' is not safe to remove.'))
        shutil.rmtree(self)
    def setmtime(self,mtime):
        os.utime(self,(time.time(),mtime))

    #--String stuff -----------------------------------------------------
    def __eq__(self, other):
        if None == other:
            return False
        elif isinstance(other,Path): 
            return self._pathNormLC == other._pathNormLC
        else:
            return self._pathNormLC == os.path.normpath(other.lower())
    def __lt__(self, other):
        return self._pathLC < (other and other.lower())
    def __le__(self, other):
        return self._pathLC <= (other and other.lower())
    def __gt__(self, other):
        return self._pathLC > (other and other.lower())
    def __ne__(self, other):
        return self._pathLC != (other and other.lower())
    def __ge__(self, other):
        return self._pathLC >= (other and other.lower())
    def __cmp__(self, other):
        return cmp(self._pathLC, (other and other.lower()))
    def __hash__(self):
        return hash(self._pathLC)
    def __contains__(self, other):
        return other.lower() in self._pathLC
    def count(self, other, *args):
        return str.count(self._pathLC, other.lower(), *args)
    def endswith(self, other, *args):
        return str.endswith(self._pathLC, other.lower(), *args)
    def find(self, other, *args):
        return str.find(self._pathLC, other.lower(), *args)
    def index(self, other, *args):
        return str.index(self._pathLC, other.lower(), *args)
    def lower(self):
        return Path(self._pathLC)
    def rfind(self, other, *args):
        return str.rfind(self._pathLC, other.lower(), *args)
    def rindex(self, other, *args):
        return str.rindex(self._pathLC, other.lower(), *args)
    def startswith(self, other, *args):
        return str.startswith(self._pathLC, other.lower(), *args)

#------------------------------------------------------------------------------
class DataDict:
    """Mixin class that handles dictionary emulation, assuming that dictionary is is 'data' attribute."""

    def __contains__(self,key):
        return key in self.data
    def __getitem__(self,key):
        return self.data[key]
    def __setitem__(self,key,value):
        self.data[key] = value
    def __delitem__(self,key):
        del self.data[key]
    def setdefault(self,key,default):
        return self.data.setdefault(key,value)
    def keys(self):
        return self.data.keys()
    def values(self):
        return self.data.values()
    def items(self):
        return self.data.items()
    def has_key(self,key):
        return self.data.has_key(key)
    def get(self,key,default=None):
        return self.data.get(key,default)

#------------------------------------------------------------------------------
class Settings(DataDict):
    """Settings/configuration dictionary with persistent storage. 
    
    Default setting for configurations are either set in bulk (by the 
    loadDefaults function) or are set as needed in the code (e.g., various 
    auto-continue settings for bash. Only settings that have been changed from 
    the default values are saved in persistent storage.

    Directly setting a value in the dictionary will mark it as changed (and thus 
    to be archived). However, an indirect change (e.g., to a value that is a 
    list) must be manually marked as changed by using the setChanged method."""
    
    def __init__(self,path,oldPath=None,safeMode=False):
        """Initialize. Read settings from pickle file.
        'oldPath' is a hack to recognize older ascii files in older locations."""
        self.safeMode = safeMode
        self.path = path 
        self.changed = []
        self.deleted = []
        self.data = {}
        #--Load
        if self.path.exists():
            try:
                ins = self.path.open('rb')
                inData = cPickle.load(ins)
                self.data.update(inData)
                ins.close()
            #--File is corrupted
            except EOFError:
                self.path.remove()
                try:
                    bakPath = self.path + '.bak'
                    if not bakPath.exists(): return
                    ins = bakPath.open('rb')
                    inData = cPickle.load(ins)
                    self.data.update(inData)
                    ins.close()
                    bakPath.copyfile(self.path)
                #--No backup or backup is corrupted.
                except EOFError:
                    return
        elif oldPath and oldPath.exists():
            ins = oldPath.open('r')
            inData = cPickle.load(ins)
            self.data.update(inData)
            ins.close()
            #--Save in new location
            cPickle.dump(inData,self.path.open('wb'),-1)

    def loadDefaults(self,defaults):
        """Add default settings to dictionary. Will not replace values that are already set."""
        for key in defaults.keys():
            if key not in self.data:
                self.data[key] = defaults[key]

    def save(self):
        """Save to pickle file. Only key/values marked as changed are saved."""
        filePath = self.path
        if self.safeMode:
            raise BoshError(_("SafeMode is active."))
        #--Data file exists?
        if filePath.exists():
            ins = filePath.open('rb')
            outData = cPickle.load(ins)
            ins.close()
            #--Delete some data?
            for key in self.deleted:
                if key in outData:
                    del outData[key]
        else:
            outData = {}
        #--Write touched data
        for key in self.changed:
            outData[key] = self.data[key]
        #--Pickle it
        tempPath = filePath+'.tmp'
        tempFile = tempPath.open('wb')
        cPickle.dump(outData,tempFile,-1)
        tempFile.close()
        renameFile(tempPath,filePath,True)

    def setChanged(self,key):
        """Marks given key as having been changed. Use if value is a dictionary, list or other object."""
        if key not in self.data:
            raise ArgumentError("No settings data for "+key)
        if key not in self.changed:
            self.changed.append(key)

    def getChanged(self,key):
        """Gets and marks as changed."""
        self.setChanged(key)
        return self.data[key]

    #--Dictionary Emulation
    def __setitem__(self,key,value):
        """Dictionary emulation. Marks key as changed."""
        if key in self.deleted: self.deleted.remove(key)
        if key not in self.changed: self.changed.append(key)
        self.data[key] = value

    def __delitem__(self,key):
        """Dictionary emulation. Marks key as deleted."""
        if key in self.changed: self.changed.remove(key)
        if key not in self.deleted: self.deleted.append(key)
        del self.data[key]

    def setdefault(self,key,value):
        """Dictionary emulation. If new key, marks key as changed."""
        if key in self.data:
            return self.data[key]
        if key in self.deleted: self.deleted.remove(key)
        if key not in self.changed: self.changed.append(key)
        self.data[key] = value
        return value

#------------------------------------------------------------------------------
class StructFile(file):
    """File reader/writer with extra functions for handling structured data."""
    def unpack(self,format,size):
        """Reads and unpacks according to format."""
        return struct.unpack(format,self.read(size))

    def pack(self,format,*data):
        """Packs data according to format."""
        self.write(struct.pack(format,*data))

#------------------------------------------------------------------------------
class TableColumn:
    """Table accessor that presents table column as a dictionary."""
    def __init__(self,table,column):
        self.table = table
        self.column = column
    #--Dictionary Emulation
    def keys(self):
        """Dictionary emulation."""
        tableData = self.table.data
        column = self.column
        return [key for key in tableData.keys() if (column in tableData[key])]
    def has_key(self,key):
        """Dictionary emulation."""
        return self.__contains__(key)
    def clear(self):
        """Dictionary emulation."""
        self.table.delColumn(self.column)
    def get(self,key,default=None):
        """Dictionary emulation."""
        return self.table.getItem(key,self.column,default)
    def __contains__(self,key):
        """Dictionary emulation."""
        tableData = self.table.data
        return tableData.has_key(key) and tableData[key].has_key(self.column)
    def __getitem__(self,key):
        """Dictionary emulation."""
        return self.table.data[key][self.column]
    def __setitem__(self,key,value):
        """Dictionary emulation. Marks key as changed."""
        self.table.setItem(key,self.column,value)
    def __delitem__(self,key):
        """Dictionary emulation. Marks key as deleted."""
        self.table.delItem(key,self.column)

#------------------------------------------------------------------------------
class Table:
    """Simple data table of rows and columns, saved in a pickle file. It is 
    currently used by modInfos to represent properties associated with modfiles, 
    where each modfile is a row, and each property (e.g. modified date or 
    'mtime') is a column.
    
    The "table" is actually a dictionary of dictionaries. E.g. 
        propValue = table['fileName']['propName']
    Rows are the first index ('fileName') and columns are the second index 
    ('propName')."""

    def __init__(self,path,oldPath=None):
        """Intialize and read data from file, if available.
        'oldPath' is a hack to recognize older ascii files in older locations."""
        self.path = Path.get(path) #--Path
        self.data = {}
        self.hasChanged = False
        #--Load
        if self.path.exists():
            try:
                ins = self.path.open('rb')
                inData = cPickle.load(ins)
                self.data.update(inData)
            #--File is corrupted
            except EOFError:
                self.path.remove()
                try:
                    bakPath = self.path + '.bak'
                    if not bakPath.exists(): return
                    ins = bakPath.open('rb')
                    inData = cPickle.load(ins)
                    self.data.update(inData)
                    ins.close()
                    bakPath.copyfile(self.path)
                #--No backup or backup is corrupted.
                except EOFError:
                    return
        elif oldPath and oldPath.exists():
            ins = oldPath.open('r')
            inData = cPickle.load(ins)
            self.data.update(inData)
            #--Save as bin file in new location
            cPickle.dump(inData,path.open('wb'),-1)

    def save(self):
        """Saves to pickle file."""
        if self.hasChanged:
            filePath = self.path
            tempPath = filePath+'.tmp'
            fileDir = filePath.head()
            fileDir.makedirs()
            tempFile = tempPath.open('wb')
            cPickle.dump(self.data,tempFile,-1)
            tempFile.close()
            renameFile(tempPath,filePath,True)
            self.hasChanged = False

    def getItem(self,row,column,default=None):
        """Get item from row, column. Return default if row,column doesn't exist."""
        data = self.data
        if row in data and column in data[row]:
            return data[row][column]
        else:
            return default

    def getColumn(self,column):
        """Returns a data accessor for column."""
        return TableColumn(self,column)

    def setItem(self,row,column,value):
        """Set value for row, column."""
        data = self.data
        if row not in data:
            data[row] = {}
        data[row][column] = value
        self.hasChanged = True

    def setItemDefault(self,row,column,value):
        """Set value for row, column."""
        data = self.data
        if row not in data:
            data[row] = {}
        self.hasChanged = True
        return data[row].setdefault(column,value)

    def delItem(self,row,column):
        """Deletes item in row, column."""
        data = self.data
        if row in data and column in data[row]:
            del data[row][column]
            self.hasChanged = True

    def delRow(self,row):
        """Deletes row."""
        data = self.data
        if row in data:
            del data[row]
            self.hasChanged = True

    def delColumn(self,column):
        """Deletes column of data."""
        data = self.data
        for rowData in data.values():
            if column in rowData:
                del rowData[column]
                self.hasChanged = True

    def moveRow(self,oldRow,newRow):
        """Renames a row of data."""
        data = self.data
        if oldRow in data:
            data[newRow] = data[oldRow]
            del data[oldRow]
            self.hasChanged = True

    def copyRow(self,oldRow,newRow):
        """Copies a row of data."""
        data = self.data
        if oldRow in data:
            data[newRow] = data[oldRow].copy()
            self.hasChanged = True

# Util Constants --------------------------------------------------------------
#--Null strings (for default empty byte arrays)
null1 = '\x00'
null2 = null1*2
null3 = null1*3
null4 = null1*4

#--Unix new lines
reUnixNewLine = re.compile(r'(?<!\r)\n')

#--Version number in tes4.hedr
reVersion = re.compile(r'^(Version:?) *([-0-9a-zA-Z\.]*\+?)',re.M)

#--Mod Extensions
reComment = re.compile('#.*')
reExGroup = re.compile('(.*?),')
reImageExt = re.compile(r'\.(gif|jpg|bmp|png)$',re.I)
reModExt  = re.compile(r'\.es[mp]$',re.I)
reEsmExt  = re.compile(r'\.esm$',re.I)
reEspExt  = re.compile(r'\.esp$',re.I)
reSaveExt = re.compile(r'(quicksave(\.bak)+|autosave(\.bak)+|\.es[rs])$',re.I)
reCsvExt  = re.compile(r'\.csv$',re.I)
reQuoted  = re.compile(r'^"(.*)"$')

# Util Functions --------------------------------------------------------------
#------------------------------------------------------------------------------
def copyattrs(source,dest,attrs):
    """Copies specified attrbutes from source object to dest object."""
    for attr in attrs:
        setattr(dest,attr,getattr(source,attr))

def cstrip(inString):
    """Convert c-string (null-terminated string) to python string."""
    zeroDex = inString.find('\x00')
    if zeroDex == -1:
        return inString
    else:
        return inString[:zeroDex]

def csvFormat(format):
    """Returns csv format for specified structure format."""
    csvFormat = ''
    for char in format:
        if char in 'bBhHiIlLqQ': csvFormat += ',%d'
        elif char in 'fd': csvFormat += ',%f'
        elif char in 's': csvFormat += ',"%s"'
    return csvFormat[1:] #--Chop leading comma

deprintOn = False
def deprint(*args):
    """Prints message along with file and line location."""
    if not deprintOn: return
    import inspect
    stack = inspect.stack()
    file,line,function = stack[1][1:4]
    print '%s %4d %s: %s' % (Path(file).tail(),line,function,' '.join(map(str,args)))

def delist(header,items):
    """Prints list as header plus items."""
    if not deprintOn: return
    import inspect
    stack = inspect.stack()
    file,line,function = stack[1][1:4]
    print '%s %4d %s: %s' % (Path(file).tail(),line,function,str(header))
    if items == None: 
        print '> None'
    else:
        for indexItem in enumerate(items): print '>%2d: %s' % indexItem

def dictFromLines(lines,sep=None):
    """Generate a dictionary from a string with lines, stripping comments and skipping empty strings."""
    temp = [reComment.sub('',x).strip() for x in lines.split('\n')]
    if sep == None or type(sep) == type(''):
        temp = dict([x.split(sep,1) for x in temp if x])
    else: #--Assume re object.
        temp = dict([sep.split(x,1) for x in temp if x])
    return temp

def getMatch(reMatch,group=0):
    """Returns the match or an empty string."""
    if reMatch: return reMatch.group(group)
    else: return ''

def intArg(arg,default=None):
    """Returns argument as an integer. If argument is a string, then it converts it using int(arg,0)."""
    if arg == None: return default
    elif isinstance(arg,types.StringType): return int(arg,0)
    else: return int(arg)

def invertDict(indict):
    """Invert a dictionary."""
    return dict([(y,x) for x,y in indict.items()])

def listFromLines(lines):
    """Generate a list from a string with lines, stripping comments and skipping empty strings."""
    temp = [reComment.sub('',x).strip() for x in lines.split('\n')]
    temp = [x for x in temp if x]
    return temp

def listSubtract(alist,blist):
    """Return a copy of first list minus items in second list."""
    result = []
    for item in alist:
        if item not in blist:
            result.append(item)
    return result

def listJoin(*inLists):
    """Joins multiple lists into a single list."""
    outList = []
    for inList in inLists:
        outList.extend(inList)
    return outList

def renameFile(oldPath,newPath,makeBack=False):
    """Moves file from oldPath to newPath. If newPath already exists then it 
    will either be moved to newPath.bak or deleted depending on makeBack."""
    if newPath.exists(): 
        if makeBack:
            backPath = newPath+'.bak'
            backPath.remove()
            newPath.rename(backPath)
        else:
            newPath.remove()
    oldPath.rename(newPath)

def listGroup(items):
    """Joins items into a list for use in a regular expression.
    E.g., a list of ('alpha','beta') becomes '(alpha|beta)'"""
    return '('+('|'.join(items))+')'

def rgbString(red,green,blue):
    """Converts red, green blue ints to rgb string."""
    return chr(red)+chr(green)+chr(blue)

def rgbTuple(rgb):
    """Converts red, green, blue string to tuple."""
    return struct.unpack('BBB',rgb)

def unQuote(inString):
    """Removes surrounding quotes from string."""
    if len(inString) >= 2 and inString[0] == '"' and inString[-1] == '"':
        return inString[1:-1]
    else:
        return inString

def winNewLines(inString):
    """Converts unix newlines to windows newlines."""
    return reUnixNewLine.sub('\r\n',inString)

# Reference (FormId)
def strFormid(formid):
    """Returns a string representation of the formid."""
    if isinstance(formid,tuple):
        return '(%s,0x%06X)' % formid
    else:
        return '%08X' % (formid,)

def genFormId(modIndex,objectIndex):
    """Generates formid from modIndex and ObjectIndex."""
    return long(objectIndex) | (long(modIndex) << 24 )

def getModIndex(formid):
    """Return the modIndex portion of a formid."""
    return int(formid >> 24)

def getObjectIndex(formid):
    """Return the objectIndex portion of a formid."""
    return int(formid & 0xFFFFFFL)

def getFormIndices(formid):
    """Returns tuple of modindex and objectindex of formid."""
    return (int(formid >> 24),int(formid & 0xFFFFFFL))

# Sorting functions
def byName(objects):
    """Sort by name attribute."""
    return sorted(objects,key=lambda a: a.name)

def byEid(records):
    """Sort by eid attribute."""
    return sorted(records,key=lambda a: a.eid)

# Log/Progress ----------------------------------------------------------------
#------------------------------------------------------------------------------
class Log:
    """Log Callable. This is the abstract/null version. Useful version should 
    override write functions.
    
    Log is divided into sections with headers. Header text is assigned (through 
    setHeader), but isn't written until a message is written under it. I.e., 
    if no message are written under a given header, then the header itself is 
    never written."""

    def __init__(self):
        """Initialize."""
        self.header = None
        self.prevHeader = None

    def setHeader(self,header,writeNow=False,doFooter=True):
        """Sets the header."""
        self.header = header
        self.doFooter = doFooter
        if writeNow: self()

    def __call__(self,message=None):
        """Callable. Writes message, and if necessary, header and footer."""
        if self.header != self.prevHeader:
            if self.prevHeader and self.doFooter:
                self.writeFooter()
            if self.header:
                self.writeHeader(self.header)
            self.prevHeader = self.header
        if message: self.writeMessage(message)

    #--Abstract/null writing functions...
    def writeHeader(self,header):
        """Write header. Abstract/null version."""
        pass
    def writeFooter(self):
        """Write mess. Abstract/null version."""
        pass
    def writeMessage(self,message):
        """Write message to log. Abstract/null version."""
        pass

#------------------------------------------------------------------------------
class LogFile(Log):
    """Log that writes messages to file."""
    def __init__(self,out):
        self.out = out
        Log.__init__(self)

    def writeHeader(self,header):
        self.out.write(header+'\n')

    def writeFooter(self):
        self.out.write('\n')

    def writeMessage(self,message):
        self.out.write(message+'\n')

#------------------------------------------------------------------------------
class Progress:
    """Progress Callable: Shows progress when called."""
    def __init__(self,full=1.0):
        if (1.0*full) == 0: raise ArgumentError('Full must be non-zero!')
        self.message = ''
        self.full = full
        self.state = 0
        self.debug = False

    def setFull(self,full):
        """Set's full and for convenience, returns self."""
        if (1.0*full) == 0: raise ArgumentError('Full must be non-zero!')
        self.full = full
        return self

    def plus(self,increment=1):
        """Increments progress by 1."""
        self.__call__(self.state+increment)

    def __call__(self,state,message=''):
        """Update progress with current state. Progress is state/full."""
        if (1.0*self.full) == 0: raise ArgumentError('Full must be non-zero!')
        if message: self.message = message
        if self.debug:
            print '%0.3f %s' % (1.0*state/self.full, self.message)
        self.doProgress(1.0*state/self.full, self.message)
        self.state = state

    def doProgress(self,progress,message):
        """Default doProgress does nothing."""
        pass

#------------------------------------------------------------------------------
class SubProgress(Progress):
    """Sub progress goes from base to ceiling."""
    def __init__(self,parent,baseFrom=0.0,baseTo='+1',full=1.0,silent=False):
        """For creating a subprogress of another progress meter.
        progress: parent (base) progress meter
        baseFrom: Base progress when this progress == 0.
        baseTo: Base progress when this progress == full
          Usually a number. But string '+1' sets it to baseFrom + 1
        full: Full meter by this progress' scale."""
        Progress.__init__(self,full)
        if baseTo == '+1': baseTo = baseFrom + 1
        if (baseFrom < 0 or baseFrom >= baseTo):
            raise ArgumentError('BaseFrom must be >= 0 and BaseTo must be > BaseFrom')
        self.parent = parent
        self.baseFrom = baseFrom
        self.scale = 1.0*(baseTo-baseFrom)
        self.silent = silent

    def __call__(self,state,message=''):
        """Update progress with current state. Progress is state/full."""
        if self.silent: message = ''
        self.parent(self.baseFrom+self.scale*state/self.full,message)
        self.state = state

#------------------------------------------------------------------------------
class ProgressFile(Progress):
    """Prints progress to file (stdout by default)."""
    def __init__(self,full=1.0,out=None):
        Progress.__init__(self,full)
        self.out = out or sys.stdout

    def doProgress(self,progress,message):
        self.out.write('%0.2f %s\n' % (progress,message))

# Mod I/O --------------------------------------------------------------------
#------------------------------------------------------------------------------
class ModError(FileError):
    """Mod Error: File is corrupted."""
    pass

#------------------------------------------------------------------------------
class ModReadError(ModError):
    """TES4 Error: Attempt to read outside of buffer."""
    def __init__(self,inName,recType,tryPos,maxPos):
        self.recType = recType
        self.tryPos = tryPos
        self.maxPos = maxPos
        if tryPos < 0:
            message = (_('%s: Attempted to read before (%d) beginning of file/buffer.')
                % (recType,tryPos))
        else:
            message = (_('%s: Attempted to read past (%d) end (%d) of file/buffer.') %
                (recType,tryPos,maxPos))
        ModError.__init__(self,inName,message)

#------------------------------------------------------------------------------
class ModSizeError(ModError):
    """TES4 Error: Record/subrecord has wrong size."""
    def __init__(self,inName,recType,readSize,maxSize,exactSize=True):
        self.recType = recType
        self.readSize = readSize
        self.maxSize = maxSize
        self.exactSize = exactSize
        if exactSize:
            messageForm = _('%s: Expected size == %d, but got: %d ')
        else:
            messageForm = _('%s: Expected size <= %d, but got: %d ')
        ModError.__init__(self,inName,messageForm % (recType,readSize,maxSize))


#------------------------------------------------------------------------------
class ModUnknownSubrecord(ModError):
    """TES4 Error: Uknown subrecord."""
    def __init__(self,inName,subType,recType):
        ModError.__init__(self,_('Extraneous subrecord (%s) in %s record.') 
            % (subType,recType))

#------------------------------------------------------------------------------
class ModReader:
    """Wrapper around an TES4 file in read mode. 
    Will throw a ModReadError if read operation fails to return correct size."""
    def __init__(self,inName,ins):
        """Initialize."""
        self.inName = inName
        self.ins = ins
        #--Get ins size
        curPos = ins.tell()
        ins.seek(0,2)
        self.size = ins.tell()
        ins.seek(curPos)

    #--IO Stream ------------------------------------------
    def seek(self,offset,whence=0,recType='----'):
        """File seek."""
        if whence == 1:
            newPos = self.ins.tell()+offset
        elif whence == 2:
            newPos = self.size + offset
        else:
            newPos = offset
        if newPos < 0 or newPos > self.size: 
            raise ModReadError(self.inName, recType,newPos,self.size)
        self.ins.seek(offset,whence)
    
    def tell(self):
        """File tell."""
        return self.ins.tell()

    def close(self):
        """Close file."""
        self.ins.close()
    
    def atEnd(self,endPos=-1,recType='----'):
        """Return True if current read position is at EOF."""
        filePos = self.ins.tell()
        if endPos == -1:
            return (filePos == self.size)
        elif filePos > endPos:
            raise ModError(self.inName, _('Exceded limit of: ')+recType)
        else:
            return (filePos == endPos)

    #--Read/unpack ----------------------------------------
    def read(self,size,recType='----'):
        """Read from file."""
        endPos = self.ins.tell() + size
        if endPos > self.size:
            raise ModSizeError(self.inName, recType,endPos,self.size)
        return self.ins.read(size)
    
    def readString(self,size,recType='----'):
        """Read string from file, stripping zero terminator."""
        return cstrip(self.read(size,recType))
    
    def unpack(self,format,size,recType='----'):
        """Read file and unpack according to struct format."""
        endPos = self.ins.tell() + size
        if endPos > self.size:
            raise ModReadError(self.inName, recType,endPos,self.size)
        return struct.unpack(format,self.ins.read(size))

    def unpackRef(self,recType='----'):
        """Read a ref (formid)."""
        return self.unpack('I',4)[0]

    def unpackRecHeader(self):
        """Unpack a record header."""
        (type,size,str0,uint1,uint2) = self.unpack('4sI4s2I',20,'REC_HEAD')
        uint0, = struct.unpack('I',str0)
        #--Record
        if type != 'GRUP':
            return (type,size,uint0,uint1,uint2)
        #--Top Group
        elif uint1 == 0:
            if str0 in bush.topTypes:
                return (type,size,str0,uint1,uint2)
            elif str0 in bush.topIgTypes:
                return (type,size,bush.topIgTypes[str0],uint1,uint2)
            else:
                raise ModError(self.inName,_('Bad Top GRUP type: ')+str0)
        #--Other groups
        else:
            return (type,size,str0,uint1,uint2)

    def unpackSubHeader(self,recType='----',expType=None,expSize=0):
        """Unpack a subrecord header. Optionally checks for match with expected type and size."""
        (type,size) = self.unpack('4sH',6,recType+'.SUB_HEAD')
        #--Extended storage?
        if type == 'XXXX':
            size = self.unpack('I',4,recType+'.XXXX.SIZE.')[0]
            type = self.unpack('4sH',6,recType+'.XXXX.TYPE')[0] #--Throw away size (always == 0)
        #--Match expected name?
        if expType and expType != type:
            raise ModError(self.inName,_('%s: Expected %s subrecord, but found %s instead.') 
                % (recType,expType,type))
        #--Match expected size?
        if expSize and expSize != size:
            raise ModSizeError(self.inName,recType+'.'+type,size,expSize,True)
        return (type,size)

    #--Find data ------------------------------------------
    def findSubRecord(self,subType,recType='----'):
        """Finds subrecord with specified type."""
        while not self.atEnd():
            (type,size) = self.unpack('4sH',6,recType+'.SUB_HEAD')
            if type == subType:
                return self.read(size,recType+'.'+subType)
            else:
                self.seek(size,1,recType+'.'+type)
        #--Didn't find it?
        else:
            return None

#------------------------------------------------------------------------------
class ModWriter:
    """Wrapper around an TES4 output stream. Adds utility functions."""
    reValidType = re.compile('^[A-Z]{4}$')

    def __init__(self,out):
        """Initialize."""
        self.out = out

    #--Stream Wrapping
    def write(self,data):
        self.out.write(data)

    def getvalue(self):
        return self.out.getvalue()

    def close(self):
        self.out.close()

    #--Additional functions.
    def pack(self,format,*data):
        self.out.write(struct.pack(format,*data))

    def packSub(self,type,data,*values):
        """Write subrecord header and data to output stream.
        Call using either packSub(type,data), or packSub(type,format,values).
        Will automatically add a prefacing XXXX size subrecord to handle data 
        with size > 0xFFFF."""
        #if not ModWriter.reValidType.match(type): raise _('Invalid type: ') + `type`
        if data == None: return
        if values: data = struct.pack(data,*values)
        if len(data) <= 0xFFFF:
            self.out.write(struct.pack('=4sH',type,len(data)))
            self.out.write(data)
        else:
            self.out.write(struct.pack('=4sHI','XXXX',4,len(data)))
            self.out.write(struct.pack('=4sH',type,0))
            self.out.write(data)

    def packSub0(self,type,data):
        """Write subrecord header and data + null terminator to output stream."""
        #if not ModWriter.reValidType.match(type): raise _('Invalid type: ') + `type`
        if data == None: return
        self.out.write(struct.pack('=4sH',type,len(data)+1))
        self.out.write(data)
        self.out.write('\x00')

    def packRef(self,type,formid):
        """Write subrecord header and formid reference."""
        #if not ModWriter.reValidType.match(type): raise _('Invalid type: ') + `type`
        if formid != None: self.out.write(struct.pack('=4sHI',type,4,formid))

# Mod Record Elements ---------------------------------------------------------
# Constants
FID = 'FID' #--Used by MelStruct classes to indicate formid elements.

#------------------------------------------------------------------------------
class MelObject:
    """An empty class used by group and structure elements for data storage."""
    def __eq__(self,other):
        """Operator: =="""
        return isinstance(other,MelObject) and self.__dict__ == other.__dict__

    def __ne__(self,other):
        """Operator: !="""
        return not (isinstance(other,MelObject) and self.__dict__ == other.__dict__)

#------------------------------------------------------------------------------
class MelBase:
    """Represents a mod record raw element. Typically used for unknown elements. 
    Also used as parent class for other element types."""

    def __init__(self,type,attr,default=None):
        """Initialize."""
        self.type, self.attr, self.default = type, attr, default
        self._debug = False

    def debug(self,on=True):
        """Sets debug flag on self."""
        self._debug = on
        return self

    def parseElements(self,*elements):
        """Parses elements and returns attrs,defaults,actions,formAttrs where:
        * attrs is tuple of attibute (names)
        * formAttrs is tuple of attributes that have formids,
        * defaults is tuple of default values for attributes
        * actions is tuple of callables to be used when loading data
        Note that each element of defaults and actions matches corresponding attr element.
        Used by struct subclasses.
        """
        formAttrs = []
        attrs,defaults,actions = [0]*len(elements),[0]*len(elements),[0]*len(elements)
        for index,element in enumerate(elements):
            if not isinstance(element,tuple): element = (element,)
            if element[0] == FID: 
                formAttrs.append(element[1])
            elif callable(element[0]): 
                actions[index] = element[0]
            attrIndex = (0,1)[callable(element[0]) or element[0] in (FID,0)]
            attrs[index] = element[attrIndex]
            defaults[index] = (0,element[-1])[len(element)-attrIndex == 2]
        return map(tuple,(attrs,defaults,actions,formAttrs))

    def getDefaulters(self,defaulters,base):
        """Registers self as a getDefault(attr) provider."""
        pass

    def getLoaders(self,loaders):
        """Adds self as loader for type."""
        loaders[self.type] = self

    def hasFormids(self,formElements):
        """Include self if has formids."""
        pass

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = self.default

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        record.__dict__[self.attr] = ins.read(size,readId)
        if self._debug: print `getattr(record,self.attr)`

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        value = record.__dict__[self.attr]
        if value != None: out.packSub(self.type,value)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is True, then formid is set
        to result of function."""
        raise AbstractError

#------------------------------------------------------------------------------
class MelFormid(MelBase):
    """Represents a mod record formid element."""

    def hasFormids(self,formElements):
        """Include self if has formids."""
        formElements.add(self)

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        record.__dict__[self.attr] = ins.unpackRef(readId)
        if self._debug: print '  %08X' % (getattr(record,self.attr),)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        value = record.__dict__.get(self.attr,None)
        if value != None: out.packRef(self.type,value)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        attr = self.attr
        result = function(record.__dict__.get(attr,None))
        if save: record.__dict__[attr] = result

#------------------------------------------------------------------------------
class MelFormids(MelBase):
    """Represents a mod record formid elements."""

    def hasFormids(self,formElements):
        """Include self if has formids."""
        formElements.add(self)

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = []

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        formid = ins.unpackRef(readId)
        record.__dict__[self.attr].append(formid)
        if self._debug: print ' ',hex(formid)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        type = self.type
        for formid in record.__dict__[self.attr]:
            out.packRef(type,formid)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        formids = record.__dict__[self.attr]
        for index,formid in enumerate(formids):
            result = function(formid)
            if save: formids[index] = result

#------------------------------------------------------------------------------
class MelFormidList(MelFormids):
    """Represents a listmod record formid elements. The only difference from 
    MelFormids is how the data is stored. For MelFormidList, the data is stored
    as a single subrecord rather than as separate subrecords."""

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        if not size: return
        formids = ins.unpack(`size/4`+'I',size,readId)
        record.__dict__[self.attr] = list(formids)
        if self._debug: 
            for formid in formids:
                print '  %08X' % (formid,)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        formids = record.__dict__[self.attr]
        if not formids: return
        out.packSub(self.type,`len(formids)`+'I',*formids)

#------------------------------------------------------------------------------
class MelGroup(MelBase):
    """Represents a group record."""

    def __init__(self,attr,*elements):
        """Initialize."""
        self.attr,self.elements,self.formElements,self.loaders = attr,elements,set(),{}

    def debug(self,on=True):
        """Sets debug flag on self."""
        for element in self.elements: element.debug(on)
        return self

    def getDefaulters(self,defaulters,base):
        """Registers self as a getDefault(attr) provider."""
        defaulters[base+self.attr] = self
        for element in self.elements:
            element.getDefaulters(defaulters,base+self.attr+'.')

    def getLoaders(self,loaders):
        """Adds self as loader for subelements."""
        for element in self.elements:
            element.getLoaders(self.loaders)
        for type in self.loaders:
            loaders[type] = self

    def hasFormids(self,formElements):
        """Include self if has formids."""
        for element in self.elements:
            element.hasFormids(self.formElements)
        if self.formElements: formElements.add(self)

    def setDefault(self,record):
        """Sets default value for record instance."""
        setattr(record,self.attr,None)

    def getDefault(self):
        """Returns a default copy of object."""
        target = MelObject()
        for element in self.elements:
            element.setDefault(target)
        return target

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        target = record.__dict__[self.attr]
        if target == None: 
            target = self.getDefault()
            record.__dict__[self.attr] = target
        self.loaders[type].loadData(target,ins,type,size,readId)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        target = record.__dict__[self.attr]
        if not target: return
        for element in self.elements:
            element.dumpData(target,out)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        target = record.__dict__[self.attr]
        if not target: return
        for element in self.formElements:
            element.mapFormids(target,function,save)

#------------------------------------------------------------------------------
class MelGroups(MelGroup):
    """Represents an array of group record."""

    def __init__(self,attr,*elements):
        """Initialize. Must have at least one element."""
        MelGroup.__init__(self,attr,*elements)
        self.type0 = self.elements[0].type

    def setDefault(self,record):
        """Sets default value for record instance."""
        setattr(record,self.attr,[])

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        if type == self.type0:
            target = self.getDefault()
            record.__dict__[self.attr].append(target)
        else:
            target = record.__dict__[self.attr][-1]
        self.loaders[type].loadData(target,ins,type,size,readId)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        for target in record.__dict__[self.attr]:
            for element in self.elements:
                element.dumpData(target,out)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        for target in record.__dict__[self.attr]:
            for element in self.formElements:
                element.mapFormids(target,function,save)

#------------------------------------------------------------------------------
class MelList(MelBase):
    """Represents a variable length array that maps to multiple subrecords. Used 
    for lists of integers, etc. (E.g., refVars in SCPT.)"""

    def __init__(self,type,format,attr):
        """Initialize."""
        self.type, self.format, self.attr = type, format, attr
        self._debug = False

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = []

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        unpacked = ins.unpack(self.format,size,readId)
        record.__dict__[self.attr].append(unpacked[0])
        if self._debug: print unpacked

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        out.packSub(self.type,self.format,*record.__dict__[self.attr])

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        type,format = self.type,self.format
        for value in record.__dict__[self.attr]:
            out.packSub(type,format,value)

#------------------------------------------------------------------------------
class MelNull(MelBase):
    """Represents an obsolete record. Reads bytes from instream, but then 
    discards them and is otherwise inactive."""

    def __init__(self,type):
        """Initialize."""
        self.type = type
        self._debug = False

    def setDefault(self,record):
        """Sets default value for record instance."""
        pass

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        ins.read(size,readId)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        pass

#------------------------------------------------------------------------------
class MelString(MelBase):
    """Represents a mod record string element."""

    def __init__(self,type,attr,default=None,maxSize=0):
        """Initialize."""
        MelBase.__init__(self,type,attr,default)
        self.maxSize = maxSize

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        value = ins.readString(size,readId)
        record.__dict__[self.attr] = value
        if self._debug: print ' ',getattr(record,self.attr)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        value = record.__dict__[self.attr]
        if value != None: 
            if self.maxSize:
                value = winNewLines(value.rstrip())
                value = value[:min(self.maxSize,len(value))]
            out.packSub0(self.type,value)

#------------------------------------------------------------------------------
class MelStrings(MelString):
    """Represents array of strings."""

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = []

    def getDefault(self):
        """Returns a default copy of object."""
        return ''

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        target = MelObject()
        record.__dict__[self.attr].append(target)
        MelStruct.loadData(self,target,ins,type,size,readId)

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        value = ins.readString(size,readId)
        record.__dict__[self.attr].append(value)
        if self._debug: print ' ',value

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        for value in record.__dict__[self.attr]:
            MelString.dumpData(self,target,out)
            if self.maxSize:
                value = winNewLines(value.rstrip())
                value = value[:min(self.maxSize,len(value))]
            out.packSub0(self.type,value)

#------------------------------------------------------------------------------
class MelStruct(MelBase):
    """Represents a structure record."""

    def __init__(self,type,format,*elements):
        """Initialize."""
        self.type, self.format = type,format
        self.attrs,self.defaults,self.actions,self.formAttrs = self.parseElements(*elements)
        self._debug = False

    def hasFormids(self,formElements):
        """Include self if has formids."""
        if self.formAttrs: formElements.add(self)

    def setDefault(self,record):
        """Sets default value for record instance."""
        recordDict = record.__dict__
        for attr,value,action in zip(self.attrs, self.defaults, self.actions):
            if action: value = action(value)
            recordDict[attr] = value

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        unpacked = ins.unpack(self.format,size,readId)
        recordDict = record.__dict__
        for attr,value,action in zip(self.attrs,unpacked,self.actions):
            if action: value = action(value)
            recordDict[attr] = value
        if self._debug: 
            print ' ',zip(self.attrs,unpacked)
            if len(unpacked) != len(self.attrs):
                print ' ',unpacked

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        recordDict = record.__dict__
        values = []
        for attr,action in zip(self.attrs,self.actions):
            value = recordDict[attr]
            if action: value = value.dump()
            values.append(value)
        try:
            out.packSub(self.type,self.format,*values)
        except struct.error:
            print self.type,self.format,values
            raise

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        recordDict = record.__dict__
        for attr in self.formAttrs:
            result = function(recordDict[attr])
            if save: recordDict[attr] = result

#------------------------------------------------------------------------------
class MelStructs(MelStruct):
    """Represents array of structured records."""

    def __init__(self,type,format,attr,*elements):
        """Initialize."""
        MelStruct.__init__(self,type,format,*elements)
        self.attr = attr

    def getDefaulters(self,defaulters,base):
        """Registers self as a getDefault(attr) provider."""
        defaulters[base+self.attr] = self

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = []

    def getDefault(self):
        """Returns a default copy of object."""
        target = MelObject()
        for attr,value,action in zip(self.attrs, self.defaults, self.actions):
            if callable(action): value = action(value)
            setattr(target,attr,value)
        return target

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        target = MelObject()
        record.__dict__[self.attr].append(target)
        MelStruct.loadData(self,target,ins,type,size,readId)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        for target in record.__dict__[self.attr]:
            MelStruct.dumpData(self,target,out)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        for target in record.__dict__[self.attr]:
            MelStruct.mapFormids(self,target,function,save)

#------------------------------------------------------------------------------
class MelTuple(MelBase):
    """Represents a fixed length array that maps to a single subrecord. 
    (E.g., the stats array for NPC_ which maps to the DATA subrecord.)"""

    def __init__(self,type,format,attr,defaults):
        """Initialize."""
        self.type, self.format, self.attr, self.defaults = type, format, attr, defaults
        self._debug = False

    def setDefault(self,record):
        """Sets default value for record instance."""
        record.__dict__[self.attr] = self.defaults[:]

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        unpacked = ins.unpack(self.format,size,readId)
        record.__dict__[self.attr] = list(unpacked)
        if self._debug: print getattr(record,self.attr)

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        #print self.type,self.format,self.attr,getattr(record,self.attr)
        out.packSub(self.type,self.format,*record.__dict__[self.attr])

#------------------------------------------------------------------------------
# Common/Special Elements

#------------------------------------------------------------------------------
class MelConditions(MelStructs):
    """Represents a set of quest/dialog conditions. Difficulty is that FID state 
    of parameters depends on function index."""
    def __init__(self):
        """Initialize."""
        MelStructs.__init__(self,'CTDA','B3sfiii4s','conditions',
            'operFlag',('unk1',null3),'compValue','ifunc','param1','param2',('unk2',null4))

    def getLoaders(self,loaders):
        """Adds self as loader for type."""
        loaders[self.type] = self
        loaders['CTDT'] = self #--Older CTDT type for ai package records.

    def getDefault(self):
        """Returns a default copy of object."""
        target = MelStructs.getDefault(self)
        target.form12 = 'ii'
        return target

    def hasFormids(self,formElements):
        """Include self if has formids."""
        formElements.add(self)

    def loadData(self,record,ins,type,size,readId):
        """Reads data from ins into record attribute."""
        if type == 'CTDA' and size != 24: 
            raise ModSizeError(ins.inName,readId,24,size,True)
        if type == 'CTDT' and size != 20: 
            raise ModSizeError(ins.inName,readId,20,size,True)
        target = MelObject()
        record.conditions.append(target)
        unpacked1 = ins.unpack('B3sfi',12,readId)
        (target.operFlag,target.unk1,target.compValue,ifunc) = unpacked1
        #--Get parameters
        if ifunc not in bush.allConditions:
            raise BoshError(_('Unknown condition function: %d') % ifunc)
        form1 = 'iI'[ifunc in bush.fid1Conditions]
        form2 = 'iI'[ifunc in bush.fid2Conditions]
        form12 = form1+form2
        unpacked2 = ins.unpack(form12,8,readId)
        (target.param1,target.param2) = unpacked2
        if size == 24: 
            target.unk2 = ins.read(4)
        else:
            target.unk2 = null4
        (target.ifunc,target.form12) = (ifunc,form12)
        if self._debug: 
            unpacked = unpacked1+unpacked2
            print ' ',zip(self.attrs,unpacked)
            if len(unpacked) != len(self.attrs):
                print ' ',unpacked

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        for target in record.conditions:
            format = 'B3sfi'+target.form12+'4s'
            out.packSub('CTDA',format,
                target.operFlag, target.unk1, target.compValue,
                target.ifunc, target.param1, target.param2, target.unk2)

    def mapFormids(self,record,function,save=False):
        """Applies function to formids. If save is true, then formid is set
        to result of function."""
        for target in record.conditions:
            form12 = target.form12
            if form12[0] == 'I':
                result = function(target.param1)
                if save: target.param1 = result
            if form12[1] == 'I':
                result = function(target.param2)
                if save: target.param2 = result

#------------------------------------------------------------------------------
class MelEffects(MelGroups):
    """Represents ingredient/potion/enchantment/spell effects."""

    #--Class Data
    seFlags = Flags(0x559E00L,Flags.getNames('hostile')) #--0x559E is probably junk.
    class MelEffectsScit(MelStruct):
        """Subclass to support alternate format."""
        def __init__(self):
            MelStruct.__init__(self,'SCIT','Ii4sI',(FID,'id',None),('school',0),
                ('visual','REHE'),(MelEffects.seFlags,'flags',0x559E00L))
        def loadData(self,record,ins,type,size,readId):
            #--Alternate formats
            if size == 16:
                attrs,actions = self.attrs,self.actions
                unpacked = ins.unpack(self.format,size,readId)
            elif size == 12:
                attrs,actions = ('id','school','flags'),(0,0,MelEffects.seFlags)
                unpacked = ins.unpack('IiI',size,readId)
            else: #--size == 4
                #--The script formid for MS40TestSpell doesn't point to a valid script.
                #--But it's not used, so... Not a problem! It's also t
                attrs,actions = ('id',),(0,)
                unpacked = ins.unpack('I',size,readId)
                if unpacked[0] & 0xFF000000L: 
                    unpacked = (0L,) #--Discard bogus MS40TestSpell formid
            #--Unpack
            recordDict = record.__dict__
            for attr,value,action in zip(attrs,unpacked,actions):
                if callable(action): value = action(value)
                recordDict[attr] = value
            if self._debug: print ' ',unpacked

    #--Instance methods
    def __init__(self,attr='effects'):
        """Initialize elements."""
        MelGroups.__init__(self,attr,
            MelBase('EFID','id','REHE'),
            MelStruct('EFIT','4s5i',('id','REHE'),'magnitude','area','duration','recipient','actorValue'),
            MelGroup('scriptEffect',
                MelEffects.MelEffectsScit(),
                MelString('FULL','full'),
                ),
            )

#------------------------------------------------------------------------------
class MelFull0(MelString):
    """Represents the main full. Use this only when there are additional FULLs
    Which means when record has magic effects."""

    def __init__(self):
        """Initialize."""
        MelString.__init__(self,'FULL','full')

#------------------------------------------------------------------------------
class MelModel(MelGroup):
    """Represents a model record."""
    typeSets = (
        ('MODL','MODB','MODT'),
        ('MOD2','MO2B','MO2T'),
        ('MOD3','MO3B','MO3T'),
        ('MOD4','MO4B','MO4T'),)

    def __init__(self,attr='model',index=0):
        """Initialize. Index is 0,2,3,4 for corresponding type id."""
        types = MelModel.typeSets[(0,index-1)[index>0]]
        MelGroup.__init__(self,attr,
            MelString(types[0],'path'),
            MelBase(types[1],'modb'),
            MelBase(types[2],'modt'),)

    def debug(self,on=True):
        """Sets debug flag on self."""
        for element in self.elements[:2]: element.debug(on)
        return self

#------------------------------------------------------------------------------
class MelOptStruct(MelStruct):
    """Represents an optional structure, where if values are null, is skipped."""

    def dumpData(self,record,out):
        """Dumps data from record to outstream."""
        recordDict = record.__dict__
        for attr in self.attrs:
            if recordDict[attr]:
                MelStruct.dumpData(self,record,out)
                break

#------------------------------------------------------------------------------
class MelSet:
    """Set of mod record elments."""

    def __init__(self,*elements):
        """Initialize."""
        self._debug = False
        self.elements = elements
        self.defaulters = {}
        self.loaders = {}
        self.formElements = set()
        self.firstFull = None
        self.full0 = None
        for element in self.elements:
            element.getDefaulters(self.defaulters,'')
            element.getLoaders(self.loaders)
            element.hasFormids(self.formElements)
            if isinstance(element,MelFull0):
                self.full0 = element
        
    def debug(self,on=True):
        """Sets debug flag on self."""
        self._debug = on
        return self

    def initRecord(self,record,header,ins,unpack):
        """Initialize record."""
        for element in self.elements:
            element.setDefault(record)
        MreRecord.__init__(record,header,ins,unpack)

    def getDefault(self,attr):
        """Returns default instance of specified instance. Only useful for 
        MelGroup, MelGroups and MelStructs."""
        return self.defaulters[attr].getDefault()

    def loadData(self,record,ins,endPos):
        """Loads data from input stream. Called by load()."""
        doFullTest = (self.full0 != None)
        recType = record.type
        loaders = self.loaders
        _debug = self._debug
        #--Read Records
        if _debug: print '\n>>>> %08X' % (record.formid,)
        while not ins.atEnd(endPos,recType):
            (type,size) = ins.unpackSubHeader(recType)
            if _debug: print type,size
            readId = recType+'.'+type
            try:
                if type not in loaders:
                    raise ModError(ins.inName,_('Unexpected subrecord: ')+readId)
                #--Hack to handle the fact that there can be two types of FULL in spell/ench/ingr records.
                elif doFullTest and type == 'FULL':
                    self.full0.loadData(record,ins,type,size,readId)
                else:
                    loaders[type].loadData(record,ins,type,size,readId)
                doFullTest = doFullTest and (type != 'EFID')
            except:
                eid = getattr(record,'eid','<<NO EID>>')
                print 'Loading: %08X..%s..%s.%s..%d..' % (record.formid,eid,record.type,type,size)
                raise
        if _debug: print '<<<<',getattr(record,'eid','[NO EID]')

    def dumpData(self,record, out):
        """Dumps state into out. Called by getSize()."""
        for element in self.elements:
            element.dumpData(record,out)

    def mapFormids(self,record,mapper,save=False):
        """Maps formids of subelements."""
        for element in self.formElements:
            element.mapFormids(record,mapper,save)

    def convertFormids(self,record, mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        if record.longFormids == toLong: return
        record.formid = mapper(record.formid)
        for element in self.formElements:
            element.mapFormids(record,mapper,True)
        record.longFormids = toLong
        record.setChanged()

    def updateMasters(self,record,masters):
        """Updates set of master names according to masters actually used."""
        if not record.longFormids: raise StateError(_("FormIds not in long format"))
        def updater(formid): 
            masters.add(formid)
        updater(record.formid)
        for element in self.formElements:
            element.mapFormids(record,updater)

    def getReport(self):
        """Returns a report of structure."""
        buff = cStringIO.StringIO()
        for element in self.elements:
            element.report(None,buff,'')
        return buff.getvalue()

# Flags
#------------------------------------------------------------------------------
class MelBipedFlags(Flags):
    """Biped flags element. Includes biped flag set by default."""
    mask = 0xFFFF
    def __init__(self,default=0L,newNames=None):
        names = Flags.getNames('head', 'hair', 'upperBody', 'lowerBody', 'hand', 'foot', 'rightRing', 'leftRing', 'amulet', 'weapon', 'backWeapon', 'sideWeapon', 'quiver', 'shield', 'torch', 'tail')
        if newNames: names.update(newNames)
        Flags.__init__(self,default,names)
        
# Mod Records 0 ---------------------------------------------------------------
#------------------------------------------------------------------------------
class MreSubrecord:
    """Generic Subrecord."""
    def __init__(self,type,size,ins=None):
        self.changed = False
        self.type = type
        self.size = size
        self.data = None 
        self.inName = ins and ins.inName
        if ins: self.load(ins)

    def load(self,ins):
        self.data = ins.read(self.size,'----.'+self.type)
    
    def setChanged(self,value=True):
        """Sets changed attribute to value. [Default = True.]"""
        self.changed = value

    def setData(self,data):
        """Sets data and size."""
        self.data = data
        self.size = len(data)

    def getSize(self):
        """Return size of self.data, after, if necessary, packing it."""
        if not self.changed: return self.size
        #--StringIO Object
        out = ModWriter(cStringIO.StringIO())
        self.dumpData(out)
        #--Done
        self.data = out.getvalue()
        data.close()
        self.size = len(self.data)
        self.setChanged(False)
        return self.size

    def dumpData(self,out):
        """Dumps state into out. Called by getSize()."""
        raise AbstractError
    
    def dump(self,out):
        if self.changed: raise StateError(_('Data changed: ')+ self.type)
        if not self.data: raise StateError(_('Data undefined: ')+self.type)
        out.packSub(self.type,self.data)

#------------------------------------------------------------------------------
class MreRecord:
    """Generic Record."""
    subtype_attr = {'EDID':'eid','FULL':'full','MODL':'model'}
    flags1 = Flags(0L,Flags.getNames(
        ( 0,'esm'),
        ( 5,'deleted'),
        ( 9,'castsShadows'),
        (10,'questItem'),
        (10,'persistent'),
        (11,'initiallyDisabled'),
        (12,'ignored'),
        (15,'visibleWhenDistant'),
        (17,'dangerous'),
        (18,'compressed'),
        (19,'cantWait'),
        ))
    #--Set at end of class data definitions.
    type_class = None 
    topTypes = None

    def __init__(self,header,ins=None,unpack=False):
        (self.type,self.size,flags1,self.formid,self.flags2) = header
        self.flags1 = MreRecord.flags1(flags1)
        self.longFormids = False #--False: Short (numeric); True: Long (espname,objectindex)
        self.changed = False
        self.subrecords = None
        self.data = ''
        self.inName = ins and ins.inName
        if ins: self.load(ins,unpack)

    def getHeader(self):
        """Returns header tuple."""
        return (self.type,self.size,int(self.flags1),self.formid,self.flags2)

    def getBaseCopy(self):
        """Returns an MreRecord version of self."""
        baseCopy = MreRecord(self.getHeader())
        baseCopy.data = self.data
        return baseCopy

    def getTypeCopy(self,mapper=None):
        """Returns a type class copy of self, optionaly mapping formids to long."""
        if self.__class__ == MreRecord:
            fullClass = MreRecord.type_class[self.type]
            myCopy = fullClass(self.getHeader())
            myCopy.data = self.data
            myCopy.load(unpack=True)
        else:
            myCopy = copy.deepcopy(self)
        if mapper and not myCopy.longFormids:
            myCopy.convertFormids(mapper,True)
        myCopy.changed = True
        myCopy.data = None
        return myCopy

    def mergeFilter(self,modSet):
        """This method is called by the bashed patch mod merger. The intention is
        to allow a record to be filtered according to the specified modSet. E.g. 
        for a list record, items coming from mods not in the modSet could be 
        removed from the list."""
        pass

    def getDecompressed(self):
        """Return self.data, first decompressing it if necessary."""
        if not self.flags1.compressed: return self.data
        import zlib
        size, = struct.unpack('I',self.data[:4])
        decomp = zlib.decompress(self.data[4:])
        if len(decomp) != size: 
            raise ModError(self.inName,
                _('Mis-sized compressed data. Expected %d, got %d.') % (size,len(decomp)))
        return decomp

    def load(self,ins=None,unpack=False):
        """Load data from ins stream or internal data buffer."""
        type = self.type
        #--Read, but don't analyze.
        if not unpack:
            self.data = ins.read(self.size,type)
        #--Unbuffered analysis?
        elif ins and not self.flags1.compressed:
            inPos = ins.tell()
            self.data = ins.read(self.size,type)
            ins.seek(inPos,0,type+'_REWIND')
            self.loadData(ins,inPos+self.size)
        #--Buffered analysis (subclasses only)
        else:
            if ins:
                self.data = ins.read(self.size,type)
            if not self.__class__ == MreRecord:
                reader = self.getReader()
                self.loadData(reader,reader.size)
                reader.close()
        #--Discard raw data?
        if unpack == 2: 
            self.data = None
            self.changed = True

    def loadData(self,ins,endPos):
        """Loads data from input stream. Called by load().

        Subclasses should actually read the data, but MreRecord just skips over 
        it (assuming that the raw data has already been read to itself. To force
        reading data into an array of subrecords, use loadSubrecords()."""
        ins.seek(endPos)

    def loadSubrecords(self):
        """This is for MreRecord only. It reads data into an array of subrecords, 
        so that it can be handled in a simplistic way."""
        self.subrecords = []
        if not self.data: return
        reader = self.getReader()
        recType = self.type
        while not reader.atEnd(reader.size,recType):
            (type,size) = reader.unpackSubHeader(recType)
            self.subrecords.append(MreSubrecord(type,size,reader))
        reader.close()

    def convertFormids(self,mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        raise AbstractError(self.type)

    def updateMasters(self,masters):
        """Updates set of master names according to masters actually used."""
        raise AbstractError(self.type)

    def setChanged(self,value=True):
        """Sets changed attribute to value. [Default = True.]"""
        self.changed = value

    def setData(self,data):
        """Sets data and size."""
        self.data = data
        self.size = len(data)
        self.changed = False

    def getSize(self):
        """Return size of self.data, after, if necessary, packing it."""
        if not self.changed: return self.size
        if self.longFormids: raise StateError(
            _('Packing Error: %s %s: Formids in long format.') % self.type,self.formid)
        #--Pack data and return size.
        out = ModWriter(cStringIO.StringIO())
        self.dumpData(out)
        data = out.getvalue()
        self.data = out.getvalue()
        out.close()
        if self.flags1.compressed:
            import zlib
            dataLen = len(self.data)
            comp = zlib.compress(self.data,6)
            self.data = struct.pack('=I',dataLen) + comp
        self.size = len(self.data)
        self.setChanged(False)
        return self.size 

    def dumpData(self,out):
        """Dumps state into data. Called by getSize(). This default version 
        just calls subrecords to dump to out."""
        if self.subrecords == None: 
            raise StateError('Subrecords not unpacked. [%s: %s %08X]' % 
                (self.inName, self.type, self.formid))
        for subrecord in self.subrecords:
            subrecord.dump(out)
    
    def dump(self,out):
        """Dumps all data to output stream."""
        if self.changed: raise StateError(_('Data changed: ')+ self.type)
        if not self.data and not self.flags1.deleted: 
            raise StateError(_('Data undefined: ')+self.type)
        out.write(struct.pack('=4s4I',self.type,self.size,int(self.flags1),self.formid,self.flags2))
        if self.size > 0: out.write(self.data)

    def getReader(self):
        """Returns a ModReader wrapped around (decompressed) self.data."""
        return ModReader(self.inName,cStringIO.StringIO(self.getDecompressed()))

    #--Accessing subrecords ---------------------------------------------------
    def getSubString(self,subType):
        """Returns the (stripped) string for a zero-terminated string record."""
        #--Common subtype expanded in self?
        attr = MreRecord.subtype_attr.get(subType,None)
        if attr and attr in self.__dict__:
            return self.__dict__[attr]
        value = None
        #--Subrecords available?
        if self.subrecords != None:
            for subrecord in self.subrecords:
                if subrecord.type == subType:
                    value = cstrip(subrecord.data)
                    break
        #--No subrecords, but have data.
        elif self.data:
            reader = self.getReader()
            recType = self.type
            while not reader.atEnd(reader.size,recType):
                (type,size) = reader.unpackSubHeader(recType)
                if type != subType:
                    reader.seek(size,1)
                else:
                    value = cstrip(reader.read(size))
                    break
            reader.close()
        #--Save and return it
        self.__dict__[attr] = value
        return value

    def setSubString(self,subType,newString):
        """Set the data for the subrecord with the specified subtype."""
        #--Common subtype expanded in self?
        attr = MreRecord.subtype_attr.get(subType,None)
        self.__dict__[attr] = newString
        self.setChanged()
        if not self.__class__ == MreRecord:
            return
        #--Get/use subrecords
        if self.subrecords == None: 
            self.loadSubrecords()
        for subrecord in self.subrecords:
            if subrecord.type == subType:
                subrecord.setData(newString+'\0')
                return
        else:
            raise StateError('Subtype not present in subrecords. [%s: %s.%s %08X]' % 
                (self.inName, self.type,subType, self.formid))

#------------------------------------------------------------------------------
class MelRecord(MreRecord):
    """Mod record built from mod record elements."""
    melSet = None #--Subclasses must define as MelSet(*mels)

    def __init__(self,header,ins=None,unpack=False):
        """Initialize."""
        self.__class__.melSet.initRecord(self,header,ins,unpack)

    def getDefault(self,attr):
        """Returns default instance of specified instance. Only useful for 
        MelGroup, MelGroups and MelStructs."""
        return self.__class__.melSet.getDefault(attr)

    def loadData(self,ins,endPos):
        """Loads data from input stream. Called by load()."""
        self.__class__.melSet.loadData(self,ins,endPos)

    def dumpData(self,out):
        """Dumps state into out. Called by getSize()."""
        self.__class__.melSet.dumpData(self,out)

    def mapFormids(self,mapper,save):
        """Applies mapper to formids of sub-elements. Will replace formid with mapped value if save == True."""
        self.__class__.melSet.mapFormids(self,mapper,save)

    def convertFormids(self,mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        self.__class__.melSet.convertFormids(self,mapper,toLong)

    def updateMasters(self,masters):
        """Updates set of master names according to masters actually used."""
        self.__class__.melSet.updateMasters(self,masters)

#------------------------------------------------------------------------------
class MreLeveledList(MelRecord):
    """Leveled item/creature/spell list.."""
    flags = Flags(0,Flags.getNames('calcFromAllLevels','calcForEachItem','useAllSpells'))
    #--Special load classes
    class MelLevListLvld(MelStruct):
        """Subclass to support alternate format."""
        def loadData(self,record,ins,type,size,readId):
            MelStruct.loadData(self,record,ins,type,size,readId)
            if record.chanceNone > 127:
                record.flags.calcFromAllLevels = True
                record.chanceNone &= 127
    class MelLevListLvlo(MelStructs):
        """Subclass to support alternate format."""
        def loadData(self,record,ins,type,size,readId):
            target = self.getDefault()
            record.__dict__[self.attr].append(target)
            format,attrs = ((self.format,self.attrs),('iI',('level','id'),))[size==8]
            unpacked = ins.unpack(format,size,readId)
            for attr,value in zip(attrs,unpacked):
                setattr(target,attr,value)
    #--Element Set
    melSet = MelSet(
        MelString('EDID','eid'),
        MelLevListLvld('LVLD','B','chanceNone'),
        MelStruct('LVLF','B',(flags,'flags',0L)),
        MelFormid('SCRI','script'),
        MelFormid('TNAM','template'),
        MelLevListLvlo('LVLO','2hI2h','entries','level','junk1',(FID,'id',None),('count',1),'junk2'),
        MelNull('DATA'),
        )

    def __init__(self,header,ins=None,unpack=False):
        """Initialize."""
        MelRecord.__init__(self,header,ins,unpack)
        self.mergeOverLast = False #--Merge overrides last mod merged
        self.mergeSources = None #--Set to list by other functions
        self.items  = None #--Set of items included in list
        self.delevs = None #--Set of items deleted by list (Delev and Relev mods)
        self.relevs = None #--Set of items relevelled by list (Relev mods)

    def mergeFilter(self,modSet):
        """Filter out items that don't come from specified modSet."""
        if not self.longFormids: raise StateError(_("FormIds not in long format"))
        self.entries = [entry for entry in self.entries if entry.id[0] in modSet]

    def mergeWith(self,other,otherMod):
        """Merges newLevl settings and entries with self.
        Requires that: self.items, other.delevs and other.relevs be defined."""
        if not self.longFormids: raise StateError(_("FormIds not in long format"))
        if not other.longFormids: raise StateError(_("FormIds not in long format"))
        #--Relevel or not?
        if other.relevs:
            self.chanceNone = other.chanceNone
            self.script = other.script
            self.template = other.template
            self.flags = other.flags()
        else:
            self.chanceNone = other.chanceNone or self.chanceNone
            self.script   = other.script or self.script
            self.template = other.template or self.template
            self.flags |= other.flags
        #--Remove items based on other.removes
        if other.delevs or other.relevs:
            removeItems = self.items & (other.delevs | other.relevs)
            self.entries = [entry for entry in self.entries if entry.id not in removeItems]
            self.items = (self.items | other.delevs) - other.relevs
        hasOldItems = bool(self.items)
        #--Add new items from other
        newItems = set()
        for entry in other.entries:
            if entry.id not in self.items:
                self.entries.append(entry)
                newItems.add(entry.id)
        if newItems:
            self.items |= newItems
            self.entries.sort(key=lambda a: a.level)
        #--Is merged list different from other? (And thus written to patch.)
        if (self.chanceNone != other.chanceNone or
            self.script != other.script or
            self.template != other.template or
            #self.flags != other.flags or
            len(self.entries) != len(other.entries)):
            self.mergeOverLast = True
        else:
            for selfEntry,otherEntry in zip(self.entries,other.entries):
                if (selfEntry.id != otherEntry.id or 
                    selfEntry.level != otherEntry.level or
                    selfEntry.count != otherEntry.count):
                    self.mergeOverLast = True
                    break
            else:
                self.mergeOverLast = False
        if self.mergeOverLast:
            self.mergeSources.append(otherMod)
        else:
            self.mergeSources = [otherMod]
        #--Done
        self.setChanged()

#------------------------------------------------------------------------------
class MreHasEffects:
    """Mixin class for magic items."""
    def getEffects(self):
        """Returns a summary of effects. Useful for alchemical catalog."""
        effects = []
        for effect in self.effects:
            mgef, actorValue = effect.id, effect.actorValue
            if mgef not in bush.actorValueEffects:
                actorValue = 0
            effects.append((mgef,actorValue))
        return effects

    def getEffectsSummary(self):
        """Return a text description of magic effects."""
        buff = cStringIO.StringIO()
        if self.effects:
            school = bush.magicEffects[self.effects[0].id][0]
            buff.write(bush.actorValues[20+school] + '\n')
        for index,effect in enumerate(self.effects):
            if effect.scriptEffect:
                effectName = effect.scriptEffect.full or 'Script Effect'
            else:
                effectName = bush.magicEffects[effect.id][1]
                if effect.id in bush.actorValueEffects: 
                    effectName = re.sub('(Attribute|Skill)',bush.actorValues[effect.actorValue],effectName)
            buff.write('o+*'[effect.recipient]+' '+effectName)
            if effect.magnitude: buff.write(' '+`effect.magnitude`+'m')
            if effect.area: buff.write(' '+`effect.area`+'a')
            if effect.duration > 1: buff.write(' '+`effect.duration`+'d')
            buff.write('\n')
        return buff.getvalue()

# Mod Records 1 ---------------------------------------------------------------
#------------------------------------------------------------------------------
class MreActi(MelRecord):
    """Activator record."""
    type = 'ACTI' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelFormid('SNAM','sound'),
        )

#------------------------------------------------------------------------------
class MreAlch(MelRecord,MreHasEffects):
    """ALCH (potion) record."""
    type = 'ALCH' 
    flags = Flags(0L,Flags.getNames('autoCalc','isFood'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFull0(),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','f','weight'),
        MelStruct('ENIT','iI','value',(flags,'flags',0L)),
        MelEffects(),
        )

#------------------------------------------------------------------------------
class MreAmmo(MelRecord):
    """Ammo (arrow) record."""
    type = 'AMMO' 
    flags = Flags(0L,Flags.getNames('notNormalWeapon'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('ENAM','enchantment'),
        MelOptStruct('ANAM','H','enchantPoints'),
        MelStruct('DATA','fIIfH','speed',(flags,'flags',0L),'value','weight','damage'),
        )

#------------------------------------------------------------------------------
class MreAnio(MelRecord):
    """Animation object record."""
    type = 'ANIO' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelModel(),
        MelFormid('DATA','animationId'),
        )

#------------------------------------------------------------------------------
class MreAppa(MelRecord):
    """Alchemical apparatus record."""
    type = 'APPA' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','=Biff',('apparatus',0),('value',25),('weight',1),('quality',10)),
        )

#------------------------------------------------------------------------------
class MreArmo(MelRecord):
    """Amor record."""
    type = 'ARMO' 
    flags = MelBipedFlags(0L,Flags.getNames((16,'hideRings'),(17,'hideAmulet'),(22,'notPlayable'),(23,'heavyArmor')))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelFormid('SCRI','script'),
        MelFormid('ENAM','enchantment'),
        MelOptStruct('ANAM','H','enchantPoints'),
        MelStruct('BMDT','I',(flags,'flags',0L)),
        MelModel('maleBody',0),
        MelModel('maleWorld',2),
        MelString('ICON','maleIcon'),
        MelModel('femaleBody',3),
        MelModel('femaleWorld',4),
        MelString('ICO2','femaleIcon'),
        MelStruct('DATA','=HIIf','strength','value','health','weight'),
        )

#------------------------------------------------------------------------------
class MreBook(MelRecord):
    """BOOK record."""
    type = 'BOOK' 
    flags = Flags(0,Flags.getNames('isScroll','isFixed'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelString('DESC','text'),
        MelFormid('SCRI','script'),
        MelFormid('ENAM','enchantment'),
        MelOptStruct('ANAM','H','enchantPoints'),
        MelStruct('DATA', '=BBif',(flags,'flags',0L),('teaches',0xFF),'value','weight'),
        )

#------------------------------------------------------------------------------
class MreBsgn(MelRecord):
    """Alchemical apparatus record."""
    type = 'BSGN' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelString('ICON','icon'),
        MelString('DESC','text'),
        MelFormids('SPLO','spells'),
        )

#------------------------------------------------------------------------------
class MreClot(MelRecord):
    """Clothing record."""
    type = 'CLOT' 
    flags = MelBipedFlags(0L,Flags.getNames((16,'hideRings'),(17,'hideAmulet'),(22,'notPlayable')))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelFormid('SCRI','script'),
        MelFormid('ENAM','enchantment'),
        MelOptStruct('ANAM','H','enchantPoints'),
        MelStruct('BMDT','I',(flags,'flags',0L)),
        MelModel('maleBody',0),
        MelModel('maleWorld',2),
        MelString('ICON','maleIcon'),
        MelModel('femaleBody',3),
        MelModel('femaleWorld',4),
        MelString('ICO2','femaleIcon'),
        MelStruct('DATA','If','value','weight'),
        )

#------------------------------------------------------------------------------
class MreCont(MelRecord):
    """Container record."""
    type = 'CONT'
    flags = Flags(0,Flags.getNames(None,'respawns'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelStructs('CNTO','Ii','items',(FID,'id'),'count'),
        MelStruct('DATA','=Bf',(flags,'flags',0L),'weight'),
        MelFormid('SNAM','soundOpen'),
        MelFormid('QNAM','soundClose'),
        )

#------------------------------------------------------------------------------
class MreCrea(MelRecord):
    """NPC Record. Non-Player Character."""
    type = 'CREA' 
    #--Main flags
    flags = Flags(0L,Flags.getNames(
        ( 1,'essential'),
        ( 2,'weaponAndShield'),
        ( 3,'respawn'),
        ( 7,'pcLevelOffset'),
        ( 9,'noLowLevel'),
        (15,'noHead'),
        (16,'noRightArm'),
        (17,'noLeftArm'),
        (18,'noCombatInWater'),
        (19,'noShadow'),
        (20,'noCorpseCheck'),
        ))
#    #--AI Service flags
    aiService = Flags(0L,Flags.getNames(
        (0,'weapons'),
        (1,'armor'),
        (2,'clothing'),
        (3,'books'),
        (4,'ingredients'),
        (7,'lights'),
        (8,'apparatus'),
        (10,'miscItems'),
        (11,'spells'),
        (12,'magicItems'),
        (13,'potions'),
        (14,'training'),
        (16,'recharge'),
        (17,'repair'),))
    #--Mel Set
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormids('SPLO','spells'),
        MelBase('NIFZ','bodyParts'),
        MelBase('NIFT','nift'),
        MelStruct('ACBS','=I3Hh2H',
            (flags,'flags',0L),'baseSpell','fatigue','barterGold',
            ('level',1),'calcMin','calcMax'),
        MelStructs('SNAM','=IB3s','factions',
            (FID,'faction',None),'rank',('unknown','IFZ')),
        MelFormid('INAM','deathItem'),
        MelFormid('SCRI','script'),
        MelStructs('CNTO','Ii','items',(FID,'item',None),('count',1)),
        MelStruct('AIDT','=4BI2BH',
            ('aggression',5),('confidence',50),('energyLevel',50),('responsibility',50),
            (aiService,'services',0L),'trainSkill','trainLevel','aiUnknown'),
        MelFormids('PKID','aiPackages'),
        MelBase('KFFZ','kffz'),
        MelStruct('DATA','=b3B4H8B','creatureType','combat','magic','stealth','soul','health','unk1','attackDamage','strength','intelligence','willpower','agility','speed','endurance','personality','luck'),
        MelStruct('RNAM','B','attackReach'),
        MelFormid('ZNAM','combatStyle'),
        MelStruct('TNAM','f','turningSpeed'),
        MelStruct('BNAM','f','baseScale'),
        MelStruct('WNAM','f','footWeight'),
        MelFormid('CSCR','inheritsSoundsFrom'),
        MelString('NAM0','bloodSpray'),
        MelString('NAM1','bloodDecal'),
        MelGroups('sounds',
            MelStruct('CSDT','I','type'),
            MelFormid('CSDI','sound'),
            MelStruct('CSDC','B','chance'),
        ),
        )

#------------------------------------------------------------------------------
class MreDial(MelRecord):
    """Dialog record."""
    type = 'DIAL' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFormids('QSTI','quests'),
        MelString('FULL','full'),
        MelStruct('DATA','B','dialType'),
    )

    def __init__(self,header,ins=None,unpack=False):
        """Initialize."""
        MelRecord.__init__(self,header,ins,unpack)
        self.infoStamp = 0 #--Stamp for info GRUP
        self.infos = []

    def loadInfos(self,ins,endPos,infoClass):
        """Load infos from ins. Called from DialBlock."""
        infos = self.infos
        while not ins.atEnd(endPos,'INFO Block'):
            #--Get record info and handle it
            header = ins.unpackRecHeader()
            recType = header[0]
            if recType == 'INFO':
                info = infoClass(header,ins,True)
                infos.append(info)
            else:
                raise ModError(_('Unexpected %s record in %s group.') 
                    % (recType,"INFO"), ins.inName)

    def dump(self,out):
        """Dumps self., then group header and then records."""
        MreRecord.dump(self,out)
        if not self.infos: return
        size = 20 + sum([20 + info.getSize() for info in self.infos])
        out.pack('4sIIII','GRUP',size,self.formid,7,self.infoStamp)
        for info in self.infos: info.dump(out)

    def updateMasters(self,masters):
        """Updates set of master names according to masters actually used."""
        MelRecord.updateMasters(self,masters)
        for info in self.infos:
            info.updateMasters(masters)

    def convertFormids(self,mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        MelRecord.convertFormids(self,mapper,toLong)
        for info in self.infos:
            info.convertFormids(mapper,toLong)

#------------------------------------------------------------------------------
class MreDoor(MelRecord):
    """Container record."""
    type = 'DOOR'
    flags = Flags(0,Flags.getNames('oblivionGate','automatic','hidden','minimalUse'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelStructs('CNTO','Ii','items',(FID,'id'),'count'),
        MelStruct('DATA','=Bf',(flags,'flags',0L),'weight'),
        MelFormid('SNAM','soundOpen'),
        MelFormid('ANAM','soundClose'),
        MelFormid('BNAM','soundLoop'),
        MelStruct('FNAM','B',(flags,'flags',0L)),
        MelFormids('TNAM','destinations'),
        )

#------------------------------------------------------------------------------
class MreEfsh(MelRecord):
    """Effect shader record."""
    type = 'EFSH' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('ICON','fillTexture'),
        MelString('ICO2','particleTexture'),
        MelBase('DATA','DATA'),
        )

#------------------------------------------------------------------------------
class MreEnch(MelRecord,MreHasEffects):
    """Enchantment record."""
    type = 'ENCH' 
    flags = Flags(0L,Flags.getNames('noAutoCalc'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFull0(), #--At least one mod has this. Odd.
        MelStruct('ENIT','3iI','itemType','chargeAmount','enchantCost',(flags,'flags',0L)),
        #--itemType = 0: Scroll, 1: Staff, 2: Weapon, 3: Apparel
        MelEffects(),
        )

#------------------------------------------------------------------------------
class MreEyes(MelRecord):
    """Eyes record."""
    type = 'EYES' 
    flags = Flags(0L,Flags.getNames('playable',))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelString('ICON','icon'),
        MelStruct('DATA','B',(flags,'flags')),
        )
    
#------------------------------------------------------------------------------
class MreFact(MelRecord):
    """Faction record."""
    type = 'FACT'
    flags = Flags(0L,Flags.getNames('hiddenFromPC','evil','specialCombat'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelStructs('XNAM','Ii','relations',(FID,'faction'),'mod'),
        MelStruct('DATA','B',(flags,'flags',0L)),
        MelStruct('CNAM','f',('crimeGoldMultiplier',1)),
        MelGroups('ranks',
            MelStruct('RNAM','i','rank'),
            MelString('MNAM','male'),
            MelString('FNAM','female'),
            MelString('INAM','insignia'),),
        )

#------------------------------------------------------------------------------
class MreFlor(MelRecord):
    """Flora (plant) record."""
    type = 'FLOR' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelFormid('PFIG','ingredient'),
        MelStruct('PFPC','4B','spring','summer','fall','winter'),
        )

#------------------------------------------------------------------------------
class MreFurn(MelRecord):
    """Furniture record."""
    type = 'FURN' 
    flags = Flags() #--Governs type of furniture and which anims are available
    #--E.g., whether it's a bed, and which of the bed entry/exit animations are available
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelStruct('MNAM','I',(flags,'activeMarkers',0L)),
        )

#------------------------------------------------------------------------------
class MreGlob(MelRecord):
    """Global record. Rather stupidly all values, despite their 
    designation (short,long,float) are stored as floats -- which means that 
    very large integers lose precision."""
    type = 'GLOB' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelBase('FNAM','format'), #-'s','l','f' for short/long/float 
        MelStruct('FLTV','f','value'),
        )

#------------------------------------------------------------------------------
class MreGmst(MelRecord):
    """Gmst record"""
    oblivionIds = None
    type = 'GMST' 
    class MelGmstValue(MelBase):
        def loadData(self,record,ins,type,size,readId):
            format = record.eid[0] #-- s|i|f
            if format == 's':
                record.value = ins.readString(size,readId)
            else:
                record.value, = ins.unpack(format,size,readId)
        def dumpData(self,record,out):
            format = record.eid[0] #-- s|i|f
            if format == 's':
                out.packSub0(self.type,record.value)
            else:
                out.packSub(self.type,format,record.value)
    melSet = MelSet(
        MelString('EDID','eid'),
        MelGmstValue('DATA','value'),
        )

    def getOblivionFormid(self):
        """Returns Oblivion.esm formid in long format for specified eid."""
        myClass = self.__class__
        if not myClass.oblivionIds:
            myClass.oblivionIds = cPickle.load(Path.get(r'Data\Oblivion_ids.pkl').open())['GMST']
        return (Path.get('Oblivion.esm'), myClass.oblivionIds[self.eid])

#------------------------------------------------------------------------------
class MreGras(MelRecord):
    """Grass record."""
    type = 'GRAS' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelModel(),
        MelBase('DATA','gdata'),
        )

#------------------------------------------------------------------------------
class MreHair(MelRecord):
    """Hair record."""
    type = 'HAIR' 
    flags = Flags(0L,Flags.getNames('playable','notMale','notFemale','fixed'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelStruct('DATA','B',(flags,'flags')),
        )
 
#------------------------------------------------------------------------------
class MreInfo(MelRecord):
    """Info (dialog entry) record."""
    type = 'INFO' 
    flags = Flags(0,Flags.getNames(
        'goodbye','random','sayOnce',None,'infoRefusal','randomEnd','runForRumors'))
    class MelInfoData(MelStruct):
        """Support older 2 byte version."""
        def loadData(self,record,ins,type,size,readId):
            if size == 2:
                record.dialType,flags = ins.unpack('BB',2,readId)
                record.flags = MreInfo.flags(flags)
                record.unk1 = 0
                if self._debug: print (record.dialType,record.flags.getTrueAttrs(),record.unk1)
            else:
                MelStruct.loadData(self,record,ins,type,size,readId)
    class MelInfoSchr(MelStruct):
        """Print only if schd record is null."""
        def dumpData(self,record,out):
            if not record.schd:
                MelStruct.dumpData(self,record,out)
    #--MelSet
    melSet = MelSet(
        MelInfoData('DATA','BBB','dialType',(flags,'flags'),'unk1'),
        MelFormid('QSTI','questId'),
        MelFormid('PNAM','prevInfo'),
        MelFormids('NAME','addTopics'),
        MelGroups('responses',
            MelStruct('TRDT','iiIB3s','emotionType','emotionValue','unk1','responseNum','unk2'),
            MelString('NAM1','responseText'),
            MelString('NAM2','actorNotes'),
            ),
        MelConditions(),
        MelFormids('TCLT','choices'),
        MelFormids('TCLF','linkFrom'),
        MelBase('SCHD','schd'), #--Old format script header?
        MelInfoSchr('SCHR','5I','unk2','numRefs','compiledSize','lastIndex','scriptType'),
        MelBase('SCDA','scriptBytes'),
        MelBase('SCTX','scriptText'),
        MelFormids('SCRO','refConstants'),
        )

#------------------------------------------------------------------------------
class MreIngr(MelRecord,MreHasEffects):
    """INGR (ingredient) record."""
    type = 'INGR' 
    flags = Flags(0L,Flags.getNames('noAutoCalc','isFood'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFull0(),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','f','weight'),
        MelStruct('ENIT','iI','value',(flags,'flags',0L)),
        MelEffects(),
        )

#------------------------------------------------------------------------------
class MreKeym(MelRecord):
    """MISC (miscellaneous item) record."""
    type = 'KEYM' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','if','value','weight'),
        )

#------------------------------------------------------------------------------
class MreLigh(MelRecord):
    """Light source record."""
    type = 'LIGH' 
    flags = Flags(0L,Flags.getNames('dynamic','canTake','negative','flickers',
        'unk1','offByDefault','flickerSlow','pulse','pulseSlow','spotlight','spotShadow'))
    #--Mel NPC DATA
    class MelLighData(MelStruct):
        """Handle older trucated DATA for LIGH subrecord."""
        def loadData(self,record,ins,type,size,readId):
            if size == 32:
                MelStruct.loadData(self,record,ins,type,size,readId)
                return
            #--Else 24 byte record (skips falloff and fov...
            unpacked = ins.unpack('iI4sIIf',size,readId)
            (record.duration, record.radius, record.color, record.flags,
                record.value, record.weight) = unpacked
            record.flags = MreLigh.flags(unpacked[3])
            if self._debug: print unpacked, record.flags.getTrueAttrs()
    melSet = MelSet(
        MelString('EDID','eid'),
        MelModel(),
        MelFormid('SCRI','script'),
        MelString('FULL','full'),
        MelString('ICON','icon'),
        MelLighData('DATA','iI4sIffIf','duration','radius','color',(flags,'flags',0L),
            'falloff',('fov',90),'value','weight'),
        MelStruct('FNAM','f','fade'),
        MelFormid('SNAM','sound'),
        )

#------------------------------------------------------------------------------
class MreLscr(MelRecord):
    """Load screen."""
    type = 'LSCR' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('ICON','icon'),
        MelString('DESC','text'),
        MelStructs('LNAM','IIhh','locations',(FID,'direct'),(FID,'indirect'),'gridy','gridx'),
        )

#------------------------------------------------------------------------------
class MreLvlc(MreLeveledList):
    """LVLC record. Leveled list for creatures."""
    type = 'LVLC' 

#------------------------------------------------------------------------------
class MreLvli(MreLeveledList):
    """LVLI record. Leveled list for items."""
    type = 'LVLI' 

#------------------------------------------------------------------------------
class MreLvsp(MreLeveledList):
    """LVSP record. Leveled list for items."""
    type = 'LVSP' 

#------------------------------------------------------------------------------
class MreMgef(MelRecord):
    """MGEF (magic effect) record."""
    type = 'MGEF' 
    #--Mel NPC DATA
    class MelMgefData(MelStruct):
        """Handle older trucated DATA for DARK subrecord."""
        def loadData(self,record,ins,type,size,readId):
            if size == 64:
                MelStruct.loadData(self,record,ins,type,size,readId)
                return
            #--Else is data for DARK record, read it all.
            unpacked = ins.unpack('IfiiiIIfI',size,readId)
            (record.flags,record.baseCost,record.unk1,record.school,
                record.resistValue,record.unk2,record.light,record.projectileSpeed,
                record.effectShader) = unpacked
            if self._debug: print unpacked
    class MelMgefEsce(MelBase):
        """Handle counter-effects array."""
        def loadData(self,record,ins,type,size,readId):
            data = ins.read(size)
            record.counterEffects = [data[first:first+4] for first in range(0,len(data),4)]
            if self._debug: print `record.counterEffects`
        def dumpData(self,record,out):
            if record.counterEffects:
                out.packSub(self.type,''.join(record.counterEffects))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelString('DESC','text'),
        MelString('ICON','icon'),
        MelModel(),
        MelMgefData('DATA','IfiiiIIf6I2f',
            'flags','baseCost','unk1','school','resistValue','unk2',(FID,'light'),'projectileSpeed',
            (FID,'effectShader'),(FID,'enchantEffect'),(FID,'castingSound'),
            (FID,'boltSound'),(FID,'hitSound'),(FID,'areaSound'),'cefEnchantment','cefBarter'),
        MelMgefEsce('ESCE','counterEffects'),
        )

#------------------------------------------------------------------------------
class MreMisc(MelRecord):
    """MISC (miscellaneous item) record."""
    type = 'MISC' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','if','value','weight'),
        )

#------------------------------------------------------------------------------
class MreNpc(MelRecord):
    """NPC Record. Non-Player Character."""
    type = 'NPC_' 
    #--Main flags
    flags = Flags(0L,Flags.getNames(
        ( 0,'female'),
        ( 1,'essential'),
        ( 3,'respawn'),
        ( 4,'autoCalc'),
        ( 7,'pcLevelOffset'),
        ( 9,'noLowLevel'),
        (13,'noRumors'),
        (14,'summonable'),
        (15,'noPersuasion'),
        (20,'canCorpseCheck'),))
    #--AI Service flags
    aiService = Flags(0L,Flags.getNames(
        (0,'weapons'),
        (1,'armor'),
        (2,'clothing'),
        (3,'books'),
        (4,'ingredients'),
        (7,'lights'),
        (8,'apparatus'),
        (10,'miscItems'),
        (11,'spells'),
        (12,'magicItems'),
        (13,'potions'),
        (14,'training'),
        (16,'recharge'),
        (17,'repair'),))
    #--Mel NPC DATA
    class MelNpcData(MelStruct):
        """Convert npc stats into skills, health, attributes."""
        def loadData(self,record,ins,type,size,readId):
            unpacked = list(ins.unpack('=21BI8B',size,readId))
            recordDict = record.__dict__
            recordDict['skills'] = unpacked[:21]
            recordDict['health'] = unpacked[21]
            recordDict['attributes'] = unpacked[22:]
            if self._debug: print unpacked[:21],unpacked[21],unpacked[22:]
        def dumpData(self,record,out):
            """Dumps data from record to outstream."""
            recordDict = record.__dict__
            values = recordDict['skills']+[recordDict['health']]+recordDict['attributes']
            out.packSub(self.type,'=21BI8B',*values)

    #--Mel NPC FNAM
    class MelNpcFnam(MelStruct):
        def dumpData(self,record,out):
            if record.fnam: MelStruct.dumpData(self,record,out)
    #--Mel Set
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelStruct('ACBS','=I3Hh2H',
            (flags,'flags',0L),'baseSpell','fatigue','barterGold',
            ('level',1),'calcMin','calcMax'),
        MelStructs('SNAM','=Ib3s','factions',
            (FID,'faction',None),'rank',('unknown','ODB')),
        MelFormid('INAM','deathItem'),
        MelFormid('RNAM','race'),
        MelFormids('SPLO','spells'),
        MelFormid('SCRI','script'),
        MelStructs('CNTO','Ii','items',(FID,'item',None),('count',1)),
        MelStruct('AIDT','=4BI2BH',
            ('aggression',5),('confidence',50),('energyLevel',50),('responsibility',50),
            (aiService,'services',0L),'trainSkill','trainLevel','aiUnknown'),
        MelFormids('PKID','aiPackages'),
        MelBase('KFFZ','kffz'),
        MelFormid('CNAM','iclass'),
        MelNpcData('DATA','',('skills',[0]*21),'health',('attributes',[0]*8)),
        MelFormid('HNAM','hair'),
        MelStruct('LNAM','f',('hairLength',1)),
        MelFormid('ENAM','eyes'),
        MelStruct('HCLR','I',('hairColor',0xFFFFFF00L)),
        MelFormid('ZNAM','combatStyle'),
        MelBase('FGGS','fggs'),
        MelBase('FGGA','fgga'),
        MelBase('FGTS','fgts'),
        MelNpcFnam('FNAM','H','fnam'),
        )

    def setRace(self,raceId):
        """Set additional race info."""
        self.race = raceId
        #--Model
        if not self.model:
            self.model = self.getDefault('model')
        if raceId in (0x23fe9,0x223c7):
            self.model.path = r"Characters\_Male\SkeletonBeast.NIF"
        else:
            self.model.path = r"Characters\_Male\skeleton.nif"
        #--FNAM
        fnams = {
            0x23fe9 : 0x3cdc ,#--Argonian
            0x224fc : 0x1d48 ,#--Breton
            0x191c1 : 0x5472 ,#--Dark Elf
            0x19204 : 0x21e6 ,#--High Elf
            0x00907 : 0x358e ,#--Imperial
            0x22c37 : 0x5b54 ,#--Khajiit
            0x224fd : 0x03b6 ,#--Nord
            0x191c0 : 0x0974 ,#--Orc
            0x00d43 : 0x61a9 ,#--Redguard
            0x00019 : 0x4477 ,#--Vampire
            0x223c8 : 0x4a2e ,#--Wood Elf
            }
        self.fnam = fnams.get(raceId,0x358e)

#------------------------------------------------------------------------------
class MrePack(MelRecord):
    """AI package record."""
    type = 'PACK' 
    flags = Flags(0,Flags.getNames(
        'offersServices','mustReachLocation','mustComplete','lockAtStart',
        'lockAtEnd','lockAtLocation','unlockAtStart','unlockAtEnd',
        'unlockAtLocation','continueIfPcNear','oncePerDay',None,
        'skipFallout','alwaysRun',None,None,
        None,'alwaysSneak','allowSwimming','allowFalls',
        'unequipArmor','unequipWeapons','defensiveCombat','useHorse',
        'noIdleAnims',))
    class MelPackPkdt(MelStruct):
        """Support older 4 byte version."""
        def loadData(self,record,ins,type,size,readId):
            if size != 4:
                MelStruct.loadData(self,record,ins,type,size,readId)
            else:
                flags,record.aiType,junk = ins.unpack('HBc',4,readId)
                record.flags = MrePack.flags(flags)
                record.unk1 = null3
                if self._debug: print (record.flags.getTrueAttrs(),record.aiType,record.unk1)
    class MelPackLT(MelStruct):
        """For PLDT and PTDT. Second element of both may be either an FID or a long, 
        depending on value of first element."""
        def hasFormids(self,formElements):
            formElements.add(self)
        def dumpData(self,record,out):
            if ((self.type == 'PLDT' and (record.locType or record.locId)) or
                (self.type == 'PTDT' and (record.targetType or record.targetId))):
                MelStruct.dumpData(self,record,out)
        def mapFormids(self,record,function,save=False):
            """Applies function to formids. If save is true, then formid is set
            to result of function."""
            if self.type == 'PLDT' and record.locType != 5:
                result = function(record.locId)
                if save: record.locId = result
            elif self.type == 'PTDT' and record.targetType != 2:
                result = function(record.targetId)
                if save: record.targetId = result
    #--MelSet
    melSet = MelSet(
        MelString('EDID','eid'),
        MelPackPkdt('PKDT','IB3s',(flags,'flags'),'aiType',('unk1',null3)),
        MelPackLT('PLDT','iIi','locType','locId','locRadius'),
        MelStruct('PSDT','BBBBi','month','day','date','time','duration'),
        MelPackLT('PTDT','iIi','targetType','targetId','targetCount'),
        MelConditions(),
        )

#------------------------------------------------------------------------------
class MreQust(MelRecord):
    """Quest record."""
    type = 'QUST' 
    questFlags = Flags(0,Flags.getNames('startGameEnabled',None,'repeatedTopics','repeatedStages'))
    stageFlags = Flags(0,Flags.getNames('complete'))
    targetFlags = Flags(0,Flags.getNames('ignoresLocks'))

    #--CDTA loader
    class MelQustLoaders(DataDict):
        """Since CDTA subrecords occur in three different places, we need
        to replace ordinary 'loaders' dictionary with a 'dictionary' that will
        return the correct element to handle the CDTA subrecord. 'Correct' 
        element is determined by which other subrecords have been encountered."""
        def __init__(self,loaders,quest,stages,targets):
            self.data = loaders
            self.type_ctda = {'EDID':quest, 'INDX':stages, 'QSTA':targets}
            self.ctda = quest #--Which ctda element loader to use next.
        def __getitem__(self,key):
            if key == 'CTDA': return self.ctda
            self.ctda = self.type_ctda.get(key, self.ctda)
            return self.data[key]
    
    #--MelSet
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFormid('SCRI','script'),
        MelString('FULL','full'),
        MelString('ICON','icon'),
        MelStruct('DATA','BB',(questFlags,'questFlags',0),'priority'),
        MelConditions(),
        MelGroups('stages',
            MelStruct('INDX','h','stage'),
            MelGroups('entries',
                MelStruct('QSDT','B','flags'),
                MelConditions(),
                MelString('CNAM','text'),
                MelStruct('SCHR','5I','unk1','numRefs','compiledSize','lastIndex','scriptType'),
                MelBase('SCDA','compiled'),
                MelBase('SCTX','script'),
                MelFormids('SCRO','refConstants'),
                ),
            ),
        MelGroups('targets',
            MelStruct('QSTA','IB3s',(FID,'target'),'flags','unk1'),
            MelConditions(),
            ),
        )
    melSet.loaders = MelQustLoaders(melSet.loaders,*melSet.elements[5:8])

#------------------------------------------------------------------------------
class MreRace(MelRecord):
    """Race record.

    This record is complex to read and write. Relatively simple problems are the 
    DATA subrecord (which contains a structured array of boosts) and the VNAM
    which can be empty or zeroed depending on relationship between voices and 
    the formid for the race. 

    The face and body data is much more complicated, with the same subrecord types 
    mapping to different attributes depending on preceding flag subrecords (NAM0, NAM1, 
    NMAN, FNAM and INDX.) These are handled by using the MelRaceDistributor class 
    to dynamically reassign melSet.loaders[type] as the flag records are encountered.

    It's a mess, but this is the shortest, clearest implementation that I could 
    think of."""

    type = 'RACE' 
    flags = Flags(0L,Flags.getNames('playable'))

    class MelRaceData(MelStruct):
        """Convert raw string of _boosts into boosts = [(av0,boost0),(av1,boost1),...]"""
        def loadData(self,record,ins,type,size,readId):
            MelStruct.loadData(self,record,ins,type,size,readId)
            unpacked = struct.unpack('14B',record._boosts)
            record.boosts = zip(unpacked[0::2],unpacked[1::2])

        def dumpData(self,record,out):
            record._boosts = struct.pack('14B',*sum(record.boosts,tuple()))
            MelStruct.dumpData(self,record,out)
    
    class MelRaceVoices(MelStruct):
        """Set voices to zero, if equal race formid. If both are zero, then don't skip dump."""
        def dumpData(self,record,out):
            if record.maleVoice   == record.formid: record.maleVoice   = 0L
            if record.femaleVoice == record.formid: record.femaleVoice = 0L
            if (record.maleVoice,record.femaleVoice) != (0,0):
                MelStruct.dumpData(self,record,out)
    
    class MelRaceModel(MelGroup):
        """Most face data, like a MelModel - MODT + ICON. Load is controlled by MelRaceDistributor."""
        def __init__(self,attr,index):
            MelGroup.__init__(self,attr,
                MelString('MODL','path'),
                MelBase('MODB','modb'),
                MelBase('ICON','icon'),)
            self.index = index

        def dumpData(self,record,out):
            out.packSub('INDX','i',self.index)
            MelGroup.dumpData(self,record,out)

    class MelRaceIcon(MelString):
        """Most body data plus eyes for face. Load is controlled by MelRaceDistributor."""
        def __init__(self,attr,index):
            MelString.__init__(self,'ICON',attr)
            self.index = index
        def dumpData(self,record,out):
            out.packSub('INDX','i',self.index)
            MelString.dumpData(self,record,out)

    class MelRaceDistributor(MelNull):
        """Handles NAM0, NAM1, MNAM, FMAN and INDX records. Distributes load 
        duties to other elements as needed."""
        def __init__(self):
            bodyAttrs = ('UpperBody','LowerBody','Hand','Foot','Tail')
            self.attrs = {
                'MNAM':tuple('male'+text for text in bodyAttrs),
                'FNAM':tuple('female'+text for text in bodyAttrs),
                'NAM0':('head', 'maleEars', 'femaleEars', 'mouth', 
                'teethLower', 'teethUpper', 'tongue', 'leftEye', 'rightEye',)
                }
            self.tailModelAttrs = {'MNAM':'maleTailModel','FNAM':'femaleTailModel'}
            self._debug = False

        def getLoaders(self,loaders):
            """Self as loader for structure types."""
            for type in ('NAM0','MNAM','FNAM','INDX'):
                loaders[type] = self
        
        def setMelSet(self,melSet):
            """Set parent melset. Need this so that can reassign loaders later."""
            self.melSet = melSet
            self.loaders = {}
            for element in melSet.elements:
                attr = element.__dict__.get('attr',None)
                if attr: self.loaders[attr] = element
        
        def loadData(self,record,ins,type,size,readId):
            if type in ('NAM0','MNAM','FNAM'):
                record._loadAttrs = self.attrs[type]
                attr = self.tailModelAttrs.get(type)
                if not attr: return
            else: #--INDX
                index, = ins.unpack('i',4,readId)
                attr = record._loadAttrs[index]
            element = self.loaders[attr]
            for type in ('MODL','MODB','ICON'):
                self.melSet.loaders[type] = element

    #--Mel Set
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelString('DESC','text'),
        MelFormids('SPLO','spells'),
        MelStructs('XNAM','Ii','relations',(FID,'faction'),'mod'),
        MelRaceData('DATA','14sH4fI','_boosts','unk1',
            'maleHeight','femaleHeight','maleWeight','femaleWeight',(flags,'flags',0L)),
        MelRaceVoices('VNAM','2I',(FID,'maleVoice'),(FID,'femaleVoice')), #--0 same as race formid.
        MelStruct('DNAM','2I',(FID,'defaultHairMale',0L),(FID,'defaultHairFemale',0L)), #--0=None
        MelStruct('CNAM','B','defaultHairColor'), #--Int corresponding to GMST sHairColorNN
        MelBase('PNAM','PNAM'),
        MelBase('UNAM','UNAM'),
        #--Male: Str,Int,Wil,Agi,Spd,End,Per,luck; Female Str,Int,...
        MelTuple('ATTR','16B','baseAttributes',[0]*16),
        #--Begin Indexed entries
        MelBase('NAM0','_nam0',''),
        MelRaceModel('head',0),
        MelRaceModel('maleEars',1),
        MelRaceModel('femaleEars',2),
        MelRaceModel('mouth',3),
        MelRaceModel('teethLower',4),
        MelRaceModel('teethUpper',5),
        MelRaceModel('tongue',6),
        MelRaceModel('leftEye',7),
        MelRaceModel('rightEye',8),
        MelBase('NAM1','_nam1',''),
        MelBase('MNAM','_mnam',''),
        MelModel('maleTailModel'),
        MelRaceIcon('maleUpperBody',0),
        MelRaceIcon('maleLowerBody',1),
        MelRaceIcon('maleHand',2),
        MelRaceIcon('maleFoot',3),
        MelRaceIcon('maleTail',4),
        MelBase('FNAM','_fnam',''),
        MelModel('femaleTailModel'),
        MelRaceIcon('femaleUpperBody',0),
        MelRaceIcon('femaleLowerBody',1),
        MelRaceIcon('femaleHand',2),
        MelRaceIcon('femaleFoot',3),
        MelRaceIcon('femaleTail',4),
        #--Normal Entries
        MelFormidList('HNAM','hair'),
        MelFormidList('ENAM','eyes'),
        MelBase('FGGS','fggs'),
        MelBase('FGGA','fgga'),
        MelBase('FGTS','fgts'),
        MelBase('SNAM','SNAM'),
        #--Distributor for face and body entries.
        MelRaceDistributor(),
        )
    melSet.elements[-1].setMelSet(melSet)

#------------------------------------------------------------------------------
class MreScpt(MelRecord):
    """Script record."""
    type = 'SCPT' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelStruct('SCHR','5I','unk1','numRefs','compiledSize','lastIndex','scriptType'),
        #--Type: 0: Object, 1: Quest, 0x100: Magic Effect
        MelBase('SCDA','compiled'),
        MelString('SCTX','text'),
        MelGroups('vars',
            MelStruct('SLSD','6I','index','unk1','unk2','unk3','varType','unk4'),
            MelString('SCVR','name')),
        MelList('SCRV','I','refVars'),
        MelFormids('SCRO','refConstants'),
    )

#------------------------------------------------------------------------------
class MreSgst(MelRecord,MreHasEffects):
    """Sigil stone record."""
    type = 'SGST' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFull0(),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelEffects(),
        MelStruct('DATA','=BIf','uses','value','weight'),
        )

#------------------------------------------------------------------------------
class MreSlgm(MelRecord):
    """Soul gem record."""
    type = 'SLGM' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelStruct('DATA','if','value','weight'),
        MelStruct('SOUL','B',('soul',0)),
        MelStruct('SLCP','B',('capacity',1)),
        )

#------------------------------------------------------------------------------
class MreSoun(MelRecord):
    """Sound record."""
    type = 'SOUN' 
    flags = Flags(0L,Flags.getNames('randomFrequencyShift', 'playAtRandom', 
        'environmentIgnored', 'randomLocation', 'menuSound', '2d', '360LFE'))
    class MelSounSndd(MelStruct):
        """SNDD is an older version of SNDX. Allow it to read in, but not set defaults or write."""
        def setDefault(self,record): return
        def dumpData(self,record,out): return
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FNAM','soundFile'),
        MelSounSndd('SNDD','=BBcBHH','minDistance', 'maxDistance', 'freqAdjustment', 'unk04', (flags,'flags'), 'unk05'), 
        MelStruct('SNDX','=BBcBHHHbb','minDistance', 'maxDistance', 'freqAdjustment', 'unk01', (flags,'flags'), 'unk02', 'staticAtten','stopTime','startTime'),
        )

#------------------------------------------------------------------------------
class MreSpel(MelRecord,MreHasEffects):
    """Spell record."""
    type = 'SPEL' 
    class SpellFlags(Flags):
        """For SpellFlags, immuneSilence activates bits 1 AND 3."""
        def __setitem__(self,index,value):
            Flags.__setitem__(self,index,value)
            if index == 1:
                Flags.__setitem__(self,3,value)
    flags = SpellFlags(0L,Flags.getNames('noAutoCalc', 'immuneToSilence', 
        'startSpell', None,'ignoreLOS','scriptEffectAlwaysApplies','disallowAbsorbReflect'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelFull0(),
        MelStruct('SPIT','4I','spellType','cost','level',(flags,'flags',0L)),
        # spellType = 0: Spell, 1: Disease, 3: Lesser Power, 4: Ability, 5: Poison
        MelEffects(),
        )

#------------------------------------------------------------------------------
class MreStat(MelRecord):
    """Static model record."""
    type = 'STAT' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelModel(),
        )

#------------------------------------------------------------------------------
class MreTes4(MelRecord):
    """TES4 Record. File header."""
    type = 'TES4' #--Used by LoadFactory
    #--Masters array element
    class MelTes4Name(MelBase):
        def setDefault(self,record): 
            record.masters = []
        def loadData(self,record,ins,type,size,readId):
            name = Path.get(ins.readString(size,readId))
            record.masters.append(name)
        def dumpData(self,record,out):
            for name in record.masters:
                out.packSub0('MAST',name)
                out.packSub('DATA','Q',0)
    #--Data elements
    melSet = MelSet(
        MelStruct('HEDR','fiI',('version',0.8),'numRecords',('nextObject',0xCE6)),
        MelBase('OFST','ofst',), #--Obsolete?
        MelBase('DELE','dele'), #--Obsolete?
        MelString('CNAM','author','',512),
        MelString('SNAM','description','',512),
        MelTes4Name('MAST','masters'),
        MelNull('DATA'),
        )

    def getNextObject(self):
        """Gets next object index and increments it for next time."""
        self.changed = True
        self.nextObject += 1
        return (self.nextObject -1)

#------------------------------------------------------------------------------
class MreTree(MelRecord):
    """Tree record."""
    type = 'TREE' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelModel(),
        MelString('ICON','icon'),
        MelBase('SNAM','snam'),
        MelBase('CNAM','cnam'),
        MelBase('BNAM','bnam'),
        )

#------------------------------------------------------------------------------
class MreWatr(MelRecord):
    """Water record."""
    type = 'WATR' 
    flags = Flags(0L,Flags.getNames('reflective'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('TNAM','texture'),
        MelStruct('ANAM','B','opacity'),
        MelStruct('FNAM','B',(flags,'flags',0)),
        MelString('MNAM','material'),
        MelFormid('SNAM','sound'),
        MelBase('DATA','DATA'),
        MelFormidList('GNAM','relatedWaters'),
        )
    
#------------------------------------------------------------------------------
class MreWeap(MelRecord):
    """Weapon record."""
    type = 'WEAP' 
    flags = Flags(0L,Flags.getNames('notNormalWeapon'))
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('FULL','full'),
        MelModel(),
        MelString('ICON','icon'),
        MelFormid('SCRI','script'),
        MelFormid('ENAM','enchantment'),
        MelOptStruct('ANAM','H','enchantPoints'),
        MelStruct('DATA','iffIIIfH','weaponType','speed','reach',(flags,'flags',0L),
            'value','health','weight','damage'),
        #--weaponType = 0: Blade 1Hand, 1: Blade 2Hand, 2: Blunt 1Hand, 3: Blunt 2Hand, 4: Staff, 5: Bow
        )

#------------------------------------------------------------------------------
class MreWthr(MelRecord):
    """Weather record."""
    type = 'WTHR' 
    melSet = MelSet(
        MelString('EDID','eid'),
        MelString('CNAM','lowerLayer'),
        MelString('DNAM','upperLayer'),
        MelModel(),
        MelTuple('NAM0','160B','colors',[0]*160),
        MelStruct('FNAM','4f','fogDayNear','fogDayFar','fogNightNear','fogNightFar'),
        MelStruct('HNAM','14f',
            'eyeAdaptSpeed', 'blurRadius', 'blurPasses', 'emissiveMult', 
            'targetLum', 'upperLumClamp', 'brightScale', 'brightClamp', 
            'lumRampNoTex', 'lumRampMin', 'lumRampMax', 'sunlightDimmer', 
            'grassDimmer', 'treeDimmer'),
        MelStruct('DATA','15B',
            'windSpeed','lowerCloudSpeed','upperCloudSpeed','transDelta',
            'sunGlare','sunDamage','rainFadeIn','rainFadeOut','boltFadeIn',
            'boltFadeOut','boltFrequency','weatherType','boltRed','boltBlue','boltGreen'),
        MelStructs('SNAM','Ii','sounds',(FID,'sound'),'type'),
        )

# MreRecord.type_class
MreRecord.type_class = dict((a.type,a) for a in (
    MreActi, MreAlch, MreAmmo, MreAnio, MreAppa, MreArmo, MreBook, MreBsgn, 
    MreClot, MreCont, MreCrea, MreDoor, MreEfsh, MreEnch, MreEyes, MreFact, MreFlor, MreFurn, 
    MreGlob, MreGmst, MreGras, MreHair, MreIngr, MreKeym, MreLigh, MreLscr, MreLvlc, MreLvli, 
    MreLvsp, MreMgef, MreMisc, MreNpc,  MrePack, MreQust, MreRace, MreScpt, MreSgst, MreSlgm,  
    MreSoun, MreSpel, MreStat, MreTree, MreTes4, MreWatr, MreWeap, MreWthr,
    ))
MreRecord.topTypes = set(MreRecord.type_class)

# Mod Blocks, File ------------------------------------------------------------
#------------------------------------------------------------------------------
class MasterMapError(BoshError):
    """Attempt to map a formid when mapping does not exist."""
    def __init__(self,modIndex):
        BoshError.__init__(self,_('No valid mapping for mod index 0x%02X') % (modIndex,))

#------------------------------------------------------------------------------
class MasterMap:
    """Serves as a map between two sets of masters."""
    def __init__(self,inMasters,outMasters):
        """Initiation."""
        self.map = {}
        for index,master in enumerate(inMasters):
            if str(master) in outMasters:
                self.map[index] = outMasters.index(master)
            else:
                self.map[index] = -1

    def __call__(self,formid,default=-1):
        """Maps a formid from first set of masters to second. If no mapping 
        is possible, then either returns default (if defined) or raises MasterMapError."""
        if not formid: return formid
        inIndex = int(formid >> 24)
        outIndex = self.map.get(inIndex,-2)
        if outIndex >= 0:
            return (long(outIndex) << 24 ) | (formid & 0xFFFFFFL)
        elif default != -1:
            return default
        else:
            raise MasterMapError(inIndex)

#------------------------------------------------------------------------------
class MasterSet(set):
    """Set of master names."""

    def add(self,element):
        """Add an element it's not empty. Special handling for tuple."""
        if isinstance(element,tuple):
            set.add(self,element[0])
        elif element: 
            set.add(self,element)

    def getOrdered(self):
        """Returns masters in proper load order."""
        return list(modInfos.getOrdered(list(self)))

#------------------------------------------------------------------------------
class LoadFactory:
    """Factory for mod representation objects."""
    def __init__(self,keepAll,*recClasses):
        """Initialize."""
        self.keepAll = keepAll
        self.recTypes = set()
        self.topTypes = set()
        self.recFactory = {}
        for recClass in recClasses:
            self.addClass(recClass)

    def addClass(self,recClass):
        """Adds specified class."""
        if isinstance(recClass,str):
            recType = recClass
            recClass = MreRecord
        else:
            recType = recClass.type
        #--Don't replace complex class with default (MreRecord) class
        if recType in self.recFactory and recClass == MreRecord: 
            return
        self.recTypes.add(recType)
        self.recFactory[recType] = recClass
        if recType in ('CELL','REFR','ACHR','ACRE','PGRD'):
            self.topTypes.add('CELL')
            self.topTypes.add('WRLD')
        elif recType in ('WRLD','ROAD','LAND'):
            self.topTypes.add('WRLD')
        elif recType is 'INFO':
            self.topTypes.add('DIAL')
        else:
            self.topTypes.add(recType)

    def getRecClass(self,type):
        """Returns class for record type or None."""
        default = self.keepAll and MreRecord or None
        return self.recFactory.get(type,default)

    def getTopClass(self,type):
        """Returns top block class for top block type, or None."""
        if type in self.topTypes:
            if type == 'DIAL':
                return DialBlock
            else:
                return TopBlock
        elif self.keepAll:
            return Block
        else:
            return None

#------------------------------------------------------------------------------
class Block:
    """Group of records and/or subgroups. This basic implementation does not 
    support unpacking, but can report its number of records and be written."""
    def __init__(self,header,loadFactory,ins=None,unpack=False):
        """Initialize."""
        (grup, self.size, self.label, self.groupType, self.stamp) = header
        self.debug = False
        self.data = None
        self.changed = False
        self.numRecords = -1
        self.loadFactory = loadFactory
        self.inName = ins and ins.inName
        if ins: self.load(ins,unpack)

    def load(self,ins=None,unpack=False):
        """Load data from ins stream or internal data buffer."""
        if self.debug: print 'GRUP load:',self.label
        #--Read, but don't analyze.
        if not unpack:
            self.data = ins.read(self.size-20,type)
        #--Analyze ins.
        elif ins:
            self.loadData(ins, ins.tell()+self.size-20)
        #--Analyze internal buffer.
        else:
            reader = self.getReader()
            self.loadData(reader,reader.size)
            reader.close()
        #--Discard raw data?
        if unpack: 
            self.data = None
            self.setChanged()

    def loadData(self,ins,endPos):
        """Loads data from input stream. Called by load()."""
        raise AbstractError

    def setChanged(self,value=True):
        """Sets changed attribute to value. [Default = True.]"""
        self.changed = value

    def getSize(self):
        """Returns size (incuding size of any group headers)."""
        if self.changed: raise AbstractError
        return self.size

    def getNumRecords(self):
        """Returns number of records, including self, unless there's no 
        subrecords, in which case, it returns 0."""
        if self.changed:
            raise AbstractError
        elif self.numRecords > -1: #--Cached value.
            return self.numRecords
        elif not self.data: #--No data >> no records, not even self.
            self.numRecords = 0
            return self.numRecords
        else:
            numSubRecords = 0
            reader = self.getReader()
            errLabel = bush.groupTypes[self.groupType]
            while not reader.atEnd(reader.size,errLabel):
                header = reader.unpackRecHeader()
                type,size = header[0:2]
                if type == 'GRUP': size = 0
                reader.seek(size,1)
                numSubRecords += 1
            self.numRecords = numSubRecords + 1
            return self.numRecords

    def dump(self,out):
        """Dumps record header and data into output file stream."""
        if self.changed:
            raise AbstractError
        if self.numRecords == -1:
            self.getNumRecords()
        if self.numRecords > 0:
            out.pack('4sI4sII','GRUP',self.size,self.label,self.groupType,self.stamp)
            out.write(self.data)
        
    def getReader(self):
        """Returns a ModReader wrapped around self.data."""
        return ModReader(self.inName,cStringIO.StringIO(self.data))

    def convertFormids(self,mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        raise AbstractError

#------------------------------------------------------------------------------
class TopBlock(Block):
    def __init__(self,header,loadFactory,ins=None,unpack=False):
        """Initialize."""
        self.records = []
        self.id_records = {}
        Block.__init__(self,header,loadFactory,ins,unpack)

    def loadData(self,ins,endPos):
        """Loads data from input stream. Called by load()."""
        debug = self.debug
        expType = self.label
        recClass = self.loadFactory.getRecClass(expType)
        errLabel = expType+' Top Block'
        records = self.records
        while not ins.atEnd(endPos,errLabel):
            #--Get record info and handle it
            header = ins.unpackRecHeader()
            #print ' ',header
            recType = header[0]
            if recType != expType:
                #print recType,expType
                raise ModError(_('Unexpected %s record in %s group.') 
                    % (recType,expType), ins.inName)
            record = recClass(header,ins,True)
            records.append(record)
        self.setChanged()

    def getActiveRecords(self,getIgnored=True,getDeleted=True):
        """Returns non-ignored records."""
        return [record for record in self.records if not record.flags1.ignored]

    def getNumRecords(self):
        """Returns number of records, including self."""
        self.numRecords = len(self.records)
        if self.numRecords: self.numRecords += 1 #--Count self
        return self.numRecords

    def getSize(self):
        """Returns size (incuding size of any group headers)."""
        if not self.changed: 
            return self.size
        else:
            return 20 + sum((20 + record.getSize()) for record in self.records)

    def dump(self,out):
        """Dumps group header and then records."""
        if not self.changed:
            out.pack('4sI4sII','GRUP',self.size,self.label,0,self.stamp)
            out.write(self.data)
        else:
            size = self.getSize()
            if size == 20: return
            out.pack('4sI4sII','GRUP',size,self.label,0,self.stamp)
            for record in self.records:
                record.dump(out)

    def updateMasters(self,masters):
        """Updates set of master names according to masters actually used."""
        for record in self.records:
            record.updateMasters(masters)

    def convertFormids(self,mapper,toLong):
        """Converts formids between formats according to mapper. 
        toLong should be True if converting to long format or False if converting to short format."""
        for record in self.records:
            record.convertFormids(mapper,toLong)
        self.id_records.clear()

    def indexRecords(self):
        """Indexes records by formid."""
        self.id_records.clear()
        for record in self.records:
            self.id_records[record.formid] = record

    def getRecord(self,formid,default=None):
        """Gets record with corresponding id.
        If record doesn't exist, returns None."""
        if not self.records: return default
        if not self.id_records: self.indexRecords()
        return self.id_records.get(formid,default)

    def getRecordByEid(self,eid,default=None):
        """Gets record by eid, or returns default."""
        if not self.records: return default
        for record in self.records:
            if record.eid == eid:
                return record
        else:
            return default

    def setRecord(self,formid,record):
        """Adds record to record list and indexed."""
        if self.records and not self.id_records: 
            self.indexRecords()
        if record.formid != formid: raise "Formids don't match: %s, %s" % (formid, record.formid)
        if formid in self.id_records:
            oldRecord = self.id_records[formid]
            index = self.records.index(oldRecord)
            self.records[index] = record
        else:
            self.records.append(record)
        self.id_records[formid] = record

    def keepRecords(self,keepIds):
        """Keeps records with formid in set keepIds. Discards the rest."""
        self.records = [record for record in self.records if record.formid in keepIds]
        self.id_records.clear()
        self.setChanged()

#------------------------------------------------------------------------------
class DialBlock(TopBlock):
    """DIAL top block of mod file."""

    def loadData(self,ins,endPos):
        """Loads data from input stream. Called by load()."""
        expType = self.label
        recClass = self.loadFactory.getRecClass(expType)
        errLabel = expType+' Top Block'
        records = self.records
        while not ins.atEnd(endPos,errLabel):
            #--Get record info and handle it
            header = ins.unpackRecHeader()
            recType = header[0]
            if recType == expType:
                record = recClass(header,ins,True)
                records.append(record)
            elif recType == 'GRUP':
                (recType,size,label,groupType,stamp) = header
                if groupType == 7:
                    record.infoStamp = stamp
                    infoClass = self.loadFactory.getRecClass('INFO')
                    record.loadInfos(ins,ins.tell()+size-20,infoClass)
                else:
                    raise ModError('Unexpected subgroup %d in DIAL group.' % (groupType,))
            else:
                raise ModError(_('Unexpected %s record in %s group.') 
                    % (recType,expType), ins.inName)
        self.setChanged()

    def getSize(self):
        """Returns size of records plus group and record headers."""
        if not self.changed: 
            return self.size
        size = 20
        for record in self.records:
            size += 20 + record.getSize()
            if record.infos: 
                size += 20 + sum(20+info.getSize() for info in record.infos)
        return size

    def getNumRecords(self):
        """Returns number of records, including self plus info records."""
        numRecords = len(self.records)
        if self.numRecords: 
            numRecords += 1 #--Count self
        for record in self.records:
            if record.infos:
                numRecords += len(record.infos) + 1
        self.numRecords = numRecords
        return numRecords

##------------------------------------------------------------------------------
#class CellBlock(Block):
#    cell = None
#    persistent = []
#    distant = []
#    temp = []
#    pgrd = None
#    land = None
#
##------------------------------------------------------------------------------
#class WorldBlock(Block):
#    world = None
#    cellBlocks = {} #cellBlock = cellBlocks[(block,subBlock)]
#
##------------------------------------------------------------------------------
#class CellTopBlock(Block):
#    cellBlocks = {} #cellBlock = cellBlocks[(block,subBlock)]
#
##------------------------------------------------------------------------------
#class WorldTopBlock(Block):
#    worldBlocks = {} #cellBlock = cellBlocks[(block,subBlock)]

#------------------------------------------------------------------------------
class ModFile:
    """TES4 file representation."""
    def __init__(self, fileInfo,loadFactory=None):
        """Initialize."""
        self.fileInfo = fileInfo
        self.loadFactory = loadFactory or LoadFactory(True)
        #--Variables to load
        self.tes4 = MreTes4(('TES4',0,0,0,0))
        self.tes4.setChanged()
        self.tops = {} #--Top groups.
        self.topsSkipped = set() #--Types skipped
        self.longFormids = False

    def __getattr__(self,topType):
        """Returns top block of specified topType, creating it, if necessary."""
        if topType in self.tops:
            return self.tops[topType]
        elif topType in bush.topTypes:
            topClass = self.loadFactory.getTopClass(topType)
            self.tops[topType] = topClass(('GRUP',0,topType,0,0),self.loadFactory)
            self.tops[topType].setChanged()
            return self.tops[topType]
        else:
            raise ArgumentError(_('Invalid top group type: ')+topType)
        
    def load(self,unpack=False,progress=None):
        """Load file."""
        progress = progress or Progress()
        #--Header
        ins = ModReader(self.fileInfo.name,file(self.fileInfo.getPath(),'rb'))
        header = ins.unpackRecHeader()
        self.tes4 = MreTes4(header,ins,True)
        #--Raw data read
        while not ins.atEnd():
            #--Get record info and handle it
            (type,size,label,groupType,stamp) = header = ins.unpackRecHeader()
            if type != 'GRUP' or groupType != 0:
                raise ModError(self.fileInfo.name,_('Improperly grouped file.'))
            topClass = self.loadFactory.getTopClass(label)
            if topClass:
                self.tops[label] = topClass(header,self.loadFactory)
                self.tops[label].load(ins,unpack and (topClass != Block))
            else:
                self.topsSkipped.add(label)
                ins.seek(size-20,1,type+'.'+label)
        #--Done Reading
        ins.close()

    def load_unpack(self):
        """Unpacks blocks."""
        for type in bush.topTypes:
            if type in self.tops and type in self.loadFactory.topTypes:
                self.tops[type].load(None,True)

    def load_UI(self):
        """Convenience function. Loads, then unpacks, then indexes."""
        self.load()
        self.load_unpack()
        #self.load_index()

    def askSave(self,hasChanged=True):
        """CLI command. If hasSaved, will ask if user wants to save the file, 
        and then save if the answer is yes. If hasSaved == False, then does nothing."""
        if not hasChanged: return
        fileName = self.fileInfo.name
        if re.match(r'\s*[yY]',raw_input('\nSave changes to '+fileName+' [y/n]?: ')):
            self.safeSave()
            print fileName,'saved.'
        else:
            print fileName,'not saved.'

    def safeSave(self):
        """Save data to file safely."""
        self.fileInfo.makeBackup()
        filePath = self.fileInfo.getPath()
        tempPath = filePath+'.tmp'
        self.save(tempPath)
        renameFile(tempPath,filePath)
        self.fileInfo.setMTime()
        self.fileInfo.extras.clear()

    def save(self,outPath=None):
        """Save data to file. 
        outPath -- Path of the output file to write to. Defaults to original file path."""
        if (not self.loadFactory.keepAll): raise StateError(_("Insufficient data to write file."))
        outPath = outPath or self.fileInfo.getPath()
        out = ModWriter(outPath.open('wb'))
        #--Mod Record
        self.tes4.setChanged()
        self.tes4.numRecords = sum(block.getNumRecords() for block in self.tops.values())
        self.tes4.getSize()
        self.tes4.dump(out)
        #--Blocks
        for type in bush.topTypes:
            if type in self.tops:
                self.tops[type].dump(out)
        out.close()

    def getLongMapper(self):
        """Returns a mapping function to map short formids to long formids."""
        masters = self.tes4.masters+[self.fileInfo.name]
        maxMaster = len(masters)-1
        def mapper(formid):
            if formid == None: return None
            if isinstance(formid,tuple): return formid
            mod,object = int(formid >> 24),int(formid & 0xFFFFFFL)
            return (masters[min(mod,maxMaster)],object)
        return mapper

    def getShortMapper(self):
        """Returns a mapping function to map long formids to short formids."""
        masters = self.tes4.masters+[self.fileInfo.name]
        indices = dict([(name,index) for index,name in enumerate(masters)])
        def mapper(formid):
            if formid == None: return None
            modName,object = formid
            mod = indices[modName]
            return (long(mod) << 24 ) | long(object)
        return mapper

    def convertToLongFormids(self,types=None):
        """Convert formids to long format (modname,objectindex)."""
        mapper = self.getLongMapper()
        if types == None: types = self.tops.keys()
        for type in types:
            if type in self.tops:
                self.tops[type].convertFormids(mapper,True)
        #--Done
        self.longFormids = True

    def convertToShortFormids(self):
        """Convert formids to short (numeric) format."""
        mapper = self.getShortMapper()
        for type in self.tops:
            self.tops[type].convertFormids(mapper,False)
        #--Done
        self.longFormids = False

    def getMastersUsed(self):
        """Updates set of master names according to masters actually used."""
        if not self.longFormids: raise StateError("ModFile formids not in long form.")
        masters = MasterSet([Path.get('Oblivion.esm')]) #--Not so good for TCs. Fix later.
        for block in self.tops.values():
            block.updateMasters(masters)
        return masters.getOrdered()

# Save I/O --------------------------------------------------------------------
#------------------------------------------------------------------------------
class SaveFileError(FileError):
    """TES4 Save File Error: File is corrupted."""
    pass

# Save Change Records ---------------------------------------------------------
class SreNPC(object):
    """NPC change record."""
    __slots__ = ('form','health','attributes','acbs','spells','factions','full','ai','skills','modifiers')
    flags = Flags(0L,Flags.getNames(
        (0,'form'),
        (2,'health'),
        (3,'attributes'),
        (4,'acbs'),
        (5,'spells'),
        (6,'factions'),
        (7,'full'),
        (8,'ai'),
        (9,'skills'), 
        (28,'modifiers'),
        ))

    class ACBS(object):
        __slots__ = ['flags','baseSpell','fatigue','barterGold','level','calcMin','calcMax']

    def __init__(self,flags=0,data=None):
        """Initialize."""
        for attr in self.__slots__:
            setattr(self,attr,None)
        if data: self.load(flags,data)

    def getDefault(self,attr):
        """Returns a default version. Only supports acbs."""
        assert(attr == 'acbs')
        acbs = SreNPC.ACBS()
        (acbs.flags, acbs.baseSpell, acbs.fatigue, acbs.barterGold, acbs.level,
                acbs.calcMin, acbs.calcMax) = (0,0,0,0,1,0,0)
        acbs.flags = MreNpc.flags(acbs.flags)
        return acbs

    def load(self,flags,data):
        """Loads variables from data."""
        ins = cStringIO.StringIO(data)
        def unpack(format,size):
            return struct.unpack(format,ins.read(size))
        flags = SreNPC.flags(flags)
        if flags.form:
            self.form, = unpack('I',4)
        if flags.attributes:
            self.attributes = list(unpack('8B',8))
        if flags.acbs:
            acbs = self.acbs = SreNPC.ACBS()
            (acbs.flags, acbs.baseSpell, acbs.fatigue, acbs.barterGold, acbs.level,
                acbs.calcMin, acbs.calcMax) = unpack('=I3Hh2H',16)
            acbs.flags = MreNpc.flags(acbs.flags)
        if flags.factions:
            self.factions = []
            num, = unpack('H',2)
            for count in range(num):
                self.factions.append(unpack('=Ib',5))
        if flags.spells:
            num, = unpack('H',2)
            self.spells = list(unpack('%dI' % (num,),4*num))
        if flags.ai:
            self.ai = ins.read(4)
        if flags.health:
            self.health, = unpack('I',4)
        if flags.modifiers:
            num, = unpack('H',2)
            self.modifiers = []
            for count in range(num):
                self.modifiers.append(unpack('=Bf',5))
        if flags.full:
            size, = unpack('B',1)
            self.full = ins.read(size)
        if flags.skills:
            self.skills = list(unpack('21B',21))
        #--Done
        ins.close()

    def getFlags(self):
        """Returns current flags set."""
        flags = SreNPC.flags()
        for attr in SreNPC.__slots__:
            setattr(flags,attr,getattr(self,attr) != None)
        return int(flags)

    def getData(self):
        """Returns self.data."""
        out = cStringIO.StringIO()
        def pack(format,*args):
            out.write(struct.pack(format,*args))
        #--Form 
        if self.form != None:
            pack('I',self.form)
        #--Attributes
        if self.attributes != None:
            pack('8B',*self.attributes)
        #--Acbs
        if self.acbs != None:
            acbs = self.acbs
            pack('=I3Hh2H',int(acbs.flags), acbs.baseSpell, acbs.fatigue, acbs.barterGold, acbs.level,
                acbs.calcMin, acbs.calcMax)
            #deprint(int(acbs.flags),acbs.flags.hex(),acbs.flags.getTrueAttrs())
        #--Factions
        if self.factions != None:
            pack('H',len(self.factions))
            for faction in self.factions:
                pack('=Ib',*faction)
        #--Spells
        if self.spells != None:
            num = len(self.spells)
            pack('H',num)
            pack('%dI' % (num,),*self.spells)
        #--AI Data
        if self.ai != None:
            out.write(self.ai)
        #--Health
        if self.health != None:
            pack('I',self.health)
        #--Modifiers
        if self.modifiers != None:
            pack('H',len(self.modifiers))
            for modifier in self.modifiers:
                pack('=Bf',*modifier)
        #--Full
        if self.full != None:
            pack('B',len(self.full))
            out.write(self.full)
        #--Skills
        if self.skills != None:
            pack('21B',*self.skills)
        #--Done
        return out.getvalue()

    def getTuple(self,formid,version):
        """Returns record as a change record tuple."""
        return (formid,35,self.getFlags(),version,self.getData())

    def dumpText(self,saveFile):
        """Returns informal string representation of data."""
        buff = cStringIO.StringIO()
        formids = saveFile.formids
        if self.form != None: 
            buff.write('Form:\n  %d' % (self.form,))
        if self.attributes != None: 
            buff.write('Attributes\n  strength %3d\n  intelligence %3d\n  willpower %3d\n  agility %3d\n  speed %3d\n  endurance %3d\n  personality %3d\n  luck %3d\n' % tuple(self.attributes))
        if self.acbs != None:
            buff.write('ACBS:\n')
            for attr in SreNPC.ACBS.__slots__:
                buff.write('  '+attr+' '+`getattr(self.acbs,attr)`+'\n')
        if self.factions != None: 
            buff.write('Factions:\n')
            for faction in self.factions:
                buff.write('  %8X %2X\n' % (formids[faction[0]],faction[1]))
        if self.spells != None: 
            buff.write('Spells:\n')
            for spell in self.spells:
                buff.write('  %8X\n' % (formids[spell],))
        if self.ai != None: 
            buff.write('AI:\n  ' + self.ai + '\n')
        if self.health != None: 
            buff.write('Health\n  '+`self.health`+'\n')
        if self.modifiers != None: 
            buff.write('Modifiers:\n')
            for modifier in self.modifiers:
                buff.write('  %s\n' % (`modifier`,))
        if self.full != None: 
            buff.write('Full:\n  '+`self.full`+'\n')
        if self.skills != None: 
            buff.write('Skills:\n  armorer %3d\n  athletics %3d\n  blade %3d\n  block %3d\n  blunt %3d\n  handToHand %3d\n  heavyArmor %3d\n  alchemy %3d\n  alteration %3d\n  conjuration %3d\n  destruction %3d\n  illusion %3d\n  mysticism %3d\n  restoration %3d\n  acrobatics %3d\n  lightArmor %3d\n  marksman %3d\n  mercantile %3d\n  security %3d\n  sneak %3d\n  speechcraft  %3d\n' % tuple(self.skills))
        return buff.getvalue()

# Save File -------------------------------------------------------------------
#------------------------------------------------------------------------------
class SaveHeader:
    """Represents selected info from a Tes4SaveGame file."""
    def __init__(self,path=None):
        """Initialize."""
        self.pcName = None
        self.pcLocation = None
        self.gameDays = 0
        self.gameTicks = 0
        self.pcLevel = 0
        self.masters = []
        self.image = None
        if path: self.load(path)

    def load(self,path):
        """Extract info from save file."""
        ins = path.open('rb')
        try:
            #--Header 
            ins.seek(34)
            headerSize, = struct.unpack('I',ins.read(4))
            #posMasters = 38 + headerSize
            #--Name, location
            ins.seek(38+4)
            size, = struct.unpack('B',ins.read(1))
            self.pcName = cstrip(ins.read(size))
            self.pcLevel, = struct.unpack('H',ins.read(2))
            size, = struct.unpack('B',ins.read(1))
            self.pcLocation = cstrip(ins.read(size))
            #--Image Data
            self.gameDays,self.gameTicks,self.gameTime,ssSize,ssWidth,ssHeight = struct.unpack('=fI16s3I',ins.read(36))
            ssData = ins.read(3*ssWidth*ssHeight)
            self.image = (ssWidth,ssHeight,ssData)
            #--Masters
            #ins.seek(posMasters)
            del self.masters[:]
            numMasters, = struct.unpack('B',ins.read(1))
            for count in range(numMasters):
                size, = struct.unpack('B',ins.read(1))
                self.masters.append(Path.get(ins.read(size)))
        #--Errors
        except:
            raise SaveFileError(path.tail(),_('File header is corrupted..'))
        #--Done
        ins.close()

    def writeMasters(self,path):
        """Rewrites masters of existing save file."""
        if not path.exists():
            raise SaveFileError(path.head(),_('File does not exist.'))
        tmpPath = path+'.tmp'
        ins = path.open('rb')
        out = tmpPath.open('wb')
        def unpack(format,size):
            return struct.unpack(format,ins.read(size))
        def pack(format,*args):
            out.write(struct.pack(format,*args))
        #--Header
        out.write(ins.read(34))
        #--SaveGameHeader
        size, = unpack('I',4)
        pack('I',size)
        out.write(ins.read(size))
        #--Skip old masters
        numMasters, = unpack('B',1)
        for count in range(numMasters):
            size, = unpack('B',1)
            ins.seek(size,1)
        #--Write new masters
        pack('B',len(self.masters))
        for master in self.masters:
            pack('B',len(master))
            out.write(master)
        #--Formids Address
        offset = out.tell() - ins.tell()
        formIdsAddress, = unpack('I',4)
        pack('I',formIdsAddress+offset)
        #--Copy remainder
        while True:
            buffer= ins.read(0x5000000)
            if not buffer: break
            out.write(buffer)
        #--Cleanup
        ins.close()
        out.close()
        renameFile(tmpPath,path)

#------------------------------------------------------------------------------
class SaveFile:
    """Represents a Tes4 Save file."""
    recordFlags = Flags(0L,Flags.getNames(
        'form','baseid','moved','havocMoved','scale','allExtra','lock','owner','unk8','unk9',
        'mapMarkerFlags','hadHavokMoveFlag','unk12','unk13','unk14','unk15',
        'emptyFlag','droppedItem','doorDefaultState','doorState','teleport',
        'extraMagic','furnMarkers','oblivionFlag','movementExtra','animation',
        'script','inventory','created','unk29','enabled'))

    def __init__(self,saveInfo=None,canSave=True):
        """Initialize."""
        self.fileInfo = saveInfo
        self.canSave = canSave
        #--File Header, Save Game Header
        self.header = None
        self.gameHeader = None
        self.pcName = None
        #--Masters
        self.masters = []
        #--Global
        self.globals = []
        self.created = []
        self.preGlobals = None #--Pre-records, pre-globals
        self.preCreated = None #--Pre-records, pre-created
        self.preRecords = None #--Pre-records, pre
        #--Records, temp effects, formids, worldspaces
        self.records = [] #--(formid,recType,flags,version,data)
        self.formid_recNum = None
        self.tempEffects = None
        self.formids = None
        self.irefs = {}  #--iref = self.irefs[formid]
        self.worldSpaces = None

    def load(self,progress=None):
        """Extract info from save file."""
        import array
        path = self.fileInfo.getPath()
        ins = StructFile(path,'rb')
        #--Progress
        fileName = self.fileInfo.name
        progress = progress or Progress()
        progress.setFull(self.fileInfo.size)
        #--Header 
        progress(0,_('Reading Header.'))
        self.header = ins.read(34)
        
        #--Save Header, pcName
        gameHeaderSize, = ins.unpack('I',4)
        self.saveNum,pcNameSize, = ins.unpack('=IB',5)
        self.pcName = cstrip(ins.read(pcNameSize))
        self.postNameHeader = ins.read(gameHeaderSize-5-pcNameSize)
        
        #--Masters
        del self.masters[:]
        numMasters, = ins.unpack('B',1)
        for count in range(numMasters):
            size, = ins.unpack('B',1)
            self.masters.append(Path.get(ins.read(size)))
        
        #--Pre-Records copy buffer
        def insCopy(buff,size,backSize=0):
            if backSize: ins.seek(-backSize,1)
            buff.write(ins.read(size+backSize))

        #--"Globals" block
        formIdsPointer,recordsNum = ins.unpack('2I',8)
        #--Pre-globals
        self.preGlobals = ins.read(8*4)
        #--Globals
        globalsNum, = ins.unpack('H',2)
        self.globals = [ins.unpack('If',8) for num in xrange(globalsNum)]
        #--Pre-Created (Class, processes, spectator, sky)
        buff = cStringIO.StringIO()
        for count in range(4):
            size, = ins.unpack('H',2)
            insCopy(buff,size,2)
        insCopy(buff,4) #--Supposedly part of created info, but sticking it here since I don't decode it.
        self.preCreated = buff.getvalue()
        #--Created (ALCH,SPEL,ENCH,WEAP,CLOTH,ARMO, etc.?)
        modReader = ModReader(self.fileInfo.name,ins)
        createdNum, = ins.unpack('I',4)
        for count in xrange(createdNum):
            progress(ins.tell(),_('Reading created...'))
            header = ins.unpack('4s4I',20)
            self.created.append(MreRecord(header,modReader))
        #--Pre-records: Quickkeys, reticule, interface, regions
        buff = cStringIO.StringIO()
        for count in range(4):
            size, = ins.unpack('H',2)
            insCopy(buff,size,2)
        self.preRecords = buff.getvalue()

        #--Records
        for count in xrange(recordsNum):
            progress(ins.tell(),_('Reading records...'))
            (formid,recType,flags,version,size) = ins.unpack('=IBIBH',12)
            data = ins.read(size)
            self.records.append((formid,recType,flags,version,data))

        #--Temp Effects, formids, worldids
        progress(ins.tell(),_('Reading formids, worldids...'))
        size, = ins.unpack('I',4)
        self.tempEffects = ins.read(size)
        #--Formids
        num, = ins.unpack('I',4)
        self.formids = array.array('I')
        self.formids.fromfile(ins,num)
        for iref,formid in enumerate(self.formids):
            self.irefs[formid] = iref

        #--WorldSpaces
        num, = ins.unpack('I',4)
        self.worldSpaces = array.array('I')
        self.worldSpaces.fromfile(ins,num)
        #--Done
        ins.close()
        progress(progress.full,_('Finished reading.'))

    def save(self,outPath=None,progress=None):
        """Save data to file. 
        outPath -- Path of the output file to write to. Defaults to original file path."""
        if (not self.canSave): raise StateError(_("Insufficient data to write file."))
        outPath = outPath or self.saveInfo.getPath()
        out = outPath.open('wb')
        def pack(format,*data):
            out.write(struct.pack(format,*data))
        #--Progress
        fileName = self.fileInfo.name
        progress = progress or Progress()
        progress.setFull(self.fileInfo.size)
        #--Header
        progress(0,_('Writing Header.'))
        out.write(self.header)
        #--Save Header
        pack('=IIB',5+len(self.pcName)+1+len(self.postNameHeader),
            self.saveNum, len(self.pcName)+1)
        out.write(self.pcName+'\x00')
        out.write(self.postNameHeader)
        #--Masters
        pack('B',len(self.masters))
        for master in self.masters:
            pack('B',len(master))
            out.write(master)
        #--FormIds Pointer, num records
        formIdsPointerPos = out.tell()
        pack('I',0) #--Temp. Will write real value later.
        pack('I',len(self.records))
        #--Pre-Globals
        out.write(self.preGlobals)
        #--Globals
        pack('H',len(self.globals))
        for iref,value in self.globals:
            pack('If',iref,value)
        #--Pre-Created
        out.write(self.preCreated)
        #--Created
        progress(0.1,_('Writing created.'))
        modWriter = ModWriter(out)
        pack('I',len(self.created))
        for record in self.created:
            record.dump(modWriter)
        #--Pre-records
        out.write(self.preRecords)
        #--Records, temp effects, formids, worldspaces
        progress(0.2,_('Writing records.'))
        for formid,recType,flags,version,data in self.records:
            pack('=IBIBH',formid,recType,flags,version,len(data))
            out.write(data)
        #--Temp Effects, formids, worldids
        pack('I',len(self.tempEffects))
        out.write(self.tempEffects)
        #--Formids
        progress(0.9,_('Writing formids, worldids.'))
        formIdsPos = out.tell()
        out.seek(formIdsPointerPos)
        pack('I',formIdsPos)
        out.seek(formIdsPos)
        pack('I',len(self.formids))
        self.formids.tofile(out)
        #--Worldspaces
        pack('I',len(self.worldSpaces))
        self.worldSpaces.tofile(out)
        #--Done
        progress(1.0,_('Writing complete.'))
        out.close()

    def safeSave(self,progress=None):
        """Save data to file safely."""
        self.fileInfo.makeBackup()
        filePath = self.fileInfo.getPath()
        tempPath = filePath+'.tmp'
        self.save(tempPath,progress)
        renameFile(tempPath,filePath)
        self.fileInfo.setMTime()

    def addMaster(self,master):
        """Adds master to masters list."""
        if master not in self.masters:
            self.masters.append(master)

    def indexRecords(self):
        """Fills out self.formid_recNum."""
        self.formid_recNum = dict((entry[0],index) for index,entry in enumerate(self.records))

    def getRecord(self,formid,default=None):
        """Returns recNum and record with corresponding formid."""
        if self.formid_recNum == None: self.indexRecords()
        recNum = self.formid_recNum.get(formid)
        if recNum == None: 
            return default
        else: 
            return self.records[recNum]

    def setRecord(self,record):
        """Sets records where record = (formid,recType,flags,version,data)."""
        if self.formid_recNum == None: self.indexRecords()
        formid = record[0]
        recNum = self.formid_recNum.get(formid,-1)
        if recNum == -1:
            self.records.append(record)
            self.formid_recNum[formid] = len(self.records)-1
        else:
            self.records[recNum] = record

    def getShortMapper(self):
        """Returns a mapping function to map long formids to short formids."""
        indices = dict([(name,index) for index,name in enumerate(self.masters)])
        def mapper(formid):
            if formid == None: return None
            modName,object = formid
            mod = indices[modName]
            return (long(mod) << 24 ) | long(object)
        return mapper

    def getFormid(self,iref,default=None):
        """Returns formid corresponding to iref."""
        if not iref: return default
        if iref >> 24 == 0xFF: return iref
        if iref >= len(self.formids): raise 'IRef from Mars.'
        return self.formids[iref]

    def getIref(self,formid):
        """Returns iref corresponding to formid, creating it if necessary."""
        iref = self.irefs.get(formid,-1)
        if iref < 0: 
            self.formids.append(formid)
            iref = self.irefs[formid] = len(self.formids) - 1
        return iref

    #--------------------------------------------------------------------------
    def logStats(self,log=None):
        """Print stats to log."""
        log = log or Log()
        doLostChanges = False
        doUnknownTypes = False
        def getMaster(modIndex):
            if modIndex < len(self.masters):
                return self.masters[modIndex]
            elif modIndex == 0xFF:
                return self.fileInfo.name
            else:
                return _('Missing Master ')+hex(modIndex)
        #--ABomb
        (tesClassSize,abombCounter,abombFloat) = self.getAbomb()
        log.setHeader(_('Abomb Counter'))
        log(_('  Integer:\t0x%08X') % (abombCounter,))
        log(_('  Float:\t%.2f') % (abombFloat,))
        #--FBomb
        log.setHeader(_('Fbomb Counter'))
        log(_('  Next in-game object: %08X') % struct.unpack('I',self.preGlobals[:4]))
        #--Array Sizes
        log.setHeader('Array Sizes')
        log('  %d\t%s' % (len(self.created),_('Created Items')))
        log('  %d\t%s' % (len(self.records),_('Records')))
        log('  %d\t%s' % (len(self.formids),_('FormIds')))
        #--Created Types
        log.setHeader(_('Created Items'))
        createdHisto = {}
        id_created = {}
        for citem in self.created:
            count,size = createdHisto.get(citem.type,(0,0))
            createdHisto[citem.type] =  (count + 1,size + citem.size)
            id_created[citem.formid] = citem
        for type in sorted(createdHisto.keys()):
            count,size = createdHisto[type]
            log('  %d\t%d kb\t%s' % (count,size/1024,type))
        #--Formids
        lostRefs = 0
        idHist = [0]*256
        for formid in self.formids:
            if formid == 0: 
                lostRefs += 1
            else:
                idHist[formid >> 24] += 1
        #--Change Records
        changeHisto = [0]*256
        modHisto = [0]*256
        typeModHisto = {}
        knownTypes = set(bush.saveRecTypes.keys())
        lostChanges = {}
        objRefBases = {}
        objRefNullBases = 0
        formids = self.formids
        for record in self.records:
            formid,type,flags,version,data = record
            if formid ==0xFEFFFFFF: continue #--Ignore intentional(?) extra formid added by patch.
            mod = formid >> 24
            if type not in typeModHisto:
                typeModHisto[type] = modHisto[:]
            typeModHisto[type][mod] += 1
            changeHisto[mod] += 1
            #--Lost Change?
            if doLostChanges and mod == 255 and not (48 <= type <= 51) and formid not in id_created:
                lostChanges[formid] = record
            #--Unknown type?
            if doUnknownTypes and type not in knownTypes:
                if mod < 255:
                    print type,hex(formid),getMaster(mod)
                    knownTypes.add(type)
                elif formid in id_created:
                    print type,hex(formid),id_created[formid].type
                    knownTypes.add(type)
            #--Obj ref parents
            if type == 49 and mod == 255 and (flags & 2):
                iref, = struct.unpack('I',data[4:8])
                count,cumSize = objRefBases.get(iref,(0,0))
                count += 1
                cumSize += len(data) + 12
                objRefBases[iref] = (count,cumSize)
                if iref >> 24 != 255 and formids[iref] == 0:
                    objRefNullBases += 1
        saveRecTypes = bush.saveRecTypes
        #--Formids log
        log.setHeader(_('FormIds'))
        log('  Refed\tChanged\tMI    Mod Name')
        log('  %d\t\t     Lost Refs (FormId == 0)' % (lostRefs))
        for modIndex,(irefed,changed) in enumerate(zip(idHist,changeHisto)):
            if irefed or changed: 
                log('  %d\t%d\t%-3d   %s' % (irefed,changed,modIndex,getMaster(modIndex)))
        #--Lost Changes
        if lostChanges:
            log.setHeader(_('LostChanges'))
            for id in sorted(lostChanges.keys()):
                type = lostChanges[id][1]
                log(hex(id)+saveRecTypes.get(type,`type`))
        for type in sorted(typeModHisto.keys()):
            modHisto = typeModHisto[type]
            log.setHeader('%d %s' % (type,saveRecTypes.get(type,_('Unknown')),))
            for modIndex,count in enumerate(modHisto):
                if count: log('  %d\t%s' % (count,getMaster(modIndex)))
            log('  %d\tTotal' % (sum(modHisto),))
        objRefBases = dict((key,value) for key,value in objRefBases.items() if value[0] > 100)
        log.setHeader(_('New ObjectRef Bases'))
        if objRefNullBases:
            log(' Null Bases: '+`objRefNullBases`)
        if objRefBases:
            log(_(' Count IRef     BaseId'))
            for iref in sorted(objRefBases.keys()):
                count,cumSize = objRefBases[iref]
                if iref >> 24 == 255:
                    parentid = iref
                else:
                    parentid = self.formids[iref]
                log('%6d %08X %08X %6d kb' % (count,iref,parentid,cumSize/1024))

    def findBloating(self,progress=None):
        """Analyzes file for bloating. Returns (createdCounts,nullRefCount)."""
        nullRefCount = 0
        createdCounts = {}
        progress = progress or Progress()
        progress.setFull(len(self.created)+len(self.records))
        #--Created objects
        progress(0,_('Scanning created objects'))
        for citem in self.created:
            if 'full' in citem.__dict__:
                full = citem.__dict__['full']
            else:
                full = citem.getSubString('FULL')
            if full:
                typeFull = (citem.type,full)
                count = createdCounts.get(typeFull,0)
                createdCounts[typeFull] = count + 1
            progress.plus()
        for key in createdCounts.keys()[:]:
            minCount = (50,100)[key[0] == 'ALCH']
            if createdCounts[key] < minCount:
                del createdCounts[key]
        #--Change records
        progress(len(self.created),_('Scanning change records.'))
        formids = self.formids
        for record in self.records:
            formid,recType,flags,version,data = record
            if recType == 49 and formid >> 24 == 0xFF and (flags & 2):
                iref, = struct.unpack('I',data[4:8])
                if iref >> 24 != 0xFF and formids[iref] == 0:
                    nullRefCount += 1
            progress.plus()
        return (createdCounts,nullRefCount)        

    def removeBloating(self,uncreateKeys,removeNullRefs=True,progress=None):
        """Removes duplicated created items and null refs."""
        numUncreated = numUnCreChanged = numUnNulled = 0
        progress = progress or Progress()
        progress.setFull((len(uncreateKeys) and len(self.created))+len(self.records))
        uncreated = set()
        #--Uncreate
        if uncreateKeys: 
            progress(0,_('Scanning created objects'))
            kept = []
            for citem in self.created:
                if 'full' in citem.__dict__:
                    full = citem.__dict__['full']
                else:
                    full = citem.getSubString('FULL')
                if full and (citem.type,full) in uncreateKeys:
                    uncreated.add(citem.formid)
                    numUncreated += 1
                else:
                    kept.append(citem) 
                progress.plus()
            self.created = kept
        #--Change records
        progress(progress.state,_('Scanning change records.'))
        formids = self.formids
        kept = []
        for record in self.records:
            formid,recType,flags,version,data = record
            if formid in uncreated:
                numUnCreChanged += 1
            elif removeNullRefs and recType == 49 and formid >> 24 == 0xFF and (flags & 2):
                iref, = struct.unpack('I',data[4:8])
                if iref >> 24 != 0xFF and formids[iref] == 0:
                    numUnNulled += 1
                else:
                    kept.append(record)
            else:
                kept.append(record)
            progress.plus()
        self.records = kept
        return (numUncreated,numUnCreChanged,numUnNulled)

    def getCreated(self,*types):
        """Return created items of specified type(s)."""
        types = set(types)
        created = [citem for citem in self.created if citem.type in types]
        created.sort(key=lambda a: a.formid)
        created.sort(key=lambda a: a.type)
        return created

    def getAbomb(self):
        """Get's animation slowing counter(?) value."""
        data = self.preCreated
        tesClassSize, = struct.unpack('H',data[:2])
        abombBytes = data[2+tesClassSize-4:2+tesClassSize]
        abombCounter, = struct.unpack('I',abombBytes)
        abombFloat, = struct.unpack('f',abombBytes)
        return (tesClassSize,abombCounter,abombFloat)

    def setAbomb(self,value=0x41000000):
        """Resets abomb counter to specified value."""
        data = self.preCreated
        tesClassSize, = struct.unpack('H',data[:2])
        if tesClassSize < 4: return
        buff = cStringIO.StringIO()
        buff.write(data)
        buff.seek(2+tesClassSize-4)
        buff.write(struct.pack('I',value))
        self.preCreated = buff.getvalue()
        buff.close()

# File System -----------------------------------------------------------------
#--------------------------------------------------------------------------------
class BsaFile:
    """Represents a BSA archive file."""

    @staticmethod
    def getHash(fileName):
        """Returns tes4's two hash values for filename.
        Based on Timeslips code with cleanup and pythonization."""
        root,ext = os.path.splitext(fileName.lower())
        #--Hash1
        chars = map(ord,root)
        hash1 = chars[-1] | ((len(chars)>2 and chars[-2]) or 0)<<8 | len(chars)<<16 | chars[0]<<24
        if   ext == '.kf':  hash1 |= 0x80
        elif ext == '.nif': hash1 |= 0x8000
        elif ext == '.dds': hash1 |= 0x8080
        elif ext == '.wav': hash1 |= 0x80000000
        #--Hash2
        uintMask, hash2, hash3 = 0xFFFFFFFF, 0, 0
        for char in chars[1:-2]:
            hash2 = ((hash2 * 0x1003F) + char ) & uintMask
        for char in map(ord,ext):
            hash3 = ((hash3 * 0x1003F) + char ) & uintMask
        hash2 = (hash2 + hash3) & uintMask
        #--Done
        return (hash2<<32) + hash1

    #--Instance Methods ------------------------------------------------------
    def __init__(self,path):
        """Initialize."""
        self.path = path
        self.folderInfos = None

    def scan(self):
        """Reports on contents."""
        ins = StructFile(self.path,'rb')
        #--Header
        ins.seek(4*4)
        (self.folderCount,self.fileCount,lenFolderNames,lenFileNames,fileFlags) = ins.unpack('5I',20)
        #--FolderInfos (Initial)
        folderInfos = self.folderInfos = []
        for index in range(self.folderCount):
            hash,subFileCount,offset = ins.unpack('Q2I',16)
            folderInfos.append([hash,subFileCount,offset])
        #--Update folderInfos
        for index,folderInfo in enumerate(folderInfos):
            fileInfos = []
            folderName = cstrip(ins.read(ins.unpack('B',1)[0]))
            folderInfos[index].extend((folderName,fileInfos))
            for index in range(folderInfo[1]):
                filePos = ins.tell()
                hash,size,offset = ins.unpack('Q2I',16)
                fileInfos.append([hash,size,offset,'',filePos])
        #--File Names
        fileNames = ins.read(lenFileNames)
        fileNames = fileNames.split('\x00')[:-1]
        namesIter = iter(fileNames)
        for folderInfo in folderInfos:
            fileInfos = folderInfo[-1]
            for index,fileInfo in enumerate(fileInfos):
                fileInfo[3] = namesIter.next()
        #--Done
        ins.close()

    def report(self,printAll=False):
        """Report on contents."""
        folderInfos = self.folderInfos
        getHash = BsaFile.getHash
        print self.folderCount,self.fileCount,sum(len(info[-1]) for info in folderInfos)
        for folderInfo in folderInfos:
            printOnce = folderInfo[-2]
            for fileInfo in folderInfo[-1]:
                hash,fileName = fileInfo[0],fileInfo[3]
                trueHash = getHash(fileName)

    def firstBackup(self,progress):
        """Make first backup, just in case!"""
        backupDir = modInfos.dir.join('Bash','Backups')
        backupDir.makedirs()
        backup = backupDir.join(self.path.tail()+'f')
        if not backup.exists():
            progress(0,_("Backing up BSA file. This will take a while..."))
            self.path.copyfile(backup)

    def updateAIText(self,files=None):
        """Update aiText with specified files. (Or remove, if files == None.)"""
        aiPath = dirs['app'].join('ArchiveInvalidation.txt')
        if not files:
            aiPath.remove()
            return
        #--Archive invalidation
        aiText = re.sub(r'\\','/','\n'.join(files))
        aiPath.open('w').write(aiText)

    def resetMTimes(self):
        """Reset dates of bsa files to 'correct' values."""
        #--Fix the data of a few archive files
        bsaTimes = (
            ('Oblivion - Meshes.bsa',1138575220),
            ('Oblivion - Misc.bsa',1139433736),
            ('Oblivion - Sounds.bsa',1138660560),
            ('Oblivion - Textures - Compressed.bsa',1138162634),
            ('Oblivion - Voices1.bsa',1138162934),
            ('Oblivion - Voices2.bsa',1138166742),
            )
        for bsaFile,mtime in bsaTimes:
            bsaPath = dirs['mods'].join(bsaFile)
            bsaPath.setmtime(mtime)

    def reset(self,progress=None):
        """Resets BSA archive hashes to correct values."""
        ios = StructFile(self.path,'r+b')
        #--Rehash
        resetCount = 0
        folderInfos = self.folderInfos
        getHash = BsaFile.getHash
        for folderInfo in folderInfos:
            for fileInfo in folderInfo[-1]:
                hash,size,offset,fileName,filePos = fileInfo
                trueHash = getHash(fileName)
                if hash != trueHash:
                    #print ' ',fileName,'\t',hex(hash-trueHash),hex(hash),hex(trueHash)
                    ios.seek(filePos)
                    ios.pack('Q',trueHash)
                    resetCount += 1
        #--Done
        ios.close()
        self.resetMTimes()
        self.updateAIText()
        return resetCount

    def invalidate(self,progress=None):
        """Invalidates entries in BSA archive and regenerates Archive Invalidation.txt."""
        reRepTexture = re.compile(r'(?<!_[gn])\.dds',re.I)
        ios = StructFile(self.path,'r+b')
        #--Rehash
        reset,inval,intxt = [],[],[]
        folderInfos = self.folderInfos
        getHash = BsaFile.getHash
        trueHashes = set()
        def setHash(filePos,newHash):
            ios.seek(filePos)
            ios.pack('Q',newHash)
            return newHash
        for folderInfo in folderInfos:
            folderName = folderInfo[-2]
            #--Actual directory files
            diskPath = modInfos.dir.join(folderName)
            #print '>>',diskPath
            diskFiles = set(diskPath.list())
            trueHashes.clear()
            nextHash = 0 #--But going in reverse order, physical 'next' == loop 'prev'
            for fileInfo in reversed(folderInfo[-1]):
                hash,size,offset,fileName,filePos = fileInfo
                fullPath = os.path.join(folderName,fileName)
                trueHash = getHash(fileName)
                plusCE = trueHash + 0xCE
                plusE = trueHash + 0xE
                #--No invalidate?
                if not (fileName in diskFiles and reRepTexture.search(fileName)):
                    if hash != trueHash:
                        setHash(filePos,trueHash)
                        reset.append(fullPath)
                    nextHash = trueHash
                #--Invalidate one way or another...
                elif not nextHash or (plusCE < nextHash and plusCE not in trueHashes):
                    nextHash = setHash(filePos,plusCE)
                    inval.append(fullPath)
                elif plusE < nextHash and plusE not in trueHashes:
                    nextHash = setHash(filePos,plusE)
                    inval.append(fullPath)
                else:
                    if hash != trueHash:
                        setHash(filePos,trueHash)
                    nextHash = trueHash
                    intxt.append(fullPath)
                trueHashes.add(trueHash)
        #--Save/Cleanup
        ios.close()
        self.resetMTimes()
        self.updateAIText(intxt)
        #--Done
        return (reset,inval,intxt)

#--------------------------------------------------------------------------------
class OblivionIni:
    """Oblivion.ini file."""
    def __init__(self):
        """Initialize."""
        self.path = dirs['saveBase'].join('Oblivion.ini')
        self.isCorrupted = False
        
    def ensureExists(self):
        """Ensures that Oblivion.ini file exists. Copies from default oblvion.ini if necessary."""
        if self.path.exists(): return
        srcPath = dirs['app'].join('Oblivion_default.ini')
        srcPath.copyfile(self.path)

    def getSetting(self,section,key,default=None):
        """Gets a single setting from the file."""
        settings = self.getSettings()
        if section in settings:
            return settings[section].get(key,default)
        else:
            return default

    def getSettings(self):
        """Gets settings for self."""
        reComment = re.compile(';.*')
        reSection = re.compile(r'^\[\s*(.+?)\s*\]$')
        reSetting = re.compile(r'(.+?)\s*=(.*)')
        #--Read ini file
        self.ensureExists()
        iniFile = self.path.open('r')
        settings = {} #settings[section][key] = value (stripped!)
        sectionSettings = None 
        for line in iniFile:
            stripped = reComment.sub('',line).strip()
            maSection = reSection.match(stripped)
            maSetting = reSetting.match(stripped)
            if maSection:
                sectionSettings = settings[maSection.group(1)] = {}
            elif maSetting:
                if sectionSettings == None:
                    sectionSettings = settings.setdefault('General',{})
                    self.isCorrupted = True
                sectionSettings[maSetting.group(1)] = maSetting.group(2).strip()
        iniFile.close()
        return settings

    def saveSetting(self,section,key,value):
        """Changes a single setting in the file."""
        settings = {section:{key:value}}
        self.saveSettings(settings)

    def saveSettings(self,settings):
        """Applies dictionary of settings to ini file. 
        Values in settings dictionary can be either actual values or 
        full key=value line ending in newline char."""
        reComment = re.compile(';.*')
        reSection = re.compile(r'^\[\s*(.+?)\s*\]$')
        reSetting = re.compile(r'(.+?)\s*=')
        #--Read init, write temp
        self.ensureExists()
        iniFile = self.path.open('r')
        tmpPath = self.path+'.tmp'
        tmpFile = tmpPath.open('w')
        section = sectionSettings = None
        for line in iniFile:
            stripped = reComment.sub('',line).strip()
            maSection = reSection.match(stripped)
            maSetting = reSetting.match(stripped)
            if maSection:
                section = maSection.group(1)
                sectionSettings = settings.get(section,{})
            elif maSetting and maSetting.group(1) in sectionSettings:
                key = maSetting.group(1)
                value = sectionSettings[key] 
                if isinstance(value,str) and value[-1] == '\n':
                    line = value
                else:
                    line = '%s=%s\n' % (key,value)
            tmpFile.write(line)
        tmpFile.close()
        iniFile.close()
        #--Done
        tmpPath.replace(self.path)

    def applyTweakFile(self,tweakPath):
        """Read Ini tweak file and apply its settings to oblivion.ini.
        Note: Will ONLY apply settings that already exist."""
        reComment = re.compile(';.*')
        reSection = re.compile(r'^\[\s*(.+?)\s*\]$')
        reSetting = re.compile(r'(.+?)\s*=')
        #--Read Tweak file
        self.ensureExists()
        tweakFile = tweakPath.open('r')
        settings = {} #settings[section][key] = "key=value\n"
        sectionSettings = None 
        for line in tweakFile:
            stripped = reComment.sub('',line).strip()
            maSection = reSection.match(stripped)
            maSetting = reSetting.match(stripped)
            if maSection:
                sectionSettings = settings[maSection.group(1)] = {}
            elif maSetting:
                sectionSettings[maSetting.group(1)] = line
        tweakFile.close()
        self.saveSettings(settings)

#------------------------------------------------------------------------------
class PluginsFullError(BoshError):
    """Usage Error: Attempt to add a mod to plugins when plugins is full."""
    def __init__(self,message=_('Load list is full.')):
        BoshError.__init__(self,message)

#------------------------------------------------------------------------------
class Plugins: 
    """Plugins.txt file. Owned by modInfos. Almost nothing else should access it directly."""
    def __init__(self):
        """Initialize."""
        self.dir = dirs['userApp']
        self.path = self.dir.join('Plugins.txt')
        self.mtime = 0
        self.size = 0
        self.selected = []
        self.selectedBad = [] #--In plugins.txt, but don't exist!
        self.selectedExtra = [] #--Where mod number would be greater than 255.
        #--Create dirs/files if necessary
        self.dir.makedirs()
        if not self.path.exists():
            self.save()

    def load(self):
        """Read data from plugins.txt file.
        NOTE: modInfos must exist and be up to date."""
        #--Read file
        self.mtime = self.path.getmtime()
        self.size = self.path.getsize()
        ins = self.path.open('r')
        #--Load Files 
        selFiles = set()
        modsDir = dirs['mods']
        del self.selected[:]
        del self.selectedBad[:]
        for line in ins:
            selFile = Path.get(reComment.sub('',line).strip())
            if not selFile: continue
            selPath = modsDir.join(selFile)
            if selFile in selFiles: #--In case it's listed twice.
                pass
            elif len(self.selected) == 255:
                self.selectedExtra.append(selFile)
            elif selFile in modInfos:
                self.selected.append(selFile)
            else:
                self.selectedBad.append(selFile)
            selFiles.add(selFile)
        #--Done
        ins.close()

    def save(self):
        """Write data to Plugins.txt file."""
        self.selected.sort(key=string.lower)
        out = self.path.open('w')
        out.write('# This file is used to tell Oblivion which data files to load.\n\n')
        for selFile in self.selected:
            out.write(selFile+'\n')
        out.close()
        self.mtime = self.path.getmtime()
        self.size = self.path.getsize()

    def hasChanged(self):
        """True if plugins.txt file has changed."""
        return ((self.mtime != self.path.getmtime()) or
            (self.size != self.path.getsize()) )

    def refresh(self,forceRefresh):
        """Load only if plugins.txt has changed."""
        hasChanged = forceRefresh or self.hasChanged()
        if hasChanged: self.load()
        return hasChanged

    def remove(self,fileName):
        """Remove specified mod from file list."""
        while fileName in self.selected:
            self.selected.remove(fileName)

#------------------------------------------------------------------------------
class MasterInfo:
    def __init__(self,name,size):
        self.oldName = self.name = Path.get(name)
        self.modInfo = modInfos.get(self.name,None)
        if self.modInfo:
            self.mtime = self.modInfo.mtime
            self.author = self.modInfo.header.author
            self.masterNames = self.modInfo.masterNames
        else:
            self.mtime = 0
            self.author = ''
            self.masterNames = tuple()
    
    def setName(self,name):
        self.name = Path.get(name)
        self.modInfo = modInfos.get(self.name,None)
        if self.modInfo:
            self.mtime = self.modInfo.mtime
            self.author = self.modInfo.header.author
            self.masterNames = self.modInfo.masterNames
        else:
            self.mtime = 0
            self.author = ''
            self.masterNames = tuple()

    def hasChanged(self):
        return (self.name != self.oldName)

    def isEsm(self):
        if self.modInfo:
            return self.modInfo.isEsm()
        else:
            return reEsmExt.search(self.name)

    def hasTimeConflict(self):
        """True if has an mtime conflict with another mod."""
        if self.modInfo:
            return self.modInfo.hasTimeConflict()
        else:
            return False

    def hasActiveTimeConflict(self):
        """True if has an active mtime conflict with another mod."""
        if self.modInfo:
            return self.modInfo.hasActiveTimeConflict()
        else:
            return False

    def isExOverLoaded(self):
        """True if belongs to an exclusion group that is overloaded."""
        if self.modInfo:
            return self.modInfo.isExOverLoaded()
        else:
            return False

    def getStatus(self):
        if not self.modInfo: 
            return 30
        else:
            return 0
    
#------------------------------------------------------------------------------
class FileInfo:
    """Abstract TES4/TES4GAME File."""
    def __init__(self,dir,name): 
        self.dir = Path.get(dir)
        self.name = Path.get(name)
        path = dir.join(name)
        if path.exists():
            self.ctime = path.getctime()
            self.mtime = path.getmtime()
            self.size = path.getsize()
        else:
            self.ctime = time.time()
            self.mtime = time.time()
            self.size = 0
        self.header = None
        self.masterNames = tuple()
        self.masterOrder = tuple()
        self.madeBackup = False
        #--Ancillary storage
        self.extras = {}
 
    def getPath(self):
        """Returns joined dir and name."""
        return self.dir.join(self.name)

    def getFileInfos(self):
        """Returns modInfos or saveInfos depending on fileInfo type."""
        raise AbstractError

    #--File type tests 
    #--Note that these tests only test extension, not the file data.
    def isMod(self):
        return reModExt.search(self.name)
    def isEsp(self):
        if not self.isMod(): return False
        if self.header:
            return int(self.header.flags1) & 1 == 0
        else:
            return reEspExt.search(self.name)
    def isEsm(self):
        if not self.isMod(): return False
        if self.header:
            return int(self.header.flags1) & 1 == 1
        else:
            return reEsmExt.search(self.name) and False
    def isInvertedMod(self):
        """Extension indicates esp/esm, but byte setting indicates opposite."""
        return self.isMod() and self.header and self.name[-3:].lower() != ('esp','esm')[int(self.header.flags1) & 1]

    def isEss(self):
        return self.name[-3:].lower() == 'ess'

    def sameAs(self,fileInfo): 
        """Returns true if other fileInfo refers to same file as this fileInfo."""
        return (
            (self.size == fileInfo.size) and
            (self.mtime == fileInfo.mtime) and
            (self.ctime == fileInfo.ctime) and
            (self.name == fileInfo.name) )

    def refresh(self): 
        path = self.dir.join(self.name)
        self.ctime = path.getctime()
        self.mtime = path.getmtime()
        self.size  = path.getsize()
        if self.header: self.getHeader()

    def getHeader(self): 
        """Read header for file."""
        raise AbstractError

    def getMasterStatus(self,masterName): 
        """Returns status of a master. Called by getStatus."""
        #--Exists?
        if masterName not in modInfos:
            return 30
        #--Okay?
        else:
            return 0
    
    def getStatus(self): 
        """Returns status of this file -- which depends on status of masters.
        0:  Good
        10: Out of order master
        30: Missing master(s)."""
        #--Worst status from masters
        if self.masterNames:
            status = max([self.getMasterStatus(masterName) for masterName in self.masterNames])
        else:
            status = 0
        #--Missing files?
        if status == 30: 
            return status
        #--Misordered?
        self.masterOrder = modInfos.getOrdered(self.masterNames)
        if self.masterOrder != self.masterNames:
            return 20
        else:
            return status

    def writeHeader(self): 
        """Writes header to file, overwriting old header."""
        raise AbstractError

    def setMTime(self,mtime=0): 
        """Sets mtime. Defaults to current value (i.e. reset)."""
        mtime = int(mtime or self.mtime)
        path = self.dir.join(self.name)
        path.setmtime(mtime)
        self.mtime = path.getmtime()
    
    def makeBackup(self, forceBackup=False): 
        """Creates backup(s) of file."""
        #--Skip backup?
        if not self in self.getFileInfos().data.values(): return
        if self.madeBackup and not forceBackup: return
        #--Backup Directory
        backupDir = self.dir.join('Bash','Backups')
        backupDir.makedirs()
        #--File Path
        original = self.dir.join(self.name)
        #--Backup
        backup = backupDir.join(self.name)
        original.copyfile(backup)
        #--First backup
        firstBackup = backup+'f'
        if not firstBackup.exists():
            original.copyfile(firstBackup)
        #--Done
        self.madeBackup = True
    
    def getStats(self): 
        """Gets file stats. Saves into self.stats."""
        stats = self.stats = {}
        raise AbstractError

    def getNextSnapshot(self): 
        """Returns parameters for next snapshot."""
        if not self in self.getFileInfos().data.values(): 
            raise StateError(_("Can't get snapshot parameters for file outside main directory."))
        destDir = self.dir.join('Bash','Snapshots')
        destDir.makedirs()
        (root,ext) = self.name.splitext()
        destName = root+'-00'+ext
        separator = '-'
        snapLast = ['00']
        #--Look for old snapshots.
        reSnap = re.compile('^'+root+'[ -]([0-9\.]*[0-9]+)'+ext+'$')
        for fileName in destDir.list():
            maSnap = reSnap.match(fileName)
            if not maSnap: continue
            snapNew = maSnap.group(1).split('.')
            #--Compare shared version numbers
            sharedNums = min(len(snapNew),len(snapLast))
            for index in range(sharedNums):
                (numNew,numLast) = (int(snapNew[index]),int(snapLast[index]))
                if numNew > numLast:
                    snapLast = snapNew
                    continue
            #--Compare length of numbers
            if len(snapNew) > len(snapLast):
                snapLast = snapNew
                continue
        #--New
        snapLast[-1] = ('%0'+`len(snapLast[-1])`+'d') % (int(snapLast[-1])+1,)
        destName = root+separator+('.'.join(snapLast))+ext
        return (destDir,destName,root+'*'+ext)

#------------------------------------------------------------------------------
class ModInfo(FileInfo):
    def getFileInfos(self):
        """Returns modInfos or saveInfos depending on fileInfo type."""
        return modInfos

    def setType(self,type): 
        """Sets the file's internal type."""
        if type not in ('esm','esp'):
            raise ArgumentError
        modFile = self.dir.join(self.name).open('r+b')
        modFile.seek(8)
        flags1 = MreRecord.flags1(struct.unpack('I',modFile.read(4))[0])
        flags1.esm = (type == 'esm')
        modFile.seek(8)
        modFile.write(struct.pack('=I',int(flags1)))
        modFile.close()
        self.header.flags1 = flags1
        self.setMTime()
        self.isMergeable = False

    def hasTimeConflict(self): 
        """True if has an mtime conflict with another mod."""
        return modInfos.hasTimeConflict(self.name)

    def hasActiveTimeConflict(self): 
        """True if has an active mtime conflict with another mod."""
        return modInfos.hasActiveTimeConflict(self.name)

    def isExOverLoaded(self):
        """True if belongs to an exclusion group that is overloaded."""
        maExGroup = reExGroup.match(self.name)
        if not (modInfos.isSelected(self.name) and maExGroup):
            return False
        else:
            exGroup = maExGroup.group(1)
            return len(modInfos.exGroup_mods.get(exGroup,'')) > 1

    def hasResources(self):
        """Returns (hasBsa,hasVoices) booleans according to presence of corresponding resources."""
        bsaPath = Path.get(self.dir.join(self.name)[:-3]+'bsa')
        voicesPath = self.dir.join('Sound','Voice',self.name)
        return [bsaPath.exists(),voicesPath.exists()]

    def setMTime(self,mtime=0): 
        """Sets mtime. Defaults to current value (i.e. reset)."""
        mtime = int(mtime or self.mtime)
        FileInfo.setMTime(self,mtime)
        modInfos.mtimes[self.name] = mtime
    
    def getHeader(self): 
        """Read header for file."""
        path = self.dir.join(self.name)
        ins = ModReader(self.name,path.open('rb'))
        try:
            recHeader = ins.unpackRecHeader()
            if recHeader[0] != 'TES4':
                raise ModError(self.name,_('Expected TES4, but got ')+recHeader[0])
            self.header = MreTes4(recHeader,ins,True)
            ins.close()
        except struct.error, rex:
            ins.close()
            raise ModError(self.name,_('Struct.error: ')+`rex`)
        except:
            ins.close()
            raise
        #--Master Names/Order
        self.masterNames = tuple(self.header.masters)
        self.masterOrder = tuple() #--Reset to empty for now
        self.isMergeable = 'Merge' in self.getBashKeys()

    def shiftBashKeys(self):
        """Shifts bash keys from bottom to top."""
        description = self.header.description
        reReturns = re.compile('\r{2,}')
        reBashKeys = re.compile('^(.+)({{BASH:[^}]*}})$',re.S)
        if reBashKeys.match(description) or reReturns.search(description):
            description = reReturns.sub('\r',description)
            description = reBashKeys.sub(r'\2\n\1',description)
            self.writeDescription(description)

    def addBashKey(self,key):
        """Adds specified Bash flag key."""
        bashKeys = self.getBashKeys()
        if key not in bashKeys: 
            bashKeys.add(key)
            self.setBashKeys(bashKeys)

    def setBashKeys(self,keys):
        """Sets bash keys as specified."""
        keys = set(keys) #--Make sure it's a set.
        if keys == self.getBashKeys(): return
        strKeys = '{{BASH:'+(','.join(sorted(keys)))+'}}'
        description = self.header.description or ''
        reBashKeys = re.compile('{{ *BASH *:[^}]*}}')
        if reBashKeys.search(description):
            description = reBashKeys.sub(strKeys,description)
        else:
            description = strKeys+'\n'+description
        self.writeDescription(description)

    def hasBashKeys(self):
        """Boolean: True if has bash keys."""
        description = self.header.description or ''
        maBashKeys = re.search('{{ *BASH *:([^}]+)}}',description)
        return bool(maBashKeys)

    def getBashKeys(self):
        """Returns any Bash flag keys."""
        description = self.header.description or ''
        maBashKeys = re.search('{{ *BASH *:([^}]+)}}',description)
        if not maBashKeys:
            return set()
        else:
            bashKeys = maBashKeys.group(1).split(',')
            bashKeys = [str.strip() for str in bashKeys]
            return set(bashKeys)

    def writeNew(self,masters=[],mtime=0): 
        """Creates a new file with the given name, masters and mtime."""
        header = MreTes4(('TES4',0,(self.isEsm() and 1 or 0),0,0))
        for master in masters:
            header.masters.append(master)
        header.setChanged()
        #--Write it
        path = self.dir.join(self.name)
        out = path.open('wb')
        header.getSize()
        header.dump(out)
        out.close()
        self.setMTime(mtime)

    def writeHeader(self): 
        """Write Header. Actually have to rewrite entire file."""
        filePath = self.dir.join(self.name)
        tempPath = filePath+'.tmp'
        ins = filePath.open('rb')
        out = tempPath.open('wb')
        try:
            #--Open original and skip over header
            reader = ModReader(self.name,ins)
            recHeader = reader.unpackRecHeader()
            if recHeader[0] != 'TES4': 
                raise ModError(self.name,_('Expected TES4, but got ')+recHeader[0])
            reader.seek(recHeader[1],1)
            #--Write new header
            self.header.getSize()
            self.header.dump(out)
            #--Write remainder
            while True:
                buffer= ins.read(0x5000000)
                if not buffer: break
                out.write(buffer)
            ins.close()
            out.close()
        except struct.error, rex:
            ins.close()
            out.close()
            raise ModError(self.name,_('Struct.error: ')+`rex`)
        except:
            ins.close()
            out.close()
            raise
        #--Remove original and replace with temp
        renameFile(tempPath,filePath)
        self.setMTime()
        self.isMergeable = 'Merge' in self.getBashKeys()

    def writeDescription(self,description): 
        """Sets description to specified text and then writes hedr."""
        description = description[:min(255,len(description))]
        self.header.description = description
        self.header.setChanged()
        self.writeHeader()

    def writeAuthor(self,author): 
        """Sets author to specified text and then writes hedr."""
        author = author[:min(512,len(author))]
        self.header.author = author
        self.header.setChanged()
        self.writeHeader()

    def writeAuthorWB(self): 
        """Marks author field with " [wb]" to indicate Wrye Bash modification."""
        author = self.header.author
        if '[wm]' not in author and len(author) <= 27:
            self.writeAuthor(author+' [wb]')

#------------------------------------------------------------------------------
class SaveInfo(FileInfo):
    def getFileInfos(self):
        """Returns modInfos or saveInfos depending on fileInfo type."""
        return saveInfos

    def getStatus(self):
        status = FileInfo.getStatus(self)
        masterOrder = self.masterOrder
        #--File size?
        if status > 0 or len(masterOrder) > len(modInfos.ordered):
            return status
        #--Current ordering?
        if masterOrder != modInfos.ordered[:len(masterOrder)]: 
            return status
        elif masterOrder == modInfos.ordered: 
            return -20
        else:
            return -10

    def getHeader(self): 
        """Read header for file."""
        try:
            path = self.dir.join(self.name)
            self.header = SaveHeader(path)
            #--Master Names/Order
            self.masterNames = tuple(self.header.masters)
            self.masterOrder = tuple() #--Reset to empty for now
        except struct.error, rex:
            raise SaveFileError(self.name,_('Struct.error: ')+`rex`)

#------------------------------------------------------------------------------
class FileInfos(DataDict):
    def __init__(self,dir,factory=FileInfo):
        """Init with specified directory and specified factory type."""
        self.dir = Path.get(dir) #--Path
        self.factory=factory
        self.data = {}
        self.table = Table(self.dir.join('Bash','Table.dat'),self.dir.join('Bash','Table.pkl'))
        self.corrupted = {} #--errorMessage = corrupted[fileName]
        #--Update table keys...
        tableData = self.table.data
        for key in self.table.data.keys():
            if not isinstance(key,Path):
                data = tableData[key]
                del tableData[key]
                tableData[Path.get(key)] = data

    #--Refresh File
    def refreshFile(self,fileName):
        try:
            fileInfo = self.factory(self.dir,fileName)
            fileInfo.getHeader()
            self.data[fileName] = fileInfo
        except FileError, error:
            self.corrupted[fileName] = error.message
            if fileName in self.data:
                del self.data[fileName]
            raise

    #--Refresh
    def refresh(self):
        data = self.data
        oldList = data.keys()
        newList = []
        added = []
        updated = []
        deleted = []
        self.dir.makedirs()
        #--Loop over files in directory
        for fileName in self.dir.list():
            #--Right file type?
            filePath = self.dir.join(fileName)
            if not filePath.isfile() or not self.rightFileType(fileName): 
                continue
            fileInfo = self.factory(self.dir,fileName)
            #--New file?
            if fileName not in oldList:
                try:
                    fileInfo.getHeader()
                #--Bad header?
                except FileError, error:
                    self.corrupted[fileName] = error.message
                    continue
                #--Good header?
                else:
                    if fileName in self.corrupted:
                        del self.corrupted[fileName]
                    added.append(fileName)
                    data[fileName] = fileInfo
            #--Updated file?
            elif not fileInfo.sameAs(data[fileName]):
                try:
                    fileInfo.getHeader()
                    data[fileName] = fileInfo
                #--Bad header?
                except FileError, error:
                    self.corrupted[fileName] = error.message
                    del self.data[fileName]
                    continue
                #--Good header?
                else:
                    if fileName in self.corrupted:
                        del self.corrupted[fileName]
                    updated.append(fileName)
            #--No change?
            newList.append(fileName)
        #--Any files deleted?
        for fileName in oldList:
            if fileName not in newList:
                deleted.append(fileName)
                del self.data[fileName]
        #--Return
        return (len(added) or len(updated) or len(deleted))

    #--Right File Type? [ABSTRACT]
    def rightFileType(self,fileName):
        """Bool: filetype (extension) is correct for subclass. [ABSTRACT]"""
        raise AbstractError

    #--Rename
    def rename(self,oldName,newName):
        """Renames member file from oldName to newName."""
        #--Update references
        fileInfo = self[oldName]
        self[newName] = self[oldName]
        del self[oldName]
        self.table.moveRow(oldName,newName)
        #--FileInfo
        fileInfo.name = newName
        #--File system
        newPath = fileInfo.dir.join(newName)
        oldPath = fileInfo.dir.join(oldName)
        renameFile(oldPath,newPath)
        #--Done
        fileInfo.madeBackup = False

    #--Delete
    def delete(self,fileName):
        """Deletes member file."""
        fileInfo = self[fileName]
        #--File
        filePath = fileInfo.dir.join(fileInfo.name)
        filePath.remove()
        #--Table
        self.table.delRow(fileName)
        #--Misc. Editor backups (mods only)
        if fileInfo.isMod():
            for ext in ('.bak','.tmp','.old'):
                backPath = filePath + ext
                backPath.remove()
        #--Backups
        backRoot = fileInfo.dir.join('Bash','Backups',fileInfo.name)
        for backPath in (backRoot,backRoot+'f'):
            backPath.remove()
        self.refresh()

    #--Move Exists
    def moveIsSafe(self,fileName,destDir):
        """Bool: Safe to move file to destDir."""
        return not destDir.join(fileName).exists()

    #--Move
    def move(self,fileName,destDir):
        """Moves member file to destDir. Will overwrite!"""
        destDir.makedirs()
        srcPath = self.dir.join(fileName)
        destPath = destDir.join(fileName)
        renameFile(srcPath,destPath)
        self.refresh()

    #--Copy
    def copy(self,fileName,destDir,destName=None,mtime=False):
        """Copies member file to destDir. Will overwrite!"""
        destDir.makedirs()
        if not destName: destName = fileName
        srcPath = self.dir.join(fileName)
        destPath = destDir.join(destName)
        destPath.remove()
        srcPath.copyfile(destPath)
        if mtime:
            if mtime == True:
                mtime = srcPath.getmtime()
            elif mtime == '+1': 
                mtime = 1 + srcPath.getmtime()
            destPath.setmtime(mtime)
        self.refresh()

#------------------------------------------------------------------------------
class ResourceReplacer: 
    """Resource Replacer. Used to apply and remove a set of resource (texture, etc.) replacement files."""
    #--Class data
    dirExts = {
        'distantlod': ['.cmp', '.lod'],
        'docs':['.txt','.html','.htm','.rtf','.doc','.gif','.jpg'],
        'facegen': ['.ctl'],
        'fonts': ['.fnt', '.tex'],
        'menus': ['.bat', '.html', '.scc', '.txt', '.xml'],
        'meshes': ['.egm', '.egt', '.fim', '.kf', '.kfm', '.nif', '.tri', '.txt'],
        'shaders': ['.sdp'],
        'sound': ['.lip', '.mp3', '.wav'],
        'textures': ['.dds', '.ifl', '.psd', '.txt'],
        'trees': ['.spt'],
        }

    def __init__(self,replacerDir,file):
        """Initialize"""
        self.replacerDir = replacerDir
        self.file = file
        self.rootDir = ''

    def isApplied(self):
        """Returns True if has been applied."""
        return self.file in settings['bosh.resourceReplacer.applied']
    
    def validate(self):
        """Does archive invalidation according to settings."""
        if settings.get('bash.replacers.autoEditBSAs',False):
            bsaPath = dirs['mods'].join('Oblivion - Textures - Compressed.bsa')
            bsaFile = BsaFile(bsaPath)
            bsaFile.scan()
            bsaFile.invalidate()

    def apply(self,progress=None):
        """Copy files to appropriate resource directories (Textures, etc.).""" 
        progress = progress or Progress()
        progress.state,progress.full = 0,1
        progress(0,_("Getting sizes."))
        self.doRoot(self.countDir,progress) #--Updates progress.full
        self.doRoot(self.applyDir,progress)
        self.validate()
        settings.getChanged('bosh.resourceReplacer.applied').append(self.file)

    def remove(self,progress=None):
        """Uncopy files from appropriate resource directories (Textures, etc.).""" 
        progress = progress or Progress()
        self.doRoot(self.removeDir,progress)
        self.validate()
        settings.getChanged('bosh.resourceReplacer.applied').remove(self.file)

    def doRoot(self,action,progress):
        """Copy/uncopy files to/from appropriate resource directories."""
        dirExts = ResourceReplacer.dirExts
        srcDir = self.rootDir = self.replacerDir.join(self.file)
        destDir = dirs['mods']
        action(srcDir,destDir,['.esp','.esm','.bsa'],progress)
        for srcFile in srcDir.list():
            srcPath  = srcDir.join(srcFile)
            if srcPath.isdir() and srcFile.lower() in dirExts:
                destPath = destDir.join(srcFile)
                action(srcPath,destPath,dirExts[srcFile.lower()],progress)
    
    def sizeDir(self,srcDir,destDir,exts,progress):
        """Determine cumulative size of files to copy.""" 
        for srcFile in srcDir.list():
            srcExt = srcFile.ext().lower()
            srcPath  = srcDir.join(srcFile)
            destPath = destDir.join(srcFile)
            if srcExt in exts:
                progress.full += srcPath.getsize()
            elif srcPath.isdir():
                self.sizeDir(srcPath,destPath,exts,progress)
    
    def countDir(self,srcDir,destDir,exts,progress):
        """Determine cumulative count of files to copy.""" 
        rootDir = self.rootDir
        for srcFile in srcDir.list():
            srcExt = srcFile.ext().lower()
            srcPath  = srcDir.join(srcFile)
            destPath = destDir.join(srcFile)
            if srcExt in exts:
                progress.full += 1
            elif srcDir != rootDir and srcPath.isdir():
                self.countDir(srcPath,destPath,exts,progress)
    
    def applyDir(self,srcDir,destDir,exts,progress):
        """Copy files to appropriate resource directories (Textures, etc.).""" 
        #print srcDir
        rootDir = self.rootDir
        progress(progress.state,srcDir[len(rootDir)+1:])
        for srcFile in srcDir.list():
            srcExt = srcFile.ext().lower()
            srcPath  = srcDir.join(srcFile)
            destPath = destDir.join(srcFile)
            if srcExt in exts:
                destDir.makedirs()
                srcPath.copyfile(destPath)
                #print '>',srcFile
                #progress.plus(srcPath.getsize())
                progress.plus()
            elif srcDir != rootDir and srcPath.isdir():
                self.applyDir(srcPath,destPath,exts,progress)
    
    def removeDir(self,srcDir,destDir,exts,progress):
        """Uncopy files from appropriate resource directories (Textures, etc.).""" 
        rootDir = self.rootDir
        for srcFile in srcDir.list():
            srcExt = srcFile.ext().lower()
            srcPath  = srcDir.join(srcFile)
            destPath = destDir.join(srcFile)
            if destPath.exists():
                if srcExt in exts:
                    destPath.remove()
                elif srcDir != rootDir and srcPath.isdir():
                    self.removeDir(srcPath,destPath,exts,progress)

    @staticmethod
    def updateInvalidator():
        """Updates ArchiveInvalidator.txt file. Use this after adding/removing resources."""
        reRepTexture = re.compile(r'(?<!_[gn])\.dds',re.I)
        #--Get files to invalidate
        fileNames = []
        def addFiles(dirtuple):
            dirPath = dirs['mods'].join(*dirtuple)
            for fileName in dirPath.list():
                filetuple = dirtuple+(fileName,)
                if dirPath.join(fileName).isdir():
                    addFiles(filetuple)
                elif reRepTexture.search(fileName):
                    fileNames.append('/'.join(filetuple))
        if dirs['mods'].join('textures').exists():
            addFiles(('textures',))
        fileNames.sort(key=string.lower)
        #--Update file
        aiAppPath = dirs['app'].join('ArchiveInvalidation.txt')
        #--Update file?
        if fileNames:
            out = file(aiAppPath,'w')
            for fileName in fileNames:
                out.write(fileName+'\n')
            out.close
        #--No files to invalidate, but ArchiveInvalidation.txt exists?
        elif aiAppPath.exists():
            aiAppPath.remove()
        #--Remove any duplicate AI.txt in the mod directory
        aiModsPath = dirs['mods'].join('ArchiveInvalidation.txt')
        aiModsPath.remove()
        #--Fix the data of a few archive files
        bsaTimes = (
            ('Oblivion - Meshes.bsa',1138575220),
            ('Oblivion - Misc.bsa',1139433736),
            ('Oblivion - Sounds.bsa',1138660560),
            ('Oblivion - Textures - Compressed.bsa',1138162634),
            ('Oblivion - Voices1.bsa',1138162934),
            ('Oblivion - Voices2.bsa',1138166742),
            )
        for bsaFile,mtime in bsaTimes:
            bsaPath = dirs['mods'].join(bsaFile)
            bsaPath.setmtime(mtime)

#------------------------------------------------------------------------------
class ModInfos(FileInfos):
    """Collection of modinfos. Represents mods in the Oblivion\Data directory."""

    def __init__(self):
        """Initialize."""
        FileInfos.__init__(self,dirs['mods'],ModInfo)
        #--MTime resetting
        self.resetMTimes = settings['bosh.modInfos.resetMTimes']
        self.mtimes = self.table.getColumn('mtime')
        self.mtimesReset = [] #--Files whose mtimes have been reset.
        #--Selection state (ordered, merged, imported)
        self.plugins = Plugins() #--Plugins instance.
        self.ordered = tuple() #--Active mods arranged in load order.
        #--Info lists/sets
        self.mtime_mods = {}
        self.mtime_selected = {}
        self.exGroup_mods = {}
        self.merged = set() #--For bash merged files
        self.imported = set() #--For bash imported files
        self.autoSorted = set() #--Files that are auto-sorted
        #--Oblivion version 
        self.version_voSize = {
            '1.1':int(_("247388848")), #--247388848
            'SI': int(_("277504985"))}
        self.size_voVersion = invertDict(self.version_voSize)
        self.voCurrent = None
        self.voAvailable = set()

    #--Refresh
    def refresh(self):
        """Update file data for additions, removals and date changes."""
        #--OBMM warning
        obmmWarn = settings.setdefault('bosh.modInfos.obmmWarn',0)
        if self.resetMTimes and obmmWarn == 0 and dirs['app'].join('obmm').exists():
            settings['bosh.modInfos.obmmWarn'] = 1
        #--Refresh FileInfos
        hasChanged = FileInfos.refresh(self)
        if hasChanged and self.resetMTimes and obmmWarn != 1 and not settings.safeMode:
            self.refreshMTimes()
        hasSorted = self.autoSort()
        #--Plugins
        plugins = self.plugins
        hasReplugged = plugins.refresh(hasChanged)
        self.refreshInfoLists()
        self.getOblivionVersions()
        #--Done
        return hasChanged or hasSorted or hasReplugged

    def refreshMTimes(self):
        """Remember/reset mtimes of member files."""
        del self.mtimesReset[:]
        for fileName, fileInfo in self.data.items():
            oldMTime = int(self.mtimes.get(fileName,fileInfo.mtime))
            self.mtimes[fileName] = oldMTime
            #--Reset mtime?
            if fileInfo.mtime != oldMTime and oldMTime  > 0:
                #print fileInfo.name, formatDate(oldMTime), formatDate(fileInfo.mtime)
                fileInfo.setMTime(oldMTime)
                self.mtimesReset.append(fileName)

    def autoSort(self):
        """Automatically sorts mods by group."""
        if settings['bosh.modInfos.obmmWarn'] == 1: return
        if settings.safeMode or not settings.get('bosh.modInfos.resetMTimes'): 
            return
        #--Get grouping info
        group_anchor = {}
        group_mods = {}
        mod_group = self.table.getColumn('group')
        for modName in self.data:
            group = mod_group.get(modName,None)
            if not group: continue
            if modName[:2] == '++':
                group_anchor[group] = modName
            else:
                if group not in group_mods: 
                    group_mods[group] = [modName]
                else:
                    group_mods[group].append(modName)
        #--Sort them
        autoSorted = self.autoSorted
        autoSorted.clear()
        changed = 0
        if not group_anchor: return changed
        for group,anchor in group_anchor.items():
            mods = group_mods.get(group,[])
            mods.sort(key=lambda a: a.root())
            mods.sort(key=lambda a: a[-1] in 'pP')
            mtime = self.data[anchor].mtime + 70
            for mod in mods:
                autoSorted.add(mod)
                modInfo = self.data[mod]
                if modInfo.mtime != mtime:
                    modInfo.setMTime(mtime)
                    changed += 1
                mtime += 70
        return changed

    def refreshInfoLists(self):
        """Refreshes various mod info lists (mtime_mods, mtime_selected, exGroup_mods, imported, exported."""
        #--Ordered
        self.ordered = self.getOrdered(self.plugins.selected)
        #--Mod mtimes
        mtime_mods = self.mtime_mods
        mtime_mods.clear()
        for modName in self.keys():
            mtime = modInfos[Path.get(modName)].mtime
            mtime_mods.setdefault(mtime,[]).append(modName)
        #--Selected mtimes
        mtime_selected = self.mtime_selected
        mtime_selected.clear()
        for modName in self.ordered:
            mtime = modInfos[Path.get(modName)].mtime
            mtime_selected.setdefault(mtime,[]).append(modName)
        #--Refresh overLoaded too..
        self.exGroup_mods.clear()
        for modName in self.ordered:
            maExGroup = reExGroup.match(modName)
            if maExGroup: 
                exGroup = maExGroup.group(1)
                mods = self.exGroup_mods.setdefault(exGroup,[])
                mods.append(modName)
        #--Refresh merged/imported lists.
        self.merged,self.imported = self.getSemiActive(self.ordered)

    def getSemiActive(self,masters):
        """Returns (merged,imported) mods made semi-active by Bashed Patch."""
        merged,imported,nullDict = set(),set(),{}
        for modName,modInfo in [(modName,self[modName]) for modName in masters]:
            if modInfo.header.author != 'BASHED PATCH': continue
            patchConfigs = self.table.getItem(modName,'bash.patch.configs',None)
            if not patchConfigs: continue
            if patchConfigs.get('PatchMerger',nullDict).get('isEnabled'):
                configChecks = patchConfigs['PatchMerger']['configChecks']
                for modName in configChecks:
                    if configChecks[modName]: merged.add(modName)
            imported.update(patchConfigs.get('ImportedMods',tuple()))
        return (merged,imported)

    def getModList(self,fileInfo=None):
        """Returns mod list as text. If fileInfo is provided will show mod list 
        for its masters. Otherwise will show currently loaded mods."""
        #--Setup
        log = LogFile(cStringIO.StringIO())
        log.out.write('[codebox]')
        if fileInfo:
            masters = set(fileInfo.header.masters)
            missing = sorted([x for x in masters if x not in self])
            log.setHeader(_('Missing Masters for: ')+fileInfo.name)
            for mod in missing:
                log('xx '+mod)
            log.setHeader(_('Masters for: ')+fileInfo.name)
            present = set(x for x in masters if x in self)
            if fileInfo.name in self: #--In case is bashed patch
                present.add(fileInfo.name)
            merged,imported = self.getSemiActive(present)
        else:
            log.setHeader(_('Active Mod Files:'))
            masters = set(self.ordered)
            merged,imported = self.merged,self.imported
        anchors = set(mod for mod in self.data if mod[0] in '.=+')
        allMods = masters | merged | imported | anchors
        allMods = self.getOrdered([x for x in allMods if x in self])
        #--List
        modIndex,header = 0, None
        for name in allMods:
            if name in masters:
                prefix = '%02X' % (modIndex)
                modIndex += 1
            elif name in anchors:
                match = re.match('^[\.+= ]*(.*?)\.es[pm]',name)
                if match: name = match.group(1)
                header = '==  ' +name
                continue
            elif name in merged:
                prefix = '++'
            else:
                prefix = '**'
            version = self.getVersion(name)
            if header:
                log(header)
                header = None
            if version: 
                log(_('%s  %s  [Version %s]') % (prefix,name,version))
            else:
                log('%s  %s' % (prefix,name))
        log('[/codebox]')
        return winNewLines(log.out.getvalue())

    #--Circular Masters
    def circularMasters(self,stack,masters=None):
        stackTop = stack[-1]
        masters = masters or (stackTop in self.data and self.data[stackTop].masterNames)
        if not masters: return False
        for master in masters:
            if master in stack: 
                return True
            if self.circularMasters(stack+[master]):
                return True
        return False

    #--Get load order
    def getOrdered(self,modNames,asTuple=True):
        """Sort list of mod names into their load order. ASSUMES MODNAMES ARE UNIQUE!!!"""
        data = self.data
        modNames = list(modNames) #--Don't do an in-place sort.
        modNames.sort()
        modNames.sort(key=lambda a: (a in data) and data[a].mtime) #--Sort on modified
        modNames.sort(key=lambda a: a[-1].lower()) #--Sort on esm/esp
        #--Match Bethesda's esm sort order
        #  - Start with masters in chronological order.
        #  - For each master, if its masters (mm's) are not already in list, 
        #    then place them ahead of master... but in REVERSE order. E.g., last
        #    grandmaster will be first to be added.
        def preMaster(modName,modDex):
            """If necessary, move grandmasters in front of master -- but in 
            reverse order."""
            if self.data.has_key(modName):
                mmNames = list(self.data[modName].masterNames[:])
                mmNames.reverse()
                for mmName in mmNames:
                    if mmName in modNames:
                        mmDex = modNames.index(mmName)
                        #--Move master in front and pre-master it too.
                        if mmDex > modDex:
                            del modNames[mmDex]
                            modNames.insert(modDex,mmName)
                            modDex = 1 + preMaster(mmName,modDex)
            return modDex
        #--Read through modNames.
        modDex = 1
        while modDex < len(modNames):
            modName = modNames[modDex]
            if modName[-1].lower() != 'm': break
            if self.circularMasters([modName]):
                modDex += 1
            else:
                modDex = 1 + preMaster(modName,modDex)
        #--Convert? and return
        if asTuple:
            return tuple(modNames)
        else:
            return modNames

    def selectExact(self,modNames):
        """Selects exactly the specified set of mods."""
        del self.plugins.selected[:]
        missing,extra = [],[]
        modSet = set(self.keys())
        for modName in modNames:
            if modName not in self: 
                missing.append(modName)
                continue
            try:
                self.select(modName,False,modSet)
            except PluginsFullError:
                extra.append(modName)
        #--Save
        self.refreshInfoLists()
        self.plugins.save()
        #--Done/Error Message
        if missing or extra:
            message = ''
            if missing:
                message += _("Some mods were unavailable and were skipped:\n")
                message += '\n* '.join(missing)
            if extra:
                if missing: message += '\n'
                message += _("Mod list is full, so some mods were skipped:\n")
                message += '\n* '.join(extra)
            return message
        else:
            return None

    #--Mod Specific ----------------------------------------------------------
    def rightFileType(self,fileName):
        """Bool: File is a mod."""
        return reModExt.search(fileName)

    #--Refresh File
    def refreshFile(self,fileName):
        try:
            FileInfos.refreshFile(self,fileName)
        finally:
            self.refreshInfoLists()

    def isSelected(self,modFile):
        """True if modFile is selected (active)."""
        return (modFile in self.ordered)

    def select(self,fileName,doSave=True,modSet=None,children=None):
        """Adds file to selected."""
        try:
            plugins = self.plugins
            children = (children or tuple()) + (fileName,)
            if fileName in children[:-1]: 
                raise BoshError(_('Circular Masters: ')+' >> '.join(children))
            #--Select masters
            if modSet == None: modSet = set(self.keys())
            for master in self[fileName].header.masters:
                if master in modSet:
                    self.select(master,False,modSet,children)
            #--Select in plugins
            if fileName not in plugins.selected:
                if len(plugins.selected) >= 255:
                    raise PluginsFullError
                plugins.selected.append(fileName)
        finally:
            if doSave:
                plugins.save()
                self.refreshInfoLists()

    def unselect(self,fileName,doSave=True):
        """Removes file from selected."""
        #--Unselect self
        plugins = self.plugins
        plugins.remove(fileName)
        #--Unselect children
        for selFile in plugins.selected[:]:
            #--Already unselected or missing?
            if not self.isSelected(selFile) or selFile not in self.data: 
                continue
            #--One of selFile's masters?
            for master in self[selFile].header.masters:
                if master == fileName:
                    self.unselect(selFile,False)
                    break
        #--Save
        if doSave: 
            self.refreshInfoLists()
            plugins.save()

    def hasTimeConflict(self,modName):
        """True if there is another mod with the same mtime."""
        mtime = self[modName].mtime
        mods = self.mtime_mods.get(mtime,tuple())
        return len(mods) > 1

    def hasActiveTimeConflict(self,modName):
        """True if there is another mod with the same mtime."""
        if not self.isSelected(modName): return False
        mtime = self[modName].mtime
        mods = self.mtime_selected.get(mtime,tuple())
        return len(mods) > 1

    #--Mod move/delete/rename -------------------------------------------------
    def rename(self,oldName,newName):
        """Renames member file from oldName to newName."""
        isSelected = self.isSelected(oldName)
        if isSelected: self.unselect(oldName)
        FileInfos.rename(self,oldName,newName)
        self.refreshInfoLists()
        if isSelected: self.select(newName)

    def delete(self,fileName):
        """Deletes member file."""
        self.unselect(fileName)
        FileInfos.delete(self,fileName)

    def move(self,fileName,destDir):
        """Moves member file to destDir."""
        self.unselect(fileName)
        FileInfos.move(self,fileName,destDir)
    
    #--Mod info/modify --------------------------------------------------------
    def getVersion(self,fileName):
        """Extracts and returns version number for fileName from header.hedr.description."""
        if not fileName in self.data or not self.data[fileName].header:
            return ''
        maVersion = reVersion.search(self.data[fileName].header.description)
        return (maVersion and maVersion.group(2)) or ''

    #--Oblivion 1.1/SI Swapping -----------------------------------------------
    def getOblivionVersions(self):
        """Returns tuple of Oblivion versions."""
        reOblivion = re.compile('^Oblivion(|_SI|_1.1).esm$')
        self.voAvailable.clear()
        for name,info in self.data.items():
            maOblivion = reOblivion.match(name)
            if maOblivion and info.size in self.size_voVersion:
                self.voAvailable.add(self.size_voVersion[info.size])
        self.voCurrent = self.size_voVersion.get(self.data[Path.get('Oblivion.esm')].size,None)

    def setOblivionVersion(self,newVersion):
        """Swaps Oblivion.esm to to specified version."""
        #--Old info
        baseName = Path.get('Oblivion.esm')
        newSize = self.version_voSize[newVersion]
        oldSize = self.data[baseName].size
        if newSize == oldSize: return
        if oldSize not in self.size_voVersion: 
            raise StateError(_("Can't match current Oblivion.esm to known version."))
        oldName = Path.get('Oblivion_'+self.size_voVersion[oldSize]+'.esm')
        if self.dir.join(oldName).exists(): 
            raise StateError(_("Can't swap: %s already exists.") % (oldName,))
        newName = Path.get('Oblivion_'+newVersion+'.esm')
        if not self.dir.join(newName).exists(): 
            raise StateError(_("Can't swap: %s doesn't exist.") % (newName,))
        #--Rename
        baseTime = self.data[baseName].mtime
        newTime  = self.data[newName].mtime
        basePath,newPath,oldPath = [self.dir.join(name) for name in (baseName,newName,oldName)]
        basePath.rename(oldPath)
        newPath.rename(basePath)
        #--Reset mtimes
        basePath.setmtime(baseTime)
        oldPath.setmtime(newTime)
        self.mtimes[oldName] = newTime
        self.voCurrent = newVersion
        #if oldName in self.data: self.data[oldName].mtime = oldPath.getmtime()
 
    #--Resource Replacers -----------------------------------------------------
    def getResourceReplacers(self):
        """Returns list of ResourceReplacer objects for subdirectories of Replacers directory."""
        replacers = {}
        replacerDir = self.dir.join('Replacers')
        if not replacerDir.exists():
            return replacers
        if 'bosh.resourceReplacer.applied' not in settings:
            settings['bosh.resourceReplacer.applied'] = []
        for name in replacerDir.list():
            path = replacerDir.join(name)
            if path.isdir():
                replacers[name] = ResourceReplacer(replacerDir,name)
        return replacers

#------------------------------------------------------------------------------
class SaveInfos(FileInfos): 
    """SaveInfo collection. Represents save directory and related info."""
    #--Init
    def __init__(self):
        self.iniMTime = 0
        self.refreshLocalSave()
        FileInfos.__init__(self,self.dir,SaveInfo)
        self.profiles = Table(dirs['saveBase'].join('BashProfiles.dat'),dirs['userApp'].join('Profiles.pkl'))

    #--Right File Type (Used by Refresh)
    def rightFileType(self,fileName):
        """Bool: File is a mod."""
        return reSaveExt.search(fileName)

    #--Refresh
    def refresh(self):
        if self.refreshLocalSave():
            self.data.clear()
            self.table.save()
            self.table = Table(self.dir.join('Bash','Table.dat'),self.dir.join('Bash','Table.pkl'))
        return FileInfos.refresh(self)

    #--Local Saves
    def getLocalSaveDirs(self):
        """Returns a list of possible local save directories, NOT including the base directory."""
        localSaveDirs = []
        baseSaves = dirs['saveBase'].join('Saves')
        if not baseSaves.exists(): return localSaveDirs
        for fileName in baseSaves.list():
            if fileName != 'Bash' and baseSaves.join(fileName).isdir():
                localSaveDirs.append(fileName)
        return localSaveDirs

    def refreshLocalSave(self):
        """Refreshes self.localSave and self.dir."""
        self.localSave = getattr(self,'localSave','Saves\\')
        self.dir = dirs['saveBase'].join(self.localSave)
        if oblivionIni.path.exists() and (oblivionIni.path.getmtime() != self.iniMTime):
            self.localSave = oblivionIni.getSetting('General','SLocalSavePath','Saves\\')
            self.iniMTime = oblivionIni.path.getmtime()
            return True
        else:
            return False

    def setLocalSave(self,localSave):
        """Sets SLocalSavePath in Oblivion.ini."""
        self.localSave = localSave
        oblivionIni.saveSetting('General','SLocalSavePath',localSave)
        self.iniMTime = oblivionIni.path.getmtime()
        self.refresh()

    #--Enabled
    def isEnabled(self,fileName):
        """True if fileName is enabled)."""
        return (fileName[-4:].lower() == '.ess')

    def enable(self,fileName,value=True):
        """Enables file by changing extension to 'ess' (True) or 'esr' (False)."""
        isEnabled = self.isEnabled(fileName)
        if isEnabled or value == isEnabled or re.match('(autosave|quicksave)',fileName,re.I):
            return fileName
        (root,ext) = fileName.splitext()
        newName = root + ((value and '.ess') or '.esr')
        self.rename(fileName,newName)
        return newName
    
    def disable(self,fileName):
        """Disables file by changing ext to '.esr'."""
        self.disable(fileName,False)

#------------------------------------------------------------------------------
class ReplacersData(DataDict): 
    def __init__(self):
        """Initialize."""
        self.dir = dirs['mods'].join("Replacers")
        self.data = {}

    #--Refresh
    def refresh(self):
        """Refresh list of screenshots."""
        newData = modInfos.getResourceReplacers()
        changed = (set(self.data) != set(newData))
        self.data = newData
        return changed

#------------------------------------------------------------------------------
class ScreensData(DataDict): 
    def __init__(self):
        """Initialize."""
        self.dir = dirs['app']
        self.data = {} #--data[Path] = (ext,mtime)

    def refresh(self):
        """Refresh list of screenshots."""
        self.dir = dirs['app']
        ssBase = Path.get(oblivionIni.getSetting('Display','SScreenShotBaseName','ScreenShot'))
        if ssBase.head():
            self.dir = self.dir.join(ssBase.head())
        newData = {}
        reImageExt = re.compile(r'\.(bmp|jpg)$',re.I)
        #--Loop over files in directory
        for fileName in self.dir.list():
            filePath = self.dir.join(fileName)
            maImageExt = reImageExt.search(fileName)
            if maImageExt and filePath.isfile(): 
                newData[fileName] = (maImageExt.group(1).lower(),filePath.getmtime())
        changed = (self.data != newData)
        self.data = newData
        return changed

    def delete(self,fileName):
        """Deletes member file."""
        filePath = self.dir.join(fileName)
        filePath.remove()
        del self.data[fileName]

#------------------------------------------------------------------------------
class Messages(DataDict): 
    def __init__(self):
        """Initialize."""
        self.path = dirs['saveBase'].join('Messages.dat')
        self.hasChanged = False
        self.data = {} #--data[hash] = (subject,author,date,text)
        self.loaded = False

    def refresh(self):
        if not self.loaded:
            self.load()
            self.loaded = True

    def load(self):
        """Refresh list of data."""
        if self.path.exists():
            ins = self.path.open('rb')
            store = cPickle.load(ins)
            self.data.clear()
            self.data.update(store['data'])
            ins.close()

    def save(self):
        """Saves to pickle file."""
        if self.hasChanged:
            store = {}
            store['data'] = self.data
            filePath = self.path
            tempPath = filePath+'.tmp'
            fileDir = filePath.head()
            fileDir.makedirs()
            tempFile = tempPath.open('wb')
            cPickle.dump(store,tempFile,-1)
            tempFile.close()
            renameFile(tempPath,filePath,True)
            self.hasChanged = False

    def delete(self,key):
        """Delete entry."""
        del self.data[key]
        self.hasChanged = True

    def search(self,term):
        """Search entries for term."""
        term = term.strip()
        if not term: return None
        items = []
        reTerm = re.compile(term,re.I)
        for key,(subject,author,date,text) in self.data.items():
            if (reTerm.search(subject) or
                reTerm.search(author) or
                reTerm.search(text)
                ):
                items.append(key)
        return items

    def writeText(self,path,*keys):
        """Return html text for each key."""
        out = path.open('w')
        out.write(bush.messagesHeader)
        for key in keys:
            out.write(self.data[key][3])
            out.write('\n<br />')
        out.write("\n</div></body></html>")
        out.close()

    def importArchive(self,path):
        """Import archive file into data."""
        #--Today, yesterday handling
        import datetime
        maPathDate = re.match(r'(\d+)\.(\d+)\.(\d+)',path.tail())
        dates = {'today':None,'yesterday':None,'previous':None}
        if maPathDate:
            year,month,day = map(int,maPathDate.groups())
            if year < 100: year = 2000+year
            dates['today'] = datetime.datetime(year,month,day)
            dates['yesterday'] = dates['today'] - datetime.timedelta(1)
        reRelDate = re.compile('(Today|Yesterday), (.*)')
        def getTime(str):
            maRelDate = reRelDate.search(str)
            if not maRelDate:
                date = time.strptime(str,'%b %d %Y, %I:%M %p')[:-1]+(0,)
                dates['previous'] = datetime.datetime(*date[0:3])
            else:
                if not dates['yesterday']:
                    dates['yesterday'] = dates['previous'] + timedelta(1)
                    dates['today'] = dates['yesterday'] + timedelta(1)
                strDay,strTime = maRelDate.groups()
                date = time.strptime(strTime,'%I:%M %p')
                ymd = dates[strDay.lower()]
                date = ymd.timetuple()[0:3]+date[3:6]+(0,0,0)
            return time.mktime(date)
        #--Html entity substitution
        from htmlentitydefs import name2codepoint
        def subHtmlEntity(match):
            entity = match.group(2)
            if match.group(1) == "#":
                return unichr(int(entity)).encode()
            else:
                cp = name2codepoint.get(entity)
                if cp:
                    return unichr(cp).encode()
                else:
                    return match.group()
        #--Re's
        reHtmlEntity = re.compile("&(#?)(\d{1,5}|\w{1,8});")
        reBody         = re.compile('<body>')
        reWrapper      = re.compile('<div id=["\']ipbwrapper["\']>') #--Will be removed
        reMessage      = re.compile('<div class="borderwrapm">')
        reMessageOld   = re.compile("<div class='tableborder'>")
        reTitle        = re.compile('<div class="maintitle">PM: (.+)</div>')
        reTitleOld     = re.compile('<div class=\'maintitle\'><img[^>]+>&nbsp;')
        reSignature    = re.compile('<div class="formsubtitle">')
        reSignatureOld = re.compile('<div class=\'pformstrip\'>')
        reSent         = re.compile('Sent (by|to) <b>(.+)</b> on (.+)</div>')
        #--Final setup, then parse the file
        (HEADER,BODY,MESSAGE) = range(3)
        mode = HEADER
        buff = None
        subject = "<No Subject>"
        ins = path.open()
        for line in ins:
            line = reMessageOld.sub('<div class="borderwrapm">',line)
            line = reTitleOld.sub('<div class="maintitle">',line)
            line = reSignatureOld.sub('<div class="formsubtitle">',line)
            #print mode,'>>',line,
            if mode == HEADER: #--header
                if reBody.search(line):
                    mode = BODY
            elif mode == BODY:
                if reMessage.search(line):
                    subject = "<No Subject>"
                    buff = cStringIO.StringIO()
                    buff.write(reWrapper.sub('',line))
                    mode = MESSAGE                    
            elif mode == MESSAGE:
                if reTitle.search(line):
                    subject = reTitle.search(line).group(1)
                    subject = reHtmlEntity.sub(subHtmlEntity,subject)
                    buff.write(line)
                elif reSignature.search(line):
                    maSent = reSent.search(line)
                    if maSent:
                        direction = maSent.group(1)
                        author = maSent.group(2)
                        date = getTime(maSent.group(3))
                        messageKey = '::'.join((subject,author,`int(date)`))
                        newSent = 'Sent %s <b>%s</b> on %s</div>' % (direction,
                            author,time.strftime('%b %d %Y, %I:%M %p',time.localtime(date)))
                        line = reSent.sub(newSent,line,1)
                        buff.write(line)
                        self.data[messageKey] = (subject,author,date,buff.getvalue())
                    buff.close()
                    buff = None
                    mode = BODY
                else:
                    buff.write(line)
        ins.close()
        self.hasChanged = True
        self.save()

# Utilities -------------------------------------------------------------------
#------------------------------------------------------------------------------
class ActorLevels:
    """Package: Functions for manipulating actor levels."""

    @staticmethod
    def dumpText(modInfo,outPath,progress=None):
        """Export NPC level data to text file."""
        outPath = Path.get(outPath)
        progress = progress or Progress()
        #--Mod levels
        progress(0,_('Loading ')+modInfo.name)
        loadFactory= LoadFactory(False,MreNpc)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        offsetFlag = 0x80
        npcLevels = {}
        for npc in modFile.NPC_.records:
            if npc.flags.pcLevelOffset:
                npcLevels[npc.formid] = (npc.eid,npc.level,npc.calcMin, npc.calcMax)
        #--Oblivion Levels (for comparison)
        progress(0.25,_('Loading Oblivion.esm'))
        obFactory= LoadFactory(False,MreNpc)
        obInfo = modInfos[Path.get('Oblivion.esm')]
        obFile = ModFile(obInfo,obFactory)
        obFile.load(True)
        obNPCs = {}
        for npc in obFile.NPC_.records:
            obNPCs[npc.formid] = npc
        #--File, column headings
        progress(0.75,_('Writing ')+outPath.tail())
        out = outPath.open('w')
        headings = (_('FormId'),_('EditorId'),_('Offset'),_('CalcMin'),_('CalcMax'),_(''),
            _('Old bOffset'),_('Old Offset'),_('Old CalcMin'),_('Old CalcMax'),)
        out.write('"'+('","'.join(headings))+'"\n')
        #--Sort by eid and print
        for formid in sorted(npcLevels.keys(),key=lambda a: npcLevels[a][0]):
            npcLevel = npcLevels[formid]
            out.write('"0x%08X","%s",%d,%d,%d' % ((formid,)+npcLevel))
            obNPC = obNPCs.get(formid,None)
            if obNPC:
                flagged = (obNPC.flags.pcLevelOffset and offsetFlag) and 1 or 0
                out.write(',,%d,%d,%d,%d' % (flagged,obNPC.level,obNPC.calcMin,obNPC.calcMax))
            out.write('\n')
        out.close()
        progress(1,_('Done'))

    @staticmethod
    def loadText(modInfo,inPath,progress=None):
        """Import NPC level data from text file."""
        inPath = Path.get(inPath)
        progress = progress or Progress()
        #--Sort and print
        progress(0,_('Reading ')+inPath.tail())
        inNPCs = {}
        ins = CsvReader(inPath)
        for fields in ins:
            if '0x' not in fields[0]: continue
            inNPCs[int(fields[0],0)] = tuple(map(int,fields[2:5]))
        ins.close()
        #--Load Mod
        progress(0.25,_('Loading ')+modInfo.name)
        loadFactory= LoadFactory(True,MreNpc)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        offsetFlag = 0x80
        npcLevels = {}
        for npc in modFile.NPC_.records:
            if npc.formid in inNPCs:
                (npc.level, npc.calcMin, npc.calcMax) = inNPCs[npc.formid]
                npc.setChanged()
        progress(0.5,_('Saving ')+modInfo.name)
        modFile.safeSave()
        progress(1.0,_('Done'))

#------------------------------------------------------------------------------
class EditorIds:
    """Editor ids for records, with functions for importing/exporting from/to mod/text file."""
    def __init__(self,types=None,aliases=None):
        """Initialize."""
        self.type_id_eid = {} #--eid = eids[type][longid]
        self.old_new = {}
        if types:
            self.types = types
        else:
            self.types = set(MreRecord.topTypes)
            #self.types.add('DIAL')
        self.aliases = aliases or {}

    def readFromMod(self,modInfo):
        """Imports eids from specified mod."""
        type_id_eid,types = self.type_id_eid,self.types
        loadFactory= LoadFactory(False,*types)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        for type in types:
            typeBlock = modFile.tops.get(type,None)
            if not typeBlock: continue
            if type not in type_id_eid: type_id_eid[type] = {}
            id_eid = type_id_eid[type]
            for record in typeBlock.getActiveRecords():
                longid = mapper(record.formid)
                eid = record.getSubString('EDID')
                if eid: id_eid[longid] = eid

    def writeToMod(self,modInfo):
        """Exports eids to specified mod."""
        type_id_eid,types = self.type_id_eid,self.types
        loadFactory= LoadFactory(True,*types)
        loadFactory.addClass(MreScpt)
        loadFactory.addClass(MreQust)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        changed = []
        for type in types:
            id_eid = type_id_eid.get(type,None)
            typeBlock = modFile.tops.get(type,None)
            if not id_eid or not typeBlock: continue
            for record in typeBlock.records:
                longid = mapper(record.formid)
                eid = record.getSubString('EDID')
                newEid = id_eid.get(longid)
                if eid and newEid and newEid != eid:
                    record.setSubString('EDID',newEid)
                    changed.append((eid,newEid))
        #--Update scripts
        old_new = dict(self.old_new)
        old_new.update(dict([(oldEid.lower(),newEid) for oldEid,newEid in changed]))
        changed.extend(self.changeScripts(modFile,old_new))
        #--Done
        if changed: modFile.safeSave()
        return changed

    def changeScripts(self,modFile,old_new):
        """Changes scripts in modfile according to changed."""
        changed = []
        if not old_new: return changed
        reWord = re.compile('\w+')
        def subWord(match):
            word = match.group(0)
            newWord = old_new.get(word.lower())
            if not newWord:
                return word
            else:
                return newWord
        #--Scripts
        byEid = lambda a: a.eid
        for script in sorted(modFile.SCPT.records,key=byEid):
            if not script.text: continue
            newText = reWord.sub(subWord,script.text)
            if newText != script.text: 
                header = '\r\n\r\n; %s %s\r\n' % (script.eid,'-'*(77-len(script.eid)))
                script.text = newText
                script.setChanged()
                changed.append((_("Script"),script.eid))
        #--Quest Scripts
        for quest in sorted(modFile.QUST.records,key=byEid):
            questChanged = False
            for stage in quest.stages:
                for entry in stage.entries:
                    oldScript = entry.script
                    if not oldScript: continue
                    newScript = reWord.sub(subWord,oldScript)
                    if newScript != oldScript:
                        entry.script = newScript
                        questChanged = True
            if questChanged:
                changed.append((_("Quest"),quest.eid))
                quest.setChanged()
        #--Done
        return changed

    def readFromText(self,textPath):
        """Imports eids from specified text file."""
        textPath = Path.get(textPath)
        type_id_eid = self.type_id_eid
        aliases = self.aliases
        ins = CsvReader(textPath)
        reNewEid = re.compile('^[a-zA-Z][a-zA-Z0-9]+$')
        for fields in ins:
            if len(fields) < 4 or fields[2][:2] != '0x': continue
            type,mod,objectIndex,eid = fields[:4]
            longid = (Path.get(aliases.get(mod,mod)),int(objectIndex[2:],16))
            if not reNewEid.match(eid): 
                continue
            elif type in type_id_eid:
                type_id_eid[type][longid] = eid
            else:
                type_id_eid[type] = {longid:eid}
            #--Explicit old to new def? (Used for script updating.)
            if len(fields) > 4:
                self.old_new[fields[4].lower()] = fields[3]
        ins.close()

    def writeToText(self,textPath):
        """Exports eids to specified text file."""
        textPath = Path.get(textPath)
        type_id_eid = self.type_id_eid
        headFormat = '"%s","%s","%s","%s"\n'
        rowFormat = '"%s","%s","0x%06X","%s"\n'
        out = file(textPath,'w')
        out.write(headFormat % (_('Type'),_('Mod Name'),_('ObjectIndex'),_('Editor Id')))
        for type in sorted(type_id_eid):
            id_eid = type_id_eid[type]
            for id in sorted(id_eid,key = lambda a: id_eid[a]):
                out.write(rowFormat % (type,id[0],id[1],id_eid[id]))
        out.close()

#------------------------------------------------------------------------------
class FormidReplacer:
    """Replaces one set of formids with another."""

    def __init__(self,types=None,aliases=None):
        """Initialize."""
        self.types = types or ('NPC_','CREA','LVLC','LVLI','LVSP','CONT','FLOR')
        self.aliases = aliases or {} #--For aliasing mod names
        self.old_new = {} #--Maps old formid to new formid
        self.old_eid = {} #--Maps old formid to old editor id
        self.new_eid = {} #--Maps new formid to new editor id

    def readFromText(self,textPath):
        """Reads replacment data from specified text file."""
        old_new,old_eid,new_eid = self.old_new,self.old_eid,self.new_eid
        textPath = Path.get(textPath)
        aliases = self.aliases
        ins = CsvReader(textPath)
        pack,unpack = struct.pack,struct.unpack
        for fields in ins:
            if len(fields) < 7 or fields[2][:2] != '0x' or fields[6][:2] != '0x': continue
            oldMod,oldObj,oldEid,newEid,newMod,newObj = fields[1:7]
            oldId = (Path.get(aliases.get(oldMod,oldMod)),int(oldObj,16))
            newId = (Path.get(aliases.get(newMod,newMod)),int(newObj,16))
            old_new[oldId] = newId
            old_eid[oldId] = oldEid
            new_eid[newId] = newEid
        ins.close()

    def updateMod(self, modInfo):
        """Updates specified mod file."""
        types = self.types
        classes = [MreRecord.type_class[type] for type in types]
        loadFactory= LoadFactory(True,*classes)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        #--Create  filtered versions of mappers.
        mapper = modFile.getShortMapper()
        masters = modFile.tes4.masters+[modFile.fileInfo.name]
        short = dict((oldId,mapper(oldId)) for oldId in self.old_eid if oldId[0] in masters)
        short.update((newId,mapper(newId)) for newId in self.new_eid if newId[0] in masters)
        old_eid = dict((short[oldId],eid) for oldId,eid in self.old_eid.items() if oldId in short)
        new_eid = dict((short[newId],eid) for newId,eid in self.new_eid.items() if newId in short)
        old_new = dict((short[oldId],short[newId]) for oldId,newId in self.old_new.items() 
            if (oldId in short and newId in short))
        if not old_new: return False
        #--Swapper function
        old_count = {} 
        def swapper(oldId):
            newId = old_new.get(oldId,None)
            if newId: 
                old_count.setdefault(oldId,0)
                old_count[oldId] += 1
                return newId
            else:
                return oldId
        #--Do swap on all records
        for type in types:
            for record in getattr(modFile,type).getActiveRecords():
                record.mapFormids(swapper,True)
                record.setChanged()
        #--Done
        if not old_count: return False
        modFile.safeSave()
        entries = [(count,old_eid[oldId],new_eid[old_new[oldId]]) for oldId,count in old_count.items()]
        entries.sort(key=lambda a: a[1])
        return '\n'.join(['%3d %s >> %s' % entry for entry in entries])

#------------------------------------------------------------------------------
class FullNames:
    """Names for records, with functions for importing/exporting from/to mod/text file."""
    defaultTypes = set((
        'ALCH', 'AMMO', 'APPA', 'ARMO', 'BOOK', 'BSGN', 'CLAS', 'CLOT', 'CONT', 'CREA', 'DOOR',
        'FACT', 'FLOR', 'INGR', 'KEYM', 'LIGH', 'MISC', 'NPC_', 'RACE', 'SGST', 'SLGM', 'SPEL',
        'WEAP',))

    def __init__(self,types=None,aliases=None):
        """Initialize."""
        self.type_id_name = {} #--(eid,name) = type_id_name[type][longid]
        self.types = types or FullNames.defaultTypes
        self.aliases = aliases or {}

    def readFromMod(self,modInfo):
        """Imports type_id_name from specified mod."""
        type_id_name,types = self.type_id_name, self.types
        loadFactory= LoadFactory(False,*types)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        for type in types:
            typeBlock = modFile.tops.get(type,None)
            if not typeBlock: continue
            if type not in type_id_name: type_id_name[type] = {}
            id_name = type_id_name[type]
            for record in typeBlock.getActiveRecords():
                longid = mapper(record.formid)
                eid = record.getSubString('EDID')
                full = record.getSubString('FULL')
                if eid and full:
                    id_name[longid] = (eid,full)

    def writeToMod(self,modInfo):
        """Exports type_id_name to specified mod."""
        type_id_name,types = self.type_id_name,self.types
        loadFactory= LoadFactory(True,*types)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        changed = {}
        for type in types:
            id_name = type_id_name.get(type,None)
            typeBlock = modFile.tops.get(type,None)
            if not id_name or not typeBlock: continue
            for record in typeBlock.records:
                longid = mapper(record.formid)
                full = record.getSubString('FULL')
                eid,newFull = id_name.get(longid,(0,0))
                if full and newFull and newFull != full:
                    record.setSubString('FULL',newFull)
                    changed[eid] = (full,newFull)
        if changed: modFile.safeSave()
        return changed

    def readFromText(self,textPath):
        """Imports type_id_name from specified text file."""
        textPath = Path.get(textPath)
        type_id_name = self.type_id_name
        aliases = self.aliases
        ins = CsvReader(textPath)
        for fields in ins:
            if len(fields) < 5 or fields[2][:2] != '0x': continue
            type,mod,objectIndex,eid,full = fields[:5]
            longid = (Path.get(aliases.get(mod,mod)),int(objectIndex[2:],16))
            if type in type_id_name:
                type_id_name[type][longid] = (eid,full)
            else:
                type_id_name[type] = {longid:(eid,full)}
        ins.close()

    def writeToText(self,textPath):
        """Exports type_id_name to specified text file."""
        textPath = Path.get(textPath)
        type_id_name = self.type_id_name
        headFormat = '"%s","%s","%s","%s","%s"\n'
        rowFormat = '"%s","%s","0x%06X","%s","%s"\n'
        out = file(textPath,'w')
        out.write(headFormat % (_('Type'),_('Mod Name'),_('ObjectIndex'),_('Editor Id'),_('Name')))
        for type in sorted(type_id_name):
            id_name = type_id_name[type]
            longids = id_name.keys()
            longids.sort(key=lambda a: id_name[a][0])
            longids.sort(key=lambda a: a[0])
            for longid in longids:
                eid,name = id_name[longid]
                out.write(rowFormat % (type,longid[0],longid[1],eid,name))
        out.close()

#------------------------------------------------------------------------------
class ItemStats:
    """Statistics for armor and weapons, with functions for importing/exporting from/to mod/text file."""

    def __init__(self,types=None,aliases=None):
        """Initialize."""
        #--type_stats[type] = ...
        #--AMMO: (eid, weight, value, damage, speed, epoints)
        #--ARMO: (eid, weight, value, health, strength)
        #--WEAP: (eid, weight, value, health, damage, speed, reach, epoints)
        self.type_stats = {'AMMO':{},'ARMO':{},'WEAP':{}}
        self.type_attrs = {
            'AMMO':('eid', 'weight', 'value', 'damage', 'speed', 'enchantPoints'),
            'ARMO':('eid', 'weight', 'value', 'health', 'strength'),
            'WEAP':('eid', 'weight', 'value', 'health', 'damage', 'speed', 'reach', 'enchantPoints'),
            }
        self.aliases = aliases or {} #--For aliasing mod names

    def readFromMod(self,modInfo):
        """Reads stats from specified mod."""
        loadFactory= LoadFactory(False,MreAmmo,MreArmo,MreWeap)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        for type in self.type_stats:
            stats, attrs = self.type_stats[type], self.type_attrs[type]
            for record in getattr(modFile,type).getActiveRecords():
                longid = mapper(record.formid)
                recordDict = record.__dict__
                stats[longid] = tuple(recordDict[attr] for attr in attrs)

    def writeToMod(self,modInfo):
        """Writes stats to specified mod."""
        loadFactory= LoadFactory(True,MreAmmo,MreArmo,MreWeap)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        mapper = modFile.getLongMapper()
        changed = {} #--changed[modName] = numChanged
        for type in self.type_stats:
            stats, attrs = self.type_stats[type], self.type_attrs[type]
            for record in getattr(modFile,type).getActiveRecords():
                longid = mapper(record.formid)
                itemStats = stats.get(longid,None)
                if not itemStats: continue
                record.__dict__.update(zip(attrs,itemStats))
                record.setChanged()
                changed[longid[0]] = 1 + changed.get(longid[0],0)
        if changed: modFile.safeSave()
        return changed

    def readFromText(self,textPath):
        """Reads stats from specified text file."""
        ammo, armor, weapons = [self.type_stats[type] for type in ('AMMO','ARMO','WEAP')]
        textPath = Path.get(textPath)
        aliases = self.aliases
        ins = CsvReader(textPath)
        pack,unpack = struct.pack,struct.unpack
        sfloat = lambda a: unpack('f',pack('f',float(a)))[0] #--Force standard precision
        for fields in ins:
            if len(fields) < 3 or fields[2][:2] != '0x': continue
            type,modName,objectStr,eid = fields[0:4]
            longid = (Path.get(aliases.get(modName,modName)),int(objectStr[2:],16))
            if type == 'AMMO':
                ammo[longid] = (eid,) + tuple(func(field) for func,field in 
                    #--(weight, value, damage, speed, enchantPoints)
                    zip((sfloat,int,int,sfloat,int),fields[4:9]))
            elif type == 'ARMO':
                armor[longid] = (eid,) + tuple(func(field) for func,field in 
                    #--(weight, value, health, strength)
                    zip((sfloat,int,int,int),fields[4:8]))
            elif type == 'WEAP':
                weapons[longid] = (eid,) + tuple(func(field) for func,field in 
                    #--(weight, value, health, damage, speed, reach, epoints)
                    zip((sfloat,int,int,int,sfloat,sfloat,int),fields[4:11]))
        ins.close()

    def writeToText(self,textPath):
        """Writes stats to specified text file."""
        out = Path.get(textPath).open('w')
        def getSortedIds(stats):
            longids = stats.keys()
            longids.sort(key=lambda a: stats[a][0])
            longids.sort(key=lambda a: a[0])
            return longids
        for type,format,header in (
            #--Ammo
            ('AMMO', csvFormat('sfiifi')+'\n',
                ('"' + '","'.join((_('Type'),_('Mod Name'),_('ObjectIndex'),
                _('Editor Id'),_('Weight'),_('Value'),_('Damage'),_('Speed'),_('EPoints'))) + '"\n')),
            #--Armor
            ('ARMO', csvFormat('sfiii')+'\n',
                ('"' + '","'.join((_('Type'),_('Mod Name'),_('ObjectIndex'),
                _('Editor Id'),_('Weight'),_('Value'),_('Health'),_('AR'))) + '"\n')),
            #--Weapons
            ('WEAP', csvFormat('sfiiiffi')+'\n',
                ('"' + '","'.join((_('Type'),_('Mod Name'),_('ObjectIndex'),
                _('Editor Id'),_('Weight'),_('Value'),_('Health'),_('Damage'),
                _('Speed'),_('Reach'),_('EPoints'))) + '"\n')),
            ):
            stats = self.type_stats[type]
            if not stats: continue
            out.write(header)
            for longid in getSortedIds(stats):
                out.write('"%s","%s","0x%06X",' % (type,longid[0],longid[1]))
                out.write(format % stats[longid])
        out.close()

#------------------------------------------------------------------------------
class ModDetails:
    """Details data for a mods file. Similar to TesCS Details view."""
    def __init__(self,modInfo=None,progress=None):
        """Initialize."""
        self.group_records = {} #--group_records[group] = [(formid0,eid0),(formid1,eid1),...]

    def readFromMod(self,modInfo,progress=None):
        """Extracts details from mod file."""
        def getRecordReader(ins,flags,size):
            """Decompress record data as needed."""
            if not MreRecord.flags1(flags).compressed:
                return (ins,ins.tell()+size)
            else:
                import zlib
                sizeCheck, = struct.unpack('I',ins.read(4))
                decomp = zlib.decompress(ins.read(size-4))
                if len(decomp) != sizeCheck: 
                    raise ModError(self.inName,
                        _('Mis-sized compressed data. Expected %d, got %d.') % (size,len(decomp)))
                reader = ModReader(modInfo.name,cStringIO.StringIO(decomp))
                return (reader,sizeCheck)
        progress = progress or Progress()
        group_records = self.group_records = {}
        records = group_records['TES4'] = []
        ins = ModReader(modInfo.name,modInfo.getPath().open('rb'))
        while not ins.atEnd():
            (type,size,str0,formid,uint2) = ins.unpackRecHeader()
            if type == 'GRUP':
                progress(1.0*ins.tell()/modInfo.size,_("Scanning: ")+str0)
                records = group_records.setdefault(str0,[])
                if str0 in ('CELL','WRLD','DIAL'):
                    ins.seek(size-20,1)
            elif type != 'GRUP':
                eid = ''
                nextRecord = ins.tell() + size
                recs,endRecs = getRecordReader(ins,str0,size)
                while recs.tell() < endRecs:
                    (type,size) = recs.unpackSubHeader()
                    if type == 'EDID':
                        eid = recs.readString(size)
                        break
                    ins.seek(size,1)
                records.append((formid,eid))
                ins.seek(nextRecord)
        ins.close()
        del group_records['TES4']

#------------------------------------------------------------------------------
class PCFaces:
    """Package: Objects and functions for working with face data."""
    flags = Flags(0L,Flags.getNames('name','race','gender','hair','eyes','iclass','stats','factions','modifiers','spells'))

    class PCFace(object):
        """Represents a face."""
        __slots__ = ('masters','eid','pcName','race','gender','eyes','hair',
            'hairLength','hairColor','fggs','fgga','fgts','level','attributes',
            'skills','health','baseSpell','fatigue','iclass','factions','modifiers','spells')
        def __init__(self):
            self.masters = []
            self.eid = self.pcName = 'generic'
            self.fggs = self.fgts = '\x00'*4*50
            self.fgga = '\x00'*4*30
            self.health = self.baseSpell = self.fatigue = self.level = 0
            self.skills = self.attributes = self.iclass = None
            self.factions = []
            self.modifiers = []
            self.spells = []

        def getGenderName(self):
            return self.gender and 'Female' or 'Male'
        
        def getRaceName(self):
            return bush.raceNames.get(self.race,_('Unknown'))

        def convertRace(self,fromRace,toRace):
            """Converts face from one race to another while preserving structure, etc."""
            for attr,num in (('fggs',50),('fgga',30),('fgts',50)):
                format = `num`+'f'
                sValues = list(struct.unpack(format,getattr(self,attr)))
                fValues = list(struct.unpack(format,getattr(fromRace,attr)))
                tValues = list(struct.unpack(format,getattr(toRace,attr)))
                for index,(sValue,fValue,tValue) in enumerate(zip(sValues,fValues,tValues)):
                    sValues[index] = sValue + fValue - tValue
                setattr(self,attr,struct.pack(format,*sValues))

    # SAVES -------------------------------------------------------------------
    @staticmethod
    def save_getNamePos(saveName,data,pcName):
        """Safely finds position of name within save ACHR data."""
        namePos = data.find(pcName)
        if namePos == -1:
            raise SaveFileError(saveName,_('Failed to find pcName in PC ACRH record.'))
        namePos2 = data.find(pcName,namePos+1)
        if namePos2 != -1:
            raise SaveFileError(saveName,_(
                'Uncertain about position of face data, probably because '
                'player character name is too short. Try renaming player '
                'character in save game.'))
        return namePos

    # Save Get ----------------------------------------------------------------
    @staticmethod
    def save_getFace(saveFile):
        """DEPRECATED. Same as save_getPlayerFace(saveFile)."""
        return PCFaces.save_getPlayerFace(saveFile)

    @staticmethod
    def save_getFaces(saveFile):
        """Returns player and created faces from a save file or saveInfo."""
        if isinstance(saveFile,SaveInfo):
            saveInfo = saveFile
            saveFile = SaveFile(saveInfo)
            saveFile.load()
        faces = PCFaces.save_getCreatedFaces(saveFile)
        playerFace = PCFaces.save_getPlayerFace(saveFile)
        faces[7] = playerFace
        return faces

    @staticmethod
    def save_getCreatedFace(saveFile,targetid):
        """Gets a particular created face."""
        return PCFaces.save_getCreatedFaces(saveFile,targetid).get(targetid)

    @staticmethod
    def save_getCreatedFaces(saveFile,targetid=None):
        """Returns created faces from savefile. If formid is supplied, will only 
        return created face with that formid.
        Note: Created NPCs do NOT use irefs!"""
        targetid = intArg(targetid)
        if isinstance(saveFile,SaveInfo):
            saveInfo = saveFile
            saveFile = SaveFile(saveInfo)
            saveFile.load()
        faces = {}
        for record in saveFile.created:
            if record.type != 'NPC_': continue
            #--Created NPC record
            if targetid and record.formid != targetid: continue
            npc = record.getTypeCopy()
            face = faces[npc.formid] = PCFaces.PCFace()
            face.masters = saveFile.masters
            for attr in ('eid','race','eyes','hair','hairLength','hairColor','fggs','fgga','fgts','level',
                'skills','health','baseSpell', 'fatigue','attributes','iclass'):
                setattr(face,attr,getattr(npc,attr))
            face.gender = (0,1)[npc.flags.female]
            face.pcName = npc.full
            #--Changed NPC Record
            PCFaces.save_getChangedNpc(saveFile,record.formid,face)
        return faces

    @staticmethod
    def save_getChangedNpc(saveFile,formid,face=None):
        """Update face with data from npc change record."""
        face = face or PCFaces.PCFace()
        changeRecord = saveFile.getRecord(formid)
        if not changeRecord:
            deprint('No change record')
            return face
        formid,recType,recFlags,version,data = changeRecord
        npc = SreNPC(recFlags,data)
        #deprint(SreNPC.flags(recFlags).getTrueAttrs())
        if npc.acbs:
            face.gender = npc.acbs.flags.female
            face.level = npc.acbs.level
            face.baseSpell = npc.acbs.baseSpell
            face.fatigue = npc.acbs.fatigue
            #deprint('level,baseSpell,fatigue:',face.level,face.baseSpell,face.fatigue)
        for attr in ('attributes','skills','health'):
            value = getattr(npc,attr)
            if value != None: 
                setattr(face,attr,value)
                #deprint(attr,value)
        #--Iref >> formid
        getFormid = saveFile.getFormid
        face.spells = [getFormid(x) for x in (npc.spells or [])]
        face.factions = [(getFormid(x),y) for x,y in (npc.factions or [])]
        face.modifiers = (npc.modifiers or [])[:]
        #delist('npc.spells:',[strFormid(x) for x in face.spells])
        #delist('npc.factions:',face.factions)
        #delist('npc.modifiers:',face.modifiers)
        return face

    @staticmethod
    def save_getPlayerFace(saveFile):
        """Extract player face from save file."""
        if isinstance(saveFile,SaveInfo):
            saveInfo = saveFile
            saveFile = SaveFile(saveInfo)
            saveFile.load()
        face = PCFaces.PCFace()
        face.pcName = saveFile.pcName
        face.masters = saveFile.masters
        #--Player ACHR
        record = saveFile.getRecord(0x14)
        data = record[-1]
        namePos = PCFaces.save_getNamePos(saveFile.fileInfo.name,data,saveFile.pcName)
        (face.fggs, face.fgga, face.fgts, face.race, face.hair, face.eyes,
            face.hairLength, face.hairColor, face.gender) = struct.unpack(
            '=200s120s200s3IfIB',data[namePos-542:namePos-1])
        classPos = namePos+len(saveFile.pcName)+1
        face.iclass, = struct.unpack('I',data[classPos:classPos+4])
        #--Iref >> formid
        getFormid = saveFile.getFormid
        face.race = getFormid(face.race)
        face.hair = getFormid(face.hair)
        face.eyes = getFormid(face.eyes)
        face.iclass = getFormid(face.iclass)
        #--Changed NPC Record
        PCFaces.save_getChangedNpc(saveFile,7,face)
        #--Done
        return face

    # Save Set ----------------------------------------------------------------
    @staticmethod
    def save_setFace(saveInfo,face,flags=0L):
        """DEPRECATED. Write a pcFace to a save file."""
        saveFile = SaveFile(saveInfo)
        saveFile.load()
        PCFaces.save_setPlayerFace(saveFile,face,flags)
        saveFile.safeSave()

    @staticmethod
    def save_setCreatedFace(saveFile,targetid,face):
        """Sets created face in savefile to specified face.
        Note: Created NPCs do NOT use irefs!"""
        targetid = intArg(targetid)
        #--Find record
        for index,record in enumerate(saveFile.created):
            if record.formid == targetid:
                npc = record.getTypeCopy()
                saveFile.created[index] = npc
                break
        else:
            raise StateError("Record %08X not found in %s." % (targetid,saveFile.fileInfo.name))
        if npc.type != 'NPC_':
            raise StateError("Record %08X in %s is not an NPC." % (targetid,saveFile.fileInfo.name))
        #--Update masters
        for formid in (face.race, face.eyes, face.hair):
            if not formid: continue
            master = face.masters[getModIndex(formid)]
            if master not in saveFile.masters:
                saveFile.masters.append(master)
        masterMap = MasterMap(face.masters,saveFile.masters)
        #--Set face
        npc.full = face.pcName
        npc.flags.female = (face.gender & 0x1)
        npc.setRace(masterMap(face.race,0x00907)) #--Default to Imperial
        npc.eyes = masterMap(face.eyes,None)
        npc.hair = masterMap(face.hair,None)
        npc.hairLength = face.hairLength
        npc.hairColor = face.hairColor
        npc.fggs = face.fggs
        npc.fgga = face.fgga
        npc.fgts = face.fgts
        npc.hairColor = face.hairColor
        #--Stats: Skip Level, baseSpell, fatigue and factions since they're discarded by game engine.
        if face.skills: npc.skills = face.skills
        if face.health: npc.health = face.health
        if face.attributes: npc.attributes = face.attributes
        if face.iclass: npc.iclass = face.iclass
        npc.setChanged()
        npc.getSize()

        #--Change record?
        changeRecord = saveFile.getRecord(npc.formid)
        if changeRecord == None: return
        formid,recType,recFlags,version,data = changeRecord
        npc = SreNPC(recFlags,data)
        #deprint(SreNPC.flags(recFlags).getTrueAttrs())
        if not npc.acbs: npc.acbs = npc.getDefault('acbs')
        npc.acbs.flags.female = face.gender
        npc.acbs.level = face.level
        npc.acbs.baseSpell = face.baseSpell
        npc.acbs.fatigue = face.fatigue
        npc.modifiers = face.modifiers[:]
        #--Formid conversion
        getIref = saveFile.getIref
        npc.spells = [getIref(x) for x in face.spells]
        npc.factions = [(getIref(x),y) for x,y in face.factions]

        #--Done
        saveFile.setRecord(npc.getTuple(formid,version))

    @staticmethod
    def save_setPlayerFace(saveFile,face,flags=0L,morphFacts=None):
        """Write a pcFace to a save file.
        flags:
            0x01 Import Name
            0x02 Import Race
            0x04 Import Gender
            0x08 Import Hair
            0x10 Import Eyes
            0x20 Import Class
            0x40 Import Stats
            0x80 Morph Factions"""
        flags = PCFaces.flags(flags)
        #deprint(flags.getTrueAttrs())
        #--Update masters
        for formid in (face.race, face.eyes, face.hair, face.iclass):
            if not formid: continue
            master = face.masters[getModIndex(formid)]
            if master not in saveFile.masters:
                saveFile.masters.append(master)
        masterMap = MasterMap(face.masters,saveFile.masters)
        
        #--Player ACHR
        #--Buffer for modified record data
        buff = cStringIO.StringIO()
        def buffPack(format,*args):
            buff.write(struct.pack(format,*args))
        def buffPackRef(oldFormId,doPack=True):
            newFormId = oldFormId and masterMap(oldFormId,None)
            if newFormId and doPack:
                newRef = saveFile.getIref(newFormId)
                buff.write(struct.pack('I',newRef))
            else:
                buff.seek(4,1)
        oldRecord = saveFile.getRecord(0x14)
        oldData = oldRecord[-1]
        namePos = PCFaces.save_getNamePos(saveFile.fileInfo.name,oldData,saveFile.pcName)
        buff.write(oldData)
        #--Modify buffer with face data.
        buff.seek(namePos-542)
        buffPack('=200s120s200s',face.fggs, face.fgga, face.fgts)
        #--Race?
        buffPackRef(face.race,flags.race)
        #--Hair, Eyes?
        buffPackRef(face.hair,flags.hair)
        buffPackRef(face.eyes,flags.eyes)
        if flags.hair:
            buffPack('=fI',face.hairLength,face.hairColor)
        else:
            buff.seek(8,1)
        #--Gender?
        if flags.gender:
            buffPack('B',face.gender)
        else:
            buff.seek(1,1)
        #--Name?
        if flags.name:
            postName = buff.getvalue()[buff.tell()+len(saveFile.pcName)+2:]
            buffPack('B',len(face.pcName)+1)
            buff.write(face.pcName+'\x00')
            buff.write(postName)
            buff.seek(-len(postName),1)
            saveFile.pcName = face.pcName
        else:
            buff.seek(len(saveFile.pcName)+2,1)
        #--Class?
        if flags.iclass and face.iclass:
            pos = buff.tell()
            newClass = masterMap(face.iclass)
            oldClass = saveFile.formids[struct.unpack('I',buff.read(4))[0]]
            customClass = saveFile.getIref(0x22843)
            if customClass not in (newClass,oldClass):
                buff.seek(pos)
                buffPackRef(newClass)
        
        newData = buff.getvalue()
        saveFile.setRecord(oldRecord[:-1]+(newData,))

        #--Player NPC
        (formid,recType,recFlags,version,data) = saveFile.getRecord(7)
        npc = SreNPC(recFlags,data)
        #--Gender
        if flags.gender and npc.acbs: 
            npc.acbs.flags.female = face.gender
        #--Stats
        if flags.stats and npc.acbs:
            npc.acbs.level = face.level
            npc.acbs.baseSpell = face.baseSpell
            npc.acbs.fatigue = face.fatigue
            npc.attributes = face.attributes
            npc.skills = face.skills
            npc.health = face.health
        #--Factions: Faction assignment doesn't work. (Probably stored in achr.)
        #--Modifiers, Spells, Name
        if flags.modifiers: npc.modifiers = face.modifiers[:]
        if flags.spells: 
            #delist('Set PC Spells:',face.spells)
            npc.spells = [saveFile.getIref(x) for x in face.spells]
        npc.full = None
        saveFile.setRecord(npc.getTuple(formid,version))
        #--Save
        buff.close()

    # Save Misc ----------------------------------------------------------------
    @staticmethod
    def save_repairHair(saveInfo):
        """Repairs hair if it has been zeroed. (Which happens if hair came from a
        cosmetic mod that has since been removed.) Returns True if repaired, False
        if no repair was necessary."""
        saveFile = SaveFile(saveInfo)
        saveFile.load()
        record = saveFile.getRecord(0x14)
        data = record[-1]
        namePos = PCFaces.save_getNamePos(saveInfo.name,data,saveFile.pcName)
        raceRef,hairRef = struct.unpack('2I',data[namePos-22:namePos-14])
        if hairRef != 0: return False
        raceForm = raceRef and saveFile.formids[raceRef]
        gender, = struct.unpack('B',data[namePos-2])
        if gender:
            hairForm = bush.raceHairFemale.get(raceForm,0x1da83)
        else:
            hairForm = bush.raceHairMale.get(raceForm,0x90475)
        hairRef = saveFile.getIref(hairForm)
        data = data[:namePos-18]+struct.pack('I',hairRef)+data[namePos-14:]
        saveFile.setRecord(record[:-1]+(data,))
        saveFile.safeSave()
        return True

    # MODS --------------------------------------------------------------------
    @staticmethod
    def mod_getFaces(modInfo):
        """Returns an array of PCFaces from a mod file."""
        #--Mod File
        loadFactory = LoadFactory(False,MreNpc)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        faces = {}
        for npc in modFile.NPC_.getActiveRecords():
            face = PCFaces.PCFace()
            face.masters = modFile.tes4.masters + [modInfo.name]
            for field in ('eid','race','eyes','hair','hairLength','hairColor',
                'fggs','fgga','fgts'):
                setattr(face,field,getattr(npc,field))
            face.gender = npc.flags.female
            face.pcName = npc.full
            face.level = npc.level
            face.skills = npc.skills
            face.health = npc.health
            face.baseSpell = npc.baseSpell
            face.fatigue = npc.fatigue
            face.attributes = npc.attributes
            faces[face.eid] = face
            #print face.pcName, face.race, face.hair, face.eyes, face.hairLength, face.hairColor
        return faces

    @staticmethod
    def mod_getRaceFaces(modInfo):
        """Returns an array of Race Faces from a mod file."""
        loadFactory = LoadFactory(False,MreRace)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        faces = {}
        for race in modFile.RACE.getActiveRecords():
            face = PCFaces.PCFace()
            face.masters = []
            for field in ('eid','fggs','fgga','fgts'):
                setattr(face,field,getattr(race,field))
            faces[face.eid] = face
        return faces

    @staticmethod
    def mod_addFace(modInfo,face):
        """Writes a pcFace to a mod file."""
        #--Mod File
        loadFactory = LoadFactory(True,MreNpc)
        modFile = ModFile(modInfo,loadFactory)
        if modInfo.getPath().exists():
            modFile.load(True)
        #--Tes4
        tes4 = modFile.tes4
        if not tes4.author: 
            tes4.author = '[wb]'
        if not tes4.description: 
            tes4.description = _('Face dump from save game.')
        if Path.get('Oblivion.esm') not in tes4.masters:
            tes4.masters.append(Path.get('Oblivion.esm'))
        masterMap = MasterMap(face.masters,tes4.masters+[modInfo.name])
        #--Eid
        npcEids = set([record.eid for record in modFile.NPC_.records])
        eidForm = ''.join(("sg", bush.raceShortNames.get(face.race,'Unk'), 
            (face.gender and 'a' or 'u'), re.sub(r'\W','',face.pcName),'%02d'))
        count,eid = 0, eidForm % (0,)
        while eid in npcEids:
            count += 1
            eid = eidForm % (count,)
        #--NPC
        npcid = genFormId(len(tes4.masters),tes4.getNextObject())
        npc = MreNpc(('NPC_',0,0x40000,npcid,0))
        npc.eid = eid
        npc.full = face.pcName
        npc.flags.female = (face.gender & 0x1)
        npc.classx = 0x237a8 #--Acrobat
        npc.setRace(masterMap(face.race,0x00907)) #--Default to Imperial
        npc.eyes = masterMap(face.eyes,None)
        npc.hair = masterMap(face.hair,None)
        npc.hairLength = face.hairLength
        npc.hairColor = face.hairColor
        npc.fggs = face.fggs
        npc.fgga = face.fgga
        npc.fgts = face.fgts
        npc.hairColor = face.hairColor
        #--Stats
        npc.level = face.level
        npc.baseSpell = face.baseSpell
        npc.fatigue = face.fatigue
        if face.skills: npc.skills = face.skills
        if face.health: npc.health = face.health
        if face.attributes: npc.attributes = face.attributes
        npc.setChanged()
        modFile.NPC_.records.append(npc)
        #--Save
        modFile.safeSave()
        return npc

#------------------------------------------------------------------------------
class SaveSpells:
    """Player spells of a savegame."""

    def __init__(self,saveInfo):
        """Initialize."""
        self.saveInfo = saveInfo
        self.saveFile = None
        self.allSpells = {} #--spells[(modName,objectIndex)] = (name,type)

    def load(self,progress=None):
        """Loads savegame and and extracts created spells from it and its masters."""
        saveFile = self.saveFile = SaveFile(self.saveInfo)
        saveFile.load()
        progress = (progress or Progress()).setFull(len(saveFile.masters)+1)
        #--Extract spells from masters
        for index,master in enumerate(saveFile.masters):
            progress(index,master)
            if master in modInfos:
                self.importMod(modInfos[master])
        #--Extract created spells
        allSpells = self.allSpells
        saveName = self.saveInfo.name
        progress(progress.full-1,saveName)
        for record in saveFile.created:
            if record.type == 'SPEL':
                allSpells[(saveName,getObjectIndex(record.formid))] = record.getTypeCopy()

    def importMod(self,modInfo):
        """Imports spell info from specified mod."""
        #--Spell list already extracted?
        if 'bash.spellList' in modInfo.extras:
            self.allSpells.update(modInfo.extras['bash.spellList'])
            return
        #--Else extract spell list
        loadFactory= LoadFactory(False,MreSpel)
        modFile = ModFile(modInfo,loadFactory)
        modFile.load(True)
        modFile.convertToLongFormids(('SPEL',))
        spells = modInfo.extras['bash.spellList'] = dict(
            [(record.formid,record) for record in modFile.SPEL.getActiveRecords()])
        self.allSpells.update(spells)

    def getPlayerSpells(self):
        """Returns players spell list from savegame. (Returns ONLY spells. I.e., not abilities, etc.)"""
        saveFile = self.saveFile
        #--Get masters and npc spell formids
        masters = saveFile.masters[:]
        maxMasters = len(masters) - 1
        (formid,recType,recFlags,version,data) = saveFile.getRecord(7)
        npc = SreNPC(recFlags,data)
        pcSpells = {} #--pcSpells[spellName] = iref
        #--NPC doesn't have any spells?
        if not npc.spells: 
            return pcSpells
        #--Get spell names to match formids
        for iref in npc.spells:
            if (iref >> 24) == 255:
                formid = iref
            else:
                formid = saveFile.formids[iref]
            modIndex,objectIndex = getFormIndices(formid)
            if modIndex == 255:
                master = self.saveInfo.name
            elif modIndex <= maxMasters:
                master = masters[modIndex]
            else: #--Bad formid?
                continue
            #--Get spell data
            record = self.allSpells.get((master,objectIndex),None)
            if record and record.full and record.spellType == 0 and formid != 0x136:
                pcSpells[record.full] = (iref,record)
        return pcSpells

    def removePlayerSpells(self,spellsToRemove):
        """Removes specified spells from players spell list."""
        (formid,recType,recFlags,version,data) = self.saveFile.getRecord(7)
        npc = SreNPC(recFlags,data)
        if npc.spells and spellsToRemove: 
            #--Remove spells and save
            npc.spells = [iref for iref in npc.spells if iref not in spellsToRemove]
            self.saveFile.setRecord(npc.getTuple(formid,version))
            self.saveFile.safeSave()
        
# Patchers 1 ------------------------------------------------------------------
#------------------------------------------------------------------------------
class PatchFile(ModFile):
    """Defines and executes patcher configuration."""
    #--Class
    mergeClasses = (
        MreActi, MreAlch, MreAmmo, MreAnio, MreAppa, MreArmo, MreBook, MreBsgn, 
        MreClot, MreCont, MreCrea, MreDoor, MreEfsh, MreEnch, MreEyes, MreFact, MreFlor, MreFurn, 
        MreGlob, MreGras, MreHair, MreIngr, MreKeym, MreLigh, MreLscr, MreLvlc, MreLvli, 
        MreLvsp, MreMgef, MreMisc, MreNpc,  MrePack, MreQust, MreRace, MreScpt, MreSgst,
        MreSlgm, MreSoun, MreSpel, MreStat, MreTree, MreWatr, MreWeap, MreWthr)

    @staticmethod
    def modIsMergeable(modInfo,progress=None):
        """Returns True or error message indicating whether specified mod is mergeable."""
        reBsa = re.compile(re.escape(modInfo.name.root())+'.*bsa$',re.I)
        for file in modInfos.dir.list():
            if reBsa.match(file):
                return _("Has BSA archive.")
        mergeTypes = set([recClass.type for recClass in PatchFile.mergeClasses])
        modFile = ModFile(modInfo,LoadFactory(False,*mergeTypes))
        modFile.load(True)
        #--Marked unmergeable?
        if 'NoMerge' in modInfo.getBashKeys():
            return _('Marked NoMerge.')
        #--Skipped over types?
        if modFile.topsSkipped: 
            return _("Unsupported types: ") + ', '.join(sorted(modFile.topsSkipped))
        #--Empty mod
        if not modFile.tops:
            return _("Empty mod.")
        #--New record
        lenMasters = len(modFile.tes4.masters)
        for type,block in modFile.tops.items():
            for record in block.getActiveRecords():
                if record.formid >> 24 >= lenMasters:
                    return _("New record %08X in block %s.") % (record.formid,type)
        #--Else
        return True

    #--Instance
    def __init__(self,modInfo,patchers):
        """Initialization."""
        ModFile.__init__(self,modInfo,None)
        self.tes4.author = 'BASHED PATCH'
        self.tes4.masters = [Path.get('Oblivion.esm')]
        self.longFormids = True
        #--New attrs
        self.aliases = {} #--Aliases from one mod name to another. Used by text file patchers.
        self.patchers = patchers
        self.keepIds = set()
        #--Mods
        patchTime = modInfo.mtime
        self.allMods = []
        self.mergeMods = []
        self.loadMods = [name for name in modInfos.ordered if modInfos[name].mtime < patchTime]
        self.loadErrorMods = []
        for patcher in self.patchers:
            patcher.beginPatch(self,self.loadMods)

    def getKeeper(self):
        """Returns a function to add formids to self.keepIds."""
        def keep(formid):
            self.keepIds.add(formid)
            return formid
        return keep

    def loadSourceData(self,progress):
        """Gives each patcher a chance to get its source data."""
        if not len(self.patchers): return
        progress = progress.setFull(len(self.patchers))
        for index,patcher in enumerate(self.patchers): 
            progress(index,_("Preparing %s...") % (patcher.getName(),))
            patcher.loadSourceData(SubProgress(progress,index))
        progress(progress.full,_('Patchers prepared.'))

    def postSourceData(self,progress):
        """Gets load factories."""
        progress(0,_("Processing."))
        def updateClasses(type_classes,newClasses):
            if not newClasses: return
            for item in newClasses:
                if not isinstance(item,str):
                    type_classes[item.type] = item
                elif item not in type_classes:
                    type_classes[item] = item
        readClasses = {}
        writeClasses = {}
        for patcher in self.patchers:
            updateClasses(readClasses, patcher.getReadClasses())
            updateClasses(writeClasses,patcher.getWriteClasses())
        self.readFactory = LoadFactory(False,*readClasses.values())
        self.loadFactory = LoadFactory(True,*writeClasses.values())
        #--Merge Factory
        self.mergeFactory = LoadFactory(False,*PatchFile.mergeClasses)

    def scanLoadMods(self,progress):
        """Scans load+merge mods."""
        if not len(self.loadMods): return
        loadSet,mergeSet = set(self.loadMods),set(self.mergeMods)
        self.allMods = modInfos.getOrdered(loadSet | mergeSet)
        nullProgress = Progress()
        progress = progress.setFull(len(self.allMods))
        for index,modName in enumerate(self.allMods):
            try:
                loadFactory = (self.readFactory,self.mergeFactory)[modName in mergeSet]
                progress(index,_("Scanning %s.") % (modName,))
                modInfo = modInfos[Path.get(modName)]
                modFile = ModFile(modInfo,loadFactory)
                modFile.load(True,SubProgress(progress,index,index+0.5))
            except ModError:
                self.loadErrorMods.append(modName)
                continue
            try:
                progress(index+0.5,_("Scanning %s...") % (modName,))
                if modName in mergeSet:
                    self.mergeModFile(modFile,loadSet,nullProgress)
                else:
                    self.scanModFile(modFile,nullProgress)
                for patcher in sorted(self.patchers,key=lambda a: a.scanOrder):
                    patcher.scanModFile(modFile,nullProgress)
                self.tes4.version = max(modFile.tes4.version, self.tes4.version)
            except:
                print _("MERGE/SCAN ERROR:"),modName
                raise
        progress(progress.full,_('Load mods scanned.'))

    def mergeModFile(self,modFile,loadSet,progress):
        """Copies contents of modFile into self."""
        doFilter = 'Filter' in modFile.fileInfo.getBashKeys()
        keep = self.getKeeper()
        modFile.convertToLongFormids()
        for blockType,block in modFile.tops.items(): 
            #--Make sure block type is also in read and write factories
            if blockType not in self.loadFactory.recTypes:
                recClass = self.mergeFactory.recFactory[blockType]
                self.readFactory.addClass(recClass)
                self.loadFactory.addClass(recClass)
            patchBlock = getattr(self,blockType)
            if not isinstance(patchBlock,TopBlock):
                raise BoshError(_("Merge unsupported for type: ")+blockType)
            filtered = []
            for record in block.getActiveRecords():
                if not doFilter or record.formid[0] in loadSet:
                    filtered.append(record)
                    record = record.getTypeCopy()
                    if doFilter: record.mergeFilter(loadSet)
                    patchBlock.setRecord(record.formid,record)
                    keep(record.formid)
            #--Filter records
            block.records = filtered

    def scanModFile(self,modFile,progress):
        """Scans file and overwrites own records with modfile records."""
        mapper = modFile.getLongMapper()
        for blockType,block in self.tops.items():
            if blockType not in modFile.tops: continue
            formids = set([record.formid for record in block.records])
            for record in modFile.tops[blockType].getActiveRecords():
                if mapper(record.formid) in formids:
                    record = record.getTypeCopy(mapper)
                    block.setRecord(record.formid,record)

    def buildPatch(self,log,progress):
        """Completes merge process. Use this when finished using scanLoadMods."""
        if not len(self.patchers): return
        log.setHeader('= '+self.fileInfo.name+' '+'='*30+'#',True)
        log("{{CONTENTS=1}}")
        #--Load Mods and error mods
        log.setHeader(_("= Overview"),True)
        log.setHeader(_("=== Date/Time"))
        log('* '+formatDate(time.time()))
        log.setHeader(_("=== Active Mods"),True)
        if self.loadErrorMods:
            log(_("**Note: Some mods had loading errors and were skipped.**"))
        for name in self.allMods:
            version = modInfos.getVersion(name)
            if name in self.loadMods:
                message = '* %02X ' % (self.loadMods.index(name),)
            else:
                message = '* ++ '
            if version: 
                message += _('%s  (Version %s)') % (name,version)
            else:
                message += name
            if name in self.loadErrorMods:
                message += _(" **Load Error -- Skipped**")
            log(message)
        #--Load Mods and error mods
        if self.aliases:
            log.setHeader(_("= Mod Aliases"))
            for key,value in sorted(self.aliases.items()):
                log('* %s >> %s' % (key,value))
        #--Patchers
        progress = SubProgress(progress,0,0.9,len(self.patchers))
        for index,patcher in enumerate(sorted(self.patchers,key=lambda a: a.editOrder)):
            progress(index,_("Completing %s...") % (patcher.getName(),))
            patcher.buildPatch(log,SubProgress(progress,index))
        #--Trim records
        progress(0.9,_("Trimming records..."))
        for block in self.tops.values():
            block.keepRecords(self.keepIds)
        progress(0.95,_("Converting formids..."))
        #--Convert masters to short formids
        self.tes4.masters = self.getMastersUsed()
        self.convertToShortFormids()
        progress(1.0,"Compiled.")

    def endPatch(self):
        """Cleanup patchers. (Mostly.)"""
        for patcher in self.patchers:
            patcher.endPatch()

#------------------------------------------------------------------------------
class Patcher:
    """Abstract base class for patcher elements."""
    scanOrder = 10
    editOrder = 10
    group = 'UNDEFINED'
    name = 'UNDEFINED'
    text = "UNDEFINED."
    tip = None
    defaultConfig = {'isEnabled':False}

    def getName(self):
        """Returns patcher name."""
        return self.__class__.name

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        """Initialization of common values to defaults."""
        self.patchFile = None
        self.scanOrder = self.__class__.scanOrder
        self.editOrder = self.__class__.editOrder
        self.isActive = True
        #--Gui stuff
        self.isEnabled = False #--Patcher is enabled.
        self.gConfigPanel = None

    def getConfig(self,configs):
        """Get config from configs dictionary and/or set to default."""
        config = configs.setdefault(self.__class__.__name__,{})
        for attr,default in self.__class__.defaultConfig.items():
            value = copy.deepcopy(config.get(attr,default))
            setattr(self,attr,value)

    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        config = configs[self.__class__.__name__] = {}
        for attr in self.__class__.defaultConfig:
            config[attr] = copy.deepcopy(getattr(self,attr))

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        self.patchFile = patchFile

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return None

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return None

    def loadSourceData(self,progress):
        """Compiles material, i.e. reads source text, esp's, etc. as necessary."""
        pass

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it. If adds record, should first convert it to long formids."""
        pass

    def buildPatch(self,log,progress):
        """Edits patch file as desired. Should write to log."""
        pass

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = None

#------------------------------------------------------------------------------
class ListPatcher(Patcher):
    """Subclass for patchers that have GUI lists of objects."""
    #--Get/Save Config
    choiceMenu = None #--List of possible choices for each config item. Item 0 is default.
    defaultConfig = {'isEnabled':False,'autoIsChecked':True,'configItems':[],'configChecks':{},'configChoices':{}}
    defaultItemCheck = True #--GUI: Whether new items are checked by default or not.
    forceItemCheck = False #--Force configChecked to True for all items
    autoRe = re.compile('^UNDEFINED$') #--Compiled re used by getAutoItems
    autoKey = None
    forceAuto = True

    #--Config Phase -----------------------------------------------------------
    def getAutoItems(self):
        """Returns list of items to be used for automatic configuration."""
        autoItems = []
        autoRe = self.__class__.autoRe
        autoKey = self.__class__.autoKey
        if isinstance(autoKey,str):
            autoKey = set((autoKey,))
        autoKey = set(autoKey)
        self.choiceMenu = self.__class__.choiceMenu
        for modInfo in modInfos.data.values():
            if autoRe.match(modInfo.name) or (autoKey & modInfo.getBashKeys()): 
                autoItems.append(modInfo.name)
                if self.choiceMenu: self.getChoice(modInfo.name)
        reFile = re.compile('_('+('|'.join(autoKey))+r')\.csv$')
        for fileName in sorted(dirs['patches'].list(),key=string.lower):
            if reFile.search(fileName):
                autoItems.append(fileName)
        return autoItems

    def getConfig(self,configs):
        """Get config from configs dictionary and/or set to default."""
        Patcher.getConfig(self,configs)
        if self.forceAuto:
            self.autoIsChecked = True
        #--Verify file existence
        newConfigItems = []
        patchesDir = dirs['patches'].list()
        for srcPath in self.configItems:
            if ((reModExt.search(srcPath) and srcPath in modInfos) or 
                reCsvExt.search(srcPath) and srcPath in patchesDir):
                    newConfigItems.append(srcPath)
        self.configItems = newConfigItems
        if self.__class__.forceItemCheck:
            for item in self.configItems:
                self.configChecks[item] = True
        #--Make sure configChoices are set (if choiceMenu exists).
        if self.choiceMenu:
            for item in self.configItems:
                self.getChoice(item)
        #--AutoItems?
        if self.autoIsChecked:
            self.getAutoItems()

    def getChoice(self,item):
        """Get default config choice."""
        return self.configChoices.setdefault(item,self.choiceMenu[0])

    def getItemLabel(self,item):
        """Returns label for item to be used in list"""
        if self.choiceMenu:
            return '%s [%s]' % (item,self.getChoice(item))
        else:
            return item

    def sortConfig(self,items):
        """Return sorted items. Default assumes mods and sorts by load order."""
        return modInfos.getOrdered(items,False)

    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        #--Toss outdated configCheck data.
        listSet = set(self.configItems)
        self.configChecks = dict([(key,value) for key,value in self.configChecks.items() if key in listSet])
        self.configChoices = dict([(key,value) for key,value in self.configChoices.items() if key in listSet])
        Patcher.saveConfig(self,configs)

    #--Patch Phase ------------------------------------------------------------
    def getConfigChecked(self):
        """Returns checked config items in list order."""
        return [item for item in self.configItems if self.configChecks[item]]

#------------------------------------------------------------------------------
class MultiTweakItem:
    """A tweak item, optionally with configuration choices."""
    def __init__(self,label,tip,key,*choices):
        """Initialize."""
        self.label = label
        self.tip = tip
        self.key = key
        self.choiceLabels = []
        self.choiceValues = []
        for choice in choices:
            self.choiceLabels.append(choice[0])
            self.choiceValues.append(choice[1:])
        #--Config
        self.isEnabled = False
        self.chosen = 0
    
    #--Config Phase -----------------------------------------------------------
    def getConfig(self,configs):
        """Get config from configs dictionary and/or set to default."""
        self.isEnabled,self.chosen = False,0
        if self.key in configs: 
            self.isEnabled,value = configs[self.key]
            if value in self.choiceValues:
                self.chosen = self.choiceValues.index(value)

    def getListLabel(self):
        """Returns label to be used in list"""
        label = self.label
        if len(self.choiceLabels) > 1:
            label += ' [' + self.choiceLabels[self.chosen] + ']'
        return label

    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        if self.choiceValues: value = self.choiceValues[self.chosen]
        else: value = None
        configs[self.key] = self.isEnabled,value

#------------------------------------------------------------------------------
class MultiTweaker(Patcher):
    """Combines a number of sub-tweaks which can be individually enabled and 
    configured through a choice menu."""
    group = _('Tweakers')
    scanOrder = 20
    editOrder = 20

    #--Config Phase -----------------------------------------------------------
    def getConfig(self,configs):
        """Get config from configs dictionary and/or set to default."""
        config = configs.setdefault(self.__class__.__name__,{})
        self.isEnabled = config.get('isEnabled',False)
        self.tweaks = copy.deepcopy(self.__class__.tweaks)
        for tweak in self.tweaks:
            tweak.getConfig(config)

    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        config = configs[self.__class__.__name__] = {}
        config['isEnabled'] = self.isEnabled
        for tweak in self.tweaks:
            tweak.saveConfig(config)
        self.enabledTweaks = [tweak for tweak in self.tweaks if tweak.isEnabled]
        self.isActive = len(self.enabledTweaks) > 0

# Patchers: 10 ----------------------------------------------------------------
#------------------------------------------------------------------------------
class AliasesPatcher(Patcher):
    """Specify mod aliases for patch files."""
    scanOrder = 10
    editOrder = 10
    group = _('General')
    name = _("Alias Mod Names")
    text = _("Specify mod aliases for reading CSV source files.")
    tip = None
    defaultConfig = {'isEnabled':True,'aliases':{}}

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        if self.isEnabled:
            self.patchFile.aliases = self.aliases

#------------------------------------------------------------------------------
class PatchMerger(ListPatcher):
    """Merges specified patches into Bashed Patch."""
    scanOrder = 10
    editOrder = 10
    group = _('General')
    name = _('Merge Patches')
    text = _("Merge patch mods into Bashed Patch.")
    autoRe = re.compile(r"^UNDEFINED$",re.I)
    autoKey = 'Merge'
    defaultItemCheck = False #--GUI: Whether new items are checked by default or not.

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        patchFile.mergeMods = self.getConfigChecked()

# Patchers: 20 ----------------------------------------------------------------
#------------------------------------------------------------------------------
class ImportPatcher(ListPatcher):
    """Subclass for patchers in group Importer."""
    group = _('Importers')
    scanOrder = 20
    editOrder = 20

    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        ListPatcher.saveConfig(self,configs)
        if self.isEnabled:
            importedMods = [item for item,value in self.configChecks.items() if value and reModExt.search(item)]
            configs['ImportedMods'].update(importedMods)

#------------------------------------------------------------------------------
class GraphicsPatcher(ImportPatcher):
    """Merges changes to graphics (models and icons)."""
    name = _('Import Graphics')
    text = _("Import graphics (models, icons, etc.) from source mods.")
    tip = text
    autoRe = re.compile(r"^UNDEFINED$",re.I)
    autoKey = 'Graphics'

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.id_data = {} #--Names keyed by long formid.
        self.srcClasses = [] #--Record classes actually provided by src mods/files.
        self.sourceMods = self.getConfigChecked()
        self.isActive = len(self.sourceMods) != 0
        #--Type Fields
        recAttrs_class = self.recAttrs_class = {}
        for recClass in (MreBsgn,MreLscr):
            recAttrs_class[recClass] = ('icon',)
        for recClass in (MreActi, MreDoor, ):
            recAttrs_class[recClass] = ('model',)
        for recClass in (MreAlch, MreAmmo, MreAppa, MreBook, MreIngr, MreKeym, MreLigh, MreMisc, MreSgst, MreSlgm, MreWeap):
            recAttrs_class[recClass] = ('icon','model')
        for recClass in (MreArmo, MreClot):
            recAttrs_class[recClass] = ('maleBody','maleWorld','maleIcon','femaleBody','femaleWorld','femaleIcon','flags')
        for recClass in (MreCrea,):
            recAttrs_class[recClass] = ('model','bodyParts','nift','bloodSpray','bloodDecal')

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return None
        return self.srcClasses

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return None
        return self.srcClasses

    def loadSourceData(self,progress):
        """Get graphics from source files."""
        if not self.isActive: return
        id_data = self.id_data
        recAttrs_class = self.recAttrs_class
        loadFactory = LoadFactory(False,*recAttrs_class.keys())
        progress.setFull(len(self.sourceMods))
        for index,srcMod in enumerate(self.sourceMods):
            if srcMod not in modInfos: continue
            srcInfo = modInfos[srcMod]
            srcFile = ModFile(srcInfo,loadFactory)
            srcFile.load(True)
            mapper = srcFile.getLongMapper()
            for recClass,recAttrs in recAttrs_class.items():
                if recClass.type not in srcFile.tops: continue
                self.srcClasses.append(recClass)
                for record in srcFile.tops[recClass.type].getActiveRecords():
                    formid = mapper(record.formid)
                    id_data[formid] = dict((attr,record.__dict__[attr]) for attr in recAttrs)
            progress.plus()
        self.isActive = len(self.srcClasses) > 0

    def scanModFile(self, modFile, progress):
        """Scan mod file against source data."""
        if not self.isActive: return 
        id_data = self.id_data
        modName = modFile.fileInfo.name
        mapper = modFile.getLongMapper()
        for recClass in self.srcClasses:
            type = recClass.type
            if type not in modFile.tops: continue
            patchBlock = getattr(self.patchFile,type)
            for record in modFile.tops[type].getActiveRecords():
                formid = record.formid
                if not record.longFormids: formid = mapper(formid)
                if formid not in id_data: continue
                for attr,value in id_data[formid].items():
                    if record.__dict__[attr] != value:
                        patchBlock.setRecord(formid,record.getTypeCopy(mapper))
                        break

    def buildPatch(self,log,progress):
        """Merge last version of record with patched graphics data as needed."""
        if not self.isActive: return
        modFile = self.patchFile
        keep = self.patchFile.getKeeper()
        id_data = self.id_data
        type_count = {}
        for recClass in self.srcClasses:
            type = recClass.type
            if type not in modFile.tops: continue
            type_count[type] = 0
            for record in modFile.tops[type].records:
                formid = record.formid
                if formid not in id_data: continue
                for attr,value in id_data[formid].items():
                    if record.__dict__[attr] != value:
                        break
                else:
                    continue 
                for attr,value in id_data[formid].items():
                    record.__dict__[attr] = value
                keep(formid)
                type_count[type] += 1
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods"))
        for mod in self.sourceMods:
            log("* " +mod)
        log(_("\n=== Modified Records"))
        for type,count in sorted(type_count.items()):
            if count: log("* %s: %d" % (type,count))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.id_data = self.sourceMods = None
        self.isActive = False

#------------------------------------------------------------------------------
class NamesPatcher(ImportPatcher):
    """Merged leveled lists mod file."""
    name = _('Import Names')
    text = _("Import names from source mods/files.")
    defaultItemCheck = False #--GUI: Whether new items are checked by default or not.
    autoRe = re.compile(r"^Oblivion.esm$",re.I)
    autoKey = 'Names'

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.id_full = {} #--Names keyed by long formid.
        self.activeTypes = [] #--Types ('ALCH', etc.) of data actually provided by src mods/files.
        self.skipTypes = [] #--Unknown types that were skipped.
        self.srcFiles = self.getConfigChecked()
        self.isActive = bool(self.srcFiles)

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return None
        return self.activeTypes

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return None
        return [MreRecord.type_class[type] for type in self.activeTypes]

    def loadSourceData(self,progress):
        """Get names from source files."""
        if not self.isActive: return
        fullNames = FullNames(aliases=self.patchFile.aliases)
        progress.setFull(len(self.srcFiles))
        for srcFile in self.srcFiles:
            srcPath = Path.get(srcFile)
            patchesDir = dirs['patches'].list()
            if reModExt.search(srcFile):
                if srcPath not in modInfos: continue
                srcInfo = modInfos[Path.get(srcFile)]
                fullNames.readFromMod(srcInfo)
            else:
                if srcPath not in patchesDir: continue
                fullNames.readFromText(dirs['patches'].join(srcFile))
            progress.plus()
        #--Finish
        id_full = self.id_full
        knownTypes = set(MreRecord.type_class.keys())
        for type,id_name in fullNames.type_id_name.items():
            if type not in knownTypes: 
                self.skipTypes.append(type)
                continue
            self.activeTypes.append(type)
            for longid,(eid,name) in id_name.items():
                id_full[longid] = name
        self.isActive = bool(self.activeTypes)

    def scanModFile(self, modFile, progress):
        """Scan modFile."""
        if not self.isActive: return 
        id_full = self.id_full
        modName = modFile.fileInfo.name
        mapper = modFile.getLongMapper()
        for type in self.activeTypes:
            if type not in modFile.tops: continue
            patchBlock = getattr(self.patchFile,type)
            id_records = patchBlock.id_records
            for record in modFile.tops[type].getActiveRecords():
                formid = record.formid
                if not record.longFormids: formid = mapper(formid)
                if formid in id_records: continue
                if formid not in id_full: continue
                if record.getSubString('FULL') != id_full[formid]:
                    patchBlock.setRecord(formid,record.getTypeCopy(mapper))

    def buildPatch(self,log,progress):
        """Make changes to patchfile."""
        if not self.isActive: return
        modFile = self.patchFile
        keep = self.patchFile.getKeeper()
        id_full = self.id_full
        type_count = {}
        for type in self.activeTypes:
            if type not in modFile.tops: continue
            type_count[type] = 0
            for record in modFile.tops[type].records:
                formid = record.formid
                if formid in id_full and record.full != id_full[formid]:
                    record.full = id_full[formid]
                    keep(formid)
                    type_count[type] += 1
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods/Files"))
        for file in self.srcFiles:
            log("* " +file)
        log(_("\n=== Renamed Items"))
        for type,count in sorted(type_count.items()):
            if count: log("* %s: %d" % (type,count))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.id_full = self.srcFiles = None
        self.isActive = False

#------------------------------------------------------------------------------
class NpcFacePatcher(ImportPatcher):
    """NPC Faces patcher, for use with TNR or similar mods."""
    name = _('Import NPC Faces')
    text = _("Import NPC face/eyes/hair from source mods. For use with TNR and similar mods.")
    autoRe = re.compile(r"^TNR .*.esp$",re.I)
    autoKey = 'NpcFaces'

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.faceData = {}
        self.faceMods = self.getConfigChecked()
        self.isActive = len(self.faceMods) != 0

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (None,(MreNpc,))[self.isActive]

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (None,(MreNpc,))[self.isActive]

    def loadSourceData(self,progress):
        """Get faces from TNR files."""
        if not self.isActive: return
        faceData = self.faceData
        loadFactory = LoadFactory(False,MreNpc)
        progress.setFull(len(self.faceMods))
        for index,faceMod in enumerate(self.faceMods):
            if faceMod not in modInfos: continue
            faceInfo = modInfos[faceMod]
            faceFile = ModFile(faceInfo,loadFactory)
            faceFile.load(True)
            faceFile.convertToLongFormids(('NPC_',))
            for npc in faceFile.NPC_.getActiveRecords():
                if npc.formid[0] != faceMod:
                    faceData[npc.formid] = (npc.fggs,npc.fgga,npc.fgts,npc.eyes,npc.hair,npc.hairLength,npc.hairColor)
            progress.plus()

    def scanModFile(self, modFile, progress):
        """Add lists from modFile."""
        modName = modFile.fileInfo.name
        if not self.isActive or modName in self.faceMods or 'NPC_' not in modFile.tops: 
            return
        mapper = modFile.getLongMapper()
        faceData,patchNpcs = self.faceData,self.patchFile.NPC_
        modFile.convertToLongFormids(('NPC_',))
        for npc in modFile.NPC_.getActiveRecords():
            if npc.formid in faceData: 
                patchNpcs.setRecord(npc.formid,npc)

    def buildPatch(self,log,progress):
        """Adds merged lists to patchfile."""
        if not self.isActive: return
        keep = self.patchFile.getKeeper()
        faceData, count = self.faceData, 0
        for npc in self.patchFile.NPC_.records:
            if npc.formid in faceData:
                (npc.fggs, npc.fgga, npc.fgts, npc.eyes,npc.hair, 
                    npc.hairLength, npc.hairColor) = faceData[npc.formid]
                npc.setChanged()
                keep(npc.formid)
                count += 1
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods"))
        for mod in self.faceMods:
            log("* " +mod)
        log(_("\n=== Faces Patched: %d") % (count,))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.faceData = self.faceMods = None
        self.isActive = False

#------------------------------------------------------------------------------
class RacePatcher(ImportPatcher):
    """Merged leveled lists mod file."""
    name = _('Import Race Info')
    text = _("Import race eyes, hair, body, voice from source mods.")
    autoRe = re.compile(r"^UNDEFINED$",re.I)
    autoKey = ('Hair','Eyes-D','Eyes-R','Eyes-E','Eyes','Body-M','Body-F','Voice-M','Voice-F')
    forceAuto = True

    #--Config Phase -----------------------------------------------------------
    def getAutoItems(self):
        """Returns list of items to be used for automatic configuration."""
        autoItems = []
        autoRe = self.__class__.autoRe
        autoKey = set(self.__class__.autoKey)
        for modInfo in modInfos.data.values():
            if autoRe.match(modInfo.name) or (autoKey & set(modInfo.getBashKeys())): 
                autoItems.append(modInfo.name)
        return autoItems

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.raceData = {} #--Race eye meshes, hair,eyes
        self.srcMods = [mod for mod in self.getConfigChecked() if mod in modInfos.ordered]
        self.isActive = True #--Always enabled to support eye filtering
        self.bodyKeys = ('Height','Weight','TailModel','UpperBody','LowerBody','Hand','Foot','Tail')
        self.eyeKeys = set(('Eyes-D','Eyes-R','Eyes-E','Eyes'))
        #--Mesh tuple for each defined eye. Derived from race records.
        defaultMesh = (r'characters\imperial\eyerighthuman.nif', r'characters\imperial\eyelefthuman.nif')
        self.eye_mesh = {}

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (None,(MreRace,MreEyes))[self.isActive]

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (None,(MreRace,MreEyes))[self.isActive]

    def loadSourceData(self,progress):
        """Get data from source files."""
        if not self.isActive or not self.srcMods: return
        loadFactory = LoadFactory(False,MreRace)
        progress.setFull(len(self.srcMods))
        for index,srcMod in enumerate(self.srcMods):
            if srcMod not in modInfos: continue
            srcInfo = modInfos[srcMod]
            srcFile = ModFile(srcInfo,loadFactory)
            srcFile.load(True)
            bashKeys = srcInfo.getBashKeys()
            if 'RACE' not in srcFile.tops: continue
            srcFile.convertToLongFormids(('RACE',))
            for race in srcFile.RACE.getActiveRecords():
                raceData = self.raceData.setdefault(race.formid,{})
                if 'Hair' in bashKeys:
                    raceHair = raceData.setdefault('hair',[])
                    for hair in race.hair:
                        if hair not in raceHair: raceHair.append(hair)
                if self.eyeKeys & bashKeys:
                    raceData['rightEye'] = race.rightEye
                    raceData['leftEye'] = race.leftEye
                    raceEyes = raceData.setdefault('eyes',[])
                    for eyes in race.eyes:
                        if eyes not in raceEyes: raceEyes.append(eyes)
                if 'Voice-M' in bashKeys:
                    raceData['maleVoice'] = race.maleVoice
                if 'Voice-F' in bashKeys:
                    raceData['femaleVoice'] = race.femaleVoice
                if 'Body-M' in bashKeys:
                    for key in ['male'+key for key in self.bodyKeys]:
                        raceData[key] = getattr(race,key)
                if 'Body-F' in bashKeys:
                    for key in ['female'+key for key in self.bodyKeys]:
                        raceData[key] = getattr(race,key)
            progress.plus()

    def scanModFile(self, modFile, progress):
        """Add appropriate records from modFile."""
        if not self.isActive: return 
        eye_mesh = self.eye_mesh
        modName = modFile.fileInfo.name
        mapper = modFile.getLongMapper()
        if 'RACE' not in modFile.tops: return
        modFile.convertToLongFormids(('RACE','EYES'))
        srcEyes = set([record.formid for record in modFile.EYES.getActiveRecords()])
        #--Race block
        patchBlock = self.patchFile.RACE
        id_records = patchBlock.id_records
        for record in modFile.RACE.getActiveRecords():
            if record.formid not in id_records: 
                patchBlock.setRecord(record.formid,record.getTypeCopy(mapper))
            for eyes in record.eyes:
                if eyes in srcEyes: 
                    eye_mesh[eyes] = (record.rightEye.path.lower(),record.leftEye.path.lower())

    def buildPatch(self,log,progress):
        """Updates races as needed."""
        debug = False
        if not self.isActive: return
        patchFile = self.patchFile
        keep = patchFile.getKeeper()
        if 'RACE' not in patchFile.tops: return
        #--Import race info
        racesPatched = []
        for race in patchFile.RACE.records:
            #~~print 'Building',race.eid
            raceData = self.raceData.get(race.formid,None)
            if not raceData: continue
            raceChanged = False
            #--Hair, Eyes
            if 'hair' in raceData and (set(race.hair) != set(raceData['hair'])):
                race.hair = raceData['hair']
                raceChanged = True
            if 'eyes' in raceData and (
                race.rightEye.path != raceData['rightEye'].path or
                race.leftEye.path  != raceData['leftEye'].path or
                set(race.eyes) != set(raceData['eyes'])
                ): 
                for attr in ('rightEye','leftEye','eyes'):
                    setattr(race,attr,raceData[attr])
                raceChanged = True
            #--Gender info (voice, body data)
            for gender in ('male','female'):
                voiceKey = gender+'Voice'
                if voiceKey in raceData:
                    if getattr(race,voiceKey) != raceData[voiceKey]:
                        setattr(race,voiceKey,raceData[voiceKey])
                        raceChanged = True
                bodyKeys = [gender+key for key in self.bodyKeys]
                if gender+'Foot' in raceData:
                    for key in bodyKeys:
                        if getattr(race,key) != raceData[key]:
                            setattr(race,key,raceData[key])
                            raceChanged = True
            if raceChanged:
                racesPatched.append(race.eid)
                keep(race.formid)
        #--Eye Mesh filtering
        racesFiltered = []
        eye_mesh = self.eye_mesh
        blueEyeMesh = eye_mesh[(Path.get('Oblivion.esm'),0x27308)]
        argonianEyeMesh = eye_mesh[(Path.get('Oblivion.esm'),0x3e91e)]
        if debug:
            print '== Eye Mesh Filtering'
            print 'blueEyeMesh',blueEyeMesh
            print 'argonianEyeMesh',argonianEyeMesh
        for eye in (
            (Path.get('Oblivion.esm'),0x1a), #--Reanimate
            (Path.get('Oblivion.esm'),0x54bb9), #--Dark Seducer
            (Path.get('Oblivion.esm'),0x54bba), #--Golden Saint
            (Path.get('Oblivion.esm'),0x5fa43), #--Ordered
            ):
            eye_mesh.setdefault(eye,blueEyeMesh)
        def setRaceEyeMesh(race,rightPath,leftPath):
            race.rightEye.path = rightPath
            race.leftEye.path = leftPath
        for race in patchFile.RACE.records:
            if debug: print '===', race.eid
            if not race.eyes: continue #--Sheogorath. Assume is handled correctly.
            raceChanged = False
            mesh_eye = {}
            for eyes in race.eyes:
                if eyes not in eye_mesh:
                    raise _('Mesh undefined for eye %s in race %s') % (strFormid(eyes),race.eid,)
                mesh = eye_mesh[eyes]
                if mesh not in mesh_eye:
                    mesh_eye[mesh] = []
                mesh_eye[mesh].append(eyes)
            currentMesh = (race.rightEye.path.lower(),race.leftEye.path.lower())
            #print race.eid, mesh_eye
            maxEyesMesh = sorted(mesh_eye.keys(),key=lambda a: len(mesh_eye[a]))[0]
            #--Single eye mesh, but doesn't match current mesh?
            if len(mesh_eye) == 1 and currentMesh != maxEyesMesh:
                setRaceEyeMesh(race,*maxEyesMesh)
                raceChanged = True
            #--Multiple eye meshes (and playable)?
            if debug:
                for mesh,eyes in mesh_eye.items():
                    print mesh
                    for eye in eyes: print ' ',strFormid(eye)
            if len(mesh_eye) > 1 and race.flags.playable:
                #--If blueEyeMesh (mesh used for vanilla eyes) is present, use that.
                if blueEyeMesh in mesh_eye and currentMesh != argonianEyeMesh:
                    setRaceEyeMesh(race,*blueEyeMesh)
                    race.eyes = mesh_eye[blueEyeMesh]
                    raceChanged = True
                elif argonianEyeMesh in mesh_eye:
                    setRaceEyeMesh(race,*argonianEyeMesh)
                    race.eyes = mesh_eye[argonianEyeMesh]
                    raceChanged = True
                #--Else figure that current eye mesh is the correct one
                elif currentMesh in mesh_eye:
                    race.eyes = mesh_eye[currentMesh]
                    raceChanged = True
                #--Else use most popular eye mesh
                else:
                    setRaceEyeMesh(race,*maxEyesMeshes)
                    race.eyes = mesh_eye[maxEyesMesh]
                    raceChanged = True
            if raceChanged:
                racesFiltered.append(race.eid)
                keep(race.formid)
        #--Done
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods"))
        for mod in self.srcMods:
            log("* " +mod)
        log(_("\n=== Race Info Imported"))
        if not racesPatched:
            log(_(". ~~None~~"))
        for eid in sorted(racesPatched):
            log("* "+eid)
        log(_("\n=== Eye Meshes Filtered"))
        if not racesFiltered:
            log(_(". ~~None~~"))
        for eid in sorted(racesFiltered):
            log("* "+eid)

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.raceData = self.srcMods = None
        self.isActive = False

#------------------------------------------------------------------------------
class StatsPatcher(ImportPatcher):
    """Merged leveled lists mod file."""
    name = _('Import Stats')
    text = _("Import ammo, armor and weapon stats from source mods/files.")
    defaultItemCheck = False #--GUI: Whether new items are checked by default or not.
    autoRe = re.compile(r"^UNDEFINED$",re.I)
    autoKey = 'Stats'

    def __init__(self):
        """Initialization of common values to defaults."""
        Patcher.__init__(self)
        self.scanOrder = 28
        self.editOrder = 28 #--Run ahead of bow patcher

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.srcFiles = self.getConfigChecked()
        self.isActive = bool(self.srcFiles)
        #--To be filled by loadSourceData
        self.id_stat = {} #--Stats keyed by long formid.
        self.activeTypes = [] #--Types ('ARMO', etc.) of data actually provided by src mods/files.
        self.typeFields = {} 

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return None
        return [MreRecord.type_class[type] for type in self.activeTypes]

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return None
        return [MreRecord.type_class[type] for type in self.activeTypes]

    def loadSourceData(self,progress):
        """Get stats from source files."""
        if not self.isActive: return
        itemStats = ItemStats(aliases=self.patchFile.aliases)
        progress.setFull(len(self.srcFiles))
        for srcFile in self.srcFiles:
            srcPath = Path.get(srcFile)
            patchesDir = dirs['patches'].list()
            if reModExt.search(srcFile):
                if srcPath not in modInfos: continue
                srcInfo = modInfos[Path.get(srcFile)]
                itemStats.readFromMod(srcInfo)
            else:
                if srcPath not in patchesDir: continue
                itemStats.readFromText(dirs['patches'].join(srcFile))
            progress.plus()
        #--Finish
        id_stat = self.id_stat
        for type in itemStats.type_stats:
            typeStats = itemStats.type_stats[type]
            if typeStats:
                self.activeTypes.append(type)
                id_stat.update(typeStats)
                self.typeFields[type] = itemStats.type_attrs[type][1:]
        self.isActive = bool(self.activeTypes)

    def scanModFile(self, modFile, progress):
        """Add affected items to patchFile."""
        if not self.isActive: return 
        id_stat = self.id_stat
        mapper = modFile.getLongMapper()
        for type in self.activeTypes:
            if type not in modFile.tops: continue
            typeFields = self.typeFields[type]
            patchBlock = getattr(self.patchFile,type)
            id_records = patchBlock.id_records
            for record in modFile.tops[type].getActiveRecords():
                formid = record.formid
                if not record.longFormids: formid = mapper(formid)
                if formid in id_records: continue
                stats = id_stat.get(formid)
                if not stats: continue
                modStats = tuple(record.__dict__[attr] for attr in typeFields)
                if modStats != stats[1:]:
                    patchBlock.setRecord(formid,record.getTypeCopy(mapper))

    def buildPatch(self,log,progress):
        """Adds merged lists to patchfile."""
        if not self.isActive: return
        patchFile = self.patchFile
        keep = self.patchFile.getKeeper()
        id_stat = self.id_stat
        allCounts = []
        for type in self.activeTypes:
            if type not in patchFile.tops: continue
            typeFields = self.typeFields[type]
            count,counts = 0,{}
            for record in patchFile.tops[type].records:
                formid = record.formid
                stats = id_stat.get(formid)
                if not stats: continue
                modStats = tuple(record.__dict__[attr] for attr in typeFields)
                if modStats == stats[1:]: continue
                for attr,value in zip(typeFields,stats[1:]):
                    record.__dict__[attr] = value
                keep(formid)
                count += 1
                counts[formid[0]] = 1 + counts.get(formid[0],0)
            allCounts.append((type,count,counts))
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods/Files"))
        for file in self.srcFiles:
            log("* " +file)
        log(_("\n=== Modified Stats"))
        for type,count,counts in allCounts:
            if not count: continue
            typeName = {'AMMO':_('Ammo'),'ARMO':_('Armor'),'WEAP':_('Weapons')}[type]
            log("* %d %s" % (count,typeName))
            for modName in sorted(counts,key=string.lower):
                log("  * %d %s" % (counts[modName],modName))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.id_stat = self.srcFiles = None
        self.isActive = False

# Patchers: 30 ----------------------------------------------------------------
#------------------------------------------------------------------------------
class AssortedTweak_BowReach(MultiTweakItem):
    """Fix bows to have reach = 1.0."""

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Bow Reach Fix"),
            _('Fix bows with zero reach. (Zero reach causes CTDs.)'),
            'BowReach',
            ('1.0',  '1.0'),
            )

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreWeap,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreWeap,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        patchRecords = patchFile.WEAP
        for record in modFile.WEAP.getActiveRecords():
            if record.weaponType == 5 and record.reach <= 0:
                record = record.getTypeCopy(mapper)
                patchRecords.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        keep = patchFile.getKeeper()
        for record in patchFile.WEAP.records:
            if record.weaponType == 5 and record.reach <= 0:
                record.reach = 1
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('=== Bow Reach Fix'))
        log(_('* %d Bows fixed') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class AssortedTweak_ConsistentRings(MultiTweakItem):
    """Sets rings to all work on same finger."""

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Right Hand Rings"),
            _('Fixes rings to unequip consistently by making them prefer the right hand.'),
            'ConsistentRings',
            ('1.0',  '1.0'),
            )

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreClot,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreClot,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        patchRecords = patchFile.CLOT
        for record in modFile.CLOT.getActiveRecords():
            if record.flags.leftRing:
                record = record.getTypeCopy(mapper)
                patchRecords.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        keep = patchFile.getKeeper()
        for record in patchFile.CLOT.records:
            if record.flags.leftRing:
                record.flags.leftRing = False
                record.flags.rightRing = True
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('=== Right Hand Rings'))
        log(_('* %d Rings fixed') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class AssortedTweak_NoLightFlicker(MultiTweakItem):
    """Remove light flickering for low end machines."""

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("No Light Flicker"),
            _('Remove flickering from lights. For use on low-end machines.'),
            'NoLightFlicker',
            ('1.0',  '1.0'),
            )
        self.flags = flags = MreLigh.flags()
        flags.flickers = flags.flickerSlow = flags.pulse = flags.pulseSlow = True

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreLigh,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreLigh,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        flickerFlags = self.flags
        mapper = modFile.getLongMapper()
        patchRecords = patchFile.LIGH
        for record in modFile.LIGH.getActiveRecords():
            if record.flags & flickerFlags:
                record = record.getTypeCopy(mapper)
                patchRecords.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        flickerFlags = self.flags
        notFlickerFlags = ~flickerFlags
        keep = patchFile.getKeeper()
        for record in patchFile.LIGH.records:
            if int(record.flags & flickerFlags):
                record.flags &= notFlickerFlags
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('=== No Light Flicker'))
        log(_('* %d Lights unflickered') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class AssortedTweak_PotionWeight(MultiTweakItem):
    """Reweighs standard potions down to 0.2."""

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Max Weight Potions"),
            _('Potion weight will be capped.'),
            'PotionWeight',
            (_('0.1'),  0.1),
            (_('0.2'),  0.2),
            (_('0.4'),  0.4),
            (_('0.6'),  0.6),
            )

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreAlch,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreAlch,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        maxWeight = self.choiceValues[self.chosen][0]
        mapper = modFile.getLongMapper()
        patchBlock = patchFile.ALCH
        id_records = patchBlock.id_records
        for record in modFile.ALCH.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            if record.weight > maxWeight and record.weight < 1:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        maxWeight = self.choiceValues[self.chosen][0]
        count = {}
        keep = patchFile.getKeeper()
        for record in patchFile.ALCH.records:
            if record.weight > maxWeight and record.weight < 1 and not ('SEFF',0) in record.getEffects():
                record.weight = maxWeight
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('=== Reweigh Potions'))
        log(_('* %d Potions Reweighed') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class AssortedTweak_StaffWeight(MultiTweakItem):
    """Reweighs staffs."""

    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Max Weight Staffs"),
            _('Staff weight will be capped.'),
            'StaffWeight',
            (_('1'),  1),
            (_('2'),  2),
            (_('3'),  3),
            (_('4'),  4),
            (_('5'),  5),
            (_('6'),  6),
            (_('7'),  7),
            (_('8'),  8),
            )

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreWeap,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreWeap,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        maxWeight = self.choiceValues[self.chosen][0]
        mapper = modFile.getLongMapper()
        patchBlock = patchFile.WEAP
        id_records = patchBlock.id_records
        for record in modFile.WEAP.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            if record.weaponType == 4 and record.weight > maxWeight:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        maxWeight = self.choiceValues[self.chosen][0]
        count = {}
        keep = patchFile.getKeeper()
        for record in patchFile.WEAP.records:
            if record.weaponType == 4 and record.weight > maxWeight:
                record.weight = maxWeight
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('=== Reweigh Staffs'))
        log(_('* %d Staffs Reweighed') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class AssortedTweaker(MultiTweaker):
    """Tweaks assorted stuff. Sub-tweaks behave like patchers themselves."""
    name = _('Tweak Assorted')
    text = _("Tweak various records in miscellaneous ways.")
    tweaks = sorted([
        AssortedTweak_BowReach(),
        AssortedTweak_ConsistentRings(),
        AssortedTweak_NoLightFlicker(),
        AssortedTweak_PotionWeight(),
        AssortedTweak_StaffWeight(),
        ],key=lambda a: a.label.lower())

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return None
        classTuples = [tweak.getReadClasses() for tweak in self.enabledTweaks]
        return sum(classTuples,tuple())

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return None
        classTuples = [tweak.getWriteClasses() for tweak in self.enabledTweaks]
        return sum(classTuples,tuple())

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive: return
        for tweak in self.enabledTweaks:
            tweak.scanModFile(modFile,progress,self.patchFile)

    def buildPatch(self,log,progress):
        """Applies individual clothes tweaks."""
        if not self.isActive: return
        log.setHeader('= '+self.__class__.name,True)
        for tweak in self.enabledTweaks:
            tweak.buildPatch(log,progress,self.patchFile)

#------------------------------------------------------------------------------
class ClothesTweak(MultiTweakItem):
    flags = {
        'amulets':0x0100,
        'hoods':  0x0002,
        'pants':  0x0008,
        'rings':  0x00C0,
        'robes':  0x000C,
        'amulets2': (1<<17),
        }

    #--Config Phase -----------------------------------------------------------
    def __init__(self,label,tip,key,*choices):
        MultiTweakItem.__init__(self,label,tip,key,*choices)
        typeKey = key[:key.find('.')]
        self.orTypeFlags = typeKey == 'rings'
        self.typeFlags = self.__class__.flags[typeKey]

    def isMyType(self,record):
        """Returns true to save record for late processing."""
        recTypeFlags = int(record.flags) & 0xFFFF
        myTypeFlags = self.typeFlags
        return (recTypeFlags == myTypeFlags or (self.orTypeFlags and 
            (recTypeFlags & myTypeFlags == recTypeFlags)))

#------------------------------------------------------------------------------
class ClothesTweak_MaxWeight(ClothesTweak):
    """Enforce a max weight for specified clothes."""
    #--Patch Phase ------------------------------------------------------------
    def buildPatch(self,patchFile,keep,log):
        """Build patch."""
        tweakCount = 0
        maxWeight = self.choiceValues[self.chosen][0]
        for record in patchFile.CLOT.records:
            if self.isMyType(record) and record.weight > maxWeight and record.weight < max(10,5*maxWeight):
                record.weight = maxWeight
                keep(record.formid)
                tweakCount += 1
        log('* %s: [%0.1f]: %d' % (self.label,maxWeight,tweakCount))

#------------------------------------------------------------------------------
class ClothesTweak_Unblock(ClothesTweak):
    """Unlimited rings, amulets."""
    #--Config Phase -----------------------------------------------------------
    def __init__(self,label,tip,key,*choices):
        ClothesTweak.__init__(self,label,tip,key,*choices)
        self.unblockFlags = self.__class__.flags[key[key.rfind('.')+1:]]

    #--Patch Phase ------------------------------------------------------------
    def buildPatch(self,patchFile,keep,log):
        """Build patch."""
        tweakCount = 0
        for record in patchFile.CLOT.records:
            if self.isMyType(record) and record.flags & self.unblockFlags:
                record.flags &= ~self.unblockFlags
                keep(record.formid)
                tweakCount += 1
        log('* %s: %d' % (self.label,tweakCount))

#------------------------------------------------------------------------------
class ClothesTweaker(MultiTweaker):
    """Patches clothes in miscellaneous ways."""
    name = _('Tweak Clothes')
    text = _("Tweak clothing weight and blocking.")
    tweaks = sorted([
        ClothesTweak_Unblock(_("Unlimited Amulets"),
            _("Wear unlimited number of amulets - but they won't display."),
            'amulets.unblock.amulets'),
        ClothesTweak_Unblock(_("Unlimited Rings"),
            _("Wear unlimited number of rings - but they won't display."),
            'rings.unblock.rings'),
        ClothesTweak_Unblock(_("Robes Show Pants"),
            _("Robes will allow pants, greaves, skirts - but they'll clip."),
            'robes.unblock.pants'),
        ClothesTweak_Unblock(_("Robes Show Amulets"),
            _("Robes will always show amulets. (Conflicts with Unlimited Amulets.)"),
            'robes.show.amulets2'),
        ClothesTweak_MaxWeight(_("Max Weight Amulets"),
            _("Amulet weight will be capped."),
            'amulets.maxWeight',
            (_('0.0'),0),
            (_('0.1'),0.1),
            (_('0.2'),0.2),
            (_('0.5'),0.5),
            ),
        ClothesTweak_MaxWeight(_("Max Weight Rings"),
            _('Ring weight will be capped.'),
            'rings.maxWeight',
            (_('0.0'),0),
            (_('0.1'),0.1),
            (_('0.2'),0.2),
            (_('0.5'),0.5),
            ),
        ClothesTweak_MaxWeight(_("Max Weight Hoods"),
            _('Hood weight will be capped.'),
            'hoods.maxWeight',
            (_('0.2'),0.2),
            (_('0.5'),0.5),
            (_('1.0'),1.0),
            ),
        ],key=lambda a: a.label.lower())

    def __init__(self):
        Patcher.__init__(self)
        self.scanOrder = 31
        self.editOrder = 31

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (None,(MreClot,))[self.isActive]

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (None,(MreClot,))[self.isActive]

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive or 'CLOT' not in modFile.tops: return
        mapper = modFile.getLongMapper()
        patchRecords = self.patchFile.CLOT
        id_records = patchRecords.id_records
        for record in modFile.CLOT.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            for tweak in self.enabledTweaks:
                if tweak.isMyType(record):
                    record = record.getTypeCopy(mapper)
                    patchRecords.setRecord(record.formid,record)
                    break

    def buildPatch(self,log,progress):
        """Applies individual clothes tweaks."""
        if not self.isActive: return
        keep = self.patchFile.getKeeper()
        log.setHeader('= '+self.__class__.name)
        for tweak in self.enabledTweaks:
            tweak.buildPatch(self.patchFile,keep,log)

#------------------------------------------------------------------------------
class GmstTweak(MultiTweakItem):
    #--Patch Phase ------------------------------------------------------------
    def buildPatch(self,patchFile,keep,log):
        """Build patch."""
        eids = ((self.key,),self.key)[isinstance(self.key,tuple)]
        for eid,value in zip(eids,self.choiceValues[self.chosen]):
            gmst = MreGmst(('GMST',0,0,0,0))
            gmst.eid,gmst.value,gmst.longFormids = eid,value,True
            formid = gmst.formid = gmst.getOblivionFormid()
            patchFile.GMST.setRecord(keep(formid),gmst)
        if len(self.choiceLabels) > 1:
            log('* %s: %s' % (self.label,self.choiceLabels[self.chosen]))
        else:
            log('* ' + self.label)

#------------------------------------------------------------------------------
class GmstTweaker(MultiTweaker):
    """Tweaks miscellaneous gmsts in miscellaneous ways."""
    name = _('Tweak Settings')
    text = _("Tweak game settings.")
    tweaks = sorted([
        GmstTweak(_('Arrow Litter Count'),
            _("Maximum number of spent arrows allowed in cell."),
            'iArrowMaxRefCount',
            ('25',25),
            ('50',50),
            ('100',100),
            ('500',500),
            ),
        GmstTweak(_('Arrow Litter Time'),
            _("Time before spent arrows fade away from cells and actors."),
            'fArrowAgeMax',
            (_('1 Minute'),60),
            (_('2 Minutes'),120),
            (_('3 Minutes'),180),
            (_('5 Minutes'),300),
            (_('10 Minutes'),600),
            (_('30 Minutes'),1800),
            (_('1 Hour'),3600),
            ),
        GmstTweak(_('Arrow Recovery from Actor'),
            _("Chance that an arrow shot into an actor can be recovered."),
            'iArrowInventoryChance',
            ('60%',60),
            ('70%',70),
            ('80%',80),
            ('90%',90),
            ('100%',100),
            ),
        GmstTweak(_('Arrow Speed'),
            _("Speed of full power arrow."),
            'fArrowSpeedMult',
            (_('x 1.2'),1500*1.2),
            (_('x 1.4'),1500*1.4),
            (_('x 1.6'),1500*1.6),
            (_('x 1.8'),1500*1.8),
            (_('x 2.0'),1500*2.0),
            (_('x 2.2'),1500*2.2),
            (_('x 2.4'),1500*2.4),
            (_('x 2.6'),1500*2.6),
            (_('x 2.8'),1500*2.8),
            (_('x 3.0'),1500*3.0),
            ),
        GmstTweak(_('Chase Camera Tightness'),
            _("Tightness of chase camera to player turning."),
            ('fChase3rdPersonVanityXYMult','fChase3rdPersonXYMult'),
            (_('x 1.5'),6,6),
            (_('x 2.0'),8,8),
            (_('x 3.0'),12,12),
            (_('x 5.0'),20,20),
            ),
        GmstTweak(_('Chase Camera Distance'),
            _("Distance camera can be moved away from PC using mouse wheel."),
            ('fVanityModeWheelMax', 'fChase3rdPersonZUnitsPerSecond','fVanityModeWheelMult'),
            (_('x 1.5'),600*1.5, 300*1.5,0.15),
            (_('x 2'),  600*2,   300*2, 0.2),
            (_('x 3'),  600*3,   300*3, 0.3),
            (_('x 5'),  600*5,   1000,  0.3),
            (_('x 10'), 600*10,  2000,  0.3),
            ),
        GmstTweak(_('Chameleon: No Refraction'),
            _("Chameleon with transparency instead of refraction effect."),
            ('fChameleonMinRefraction','fChameleonMaxRefraction'),
            ('',0,0),
            ),
        GmstTweak(_('Compass: Disable'),
            _("No quest and/or points of interest markers on compass."),
            'iMapMarkerRevealDistance',
            (_('Quests'),1803),
            (_('POIs'),1802),
            (_('Quests and POIs'),1801),
            ),
        GmstTweak(_('Compass: POI Recognition'),
            _("Distance at which POI markers begin to show on compass."),
            'iMapMarkerVisibleDistance',
            (_('x 0.25'),3000),
            (_('x 0.50'),6000),
            (_('x 0.75'),9000),
            ),
        GmstTweak(_('Essential NPC Unconsciousness'),
            _("Time which essential NPCs stay unconscious."),
            'fEssentialDeathTime',
            (_('20 Seconds'),20),
            (_('30 Seconds'),30),
            (_('1 Minute'),60),
            (_('3 Minutes'),3*60),
            (_('5 Minutes'),5*60),
            ),
        GmstTweak(_('Fatigue from Running/Encumbrance'),
            _("Fatigue cost of running and encumbrance."),
            ('fFatigueRunBase','fFatigueRunMult'),
            ('x 2',16,8),
            ('x 3',24,12),
            ('x 4',32,16),
            ('x 5',40,20),
            ),
        GmstTweak(_('Horse Turning Speed'),
            _("Speed at which horses turn."),
            'iHorseTurnDegreesPerSecond',
            (_('x 1.5'),68),
            (_('x 2.0'),90),
            ),
        GmstTweak(_('Jump Higher'),
            _("Maximum height player can jump to."),
            'fJumpHeightMax',
            (_('x 1.1'),164*1.1),
            (_('x 1.2'),164*1.2),
            (_('x 1.4'),164*1.4),
            (_('x 1.6'),164*1.6),
            ),
        GmstTweak(_('PC Death Camera'),
            _("Time after player's death before reload menu appears."),
            'fPlayerDeathReloadTime',
            (_('15 Seconds'),15),
            (_('30 Seconds'),30),
            (_('1 Minute'),60),
            (_('5 Minute'),300),
            (_('Unlimited'),9999999),
            ),
        GmstTweak(_('Cell Respawn Time'),
            _("Time before unvisited cell respawns. But longer times increase save sizes."),
            'iHoursToRespawnCell',
            (_('1 Day'),24*1),
            (_('3 Days'),24*3),
            (_('5 Days'),24*5),
            (_('10 Days'),24*10),
            (_('20 Days'),24*20),
            (_('1 Month'),24*30),
            (_('6 Months'),24*182),
            (_('1 Year'),24*365),
            ),
        #--NEW
        GmstTweak(_('Combat: Recharge Weapons'),
            _("Allow recharging weapons during combat."),
            ('iAllowRechargeDuringCombat'),
            (_('Allow'),1),
            (_('Disallow'),0),
            ),
        GmstTweak(_('Magic Bolt Speed'),
            _("Speed of magic bolt/projectile."),
            'fMagicProjectileBaseSpeed',
            (_('x 1.2'),1000*1.2),
            (_('x 1.4'),1000*1.4),
            (_('x 1.6'),1000*1.6),
            (_('x 1.8'),1000*1.8),
            (_('x 2.0'),1000*2.0),
            (_('x 2.2'),1000*2.2),
            (_('x 2.4'),1000*2.4),
            (_('x 2.6'),1000*2.6),
            (_('x 2.8'),1000*2.8),
            (_('x 3.0'),1000*3.0),
            ),
        GmstTweak(_('Msg: Equip Misc. Item'),
            _("Message upon equipping misc. item."),
            ('sCantEquipGeneric'),
            (_('[None]'),' '),
            (_('.'),'.'),
            (_('Hmm...'),_('Hmm...')),
            ),
    ],key=lambda a: a.label.lower())

    #--Patch Phase ------------------------------------------------------------
    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (None,(MreGmst,))[self.isActive]

    def buildPatch(self,log,progress):
        """Edits patch file as desired. Will write to log."""
        if not self.isActive: return
        keep = self.patchFile.getKeeper()
        log.setHeader('= '+self.__class__.name)
        for tweak in self.enabledTweaks:
            tweak.buildPatch(self.patchFile,keep,log)

#------------------------------------------------------------------------------
class NamesTweak_Body(MultiTweakItem):
    """Names tweaker for armor and clothes."""

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreRecord.type_class[self.key],)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreRecord.type_class[self.key],)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        patchBlock = getattr(patchFile,self.key)
        id_records = patchBlock.id_records
        for record in getattr(modFile,self.key).getActiveRecords():
            if record.full and mapper(record.formid) not in id_records:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        format = self.choiceValues[self.chosen][0]
        showStat = '%02d' in format
        keep = patchFile.getKeeper()
        for record in getattr(patchFile,self.key).records:
            if not record.full: continue
            flags = record.flags
            if flags.head or flags.hair: type = 'H'
            elif flags.rightRing or flags.leftRing: type = 'R'
            elif flags.amulet: type = 'A'
            elif flags.upperBody: type = 'C'
            elif flags.lowerBody: type = 'P'
            elif flags.hand: type = 'G'
            elif flags.foot: type = 'B'
            elif flags.tail: type = 'T'
            elif flags.shield: type = 'S'
            else: continue
            if record.type == 'ARMO':
                type += 'LH'[record.flags.heavyArmor]
            if showStat:
                record.full = format % (type,record.strength/100) + record.full
            else:
                record.full = format % (type,) + record.full
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log(_('* %d %s') % (sum(count.values()),self.label))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class NamesTweak_Potions(MultiTweakItem):
    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Potions"),
            _('Label potions to sort by type and effect.'),
            'ALCH',
            (_('XD Illness'),  '%s '),
            (_('XD. Illness'), '%s. '),
            (_('XD - Illness'),'%s - '),
            (_('(XD) Illness'),'(%s) '),
            )

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreAlch,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreAlch,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        patchBlock = patchFile.ALCH
        id_records = patchBlock.id_records
        for record in modFile.ALCH.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            record = record.getTypeCopy(mapper)
            patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        format = self.choiceValues[self.chosen][0]
        poisonEffects = bush.poisonEffects
        keep = patchFile.getKeeper()
        reOldLabel = re.compile('^(-|X) ')
        reOldEnd = re.compile(' -$')
        effectId_school = dict((key,value[0]) for key,value in bush.magicEffects.items())
        for record in patchFile.ALCH.records:
            if not record.full: continue
            school = 6 #--Default to 6 (U: unknown)
            for index,effect in enumerate(record.effects):
                effectId = effect.id
                if index == 0:
                    if effect.scriptEffect:
                        school = effect.scriptEffect.school
                    else:
                        school = effectId_school.get(effectId,6)
                #--Non-hostile effect?
                if effect.scriptEffect:
                    if not effect.scriptEffect.flags.hostile:
                        isPoison = False
                        break
                elif effectId not in poisonEffects:
                    isPoison = False
                    break
            else:
                isPoison = True
            full = reOldLabel.sub('',record.full) #--Remove existing label
            full = reOldEnd.sub('',full)
            if record.flags.isFood:
                record.full = '.'+full
            else:
                label = ('','X')[isPoison] + 'ACDIMRU'[school]
                record.full = format % (label,) + full
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log(_('* %d %s') % (sum(count.values()),self.label))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class NamesTweak_Scrolls(MultiTweakItem):
    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Notes and Scrolls"),
            _('Mark notes and scrolls to sort separately from books'),
            'scrolls',
            (_('~Fire Ball'),  '~'),
            (_('~D Fire Ball'),  '~%s '),
            (_('~D. Fire Ball'), '~%s. '),
            (_('~D - Fire Ball'),'~%s - '),
            (_('~(D) Fire Ball'),'~(%s) '),
            ('----','----'),
            (_('.Fire Ball'),  '.'),
            (_('.D Fire Ball'),  '.%s '),
            (_('.D. Fire Ball'), '.%s. '),
            (_('.D - Fire Ball'),'.%s - '),
            (_('.(D) Fire Ball'),'.(%s) '),
            )

    #--Config Phase -----------------------------------------------------------
    def saveConfig(self,configs):
        """Save config to configs dictionary."""
        MultiTweakItem.saveConfig(self,configs)
        rawFormat = self.choiceValues[self.chosen][0]
        self.orderFormat = ('~.','.~')[rawFormat[0] == '~']
        self.magicFormat = rawFormat[1:]

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreBook,MreEnch)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreBook,MreEnch)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        #--Scroll Enchantments
        if self.magicFormat:
            patchBlock = patchFile.ENCH
            id_records = patchBlock.id_records
            for record in modFile.ENCH.getActiveRecords():
                if mapper(record.formid) in id_records: continue
                if record.itemType == 0:
                    record = record.getTypeCopy(mapper)
                    patchBlock.setRecord(record.formid,record)
        #--Books
        patchBlock = patchFile.BOOK
        id_records = patchBlock.id_records
        for record in modFile.BOOK.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            if record.flags.isScroll and not record.flags.isFixed:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        reOldLabel = re.compile('^(\([ACDIMR]\d\)|\w{3,6}:) ')
        orderFormat, magicFormat = self.orderFormat, self.magicFormat
        keep = patchFile.getKeeper()
        id_ench = patchFile.ENCH.id_records
        effectId_school = dict((key,value[0]) for key,value in bush.magicEffects.items())
        for record in patchFile.BOOK.records:
            if not record.full or not record.flags.isScroll or record.flags.isFixed: continue
            #--Magic label
            isEnchanted = bool(record.enchantment)
            if magicFormat and isEnchanted:
                school = 6 #--Default to 6 (U: unknown)
                enchantment = id_ench.get(record.enchantment)
                if enchantment and enchantment.effects:
                    effect = enchantment.effects[0]
                    effectId = effect.id
                    if effect.scriptEffect:
                        school = effect.scriptEffect.school
                    else:
                        school = effectId_school.get(effectId,6)
                record.full = reOldLabel.sub('',record.full) #--Remove existing label
                record.full = magicFormat % ('ACDIMRU'[school],) + record.full
            #--Ordering
            record.full = orderFormat[isEnchanted] + record.full
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log(_('* %d %s') % (sum(count.values()),self.label))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class NamesTweak_Spells(MultiTweakItem):
    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Spells"),
            _('Label spells to sort by school and level.'),
            'SPEL',
            (_('Fire Ball'),  'NOTAGS'),
            ('----','----'),
            (_('D Fire Ball'),  '%s '),
            (_('D. Fire Ball'), '%s. '),
            (_('D - Fire Ball'),'%s - '),
            (_('(D) Fire Ball'),'(%s) '),
            ('----','----'),
            (_('D2 Fire Ball'),  '%s%d '),
            (_('D2. Fire Ball'), '%s%d. '),
            (_('D2 - Fire Ball'),'%s%d - '),
            (_('(D2) Fire Ball'),'(%s%d) '),
            )

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreSpel,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreSpel,)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        patchBlock = patchFile.SPEL
        id_records = patchBlock.id_records
        for record in modFile.SPEL.getActiveRecords():
            if mapper(record.formid) in id_records: continue
            if record.spellType == 0:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        format = self.choiceValues[self.chosen][0]
        removeTags = '%s' not in format
        showLevel = '%d' in format
        keep = patchFile.getKeeper()
        reOldLabel = re.compile('^(\([ACDIMR]\d\)|\w{3,6}:) ')
        effectId_school = dict((key,value[0]) for key,value in bush.magicEffects.items())
        for record in patchFile.SPEL.records:
            if record.spellType != 0 or not record.full: continue
            school = 6 #--Default to 6 (U: unknown)
            if record.effects:
                effect = record.effects[0]
                effectId = effect.id
                if effect.scriptEffect:
                    school = effect.scriptEffect.school
                else:
                    school = effectId_school.get(effectId,6)
            newFull = reOldLabel.sub('',record.full) #--Remove existing label
            if not removeTags:
                if showLevel:
                    newFull = format % ('ACDIMRU'[school],record.level) + newFull
                else:
                    newFull = format % ('ACDIMRU'[school],) + newFull
            if newFull != record.full:
                record.full = newFull
                keep(record.formid)
                srcMod = record.formid[0]
                count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log(_('* %d %s') % (sum(count.values()),self.label))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class NamesTweak_Weapons(MultiTweakItem):
    #--Config Phase -----------------------------------------------------------
    def __init__(self):
        MultiTweakItem.__init__(self,_("Weapons"),
            _('Label ammo and weapons to sort by type and damage.'),
            'WEAP',
            (_('B Iron Bow'),  '%s '),
            (_('B. Iron Bow'), '%s. '),
            (_('B - Iron Bow'),'%s - '),
            (_('(B) Iron Bow'),'(%s) '),
            ('----','----'),
            (_('B08 Iron Bow'),  '%s%02d '),
            (_('B08. Iron Bow'), '%s%02d. '),
            (_('B08 - Iron Bow'),'%s%02d - '),
            (_('(B08) Iron Bow'),'(%s%02d) '),
            )

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreAmmo,MreWeap)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreAmmo,MreWeap)

    def scanModFile(self,modFile,progress,patchFile):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        mapper = modFile.getLongMapper()
        for blockType in ('AMMO','WEAP'):
            modBlock = getattr(modFile,blockType)
            patchBlock = getattr(patchFile,blockType)
            id_records = patchBlock.id_records
            for record in modBlock.getActiveRecords():
                if mapper(record.formid) not in id_records:
                    record = record.getTypeCopy(mapper)
                    patchBlock.setRecord(record.formid,record)
                
    def buildPatch(self,log,progress,patchFile):
        """Edits patch file as desired. Will write to log."""
        count = {}
        format = self.choiceValues[self.chosen][0]
        showStat = '%02d' in format
        keep = patchFile.getKeeper()
        for record in patchFile.AMMO.records:
            if not record.full: continue
            if showStat:
                record.full = format % ('A',record.damage) + record.full
            else:
                record.full = format % ('A',) + record.full
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        for record in patchFile.WEAP.records:
            if not record.full: continue
            if showStat:
                record.full = format % ('CDEFGB'[record.weaponType],record.damage) + record.full
            else:
                record.full = format % ('CDEFGB'[record.weaponType],) + record.full
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log(_('* %d %s') % (sum(count.values()),self.label))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class NamesTweaker(MultiTweaker):
    """Tweaks record full names in various ways."""
    name = _('Tweak Names')
    text = _("Tweak object names to show type and/or quality.")
    tweaks = sorted([
        NamesTweak_Body(_("Armor"),_("Rename armor to sort by type."),'ARMO',
            (_('BL Leather Boots'),  '%s '),
            (_('BL. Leather Boots'), '%s. '),
            (_('BL - Leather Boots'),'%s - '),
            (_('(BL) Leather Boots'),'(%s) '),
            ('----','----'),
            (_('BL02 Leather Boots'),  '%s%02d '),
            (_('BL02. Leather Boots'), '%s%02d. '),
            (_('BL02 - Leather Boots'),'%s%02d - '),
            (_('(BL02) Leather Boots'),'(%s%02d) '),
            ),
        NamesTweak_Body(_("Clothes"),_("Rename clothes to sort by type."),'CLOT',
            (_('P Grey Trowsers'),  '%s '),
            (_('P. Grey Trowsers'), '%s. '),
            (_('P - Grey Trowsers'),'%s - '),
            (_('(P) Grey Trowsers'),'(%s) '),
            ),
        NamesTweak_Potions(),
        NamesTweak_Scrolls(),
        NamesTweak_Spells(),
        NamesTweak_Weapons(),
        ],key=lambda a: a.label.lower())

    def __init__(self):
        Patcher.__init__(self)
        self.scanOrder = 32
        self.editOrder = 32

    #--Patch Phase ------------------------------------------------------------
    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return None
        classTuples = [tweak.getReadClasses() for tweak in self.enabledTweaks]
        return sum(classTuples,tuple())

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return None
        classTuples = [tweak.getWriteClasses() for tweak in self.enabledTweaks]
        return sum(classTuples,tuple())

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive: return
        for tweak in self.enabledTweaks:
            tweak.scanModFile(modFile,progress,self.patchFile)

    def buildPatch(self,log,progress):
        """Applies individual clothes tweaks."""
        if not self.isActive: return
        log.setHeader('= '+self.__class__.name,True)
        for tweak in self.enabledTweaks:
            tweak.buildPatch(log,progress,self.patchFile)

# Patchers: 40 ----------------------------------------------------------------
#------------------------------------------------------------------------------
class AlchemicalCatalogs(Patcher):
    """Updates COBL alchemical catalogs."""
    group = _('Special')
    scanOrder = 40
    editOrder = 40
    name = _('Alchemical Catalogs')
    text = _("Update COBL's catalogs of alchemical ingredients and effects. Will only run if Cobl Main.esm is loaded.")

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.isActive = ('COBL Main.esm' in loadMods)
        self.id_ingred = {}

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return tuple()
        return (MreIngr,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return tuple()
        return (MreBook,)

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive: return
        id_ingred = self.id_ingred
        mapper = modFile.getLongMapper()
        for record in modFile.INGR.getActiveRecords():
            if not record.full: continue #--Ingredient must have name!
            effects = record.getEffects()
            if not ('SEFF',0) in effects:
                id_ingred[mapper(record.formid)] = (record.eid, record.full, effects)

    def buildPatch(self,log,progress):
        """Edits patch file as desired. Will write to log."""
        if not self.isActive: return
        #--Setup
        effectNames = dict((key,value[1]) for key,value in bush.magicEffects.items())
        for mgef in effectNames:
            effectNames[mgef] = re.sub('(Attribute|Skill)','',effectNames[mgef])
        actorEffects = bush.actorValueEffects
        actorNames = bush.actorValues
        keep = self.patchFile.getKeeper()
        #--Book generatator
        def getBook(objectId,eid,full,value,icon,modelPath,modb):
            book = MreBook(('BOOK',0,0,0,0))
            book.longFormids = True
            book.changed = True
            book.eid = eid
            book.full = full
            book.value = value
            book.weight = 0.2
            book.formid = (Path.get('Cobl Main.esm'),objectId)
            book.text = _("Salan's Catalog of %s\r\n\r\n") % (full,)
            book.icon = icon
            book.model = book.getDefault('model')
            book.model.path = modelPath
            book.model.modb = modb
            book.modb = book
            self.patchFile.BOOK.setRecord(keep(book.formid),book)
            return book
        #--Ingredients Catalog
        id_ingred = self.id_ingred
        icon,modPath,modb = ('Clutter\IconBook9.dds','Clutter\Books\Octavo02.NIF','\x03>@A')
        for (num,objectId,full,value) in bush.ingred_alchem:
            book = getBook(objectId,'cobCatAlchemIngreds'+`num`,full,value,icon,modPath,modb)
            buff = cStringIO.StringIO()
            buff.write(book.text)
            for eid,full,effects in sorted(id_ingred.values(),key=lambda a: a[1]):
                buff.write(full+'\r\n')
                for mgef,actorValue in effects[:num]:
                    effectName = effectNames[mgef]
                    if mgef in actorEffects: effectName += actorNames[actorValue]
                    buff.write('  '+effectName+'\r\n')
            book.text = buff.getvalue()
        #--Get Ingredients by Effect
        effect_ingred = {}
        for formid,(eid,full,effects) in id_ingred.items():
            for index,(mgef,actorValue) in enumerate(effects):
                effectName = effectNames[mgef]
                if mgef in actorEffects: effectName += actorNames[actorValue]
                if effectName not in effect_ingred: effect_ingred[effectName] = []
                effect_ingred[effectName].append((index,full))
        #--Effect catalogs
        icon,modPath,modb = ('Clutter\IconBook7.dds','Clutter\Books\Octavo01.NIF','\x03>@A')
        for (num,objectId,full,value) in bush.effect_alchem:
            book = getBook(objectId,'cobCatAlchemEffects'+`num`,full,value,icon,modPath,modb)
            buff = cStringIO.StringIO()
            buff.write(book.text)
            for effectName in sorted(effect_ingred.keys()):
                effects = [indexFull for indexFull in effect_ingred[effectName] if indexFull[0] < num] 
                if effects:
                    buff.write(effectName+'\r\n')
                    for (index,full) in sorted(effects,key=lambda a: a[1]):
                        exSpace = ('',' ')[index == 0]
                        buff.write(' '+`index + 1`+exSpace+' '+full+'\r\n')
            book.text = buff.getvalue()
        #--Log
        log.setHeader(_('= Alchemical Catalogs'))
        log(_('* %d Ingredients Cataloged') % (len(id_ingred),))
        log(_('* %d Effects Cataloged') % (len(effect_ingred)))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.id_ingred = None

#------------------------------------------------------------------------------
class ListsMerger(ListPatcher):
    """Merged leveled lists mod file."""
    group = _('Special')
    scanOrder = 45
    editOrder = 45
    name = _('Leveled Lists')
    text = _("Merges changes to leveled lists from all active mods. The list below is for overriding the Delev/Relev tags of selected mods (whether active or inactive). Only advanced users should modify this list!")
    #tip = _("Merge changes to leveled lists.")
    choiceMenu = ('Auto','----','Delev','Relev') #--List of possible choices for each config item. Item 0 is default.
    autoKey = ('Delev','Relev')
    forceAuto = False
    forceItemCheck = True #--Force configChecked to True for all items

    #--Static------------------------------------------------------------------
    @staticmethod
    def getDefaultTags():
        tags = {}
        for fileName in ('Leveled Lists.csv','My Leveled Lists.csv'):
            textPath = dirs['patches'].join(fileName)
            if textPath.exists():
                reader = CsvReader(textPath)
                for fields in reader:
                    if len(fields) < 2 or not fields[0] or fields[1] not in ('DR','R','D','RD',''): continue
                    tags[Path.get(fields[0])] = fields[1]
                reader.close()
        return tags

    #--Config Phase -----------------------------------------------------------
    def getChoice(self,item):
        """Get default config choice."""
        choice = self.configChoices.get(item)
        if not isinstance(choice,set): choice = set(('Auto',))
        if 'Auto' in choice:
            if item in modInfos:
                choice = set(('Auto',))
                bashKeys = modInfos[item].getBashKeys()
                for key in ('Delev','Relev'):
                    if key in bashKeys: choice.add(key)
        self.configChoices[item] = choice
        return choice

    def getItemLabel(self,item):
        """Returns label for item to be used in list"""
        choice = map(lambda a: a[0],self.configChoices.get(item,tuple()))
        if choice:
            return '%s [%s]' % (item,''.join(sorted(choice)))
        else:
            return item

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.listTypes = ('LVLC','LVLI','LVSP')
        self.type_list = dict([(type,{}) for type in self.listTypes])
        self.masterItems = {}
        self.mastersScanned = set()
        self.levelers = None #--Will initialize later

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (MreLvlc,MreLvli,MreLvsp)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (MreLvlc,MreLvli,MreLvsp)

    def scanModFile(self, modFile, progress):
        """Add lists from modFile."""
        #--Level Masters (complete initialization)
        if self.levelers == None:
            allMods = set(self.patchFile.allMods)
            self.levelers = [leveler for leveler in self.getConfigChecked() if leveler in allMods]
            self.delevMasters = set()
            for leveler in self.levelers:
                self.delevMasters.update(modInfos[leveler].header.masters)
        #--Begin regular scan
        modName = modFile.fileInfo.name
        modFile.convertToLongFormids(self.listTypes)
        #--PreScan for later Relevs/Delevs?
        if modName in self.delevMasters:
            for type in self.listTypes:
                for levList in getattr(modFile,type).getActiveRecords():
                    masterItems = self.masterItems.setdefault(levList.formid,{})
                    masterItems[modName] = set([entry.id for entry in levList.entries])
            self.mastersScanned.add(modName)
        #--Relev/Delev setup
        configChoice = self.configChoices.get(modName,tuple())
        isRelev = ('Relev' in configChoice)
        isDelev = ('Delev' in configChoice)
        #--Scan
        for type in self.listTypes:
            levLists = self.type_list[type]
            newLevLists = getattr(modFile,type)
            for newLevList in newLevLists.getActiveRecords():
                listId = newLevList.formid
                isListOwner = (listId[0] == modName)
                #--Items, delevs and relevs sets
                newLevList.items = items = set([entry.id for entry in newLevList.entries])
                if not isListOwner:
                    #--Relevs
                    newLevList.relevs = (set(),items.copy())[isRelev]
                    #--Delevs: all items in masters minus current items
                    newLevList.delevs = delevs = set()
                    if isDelev:
                        id_masterItems = self.masterItems.get(newLevList.formid)
                        if id_masterItems:
                            for masterName in modFile.tes4.masters:
                                if masterName in id_masterItems:
                                    delevs |= id_masterItems[masterName]
                            delevs -= items
                            newLevList.items |= delevs
                #--Cache/Merge
                if isListOwner:
                    levList = copy.deepcopy(newLevList)
                    levList.mergeSources = []
                    levLists[listId] = levList
                elif listId not in levLists: 
                    levList = copy.deepcopy(newLevList)
                    levList.mergeSources = [modName]
                    levLists[listId] = levList
                else:
                    levLists[listId].mergeWith(newLevList,modName)

    def buildPatch(self,log,progress):
        """Adds merged lists to patchfile."""
        keep = self.patchFile.getKeeper()
        #--Relevs/Delevs List
        log.setHeader('= '+self.__class__.name,True)
        log.setHeader(_('=== Delevelers/Relevelers'))
        for leveler in self.levelers:
            log('* '+self.getItemLabel(leveler))
        #--Save to patch file
        for label, type in ((_('Creature'),'LVLC'), (_('Item'),'LVLI'), (_('Spell'),'LVSP')):
            log.setHeader(_('=== Merged %s Lists') % (label,))
            patchBlock = getattr(self.patchFile,type)
            levLists = self.type_list[type]
            for record in sorted(levLists.values(),key=lambda a: a.eid):
                if not record.mergeOverLast: continue
                formid = record.formid
                patchBlock.setRecord(keep(formid),levLists[formid])
                log('* '+record.eid)
                for mod in record.mergeSources:
                    log('  * ' + self.getItemLabel(mod))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = None
        self.type_list = None

#------------------------------------------------------------------------------
class MFactMarker(ImportPatcher):
    """Mark factions that player can acquire while morphing."""
    group = _('Special')
    scanOrder = 40
    editOrder = 40
    name = _('Morph Factions')
    text = _("Mark factions that player can acquire while morphing. Requires Cobl 2.18 and Wrye Morph or similar.")
    autoRe = re.compile(r"^UNDEFINED$",re.I)
    autoKey = 'MFact'

    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.id_info = {} #--Morphable factions keyed by formid
        self.srcFiles = self.getConfigChecked()
        self.isActive = bool(self.srcFiles)

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        return (None,(MreFact,))[self.isActive]

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        return (None,(MreFact,))[self.isActive]

    def loadSourceData(self,progress):
        """Get names from source files."""
        if not self.isActive: return
        aliases = self.patchFile.aliases
        id_info = self.id_info
        for srcFile in self.srcFiles:
            textPath = dirs['patches'].join(srcFile)
            if not textPath.exists(): continue
            ins = CsvReader(textPath)
            for fields in ins:
                if len(fields) < 6 or fields[1][:2] != '0x': 
                    continue
                mod,objectIndex = fields[:2]
                longid = (Path.get(aliases.get(mod,mod)),int(objectIndex,0))
                morphName = fields[4].strip()
                rankName = fields[5].strip()
                if not morphName: continue
                if not rankName: rankName = _('Member')
                id_info[longid] = (morphName,rankName)
            ins.close()

    def scanModFile(self, modFile, progress):
        """Scan modFile."""
        if not self.isActive: return 
        id_info = self.id_info
        modName = modFile.fileInfo.name
        mapper = modFile.getLongMapper()
        patchBlock = self.patchFile.FACT
        for record in modFile.FACT.getActiveRecords():
            formid = record.formid
            if not record.longFormids: formid = mapper(formid)
            if formid in id_info:
                patchBlock.setRecord(formid,record.getTypeCopy(mapper))

    def buildPatch(self,log,progress):
        """Make changes to patchfile."""
        if not self.isActive: return
        id_info = self.id_info
        modFile = self.patchFile
        keep = self.patchFile.getKeeper()
        mFactLong = (Path("Cobl Main.esm"),0x33FB)
        changed = {}
        for record in modFile.FACT.getActiveRecords():
            if record.formid not in id_info: continue
            for relation in record.relations:
                if relation.faction == mFactLong:
                    morphFacts[formid] = faction.eid
                    break
            else:
                record.flags.hiddenFromPC = False
                relation = record.getDefault('relations')
                relation.faction = mFactLong
                relation.mod = 10
                record.relations.append(relation)
                mname,rankName = id_info[record.formid]
                record.full = mname
                if not record.ranks: 
                    record.ranks = [record.getDefault('ranks')]
                for rank in record.ranks:
                    if not rank.male: rank.male = rankName
                    if not rank.female: rank.female = rank.male
                    if not rank.insignia: 
                        rank.insignia = r'Menus\Stats\Cobl\generic%02d.dds' % (rank.rank,)
                keep(record.formid)
                mod = record.formid[0]
                changed[mod] = changed.setdefault(mod,0) + 1
        log.setHeader('= '+self.__class__.name)
        log(_("=== Source Mods/Files"))
        for file in self.srcFiles:
            log("* " +file)
        log(_("\n=== Morphable Factions"))
        for mod in sorted(changed):
            log("* %s: %d" % (mod,changed[mod]))

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.id_info = self.srcFiles = None
        self.isActive = False

#------------------------------------------------------------------------------
class PowerExhaustion(Patcher):
    """Modifies most Greater power to work with Wrye's Power Exhaustion mod."""
    group = _('Special')
    scanOrder = 40
    editOrder = 40
    name = _('Power Exhaustion')
    text = _("Modify greater powers to work with Power Exhaustion mod. Will only run if Power Exhaustion mod is installed and active.")

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.isActive = ('Power Exhaustion.esp' in loadMods)
        self.id_exhaustion = dict([((Path.get(name),num),exh) for (name,num),exh in bush.id_exhaustion.items()])

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return tuple()
        return (MreSpel,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return tuple()
        return (MreSpel,)

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive: return
        mapper = modFile.getLongMapper()
        patchRecords = self.patchFile.SPEL
        for record in modFile.SPEL.getActiveRecords():
            if not record.spellType == 2: continue
            record = record.getTypeCopy(mapper)
            if record.formid in self.id_exhaustion or ('FOAT',5) in record.getEffects():
                patchRecords.setRecord(record.formid,record)
                continue

    def buildPatch(self,log,progress):
        """Edits patch file as desired. Will write to log."""
        if not self.isActive: return
        count = {}
        exhaustId = (Path.get('Power Exhaustion.esp'),0xCE7)
        keep = self.patchFile.getKeeper()
        for record in self.patchFile.SPEL.records:
            #--Skip this one?
            if record.spellType != 2: continue
            if record.formid not in self.id_exhaustion and ('FOAT',5) not in record.getEffects(): 
                continue
            newEffects = []
            duration = self.id_exhaustion.get(record.formid,0)
            for effect in record.effects:
                if effect.id == 'FOAT' and effect.actorValue == 5 and effect.magnitude == 1:
                    duration = effect.duration
                else:
                    newEffects.append(effect)
            if not duration: continue
            record.effects = newEffects
            #--Okay, do it
            record.full = '+'+record.full
            record.spellType = 3 #--Lesser power
            effect = record.getDefault('effects')
            effect.id = 'SEFF'
            effect.duration = duration
            scriptEffect = record.getDefault('effects.scriptEffect')
            scriptEffect.full = _("Power Exhaustion")
            scriptEffect.id = exhaustId
            scriptEffect.school = 2
            scriptEffect.visual = '\x00\x00\x00\x00'
            scriptEffect.flags.hostile = False
            effect.scriptEffect = scriptEffect
            record.effects.append(effect)
            keep(record.formid)
            srcMod = record.formid[0]
            count[srcMod] = count.get(srcMod,0) + 1
        #--Log
        log.setHeader(_('= Power Exhaustion'))
        log(_('* %d Powers Tweaked') % (sum(count.values()),))
        for srcMod in sorted(count.keys()):
            log('  * %3d %s' % (count[srcMod],srcMod))

#------------------------------------------------------------------------------
class SEWorldEnforcer(Patcher):
    """Suspends Cyrodiil quests while in Shivering Isles."""
    group = _('Special')
    scanOrder = 40
    editOrder = 40
    name = _('SEWorld Tests')
    text = _("Suspends Cyrodiil quests while in Shivering Isles. I.e. re-instates GetPlayerInSEWorld tests as necessary.")

    #--Config Phase -----------------------------------------------------------
    #--Patch Phase ------------------------------------------------------------
    def beginPatch(self,patchFile,loadMods):
        """Prepare to handle specified patch mod. All functions are called after this."""
        Patcher.beginPatch(self,patchFile,loadMods)
        self.cyrodiilQuests = set()
        if 'Oblivion.esm' in loadMods:
            loadFactory = LoadFactory(False,MreQust)
            modInfo = modInfos[Path.get('Oblivion.esm')]
            modFile = ModFile(modInfo,loadFactory)
            modFile.load(True)
            mapper = modFile.getLongMapper()
            for record in modFile.QUST.getActiveRecords():
                for condition in record.conditions:
                    if condition.ifunc == 365 and condition.compValue == 0:
                        self.cyrodiilQuests.add(mapper(record.formid))
                        break
        self.isActive = bool(self.cyrodiilQuests)

    def getReadClasses(self):
        """Returns load factory classes needed for reading."""
        if not self.isActive: return tuple()
        return (MreQust,)

    def getWriteClasses(self):
        """Returns load factory classes needed for writing."""
        if not self.isActive: return tuple()
        return (MreQust,)

    def scanModFile(self,modFile,progress):
        """Scans specified mod file to extract info. May add record to patch mod, 
        but won't alter it."""
        if not self.isActive: return
        if modFile.fileInfo.name == Path('Oblivion.esm'): return
        cyrodiilQuests = self.cyrodiilQuests
        mapper = modFile.getLongMapper()
        patchBlock = self.patchFile.QUST
        for record in modFile.QUST.getActiveRecords():
            formid = mapper(record.formid)
            if formid not in cyrodiilQuests: continue
            for condition in record.conditions:
                if condition.ifunc == 365: break #--365: playerInSeWorld
            else:
                record = record.getTypeCopy(mapper)
                patchBlock.setRecord(record.formid,record)

    def buildPatch(self,log,progress):
        """Edits patch file as desired. Will write to log."""
        if not self.isActive: return
        cyrodiilQuests = self.cyrodiilQuests
        patchFile = self.patchFile
        keep = patchFile.getKeeper()
        patched = []
        for record in patchFile.QUST.getActiveRecords():
            if record.formid not in cyrodiilQuests: continue
            for condition in record.conditions:
                if condition.ifunc == 365: break #--365: playerInSeWorld
            else:
                condition = record.getDefault('conditions')
                condition.ifunc = 365
                record.conditions.insert(0,condition)
                keep(record.formid)
                patched.append(record.eid)
        log.setHeader('= '+self.__class__.name)
        log(_('===Quests Patched: %d') % (len(patched),))
        for eid in sorted(patched,key=string.lower):
            log('* '+eid)

    def endPatch(self):
        """Do any cleanup required."""
        self.patchFile = self.cyrodiilQuests = None

# Initialization --------------------------------------------------------------
def initDirs():
    #--User Shell Folders
    reEnv = re.compile('%(\w+)%')
    envDefs = os.environ
    def subEnv(match):
        key = match.group(1).upper()
        if not envDefs.get(key):
            raise BoshError(_('Undefined env variable. [%s]') % (key,))
        return envDefs[key]
    def getUserShellFolder(folderKey):
        import _winreg
        regKey = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER,
            r'Software\Microsoft\Windows\CurrentVersion\Explorer\User Shell Folders')
        path = _winreg.QueryValueEx(regKey,folderKey)[0]
        regKey.Close()
        path = path.encode(locale.getpreferredencoding())
        path = reEnv.sub(subEnv,path)
        path = Path.get(path)
        return path
    personalDir = getUserShellFolder('Personal')
    localAppDataDir = getUserShellFolder('Local AppData')

    #--User sub folders
    dirs['saveBase'] = personalDir.join(r'My Games','Oblivion')
    dirs['userApp'] = localAppDataDir.join('Oblivion')

    #--App Directories... Assume bash is in right place.
    dirs['app'] = Path.get(os.getcwd()).head()
    dirs['mods'] = dirs['app'].join('Data')
    dirs['patches'] = dirs['mods'].join('Bash Patches')
    
    #--Error checks
    dictDump = '\n'.join('  '+key+': '+`envDefs[key]` for key in sorted(envDefs))
    if not personalDir.exists():
        raise BoshError(_("Personal folder does not exist\nPersonal folder: %s\nAdditional info:\n%s") 
            % (personalDir,dictDump))
    if not localAppDataDir.exists():
        raise BoshError(_("Local app data folde does not exist.\nLocal app data folder: %s\nAdditional info:\n%s") 
            % (localAppDataDir, dictDump))
    if not dirs['app'].join('Oblivion.exe').exists():
        print dirs['app'].join('Oblivion.exe')
        raise BoshError(_("Install Error\nFailed to find Oblivion.exe in %s.\nNote that the Mopy folder should be in the same folder as Oblivion.exe.") % (dirs['app'],)) 

def initSettings(safeMode=False):
    global settings
    settings = Settings(
        dirs['saveBase'].join('BashSettings.dat'),
        dirs['userApp'].join('bash config.pkl'),
        safeMode=safeMode)
    settings.loadDefaults(settingDefaults)

# Main ------------------------------------------------------------------------
if __name__ == '__main__':
    print _('Compiled')

