# GPL License and Copyright Notice ============================================
#  This file is part of Wrye Bash.
#
#  Wrye Bash is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  Wrye Bash is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Wrye Bash; if not, write to the Free Software Foundation,
#  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  Wrye Bash copyright (C) 2005, 2006, 2007 Wrye 
#
# =============================================================================

"""This module defines static data for use by other modules in the Wrye Bash package. 
Its use should generally be restricted to large chunks of data and/or chunks of data 
that are used by multiple objects."""

# Localization ----------------------------------------------------------------
# Locale: String Translation --------------------------------------------------
import cPickle, locale, re, os
reTrans = re.compile(r'^([ :=\.]*)(.+?)([ :=\.]*$)')
def compileTranslator(txtPath,pklPath):
    """Compiles specified txtFile into pklFile."""
    reSource = re.compile(r'^=== ')
    reValue = re.compile(r'^>>>>\s*$')
    reBlank = re.compile(r'^\s*$')
    reNewLine = re.compile(r'\\n')
    #--Scan text file
    translator = {}
    def addTranslation(key,value):
        key,value   = key[:-1],value[:-1]
        #print `key`, `value`
        if key and value:
            key = reTrans.match(key).group(2)
            value = reTrans.match(value).group(2)
            translator[key] = value
    key,value,mode = '','',0
    textFile = file(txtPath)
    for line in textFile:
        #--Blank line. Terminates key, value pair
        if reBlank.match(line):
            addTranslation(key,value)
            key,value,mode = '','',0
        #--Begin key input?
        elif reSource.match(line):
            addTranslation(key,value)
            key,value,mode = '','',1
        #--Begin value input?
        elif reValue.match(line):
            mode = 2
        elif mode == 1:
            key += line
        elif mode == 2:
            value += line
    addTranslation(key,value) #--In case missed last pair
    textFile.close()
    #--Write translator to pickle
    filePath = pklPath
    tempPath = filePath+'.tmp'
    cPickle.dump(translator,open(tempPath,'w'))
    if os.path.exists(filePath): os.remove(filePath)
    os.rename(tempPath,filePath)
    
#--Do translator test and set
language = locale.getlocale()[0].split('_',1)[0]
languagePkl, languageTxt = (os.path.join('locale',language+ext) for ext in ('.pkl','.txt'))
#--Recompile pkl file?
if os.path.exists(languageTxt) and (
    not os.path.exists(languagePkl) or (
        os.path.getmtime(languageTxt) > os.path.getmtime(languagePkl)
        )
    ):
    compileTranslator(languageTxt,languagePkl)
#--Use dictionary from pickle as translator
if os.path.exists(languagePkl):
    pklFile = open(languagePkl)
    reEscQuote = re.compile(r"\\'")
    _translator = cPickle.load(pklFile)
    pklFile.close()
    def _(text,encode=True):
        if encode: text = reEscQuote.sub("'",text.encode('string_escape'))
        head,core,tail = reTrans.match(text).groups()
        if core and core in _translator: 
            text = head+_translator[core]+tail
        if encode: text = text.decode('string_escape')
        return text
else:
    def _(text,encode=True): return text

# Imports ---------------------------------------------------------------------
import struct

# Group/Top Types -------------------------------------------------------------
groupTypes = [
    _('Top (Type)'),
    _('World Children'),
    _('Int Cell Block'),
    _('Int Cell Sub-Block'),
    _('Ext Cell Block'),
    _('Ext Cell Sub-Block'),
    _('Cell Children'),
    _('Topic Children'),
    _('Cell Persistent Childen'),
    _('Cell Temporary Children'),
    _('Cell Visible Distant Children'),
]

#--Top types in Oblivion order.
topTypes = ['GMST', 'GLOB', 'CLAS', 'FACT', 'HAIR', 'EYES', 'RACE', 'SOUN', 'SKIL', 
    'MGEF', 'SCPT', 'LTEX', 'ENCH', 'SPEL', 'BSGN', 'ACTI', 'APPA', 'ARMO', 'BOOK', 
    'CLOT', 'CONT', 'DOOR', 'INGR', 'LIGH', 'MISC', 'STAT', 'GRAS', 'TREE', 'FLOR', 
    'FURN', 'WEAP', 'AMMO', 'NPC_', 'CREA', 'LVLC', 'SLGM', 'KEYM', 'ALCH', 'SBSP', 
    'SGST', 'LVLI', 'WTHR', 'CLMT', 'REGN', 'CELL', 'WRLD', 'DIAL', 'QUST', 'IDLE', 
    'PACK', 'CSTY', 'LSCR', 'LVSP', 'ANIO', 'WATR', 'EFSH']

#--Dict mapping 'ignored' top types to un-ignored top types.
topIgTypes = dict([(struct.pack('I',(struct.unpack('I',type)[0]) | 0x1000),type) for type in topTypes])

# Race Info -------------------------------------------------------------------
raceNames = {
    0x23fe9 : _('Argonian'),
    0x224fc : _('Breton'),
    0x191c1 : _('Dark Elf'),
    0x19204 : _('High Elf'),
    0x00907 : _('Imperial'),
    0x22c37 : _('Khajiit'),
    0x224fd : _('Nord'),
    0x191c0 : _('Orc'),
    0x00d43 : _('Redguard'),
    0x00019 : _('Vampire'),
    0x223c8 : _('Wood Elf'),
    }

raceShortNames = {
    0x23fe9 : 'Arg',
    0x224fc : 'Bre',
    0x191c1 : 'Dun',
    0x19204 : 'Alt',
    0x00907 : 'Imp',
    0x22c37 : 'Kha',
    0x224fd : 'Nor',
    0x191c0 : 'Orc',
    0x00d43 : 'Red',
    0x223c8 : 'Bos',
    }

raceHairMale = {
    0x23fe9 : 0x64f32, #--Arg
    0x224fc : 0x90475, #--Bre
    0x191c1 : 0x64214, #--Dun
    0x19204 : 0x7b792, #--Alt
    0x00907 : 0x90475, #--Imp
    0x22c37 : 0x653d4, #--Kha
    0x224fd : 0x1da82, #--Nor
    0x191c0 : 0x66a27, #--Orc
    0x00d43 : 0x64215, #--Red
    0x223c8 : 0x690bc, #--Bos
    }

raceHairFemale = {
    0x23fe9 : 0x64f33, #--Arg
    0x224fc : 0x1da83, #--Bre
    0x191c1 : 0x1da83, #--Dun
    0x19204 : 0x690c2, #--Alt
    0x00907 : 0x1da83, #--Imp
    0x22c37 : 0x653d0, #--Kha
    0x224fd : 0x1da83, #--Nor
    0x191c0 : 0x64218, #--Orc
    0x00d43 : 0x64210, #--Red
    0x223c8 : 0x69473, #--Bos
    }

#--Race specials from Oblvion.esm. 
#..Does NOT account for changes from mods!
raceSpecials = {
    0x23fe9 : #--Argonian
        (0x47ace, 0x47acd, 0x47ad6),
    0x224fc : #--Breton
        (0x47ad2, 0x47ad1, 
         0x47ad0,),
    0x191c1 : #--Dark Elf
        (0x47ad4, 
         0x47ad5, ),
    0x19204 : #--High Elf
        (0x47adc, 0x47ad7, 0x14d52, ),
    0x00907 : #--Imperial
        (0x47ade, 0x47add, ),
    0x22c37 : #--Khajiit
        (0x47ae0, 0x47adf,),
    0x224fd : #--Nord
        (0x47ae2, 
         0x47ae4, 0x47ae3, ),
    0x191c0 : #--Orc
        (0x47acf, 
         0x47ad3, ),
    0x00d43 : #--Redguard
        (0x47ae6, 0x47ae5, 
         0x47ae7, ),
    0x223c8 : #--Wood Elf
        (0x47ae8, 
         0x47ae9),
    }

# Function Info ---------------------------------------------------------------
conditionFunctionData = ( #--0: no param; 1: int param; 2: formid param
    (153, 'CanHaveFlames', 0, 0),
    (127, 'CanPayCrimeGold', 0, 0),
    ( 14, 'GetActorValue', 1, 0),
    ( 61, 'GetAlarmed', 0, 0),
    (190, 'GetAmountSoldStolen', 0, 0),
    (  8, 'GetAngle', 1, 0),
    ( 81, 'GetArmorRating', 0, 0),
    (274, 'GetArmorRatingUpperBody', 0, 0),
    ( 63, 'GetAttacked', 0, 0),
    (264, 'GetBarterGold', 0, 0),
    (277, 'GetBaseActorValue', 1, 0),
    (229, 'GetClassDefaultMatch', 0, 0),
    ( 41, 'GetClothingValue', 0, 0),
    (122, 'GetCrime', 2, 1),
    (116, 'GetCrimeGold', 0, 0),
    (110, 'GetCurrentAIPackage', 0, 0),
    (143, 'GetCurrentAIProcedure', 0, 0),
    ( 18, 'GetCurrentTime', 0, 0),
    (148, 'GetCurrentWeatherPercent', 0, 0),
    (170, 'GetDayOfWeek', 0, 0),
    ( 46, 'GetDead', 0, 0),
    ( 84, 'GetDeadCount', 2, 0),
    (203, 'GetDestroyed', 0, 0),
    ( 45, 'GetDetected', 2, 0),
    (180, 'GetDetectionLevel', 2, 0),
    ( 35, 'GetDisabled', 0, 0),
    ( 39, 'GetDisease', 0, 0),
    ( 76, 'GetDisposition', 2, 0),
    (  1, 'GetDistance', 2, 0),
    (215, 'GetDoorDefaultOpen', 0, 0),
    (182, 'GetEquipped', 2, 0),
    ( 73, 'GetFactionRank', 2, 0),
    ( 60, 'GetFactionRankDifference', 2, 2),
    (128, 'GetFatiguePercentage', 0, 0),
    (288, 'GetFriendHit', 2, 0),
    (160, 'GetFurnitureMarkerID', 0, 0),
    ( 74, 'GetGlobalValue', 2, 0),
    ( 48, 'GetGold', 0, 0),
    ( 99, 'GetHeadingAngle', 2, 0),
    (318, 'GetIdleDoneOnce', 0, 0),
    (338, 'GetIgnoreFriendlyHits', 0, 0),
    ( 67, 'GetInCell', 2, 0),
    (230, 'GetInCellParam', 2, 2),
    ( 71, 'GetInFaction', 2, 0),
    ( 32, 'GetInSameCell', 2, 0),
    (305, 'GetInvestmentGold', 0, 0),
    (310, 'GetInWorldspace', 2, 0),
    ( 91, 'GetIsAlerted', 0, 0),
    ( 68, 'GetIsClass', 2, 0),
    (228, 'GetIsClassDefault', 2, 0),
    ( 64, 'GetIsCreature', 0, 0),
    (161, 'GetIsCurrentPackage', 2, 0),
    (149, 'GetIsCurrentWeather', 2, 0),
    (237, 'GetIsGhost', 0, 0),
    ( 72, 'GetIsID', 2, 0),
    (254, 'GetIsPlayableRace', 0, 0),
    (224, 'GetIsPlayerBirthsign', 2, 0),
    ( 69, 'GetIsRace', 2, 0),
    (136, 'GetIsReference', 2, 0),
    ( 70, 'GetIsSex', 1, 0),
    (246, 'GetIsUsedItem', 2, 0),
    (247, 'GetIsUsedItemType', 1, 0),
    ( 47, 'GetItemCount', 2, 0),
    (107, 'GetKnockedState', 0, 0),
    ( 80, 'GetLevel', 0, 0),
    ( 27, 'GetLineOfSight', 2, 0),
    (  5, 'GetLocked', 0, 0),
    ( 65, 'GetLockLevel', 0, 0),
    (320, 'GetNoRumors', 0, 0),
    (255, 'GetOffersServicesNow', 0, 0),
    (157, 'GetOpenState', 0, 0),
    (193, 'GetPCExpelled', 2, 0),
    (199, 'GetPCFactionAttack', 2, 0),
    (195, 'GetPCFactionMurder', 2, 0),
    (197, 'GetPCFactionSteal', 2, 0),
    (201, 'GetPCFactionSubmitAuthority', 2, 0),
    (249, 'GetPCFame', 0, 0),
    (132, 'GetPCInFaction', 2, 0),
    (251, 'GetPCInfamy', 0, 0),
    (129, 'GetPCIsClass', 2, 0),
    (130, 'GetPCIsRace', 2, 0),
    (131, 'GetPCIsSex', 1, 0),
    (312, 'GetPCMiscStat', 1, 0),
    (225, 'GetPersuasionNumber', 0, 0),
    ( 98, 'GetPlayerControlsDisabled', 0, 0),
    (365, 'GetPlayerInSEWorld',0,0),
    (362, 'GetPlayerHasLastRiddenHorse', 0, 0),
    (  6, 'GetPos', 1, 0),
    ( 56, 'GetQuestRunning', 2, 0),
    ( 79, 'GetQuestVariable', 2, 1),
    ( 77, 'GetRandomPercent', 0, 0),
    (244, 'GetRestrained', 0, 0),
    ( 24, 'GetScale', 0, 0),
    ( 53, 'GetScriptVariable', 2, 1),
    ( 12, 'GetSecondsPassed', 0, 0),
    ( 66, 'GetShouldAttack', 2, 0),
    (159, 'GetSitting', 0, 0),
    ( 49, 'GetSleeping', 0, 0),
    ( 58, 'GetStage', 2, 0),
    ( 59, 'GetStageDone', 2, 1),
    ( 11, 'GetStartingAngle', 1, 0),
    ( 10, 'GetStartingPos', 1, 0),
    ( 50, 'GetTalkedToPC', 0, 0),
    (172, 'GetTalkedToPCParam', 2, 0),
    (361, 'GetTimeDead', 0, 0),
    (315, 'GetTotalPersuasionNumber', 0, 0),
    (144, 'GetTrespassWarningLevel', 0, 0),
    (242, 'GetUnconscious', 0, 0),
    (259, 'GetUsedItemActivate', 0, 0),
    (258, 'GetUsedItemLevel', 0, 0),
    ( 40, 'GetVampire', 0, 0),
    (142, 'GetWalkSpeed', 0, 0),
    (108, 'GetWeaponAnimType', 0, 0),
    (109, 'GetWeaponSkillType', 0, 0),
    (147, 'GetWindSpeed', 0, 0),
    (154, 'HasFlames', 0, 0),
    (214, 'HasMagicEffect', 2, 0),
    (227, 'HasVampireFed', 0, 0),
    (353, 'IsActor', 0, 0),
    (314, 'IsActorAVictim', 0, 0),
    (313, 'IsActorEvil', 0, 0),
    (306, 'IsActorUsingATorch', 0, 0),
    (280, 'IsCellOwner', 2, 2),
    (267, 'IsCloudy', 0, 0),
    (150, 'IsContinuingPackagePCNear', 0, 0),
    (163, 'IsCurrentFurnitureObj', 2, 0),
    (162, 'IsCurrentFurnitureRef', 2, 0),
    (354, 'IsEssential', 0, 0),
    (106, 'IsFacingUp', 0, 0),
    (125, 'IsGuard', 0, 0),
    (282, 'IsHorseStolen', 0, 0),
    (112, 'IsIdlePlaying', 0, 0),
    (289, 'IsInCombat', 0, 0),
    (332, 'IsInDangerousWater', 0, 0),
    (300, 'IsInInterior', 0, 0),
    (146, 'IsInMyOwnedCell', 0, 0),
    (285, 'IsLeftUp', 0, 0),
    (278, 'IsOwner', 2, 0),
    (176, 'IsPCAMurderer', 0, 0),
    (175, 'IsPCSleeping', 0, 0),
    (171, 'IsPlayerInJail', 0, 0),
    (358, 'IsPlayerMovingIntoNewSpace', 0, 0),
    (339, 'IsPlayersLastRiddenHorse', 0, 0),
    (266, 'IsPleasant', 0, 0),
    ( 62, 'IsRaining', 0, 0),
    (327, 'IsRidingHorse', 0, 0),
    (287, 'IsRunning', 0, 0),
    (103, 'IsShieldOut', 0, 0),
    (286, 'IsSneaking', 0, 0),
    ( 75, 'IsSnowing', 0, 0),
    (223, 'IsSpellTarget', 2, 0),
    (185, 'IsSwimming', 0, 0),
    (141, 'IsTalking', 0, 0),
    (265, 'IsTimePassing', 0, 0),
    (102, 'IsTorchOut', 0, 0),
    (145, 'IsTrespassing', 0, 0),
    (329, 'IsTurnArrest', 0, 0),
    (111, 'IsWaiting', 0, 0),
    (101, 'IsWeaponOut', 0, 0),
    (309, 'IsXBox', 0, 0),
    (104, 'IsYielding', 0, 0),
    ( 36, 'MenuMode', 1, 0),
    ( 42, 'SameFaction', 2, 0),
    (133, 'SameFactionAsPC', 0, 0),
    ( 43, 'SameRace', 2, 0),
    (134, 'SameRaceAsPC', 0, 0),
    ( 44, 'SameSex', 2, 0),
    (135, 'SameSexAsPC', 0, 0),
    (323, 'WhichServiceMenu', 0, 0),
    )
allConditions = set(entry[0] for entry in conditionFunctionData)
fid1Conditions = set(entry[0] for entry in conditionFunctionData if entry[2] == 2)
fid2Conditions = set(entry[0] for entry in conditionFunctionData if entry[3] == 2)

# Magic Info ------------------------------------------------------------------
weaponTypes = (
    _('Blade (1 Handed)'),
    _('Blade (2 Handed)'),
    _('Blunt (1 Handed)'),
    _('Blunt (2 Handed)'),
    _('Staff'),
    _('Bow'),
    )

magicEffects = {
    'ABAT': (5,'Absorb Attribute'),
    'ABFA': (5,'Absorb Fatigue'),
    'ABHE': (5,'Absorb Health'),
    'ABSK': (5,'Absorb Skill'),
    'ABSP': (5,'Absorb Magicka'),
    'BABO': (1,'Bound Boots'),
    'BACU': (1,'Bound Cuirass'),
    'BAGA': (1,'Bound Gauntlets'),
    'BAGR': (1,'Bound Greaves'),
    'BAHE': (1,'Bound Helmet'),
    'BASH': (1,'Bound Shield'),
    'BRDN': (0,'Burden'),
    'BW01': (1,'Bound Order Weapon 1'),
    'BW02': (1,'Bound Order Weapon 2'),
    'BW03': (1,'Bound Order Weapon 3'),
    'BW04': (1,'Bound Order Weapon 4'),
    'BW05': (1,'Bound Order Weapon 5'),
    'BW06': (1,'Bound Order Weapon 6'),
    'BW07': (1,'Summon Staff of Sheogorath'),
    'BW08': (1,'Bound Priest Dagger'),
    'BWAX': (1,'Bound Axe'),
    'BWBO': (1,'Bound Bow'),
    'BWDA': (1,'Bound Dagger'),
    'BWMA': (1,'Bound Mace'),
    'BWSW': (1,'Bound Sword'),
    'CALM': (3,'Calm'),
    'CHML': (3,'Chameleon'),
    'CHRM': (3,'Charm'),
    'COCR': (3,'Command Creature'),
    'COHU': (3,'Command Humanoid'),
    'CUDI': (5,'Cure Disease'),
    'CUPA': (5,'Cure Paralysis'),
    'CUPO': (5,'Cure Poison'),
    'DARK': (3,'DO NOT USE - Darkness'),
    'DEMO': (3,'Demoralize'),
    'DGAT': (2,'Damage Attribute'),
    'DGFA': (2,'Damage Fatigue'),
    'DGHE': (2,'Damage Health'),
    'DGSP': (2,'Damage Magicka'),
    'DIAR': (2,'Disintegrate Armor'),
    'DISE': (2,'Disease Info'), #--Formid == 0
    'DIWE': (2,'Disintegrate Weapon'),
    'DRAT': (2,'Drain Attribute'),
    'DRFA': (2,'Drain Fatigue'),
    'DRHE': (2,'Drain Health'),
    'DRSK': (2,'Drain Skill'),
    'DRSP': (2,'Drain Magicka'),
    'DSPL': (4,'Dispel'),
    'DTCT': (4,'Detect Life'),
    'DUMY': (2,'Mehrunes Dagon'), #--Formid == 0 
    'FIDG': (2,'Fire Damage'),
    'FISH': (0,'Fire Shield'),
    'FOAT': (5,'Fortify Attribute'),
    'FOFA': (5,'Fortify Fatigue'),
    'FOHE': (5,'Fortify Health'),
    'FOMM': (5,'Fortify Magicka Multiplier'),
    'FOSK': (5,'Fortify Skill'),
    'FOSP': (5,'Fortify Magicka'),
    'FRDG': (2,'Frost Damage'),
    'FRNZ': (3,'Frenzy'),
    'FRSH': (0,'Frost Shield'),
    'FTHR': (0,'Feather'),
    'INVI': (3,'Invisibility'),
    'LGHT': (3,'Light'),
    'LISH': (0,'Shock Shield'),
    'LOCK': (0,'DO NOT USE - Lock'),
    'MYHL': (1,'Summon Mythic Dawn Helm'),
    'MYTH': (1,'Summon Mythic Dawn Armor'),
    'NEYE': (3,'Night-Eye'),
    'OPEN': (0,'Open'),
    'PARA': (3,'Paralyze'),
    'POSN': (2,'Poison Info'),
    'RALY': (3,'Rally'),
    'REAN': (1,'Reanimate'),
    'REAT': (5,'Restore Attribute'),
    'REDG': (4,'Reflect Damage'),
    'REFA': (5,'Restore Fatigue'),
    'REHE': (5,'Restore Health'),
    'RESP': (5,'Restore Magicka'),
    'RFLC': (4,'Reflect Spell'),
    'RSDI': (5,'Resist Disease'),
    'RSFI': (5,'Resist Fire'),
    'RSFR': (5,'Resist Frost'),
    'RSMA': (5,'Resist Magic'),
    'RSNW': (5,'Resist Normal Weapons'),
    'RSPA': (5,'Resist Paralysis'),
    'RSPO': (5,'Resist Poison'),
    'RSSH': (5,'Resist Shock'),
    'RSWD': (5,'Resist Water Damage'), #--Formid == 0
    'SABS': (4,'Spell Absorption'),
    'SEFF': (0,'Script Effect'),
    'SHDG': (2,'Shock Damage'),
    'SHLD': (0,'Shield'),
    'SLNC': (3,'Silence'),
    'STMA': (2,'Stunted Magicka'),
    'STRP': (4,'Soul Trap'),
    'SUDG': (2,'Sun Damage'),
    'TELE': (4,'Telekinesis'),
    'TURN': (1,'Turn Undead'),
    'VAMP': (2,'Vampirism'),
    'WABR': (0,'Water Breathing'),
    'WAWA': (0,'Water Walking'),
    'WKDI': (2,'Weakness to Disease'),
    'WKFI': (2,'Weakness to Fire'),
    'WKFR': (2,'Weakness to Frost'),
    'WKMA': (2,'Weakness to Magic'),
    'WKNW': (2,'Weakness to Normal Weapons'),
    'WKPO': (2,'Weakness to Poison'),
    'WKSH': (2,'Weakness to Shock'),
    'Z001': (1,'Summon Rufio\'s Ghost'),
    'Z002': (1,'Summon Ancestor Guardian'),
    'Z003': (1,'Summon Spiderling'),
    'Z004': (1,'Summon Flesh Atronach'),
    'Z005': (1,'Summon Bear'),
    'Z006': (1,'Summon Gluttonous Hunger'),
    'Z007': (1,'Summon Ravenous Hunger'),
    'Z008': (1,'Summon Voracious Hunger'),
    'Z009': (1,'Summon Dark Seducer'),
    'Z010': (1,'Summon Golden Saint'),
    'Z011': (1,'Wabba Summon'),
    'Z012': (1,'Summon Decrepit Shambles'),
    'Z013': (1,'Summon Shambles'),
    'Z014': (1,'Summon Replete Shambles'),
    'Z015': (1,'Summon Hunger'),
    'Z016': (1,'Summon Mangled Flesh Atronach'),
    'Z017': (1,'Summon Torn Flesh Atronach'),
    'Z018': (1,'Summon Stitched Flesh Atronach'),
    'Z019': (1,'Summon Sewn Flesh Atronach'),
    'ZCLA': (1,'Summon Clannfear'),
    'ZDAE': (1,'Summon Daedroth'),
    'ZDRE': (1,'Summon Dremora'),
    'ZDRL': (1,'Summon Dremora Lord'),
    'ZFIA': (1,'Summon Flame Atronach'),
    'ZFRA': (1,'Summon Frost Atronach'),
    'ZGHO': (1,'Summon Ghost'),
    'ZHDZ': (1,'Summon Headless Zombie'),
    'ZLIC': (1,'Summon Lich'),
    'ZSCA': (1,'Summon Scamp'),
    'ZSKA': (1,'Summon Skeleton Guardian'),
    'ZSKC': (1,'Summon Skeleton Champion'),
    'ZSKE': (1,'Summon Skeleton'),
    'ZSKH': (1,'Summon Skeleton Hero'),
    'ZSPD': (1,'Summon Spider Daedra'),
    'ZSTA': (1,'Summon Storm Atronach'),
    'ZWRA': (1,'Summon Faded Wraith'),
    'ZWRL': (1,'Summon Gloom Wraith'),
    'ZXIV': (1,'Summon Xivilai'),
    'ZZOM': (1,'Summon Zombie'),
    }

poisonEffects = set((
    'ABAT', #--Absorb Attribute
    'ABFA', #--Absorb Fatigue
    'ABHE', #--Absorb Health
    'ABSK', #--Absorb Skill
    'ABSP', #--Absorb Magicka
    'BRDN', #--Burden
    'DEMO', #--Demoralize
    'DGAT', #--Damage Attribute
    'DGFA', #--Damage Fatigue
    'DGHE', #--Damage Health
    'DGSP', #--Damage Magicka
    'DIAR', #--Disintegrate Armor
    'DIWE', #--Disintegrate Weapon
    'DRAT', #--Drain Attribute
    'DRFA', #--Drain Fatigue
    'DRHE', #--Drain Health
    'DRSK', #--Drain Skill
    'DRSP', #--Drain Magicka
    'FIDG', #--Fire Damage
    'FRDG', #--Frost Damage
    'FRNZ', #--Frenzy
    'PARA', #--Paralyze
    'SHDG', #--Shock Damage
    'SLNC', #--Silence
    'STMA', #--Stunted Magicka
    'STRP', #--Soul Trap
    'SUDG', #--Sun Damage
    'TURN', #--Turn Undead
    'WKDI', #--Weakness to Disease
    'WKFI', #--Weakness to Fire
    'WKFR', #--Weakness to Frost
    'WKMA', #--Weakness to Magic
    'WKNW', #--Weakness to Normal Weapons
    'WKPO', #--Weakness to Poison
    'WKSH', #--Weakness to Shock
    ))

actorValueEffects = set([
    'ABAT', #--Absorb Attribute
    'ABSK', #--Absorb Skill
    'DGAT', #--Damage Attribute
    'DRAT', #--Drain Attribute
    'DRSK', #--Drain Skill
    'FOAT', #--Fortify Attribute
    'FOSK', #--Fortify Skill
    'REAT', #--Restore Attribute
    ])
    
actorValues = [
    'Strength', #--00
    'Intelligence',
    'Willpower',
    'Agility',
    'Speed',
    'Endurance',
    'Personality',
    'Luck',
    'Health',
    'Magicka',

    'Fatigue', #--10
    'Encumbrance',
    'Armorer',
    'Athletics',
    'Blade',
    'Block',
    'Blunt',
    'Hand To Hand',
    'Heavy Armor',
    'Alchemy',

    'Alteration', #--20
    'Conjuration',
    'Destruction',
    'Illusion',
    'Mysticism',
    'Restoration',
    'Acrobatics',
    'Light Armor',
    'Marksman',
    'Mercantile',

    'Security', #--30
    'Sneak',
    'Speechcraft',
    'Aggression',
    'Confidence',
    'Energy',
    'Responsibility',
    'Bounty',
    'UNKNOWN 38',
    'UNKNOWN 39',

    'MagickaMultiplier', #--40
    'NightEyeBonus',
    'AttackBonus',
    'DefendBonus',
    'CastingPenalty',
    'Blindness',
    'Chameleon',
    'Invisibility',
    'Paralysis',
    'Silence',

    'Confusion', #--50
    'DetectItemRange',
    'SpellAbsorbChance',
    'SpellReflectChance',
    'SwimSpeedMultiplier',
    'WaterBreathing',
    'WaterWalking',
    'StuntedMagicka',
    'DetectLifeRange',
    'ReflectDamage',

    'Telekinesis', #--60
    'ResistFire',
    'ResistFrost',
    'ResistDisease',
    'ResistMagic',
    'ResistNormalWeapons',
    'ResistParalysis',
    'ResistPoison',
    'ResistShock',
    'Vampirism',

    'Darkness', #--70
    'ResistWaterDamage', 
    ]

acbs = {
    'Armorer': 0,
    'Athletics': 1,
    'Blade': 2,
    'Block': 3,
    'Blunt': 4,
    'Hand to Hand': 5,
    'Heavy Armor': 6,
    'Alchemy': 7,
    'Alteration': 8,
    'Conjuration': 9,
    'Destruction': 10,
    'Illusion': 11,
    'Mysticism': 12,
    'Restoration': 13,
    'Acrobatics': 14,
    'Light Armor': 15,  
    'Marksman': 16,
    'Mercantile': 17,
    'Security': 18,
    'Sneak': 19,
    'Speechcraft': 20,
    'Health': 21,
    'Strength': 25,
    'Intelligence': 26,
    'Willpower': 27,
    'Agility': 28,
    'Speed': 29,
    'Endurance': 30,
    'Personality': 31,
    'Luck': 32,
    }

 # Save File Info --------------------------------------------------------------
saveRecTypes = {
    6 : _('Faction'),
    20: _('Armor'),
    21: _('Book'),
    22: _('Clothing'),
    26: _('Light'),
    27: _('Misc. Item'),
    33: _('Weapon'),
    35: _('NPC'),
    36: _('Creature'),
    39: _('Key'),
    48: _('Cell'),
    49: _('Object Ref'),
    50: _('NPC Ref'),
    51: _('Creature Ref'),
    58: _('Dialog Entry'),
    59: _('Quest'),
    61: _('AI Package'),
    }

# Alchemical Catalogs ---------------------------------------------------------
ingred_alchem = (
    (1,0xCED,_('Alchemical Ingredients I'),250),
    (2,0xCEC,_('Alchemical Ingredients II'),500),
    (3,0xCEB,_('Alchemical Ingredients III'),1000),
    (4,0xCE7,_('Alchemical Ingredients IV'),2000),
    )
effect_alchem = (
    (1,0xCEA,_('Alchemical Effects I'),500),
    (2,0xCE9,_('Alchemical Effects II'),1000),
    (3,0xCE8,_('Alchemical Effects III'),2000),
    (4,0xCE6,_('Alchemical Effects IV'),4000),
    )

# Power Exhaustion ------------------------------------------------------------
id_exhaustion = {
    ('Oblivion.esm', 0x014D23): 9, # AbPilgrimsGrace
    ('Oblivion.esm', 0x022A43): 7, # BSLoverKiss
    ('Oblivion.esm', 0x022A3A): 7, # BSRitualMaraGift
    ('Oblivion.esm', 0x022A63): 7, # BSSerpent
    ('Oblivion.esm', 0x022A66): 7, # BSShadowMoonshadow
    ('Oblivion.esm', 0x022A6C): 7, # BSTower
    ('Oblivion.esm', 0x0CB623): 7, # BSTowerWarden
    ('Oblivion.esm', 0x06B69D): 7, # DoomstoneAetherius
    ('Oblivion.esm', 0x06A8EE): 7, # DoomstoneApprentice
    ('Oblivion.esm', 0x06A8EF): 7, # DoomstoneAtronach
    ('Oblivion.esm', 0x06B6A3): 7, # DoomstoneDragon
    ('Oblivion.esm', 0x06B69F): 7, # DoomstoneJode
    ('Oblivion.esm', 0x06B69E): 7, # DoomstoneJone
    ('Oblivion.esm', 0x06A8F2): 7, # DoomstoneLady
    ('Oblivion.esm', 0x06A8F3): 7, # DoomstoneLord
    ('Oblivion.esm', 0x06A8F5): 7, # DoomstoneLover
    ('Oblivion.esm', 0x06A8ED): 7, # DoomstoneMage
    ('Oblivion.esm', 0x06B6A1): 7, # DoomstoneMagnus
    ('Oblivion.esm', 0x06B6B1): 7, # DoomstoneNirn
    ('Oblivion.esm', 0x06A8EC): 7, # DoomstoneRitualMarasMercy
    ('Oblivion.esm', 0x06A8EB): 7, # DoomstoneRitualMarasMilk
    ('Oblivion.esm', 0x06A8F8): 7, # DoomstoneSerpent
    ('Oblivion.esm', 0x06A8F6): 7, # DoomstoneShadow
    ('Oblivion.esm', 0x06B6A2): 7, # DoomstoneShezarr
    ('Oblivion.esm', 0x06B6A0): 7, # DoomstoneSithian
    ('Oblivion.esm', 0x06A8F1): 7, # DoomstoneSteed
    ('Oblivion.esm', 0x06A8F4): 7, # DoomstoneThief
    ('Oblivion.esm', 0x06A8F7): 7, # DoomstoneTower
    ('Oblivion.esm', 0x008E53): 7, # DoomstoneTowerArmor
    ('Oblivion.esm', 0x06A8F0): 7, # DoomstoneWarrior
    ('Oblivion.esm', 0x047AD0): 7, # PwRaceBretonShield
    ('Oblivion.esm', 0x047AD5): 7, # PwRaceDarkElfGuardian
    ('Oblivion.esm', 0x047ADE): 7, # PwRaceImperialAbsorbFatigue
    ('Oblivion.esm', 0x047ADD): 7, # PwRaceImperialCharm
    ('Oblivion.esm', 0x047ADF): 7, # PwRaceKhajiitDemoralize
    ('Oblivion.esm', 0x047AE4): 7, # PwRaceNordFrostDamage
    ('Oblivion.esm', 0x047AE3): 7, # PwRaceNordShield
    ('Oblivion.esm', 0x047AD3): 7, # PwRaceOrcBerserk
    ('Oblivion.esm', 0x047AE7): 7, # PwRaceRedguardFortify
    ('Oblivion.esm', 0x047AE9): 7, # PwRaceWoodElfCommandCreature
    ('Oblivion.esm', 0x03BEDB): 7, # VampireEmbraceofShadows
    ('Oblivion.esm', 0x03BEDC): 7, # VampireReignofTerror
    ('Oblivion.esm', 0x03BED9): 7, # VampireSeduction    
    #--Shivering Isles
    ('Oblivion.esm', 0x08F024): 7, # SE02BlessingDementia
    ('Oblivion.esm', 0x08F023): 7, # SE02BlessingMania
    ('Oblivion.esm', 0x03161E): 5, # SE07SaintSpell
    ('Oblivion.esm', 0x03161D): 5, # SE07SeducerSpell
    ('Oblivion.esm', 0x05DD22): 3, # SE14WeatherSpell
    ('Oblivion.esm', 0x081C35): 7, # SE44Frenzy
    ('Oblivion.esm', 0x018DBD): 6, # SEPwSummonDarkSeducer
    ('Oblivion.esm', 0x014B35): 6, # SEPwSummonFleshAtronach
    ('Oblivion.esm', 0x018DBC): 6, # SEPwSummonGoldenSaint
    ('Oblivion.esm', 0x050C76): 3, # SE09PwGKHead1
    ('Oblivion.esm', 0x050C77): 3, # SE09PwGKHead2
    ('Oblivion.esm', 0x050C78): 3, # SE09PwGKHeart1
    ('Oblivion.esm', 0x050C79): 3, # SE09PwGKHeart2
    ('Oblivion.esm', 0x050C7A): 3, # SE09PwGKLeftArm1
    ('Oblivion.esm', 0x050C7B): 3, # SE09PwGKLeftArm2
    ('Oblivion.esm', 0x050C7C): 3, # SE09PwGKLeftArm3
    ('Oblivion.esm', 0x050C82): 3, # SE09PwGKLegs1
    ('Oblivion.esm', 0x050C83): 3, # SE09PwGKLegs2
    ('Oblivion.esm', 0x050C7D): 3, # SE09PwGKRightArm1
    ('Oblivion.esm', 0x050C7E): 3, # SE09PwGKRightArm2
    ('Oblivion.esm', 0x050C7F): 3, # SE09PwGKRightArm3
    ('Oblivion.esm', 0x050C80): 3, # SE09PwGKTorso1
    ('Oblivion.esm', 0x050C81): 3, # SE09PwGKTorso2
    ('Oblivion.esm', 0x08E93F): 3, # SESuicidePower
    
    #--Orrery
    ('DLCOrrery.esp', 0x11DC5F): 7, # Masser's Might
    ('DLCOrrery.esp', 0x11DC60): 7, # Masser's Grace
    ('DLCOrrery.esp', 0x11DC62): 7, # Secunda's Will
    ('DLCOrrery.esp', 0x11DC64): 7, # Secunda's Opportunity
    ('DLCOrrery.esp', 0x11DC66): 7, # Masser's Alacrity
    ('DLCOrrery.esp', 0x11DC68): 7, # Secunda's Magnetism
    ('DLCOrrery.esp', 0x11DC6A): 7, # Secunda's Brilliance
    ('DLCOrrery.esp', 0x11DC6C): 7, # Masser's Courage
    }

# Repair Factions -------------------------------------------------------------
#--Formids for npcs which legitimately have no faction membership
repairFactions_legitNullSpells = set((
    #--MS47 Aleswell Invisibility
    0x0002F85F, #Sakeepa
    0x0002F861, #ShagolgroBumph
    0x0002F864, #DiramSerethi
    0x0002F865, #AdosiSerethi
    0x0002F866, #UrnsiSerethi
    ))

repairFactions_legitNullFactions = set((
    #0x00012106, #SEThadon (Between SE07 and SE12) Safer to leave in.
    #0x00012107, #SESyl (Between SE07 and SE12) Safer to leave in.
    #0x00031540, #Mirisa (Only in Cropsford, but doesn't hurt to leave her in it.)
    ))

repairFactions_legitDroppedFactions = set((
    (0x000034CC,0x000034B9), #UlrichLeland CheydinhalGuardFaction
    (0x000034CC,0x000034BB), #UlrichLeland CheydinhalCastleFaction
    (0x000055C2,0x00090E31), #CheydinhalGuardCastlePostDay01 CheydinhalCorruptGuardsFactionMS10
    (0x000055C4,0x00090E31), #CheydinhalGuardCastlePostNight01 CheydinhalCorruptGuardsFactionMS10
    (0x000055C5,0x00090E31), #CheydinhalGuardCastlePostNight02 CheydinhalCorruptGuardsFactionMS10
    (0x000055C7,0x00090E31), #CheydinhalGuardCityPatrolDay02 CheydinhalCorruptGuardsFactionMS10
    (0x000055C8,0x00090E31), #CheydinhalGuardCityPatrolNight01 CheydinhalCorruptGuardsFactionMS10
    (0x000055C9,0x00090E31), #CheydinhalGuardCityPatrolNight02 CheydinhalCorruptGuardsFactionMS10
    (0x000055CB,0x00090E31), #CheydinhalGuardCityPostDay02 CheydinhalCorruptGuardsFactionMS10
    (0x000055CC,0x00090E31), #CheydinhalGuardCityPostNight01 CheydinhalCorruptGuardsFactionMS10
    (0x000055CD,0x00090E31), #CheydinhalGuardCityPostNight02 CheydinhalCorruptGuardsFactionMS10
    (0x000055D2,0x00090E31), #CheydinhalGuardCastlePatrolDay01 CheydinhalCorruptGuardsFactionMS10
    (0x000055D3,0x00090E31), #CheydinhalGuardCastlePatrolNight01 CheydinhalCorruptGuardsFactionMS10
    (0x000055D4,0x00090E31), #CheydinhalGuardCountEscort CheydinhalCorruptGuardsFactionMS10
    (0x000055D5,0x00090E31), #CheydinhalGuardJailorDay CheydinhalCorruptGuardsFactionMS10
    (0x000055D6,0x00090E31), #CheydinhalGuardJailorNight CheydinhalCorruptGuardsFactionMS10
    (0x0000BD60,0x00091ADB), #Larthjar Prisoners
    (0x00012106,0x0001557F), #SEThadon SENewSheothBliss
    (0x00012106,0x0001AD45), #SEThadon SEManiaFaction
    (0x00012106,0x00056036), #SEThadon SE07ManiaHouseFaction
    (0x00012107,0x00013A69), #SESyl SENewSheothFaction
    (0x00012107,0x00015580), #SESyl SENewSheothCrucible
    (0x00012107,0x0001723D), #SESyl SE07ASylFaction
    (0x00012107,0x0001AD46), #SESyl SEDementiaFaction
    (0x00012107,0x0007E0CB), #SESyl SE07DementiaHouseFaction
    (0x0001CF76,0x00009275), #Seridur ICFaction
    (0x0001CF76,0x000947B9), #Seridur SeridurHouseFaction
    (0x0001CF76,0x000980DD), #Seridur OrderoftheVirtuousBlood
    (0x000222A8,0x00028D98), #ReynaldJemane ChorrolFaction
    (0x00023999,0x00028D98), #Jauffre ChorrolFaction
    (0x00023E86,0x000034B7), #GuilbertJemane NewlandsLodgeFaction
    (0x00023E86,0x000034BA), #GuilbertJemane CheydinhalFaction
    (0x00023F2A,0x0001EE1E), #Baurus BladesCG
    (0x00024165,0x0002228F), #Maglir FightersGuild
    (0x00024E0A,0x000272BE), #Jeelius MythicDawnPrisoner
    (0x00026D9A,0x00009275), #AudensAvidius ICFaction
    (0x00026D9A,0x0003486F), #AudensAvidius ImperialWatch
    (0x00026D9A,0x00083595), #AudensAvidius CourierCustomers
    (0x00026D9A,0x0018B117), #AudensAvidius ImperialLegion
    (0x0002AB4E,0x0002AB4D), #Umbacano UmbacanoFaction
    (0x0002AF3E,0x0002AEFA), #Srazirr ClaudeMaricThugFaction
    (0x0002CD21,0x00022296), #Falcar MagesGuild
    (0x0002D01E,0x00035EA9), #Jskar BrumaFaction
    (0x0002D8C0,0x00022296), #Kalthar MagesGuild
    (0x00032940,0x000C48A1), #MG16NecromancerMale1 MG16FortOntusMageFaction
    (0x00032941,0x000C48A1), #MG16NecromancerFemale2 MG16FortOntusMageFaction
    (0x00032943,0x000C48A1), #MG16NecromancerMale2 MG16FortOntusMageFaction
    (0x00033907,0x0003B3F6), #Martin KvatchFaction
    (0x00033B8B,0x000C48A1), #MG16NecromancerFemale3 MG16FortOntusMageFaction
    (0x00033B8D,0x000C48A1), #MG16NecromancerMale3 MG16FortOntusMageFaction
    (0x0003486E,0x00009275), #HieronymusLex ICFaction
    (0x0003486E,0x0003486F), #HieronymusLex ImperialWatch
    (0x00034E86,0x00029F82), #Cingor MythicDawn
    (0x00034EAD,0x00022296), #Caranya MagesGuild
    (0x0003529A,0x00024164), #MyvrynaArano ThievesGuild
    (0x0003529A,0x0003AB39), #MyvrynaArano ICWaterfrontResident
    (0x0003563B,0x00028E77), #Amusei SkingradFaction
    (0x00035649,0x00028E77), #MercatorHosidus SkingradFaction
    (0x00035649,0x0002A09C), #MercatorHosidus SkingradCastleFaction
    (0x00035ECB,0x00035EA9), #Jearl BrumaFaction
    (0x0003628D,0x00009274), #VelwynBenirus AnvilFaction
    (0x0004EF69,0x00090E31), #CheydinhalGuardCityPostDay03 CheydinhalCorruptGuardsFactionMS10
    (0x0004EFF9,0x00090E31), #CheydinhalGuardCityPostDay04 CheydinhalCorruptGuardsFactionMS10
    (0x0004EFFA,0x00090E31), #CheydinhalGuardCityPostNight04 CheydinhalCorruptGuardsFactionMS10
    ))

# Messages Text ===============================================================
messagesHeader = """<html>
<head>
<meta http-equiv="content-type" content="text/html; charset=iso-8859-1" />
	<title>Private Message Archive</title>
	<style type="text/css">
		html{
			overflow-x: auto;
		}
		
		body{
			background-color: #fff;
			color: #000;
			font-family: Verdana, Tahoma, Arial, sans-serif;
			font-size: 11px;
			margin:0px;
			padding:0px;
			text-align:center;
		   }
		   
		a:link, a:visited, a:active{
			color: #000;
			text-decoration: underline;
		}
		
		a:hover{
			color: #465584;
			text-decoration:underline;
		}
		
		img{
			border: 0;
			vertical-align: middle;
		}
				
		#ipbwrapper{
			margin: 0 auto 0 auto;
			text-align: left;
			width: 95%;
		}
		
		.post1{
			background-color: #F5F9FD;
		}
		
		.post2{
			background-color: #EEF2F7;
		}
	
		/* Common elements */
		.row1{
			background-color: #F5F9FD;
		}
		
		.row1{
			background-color: #DFE6EF;
		}
		
		.row3{
			background-color: #EEF2F7;
		}
		
		.row2{
			background-color: #E4EAF2;
		}
	
		/* tableborders gives the white column / row lines effect */
		.plainborder{
			background-color: #F5F9FD
			border: 1px solid #345487;
		}
		
		.tableborder{
			background-color: #FFF;
			border: 1px solid #345487;
			margin: 0;
			padding: 0;
		}
		
		.tablefill{
			background-color: #F5F9FD;
			border: 1px solid #345487;
			padding: 6px;
		}
		
		.tablepad{
			background-color: #F5F9FD;
			padding:6px;
		}
		
		.tablebasic{
			border: 0;
			margin: 0;
			padding: 0;
			width:100%;
		}
	
		.pformstrip{
			background-color: #D1DCEB;
			color: #3A4F6C;
			font-weight: bold;
			margin-top:1px
			padding:7px;
		}
		
		#QUOTE{
			background-color: #FAFCFE;
			border: 1px solid #000;
			color: #465584;
			font-family: Verdana, Arial;
			font-size: 11px;
			padding: 2px;
		}
		
		#CODE{
			background-color: #FAFCFE;
			border: 1px solid #000;
			color: #465584;
			font-family: Courier, Courier New, Verdana, Arial;
			font-size: 11px;
			padding: 2px;
		}
		/* Main table top (dark blue gradient by default) */
		.maintitle{
			background-color: #D1DCEB;
			color: #FFF;
			font-weight: bold;
			padding:8px 0px 8px 5px;
			vertical-align:middle;
		}
		
		.maintitle a:link, .maintitle  a:visited, .maintitle  a:active{
			color: #fff;
			text-decoration: none;
		}
		
		.maintitle a:hover{
			text-decoration: underline;
		}
		
		/* Topic View elements */
		.signature{
			color: #339;
			font-size: 10px;
			line-height:150%;
		}
		
		.postdetails{
			font-size: 10px;
		}
		
		.postcolor{
			font-size: 12px;
			line-height: 160%;
		}
	</style>
</head>
<body><div id="ipbwrapper">\n"""